/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/popup/drawingframemenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/baseframework/view/popup/floatingmenu',
    'io.ox/office/drawinglayer/view/drawingframe'
], function (Utils, KeyCodes, FloatingMenu, DrawingFrame) {

    'use strict';

    // class DrawingFrameMenu =================================================

    /**
     * A floating menu attached to the frame node of a specific drawing object.
     *
     * @constructor
     *
     * @extends FloatingMenu
     *
     * @param {EditView} docView
     *  The document view that contains this instance.
     *
     * @param {String} drawingType
     *  The types of drawing objects supported by this menu, as space-separated
     *  list.
     */
    var DrawingFrameMenu = FloatingMenu.extend({ constructor: function (docView, drawingType, titleLabel, initOptions) {

        // self reference
        var self = this;

        // the current drawing frame used as menu anchor
        var drawingFrame = null;

        // convert space-separated list of drawing object types to flag set
        var drawingTypeSet = Utils.makeSet(drawingType.split(/\s+/));

        var enterBusyVisibleState = false;

        // base constructor ---------------------------------------------------

        FloatingMenu.call(this, docView, titleLabel, Utils.extendOptions({
            anchorBorder: 'right left'
        }, initOptions, {
            anchor: function () { return drawingFrame; },
            anchorPadding: 6,
            coverAnchor: true
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the position of the menu, after the drawing selection has
         * changed. If no matching drawing object is selected, the menu will be
         * hidden.
         */
        function renderDrawingSelectionHandler(event, drawingFrames) {

            // do nothing in read-only mode
            if (!docView.isEditable()) { return; }

            // hide menu if selection is anything but a single drawing object of the specified type (TODO: support multi-selections?)
            if ((drawingFrames.length === 1) && (DrawingFrame.getDrawingType(drawingFrames[0]) in drawingTypeSet)) {

                // workaround for Bug 51392, normal group event did not work as expected
                if (self.isVisible()) {
                    self.getNode().find('.group.text-field.focused input').trigger(new $.Event('keydown', { keyCode: KeyCodes.ESCAPE }));
                    self.show();
                }

                drawingFrame = drawingFrames[0];
                self.refreshImmediately();
            } else {
                drawingFrame = null;
                self.hide();
            }
        }

        // initialization -----------------------------------------------------

        // hide the menu automatically in read-only mode
        this.listenTo(docView.getApp(), 'docs:editmode:leave docs:beforequit', function () {
            self.hide();
            enterBusyVisibleState = false;
        });

        // adjust position and visibility of the menu according to the current selection
        this.listenTo(docView, 'render:drawingselection', renderDrawingSelectionHandler);

        // listen to busy mode enter and hide the menu because it's above the busy screen
        this.listenTo(docView, 'busy:enter', function () {
            enterBusyVisibleState = self.isVisible();
            // self.hide();
        });

        // show the menu after busy mode leave if it was visible on busy:enter
        this.listenTo(docView, 'busy:leave', function () {
            if (enterBusyVisibleState) {
                // self.show();
                enterBusyVisibleState = false;
            }
        });

        self.on('popup:hide', function () {
            docView.getApp().getController().update();
            drawingFrame = null;
        });

        // destroy all class members
        this.registerDestructor(function () {
            self = initOptions = docView = null;
        });

    } }); // class DrawingFrameMenu

    // exports ================================================================

    return DrawingFrameMenu;

});
