/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/range'
], function (SheetHelper, Range) {

    'use strict';

    // class Range ============================================================

    describe('Spreadsheet class Range', function () {

        it('should exist', function () {
            expect(Range).to.be.a('function');
        });
        it('should be extendable', function () {
            expect(Range).itself.to.respondTo('extend');
        });

        // private helpers ----------------------------------------------------

        var i = SheetHelper.i;
        var a = SheetHelper.a;
        var r = SheetHelper.r;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            var a1 = a('A2'), a2 = a('C4');
            it('should create a range address', function () {
                var r1 = new Range(a1, a2);
                expect(r1.start).to.equal(a1);
                expect(r1.end).to.equal(a2);
            });
            it('should create a range address from a single address', function () {
                var r1 = new Range(a1);
                expect(r1.start).to.equal(a1);
                expect(r1.end).to.deep.equal(a1);
                expect(r1.end).to.not.equal(a1);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('create');
            });
            it('should return the adjusted range address', function () {
                var r1 = new Range(a('A2'), a('C4'));
                expect(Range.create(0, 1, 2, 3)).to.deep.equal(r1);
                expect(Range.create(2, 1, 0, 3)).to.deep.equal(r1);
                expect(Range.create(0, 3, 2, 1)).to.deep.equal(r1);
                expect(Range.create(2, 3, 0, 1)).to.deep.equal(r1);
            });
        });

        describe('method "createFromIntervals"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromIntervals');
            });
            it('should create range address from intervals', function () {
                expect(Range.createFromIntervals(i('B:C'), i('3:4'))).to.deep.equal(r('B3:C4'));
            });
        });

        describe('method "createFromColInterval"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromColInterval');
            });
            it('should create range address from column interval and rows', function () {
                expect(Range.createFromColInterval(i('B:C'), 2, 3)).to.deep.equal(r('B3:C4'));
                expect(Range.createFromColInterval(i('B:C'), 2)).to.deep.equal(r('B3:C3'));
            });
        });

        describe('method "createFromRowInterval"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromRowInterval');
            });
            it('should create range address from row interval and columns', function () {
                expect(Range.createFromRowInterval(i('3:4'), 1, 2)).to.deep.equal(r('B3:C4'));
                expect(Range.createFromRowInterval(i('3:4'), 1)).to.deep.equal(r('B3:B4'));
            });
        });

        describe('method "createFromAddresses"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromAddresses');
            });
            it('should return the adjusted range address', function () {
                expect(Range.createFromAddresses(a('B3'), a('D5'))).to.deep.equal(r('B3:D5'));
                expect(Range.createFromAddresses(a('B5'), a('D3'))).to.deep.equal(r('B3:D5'));
                expect(Range.createFromAddresses(a('D3'), a('B5'))).to.deep.equal(r('B3:D5'));
                expect(Range.createFromAddresses(a('D5'), a('B3'))).to.deep.equal(r('B3:D5'));
            });
            it('should return range address for single cell', function () {
                expect(Range.createFromAddresses(a('B3'))).to.deep.equal(r('B3:B3'));
            });
            it('should clone addresses', function () {
                var a1 = a('A2'), a2 = a('C4'), r1 = Range.createFromAddresses(a1, a2);
                expect(r1.start).to.deep.equal(a1);
                expect(r1.start).to.not.equal(a1);
                expect(r1.end).to.deep.equal(a2);
                expect(r1.end).to.not.equal(a2);
            });
        });

        describe('method "createFromAddressAndSize"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromAddressAndSize');
            });
            it('should return the resulting range address', function () {
                expect(Range.createFromAddressAndSize(a('B3'), 1, 1)).to.deep.equal(r('B3:B3'));
                expect(Range.createFromAddressAndSize(a('B3'), 4, 1)).to.deep.equal(r('B3:E3'));
                expect(Range.createFromAddressAndSize(a('B3'), 1, 3)).to.deep.equal(r('B3:B5'));
                expect(Range.createFromAddressAndSize(a('B3'), 4, 3)).to.deep.equal(r('B3:E5'));
            });
            it('should clone addresses', function () {
                var a1 = a('A2'), r1 = Range.createFromAddressAndSize(a1, 1, 1);
                expect(r1.start).to.deep.equal(a1);
                expect(r1.start).to.not.equal(a1);
                expect(r1.end).to.deep.equal(a1);
                expect(r1.end).to.not.equal(a1);
            });
        });

        describe('method "parse"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('parse');
            });
            var r1 = Range.create(0, 1, 2, 3);
            it('should return the correct range address', function () {
                expect(Range.parse('A2:C4')).to.deep.equal(r1);
                expect(Range.parse('A1:ZZZZ99999999')).to.deep.equal(Range.create(0, 0, 475253, 99999998));
            });
            it('should return the adjusted range address', function () {
                expect(Range.parse('C2:A4')).to.deep.equal(r1);
                expect(Range.parse('A4:C2')).to.deep.equal(r1);
                expect(Range.parse('C4:A2')).to.deep.equal(r1);
            });
            it('should accept lower-case columns and leading zero characters', function () {
                expect(Range.parse('a01:aa010')).to.deep.equal(Range.create(0, 0, 26, 9));
                expect(Range.parse('A001:aA0010')).to.deep.equal(Range.create(0, 0, 26, 9));
            });
            it('should return null for invalid range names', function () {
                expect(Range.parse('')).to.equal(null);
                expect(Range.parse('A')).to.equal(null);
                expect(Range.parse('1')).to.equal(null);
                expect(Range.parse('A1')).to.equal(null);
                expect(Range.parse('$')).to.equal(null);
                expect(Range.parse('$A$1:$A$1')).to.equal(null);
                expect(Range.parse('A0:A1')).to.equal(null);
                expect(Range.parse('AAAAA1:AAAAA1')).to.equal(null);
                expect(Range.parse('A100000000:A100000000')).to.equal(null);
            });
            it('should return null if custom maximum does not fit', function () {
                expect(Range.parse('B2:C3', a('C3'))).to.be.an.instanceof(Range);
                expect(Range.parse('B2:C3', a('C2'))).to.equal(null);
                expect(Range.parse('B2:C3', a('B3'))).to.equal(null);
            });
        });

        describe('method "compare"', function () {
            var r1 = r('C3:E5');
            it('should exist', function () {
                expect(Range).itself.to.respondTo('compare');
            });
            it('should return 0 for equal ranges', function () {
                expect(Range.compare(r1, r1)).to.equal(0);
            });
            it('should return negative value for ranges in order', function () {
                expect(Range.compare(r1, r('B4:D4'))).to.be.below(0);
                expect(Range.compare(r1, r('D3:D4'))).to.be.below(0);
                expect(Range.compare(r1, r('C3:D6'))).to.be.below(0);
                expect(Range.compare(r1, r('C3:F5'))).to.be.below(0);
            });
            it('should return positive value for ranges in reversed order', function () {
                expect(Range.compare(r1, r('D2:F6'))).to.be.above(0);
                expect(Range.compare(r1, r('B3:F6'))).to.be.above(0);
                expect(Range.compare(r1, r('C3:F4'))).to.be.above(0);
                expect(Range.compare(r1, r('C3:D5'))).to.be.above(0);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "key"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('key');
            });
            it('should return unique key for range address', function () {
                expect(r('A2:A2').key()).to.equal('0,1:0,1');
                expect(r('B3:D5').key()).to.equal('1,2:3,4');
                expect(r('A1:ZZ1000').key()).to.equal('0,0:701,999');
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('clone');
            });
            it('should return a clone', function () {
                var r1 = r('A2:C4'), r2 = r1.clone();
                expect(r2).to.be.an.instanceof(Range);
                expect(r2).to.not.equal(r1);
                expect(r2.start).to.not.equal(r1.start);
                expect(r2.end).to.not.equal(r1.end);
                expect(r2).to.deep.equal(r1);
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('equals');
            });
            var range = r('A2:C4');
            it('should return true for equal ranges', function () {
                expect(r('A2:A4').equals(r('A2:A4'))).to.equal(true);
                expect(r('A2:C2').equals(r('A2:C2'))).to.equal(true);
                expect(r('A2:C4').equals(r('A2:C4'))).to.equal(true);
                expect(range.equals(range)).to.equal(true);
            });
            it('should return false for different ranges', function () {
                expect(range.equals(r('A2:C5'))).to.equal(false);
                expect(range.equals(r('A2:D4'))).to.equal(false);
                expect(range.equals(r('A3:C4'))).to.equal(false);
                expect(range.equals(r('B2:C4'))).to.equal(false);
            });
        });

        describe('method "differs"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('differs');
            });
            var range = r('A2:C4');
            it('should return false for equal ranges', function () {
                expect(r('A2:A4').differs(r('A2:A4'))).to.equal(false);
                expect(r('A2:C2').differs(r('A2:C2'))).to.equal(false);
                expect(r('A2:C4').differs(r('A2:C4'))).to.equal(false);
                expect(range.differs(range)).to.equal(false);
            });
            it('should return true for different ranges', function () {
                expect(range.differs(r('A2:C5'))).to.equal(true);
                expect(range.differs(r('A2:D4'))).to.equal(true);
                expect(range.differs(r('A3:C4'))).to.equal(true);
                expect(range.differs(r('B2:C4'))).to.equal(true);
            });
        });

        describe('method "compareTo"', function () {
            var r1 = r('C3:E5');
            it('should exist', function () {
                expect(Range).to.respondTo('compareTo');
            });
            it('should return 0 for equal ranges', function () {
                expect(r1.compareTo(r1)).to.equal(0);
            });
            it('should return negative value for ranges in order', function () {
                expect(r1.compareTo(r('B4:D4'))).to.be.below(0);
                expect(r1.compareTo(r('D3:D4'))).to.be.below(0);
                expect(r1.compareTo(r('C3:D6'))).to.be.below(0);
                expect(r1.compareTo(r('C3:F5'))).to.be.below(0);
            });
            it('should return positive value for ranges in reversed order', function () {
                expect(r1.compareTo(r('D2:F6'))).to.be.above(0);
                expect(r1.compareTo(r('B3:F6'))).to.be.above(0);
                expect(r1.compareTo(r('C3:F4'))).to.be.above(0);
                expect(r1.compareTo(r('C3:D5'))).to.be.above(0);
            });
        });

        describe('method "cols"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('cols');
            });
            it('should return the correct count', function () {
                expect(r('A2:A3').cols()).to.equal(1);
                expect(r('C2:C3').cols()).to.equal(1);
                expect(r('C2:D3').cols()).to.equal(2);
                expect(r('C2:E3').cols()).to.equal(3);
            });
        });

        describe('method "rows"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('rows');
            });
            it('should return the correct count', function () {
                expect(r('B1:C1').rows()).to.equal(1);
                expect(r('B3:C3').rows()).to.equal(1);
                expect(r('B3:C4').rows()).to.equal(2);
                expect(r('B3:C5').rows()).to.equal(3);
            });
        });

        describe('method "size"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('size');
            });
            it('should return the correct coumn count', function () {
                expect(r('B1:C1').size(true)).to.equal(2);
            });
            it('should return the correct row count', function () {
                expect(r('B1:C1').size(false)).to.equal(1);
            });
        });

        describe('method "cells"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('cells');
            });
            it('should return the correct count', function () {
                expect(r('A1:A1').cells()).to.equal(1);
                expect(r('C3:C3').cells()).to.equal(1);
                expect(r('C3:D3').cells()).to.equal(2);
                expect(r('C3:E3').cells()).to.equal(3);
                expect(r('C3:C4').cells()).to.equal(2);
                expect(r('C3:C5').cells()).to.equal(3);
                expect(r('C3:D4').cells()).to.equal(4);
                expect(r('C3:E5').cells()).to.equal(9);
            });
        });

        describe('method "singleCol"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('singleCol');
            });
            it('should return the correct result', function () {
                expect(r('B3:B3').singleCol()).to.equal(true);
                expect(r('B3:D3').singleCol()).to.equal(false);
                expect(r('B3:B5').singleCol()).to.equal(true);
                expect(r('B3:D5').singleCol()).to.equal(false);
            });
        });

        describe('method "singleRow"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('singleRow');
            });
            it('should return the correct result', function () {
                expect(r('B3:B3').singleRow()).to.equal(true);
                expect(r('B3:D3').singleRow()).to.equal(true);
                expect(r('B3:B5').singleRow()).to.equal(false);
                expect(r('B3:D5').singleRow()).to.equal(false);
            });
        });

        describe('method "singleLine"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('singleLine');
            });
            it('should return the correct result for columns', function () {
                expect(r('B3:B3').singleLine(true)).to.equal(true);
                expect(r('B3:D3').singleLine(true)).to.equal(false);
                expect(r('B3:B5').singleLine(true)).to.equal(true);
                expect(r('B3:D5').singleLine(true)).to.equal(false);
            });
            it('should return the correct result for rows', function () {
                expect(r('B3:B3').singleLine(false)).to.equal(true);
                expect(r('B3:D3').singleLine(false)).to.equal(true);
                expect(r('B3:B5').singleLine(false)).to.equal(false);
                expect(r('B3:D5').singleLine(false)).to.equal(false);
            });
        });

        describe('method "single"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('single');
            });
            it('should return the correct result', function () {
                expect(r('B3:B3').single()).to.equal(true);
                expect(r('B3:D3').single()).to.equal(false);
                expect(r('B3:B5').single()).to.equal(false);
                expect(r('B3:D5').single()).to.equal(false);
            });
        });

        describe('method "equalSize"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('equalSize');
            });
            it('should return the correct result', function () {
                var r1 = r('B3:D5');
                expect(r1.equalSize(r('B3:D5'))).to.equal(true);
                expect(r1.equalSize(r('C3:E5'))).to.equal(true);
                expect(r1.equalSize(r('B4:D6'))).to.equal(true);
                expect(r1.equalSize(r('B3:C5'))).to.equal(false);
                expect(r1.equalSize(r('B3:E5'))).to.equal(false);
                expect(r1.equalSize(r('B3:D4'))).to.equal(false);
                expect(r1.equalSize(r('B3:D6'))).to.equal(false);
            });
        });

        describe('method "sizeFitsInto"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('sizeFitsInto');
            });
            var r1 = r('B1:E6');
            it('should return false for mismatching sizes', function () {
                expect(r('A1:C1').sizeFitsInto(r1)).to.equal(false);
                expect(r('A1:A4').sizeFitsInto(r1)).to.equal(false);
            });
            it('should return true for matching sizes', function () {
                expect(r('A1:A1').sizeFitsInto(r1)).to.equal(true);
                expect(r('A1:B1').sizeFitsInto(r1)).to.equal(true);
                expect(r('A1:D1').sizeFitsInto(r1)).to.equal(true);
                expect(r('A1:A2').sizeFitsInto(r1)).to.equal(true);
                expect(r('A1:A3').sizeFitsInto(r1)).to.equal(true);
                expect(r('A1:A6').sizeFitsInto(r1)).to.equal(true);
            });
        });

        describe('method "startsAt"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('startsAt');
            });
            it('should return the correct result', function () {
                var r1 = r('B3:D5');
                expect(r1.startsAt(a('B3'))).to.equal(true);
                expect(r1.startsAt(a('B2'))).to.equal(false);
                expect(r1.startsAt(a('B4'))).to.equal(false);
                expect(r1.startsAt(a('A3'))).to.equal(false);
                expect(r1.startsAt(a('D3'))).to.equal(false);
                expect(r1.startsAt(a('D5'))).to.equal(false);
            });
        });

        describe('method "endsAt"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('endsAt');
            });
            it('should return the correct result', function () {
                var r1 = r('B3:D5');
                expect(r1.endsAt(a('D5'))).to.equal(true);
                expect(r1.endsAt(a('D4'))).to.equal(false);
                expect(r1.endsAt(a('D6'))).to.equal(false);
                expect(r1.endsAt(a('C5'))).to.equal(false);
                expect(r1.endsAt(a('E5'))).to.equal(false);
                expect(r1.endsAt(a('B3'))).to.equal(false);
            });
        });

        describe('method "indexAt"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('indexAt');
            });
            it('should return the correct result', function () {
                var r1 = r('B3:D5');
                expect(r1.indexAt(a('B3'))).to.equal(0);
                expect(r1.indexAt(a('C3'))).to.equal(1);
                expect(r1.indexAt(a('D3'))).to.equal(2);
                expect(r1.indexAt(a('B4'))).to.equal(3);
                expect(r1.indexAt(a('C4'))).to.equal(4);
                expect(r1.indexAt(a('D4'))).to.equal(5);
                expect(r1.indexAt(a('B5'))).to.equal(6);
                expect(r1.indexAt(a('C5'))).to.equal(7);
                expect(r1.indexAt(a('D5'))).to.equal(8);
            });
        });

        describe('method "addressAt"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('addressAt');
            });
            it('should return the correct result', function () {
                var r1 = r('B3:D5');
                expect(r1.addressAt(0)).to.deep.equal(a('B3'));
                expect(r1.addressAt(1)).to.deep.equal(a('C3'));
                expect(r1.addressAt(2)).to.deep.equal(a('D3'));
                expect(r1.addressAt(3)).to.deep.equal(a('B4'));
                expect(r1.addressAt(4)).to.deep.equal(a('C4'));
                expect(r1.addressAt(5)).to.deep.equal(a('D4'));
                expect(r1.addressAt(6)).to.deep.equal(a('B5'));
                expect(r1.addressAt(7)).to.deep.equal(a('C5'));
                expect(r1.addressAt(8)).to.deep.equal(a('D5'));
            });
        });

        describe('method "containsCol"', function () {
            var range = r('C2:D9');
            it('should exist', function () {
                expect(Range).to.respondTo('containsCol');
            });
            it('should return false for columns left of the range', function () {
                expect(range.containsCol(0)).to.equal(false);
                expect(range.containsCol(1)).to.equal(false);
            });
            it('should return true for columns inside the range', function () {
                expect(range.containsCol(2)).to.equal(true);
                expect(range.containsCol(3)).to.equal(true);
            });
            it('should return false for columns right of the range', function () {
                expect(range.containsCol(4)).to.equal(false);
                expect(range.containsCol(9)).to.equal(false);
            });
        });

        describe('method "containsRow"', function () {
            var range = r('B3:I4');
            it('should exist', function () {
                expect(Range).to.respondTo('containsRow');
            });
            it('should return false for rows above the range', function () {
                expect(range.containsRow(0)).to.equal(false);
                expect(range.containsRow(1)).to.equal(false);
            });
            it('should return true for rows inside the range', function () {
                expect(range.containsRow(2)).to.equal(true);
                expect(range.containsRow(3)).to.equal(true);
            });
            it('should return false for rows below the range', function () {
                expect(range.containsRow(4)).to.equal(false);
                expect(range.containsRow(9)).to.equal(false);
            });
        });

        describe('method "containsIndex"', function () {
            var range = r('C2:C2');
            it('should exist', function () {
                expect(Range).to.respondTo('containsIndex');
            });
            it('should check column indexes', function () {
                expect(range.containsIndex(1, true)).to.equal(false);
                expect(range.containsIndex(2, true)).to.equal(true);
                expect(range.containsIndex(3, true)).to.equal(false);
            });
            it('should check row indexes', function () {
                expect(range.containsIndex(0, false)).to.equal(false);
                expect(range.containsIndex(1, false)).to.equal(true);
                expect(range.containsIndex(2, false)).to.equal(false);
            });
        });

        describe('method "containsAddress"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('containsAddress');
            });
            var range = r('B4:C5');
            it('should return false for cells outside the range', function () {
                'A3 B3 C3 D3 A4 D4 A5 D5 A6 B6 C6 D6'.split(' ').forEach(function (str) {
                    expect(range.containsAddress(a(str))).to.equal(false);
                });
            });
            it('should return true for cells inside the range', function () {
                'B4 C4 B5 C5'.split(' ').forEach(function (str) {
                    expect(range.containsAddress(a(str))).to.equal(true);
                });
            });
        });

        describe('method "isBorderAddress"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('isBorderAddress');
            });
            var range = r('B4:D6');
            it('should return false for outer cells', function () {
                'A3 B3 C3 D3 E3 A4 E4 A5 E5 A6 E6 A7 B7 C7 D7 E7'.split(' ').map(a).forEach(function (address) {
                    expect(range.isBorderAddress(address)).to.equal(false);
                });
            });
            it('should return false for inner cells', function () {
                expect(range.isBorderAddress(a('C5'))).to.equal(false);
            });
            it('should return true for border cells', function () {
                'B4 C4 D4 B5 D5 B6 C6 D6'.split(' ').map(a).forEach(function (address) {
                    expect(range.isBorderAddress(address)).to.equal(true);
                });
            });
        });

        describe('method "contains"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('contains');
            });
            var range = r('C3:F6');
            it('should return false for ranges outside the range', function () {
                'A1:B8 G1:H8 A1:H2 A7:H8'.split(' ').forEach(function (str) {
                    expect(range.contains(r(str))).to.equal(false);
                });
            });
            it('should return false for ranges partly overlapping the range', function () {
                'A1:C8 F1:H8 A1:H3 A6:H8 D1:E3 D6:E8 A4:C5 F5:H6'.split(' ').forEach(function (str) {
                    expect(range.contains(r(str))).to.equal(false);
                });
            });
            it('should return true for ranges inside the range', function () {
                'C3:C3 C3:F3 F3:F3 F3:F6 F6:F6 C6:F6 C6:C6 C3:C6 C3:F6 D4:E5'.split(' ').forEach(function (str) {
                    expect(range.contains(r(str))).to.equal(true);
                });
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('overlaps');
            });
            var range = r('B3:C4');
            it('should return false for distinct ranges', function () {
                expect(range.overlaps(r('D3:E4'))).to.equal(false);
                expect(r('D3:E4').overlaps(range)).to.equal(false);
                expect(range.overlaps(r('B5:C6'))).to.equal(false);
                expect(r('B5:C6').overlaps(range)).to.equal(false);
            });
            it('should return true for partly overlapping ranges', function () {
                expect(range.overlaps(r('C4:D5'))).to.equal(true);
                expect(range.overlaps(r('C2:D3'))).to.equal(true);
                expect(range.overlaps(r('A2:B3'))).to.equal(true);
                expect(range.overlaps(r('A4:B5'))).to.equal(true);
            });
            it('should return true for ranges containing each other', function () {
                expect(range.overlaps(r('B3:B3'))).to.equal(true);
                expect(range.overlaps(r('B3:B4'))).to.equal(true);
                expect(range.overlaps(r('B3:C4'))).to.equal(true);
                expect(r('B3:B4').overlaps(range)).to.equal(true);
                expect(r('B3:B3').overlaps(range)).to.equal(true);
                expect(range.overlaps(range)).to.equal(true);
            });
        });

        describe('method "colInterval"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('colInterval');
            });
            it('should return the column interval', function () {
                expect(r('B3:B3').colInterval()).to.deep.equal(i('B:B'));
                expect(r('B3:C4').colInterval()).to.deep.equal(i('B:C'));
                expect(r('B3:D5').colInterval()).to.deep.equal(i('B:D'));
                expect(r('C4:D5').colInterval()).to.deep.equal(i('C:D'));
                expect(r('D5:D5').colInterval()).to.deep.equal(i('D:D'));
            });
        });

        describe('method "rowInterval"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('rowInterval');
            });
            it('should return the row interval', function () {
                expect(r('B3:B3').rowInterval()).to.deep.equal(i('3:3'));
                expect(r('B3:C4').rowInterval()).to.deep.equal(i('3:4'));
                expect(r('B3:D5').rowInterval()).to.deep.equal(i('3:5'));
                expect(r('C4:D5').rowInterval()).to.deep.equal(i('4:5'));
                expect(r('D5:D5').rowInterval()).to.deep.equal(i('5:5'));
            });
        });

        describe('method "interval"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('interval');
            });
            it('should return the column interval', function () {
                expect(r('B3:C4').interval(true)).to.deep.equal(i('B:C'));
                expect(r('B3:D5').interval(true)).to.deep.equal(i('B:D'));
                expect(r('C4:D5').interval(true)).to.deep.equal(i('C:D'));
            });
            it('should return the row interval', function () {
                expect(r('B3:C4').interval(false)).to.deep.equal(i('3:4'));
                expect(r('B3:D5').interval(false)).to.deep.equal(i('3:5'));
                expect(r('C4:D5').interval(false)).to.deep.equal(i('4:5'));
            });
        });

        describe('method "colRange"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('colRange');
            });
            it('should return the column range', function () {
                expect(r('B3:D5').colRange(0)).to.deep.equal(r('B3:B5'));
                expect(r('B3:D5').colRange(1)).to.deep.equal(r('C3:C5'));
                expect(r('B3:D5').colRange(1, 2)).to.deep.equal(r('C3:D5'));
            });
        });

        describe('method "rowRange"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('rowRange');
            });
            it('should return the row range', function () {
                expect(r('B3:D5').rowRange(0)).to.deep.equal(r('B3:D3'));
                expect(r('B3:D5').rowRange(1)).to.deep.equal(r('B4:D4'));
                expect(r('B3:D5').rowRange(1, 2)).to.deep.equal(r('B4:D5'));
            });
        });

        describe('method "lineRange"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('lineRange');
            });
            it('should return the column range', function () {
                expect(r('B3:D5').lineRange(true, 0)).to.deep.equal(r('B3:B5'));
                expect(r('B3:D5').lineRange(true, 1)).to.deep.equal(r('C3:C5'));
                expect(r('B3:D5').lineRange(true, 1, 2)).to.deep.equal(r('C3:D5'));
            });
            it('should return the row range', function () {
                expect(r('B3:D5').lineRange(false, 0)).to.deep.equal(r('B3:D3'));
                expect(r('B3:D5').lineRange(false, 1)).to.deep.equal(r('B4:D4'));
                expect(r('B3:D5').lineRange(false, 1, 2)).to.deep.equal(r('B4:D5'));
            });
        });

        describe('method "leadingCol"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('leadingCol');
            });
            it('should return the leading column range', function () {
                expect(r('B3:D5').leadingCol()).to.deep.equal(r('B3:B5'));
            });
        });

        describe('method "trailingCol"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('trailingCol');
            });
            it('should return the trailing column range', function () {
                expect(r('B3:D5').trailingCol()).to.deep.equal(r('D3:D5'));
            });
        });

        describe('method "headerRow"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('headerRow');
            });
            it('should return the header row range', function () {
                expect(r('B3:D5').headerRow()).to.deep.equal(r('B3:D3'));
            });
        });

        describe('method "footerRow"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('footerRow');
            });
            it('should return the footer row range', function () {
                expect(r('B3:D5').footerRow()).to.deep.equal(r('B5:D5'));
            });
        });

        describe('method "mirror"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('mirror');
            });
            it('should return the mirrored range', function () {
                expect(r('A2:D3').mirror()).to.deep.equal(r('B1:C4'));
                expect(r('B1:C4').mirror()).to.deep.equal(r('A2:D3'));
            });
        });

        describe('method "expand"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('expand');
            });
            it('should return the expanded range', function () {
                var r1 = r('B3:D5'), r2 = r1.expand(a('C4'));
                expect(r2).to.deep.equal(r1);
                expect(r2).to.not.equal(r1);
                expect(r1.expand(a('A4'))).to.deep.equal(r('A3:D5'));
                expect(r1.expand(a('A1'))).to.deep.equal(r('A1:D5'));
                expect(r1.expand(a('C1'))).to.deep.equal(r('B1:D5'));
                expect(r1.expand(a('E1'))).to.deep.equal(r('B1:E5'));
                expect(r1.expand(a('E4'))).to.deep.equal(r('B3:E5'));
                expect(r1.expand(a('E7'))).to.deep.equal(r('B3:E7'));
                expect(r1.expand(a('C7'))).to.deep.equal(r('B3:D7'));
                expect(r1.expand(a('A7'))).to.deep.equal(r('A3:D7'));
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('boundary');
            });
            it('should return the bounding range', function () {
                expect(r('B3:C4').boundary(r('C4:D5'))).to.deep.equal(r('B3:D5'));
                expect(r('C3:D4').boundary(r('B4:C5'))).to.deep.equal(r('B3:D5'));
                expect(r('B4:C5').boundary(r('C3:D4'))).to.deep.equal(r('B3:D5'));
                expect(r('C4:D5').boundary(r('B3:C4'))).to.deep.equal(r('B3:D5'));
                expect(r('C4:C4').boundary(r('B3:D5'))).to.deep.equal(r('B3:D5'));
                expect(r('B3:D5').boundary(r('C4:C4'))).to.deep.equal(r('B3:D5'));
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('intersect');
            });
            it('should return null for distinct ranges', function () {
                expect(r('B3:C4').intersect(r('D3:E4'))).to.equal(null);
                expect(r('D3:E4').intersect(r('B3:C4'))).to.equal(null);
                expect(r('B3:C4').intersect(r('B5:C6'))).to.equal(null);
                expect(r('B5:C6').intersect(r('B3:C4'))).to.equal(null);
            });
            it('should return intersection range for overlapping ranges', function () {
                var range = r('B3:E5');
                expect(range.intersect(r('A2:B3'))).to.deep.equal(r('B3:B3'));
                expect(range.intersect(r('A2:B6'))).to.deep.equal(r('B3:B5'));
                expect(range.intersect(r('A4:C6'))).to.deep.equal(r('B4:C5'));
                expect(range.intersect(r('A4:F4'))).to.deep.equal(r('B4:E4'));
                expect(range.intersect(r('C1:D8'))).to.deep.equal(r('C3:D5'));
                expect(range.intersect(r('A1:F8'))).to.deep.equal(r('B3:E5'));
                expect(range.intersect(r('B3:E5'))).to.deep.equal(r('B3:E5'));
                expect(range.intersect(r('C3:D5'))).to.deep.equal(r('C3:D5'));
                expect(range.intersect(r('C4:C4'))).to.deep.equal(r('C4:C4'));
                expect(range.intersect(range)).to.deep.equal(range);
                expect(range.intersect(range)).to.not.equal(range);
            });
        });

        describe('method "getStart"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('getStart');
            });
            it('should return the correct index', function () {
                var r1 = r('B3:D5');
                expect(r1.getStart(true)).to.equal(1);
                expect(r1.getStart(false)).to.equal(2);
            });
        });

        describe('method "getEnd"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('getEnd');
            });
            it('should return the correct index', function () {
                var r1 = r('B3:D5');
                expect(r1.getEnd(true)).to.equal(3);
                expect(r1.getEnd(false)).to.equal(4);
            });
        });

        describe('method "setStart"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('setStart');
            });
            it('should change the correct index', function () {
                var r1 = r('A1:F6');
                expect(r1).to.deep.equal(r('A1:F6'));
                r1.setStart(1, true);
                expect(r1).to.deep.equal(r('B1:F6'));
                r1.setStart(2, false);
                expect(r1).to.deep.equal(r('B3:F6'));
            });
            it('should return itself', function () {
                var r1 = r('A1:F6');
                expect(r1.setStart(1, true)).to.equal(r1);
            });
        });

        describe('method "setEnd"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('setEnd');
            });
            it('should change the correct index', function () {
                var r1 = r('A1:F6');
                expect(r1).to.deep.equal(r('A1:F6'));
                r1.setEnd(3, true);
                expect(r1).to.deep.equal(r('A1:D6'));
                r1.setEnd(4, false);
                expect(r1).to.deep.equal(r('A1:D5'));
            });
            it('should return itself', function () {
                var r1 = r('A1:F6');
                expect(r1.setEnd(1, true)).to.equal(r1);
            });
        });

        describe('method "setBoth"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('setBoth');
            });
            it('should change the correct index', function () {
                var r1 = r('A1:F6');
                expect(r1).to.deep.equal(r('A1:F6'));
                r1.setBoth(3, true);
                expect(r1).to.deep.equal(r('D1:D6'));
                r1.setBoth(4, false);
                expect(r1).to.deep.equal(r('D5:D5'));
            });
            it('should return itself', function () {
                var r1 = r('A1:F6');
                expect(r1.setBoth(1, true)).to.equal(r1);
            });
        });

        describe('method "moveBoth"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('moveBoth');
            });
            it('should change the correct indexes', function () {
                var r1 = r('A1:F6');
                expect(r1).to.deep.equal(r('A1:F6'));
                r1.moveBoth(3, true);
                expect(r1).to.deep.equal(r('D1:I6'));
                r1.moveBoth(4, false);
                expect(r1).to.deep.equal(r('D5:I10'));
            });
            it('should return itself', function () {
                var r1 = r('A1:F6');
                expect(r1.moveBoth(1, true)).to.equal(r1);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('iterator');
            });
            it('should return a horizontal address iterator', function () {
                var it = r('B3:C4').iterator();
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: a('B3') });
                expect(it.next()).to.deep.equal({ done: false, value: a('C3') });
                expect(it.next()).to.deep.equal({ done: false, value: a('B4') });
                expect(it.next()).to.deep.equal({ done: false, value: a('C4') });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should return a horizontal reverse address iterator', function () {
                var it = r('B3:C4').iterator({ reverse: true });
                expect(it.next()).to.deep.equal({ done: false, value: a('C4') });
                expect(it.next()).to.deep.equal({ done: false, value: a('B4') });
                expect(it.next()).to.deep.equal({ done: false, value: a('C3') });
                expect(it.next()).to.deep.equal({ done: false, value: a('B3') });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should return a vertical address iterator', function () {
                var it = r('B3:C4').iterator({ columns: true });
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: a('B3') });
                expect(it.next()).to.deep.equal({ done: false, value: a('B4') });
                expect(it.next()).to.deep.equal({ done: false, value: a('C3') });
                expect(it.next()).to.deep.equal({ done: false, value: a('C4') });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should return a vertical reverse address iterator', function () {
                var it = r('B3:C4').iterator({ reverse: true, columns: true });
                expect(it.next()).to.deep.equal({ done: false, value: a('C4') });
                expect(it.next()).to.deep.equal({ done: false, value: a('C3') });
                expect(it.next()).to.deep.equal({ done: false, value: a('B4') });
                expect(it.next()).to.deep.equal({ done: false, value: a('B3') });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('toString');
            });
            it('should stringify the range', function () {
                expect(r('A2:C4').toString()).to.equal('A2:C4');
                expect(r('A1:ZZZZ99999999').toString()).to.equal('A1:ZZZZ99999999');
            });
            it('should stringify implicitly', function () {
                expect('<' + r('A2:C4') + '>').to.equal('<A2:C4>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('toJSON');
            });
            var r1 = r('A3:B4'), r2 = { start: [0, 2], end: [1, 3] };
            it('should convert to JSON data', function () {
                expect(r1.toJSON()).to.deep.equal(r2);
            });
            it('should stringify implicitly', function () {
                expect(JSON.parse(JSON.stringify(r1))).to.deep.equal(r2);
            });
        });
    });

    // ========================================================================
});
