/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/mathfuncs'
], function (AppHelper, SheetHelper, MathFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var r3d = SheetHelper.r3d;
    var mat = SheetHelper.mat;

    // module MathFuncs =======================================================

    describe('Spreadsheet module MathFuncs', function () {

        // the operations to be applied at the test document
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'A2', contents: [
                [1, 2, 3, 'abc', '4', true, null, null, '',  5,   ErrorCode.DIV0],
                [1, 2, 4, 8,     16,  32,   64,   128,  256, 512, 1024]
            ] },

            // sheet with auto-filter with active filter rule and manually hidden row (test for SUBTOTAL, bug 49357)
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeRows', sheet: 1, start: 2, attrs: { row: { visible: false } } },
            { name: 'changeRows', sheet: 1, start: 7, attrs: { row: { visible: false } } },
            { name: 'insertTable', sheet: 1, start: 'A1', end: 'A4', attrs: { table: { headerRow: true, filtered: true } } },
            { name: 'changeTableColumn', sheet: 1, col: 0, attrs: { filter: { type: 'discrete', entries: ['1', '4'] } } },
            { name: 'changeCells', sheet: 1, start: 'A1', contents: [['head'], [1], [2], [4]] },
            { name: 'changeCells', sheet: 1, start: 'A6', contents: [['head'], [1], [2], [4]] },

            // sheet with auto-filter without active filter rule and manually hidden row (test for SUBTOTAL, bug 49357)
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'changeRows', sheet: 2, start: 7, attrs: { row: { visible: false } } },
            { name: 'insertTable', sheet: 2, start: 'A1', end: 'A4', attrs: { table: { headerRow: true, filtered: true } } },
            { name: 'changeCells', sheet: 2, start: 'A1', contents: [['head'], [1], [2], [4]] },
            { name: 'changeCells', sheet: 2, start: 'A6', contents: [['head'], [1], [2], [4]] }
        ];

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml', OPERATIONS);
        var moduleTester = SheetHelper.createFunctionModuleTester(MathFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('AGGREGATE', function (AGGREGATE) {
            var mat1 = mat([[3, 2, ErrorCode.NUM, 4, 5, 6]]);
            var mat2 = mat([[3, 2, 4, 5, 6]]);
            var mat3 = mat([[3, 2, 4, 5, 6, ErrorCode.NUM, 2]]);
            it('should return the aggregate of its arguments', function () {
                // AVERAGE
                expect(AGGREGATE(1, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(1, 1, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(1, null, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(1, 2, mat1)).to.equal(4);
                expect(AGGREGATE(1, 3, mat1)).to.equal(4);
                expect(AGGREGATE(1, 6, mat1)).to.equal(4);
                expect(AGGREGATE(1, 7, mat1)).to.equal(4);
                // COUNT
                expect(AGGREGATE(2, 6, mat1)).to.equal(5);
                // COUNTA
                expect(AGGREGATE(3, 6, mat1)).to.equal(5);
                expect(AGGREGATE(3, 4, mat1)).to.equal(6);
                // MAX
                expect(AGGREGATE(4, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(4, 6, mat1)).to.equal(6);
                // MIN
                expect(AGGREGATE(5, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(5, 6, mat1)).to.equal(2);
                // PRODUCT
                expect(AGGREGATE(6, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(6, 4, mat2)).to.equal(720);
                expect(AGGREGATE(6, 6, mat1)).to.equal(720);
                // STDEV.S
                expect(AGGREGATE(7, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(7, 6, mat1)).to.be.closeTo(1.58113883, 1e-6);
                // STDEV.P
                expect(AGGREGATE(8, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(8, 6, mat1)).to.be.closeTo(1.414213562, 1e-6);
                // SUM
                expect(AGGREGATE(9, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(9, 4, mat2)).to.equal(20);
                expect(AGGREGATE(9, 6, mat1)).to.equal(20);
                // VAR.S
                expect(AGGREGATE(10, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(10, 4, mat2)).to.equal(2.5);
                expect(AGGREGATE(10, 6, mat1)).to.equal(2.5);
                // VAR.P
                expect(AGGREGATE(11, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(11, 4, mat2)).to.equal(2);
                expect(AGGREGATE(11, 6, mat1)).to.equal(2);
                // MEDIAN
                expect(AGGREGATE(12, 4, mat1)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(12, 4, mat2)).to.equal(4);
                expect(AGGREGATE(12, 6, mat1)).to.equal(4);
                // MODE.SNGL
                expect(AGGREGATE(13, 4, mat3)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(13, 6, mat3)).to.equal(2);
                // LARGE
                expect(AGGREGATE(14, 4, mat1, 3)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(14, 6, mat1, 3)).to.equal(4);
                // SMALL
                expect(AGGREGATE(15, 4, mat1, 3)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(15, 6, mat1, 3)).to.equal(4);
                // PERCENTILE.INC
                expect(AGGREGATE(16, 4, mat1, 0.4)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(16, 6, mat1, 0.4)).to.equal(3.6);
                // QUARTILE.INC
                expect(AGGREGATE(17, 4, mat1, 1.5)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(17, 6, mat1, 1.5)).to.equal(3);
                // PERCENTILE.EXC
                expect(AGGREGATE(18, 4, mat1, 0.4)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(18, 6, mat1, 0.4)).to.be.closeTo(3.4, 1e-6);
                // QUARTILE.EXC
                expect(AGGREGATE(19, 4, mat1, 1.5)).to.equal(ErrorCode.NUM);
                expect(AGGREGATE(19, 6, mat1, 1.5)).to.equal(2.5);
            });
        });

        moduleTester.testFunction('CEILING.MATH', function (CEILING_MATH) {
            it('should return the ceiling of its arguments', function () {
                expect(CEILING_MATH(6.7)).to.equal(7);
                expect(CEILING_MATH(24.3, 5)).to.equal(25);
                expect(CEILING_MATH(22, 5)).to.equal(25);
                expect(CEILING_MATH(-8.1, 2)).to.equal(-8);
            });
            it('should return the ceiling of its arguments with assigned mode', function () {
                expect(CEILING_MATH(-5.4, 1, -1)).to.equal(-6);
                expect(CEILING_MATH(-5.4, 1, 1)).to.equal(-6);
                expect(CEILING_MATH(-5.4, 1, 0)).to.equal(-5);
                expect(CEILING_MATH(-5.4, 1)).to.equal(-5);

                expect(CEILING_MATH(5.4, 1, -1)).to.equal(6);
                expect(CEILING_MATH(5.4, 1, 1)).to.equal(6);
                expect(CEILING_MATH(5.4, 1, 0)).to.equal(6);
                expect(CEILING_MATH(5.4, 1)).to.equal(6);

                expect(CEILING_MATH(-5.4, 4, -1)).to.equal(-8);
                expect(CEILING_MATH(-5.4, 4, 1)).to.equal(-8);
                expect(CEILING_MATH(-5.4, 4, 0)).to.equal(-4);
                expect(CEILING_MATH(-5.4, 4)).to.equal(-4);
            });
        });

        moduleTester.testFunction('FLOOR.MATH', function (FLOOR_MATH) {
            it('should return the floor of its arguments', function () {
                expect(FLOOR_MATH(6.7)).to.equal(6);
                expect(FLOOR_MATH(24.3, 5)).to.equal(20);
                expect(FLOOR_MATH(22, 5)).to.equal(20);
                expect(FLOOR_MATH(-8.1, 2)).to.equal(-10);
            });
            it('should return the floor of its arguments with assigned mode', function () {
                expect(FLOOR_MATH(-5.6, 1, -1)).to.equal(-5);
                expect(FLOOR_MATH(-5.6, 1, 1)).to.equal(-5);
                expect(FLOOR_MATH(-5.6, 1, 0)).to.equal(-6);
                expect(FLOOR_MATH(-5.6, 1)).to.equal(-6);

                expect(FLOOR_MATH(5.6, 1, -1)).to.equal(5);
                expect(FLOOR_MATH(5.6, 1, 1)).to.equal(5);
                expect(FLOOR_MATH(5.6, 1, 0)).to.equal(5);
                expect(FLOOR_MATH(5.6, 1)).to.equal(5);

                expect(FLOOR_MATH(-5.6, 4, -1)).to.equal(-4);
                expect(FLOOR_MATH(-5.6, 4, 1)).to.equal(-4);
                expect(FLOOR_MATH(-5.6, 4, 0)).to.equal(-8);
                expect(FLOOR_MATH(-5.6, 4)).to.equal(-8);
            });
        });

        moduleTester.testFunction('SERIESSUM', function (SERIESSUM) {
            it('should return the sum of a power series', function () {
                expect(SERIESSUM(2, 2, 1, 1)).to.equal(4);
                expect(SERIESSUM(2, 2, 1, '1')).to.equal(4);
                expect(SERIESSUM(2, 2, 1, mat([[1]]))).to.equal(4);
                expect(SERIESSUM(2, 2, 1, mat([[1, 1]]))).to.equal(12);
                expect(SERIESSUM(2, 2, 1, mat([[1, 1, 1]]))).to.equal(28);
                expect(SERIESSUM(2, 2, 1, mat([[1, 1], [1, 1]]))).to.equal(60);
                expect(SERIESSUM(2, 2, 1, mat([[1, 1], [2, 1]]))).to.equal(76);
                expect(SERIESSUM(2, 2, 2, mat([[1, 1], [1, 1]]))).to.equal(340);
                expect(SERIESSUM(2, 2, 2, mat([[1, 2], [3, 4]]))).to.equal(1252);
            });
            it('should throw #VALUE! error code for invalid data', function () {
                expect(SERIESSUM(2, 2, 1, 'abc')).to.equal(ErrorCode.VALUE);
                expect(SERIESSUM(2, 2, 1, mat([[1, 'abc']]))).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('SUBTOTAL', function (SUBTOTAL) {
            var mat1 = mat([[3, 2, ErrorCode.NUM, 4, 5, 6]]);
            var mat2 = mat([[3, 2, 4, 5, 6]]);
            it('should return the subtotal of its arguments', function () {
                // AVERAGE
                expect(SUBTOTAL(1, mat1)).to.equal(ErrorCode.NUM);
                expect(SUBTOTAL(55, mat1)).to.equal(ErrorCode.VALUE);
                expect(SUBTOTAL(1, mat2)).to.equal(4);
                // COUNT
                expect(SUBTOTAL(2, mat2)).to.equal(5);
                expect(SUBTOTAL(102, mat2)).to.equal(5);
                // COUNTA
                expect(SUBTOTAL(3, mat2)).to.equal(5);
                expect(SUBTOTAL(103, mat2)).to.equal(5);
                // MAX
                expect(SUBTOTAL(4, mat2)).to.equal(6);
                expect(SUBTOTAL(104, mat2)).to.equal(6);
                // MIN
                expect(SUBTOTAL(5, mat2)).to.equal(2);
                expect(SUBTOTAL(105, mat2)).to.equal(2);
                // PRODUCT
                expect(SUBTOTAL(6, mat2)).to.equal(720);
                expect(SUBTOTAL(106, mat2)).to.equal(720);
                // STDEV.S
                expect(SUBTOTAL(7, mat2)).to.be.closeTo(1.58113883, 1e-6);
                expect(SUBTOTAL(107, mat2)).to.be.closeTo(1.58113883, 1e-6);
                // STDEV.P
                expect(SUBTOTAL(8, mat2)).to.be.closeTo(1.414213562, 1e-6);
                expect(SUBTOTAL(108, mat2)).to.be.closeTo(1.414213562, 1e-6);
                // SUM
                expect(SUBTOTAL(9, mat2)).to.equal(20);
                expect(SUBTOTAL(109, mat2)).to.equal(20);
                // VAR.S
                expect(SUBTOTAL(10, mat2)).to.equal(2.5);
                expect(SUBTOTAL(110, mat2)).to.equal(2.5);
                // VAR.P
                expect(SUBTOTAL(11, mat2)).to.equal(2);
                expect(SUBTOTAL(111, mat2)).to.equal(2);
            });
            it('should skip all hidden rows in a sheet with active auto-filter', function () {
                expect(SUBTOTAL(9, r3d('1:1!A2:A4'))).to.equal(5);
                expect(SUBTOTAL(109, r3d('1:1!A2:A4'))).to.equal(5);
                expect(SUBTOTAL(9, r3d('1:1!A7:A9'))).to.equal(5);
                expect(SUBTOTAL(109, r3d('1:1!A7:A9'))).to.equal(5);
            });
            it('should include hidden rows in a sheet without active auto-filter', function () {
                expect(SUBTOTAL(9, r3d('2:2!A2:A4'))).to.equal(7);
                expect(SUBTOTAL(109, r3d('2:2!A2:A4'))).to.equal(7);
                expect(SUBTOTAL(9, r3d('2:2!A7:A9'))).to.equal(7);
                expect(SUBTOTAL(109, r3d('2:2!A7:A9'))).to.equal(5);
            });
        });

        moduleTester.testFunction('SUM', function (SUM) {
            it('should return the sum of its arguments', function () {
                expect(SUM(1, 2, 3)).to.equal(6);
                expect(SUM(1, true, 3)).to.equal(5);
                expect(SUM(1, '2', 3)).to.equal(6);
                expect(SUM(1, null, 3)).to.equal(4);
            });
            it('should return the sum of a matrix', function () {
                expect(SUM(mat([[1, 2], [3, 4]]))).to.equal(10);
                expect(SUM(mat([[1, true], [3, 4]]))).to.equal(8);
                expect(SUM(mat([[1, '2'], [3, 4]]))).to.equal(8);
                expect(SUM(mat([[1, 'abc'], [3, 4]]))).to.equal(8);
                expect(SUM(mat([[1, ''], [3, 4]]))).to.equal(8);
            });
            it('should throw #VALUE! error for invalid arguments', function () {
                expect(SUM(1, 'abc', 3)).to.equal(ErrorCode.VALUE);
                expect(SUM(1, '', 3)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('SUMIF', function (SUMIF) {
            it('should return the sum of the matching cells', function () {
                expect(SUMIF(r3d('0:0!A2:J2'), '>2')).to.equal(8);
            });
            it('should return the sum of another range', function () {
                expect(SUMIF(r3d('0:0!A2:K2'), '>2', r3d('0:0!A3:K3'))).to.equal(516);
            });
            it('should ignore the size of the data range', function () {
                expect(SUMIF(r3d('0:0!A2:K2'), '>2', r3d('0:0!A3:A4'))).to.equal(516);
            });
        });

        moduleTester.testFunction('SUMIFS', function (SUMIFS) {
            it('should return the sum of the matching cells', function () {
                expect(SUMIFS(r3d('0:0!A3:J3'), r3d('0:0!A2:J2'), '>2')).to.equal(516);
                expect(SUMIFS(r3d('0:0!A3:J3'), r3d('0:0!A2:J2'), '>2', r3d('0:0!A3:J3'), '<32')).to.equal(4);
            });
        });
    });

    // ========================================================================
});
