/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/editframework/model/attributepool'
], function (ValueMap, AttributePool) {

    'use strict';

    // class AttributePool ====================================================

    describe('EditFramework class AttributePool', function () {

        it('should exist', function () {
            expect(AttributePool).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var docModel = { getWindowId: _.constant('window-0') };
        var attrPool = new AttributePool(docModel);

        var EXP_DEFAULTS_1 = { name1: 42, name2: 'value2', name4: 0 };
        var EXP_DEFAULTS_2 = { name1: true, name3: [42] };

        var EXP_NULL_1 = { name1: null, name4: null };
        var EXP_NULL_2 = { name3: null };

        // public methods -----------------------------------------------------

        it('should support events', function () {
            expect(attrPool).to.respondTo('on');
            expect(attrPool).to.respondTo('one');
            expect(attrPool).to.respondTo('off');
        });

        describe('method "isSupportedAttribute"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('isSupportedAttribute');
            });
            it('should return false for the empty pool', function () {
                expect(attrPool.isSupportedAttribute('family1', 'name1')).to.equal(false);
            });
        });

        describe('method "registerAttributes"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('registerAttributes');
            });
            it('should register new attributes', function () {
                expect(attrPool.registerAttributes('family1', {
                    name1: { def: 42 },
                    name2: { def: 'value1', special: true },
                    name4: { def: 0 }
                })).to.equal(attrPool);
                expect(attrPool.registerAttributes('family2', {
                    name1: { def: true },
                    name3: { def: [42], special: true }
                })).to.equal(attrPool);
                expect(attrPool.isSupportedAttribute('family1', 'name1')).to.equal(true);
                expect(attrPool.isSupportedAttribute('family1', 'name1', { special: true })).to.equal(true);
                expect(attrPool.isSupportedAttribute('family1', 'name2')).to.equal(false);
                expect(attrPool.isSupportedAttribute('family1', 'name2', { special: true })).to.equal(true);
                expect(attrPool.isSupportedAttribute('family1', 'name3')).to.equal(false);
                expect(attrPool.isSupportedAttribute('family1', 'name3', { special: true })).to.equal(false);
                expect(attrPool.isSupportedAttribute('family1', 'name4')).to.equal(true);
                expect(attrPool.isSupportedAttribute('family1', 'name4', { special: true })).to.equal(true);
                expect(attrPool.isSupportedAttribute('family2', 'name1')).to.equal(true);
                expect(attrPool.isSupportedAttribute('family2', 'name2')).to.equal(false);
                expect(attrPool.isSupportedAttribute('family2', 'name3')).to.equal(false);
                expect(attrPool.isSupportedAttribute('family2', 'name3', { special: true })).to.equal(true);
                expect(attrPool.isSupportedAttribute('family2', 'name4')).to.equal(false);
                expect(attrPool.isSupportedAttribute('family3', 'name1')).to.equal(false);
            });
            it('should update existing atttribute definitions', function () {
                attrPool.registerAttributes('family1', { name2: { def: 'value2' } });
                attrPool.registerAttributes('family2', { name1: { special: true }, name3: { special: false } });
                expect(attrPool.isSupportedAttribute('family1', 'name2')).to.equal(false);
                expect(attrPool.isSupportedAttribute('family1', 'name2', { special: true })).to.equal(true);
                expect(attrPool.isSupportedAttribute('family2', 'name1')).to.equal(false);
                expect(attrPool.isSupportedAttribute('family2', 'name1', { special: true })).to.equal(true);
                expect(attrPool.isSupportedAttribute('family2', 'name3')).to.equal(true);
            });
        });

        describe('method "getRegisteredAttributes"', function () {
            var defMap1, defMap2, defMap3;
            it('should exist', function () {
                expect(attrPool).to.respondTo('getRegisteredAttributes');
            });
            it('should return the definition map', function () {
                defMap1 = attrPool.getRegisteredAttributes('family1');
                expect(defMap1).to.be.an.instanceof(ValueMap);
                expect(defMap1.has('name1')).to.equal(true);
                expect(defMap1.has('name2')).to.equal(true);
                expect(defMap1.has('name3')).to.equal(false);
                defMap2 = attrPool.getRegisteredAttributes('family2');
                expect(defMap2).to.be.an.instanceof(ValueMap);
                expect(defMap2.has('name1')).to.equal(true);
                expect(defMap2.has('name2')).to.equal(false);
                expect(defMap2.has('name3')).to.equal(true);
                defMap3 = attrPool.getRegisteredAttributes('family3');
                expect(defMap3).to.equal(null);
            });
            it('should return a map with the method "isSupported"', function () {
                expect(defMap1).to.respondTo('isSupported');
                expect(defMap1.isSupported('name1')).to.equal(true);
                expect(defMap1.isSupported('name1', { special: true })).to.equal(true);
                expect(defMap1.isSupported('name2')).to.equal(false);
                expect(defMap1.isSupported('name2', { special: true })).to.equal(true);
                expect(defMap1.isSupported('name3')).to.equal(false);
                expect(defMap1.isSupported('name3', { special: true })).to.equal(false);
            });
            it('should return a map with the method "forEachSupported"', function () {
                expect(defMap1).to.respondTo('forEachSupported');
                var spy1 = sinon.spy();
                expect(defMap1.forEachSupported({ name1: 1, name2: 2, name3: 3 }, spy1)).to.equal(defMap1);
                expect(spy1.callCount).to.equal(1);
                expect(spy1.alwaysCalledOn(defMap1)).to.equal(true);
                expect(spy1.calledWithExactly(1, 'name1', defMap1.get('name1'))).to.equal(true);
                var spy2 = sinon.spy();
                expect(defMap1.forEachSupported({ name1: 1, name2: 2, name3: 3 }, spy2, { special: true })).to.equal(defMap1);
                expect(spy2.callCount).to.equal(2);
                expect(spy2.alwaysCalledOn(defMap1)).to.equal(true);
                expect(spy2.calledWithExactly(1, 'name1', defMap1.get('name1'))).to.equal(true);
                expect(spy2.calledWithExactly(2, 'name2', defMap1.get('name2'))).to.equal(true);
                var spy3 = sinon.spy();
                expect(defMap1.forEachSupported(null, spy3)).to.equal(defMap1);
                expect(spy3.callCount).to.equal(0);
            });
        });

        describe('method "getDefaultValue"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('getDefaultValue');
            });
            it('should return the default value', function () {
                expect(attrPool.getDefaultValue('family1', 'name1')).to.equal(42);
                expect(attrPool.getDefaultValue('family1', 'name2')).to.equal('value2');
                expect(attrPool.getDefaultValue('family1', 'name3')).to.equal(null);
                expect(attrPool.getDefaultValue('family2', 'name1')).to.equal(true);
                expect(attrPool.getDefaultValue('family2', 'name2')).to.equal(null);
                expect(attrPool.getDefaultValue('family2', 'name3')).to.deep.equal([42]);
                expect(attrPool.getDefaultValue('family3', 'name1')).to.equal(null);
            });
        });

        describe('method "getDefaultValues"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('getDefaultValues');
            });
            it('should return the default values', function () {
                expect(attrPool.getDefaultValues('family1')).to.deep.equal(EXP_DEFAULTS_1);
                expect(attrPool.getDefaultValues('family2')).to.deep.equal(EXP_DEFAULTS_2);
                expect(attrPool.getDefaultValues('family3')).to.equal(null);
            });
        });

        describe('method "getDefaultValueSet"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('getDefaultValueSet');
            });
            it('should return the default values for a single family', function () {
                expect(attrPool.getDefaultValueSet('family1')).to.deep.equal({ family1: EXP_DEFAULTS_1 });
                expect(attrPool.getDefaultValueSet('family2')).to.deep.equal({ family2: EXP_DEFAULTS_2 });
                expect(attrPool.getDefaultValueSet('family3')).to.deep.equal({});
            });
            it('should return the default values for an array of families', function () {
                expect(attrPool.getDefaultValueSet(['family1', 'family3'])).to.deep.equal({ family1: EXP_DEFAULTS_1 });
                expect(attrPool.getDefaultValueSet(['family1', 'family2'])).to.deep.equal({ family1: EXP_DEFAULTS_1, family2: EXP_DEFAULTS_2 });
                expect(attrPool.getDefaultValueSet(['family3', 'family4'])).to.deep.equal({});
            });
            it('should return the default values for a set of families', function () {
                expect(attrPool.getDefaultValueSet({ family1: 1, family3: 1 })).to.deep.equal({ family1: EXP_DEFAULTS_1 });
                expect(attrPool.getDefaultValueSet({ family1: 0, family2: 0 })).to.deep.equal({ family1: EXP_DEFAULTS_1, family2: EXP_DEFAULTS_2 });
                expect(attrPool.getDefaultValueSet({ family3: 1, family4: 1 })).to.deep.equal({});
            });
        });

        describe('method "changeDefaultValues"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('changeDefaultValues');
            });
            it('should change the specified default values', function () {
                var spy = sinon.spy();
                attrPool.on('change:defaults', spy);
                expect(attrPool.changeDefaultValues({ family1: { name3: 1, name4: 1 }, family2: { name3: [42] }, family3: { name1: 1 } })).to.equal(attrPool);
                attrPool.off('change:defaults', spy);
                expect(attrPool.getDefaultValue('family1', 'name1')).to.equal(42);
                expect(attrPool.getDefaultValue('family1', 'name3')).to.equal(null);
                expect(attrPool.getDefaultValue('family1', 'name4')).to.equal(1);
                expect(attrPool.getDefaultValue('family3', 'name1')).to.equal(null);
                expect(spy.callCount).to.equal(1);
                expect(spy.getCall(0).args[3]).to.deep.equal({ family1: { name4: 1 } });
            });
        });

        describe('method "buildNullValues"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('buildNullValues');
            });
            it('should return the null values', function () {
                expect(attrPool.buildNullValues('family1')).to.deep.equal(EXP_NULL_1);
                expect(attrPool.buildNullValues('family2')).to.deep.equal(EXP_NULL_2);
                expect(attrPool.buildNullValues('family3')).to.equal(null);
            });
        });

        describe('method "buildNullValueSet"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('buildNullValueSet');
            });
            it('should return the null values for a single family', function () {
                expect(attrPool.buildNullValueSet('family1')).to.deep.equal({ family1: EXP_NULL_1 });
                expect(attrPool.buildNullValueSet('family2')).to.deep.equal({ family2: EXP_NULL_2 });
                expect(attrPool.buildNullValueSet('family3')).to.deep.equal({});
            });
            it('should return the null values for an array of families', function () {
                expect(attrPool.buildNullValueSet(['family1', 'family3'])).to.deep.equal({ family1: EXP_NULL_1 });
                expect(attrPool.buildNullValueSet(['family1', 'family2'])).to.deep.equal({ family1: EXP_NULL_1, family2: EXP_NULL_2 });
                expect(attrPool.buildNullValueSet(['family3', 'family4'])).to.deep.equal({});
            });
            it('should return the null values for a set of families', function () {
                expect(attrPool.buildNullValueSet({ family1: 1, family3: 1 })).to.deep.equal({ family1: EXP_NULL_1 });
                expect(attrPool.buildNullValueSet({ family1: 0, family2: 0 })).to.deep.equal({ family1: EXP_NULL_1, family2: EXP_NULL_2 });
                expect(attrPool.buildNullValueSet({ family3: 1, family4: 1 })).to.deep.equal({});
            });
        });

        describe('method "extendAttributes"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('extendAttributes');
            });
        });

        describe('method "extendAttributeSet"', function () {
            it('should exist', function () {
                expect(attrPool).to.respondTo('extendAttributeSet');
            });
        });
    });

    // ========================================================================
});
