/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('cellmatrix/celltable', [
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'perf!cellmatrix/newmatrix32'
], function (ValueMap, BaseObject, SheetUtils, CellMatrix) {

    'use strict';

    // constants ==============================================================

    // convenience shortcuts
    var AddressSet = SheetUtils.AddressSet;
    var RangeSet = SheetUtils.RangeSet;

    // class SharedFormulaModel ===============================================

    /**
     * A structure that is used to collect data about a shared formula in the
     * cell collection.
     *
     * @constructor
     *
     * @property {SheetModel} sheetModel
     *  The model of the sheet containing this shared formula.
     *
     * @property {number} index
     *  The index of the shared formula, as used for the 'si' property of cell
     *  models.
     *
     * @property {Address|null} anchorAddress
     *  The address of the anchor cell; or null, if the anchor cell has been
     *  detached from the shared formula (used e.g. as intermediate state while
     *  changing the cells in the cell collection, and updating the shared
     *  formula afterwards).
     *
     * @property {AddressSet} addressSet
     *  The addresses of all cells that are part of the shared formula.
     *
     * @property {Address|null} refAddress
     *  The reference address used to parse the formula expression of the
     *  shared formula. This address remains intact when deleting the anchor
     *  address, in order to be able to generate a relocated formula expression
     *  for a new anchor address afterwards. Will be null for uninitialized
     *  model instances (no anchor cell with formula available).
     *
     * @property {TokenArray|null} tokenArray
     *  The parsed formula expression of the shared formula. Will be null for
     *  uninitialized model instances (no anchor cell with formula available).
     */
    function SharedFormulaModel(sheetModel, index) {

        // the sheet model owning the shared formula
        this.sheetModel = sheetModel;
        // the shared index (will never change)
        this.index = index;

        // the address of the current anchor cell in the cell collection
        this.anchorAddress = null;
        // the addresses of all cells that are currently part of the shared formula
        this.addressSet = new AddressSet();
        // the reference address for the formula definition (may differ from the anchor address)
        this.refAddress = null;
        // the parsed formula expression for the reference address
        this.tokenArray = null;
    }

    // public methods ---------------------------------------------------------

    /**
     * Inserts a new cell into this shared formula.
     *
     * @param {CellModel} cellModel
     *  The cell model to be inserted.
     */
    SharedFormulaModel.prototype.insertAddress = function (cellModel) {
        if (cellModel.isSharedAnchor()) {
            this.anchorAddress = this.refAddress = cellModel.a.clone();
            this.tokenArray = this.sheetModel.createCellTokenArray();
            this.tokenArray.parseFormula('op', cellModel.f || '', { refAddress: this.refAddress });
        }
        this.addressSet.insert(cellModel.a.clone());
    };

    /**
     * Removes a cell from this shared formula.
     *
     * @param {CellModel} cellModel
     *  The cell model to be removed.
     */
    SharedFormulaModel.prototype.removeAddress = function (cellModel) {
        if (this.anchorAddress && this.anchorAddress.equals(cellModel.a)) {
            this.anchorAddress = null;
        }
        this.addressSet.remove(cellModel.a);
    };

    /**
     * Returns the formula expression of this shared formula, relocated to the
     * specified target address.
     *
     * @param {string} grammarId
     *  The identifier of the formula grammar for the formula expression. See
     *  SpreadsheetDocument.getFormulaGrammar() for more details.
     *
     * @param {Address} targetAddress
     *  The target address to generate the formula expression for.
     *
     * @returns {string|null}
     *  The formula expression for the specified target address; or null, is
     *  this instance is invalid (no reference address, no formula available).
     */
    SharedFormulaModel.prototype.getFormula = function (grammarId, targetAddress) {
        return this.refAddress ? this.tokenArray.getFormula(grammarId, { refAddress: this.refAddress, targetAddress: targetAddress }) : null;
    };

    // class CellTable ========================================================

    var CellTable = BaseObject.extend({ constructor: function (sheetModel) {

        // base constructor
        BaseObject.call(this, sheetModel);

        // the sheet model and its containers
        this._sheetModel = sheetModel;
        this._colCollection = sheetModel.getColCollection();
        this._rowCollection = sheetModel.getRowCollection();

        // all cell models mapped by address key
        this._modelMap = new ValueMap();

        // column matrix for all cell models, used for cell auto-styles
        this._allColMatrix = new CellMatrix(true);
        // row matrix for all cell models, used for cell auto-styles
        this._allRowMatrix = new CellMatrix(false);

        // column matrix for non-blank cell models, used for value iterators (formulas)
        this._valColMatrix = new CellMatrix(true);
        // row matrix for non-blank cell models, used for value iterators (formulas)
        this._valRowMatrix = new CellMatrix(false);

        // the addresses of shared formulas, mapped by shared index
        this._sharedModelMap = new ValueMap();
        // the range addresses of all matrix formulas
        this._matrixRangeSet = new RangeSet();

        // update cached visibility flags of column vectors if columns will be hidden or shown
        this.listenTo(this._colCollection, 'change:entries', function (_event, interval, changeInfo) {
            if (changeInfo.visibilityChanged) {
                var visible = this._colCollection.isEntryVisible(interval.first);
                this._allColMatrix.forEachVector(interval, function (vector) { vector.visible = visible; });
            }
        }.bind(this));

        // update cached visibility flags of row vectors if rows will be hidden or shown
        this.listenTo(this._rowCollection, 'change:entries', function (_event, interval, changeInfo) {
            if (changeInfo.visibilityChanged) {
                var visible = this._rowCollection.isEntryVisible(interval.first);
                this._allRowMatrix.forEachVector(interval, function (vector) { vector.visible = visible; });
            }
        }.bind(this));
    } });

    // public methods ---------------------------------------------------------

    /**
     * Returns the cell model with the specified cell address from this table.
     *
     * @attention
     *  The method is provided for optimized access to internal cell data. The
     *  cell model MUST NOT BE CHANGED in order to retain the internal
     *  consistency of the cell table!
     *
     * @param {Address} address
     *  The address of the cell.
     *
     * @returns {CellModel|null}
     *  The cell model; or null, if the cell does not exist in this cell table.
     *  The contents of the cell model MUST NOT be changed!
     */
    CellTable.prototype.getCellModel = function (address) {
        return this._modelMap.get(address.key(), null);
    };

    // private methods --------------------------------------------------------

    /**
     * Inserts the address of the passed cell model into the model of a shared
     * formula, if the cell refers to a shared formula; or registers the
     * bounding range of a matrix formula.
     *
     * @param {CellModel} cellModel
     *  The cell model to be inserted into this cell table.
     */
    CellTable.prototype._registerFormulaCell = function (cellModel) {

        // shared formulas: "si" property contains shared index, "sr" property denotes anchor cell
        if (cellModel.isSharedFormula()) {
            var sharedModel = this._sharedModelMap.getOrConstruct(cellModel.si, SharedFormulaModel, this._sheetModel, cellModel.si);
            sharedModel.insertAddress(cellModel);
        }

        // matrix formulas: "mr" property denotes top-left cell
        if (cellModel.isMatrixAnchor()) {
            this._matrixRangeSet.insert(cellModel.mr.clone());
        }
    };

    /**
     * Removes the address of the passed cell model from the model of a shared
     * formula, if the cell refers to a shared formula; or removes the bounding
     * range of a matrix formula.
     *
     * @param {CellModel} cellModel
     *  The cell model to be removed from this cell table.
     */
    CellTable.prototype._unregisterFormulaCell = function (cellModel) {

        // shared formulas: "si" property contains shared index, "sr" property denotes anchor cell
        if (cellModel.isSharedFormula()) {
            this._sharedModelMap.with(cellModel.si, function (sharedModel) {
                sharedModel.removeAddress(cellModel);
            });
        }

        // matrix formulas: "mr" property denotes top-left cell
        if (cellModel.isMatrixAnchor()) {
            this._matrixRangeSet.remove(cellModel.mr);
        }
    };

    /**
     * Inserts the passed cell model into the internal containers.
     *
     * @param {CellModel} cellModel
     *  The cell model to be inserted into this cell table. This table DOES NOT
     *  take ownership of the cell models; and it DOES NOT check whether it
     *  already contains a cell model with the same address as the passed cell
     *  model (it is up to the caller to ensure uniqueness of the cell models).
     */
    CellTable.prototype._insertModel = function (cellModel) {

        // insert the cell model into the map
        this._modelMap.insert(cellModel.key(), cellModel);

        // insert the cell model into the column matrix for all cells
        cellModel.cv = this._allColMatrix.insertModel(cellModel);
        if (cellModel.cv.length === 1) {
            cellModel.cv.visible = this._colCollection.isEntryVisible(cellModel.cv.index);
        }

        // insert the cell model into the row matrix for all cells
        cellModel.rv = this._allRowMatrix.insertModel(cellModel);
        if (cellModel.rv.length === 1) {
            cellModel.rv.visible = this._rowCollection.isEntryVisible(cellModel.rv.index);
        }

        // insert a non-blank cell model into the matrixes for value cells
        if (!cellModel.isBlank()) {
            this._valColMatrix.insertModel(cellModel);
            this._valRowMatrix.insertModel(cellModel);
        }

        // register shared formulas and matrix formulas in the respective containers
        this._registerFormulaCell(cellModel);
    };

    /**
     * Removes the passed cell model from the internal containers.
     *
     * @param {CellModel} cellModel
     *  The cell model to be removed from this cell table.
     */
    CellTable.prototype._removeModel = function (cellModel) {

        // unregister shared formulas and matrix formulas in the respective containers
        this._unregisterFormulaCell(cellModel);

        // remove a non-blank cell model from the matrixes for value cells
        if (!cellModel.isBlank()) {
            this._valColMatrix.removeModel(cellModel);
            this._valRowMatrix.removeModel(cellModel);
        }

        // remove the cell model from the matrixes for all cells
        this._allColMatrix.removeModel(cellModel);
        this._allRowMatrix.removeModel(cellModel);
        cellModel.ci = cellModel.ri = null;

        // remove the cell model from the map
        this._modelMap.remove(cellModel.key());
    };

    // exports ================================================================

    return CellMatrix;

});
