/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('cellmatrix/cellmatrix', [
    'io.ox/office/tk/utils/iterator',
    'perf!cellmatrix/indexarray',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/cellmodel'
], function (Iterator, IndexArray, SheetUtils, CellModel) {

    'use strict';

    // constants ==============================================================

    // convenience shortcuts
    var FilterIterator = Iterator.FilterIterator;
    var NestedIterator = Iterator.NestedIterator;
    var Interval = SheetUtils.Interval;

    // private functions ======================================================

    function VERT_VIS_VECTOR_FILTER(cellVector) { return cellVector.first().cv.visible; }
    function VERT_HID_VECTOR_FILTER(cellVector) { return !cellVector.first().cv.visible; }
    function HORI_VIS_VECTOR_FILTER(cellVector) { return cellVector.first().rv.visible; }
    function HORI_HID_VECTOR_FILTER(cellVector) { return !cellVector.first().rv.visible; }

    function VERT_VIS_MODEL_FILTER(cellModel) { return cellModel.cv.visible; }
    function VERT_HID_MODEL_FILTER(cellModel) { return !cellModel.cv.visible; }
    function HORI_VIS_MODEL_FILTER(cellModel) { return cellModel.rv.visible; }
    function HORI_HID_MODEL_FILTER(cellModel) { return !cellModel.rv.visible; }

    /**
     * Returns an iterator that combines the passed iterator with an optional
     * filter predicate callback function.
     *
     * @param {Iterator<T>} iterator
     *  The iterator to be filtered.
     *
     * @param {(T => boolean)|null} filter
     *  The filter predicate callback to be used to filter the iterator. If set
     *  to null, the original iterator will be returned.
     *
     * @returns {Iterator<T>}
     *  An instance of FilterIterator<T>, if a filter predicate has been
     *  passed, otherwise the original iterator passed to this function.
     */
    function createFilterIterator/*<T>*/(iterator, filter) {
        return filter ? new FilterIterator(iterator, filter) : iterator;
    }

    /**
     * Returns a callback function that invokes the passed callback function,
     * if the filter predicate matches a sequence value.
     *
     * @param {T => any} callback
     *  The callback function to be wrapped with the filter predicate.
     *
     * @param {(T => any)|null} filter
     *  The filter predicate callback to be used to filter the invocations of
     *  the passed callback function. If set to null, the original callback
     *  function will be returned (no filtering).
     *
     * @param {any} [defValue]
     *  The default value to return for elements that do not match the filter
     *  predicate.
     *
     * @returns {T => any}
     *  A new callback function that invokes the passed callback function only
     *  if the passed-in value matches the filter predicate. Returns either the
     *  result of the callback function (if the filter matches), or the passed
     *  default value.
     */
    function createFilterCallback/*<T>*/(callback, filter, defValue) {
        return filter ? function (element) { return filter(element) ? callback(element) : defValue; } : callback;
    }

    // class CellVector =======================================================

    /**
     * A sorted array of cell models from a single column/row in the cell
     * table. Instances of this class are contained in the internal array of
     * CellMatrix instances.
     *
     * @param {number} index
     *  The index of the column/row represented by this vector.
     *
     * @param {boolean} vertical
     *  Whether to create a column vector (true), or a row vector (false).
     */
    var CellVector = IndexArray/*<CellModel>*/.extend(function (index, vertical) {

        // base constructor
        IndexArray.call(this, vertical ? CellModel.row : CellModel.col);

        // the column/row index of all cells in this vector
        this.index = index;
    });

    // class CellMatrix =======================================================

    /**
     * An instance of this class represents a sparse two-dimensional matrix of
     * cell models. The double-sorted data structure allows fast binary lookup,
     * and fast iterator implementations for existing cell models in large cell
     * ranges.
     *
     * Example: An instance of this class is used as row matrix. In this case,
     * it will contain several row vectors that contain the cell models in the
     * respective rows sorted by column index.
     *
     * @constructor
     *
     * @param {boolean} vertical
     *  Whether this matrix contains column vectors (true), or row vectors
     *  (false).
     */
    function CellMatrix(vertical) {

        // the sorted array of cell vectors
        this._array = new IndexArray/*<CellVector>*/(function (entry) { return entry.index; });
        // resolve column/row index for cell vectors from cell model
        this._resolveIndex = vertical ? CellModel.col : CellModel.row;
        // the orientation of the cell vectors
        this._vertical = vertical;
    }

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this matrix is empty.
     *
     * @returns {boolean}
     *  Whether this matrix is empty.
     */
    CellMatrix.prototype.empty = function () {
        return this._array.empty();
    };

    /**
     * Returns the smallest index interval containing all cell vectors in this
     * matrix.
     *
     * @returns {Interval|null}
     *  The smallest index interval containing all cell vectors in this matrix;
     *  or null, if this matrix is empty.
     */
    CellMatrix.prototype.getUsedInterval = function () {
        return this.empty() ? null : new Interval(this._array.first().index, this._array.last().index);
    };

    /**
     * Creates an iterator that visits all cell vectors contained in this
     * matrix that are covered by the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the passed index intervals will be visited in
     *      reversed order, as well as the cell vectors covered by each of the
     *      intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cell vectors (associated to visible
     *      columns/rows) will be visited. If set to false, only hidden cell
     *      vectors (associated to hidden columns/rows) will be visited. By
     *      default, all cell vectors will be visited.
     *
     * @returns {Iterator<CellVector>}
     *  The new iterator.
     */
    CellMatrix.prototype.createVectorIterator = function (intervals, options) {

        // create an iterator that visits all cell vectors in the index intervals
        var iterator = this._array.iteratorIn(intervals, options && options.reverse);

        // filter for visible or hidden cell vectors
        var filter = this._createVectorFilter(options);
        return createFilterIterator(iterator, filter);
    };

    /**
     * Invokes the specified callback function for all cell vectors contained
     * in this matrix that are covered by the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {CellVector => void} callback
     *  The callback function to be invoked for each matching cell vector.
     *  Receives the cell vector as first parameter. The return value will be
     *  ignored.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the passed index intervals will be visited in
     *      reversed order, as well as the cell vectors covered by each of the
     *      intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cell vectors (associated to visible
     *      columns/rows) will be visited. If set to false, only hidden cell
     *      vectors (associated to hidden columns/rows) will be visited. By
     *      default, all cell vectors will be visited.
     */
    CellMatrix.prototype.forEachVector = function (intervals, callback, options) {

        // invokes the callback function according to visibility of the vector
        var filter = this._createVectorFilter(options);
        var invoke = createFilterCallback(callback, filter);

        // visit the passed intervals in the specified order
        this._array.forEachIn(intervals, invoke, options && options.reverse);
    };

    /**
     * Invokes the specified predicate callback function for all cell vectors
     * contained in this matrix that are covered by the passed index intervals,
     * until it returns a truthy value.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {CellVector => any} predicate
     *  The predicate callback function to be invoked for each matching cell
     *  vector. Receives the cell vector as first parameter. If this function
     *  returns a truthy value, the loop will stop immediately.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the passed index intervals will be visited in
     *      reversed order, as well as the cell vectors covered by each of the
     *      intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cell vectors (associated to visible
     *      columns/rows) will be visited. If set to false, only hidden cell
     *      vectors (associated to hidden columns/rows) will be visited. By
     *      default, all cell vectors will be visited.
     *
     * @returns {boolean}
     *  Whether the predicate callback function has returned a truthy value for
     *  a cell vector.
     */
    CellMatrix.prototype.someVector = function (intervals, predicate, options) {

        // invokes the callback function according to visibility of the vector
        var filter = this._createVectorFilter(options);
        var invoke = createFilterCallback(predicate, filter, false);

        // visit the passed intervals in the specified order
        return this._array.someIn(intervals, invoke, options && options.reverse);
    };

    /**
     * Invokes the specified predicate callback function for all cell vectors
     * contained in this matrix that are covered by the passed index intervals,
     * until it returns a falsy value.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {CellVector => any} predicate
     *  The predicate callback function to be invoked for each matching cell
     *  vector. Receives the cell vector as first parameter. If this function
     *  returns a falsy value, the loop will stop immediately.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the passed index intervals will be visited in
     *      reversed order, as well as the cell vectors covered by each of the
     *      intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cell vectors (associated to visible
     *      columns/rows) will be visited. If set to false, only hidden cell
     *      vectors (associated to hidden columns/rows) will be visited. By
     *      default, all cell vectors will be visited.
     *
     * @returns {boolean}
     *  Whether the predicate callback function has returned a truthy value for
     *  all cell vectors.
     */
    CellMatrix.prototype.everyVector = function (intervals, predicate, options) {

        // invokes the callback function according to visibility of the vector
        var filter = this._createVectorFilter(options);
        var invoke = createFilterCallback(predicate, filter, true);

        // visit the passed intervals in the specified order
        return this._array.everyIn(intervals, invoke, options && options.reverse);
    };

    /**
     * Creates an iterator that visits all existing cell models in this matrix
     * covered by the ranges formed by the passed index intervals. The cell
     * models will be visited vector-by-vector.
     *
     * @param {IntervalArray|Interval} outerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all cell vectors of this matrix to be visited.
     *
     * @param {IntervalArray|Interval} innerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all column/row indexes to be visited in each cell vector.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the cell models will be visited starting from the
     *      bottom-right edge of the last inner/outer intervals, and ending in
     *      the top-left edge of the first inner/outer intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cells (cells in visible columns AND
     *      visible rows) will be visited. If set to false, only hidden cells
     *      (cells in hidden columns OR hidden rows) will be visited. By
     *      default, all cells will be visited.
     *
     * @returns {Iterator<CellModel>}
     *  The new iterator.
     */
    CellMatrix.prototype.createModelIterator = function (outerIntervals, innerIntervals, options) {

        // the iteration order
        var reverse = options && options.reverse;
        // create an iterator that visits the cell vectors in the passed outer intervals
        var iterator = this.createVectorIterator(outerIntervals, options);

        // create an iterator that visits the inner intervals for every single vector, and returns cell models
        iterator = new NestedIterator(iterator, function (vector) {
            return vector.iteratorIn(innerIntervals, reverse);
        });

        // filter for visible or hidden cell models
        var filter = this._createModelFilter(options);
        return createFilterIterator(iterator, filter);
    };

    /**
     * Invokes the specified callback function for all existing cell models in
     * this matrix covered by the ranges formed by the passed index intervals.
     * The cell models will be visited vector-by-vector.
     *
     * @param {IntervalArray|Interval} outerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all cell vectors of this matrix to be visited.
     *
     * @param {IntervalArray|Interval} innerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all column/row indexes to be visited in each cell vector.
     *
     * @param {CellModel => void} callback
     *  The callback function to be invoked for each cell model. Receives the
     *  cell model as first parameter. The return value will be ignored.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the cell models will be visited starting from the
     *      bottom-right edge of the last inner/outer intervals, and ending in
     *      the top-left edge of the first inner/outer intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cells (cells in visible columns AND
     *      visible rows) will be visited. If set to false, only hidden cells
     *      (cells in hidden columns OR hidden rows) will be visited. By
     *      default, all cells will be visited.
     */
    CellMatrix.prototype.forEachModel = function (outerIntervals, innerIntervals, callback, options) {

        // the iteration order
        var reverse = options && options.reverse;
        // invoke the callback function according to visibility of the vector
        var filter = this._createModelFilter(options);
        var invoke = createFilterCallback(callback, filter);

        // visit the index vectors in the passed outer intervals
        this.forEachVector(outerIntervals, function (vector) {
            vector.forEachIn(innerIntervals, invoke, reverse);
        }, options);
    };

    /**
     * Invokes the specified predicate callback function for all existing cell
     * models in this matrix covered by the ranges formed by the passed index
     * intervals, until it returns a truthy value. The cell models will be
     * visited vector-by-vector.
     *
     * @param {IntervalArray|Interval} outerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all cell vectors of this matrix to be visited.
     *
     * @param {IntervalArray|Interval} innerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all column/row indexes to be visited in each cell vector.
     *
     * @param {CellModel => any} predicate
     *  The callback function to be invoked for each cell model. Receives the
     *  cell model as first parameter. If this function returns a truthy value,
     *  the loop will stop immediately.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the cell models will be visited starting from the
     *      bottom-right edge of the last inner/outer intervals, and ending in
     *      the top-left edge of the first inner/outer intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cells (cells in visible columns AND
     *      visible rows) will be visited. If set to false, only hidden cells
     *      (cells in hidden columns OR hidden rows) will be visited. By
     *      default, all cells will be visited.
     *
     * @returns {boolean}
     *  Whether the predicate callback function has returned a truthy value for
     *  a cell model.
     */
    CellMatrix.prototype.someModel = function (outerIntervals, innerIntervals, predicate, options) {

        // the iteration order
        var reverse = options && options.reverse;
        // invoke the callback function according to visibility of the vector
        var filter = this._createModelFilter(options);
        var invoke = createFilterCallback(predicate, filter, false);

        // visit the index vectors in the passed outer intervals
        return this.someVector(outerIntervals, function (vector) {
            return vector.someIn(innerIntervals, invoke, reverse);
        }, options);
    };

    /**
     * Invokes the specified predicate callback function for all existing cell
     * models in this matrix covered by the ranges formed by the passed index
     * intervals, until it returns a falsy value. The cell models will be
     * visited vector-by-vector.
     *
     * @param {IntervalArray|Interval} outerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all cell vectors of this matrix to be visited.
     *
     * @param {IntervalArray|Interval} innerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all column/row indexes to be visited in each cell vector.
     *
     * @param {CellModel => any} predicate
     *  The callback function to be invoked for each cell model. Receives the
     *  cell model as first parameter. If this function returns a falsy value,
     *  the loop will stop immediately.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the cell models will be visited starting from the
     *      bottom-right edge of the last inner/outer intervals, and ending in
     *      the top-left edge of the first inner/outer intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cells (cells in visible columns AND
     *      visible rows) will be visited. If set to false, only hidden cells
     *      (cells in hidden columns OR hidden rows) will be visited. By
     *      default, all cells will be visited.
     *
     * @returns {boolean}
     *  Whether the predicate callback function has returned a truthy value for
     *  all cell models.
     */
    CellMatrix.prototype.everyModel = function (outerIntervals, innerIntervals, predicate, options) {

        // the iteration order
        var reverse = options && options.reverse;
        // invoke the callback function according to visibility of the vector
        var filter = this._createModelFilter(options);
        var invoke = createFilterCallback(predicate, filter, true);

        // visit the index vectors in the passed outer intervals
        return this.everyVector(outerIntervals, function (vector) {
            return vector.everyIn(innerIntervals, invoke, reverse);
        }, options);
    };

    /**
     * Inserts the passed cell model into this matrix.
     *
     * @param {CellModel} cellModel
     *  The cell model to be inserted into this matrix. This matrix DOES NOT
     *  take ownership of the cell models; and it DOES NOT check whether it
     *  already contains a cell model with the same address as the passed cell
     *  model (it is up to the caller to ensure uniqueness of the cell models).
     *
     * @returns {CellVector}
     *  The cell vector containing the cell model, for further processing.
     */
    CellMatrix.prototype.insertModel = function (cellModel) {

        // get existing cell vector, or create a new cell vector
        var index = this._resolveIndex(cellModel);
        var vector = this._array.getOrCreate(index, function () {
            return new CellVector(index, this._vertical);
        }, this);

        // insert the respective address index into the cell vector
        vector.insert(cellModel);

        // return the cell vector for further processing
        return vector;
    };

    /**
     * Removes an existing cell model from this matrix.
     *
     * @param {CellModel} cellModel
     *  The cell model to be removed from this matrix.
     *
     * @returns {CellVector|null}
     *  The cell vector that has contained the cell model, for further
     *  processing; or null, if no such cell vector has been found, or if the
     *  existing cell vector became empty and has been deleted.
     */
    CellMatrix.prototype.removeModel = function (cellModel) {

        // get existing cell vector, or create a new cell vector
        var index = this._resolveIndex(cellModel);
        var vector = this._array.get(index);

        // nothing more to do, if the vector does not exist
        if (!vector) { return null; }

        // remove the respective cell model from the cell vector
        vector.remove(cellModel);

        // remove the entire vector, if it became empty
        if (vector.empty()) {
            this._array.remove(vector);
            return null;
        }

        // return the cell vector for further processing
        return vector;
    };

    // private methods --------------------------------------------------------

    /**
     * Returns a filter predicate callback function that returns whether a cell
     * vector matches the specified visibility of its column/row.
     *
     * @param {object} [options]
     *  Optional parameters passed to other methods of this class. Uses the
     *  boolean option "visible" to decide which filter predicate to return.
     *
     * @returns {(CellVector => boolean)|null}
     *  The filter predicate matching the passed visibility state; or null, if
     *  no visibility flag has been passed.
     */
    CellMatrix.prototype._createVectorFilter = function (options) {
        switch (options && options.visible) {
            case true:  return this._vertical ? VERT_VIS_VECTOR_FILTER : HORI_VIS_VECTOR_FILTER;
            case false: return this._vertical ? VERT_HID_VECTOR_FILTER : HORI_HID_VECTOR_FILTER;
            default:    return null;
        }
    };

    /**
     * Returns a filter predicate callback function that returns whether a cell
     * model matches the specified visibility of its column/row.
     *
     * @param {object} [options]
     *  Optional parameters passed to other methods of this class. Uses the
     *  boolean option "visible" to decide which filter predicate to return.
     *
     * @returns {(CellModel => boolean)|null}
     *  The filter predicate matching the passed visibility state; or null, if
     *  no visibility flag has been passed.
     */
    CellMatrix.prototype._createModelFilter = function (visible) {
        switch (visible) {
            case true:  return this._vertical ? HORI_VIS_MODEL_FILTER : VERT_VIS_MODEL_FILTER;
            case false: return this._vertical ? HORI_HID_MODEL_FILTER : VERT_HID_MODEL_FILTER;
            default:    return null;
        }
    };

    // exports ================================================================

    return CellMatrix;

});
