/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/iterator', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/class'
], function (Utils, Class) {

    'use strict';

    // private global functions ===============================================

    // generator function for finished result objects
    var DONE_GEN = _.constant({ done: true });

    /**
     * Returns the minimum of the passed offsets (either of the parameters can
     * be null).
     */
    function getMinOffset(offset1, offset2) {
        return (offset1 === null) ? offset2 : (offset2 === null) ? offset1 : Math.min(offset1, offset2);
    }

    // class Iterator =========================================================

    /**
     * Represents a generic iterator used to visit an arbitrary sequence of
     * values provided by a generator callback function. Iterators can be used
     * in synchronous code (see static helper methods of this class), as well
     * as in asychronous code (see the iterator methods of the class
     * BaseObject).
     *
     * Instances of this class and its sub classes implement the standard
     * EcmaScript iterator protocol, i.e. they provide the method next() that
     * returns a result object with the boolean property "done" and the
     * property "value".
     *
     * If the property "done" of the result object is set to true, the iterator
     * has visited all available values in the sequence. The result object does
     * not contain any other properties (no property "value").
     *
     * @constructor
     */
    var Iterator = Class.extendable(function () {}); // class Iterator

    // protected methods ------------------------------------------------------

    /**
     * Generates the result objects for the iterator. If the return value of
     * the callback function is an object, it will be set as result of the
     * iterator step. If the object contains a property "done" set to true, the
     * iterator switches to finished state. Otherwise, the property "done" set
     * to false will be added to the result object automatically. If the return
     * value is not an object, the iterator will skip the result, and will
     * continue to fetch results from the callback function until a result
     * object will be returned.
     *
     * @protected
     *  This method will be called from the implementation of next() and is
     *  intended to be overwritten by sub classes.
     */
    Iterator.prototype._generate = DONE_GEN;

    // public methods ---------------------------------------------------------

    /**
     * Returns a result object for the next value in the sequence represented
     * by this iterator.
     *
     * @returns {Object}
     *  An iterator result object with the boolean property "done", and the
     *  property "value", if the value of the property "done" is false. Sub
     *  classes may add more value properties to the result object.
     */
    Iterator.prototype.next = function () {

        // loop until the generator returns a result object
        var result = null;
        while (!result || (typeof result !== 'object')) { result = this._generate(); }

        // exit if the iterator has left the last sequence value
        if (result.done === true) { return this.abort(); }

        // add the "done" property to the result object
        result.done = false;
        return result;
    };

    /**
     * Immediately sets this iterator into finished state. All following calls
     * of the method "next()" will return a result object with the property
     * "done" set to true.
     *
     * @returns {Object}
     *  The result object for finished state containing a "done" property with
     *  the value true, as a convenience shortcut for implementations in sub
     *  classes.
     */
    Iterator.prototype.abort = function () {
        // overwrite the protoype method with a method always returning finished results
        return (this.next = DONE_GEN)();
    };

    // constants --------------------------------------------------------------

    /**
     * An empty iterator that represents the empty sequence and is always in
     * finished state.
     *
     * @type Iterator
     * @constant
     */
    Iterator.EMPTY = new Iterator();

    // class SingleIterator ===================================================

    /**
     * A simple iterator that will visit a single result object once.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Object} result
     *  The result to be returned by the first invocation of the method next()
     *  of the iterator. MUST contain a property "value". The property "done"
     *  with value false may be missing and will be added to the result object
     *  automatically.
     */
    var SingleIterator = Iterator.extend(function (result) {

        // base constructor
        Iterator.call(this);

        // the result to be returned by the first call of next()
        this._result = result;

    }); // class SingleIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    SingleIterator.prototype._generate = function () {
        this.abort();
        return this._result;
    };

    // class IndexIterator ====================================================

    /**
     * An iterator that will generate an increasing (or decreasing) sequence of
     * integer indexes upon calling its next() method.
     *
     * The result objects will contain the current index as value.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Number} size
     *  The length of the index sequence to be generated. The iterator will
     *  generate all indexes starting from zero, that are less than this value
     *  (unless the option 'offset' has been used, see below).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Number} [options.offset=0]
     *      If set to any number, it will be added to the indexes generated by
     *      this iterator.
     *  - {Boolean} [options.reverse=false]
     *      If set to true, the indexes will be generated in reversed order.
     */
    var IndexIterator = Iterator.extend(function (size, options) {

        // base constructor
        Iterator.call(this);

        // the size of the index sequence
        this._size = size;
        // the offset to be added to the generated indexes
        this._off = Utils.getNumberOption(options, 'offset', 0);
        // whether to iterate in reversed order
        this._rev = Utils.getBooleanOption(options, 'reverse', false);
        // the next index to be visited
        this._i = this._rev ? (size - 1) : 0;

    }); // class IndexIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    IndexIterator.prototype._generate = function () {

        var index = this._i;

        // exit if the iterator has reached the last index
        if (this._rev ? (index < 0) : (index >= this._size)) { return this.abort(); }

        // create the result object
        var result = { value: index + this._off };
        this._i += this._rev ? -1 : 1;
        return result;
    };

    // class IntervalIterator =================================================

    /**
     * An iterator that will generate an increasing (or decreasing) sequence of
     * integer indexes inside the specified closed interval upon calling its
     * next() method.
     *
     * @constructor
     *
     * @extends IndexIterator
     *
     * @param {Number} first
     *  The first index to be generated by the iterator.
     *
     * @param {Number} last
     *  The last index to be generated by the iterator.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.reverse=false]
     *      If set to true, the indexes will be generated in reversed order.
     *      The index passed to the parameter "last" will be generated first.
     */
    var IntervalIterator = IndexIterator.extend(function (first, last, options) {

        // base constructor
        IndexIterator.call(this, last - first + 1, { offset: first, reverse: options && options.reverse });

    }); // class IntervalIterator

    // class ArrayIterator ====================================================

    /**
     * An iterator that visits the elements of an array, or another array-like
     * object, upon calling its next() method.
     *
     * The result objects will contain the following value properties:
     *  - {Any} value
     *      The value of the array element currently visited.
     *  - {Number} index
     *      The index of the array element currently visited.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Array} array
     *  The array, or another array-like object.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.reverse=false]
     *      If set to true, the array will be visited in reverse order, from
     *      its last to its first element.
     *  - {Number} [options.begin]
     *      If specified, only the array elements with an index greater than or
     *      equal to this value (in reversed mode: with an index less than or
     *      equal to this value) will be visited. If omitted, all elements from
     *      the beginning (in reverse mode: from the end) of the array will be
     *      visited.
     *  - {Number} [options.end]
     *      If specified, only the array elements with an index less than this
     *      value (in reverse mode: with an index greater than this value) will
     *      be visited (half-open interval!). If omitted, all elements to the
     *      end (in reverse mode: to the beginning) of the array will be
     *      visited.
     */
    var ArrayIterator = Iterator.extend(function (array, options) {

        // base constructor
        Iterator.call(this);

        // the array to be iterated over
        this._arr = array;
        // whether to iterate in reversed order
        this._rev = Utils.getBooleanOption(options, 'reverse', false);
        // the first array index
        this._beg = Utils.getIntegerOption(options, 'begin', this._rev ? (array.length - 1) : 0);
        // one beyond the last array index
        this._end = Utils.getIntegerOption(options, 'end', this._rev ? -1 : array.length);
        // the index of the next array element to be visited
        this._i = this._beg;

    }); // class ArrayIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    ArrayIterator.prototype._generate = function () {

        var index = this._i;

        // exit if the iterator has reached the last element
        if (this._rev ? (index <= this._end) : (index >= this._end)) { return this.abort(); }

        // create the result object
        var result = { value: this._arr[index], index: index };
        this._i += this._rev ? -1 : 1;
        return result;
    };

    // class ObjectIterator ===================================================

    /**
     * An iterator that visits the properties of an object upon calling its
     * next() method.
     *
     * The result objects will contain the following value properties:
     *  - {Any} value
     *      The value of the object property currently visited.
     *  - {String} key
     *      The name of the object property currently visited.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Object} object
     *  The object whose properties will be visited by the iterator. The order
     *  of the properties is undefined.
     */
    var ObjectIterator = Iterator.extend(function (object) {

        // base constructor
        Iterator.call(this);

        // the object to be iterated over
        this._obj = object;
        // the property names of the object, as array
        this._keys = Object.keys(object);
        // the index of the next key to be visited
        this._i = 0;

    }); // class ObjectIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    ObjectIterator.prototype._generate = function () {

        // exit if the iterator has reached the last property
        var length = this._keys.length;
        if (this._i >= length) { return this.abort(); }

        // create the result object
        var object = this._obj;
        var key = this._keys[this._i];
        var result = (key in object) ? { value: object[key], key: key } : null;
        this._i += 1;
        return result;
    };

    // class GeneratorIterator ================================================

    /**
     * An iterator that receives the iterator results from a generator callback
     * function.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Function} generator
     *  A callback function that generates the result objects for the iterator.
     *  If the return value of the callback function is an object, it will be
     *  set as result of the iterator step. If the object contains a property
     *  "done" set to true, the iterator switches to finished state. Otherwise,
     *  the property "done" set to false will be added to the result object
     *  automatically. If the return value is not an object, the iterator will
     *  skip the result, and will continue to fetch results from the callback
     *  function until a result object will be returned.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    var GeneratorIterator = Iterator.extend(function (generator, context) {

        // base constructor
        Iterator.call(this);

        // the generator, bound to the passed context
        this._generate = generator.bind(context);

    }); // class GeneratorIterator

    // class TransformIterator ================================================

    /**
     * An iterator that transforms (and optionally filters) the values of the
     * passed source iterator.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function|String} transform
     *  A callback function that receives the result objects of the source
     *  sequence, and returns the modified result object, or a new result
     *  object, or a falsy value to filter the result. Alternatively, can be
     *  the name of a property in the result objects returned by the source
     *  sequence that will be used as iterator value. If set to a callback
     *  function, it will receive the following parameters:
     *  (1) {Any} value
     *      The current value received from the source sequence.
     *  (2) {Object} result
     *      The current and complete result object of the source sequence, with
     *      the standard properties "done" (always false) and "value", and all
     *      other non-standard value properties returned by the source
     *      sequence. Can be modified in-place and returned.
     *  If the return value is an object, it will be set as result object of
     *  the iterator step. If the object contains a property "done" set to
     *  true, the transformation iterator stops regardless if the source
     *  sequence is already done. Otherwise, the property "done" set to false
     *  will be added to the result object automatically. If the return value
     *  is not an object, the transformation iterator will skip the result and
     *  continue to fetch values from the source sequence.
     *
     * @param {Object} [context]
     *  The calling context for the transform callback function.
     */
    var TransformIterator = Iterator.extend(function (iterable, transform, context) {

        // base constructor
        Iterator.call(this);

        // the iterator for the passed sequence
        this._iter = Iterator.from(iterable);
        // convert a property name passed as transformator to a function that replaces the property "value"
        this._transform = (typeof transform === 'string') ? function (object, result) {
            result.value = object[transform];
            return result;
        } : transform.bind(context);

    }); // class TransformIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    TransformIterator.prototype._generate = function () {
        var result = this._iter.next();
        return result.done ? result : this._transform(result.value, result);
    };

    // class FilterIterator ===================================================

    /**
     * An iterator that filters the values of the passed source sequence
     * according to a truth test.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function|String} filter
     *  A callback predicate function that receives the result objects of the
     *  source sequence, and returns whether the resulting filter iterator will
     *  visit that result; or the name of a property in the result objects
     *  returned by the source sequence that will be checked for truthness. If
     *  set to a callback function, it will receive the following parameters:
     *  (1) {Any} value
     *      The current value received from the source sequence.
     *  (2) {Object} result
     *      The current and complete result object of the source sequence, with
     *      the standard properties "done" (always false) and "value", and all
     *      other non-standard value properties returned by the source
     *      sequence.
     *  If the return value is truthy, the resulting filter iterator will visit
     *  the passed result.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     */
    var FilterIterator = Iterator.extend(function (iterable, filter, context) {

        // base constructor
        Iterator.call(this);

        // the iterator for the passed sequence
        this._iter = Iterator.from(iterable);
        // convert a property name passed as predicate to a function
        this._filter = (typeof filter === 'string') ? _.property(filter) : filter.bind(context);

    }); // class FilterIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    FilterIterator.prototype._generate = function () {
        var result = this._iter.next();
        return result.done ? result : this._filter(result.value, result) ? result : null;
    };

    // class ReduceIterator ===================================================

    /**
     * An iterator that tries to reduce the elements of the source sequence by
     * combining the result values of consecutive sequence values.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} reduce
     *  A callback function that receives two consecutive result objects of the
     *  source sequence, and returns a new result object, if the source results
     *  can be combined. Receives the following parameters:
     *  (1) {Object} result1
     *      The first result object of the source sequence, with the standard
     *      properties "done" (always false) and "value", and all other
     *      non-standard value properties returned by the source sequence.
     *  (2) {Object} result2
     *      The second result object of the source sequence, with the standard
     *      properties "done" (always false) and "value", and all other
     *      non-standard value properties returned by the source sequence.
     *  If the return value is an object, it will be used as combined iterator
     *  result, and the reducing iterator continues to try combining subsequent
     *  results of the source sequence. If the returned object contains a
     *  property "done" set to true, the reducing iterator stops regardless if
     *  the source sequence is already done. Otherwise, the property "done" set
     *  to false will be added to the result object automatically. If the
     *  return value is not an object, the passed results cannot be combined,
     *  and the reducing iterator visits the first result. Either of the passed
     *  result objects can be modified in-place, and can be returned by the
     *  callback function.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    var ReduceIterator = Iterator.extend(function (iterable, reduce, context) {

        // base constructor
        Iterator.call(this);

        // the iterator for the passed sequence
        this._iter = Iterator.from(iterable);
        // the bound callback function
        this._reduce = reduce.bind(context);
        // cached iterator result that may be combined with its successors
        this._pending = this._iter.next();

    }); // class ReduceIterator

    // private methods --------------------------------------------------------

    /**
     * Replaces the current pending result with the passed result, and returns
     * the old pending result.
     *
     * @private
     */
    ReduceIterator.prototype._replacePending = function (nextResult) {
        var result = this._pending;
        this._pending = nextResult;
        return result;
    };

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    ReduceIterator.prototype._generate = function () {

        // no more data left for the iterator
        if (this._pending.done) { return this.abort(); }

        // fetch the next result from the iterator; return the cached result from
        // preceding call, if the original iterator is done
        var nextResult = this._iter.next();
        if (nextResult.done) { return this._replacePending(nextResult); }

        // try to combine two consecutive results via the callback function
        var reducedResult = this._reduce(this._pending, nextResult);

        // the results cannot be combined: cache the new result, and return the old cached result
        if (!reducedResult || !(reducedResult instanceof Object)) {
            return this._replacePending(nextResult);
        }

        // the callback may stop the iteration by returning a 'done' object
        if (reducedResult.done === true) {
            return this._replacePending(reducedResult);
        }

        // use the new result as pending result, and continue with the next iterator result
        this._pending = reducedResult;
        return null;
    };

    // class NestedIterator ===================================================

    /**
     * An iterator that combines an outer value sequence with multiple inner
     * sequences. For each single value of the outer sequence, a new inner
     * iterator will be created by invoking the passed generator function.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Any} outerIterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} innerGenerator
     *  A callback function that creates and returns a value sequence for each
     *  value of the outer sequence. Receives the following parameters:
     *  (1) {Any} outerValue
     *      The current value received from the outer sequence.
     *  (2) {Object} outerResult
     *      The current result object of the iterator visiting the outer
     *      sequence, with the standard properties "done" (always false) and
     *      "value", and all other non-standard value properties returned by
     *      the outer sequence.
     *  Must return an iterable sequence accepted by the static method
     *  Iterator.from(), or the value null. The returned sequence may be empty.
     *  In the latter cases (value null, or empty sequence), the next value of
     *  the outer sequence will be fetched until a non-empty inner sequence has
     *  been found.
     *
     * @param {Function} [transformResults]
     *  A callback function that returns the result object by combining the
     *  result objects of the outer and inner iterators. If omitted, the result
     *  objects of the inner iterators will be returned as received. The
     *  callback function receives the following parameters:
     *  (1) {Object} outerResult
     *      The current result object of the iterator visiting the outer value
     *      sequence, with the standard properties "done" (always false) and
     *      "value", and all other non-standard value properties returned by
     *      the outer sequence. This result object MUST NOT be modified!
     *  (2) {Object} innerResult
     *      The current result object of the iterator visiting the inner value
     *      sequence, with the standard properties "done" (always false) and
     *      "value", and all other non-standard value properties returned by
     *      the inner sequence. This result object CAN be modified in-place and
     *      returned.
     *  If the return value is an object, it will be set as result object of
     *  the nested iterator. If the object contains a property "done" set to
     *  true, the nested iterator stops regardless if any of the source value
     *  sequences are already done. Otherwise, the property "done" set to false
     *  will be added to the result object automatically. If the return value
     *  is not an object, the nested iterator will skip the result silently.
     *
     * @param {Object} [context]
     *  The calling context for the callback functions.
     */
    var NestedIterator = Iterator.extend(function (outerIterable, innerGenerator, transformResults, context) {

        // base constructor
        Iterator.call(this);

        // iterator for the outer source sequence
        this._iter1 = Iterator.from(outerIterable);
        // the current result of the outer iterator (used multiple times during inner iterator)
        this._result1 = this._iter1.next();
        // the bound generator function for the inner sequences
        this._gen2 = innerGenerator.bind(context);
        // the current inner iterator
        this._iter2 = null;
        // the bound transformation function for the results
        this._transform = transformResults ? transformResults.bind(context) : null;

    }); // class NestedIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    NestedIterator.prototype._generate = function () {

        // exit if the outer iterator is done
        if (this._result1.done) { return this.abort(); }

        // create a new inner iterator on demand (callback function may return null to signal an empty iterator)
        if (!this._iter2) {
            this._iter2 = Iterator.from(this._gen2(this._result1.value, this._result1));
        }

        // the result from the inner iterator
        var innerResult = this._iter2.next();

        // next value from outer iterator, if inner iterator is done
        if (innerResult.done) {
            this._result1 = this._iter1.next();
            this._iter2 = null;
            return null;
        }

        // create the result for the nested iterator, skip non-object results
        return this._transform ? this._transform(this._result1, innerResult) : innerResult;
    };

    // class SerialIterator ===================================================

    /**
     * An iterator that visits the results provided by the passed value
     * sequences one after the other.
     *
     * @constructor
     *
     * @extends NestedIterator
     *
     * @param {Any} [iterable1 ...]
     *  An arbitrary number of source sequences to be iterated over. Each value
     *  will be passed to the static method Iterator.from() to create an
     *  iterator object.
     */
    var SerialIterator = NestedIterator.extend(function () {

        // base constructor
        NestedIterator.call(this, _.map(arguments, Iterator.from), _.identity);

    }); // class SerialIterator

    // class MultiIteratorState ===============================================

    /**
     * A helper class representing the state of a single source sequence in an
     * iterator working on multiple source sequences.
     *
     * @constructor
     *
     * @private
     */
    function MultiIteratorState(index, iterables, indexers, context) {

        // the sequence index
        this.index = index;
        // convert sequence to iterator
        this.iter = Iterator.from(iterables[index]);
        // the bound indexer callback function
        this.indexer = (indexers instanceof Function) ? indexers.bind(context) : indexers[index].bind(context);
        // the cached next (not yet visited) result object
        this.result = null;
        // the current offset returned by the indexer
        this.offset = null;

        // fetch first iterator result and offset
        this.next();

    } // MultiIteratorState

    // public methods ---------------------------------------------------------

    /**
     * Fetches the next result from the wrapped source sequence, and updates
     * the indexer offset.
     *
     * @returns {Object}
     *  The iterator result object.
     */
    MultiIteratorState.prototype.next = function () {
        var result = this.result = this.iter.next();
        this.offset = result.done ? null : this.indexer(result.value, this.result, this.index);
        return result;
    };

    // class MultiIteratorBase ================================================

    /**
     * Implementation helper base class for iterator classes taking multiple
     * source sequences and indexer callback functions.
     *
     * @constructor
     *
     * @extends Iterator
     *
     * @param {Array<Any>} iterables
     *  An arbitrary number of source sequences to be iterated over. Each value
     *  will be passed to the static method Iterator.from() to create an
     *  iterator object.
     *
     * @param {Function|Array<Function>} indexers
     *  The callback functions that convert the values of the source sequences
     *  to a numeric offset or sorting index. If set to a single function, it
     *  will be used for all source sequences. Otherwise, the array MUST have
     *  the same length as the array containing the source sequences. Each
     *  callback function receives the following parameters:
     *  (1) {Any} value
     *      The current value of the respective source sequence.
     *  (2) {Object} result
     *      The complete result object of the respective source iterator.
     *  (3) {Number} index
     *      The array index of the source sequence (may be useful for a single
     *      callback function used for all sequences).
     *  The callback functions MUST return finite numbers. The returned indexes
     *  MUST grow strictly per source sequence for all its results.
     *
     * @param {Object} [context]
     *  The calling context for the indexer callback functions.
     *
     * @property {Array<MultiIteratorState>} _states
     *  The state descriptors for all source sequences.
     */
    var MultiIteratorBase = Iterator.extend(function (iterables, indexers, context) {

        // base constructor
        Iterator.call(this);

        // the state descriptors for all source sequences
        this._states = _.times(iterables.length, function (index) {
            return new MultiIteratorState(index, iterables, indexers, context);
        });

    }); // class MultiIteratorBase

    // class OrderedIterator ==================================================

    /**
     * An iterator that visits the values provided by the passed sequences in a
     * specific order, according to some offset or sorting index for the values
     * of all sequences.
     *
     * The ordered iterator will pick the result of the sequence iterator with
     * the smallest sort index, according to the return values of the indexers.
     * The result objects returned by the ordered iterator will be shallow
     * copies of the result objects of the sequence iterators, with the
     * following additional properties:
     *  - {Number} offset
     *      The offset or sorting index of the result of the source iterator,
     *      as provided by the indexer callback function.
     *  - {Number} srcIndex
     *      The array index of the source sequence that provided the current
     *      value.
     *
     * @constructor
     *
     * @extends MultiIteratorBase
     *
     * @param {Array<Any>} iterables
     *  An arbitrary number of source sequences to be iterated over. Each value
     *  will be passed to the static method Iterator.from() to create an
     *  iterator object.
     *
     * @param {Function|Array<Function>} indexers
     *  The callback functions that convert the values of the source sequences
     *  to a numeric offset or sorting index that will be used to decide which
     *  value of the sequences will be visited next. If set to a single
     *  function, it will be used for all source sequences. Otherwise, the
     *  array MUST have the same length as the array containing the source
     *  sequences. Each callback function receives the following parameters:
     *  (1) {Any} value
     *      The current value of the respective source sequence.
     *  (2) {Object} result
     *      The complete result object of the respective source iterator.
     *  (3) {Number} index
     *      The array index of the source sequence (may be useful for a single
     *      callback function used for all sequences).
     *  The callback functions MUST return finite numbers. The returned indexes
     *  MUST grow strictly per source sequence for all its results.
     *
     * @param {Object} [context]
     *  The calling context for the indexer callback functions.
     */
    var OrderedIterator = MultiIteratorBase.extend(function (iterables, indexers, context) {

        // base constructor
        MultiIteratorBase.call(this, iterables, indexers, context);

    }); // class OrderedIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    OrderedIterator.prototype._generate = function () {

        // find the iterator result with the minimum offset
        var state = this._states.reduce(function (state1, state2) {
            // skip finished sequences (state1 may still be null)
            if (state2.result.done) { return state1; }
            // return first active seuence (if state1 is still null)
            if (!state1) { return state2; }
            // pick sequence with lower offset (prefer first sequence if offsets are equal)
            return (state1.offset <= state2.offset) ? state1 : state2;
        }, null);

        // early exit, if all iterators are done
        if (!state) { return this.abort(); }

        // create the result object to be returned
        var result = _.extend({ offset: state.offset, srcIndex: state.index }, state.result);

        // step the found iterator ahead
        state.next();

        return result;
    };

    // class ParallelIterator =================================================

    /**
     * An iterator that visits the values provided by the passed sequences
     * simultaneously, according to some offset or sorting index for the values
     * of all iterators.
     *
     * The parallel iterator will visit the result objects of the sequence
     * iterators simultaneously. The iterator will provide all results of the
     * source sequences that have an equal sorting index, and will pass the
     * value null as result for all sequences that have skipped that sorting
     * index.  The result objects of the parallel iterator will contain the
     * following value  properties:
     *  - {Array<Any>} value
     *      The values of the source sequences, or nullfor all iterators that
     *      did not provide a result object for the current sorting index that
     *      is contained in the property "offset". Note that a source sequence
     *      may provide the value null by itself. To detect a skipped sequence,
     *      use the "results" array property.
     *  - {Array<Object|Null>} results
     *      The complete result objects of the iterators visiting the source
     *      sequences, or null for iterators that did not provide a result
     *      object for the current sorting index that is contained in the
     *      property "offset". At least one element in this array will be a
     *      valid result object. This array MUST NOT be changed!
     *  - {Number} offset
     *      The offset or sorting index of the results of the source iterators,
     *      as provided by the indexer callback functions.
     *  - {Boolean} complete
     *      Whether all source sequences have provided a result object for the
     *      current sorting index (i.e. whether all elements in the property
     *      "results" are result objects). This property is provided for
     *      convenience and to improve performance.
     *
     * @constructor
     *
     * @extends MultiIteratorBase
     *
     * @param {Array<Any>} iterables
     *  An array of objects implementing the standard EcmaScript iterator
     *  protocol (each iterator must provide a method next() that returns an
     *  object with the properties 'done' and 'value').
     *
     * @param {Function|Array<Function>} indexers
     *  The callback functions that convert the results of the source iterators
     *  to an offset or sorting index that will be used to create the combined
     *  results of the new parallel iterator. If set to a single function, it
     *  will be used for all source iterators. Otherwise, the array MUST have
     *  the same length as the array containing the source iterators. Each
     *  callback function receives the following parameters:
     *  (1) {Any} value
     *      The current value of the respective source sequence.
     *  (2) {Object} result
     *      The complete result object of the respective source iterator.
     *  (3) {Number} index
     *      The array index of the source iterator (may be useful for a single
     *      callback function used for all iterators).
     *  The callback functions MUST return finite numbers. The returned indexes
     *  MUST grow strictly per source sequence for all its results.
     *
     * @param {Object} [context]
     *  The calling context for the indexer callback functions.
     */
    var ParallelIterator = MultiIteratorBase.extend(function (iterables, indexers, context) {

        // base constructor
        MultiIteratorBase.call(this, iterables, indexers, context);

        // next offset to be visited
        this._offset = this._states.reduce(function (offset, state) {
            return getMinOffset(offset, state.offset);
        }, null);

    }); // class ParallelIterator

    // protected methods ------------------------------------------------------

    /**
     * Generator callback function used by base class implementation.
     *
     * @protected
     */
    ParallelIterator.prototype._generate = function () {

        // early exit if all iterators are done
        if (this._offset === null) { return this.abort(); }

        // the result object to be returned, with arrays for the results and values of all sequences
        var result = { value: [], results: [], complete: true, offset: this._offset };
        // offset to be visited in next iteration step (minimum of all offsets greater than currOffset)
        var nextOffset = null;

        // collect the results into an array, and step ahead all affected iterators
        this._states.forEach(function (state) {

            // skip current iterator, if it is done (offset is null), or if offset is greater than the current offset
            var offset = state.offset;
            if ((offset === null) || (this._offset < offset)) {
                nextOffset = getMinOffset(nextOffset, offset);
                result.value.push(null);
                result.results.push(null);
                result.complete = false;
                return;
            }

            // result of the current iterator will be part of the result of the parallel iterator
            result.value.push(state.result.value);
            result.results.push(state.result);

            // step the found iterator ahead
            state.next();

            // update next offset to be visited
            nextOffset = getMinOffset(nextOffset, state.offset);

        }, this);

        this._offset = nextOffset;

        return result;
    };

    // class Iterator (static part) ===========================================

    /**
     * Returns an iterator object suitable for the passed object or sequence.
     *
     * @param {Iterator|Array|Function|Object} iterable
     *  A sequence to be iterated over. If this value is an instance of class
     *  Iterator, it will be rturned as passed. If the value is an array, a new
     *  instance of class ArrayIterator will be returned. If the value is a
     *  function, a new instance of class GeneratorIterator will be returned.
     *  If the value is an object with a method "iterator()", it will be called
     *  and its result will be returned. Otherwise, an empty iterator will be
     *  returned.
     *
     * @returns {Iterator}
     *  An iterator object suitable for the passed object or sequence.
     */
    Iterator.from = function (iterable) {
        if (!iterable || !(iterable instanceof Object)) { return Iterator.EMPTY; }
        if (iterable instanceof Iterator) { return iterable; }
        if (iterable.iterator instanceof Function) { return iterable.iterator(); }
        // check array type after "iterator()" method (e.g. class TypedArray)
        if (iterable instanceof Array) { return new ArrayIterator(iterable); }
        if (iterable instanceof Function) { return new GeneratorIterator(iterable); }
        return Iterator.EMPTY;
    };

    /**
     * Collects all result values of the passed source sequence in a plain
     * JavaScript array.
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Array} [array]
     *  The array to be filled. The values of the source sequence will be
     *  appended to the array. If omitted, a bew empty array will be created
     *  and returned.
     *
     * @returns {Array}
     *  An array with all values returned by the sequence.
     */
    Iterator.toArray = function (iterable, array) {
        var iterator = Iterator.from(iterable);
        array = array || [];
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            array.push(result.value);
        }
        return array;
    };

    /**
     * Invokes the passed callback function for all values of the passed source
     * sequence.
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each value of the passed
     *  source sequence. Receives the following parameters:
     *  (1) {Any} value
     *      The current value from the source sequence.
     *  (2) {Object} result
     *      The complete iterator result object from the source iterator.
     *  If the callback function returns the Utils.BREAK object, the iteration
     *  process will be stopped immediately.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the callback function has
     *  returned Utils.BREAK to stop the iteration process; otherwise
     *  undefined.
     */
    Iterator.forEach = function (iterable, callback, context) {
        var iterator = Iterator.from(iterable);
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            if (callback.call(context, result.value, result) === Utils.BREAK) {
                return Utils.BREAK;
            }
        }
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for at least one value of the passed source sequence.
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each value of
     *  the passed source sequence until it returns a truthy value. After that,
     *  iteration will be stopped immediately, the remaining values of the
     *  source sequence will not be visited anymore. Receives the following
     *  parameters:
     *  (1) {Any} value
     *      The current value from the source sequence.
     *  (2) {Object} result
     *      The complete iterator result object from the source iterator.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for at
     *  least one value of the passed sequence.
     */
    Iterator.some = function (iterable, predicate, context) {
        var iterator = Iterator.from(iterable);
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            if (predicate.call(context, result.value, result)) { return true; }
        }
        return false;
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for all values of the passed source sequence.
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each value of
     *  the passed source sequence until it returns a falsy value.  After that,
     *  iteration will be stopped immediately, the remaining values of the
     *  source sequence will not be visited anymore. Receives the following
     *  parameters:
     *  (1) {Any} value
     *      The current value from the source sequence.
     *  (2) {Object} result
     *      The complete iterator result object from the source iterator.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for all
     *  values of the passed sequence.
     */
    Iterator.every = function (iterable, predicate, context) {
        var iterator = Iterator.from(iterable);
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            if (!predicate.call(context, result.value, result)) { return false; }
        }
        return true;
    };

    /**
     * Invokes the passed callback function for all values of the passed source
     * sequence, and returns the return values as plain JavaScript array.
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each value of the passed
     *  source sequence. Receives the following parameters:
     *  (1) {Any} value
     *      The current value from the source sequence.
     *  (2) {Object} result
     *      The complete iterator result object from the source iterator.
     *  The return value of the callback function will be inserted into the
     *  result array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Array<Any>}
     *  The return values of the callback function.
     */
    Iterator.map = function (iterable, callback, context) {
        var iterator = Iterator.from(iterable);
        var array = [];
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            array.push(callback.call(context, result.value, result));
        }
        return array;
    };

    /**
     * Invokes the passed callback function for all values of the passed source
     * sequence, and reduces these values to a single result value.
     *
     * @param {Any} initial
     *  The initial result value passed to the first invocation of the callback
     *  function (if the source sequence is empty, this value becomes the final
     *  result value immediately).
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each value of the passed
     *  source sequence. Receives the following parameters:
     *  (1) {Any} prevValue
     *      The result value of the previous invocation of this callback
     *      function (or the initial value on first invocation).
     *  (2) {Any} value
     *      The current value from the source sequence.
     *  (3) {Object} result
     *      The complete iterator result object from the source iterator.
     *  The return value of the callback function will become the new result
     *  value that will be passed as "prevValue" with the next invocation.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The final result (the last return value of the callback function).
     */
    Iterator.reduce = function (initial, iterable, callback, context) {
        var iterator = Iterator.from(iterable);
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            initial = callback.call(context, initial, result.value, result);
        }
        return initial;
    };

    /**
     * Returns the first value of the source sequence that passes a truth test.
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each value of
     *  the passed source sequence until it returns a truthy value. After that,
     *  iteration will be stopped immediately, the remaining values of the
     *  source sequence will not be visited anymore. Receives the following
     *  parameters:
     *  (1) {Any} value
     *      The current value from the source sequence.
     *  (2) {Object} result
     *      The complete iterator result object from the source iterator.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Object|Null}
     *  The complete iterator result of the first value from the passed source
     *  sequence that passes the truth test; or null, if no such value exists.
     */
    Iterator.find = function (iterable, predicate, context) {
        var iterator = Iterator.from(iterable);
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            if (predicate.call(context, result.value, result)) { return result; }
        }
        return null;
    };

    /**
     * Returns the last value of the source sequence that passes a truth test.
     *
     * @param {Any} iterable
     *  The source sequence to be iterated over. Will be passed to the static
     *  method Iterator.from() to create an iterator object.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each value of
     *  the passed source sequence. Receives the following parameters:
     *  (1) {Any} value
     *      The current value from the source sequence.
     *  (2) {Object} result
     *      The complete iterator result object from the source iterator.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Object|Null}
     *  The complete iterator result of the last value from the passed source
     *  sequence that passes the truth test; or null, if no such value exists.
     */
    Iterator.findLast = function (iterable, predicate, context) {
        var iterator = Iterator.from(iterable);
        var lastResult = null;
        for (var result = iterator.next(); !result.done; result = iterator.next()) {
            if (predicate.call(context, result.value, result)) { lastResult = result; }
        }
        return lastResult;
    };
    // classes ----------------------------------------------------------------

    Iterator.SingleIterator = SingleIterator;
    Iterator.IndexIterator = IndexIterator;
    Iterator.IntervalIterator = IntervalIterator;
    Iterator.ArrayIterator = ArrayIterator;
    Iterator.ObjectIterator = ObjectIterator;
    Iterator.GeneratorIterator = GeneratorIterator;
    Iterator.TransformIterator = TransformIterator;
    Iterator.FilterIterator = FilterIterator;
    Iterator.ReduceIterator = ReduceIterator;
    Iterator.NestedIterator = NestedIterator;
    Iterator.SerialIterator = SerialIterator;
    Iterator.OrderedIterator = OrderedIterator;
    Iterator.ParallelIterator = ParallelIterator;

    // exports ================================================================

    return Iterator;

});
