/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/tooltip', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/popup/basepopup'
], function (Utils, Forms, BasePopup) {

    'use strict';

    // class ToolTip ==========================================================

    /**
     * Wrapper class for a DOM node used as tooltip info element, shown on top
     * of the application window, and relative to an arbitrary DOM node.
     *
     * @constructor
     *
     * @extends BasePopup
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application owning the
     *  tooltip object, or an object with a method 'getWindowId' that returns
     *  such a window identifier. Used for debugging and logging of running
     *  timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Initial options to control the properties of the tooltip element.
     *  Supports all options also supported by the base class BasePopup. The
     *  default value of the option 'anchorPadding' will be set to 3 though.
     *  Additionally, the following options are supported:
     *  - {Boolean} [initOptions.hoverStyle=false]
     *      If set to true, the tool tip will be shown with dark background and
     *      light text color, and with a small arrow pointing towards the
     *      anchor node. Intended to be used for tool tips that will be shown
     *      automatically on hovering an element, with styling similar to the
     *      tool tips provided by the Bootstrap library.
     */
    var ToolTip = BasePopup.extend({ constructor: function (windowId, initOptions) {

        // base constructor ---------------------------------------------------

        BasePopup.call(this, windowId, Utils.extendOptions({
            restrictSize: 'width',
            minSize: 150,
            maxSize: 450,
            anchorPadding: 3
        }, initOptions));

        // methods ------------------------------------------------------------

        /**
         * Inserts the passed plain text into the tooltip.
         *
         * @param {String} text
         *  The text to be inserted into the tooltip.
         *
         * @returns {ToolTip}
         *  A reference to this instance.
         */
        this.setText = function (text) {
            this.clearContents(); // clean up and trigger events
            if (text) { this.getContentNode().text(text); }
            // immediately refresh position (prevent display glitches with oversized and thus wrapping text)
            this.refreshImmediately();
            return this;
        };

        // initialization -----------------------------------------------------

        // add tooltip styling
        this.getNode().addClass('popup-tooltip');

        // special styling for tool tips with hover style (Bootstrap styling)
        if (Utils.getBooleanOption(initOptions, 'hoverStyle', false)) {
            this.getNode().addClass('hover-style').append('<div class="hover-arrow">');
        }

    } }); // class ToolTip

    // global hover tool tip ==================================================

    // the current anchor node to show the global tool tip for
    var hoverNode = null;

    // the global tool tip instance (will live forever
    var hoverToolTip = new ToolTip(ToolTip.SINGLETON, {
        anchor: function () { return hoverNode; },
        anchorAlign: 'center',
        anchorBorder: ['bottom', 'top'],
        hoverStyle: true
    });

    var hoverSelector = '.io-ox-office-main .' + Forms.APP_TOOLTIP_CLASS;

    $('body').on('mouseenter', hoverSelector, function (event) {
        hoverNode = $(event.currentTarget);
        hoverToolTip.setText(hoverNode.attr('data-original-title')).show();

        // fix for Bug 52025
        var node = hoverToolTip.getNode();
        var arrow = node.find('.hover-arrow');

        var holderOffset = hoverNode.offset();
        var popupOffset = node.offset();

        arrow.css('left', Math.round(holderOffset.left + (hoverNode.width() / 2) - popupOffset.left));
    });

    $('body').on('mouseleave mouseup', hoverSelector, function () {
        hoverToolTip.hide();
    });

    // exports ================================================================

    return ToolTip;

});
