/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/io', [
    'io.ox/core/http',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/scheduler'
], function (CoreHTTP, Utils, Scheduler) {

    'use strict';

    // static class IO ========================================================

    /**
     * Provides static methods for server/client communication and host file
     * system access.
     */
    var IO = {};

    // constants --------------------------------------------------------------

    /**
     * The name of the document filter server module.
     */
    IO.FILTER_MODULE_NAME = 'oxodocumentfilter';

    /**
     * The name of the document converter server module.
     */
    IO.CONVERTER_MODULE_NAME = 'oxodocumentconverter';

    // public methods ---------------------------------------------------------

    /**
     * Sends a request to the server and returns a promise waiting for the
     * response.
     *
     * @param {String} module
     *  The name of the server module that will receive the request.
     *
     * @param {Object} params
     *  Parameters that will be inserted into the request URL (method GET), or
     *  into the request body (method POST).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.method='GET']
     *      The request method. Must be an upper-case string (for example,
     *      'GET', 'POST', etc.). Defaults to 'GET'.
     *
     * @returns {jQuery.Promise}
     *  The promise of the request. Will be resolved with the 'data' object
     *  returned by the response, if available; or the valid return value of
     *  the result filter callback function, if specified. Otherwise, the
     *  promise will be rejected. Contains the additional method 'abort()' that
     *  allows to abort the running request which rejects the promise. Calling
     *  this method has no effect, if the request is finished already.
     */
    IO.sendRequest = function (module, params, options) {

        // extract the request method
        var method = Utils.getStringOption(options, 'method', 'GET');
        // properties passed to the server request
        var requestProps = { module: module, params: params };
        // send the AJAX request
        var ajaxRequest = CoreHTTP[method](requestProps);
        // deferred for synchronous abort (54803)
        var deferred = Scheduler.createDeferred('IO', 'IO.sendRequest');
        // the corresponding promise object
        var promise = deferred.promise();

        ajaxRequest.done(function (response) {
            if (Utils.getBooleanOption(response, 'hasErrors', false)) {
                deferred.reject(response);
            } else {
                deferred.resolve(response);
            }
        });

        ajaxRequest.fail(function (response) {
            deferred.reject(response);
        });

        // add an abort() method
        promise.abort = function () {
            ajaxRequest.abort();
            deferred.reject('abort');
        };

        return promise;
    };

    /**
     * Loads the specified JSON source file.
     *
     * @param {String} fileName
     *  The name of the file, relative to the application base path, as used by
     *  RequireJS, without file extension (which is assumed to be 'json').
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the JSON data loaded from the
     *  specified source file, or that will be rejected on any error (e.g.,
     *  source file not found, or invalid JSON data).
     */
    IO.loadJSON = function (fileName) {

        // use the text loader of RequireJS to load the JSON file
        return require(['text!' + fileName + '.json']).then(function (response) {

            // check response
            if ((typeof response !== 'string') || (response.length === 0)) {
                Utils.error('IO.loadJSON(): invalid response');
                return new $.Deferred().reject(response);
            }

            // parse JSON string to object
            try {
                return JSON.parse(response);
            } catch (err) {
                Utils.exception(err);
                return new $.Deferred().reject(err);
            }
        });
    };

    /**
     * Fetches the source code of the specified application in the background.
     *
     * @param {String} modulePath
     *  The path to the application main module. The implementation will fetch
     *  the file 'main.js' inside this directory.
     */
    IO.prefetchModuleSource = _.memoize(function (modulePath) {

        // determine the module for server-side initialization
        var module = modulePath.substr(modulePath.lastIndexOf('/') + 1);

        // send prefetch request to server
        if (module.length > 0) {
            IO.sendRequest(IO.FILTER_MODULE_NAME, { action: 'prefetchofficeparts', part: module });
        }

        // fetch source code for module
        _.delay(function () {
            Utils.log('IO.prefetchModuleSource(): fetching ' + modulePath + ' ...');
            var t0 = Date.now();
            require([modulePath + '/main'], function () {
                Utils.log('... fetching ' + modulePath + ' done in ' + (Date.now() - t0) + 'ms');
            });
        }, 2500);
    });

    // exports ================================================================

    return IO;

});
