/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/dialog/querydialog', [
    'io.ox/office/tk/dialog/basedialog',
    'gettext!io.ox/office/tk/main'
], function (BaseDialog, gt) {

    'use strict';

    // constants ==============================================================

    var YES_LABEL = /*#. button label in a Yes/No dialog */ gt('Yes');
    var NO_LABEL = /*#. button label in a Yes/No dialog */ gt('No');

    // class QueryDialog ======================================================

    /**
     * Creates a simple dialog with a message text and Yes and No buttons.
     * Optionally, the dialog may contain an additional Cancel button next to
     * the Yes and No buttons.
     *
     * By default, the promise returned by the public method show() of an
     * instance of this dialog will be resolved with the boolean value true
     * when pressing the Yes button, and rejected when pressing the No button.
     *
     * If the dialog has been opened with an additional Cancel button (see
     * constructor option "cancel" below), the promise returned by the public
     * method show() will behave differently: When pressing the Yes button, the
     * promise will be resolved with the boolean value true. When pressing the
     * No button, the promise will be resolved with the boolean value false.
     * Otherwise (dialog canceled), the promise will be rejected.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application that is
     *  creating the dialog, or an object with a method getWindowId() that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {String} message
     *  The message text shown in the dialog body element.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  BaseDialog. The default value of the option "okLabel" will be changed
     *  to the translated string "Yes", and the default value of the option
     *  "cancelLabel" will be changed to the translated string "No".
     *  Additionally, the following options are supported:
     *  - {Boolean} [initOptions.cancel=false]
     *      Whether to show an additional Cancel button next to the Yes and No
     *      buttons. See comments above about the behavioral changes for this
     *      case.
     */
    var QueryDialog = BaseDialog.extend(function (windowId, message, initOptions) {

        // whether to show an additional Cancel button
        var cancelMode = initOptions && initOptions.cancel;

        // base constructor
        BaseDialog.call(this, windowId, _.extend({
            width: 400
        }, initOptions, {
            okLabel: YES_LABEL,
            cancelLabel: cancelMode ? null : NO_LABEL
        }));

        // create a <label> element with the message text
        this.append($('<label>').text(message));

        // return true for OK (Yes) button
        this.setOkHandler(_.constant(true));

        // add an own No button in Yes/No/Cancel mode
        if (cancelMode) {
            this.createButton('no', NO_LABEL, {
                insertBefore: BaseDialog.OK_ACTION,
                actionHandler: _.constant(false)
            });
        }

    }); // class QueryDialog

    // exports ================================================================

    return QueryDialog;

});
