/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/spinfield', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/control/textfield',
    'io.ox/office/settings/units'
], function (Utils, KeyCodes, Forms, Tracking, TextField, Units) {

    'use strict';

    // class SpinField ========================================================

    /**
     * Creates a numeric field control with additional controls used to spin
     * (decrease and increase) the value.
     *
     * @constructor
     *
     * @extends TextField
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application owning the
     *  spin field object, or an object with a method 'getWindowId' that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the TextField base class.
     *  If the option 'validator' exists, it MUST be an instance of the class
     *  TextField.NumberValidator (or a sub class). Alternatively, the options
     *  supported by the constructor of the class TextField.NumberValidator can
     *  be passed. In this case, a new instance of TextField.NumberValidator
     *  will be created automatically based on these options. Additionally, the
     *  following options are supported:
     */
    function SpinField(windowId, initOptions) {

        // self reference
        var self = this;

        // the number validator of this spin field
        var validator = Utils.getObjectOption(initOptions, 'validator') || new TextField.NumberValidator(initOptions);

        // whether the unit is percentage
        var percentageUnit = Utils.getBooleanOption(initOptions, 'percentageUnit', false);

        // initial click on the spin button has to wait for control focus
        var pendingStep = null;

        // base constructors --------------------------------------------------

        TextField.call(this, windowId, Utils.extendOptions({ keyboard: 'number' }, initOptions, { validator: validator }));

        // private methods ----------------------------------------------------

        function getSmallStep() {
            return percentageUnit ? 1 : Units.getStandardUnitStep();
        }

        function getLargeStep() {
            return getSmallStep() * 5;
        }

        function getOldValue(unit) {
            var fieldValue = self.getFieldValue();

            return percentageUnit ? parseInt(fieldValue, 10) : Utils.convertHmmToLength(fieldValue, unit);
        }

        function convertNewValue(newValue, unit) {
            return percentageUnit ? newValue : Utils.convertLengthToHmm(newValue, unit);
        }

        /**
         * Changes the current value of the spin field, according to the passed
         * step of the general unit.
         */
        function changeValue(step) {
            // the measurement unit to be shown in the spin filed
            var unit = Units.getStandardUnit();
            // the current value of the spin field
            var oldValue = getOldValue(unit);
            // the new value
            var newValue = oldValue + step;

            newValue = convertNewValue(newValue, unit);
            // set new value, after restricting to minimum/maximum
            newValue = validator.restrictValue(newValue);
            self.setValue(newValue);
        }

        /**
         * Creates and returns a spin button. Initializes tracking used to
         * change the current value.
         */
        function createSpinButton(increase) {

            // create the spin button element
            var spinButton = $(Forms.createButtonMarkup({ focusable: false })).addClass(increase ? 'spin-up' : 'spin-down');

            // enables or disables node tracking according to own enabled state
            function initializeTracking() {
                if (self.isEnabled()) {
                    Tracking.enableTracking(spinButton, { autoRepeat: true });
                } else {
                    Tracking.disableTracking(spinButton);
                }
            }

            // handler for 'tracking:start' and 'tracking:repeat' events: change the control value
            function trackingStartHandler() {
                var step = increase ? getSmallStep() : -getSmallStep();
                // bug 46846: initial click on a spin button before this control is focused needs
                // to be deferred until the base class TextField has processed the focus events
                if (self.hasFocus()) {
                    changeValue(step);
                } else {
                    pendingStep = step;
                }
            }

            // handler for 'tracking:end' and 'tracking:cancel' events: return focus to text field
            function trackingEndHandler() {
                self.executeDelayed(function () { Utils.setFocus(self.getInputNode()); }, 'SpinField.createSpinButton');
            }

            // add the caret symbol
            spinButton.append(Forms.createCaretMarkup(increase ? 'up' : 'down'));

            // enable/disable node tracking according to own enabled state
            self.on('group:enable', initializeTracking);
            initializeTracking();

            // register tracking event listeners to change the current value
            spinButton.on({
                'tracking:start tracking:repeat': trackingStartHandler,
                'tracking:end tracking:cancel': trackingEndHandler
            });

            // enable/disable the button according to the current value
            self.registerUpdateHandler(function (value) {
                var enabled = increase ? (value < validator.getMax()) : (value > validator.getMin());
                Forms.enableNodes(spinButton, enabled);
            });

            return spinButton;
        }

        /**
         * Handles keyboard events in the text field.
         */
        function inputKeyDownHandler(event) {

            if (KeyCodes.hasModifierKeys(event)) { return; }

            switch (event.keyCode) {
                case KeyCodes.UP_ARROW:
                    changeValue(getSmallStep());
                    return false;
                case KeyCodes.DOWN_ARROW:
                    changeValue(-getSmallStep());
                    return false;
                case KeyCodes.PAGE_UP:
                    changeValue(getLargeStep());
                    return false;
                case KeyCodes.PAGE_DOWN:
                    changeValue(-getLargeStep());
                    return false;
            }
        }

        /**
         * Handles updates of the input text field
         */
        function updateHandler(value) {
            this.getInputNode().attr({ 'aria-valuenow': value, 'aria-valuetext': this.getInputNode().val() });
        }

        // initialization -----------------------------------------------------

        // add special marker class used to adjust formatting, add spin buttons
        this.getNode().addClass('spin-field').append(
            $('<div>').addClass('spin-wrapper').append(createSpinButton(true), createSpinButton(false))
        );

        // add ARIA attributes (note: role 'spinbutton' would be correct but doesn't work with VoiceOver)
        this.getInputNode().attr({ role: 'slider', 'aria-valuemin': validator.getMin(), 'aria-valuemax': validator.getMax() });

        // register event handlers
        this.getInputNode().on('keydown', inputKeyDownHandler);

        // bug 46846: initial click on a spin button before this control is focused needs
        // to be deferred until the base class TextField has processed the focus events
        this.on('group:focus', function () {
            if (pendingStep !== null) {
                changeValue(pendingStep);
                pendingStep = null;
            }
        });

        // register input field update handler
        this.registerUpdateHandler(updateHandler);

    } // class SpinField

    // exports ================================================================

    // derive this class from class TextField
    return TextField.extend({ constructor: SpinField });

});
