/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/view/toolbars', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/textframework/view/controls',
    'gettext!io.ox/office/textframework/main'
], function (Utils, ToolBar, Labels, Controls, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CompoundButton = Controls.CompoundButton;

    // static class TextBaseToolBars ==========================================

    /**
     * A static map with tool bar classes for text editing applications.
     *
     * @type Object
     * @constant
     */
    var TextBaseToolBars = {};

    // class FontFamilyToolBar ================================================

    /**
     * A tool bar with controls to select the font family, and the font size.
     *
     * The tool bar will contain the following controls:
     *  (1) A font family picker (controller key 'character/fontname').
     *  (2) A font size picker (controller key 'character/fontsize').
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextBaseView} docView
     *  The document view instance containing this tool bar.
     */
    TextBaseToolBars.FontFamilyToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        ToolBar.call(this, docView, {
            shrinkToMenu: { icon: 'fa-font', tooltip: Labels.FONT_HEADER_LABEL }
        });

        // initialization -----------------------------------------------------

        // create the controls of this tool bar
        this.addGroup('character/fontname', new Controls.FontFamilyPicker(docView));
        this.addGap();
        this.addGroup('character/fontsize', new Controls.FontSizePicker(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class FontFamilyToolBar

    // class FormatPainterToolBar =============================================

    /**
     * A tool bar with a controls to copy the formatting of the selection.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextframeworkView} docView
     *  The document view instance containing this tool bar.
     */
    TextBaseToolBars.FormatPainterToolBar = ToolBar.extend({ constructor: function (docView) {

        var docModel = docView.getApp().getModel();

        // base constructor
        ToolBar.call(this, docView);

        // register a controller item for the drop-down menu that will be used as selector in Selenium tests
        docView.getApp().getController().registerDefinition('format/painter', {
            parent: 'document/editable',
            enable: function () {
                var selection = docModel.getSelection();
                return (!selection.isMultiSelection() && !selection.isSlideSelection());
            },
            get: function () { return docModel.isFormatPainterActive(); },
            set: function (state) { docModel.handleFormatPainter(state, { reset: true }); }
        });

        this.listenTo(docModel.getSelection(), 'update', function (jqe, e) {
            if (docModel.getSelection().isSlideSelection() || Utils.TOUCHDEVICE) { return; }

            if (e && e.event && e.event.type === 'mouseup' && docModel.isFormatPainterActive()) {
                docModel.handleFormatPainter(false);

            } else if (docModel.isFormatPainterActive() && docModel.getSelection().isSlideSelection()) {
                docModel.setFormatPainterState(false);
            }
        });

        this.listenTo(docModel.getSelection(), 'change', function () {
            if (docModel.getSelection().isSlideSelection()) { return; }

            if (docModel.getSelection().isDrawingSelection()) {
                docModel.handleFormatPainter(false);

            } else if (docModel.isFormatPainterActive() && docModel.getSelection().isSlideSelection()) {
                docModel.setFormatPainterState(false);

            } else if (docModel.isFormatPainterActive() && Utils.TOUCHDEVICE) {
                docModel.handleFormatPainter(false);
            }
        });

        // create the controls of this tool bar
        this.addGroup('format/painter', new Button(docView, { icon: 'docs-format-painter', tooltip: Labels.FORMAT_PAINTER_LABEL, toggle: true, dropDownVersion: { label: Labels.FORMAT_PAINTER_LABEL } }));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = docModel = null;
        });

    } }); // class FormatPainterToolBar

    // class FontStyleToolBar =================================================

    /**
     * A tool bar with controls to change different font style attributes.
     *
     * The tool bar will contain the following controls:
     *  (1) A bold button (controller key 'character/bold').
     *  (2) An italic button (controller key 'character/italic').
     *  (3) An underline button (controller key 'character/underline').
     *  (4) A strikeout button (controller key 'character/strike').
     *  (5) An escapement picker (controller key 'character/vertalign').
     *  (6) A reset button (controller key 'character/reset').
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextBaseView} docView
     *  The document view instance containing this tool bar.
     */
    TextBaseToolBars.FontStyleToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        ToolBar.call(this, docView, {
            shrinkToMenu: _.extend({
                splitKey: 'character/bold',
                caretTooltip: Labels.MORE_FONT_STYLES_LABEL
            }, Labels.BOLD_BUTTON_OPTIONS)
        });

        // private methods ----------------------------------------------------

        /**
         * Inserts advanced font style controls into the passed view component.
         */
        function createAdvancedControls(component, sepClasses, options) {
            component.addGroup('character/strike', new Button(docView, Utils.extendOptions(Labels.STRIKEOUT_BUTTON_OPTIONS, options)));
            component.addSeparator({ classes: sepClasses });
            component.addGroup('character/vertalign', new Controls.TextPositionGroup(docView, options));
            component.addSeparator({ classes: sepClasses });
            component.addGroup('character/reset', new Button(docView, Utils.extendOptions(Labels.CLEAR_FORMAT_BUTTON_OPTIONS, options)));
        }

        // initialization -----------------------------------------------------

        // create the generic controls of this tool bar
        this.addGroup('character/bold',      new Button(docView, Labels.BOLD_BUTTON_OPTIONS));
        this.addGroup('character/italic',    new Button(docView, Labels.ITALIC_BUTTON_OPTIONS));
        this.addGroup('character/underline', new Button(docView, Labels.UNDERLINE_BUTTON_OPTIONS));

        // create a drop-down menu with advanced font style options
        var moreButton = new CompoundButton(docView, {
            icon: 'docs-font-format',
            tooltip: Labels.MORE_FONT_STYLES_LABEL,
            dropDownVersion: { visible: false }
        });
        this.addSeparator({ classes: 'vertical-only' });
        this.addGroup('character/format', moreButton);

        // register a controller item for the drop-down menu that will be used as selector in Selenium tests
        docView.getApp().getController().registerDefinition('character/format', { parent: 'app/valid' });

        // create the advanced controls for the regular drop-down menu, and for the shrunken toolbar
        createAdvancedControls(moreButton, null);
        createAdvancedControls(this, 'hidden', { dropDownVersion: { visible: true } });

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class FontStyleToolBar

    // class FontStyleToolBar =================================================

    //#. fill color behind single characters (instead of entire paragraph/cell)
    var TEXT_HIGHLIGHT_COLOR_LABEL = gt('Text highlight color');

    /**
     * A tool bar with controls to change the font color attributes.
     *
     * The tool bar will contain the following controls:
     *  (1) A text color button (controller key 'character/color').
     *  (2) A text fill color button (controller key 'character/fillcolor').
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextBaseView} docView
     *  The document view instance containing this tool bar.
     */
    TextBaseToolBars.FontColorToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        ToolBar.call(this, docView, {
            shrinkToMenu: { icon: 'docs-color', tooltip: gt('Colors') }
        });

        // initialization -----------------------------------------------------

        // create the controls of this tool bar
        this.addGroup('character/color',     new Controls.ColorPicker(docView, null, Labels.TEXT_COLORPICKER_OPTIONS));
        this.addGroup('character/fillcolor', new Controls.FillColorPicker(docView, { icon: 'docs-font-fill-color', tooltip: TEXT_HIGHLIGHT_COLOR_LABEL, title: TEXT_HIGHLIGHT_COLOR_LABEL }), { visibleKey: 'character/fillcolor' });

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class FontColorToolBar

    // class ListStyleToolBar =================================================

    //#. indentation of lists (one list level up)
    var DEC_LEVEL_LABEL = gt('Demote one level');
    //#. indentation of lists (one list level down)
    var INC_LEVEL_LABEL = gt('Promote one level');

    /**
     * A tool bar with controls to change the settings of numbered lists, or
     * bullet lists.
     *
     * The tool bar will contain the following controls:
     *  (1) A bullet list picker (controller key 'paragraph/list/bullet').
     *  (2) A numbered list picker (controller key 'paragraph/list/numbered').
     *  (3) A decrement level button (controller key 'paragraph/list/decindent').
     *  (4) An increment level button (controller key 'paragraph/list/incindent').
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextBaseView} docView
     *  The document view instance containing this tool bar.
     */
    TextBaseToolBars.ListStyleToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        ToolBar.call(this, docView, {
            shrinkToMenu: { icon: 'docs-lists', tooltip: Labels.LIST_SETTINGS_LABEL }
        });

        // initialization -----------------------------------------------------

        // create the controls of this tool bar
        this.addGroup('paragraph/list/bullet', new Controls.BulletListStylePicker(docView));
        this.addGap();
        this.addGroup('paragraph/list/numbered', new Controls.NumberedListStylePicker(docView));
        this.addGap();
        this.addGroup('paragraph/list/decindent', new Button(docView, { icon: 'docs-list-dec-level', tooltip: DEC_LEVEL_LABEL, dropDownVersion: { label: DEC_LEVEL_LABEL } }));
        this.addGroup('paragraph/list/incindent', new Button(docView, { icon: 'docs-list-inc-level', tooltip: INC_LEVEL_LABEL, dropDownVersion: { label: INC_LEVEL_LABEL } }));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class ListStyleToolBar

    // class BoxAlignmentToolBar ==============================================

    /**
     * A tool bar with controls to change the settings of numbered lists, or
     * bullet lists.
     *
     * The tool bar will contain the following controls:
     *  (1) A picker for horizontal and vertical paragraph alignment.
     *  (2) A picker for line height and paragraph spacing.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextBaseView} docView
     *  The document view instance containing this tool bar.
     */
    TextBaseToolBars.BoxAlignmentToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        ToolBar.call(this, docView);

        // initialization -----------------------------------------------------

        // create the controls of this tool bar
        this.addGroup(null, new Controls.TextBoxAlignmentPicker(docView));
        this.addGroup(null, new Controls.ParagraphSpacingPicker(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class BoxAlignmentToolBar

    // exports ================================================================

    return TextBaseToolBars;

});
