/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/textframework/view/dialog/imagecropdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/dialog/basedialog',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/textframework/view/controls',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/utils/imageutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/drawing/imagecropframe',
    'gettext!io.ox/office/textframework/main',
    'less!io.ox/office/textframework/view/dialog/imagecropdialog'
], function (Utils, KeyCodes, Forms, Tracking, BaseDialog, EditControls, Controls, AttributeUtils, ImageUtils, Operations, Position, ImageCropFrame, gt) {

    'use strict';

    // Global constants and variables
    var imageCropTitle = /*#. Change image crop properties from dialog menu*/ gt('Crop position');
    var dialogWidth = Utils.SMALL_DEVICE ? 260 : 300;
    // global values that save last left and top position of dialog
    var savedLeftVal;
    var savedTopVal;

    // class ImageCropDialog ==================================================

    /**
     * The image crop properties dialog.
     * Provides different cropping properties.
     *
     *
     * The changes will be applied to the current document after clicking the 'Ok' button.
     *
     * The dialog itself is shown by calling this.execute. The caller can react on successful
     * finalization of the dialog within an appropriate done-Handler, chained to the
     * this.execute function call.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {AppView} docView
     *  The document view containing this dialog instance.
     * @param {ImageCropFrame} ImageCropFrame
     *  The image crop frame instance.
     */
    var ImageCropDialog = BaseDialog.extend(function (docView, ImageCropFrame) {

        var self = this;
        var docModel = docView.getDocModel();
        var isTextApp = docModel.getApp().isTextApp();
        var startOffset;
        var dialogNode = null;
        var headerNode;
        // main container node for controls
        var mainControlArea = null;
        // sub-containers for image and crop position controls
        var imagePositionControl = null, cropPositionControl = null;
        // initial properties used for comparing and detecting change
        var initialValues = [];
        // keeps collection of all spinners in dialog
        var spinnerCollection = [];

        // base constructor ---------------------------------------------------

        BaseDialog.call(this, docView, { title: imageCropTitle, width: dialogWidth });

        // private methods ----------------------------------------------------

        /**
         * Spinner wrapper.
         * @param {Number} min Value
         * @param {Number} max Value
         * @param {String} label String for the label
         * @param {String} tooltip
         */
        var Spinner = EditControls.SpinField.extend({ constructor: function (min, max, dataValue, label, tooltip) {
            var spinnerValue,
                spinnerNode,
                cleanLabel = label ? Utils.trimAndCleanString(label.toLowerCase().replace(' ', '')) : false,
                labelNode = Forms.createElementMarkup('label', { attributes: { class: 'spin-field-label', for: cleanLabel }, content: label });

            EditControls.UnitField.call(this, docView, {
                width: 70,
                tooltip: tooltip,
                min: min,
                max: max,
                smallerVersion: { hide: true }
            });

            this.setValue = function (value) {
                if (spinnerValue) { //crop frame position fields changes are related to image field values, need update
                    var diff = value - spinnerValue;
                    var attrName = this.getAttributeName();
                    var linkNum = attrName === 'cropWidth' || attrName === 'left' ? 2 : (attrName === 'cropHeight' || attrName === 'top') ? 3 : null;
                    var diffVal = attrName === 'cropWidth' || attrName === 'cropHeight' ? diff / 2 : diff;
                    if (spinnerCollection.length && linkNum) {
                        var linkedSpinner = spinnerCollection[linkNum];
                        linkedSpinner.setValue(linkedSpinner.getValue() - diffVal);
                    }
                }

                spinnerValue = value;
                this.setFieldValue(spinnerValue);
                this.getInputNode().trigger('input');

                if (spinnerValue && spinnerCollection.length) {
                    updateImagePosition();
                }
            };

            this.triggerChange = function (value, options) {
                if (!_.isNull(value)) {
                    this.setValue(value);
                    if (options.sourceEvent.keyCode === 13) {
                        self.invoke('ok');
                    }
                } else {
                    this.setValue(0);
                }
            };

            /**
             * Returns the node of the spinner
             * @returns {jQuery} the spinner node.
             */
            this.node = function () {
                return spinnerNode || (spinnerNode = $(this.getNode()));
            };

            /**
             * Returns the label for the spinner.
             * @returns {String} markup of the label.
             */
            this.label = function () {
                return labelNode;
            };

            /**
             * Returns the value of the spinner.
             * @returns {String} markup of the label.
             */
            this.getValue = function () {
                return spinnerValue;
            };

            /**
             * Returns the name of changed attribute.
             * @returns {String} name of the attribute.
             */
            this.getAttributeName = function () {
                return dataValue;
            };
        } });

        /**
         * Initializes mouse/touch tracking to move this menu around.
         */
        function trackingStartHandler(event) {
            dialogNode = self.getBody().parents('.io-ox-office-dialog');
            startOffset = dialogNode.offset();
            var parentOffset = dialogNode.parent().offset();

            startOffset.left -= parentOffset.left;
            startOffset.top -=  parentOffset.top;
            event.preventDefault();
        }

        /**
         * Moves the menu while mouse/touch tracking is active.
         */
        function trackingMoveHandler(event) {
            if (!dialogNode || !dialogNode.length) {
                Utils.error('ImageCropDialog.trackingMoveHandler(): dialog node not found!');
                return;
            }
            var appWindowNode = $(window);
            var marginBoundary = 5;
            var rightBoundary = appWindowNode.width() - dialogNode.width() - marginBoundary;
            var bottomBoundary = appWindowNode.height() - (Utils.SMALL_DEVICE ? self.getHeader().outerHeight() : dialogNode.height()) - marginBoundary;
            var leftValue = Utils.minMax(startOffset.left + event.offsetX, marginBoundary, rightBoundary);
            var topValue = Utils.minMax(startOffset.top + event.offsetY, marginBoundary, bottomBoundary);

            dialogNode.css({ left: leftValue, top: topValue, margin: 0 });
            savedLeftVal = leftValue;
            savedTopVal = topValue;
            event.preventDefault();
        }

        function getAllSpinnerValues(spinnerCollection) {
            var values = [];

            _.each(spinnerCollection, function (spinner) {
                values.push({ name: spinner.getAttributeName(), value: spinner.getValue() });
            });

            return values;
        }

        function getVal(collection, name) {
            var item = _.find(collection, function (item) {
                return item.name === name;
            });

            return item ? item.value : null;
        }

        function checkChanges(initialValues, spinnerCollection) {
            var changes = [];
            var values = getAllSpinnerValues(spinnerCollection);

            _.each(values, function (spinnerValObj, index) {
                if (spinnerValObj.value !== initialValues[index]) {
                    changes.push(spinnerValObj);
                }
            });
            return changes;
        }

        function updateImagePosition(initVals) {
            var drawing = docModel.getSelection().getSelectedDrawing();
            var drawingAttrs = AttributeUtils.getExplicitAttributes(drawing).drawing;
            var values = getAllSpinnerValues(spinnerCollection);
            if (initVals) { // setting initial values, overwrite current ones in structure
                _.each(values, function (oneVal, index) {
                    oneVal.value = initVals[index];
                });
            }

            var cropWidth = getVal(values, 'cropWidth');
            var cropHeight = getVal(values, 'cropHeight');
            var cropProps = ImageCropFrame.calculateCropOperation(cropWidth, cropHeight, getVal(values, 'imageWidth'),  getVal(values, 'imageHeight'),  getVal(values, 'offsetX'),  getVal(values, 'offsetY'));
            if (cropProps && !_.isEmpty(cropProps)) {
                var horzSettings = ImageUtils.calculateBitmapSettings(docModel.getApp(), cropWidth, cropProps.cropLeft, cropProps.cropRight);
                var vertSettings = ImageUtils.calculateBitmapSettings(docModel.getApp(), cropHeight, cropProps.cropTop, cropProps.cropBottom);

                drawing.find('img').css({
                    left: horzSettings.offset,
                    top: vertSettings.offset,
                    width: horzSettings.size,
                    height: vertSettings.size
                });
            }
            var leftVal = getVal(values, 'left');
            var topVal = getVal(values, 'top');
            if (drawingAttrs.anchorVertBase === 'paragraph' && isTextApp) {
                var pageAttrs  = _.copy(docModel.getStyleCollection('page').getElementAttributes(docModel.getNode()).page, true);
                leftVal -= pageAttrs.marginLeft;
                topVal -= pageAttrs.marginTop;
            }
            var drawingCssPros = {
                left: Utils.convertHmmToLength(leftVal, 'px'),
                top: Utils.convertHmmToLength(topVal, 'px'),
                width: Utils.convertHmmToLength(cropWidth, 'px'),
                height: Utils.convertHmmToLength(cropHeight, 'px')
            };
            drawing.css(drawingCssPros);
            var drawingSelectionNode = drawing.data('selection');
            if (drawingSelectionNode) {
                if (isTextApp) {
                    docModel.getSelection().updateOverlaySelection(drawing);
                } else {
                    drawingSelectionNode.css(drawingCssPros);
                }
            }
        }

        /**
         * Initialize controls of the dialog.
         */
        function initControls() {
            var minSizeBoundary = 0;
            var maxSizeBoundary = 30000;
            var minOffsetBoundary = -30000;
            var maxOffsetBoundary = 30000;
            var drawing = docModel.getSelection().getSelectedDrawing();
            var explAttrs = AttributeUtils.getExplicitAttributes(drawing);
            var imageAttrs = explAttrs.image;
            var drawingAttrs = explAttrs.drawing;
            var cropLeft = (imageAttrs && imageAttrs.cropLeft) || 0;
            var cropRight = (imageAttrs && imageAttrs.cropRight) || 0;
            var cropTop = (imageAttrs && imageAttrs.cropTop) || 0;
            var cropBottom = (imageAttrs && imageAttrs.cropBottom) || 0;
            var horzSettings = ImageUtils.calculateBitmapSettings(docModel.getApp(), explAttrs.drawing.width, cropLeft, cropRight);
            var vertSettings = ImageUtils.calculateBitmapSettings(docModel.getApp(), explAttrs.drawing.height, cropTop, cropBottom);
            var iniImageWidth = Utils.convertLengthToHmm(parseInt(horzSettings.size, 10), 'px');
            var iniImageHeight = Utils.convertLengthToHmm(parseInt(vertSettings.size, 10), 'px');
            var iniImageX = iniImageWidth * (((cropLeft - cropRight) / 2) / 100) * -1;
            var iniImageY = iniImageHeight * (((cropTop - cropBottom) / 2) / 100) * -1;
            var imageWidthTooltip = /*#. Tooltip message for setting image width*/ gt('Set image width');
            var imageHeightTooltip = /*#. Tooltip message for setting image height*/ gt('Set image height');
            var imageHorzOffTooltip = /*#. Tooltip message for setting horizontal offset of the image*/ gt('Set horizontal offset');
            var imageVertOffTooltip = /*#. Tooltip message for setting vertical offset of the image*/ gt('Set vertical offset');
            var cropWidthTooltip = /*#. Tooltip message for setting width of cropping frame*/ gt('Set crop position width');
            var cropHeightTooltip = /*#. Tooltip message for setting height of cropping frame*/ gt('Set crop position height');
            var cropLeftTooltip = /*#. Tooltip message for setting left position of cropping frame*/ gt('Set left position');
            var cropTopTooltip = /*#. Tooltip message for setting top position of cropping frame*/ gt('Set top position');

            // image spinner fields
            var imageHeadingNode = $('<h4 class="dialog-heading">').text(gt('Image position'));
            // -> image width
            var imageWidthSpinner = new Spinner(minSizeBoundary, maxSizeBoundary, 'imageWidth', gt('Width'), imageWidthTooltip);
            var imageWidthNode = $('<div class="ind-wrapper image-width">').append(imageWidthSpinner.label(), imageWidthSpinner.node());
            // -> image height
            var imageHeightSpinner = new Spinner(minSizeBoundary, maxSizeBoundary, 'imageHeight', gt('Height'), imageHeightTooltip);
            var imageHeightNode = $('<div class="ind-wrapper image-height">').append(imageHeightSpinner.label(), imageHeightSpinner.node());
            // -> offset x
            var offsetXSpinner = new Spinner(minOffsetBoundary, maxOffsetBoundary, 'offsetX', gt('Offset X'), imageHorzOffTooltip);
            var offsetXNode = $('<div class="ind-wrapper offset-x">').append(offsetXSpinner.label(), offsetXSpinner.node());
            // -> offset y
            var offsetYSpinner = new Spinner(minOffsetBoundary, maxOffsetBoundary, 'offsetY', gt('Offset Y'), imageVertOffTooltip);
            var offsetYNode = $('<div class="ind-wrapper offset-y">').append(offsetYSpinner.label(), offsetYSpinner.node());

            imageWidthSpinner.setValue(iniImageWidth);
            imageHeightSpinner.setValue(iniImageHeight);
            offsetXSpinner.setValue(iniImageX);
            offsetYSpinner.setValue(iniImageY);
            imagePositionControl.append(imageHeadingNode, imageWidthNode, imageHeightNode, offsetXNode, offsetYNode);

            // crop spinner fields
            var cropHeadingNode = $('<h4 class="dialog-heading">').text(gt('Crop frame position'));
            // -> crop width
            var cropWidthSpinner = new Spinner(minSizeBoundary, maxSizeBoundary, 'cropWidth', gt('Width'), cropWidthTooltip);
            var cropWidthNode = $('<div class="ind-wrapper image-width">').append(cropWidthSpinner.label(), cropWidthSpinner.node());
            // -> crop height
            var cropHeightSpinner = new Spinner(minSizeBoundary, maxSizeBoundary, 'cropHeight', gt('Height'), cropHeightTooltip);
            var cropHeightNode = $('<div class="ind-wrapper image-height">').append(cropHeightSpinner.label(), cropHeightSpinner.node());
            // -> left position
            var leftPosSpinner = new Spinner(minOffsetBoundary, maxOffsetBoundary, 'left', gt('Left'), cropLeftTooltip);
            var leftPosNode = $('<div class="ind-wrapper left-pos">').append(leftPosSpinner.label(), leftPosSpinner.node());
            // -> top position
            var topPosSpinner = new Spinner(minOffsetBoundary, maxOffsetBoundary, 'top', gt('Top'), cropTopTooltip);
            var topPosNode = $('<div class="ind-wrapper top-pos">').append(topPosSpinner.label(), topPosSpinner.node());

            cropWidthSpinner.setValue(drawingAttrs.width);
            cropHeightSpinner.setValue(drawingAttrs.height);

            var leftValue, topValue;
            if (drawingAttrs.inline) { // inline - left and top are disabled
                leftPosSpinner.enable(false).hide();
                topPosSpinner.enable(false).hide();
                leftValue = null;
                topValue = null;
            } else {
                // paragraph aligned
                if (drawingAttrs.anchorVertBase !== 'page' && isTextApp) {
                    var pageAttrs  = _.copy(docModel.getStyleCollection('page').getElementAttributes(docModel.getNode()).page, true);
                    leftValue = drawingAttrs.anchorHorOffset + pageAttrs.marginLeft;
                    topValue = drawingAttrs.anchorVertOffset + pageAttrs.marginTop;
                } else if (drawingAttrs.anchorVertBase === 'page' && isTextApp) {
                    leftValue = drawingAttrs.anchorHorOffset;
                    topValue = drawingAttrs.anchorVertOffset;
                } else { // default absolute position
                    leftValue = drawingAttrs.left;
                    topValue = drawingAttrs.top;
                }
                leftPosSpinner.setValue(leftValue);
                topPosSpinner.setValue(topValue);
            }
            cropPositionControl.append(cropHeadingNode, cropWidthNode, cropHeightNode, leftPosNode, topPosNode);

            // append all containers to control area
            mainControlArea.append(imagePositionControl, cropPositionControl);
            initialValues.push(iniImageWidth, iniImageHeight, iniImageX, iniImageY, drawingAttrs.width, drawingAttrs.height, leftValue, topValue);
            spinnerCollection.push(imageWidthSpinner, imageHeightSpinner, offsetXSpinner, offsetYSpinner, cropWidthSpinner, cropHeightSpinner, leftPosSpinner, topPosSpinner);
        }

        // initialization -----------------------------------------------------

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        mainControlArea = $('<div class="control-area">');
        imagePositionControl = $('<div class="image-pos-area">');
        cropPositionControl = $('<div class="crop-pos-area">');

        // initialize the body element of the dialog
        this.getBody()
            .addClass('io-ox-office-image-dialog-crop-position')
            .toggleClass('mobile', !!Utils.SMALL_DEVICE)
            .append(mainControlArea);

        // create the layout of the dialog
        initControls();

        this.getFooter().find('button').addClass('truncate-footer-btn');

        // handler for the OK button
        this.setOkHandler(function () {
            if (!checkChanges(initialValues, spinnerCollection).length) { return; }

            var opProperties = { start: docModel.getSelection().getStartPosition(), attrs: { image: {} } };
            var generator = docModel.createOperationGenerator();
            var zoomPercentage = docView.getZoomFactor() * 100;
            var target = docModel.getActiveTarget();
            var values = getAllSpinnerValues(spinnerCollection);
            var cropWidth = getVal(values, 'cropWidth');
            var cropHeight = getVal(values, 'cropHeight');
            var cropProps = ImageCropFrame.calculateCropOperation(cropWidth, cropHeight, getVal(values, 'imageWidth'),  getVal(values, 'imageHeight'),  getVal(values, 'offsetX'),  getVal(values, 'offsetY'));
            if (_.isEmpty(cropProps)) { return; }

            opProperties.attrs.image = cropProps;
            var left = getVal(values, 'left');
            var top = getVal(values, 'top');
            var drawing = docModel.getSelection().getSelectedDrawing();
            var drawingAttrs = AttributeUtils.getExplicitAttributes(drawing).drawing;
            var pageAligned = drawingAttrs.anchorVertBase === 'page' && isTextApp;
            var isInline = drawingAttrs.inline;

            if (cropWidth !== initialValues[4]) { opProperties.attrs.drawing = _.extend({}, opProperties.attrs.drawing, { width: cropWidth }); }
            if (cropHeight !== initialValues[5]) { opProperties.attrs.drawing = _.extend({}, opProperties.attrs.drawing, { height: cropHeight }); }

            if (isTextApp && !pageAligned  && !isInline) { // OX Text: convert different aligned drawing to page aligned
                var pos = Position.getPixelPositionOfSelectionInSelectionLayer(docModel.getRootNode(), docModel.getNode(), drawing, zoomPercentage);
                left = Utils.convertLengthToHmm(pos.x, 'px');
                top = Utils.convertLengthToHmm(pos.y, 'px');
                opProperties.attrs.drawing = _.extend({}, opProperties.attrs.drawing, { anchorHorOffset: left });
                opProperties.attrs.drawing = _.extend({}, opProperties.attrs.drawing, { anchorVertOffset: top });

                opProperties.attrs.drawing.anchorHorBase = 'page';
                opProperties.attrs.drawing.anchorVertBase = 'page';
                opProperties.attrs.drawing.anchorHorAlign = 'offset';
                opProperties.attrs.drawing.anchorVertAlign = 'offset';
                opProperties.attrs.drawing.inline = false;
            } else if (isTextApp && pageAligned) {
                if (left !== initialValues[6]) { opProperties.attrs.drawing = _.extend({}, opProperties.attrs.drawing, { anchorHorOffset: left }); }
                if (top !== initialValues[7]) { opProperties.attrs.drawing = _.extend({}, opProperties.attrs.drawing, { anchorVertOffset: top }); }
            } else {
                if (!isInline) {
                    if (left !== initialValues[6]) { opProperties.attrs.drawing = _.extend({}, opProperties.attrs.drawing, { left: left }); }
                    if (top !== initialValues[7]) { opProperties.attrs.drawing = _.extend({}, opProperties.attrs.drawing, { top: top }); }
                }
            }
            if (target) { opProperties.target = target; }

            generator.generateOperation(Operations.SET_ATTRIBUTES, opProperties);
            docModel.applyOperations(generator);
        });

        // on cancel restore original values
        this.setCancelHandler(function () {
            if (checkChanges(initialValues, spinnerCollection).length) {
                updateImagePosition(initialValues);
            }
        });

        // set initial control values
        this.on('show', function () {
            // blocking keyboard input during applying of operations
            docModel.setBlockKeyboardEvent(true);

            // use last stored position of the dialog, if available
            if (savedLeftVal && savedTopVal) {
                dialogNode = self.getBody().parents('.io-ox-office-dialog');
                dialogNode.css({ left: savedLeftVal, top: savedTopVal, margin: 0 });
            }
        });

        headerNode = this.getHeader().css('cursor', 'move');
        Tracking.enableTracking(headerNode);
        headerNode.on({ 'tracking:start': trackingStartHandler, 'tracking:move': trackingMoveHandler });

        this.registerDestructor(function () {
            docModel.setBlockKeyboardEvent(false);
            mainControlArea = headerNode = imagePositionControl = cropPositionControl = null;
            initialValues = spinnerCollection = [];
            self = docModel = null;
        });

    }); // class ImageCropDialog

    // exports ================================================================

    return ImageCropDialog;

});
