/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/utils/dom', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/tk/forms',
    'gettext!io.ox/office/textframework/main'
], function (Utils, KeyCodes, DrawingFrame, AttributeUtils, Forms, gt) {

    'use strict';

    var // key codes of keys that change the text cursor position backwards
        ARROW_CURSOR_KEYS = _([KeyCodes.UP_ARROW, KeyCodes.RIGHT_ARROW, KeyCodes.DOWN_ARROW, KeyCodes.LEFT_ARROW]),

        // key codes of keys that change the text cursor position backwards
        BACKWARD_CURSOR_KEYS = _([KeyCodes.PAGE_UP, KeyCodes.HOME, KeyCodes.LEFT_ARROW, KeyCodes.UP_ARROW]),

        // key codes of keys that change the text cursor position forwards
        FORWARD_CURSOR_KEYS = _([KeyCodes.PAGE_DOWN, KeyCodes.END, KeyCodes.RIGHT_ARROW, KeyCodes.DOWN_ARROW]),

        // key codes of keys that will be passed directly to the browser
        IGNORABLE_KEYS = _([
            KeyCodes.SHIFT, KeyCodes.CONTROL, KeyCodes.ALT,
            KeyCodes.CAPS_LOCK, KeyCodes.NUM_LOCK, KeyCodes.SCROLL_LOCK,
            KeyCodes.BREAK, KeyCodes.PRINT, KeyCodes.SELECT,
            KeyCodes.LEFT_WINDOWS, KeyCodes.RIGHT_WINDOWS,
            KeyCodes.F5,
            KeyCodes.MOZ_COMMAND
        ]);

    // static class DOM =======================================================

    /**
     * Provides classes representing DOM points (DOM.Point) and ranges
     * (DOM.Range), and static helper methods for basic editor DOM
     * manipulation, and access the browser selection.
     */
    var DOM = {};

    // class DOM.Point ========================================================

    /**
     * A DOM text point contains a 'node' attribute referring to a DOM node,
     * and an 'offset' attribute containing an integer offset specifying the
     * position in the contents of the node.
     *
     * @constructor
     *
     * @param {Node|jQuery} node
     *  The DOM node selected by this DOM.Point instance. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Number} [offset]
     *  An integer offset relative to the DOM node specifying the position in
     *  the node's contents. If the node is a text node, the offset represents
     *  the character position in the node text. If the node is an element
     *  node, the offset specifies the index of a child node of this node. The
     *  value of the offset may be equal to the text length respectively the
     *  number of child nodes, in this case the DOM point refers to the
     *  position directly after the node's contents. If omitted, this DOM.Point
     *  instance refers to the start of the entire node, instead of specific
     *  contents.
     */
    DOM.Point = function (node, offset) {
        this.node = Utils.getDomNode(node);
        this.offset = offset;
    };

    // methods ----------------------------------------------------------------

    /**
     * Returns a new clone of this DOM point.
     */
    DOM.Point.prototype.clone = function () {
        return new DOM.Point(this.node, this.offset);
    };

    /**
     * Validates the offset of this DOM point. Restricts the offset to the
     * available index range according to the node's contents, or
     * initializes the offset, if it is missing.
     *
     * If this instance points to a text node, the offset will be
     * restricted to the text in the node, or set to zero if missing.
     *
     * If this instance points to an element node, the offset will be
     * restricted to the number of child nodes in the node. If the offset
     * is missing, it will be set to the index of the node in its siblings,
     * and the node will be replaced by its parent node.
     *
     * @returns {DOM.Point}
     *  A reference to this instance.
     */
    DOM.Point.prototype.validate = function () {

        // element: if offset is missing, take own index and refer to the parent node
        if (this.node.nodeType === 1) {
            if (_.isNumber(this.offset)) {
                this.offset = Utils.minMax(this.offset, 0, this.node.childNodes.length);
            } else {
                this.offset = $(this.node).index();
                this.node = this.node.parentNode;
            }

        // text node: if offset is missing, use zero
        } else if (this.node.nodeType === 3) {
            if (_.isNumber(this.offset)) {
                this.offset = Utils.minMax(this.offset, 0, this.node.nodeValue.length);
            } else {
                this.offset = 0;
            }
        }

        return this;
    };

    /**
     * Converts this DOM point to a human readable string representation.
     */
    DOM.Point.prototype.toString = function () {

        // generates a readable description of the passed node and offset
        function getNodeName(node, offset) {

            var // full string representation of this DOM Point
                result = node.nodeName.toLowerCase();

            if ((node.nodeType === 1) && (node.className.length > 0)) {
                // add class names of an element
                result += '.' + node.className.replace(/ /g, '.');
            } else if (node.nodeType === 3) {
                // add some text of a text node
                result += '"' + node.nodeValue.substr(0, 10) + ((node.nodeValue.length > 10) ? Utils.ELLIPSIS_CHAR : '') + '"';
            }

            if (_.isNumber(offset)) {
                result += ':' + offset;
            }

            return result;
        }

        if (this.node && this.node.parentNode) {
            return getNodeName(this.node.parentNode, $(this.node).index()) + '>' + getNodeName(this.node, this.offset);
        } else {
            return 'undefined node';
        }
    };

    // static methods ---------------------------------------------------------

    /**
     * Creates and returns a valid DOM.Point instance for the passed DOM node.
     * If the passed node is a text node, the DOM point will refer to its first
     * character, otherwise the DOM point will contain the parent node and the
     * child index of the passed node as offset.
     *
     * @param {Node|jQuery} node
     *  The DOM node selected by the created DOM.Point instance. If this object
     *  is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {DOM.Point}
     *  A new DOM.Point instance referring to the passed node.
     */
    DOM.Point.createPointForNode = function (node) {
        return new DOM.Point(node).validate();
    };

    /**
     * Returns whether the two passed DOM points are equal.
     *
     * @param {DOM.Point} point1
     *  The first DOM point. Must be valid (see DOM.Point.validate() method for
     *  details).
     *
     * @param {DOM.Point} point2
     *  The second DOM point. Must be valid (see DOM.Point.validate() method
     *  for details).
     *
     * @returns {Boolean}
     *  Whether the DOM points are equal.
     */
    DOM.Point.equalPoints = function (point1, point2) {
        return (point1.node === point2.node) && (point1.offset === point2.offset);
    };

    /**
     * Returns an integer indicating how the two DOM points are located to each
     * other.
     *
     * @param {DOM.Point} point1
     *  The first DOM point. Must be valid (see DOM.Point.validate() method for
     *  details).
     *
     * @param {DOM.Point} point2
     *  The second DOM point. Must be valid (see DOM.Point.validate() method
     *  for details).
     *
     * @returns {Number}
     *  The value zero, if the DOM points are equal, a negative number, if
     *  point1 precedes point2, or a positive number, if point1 follows point2.
     */
    DOM.Point.comparePoints = function (point1, point2) {

        // Returns the index of the inner node's ancestor in the outer node's
        // children list. 'outerNode' MUST contain 'innerNode'.
        function calculateOffsetInOuterNode(outerNode, innerNode) {
            while (innerNode.parentNode !== outerNode) {
                innerNode = innerNode.parentNode;
            }
            return $(innerNode).index();
        }

        // equal nodes: compare by offset
        if (point1.node === point2.node) {
            return point1.offset - point2.offset;
        }

        // Node in point1 contains the node in point2: point1 is before point2,
        // if offset of point1 (index of its child node) is less than or equal
        // to the offset of point2's ancestor node in the children of point1's
        // node. If offsets are equal, point2 is a descendant of the child node
        // pointed to by point1 and therefore located after point1.
        if (Utils.containsNode(point1.node, point2.node)) {
            return (point1.offset <= calculateOffsetInOuterNode(point1.node, point2.node)) ? -1 : 1;
        }

        // Node in point2 contains the node in point1: see above, reversed.
        if (Utils.containsNode(point2.node, point1.node)) {
            return (calculateOffsetInOuterNode(point2.node, point1.node) < point2.offset) ? -1 : 1;
        }

        // Neither node contains the other: compare nodes regardless of offset.
        return Utils.compareNodes(point1.node, point2.node);
    };

    // class DOM.Range ========================================================

    /**
     * A DOM text range represents a half-open range in the DOM tree. It
     * contains 'start' and 'end' attributes referring to DOM point objects.
     *
     * @constructor
     *
     * @param {DOM.Point} start
     *  The DOM point where the range starts.
     *
     * @param {DOM.Point} [end]
     *  The DOM point where the range ends. If omitted, uses the start position
     *  to construct a collapsed range (a simple 'cursor').
     */
    DOM.Range = function (start, end) {
        this.start = start;
        this.end = _.isObject(end) ? end : _.clone(start);
    };

    // methods ----------------------------------------------------------------

    /**
     * Returns a new clone of this DOM range.
     */
    DOM.Range.prototype.clone = function () {
        return new DOM.Range(this.start.clone(), this.end.clone());
    };

    /**
     * Validates the start and end position of this DOM range. See method
     * DOM.Point.validate() for details.
     */
    DOM.Range.prototype.validate = function () {
        this.start.validate();
        this.end.validate();
        return this;
    };

    /**
     * Swaps start and end position, if the start position is located after
     * the end position in the DOM tree.
     */
    DOM.Range.prototype.adjust = function () {
        if (DOM.Point.comparePoints(this.start, this.end) > 0) {
            var tmp = this.start;
            this.start = this.end;
            this.end = tmp;
        }
        return this;
    };

    /**
     * Returns whether the two passed DOM ranges are equal.
     *
     * @param {DOM.Range} range1
     *  The first DOM range. Must be valid (see DOM.Range.validate() method for
     *  details).
     *
     * @param {DOM.Range} range2
     *  The second DOM range. Must be valid (see DOM.Range.validate() method
     *  for details).
     *
     * @returns {Boolean}
     *  Whether the DOM ranges are equal.
     */
    DOM.Range.equalRanges = function (range1, range2) {
        return (DOM.Point.equalPoints(range1.start, range2.start) && DOM.Point.equalPoints(range1.end, range2.end));
    };

    /**
     * Returns whether the two passed DOM ranges have the same offsets (the nodes might be different).
     * This can be used for column selections, where the row might be different, but the offset are
     * constant inside a column.
     *
     * @param {DOM.Range} range1
     *  The first DOM range. Must be valid (see DOM.Range.validate() method for
     *  details).
     *
     * @param {DOM.Range} range2
     *  The second DOM range. Must be valid (see DOM.Range.validate() method
     *  for details).
     *
     * @returns {Boolean}
     *  Whether the DOM ranges have the same offsets.
     */
    DOM.Range.equalRangeOffsets = function (range1, range2) {
        return (range1.start.offset === range2.start.offset) && (range1.end.offset === range2.end.offset);
    };

    /**
     * Returns whether the DOM range is collapsed, i.e. start position and
     * end position are equal.
     *
     * @returns {Boolean}
     *  Whether this DOM range is collapsed.
     */
    DOM.Range.prototype.isCollapsed = function () {
        return DOM.Point.equalPoints(this.start, this.end);
    };

    /**
     * Converts this DOM range to a human readable string representation.
     */
    DOM.Range.prototype.toString = function () {
        return '[start=' + this.start + ', end=' + this.end + ']';
    };

    // static methods ---------------------------------------------------------

    /**
     * Creates a new DOM.Range instance from the passed nodes and offsets.
     *
     * @param {Node|jQuery} startNode
     *  The DOM node used for the start point of the created range. If this
     *  object is a jQuery collection, uses the first DOM node it contains.
     *
     * @param {Number} [startOffset]
     *  The offset for the start point of the created range.
     *
     * @param {Node|jQuery} [endNode]
     *  The DOM node used for the end point of the created range. If this
     *  object is a jQuery collection, uses the first DOM node it contains. If
     *  omitted, creates a collapsed range by cloning the start position.
     *
     * @param {Number} [endOffset]
     *  The offset for the end point of the created range. Not used, if endNode
     *  has been omitted.
     *
     * @returns {DOM.Range}
     *  The new DOM range object.
     */
    DOM.Range.createRange = function (startNode, startOffset, endNode, endOffset) {
        return new DOM.Range(new DOM.Point(startNode, startOffset), _.isObject(endNode) ? new DOM.Point(endNode, endOffset) : undefined);
    };

    /**
     * Creates and returns a valid DOM.Range instance for the passed DOM node.
     * If the passed node is a text node, the DOM range will select its entire
     * text, otherwise the DOM range will contain the parent node and the
     * child index of the passed node as start offset, and the next child index
     * as end offset, effectively selecting the entire node.
     *
     * @param {Node|jQuery} node
     *  The DOM node selected by the created DOM.Range instance. If this object
     *  is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {DOM.Range}
     *  A new DOM.Range instance referring to the passed node.
     */
    DOM.Range.createRangeForNode = function (node) {
        var range = new DOM.Range(DOM.Point.createPointForNode(node));
        if (range.end.node.nodeType === 1) {
            range.end.offset += 1;
        } else if (range.end.node.nodeType === 3) {
            range.end.offset = range.end.node.nodeValue.length;
        }
        return range;
    };

    // key codes ==============================================================

    /**
     * Returns whether the passed key code is a cursor navigation key that
     * moves the cursor backwards in the document.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isBackwardCursorKey = function (keyCode) {
        return BACKWARD_CURSOR_KEYS.contains(keyCode);
    };

    /**
     * Returns whether the passed key code is a cursor navigation key that
     * moves the cursor forwards in the document.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isForwardCursorKey = function (keyCode) {
        return FORWARD_CURSOR_KEYS.contains(keyCode);
    };

    /**
     * Returns whether the passed key code is a cursor navigation key.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isCursorKey = function (keyCode) {
        return DOM.isBackwardCursorKey(keyCode) || DOM.isForwardCursorKey(keyCode);
    };

    /**
     * Returns whether the passed key code is an 'arrow' cursor navigation key.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isArrowCursorKey = function (keyCode) {
        return ARROW_CURSOR_KEYS.contains(keyCode);
    };

    /**
     * Returns whether the passed key code has to be ignored silently.
     *
     * @param {Number} keyCode
     *  The key code from a 'keydown' browser event.
     */
    DOM.isIgnorableKey = function (keyCode) {
        return IGNORABLE_KEYS.contains(keyCode);
    };

    // pages ==================================================================

    /**
     * A jQuery selector that matches elements representing a page.
     */
    DOM.PAGE_NODE_SELECTOR = 'div.page';

    /**
     * Creates a new page element.
     *
     * @returns {jQuery}
     *  A page element, as jQuery object.
     */
    DOM.createPageNode = function () {
        return $('<div class="page formatted-content"><div class="pagecontent"></div></div>');
    };

    /**
     * Returns whether the passed node is a page element.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is a page element.
     */
    DOM.isPageNode = function (node) {
        return $(node).is(DOM.PAGE_NODE_SELECTOR);
    };

    /**
     * The CSS class used to mark the application content root node.
     *
     * @constant
     */
    DOM.APP_CONTENT_ROOT = 'app-content-root';

    /**
     * A jQuery selector that matches nodes representing the application content root.
     *
     * @constant
     */
    DOM.APP_CONTENT_ROOT_SELECTOR = '.' + DOM.APP_CONTENT_ROOT;

    /**
     * The class name for the page content node.
     */
    DOM.PAGECONTENT_NODE_CLASSNAME = 'pagecontent';

    /**
     * A jQuery selector that matches elements representing a page content node.
     */
    DOM.PAGECONTENT_NODE_SELECTOR = 'div.pagecontent';

    /**
     * The class name for the comment layer node.
     */
    DOM.COMMENTLAYER_CLASS = 'commentlayer';

    /**
     * The class name for the comment bubble layer node.
     */
    DOM.COMMENTBUBBLELAYER_CLASS = 'commentbubblelayer';

    /**
     * A jQuery selector that matches elements representing a comment layer node.
     */
    DOM.COMMENTLAYER_NODE_SELECTOR = 'div.' + DOM.COMMENTLAYER_CLASS;

    /**
     * A jQuery selector that matches elements representing a comment bubble layer node.
     */
    DOM.COMMENTBUBBLELAYER_NODE_SELECTOR = 'div.' + DOM.COMMENTBUBBLELAYER_CLASS;

    /**
     * The class name for marking paragraphs that must not be spellchecked, for
     * example template texts in empty text frames.
     */
    DOM.NO_SPELLCHECK_CLASSNAME = 'nospellcheck';

    /**
     * The attribute name for the image source property that is saved at the drawing
     * node during fast load or in local storage usage.
     */
    DOM.IMAGESOURCE_ATTR = '__imagesource__';

    /**
     * The class name for hiding nodes directly after attaching the html string
     * (fast load or local storage) to the DOM (performance).
     */
    DOM.HIDDENAFTERATTACH_CLASSNAME = 'hiddenafterattach';

    /**
     * Returns whether the passed node is a page content node that contains all
     * top-level content nodes (paragraphs and tables).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a page content node.
     */
    DOM.isPageContentNode = function (node) {
        return $(node).hasClass(DOM.PAGECONTENT_NODE_CLASSNAME);
    };

    /**
     * Returns the container node of a page element that contains all top-level
     * content nodes (paragraphs and tables).
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed page that contains all top-level
     *  content nodes (paragraphs and tables).
     */
    DOM.getPageContentNode = function (pageNode) {
        return $(pageNode).children(DOM.PAGECONTENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is the application content root node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is the application content root node.
     */
    DOM.isAppContentRootNode = function (node) {
        return $(node).is(DOM.APP_CONTENT_ROOT_SELECTOR);
    };

    /**
     * Returns collection of first header wrapper, pagecontent node, and last footer wrapper nodes.
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  Collection of first header wrapper, pagecontent node, and last footer wrapper nodes.
     */
    DOM.getContentChildrenOfPage = function (pageNode) {
        return $(pageNode).children(DOM.PAGE_CHILDREN_CONTENT_SELECTOR);
    };

    /**
     * Checking, if a specified node is a direct child of the page content node.
     *
     * @param {HTMLElement|jQuery} node
     *  The node that needs to be checked.
     *
     * @returns {Boolean}
     *  Whether a specified node is a direct child of the page content node.
     */
    DOM.isChildOfPageContentNode = function (node) {
        return DOM.isPageContentNode($(node).parent());
    };

    /**
     * Checking, if a specified node is a direct child of a marginal content node.
     *
     * @param {HTMLElement|jQuery} node
     *  The node that needs to be checked.
     *
     * @returns {Boolean}
     *  Whether a specified node is a direct child of a marginal content node.
     */
    DOM.isChildOfMarginalContentNode = function (node) {
        return DOM.isMarginalContentNode($(node).parent());
    };

    /**
     * Returns the container node of comments.
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node for all comments.
     */
    DOM.getCommentLayerNode = function (pageNode) {
        return $(pageNode).children(DOM.COMMENTLAYER_NODE_SELECTOR);
    };

    /**
     * Returns the container node of the bubble comments.
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node for all comment bubbles.
     */
    DOM.getCommentBubbleLayerNode = function (pageNode) {
        return $(pageNode).children(DOM.COMMENTBUBBLELAYER_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a comment layer node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a comment layer node.
     */
    DOM.isCommentLayerNode = function (node) {
        return $(node).hasClass(DOM.COMMENTLAYER_CLASS);
    };

    /**
     * Returning whether a specified node is inside a comment.
     *
     * @param {Node|jQuery} rootNode
     *  The root node until which the search is executed.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Booelan}
     *  Whether the specified node is located inside a comment.
     */
    DOM.isInsideCommentNode = function (rootNode, node) {
        return $(node).parentsUntil(rootNode, DOM.COMMENTNODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is within the comment layer node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is within the comment layer node.
     */
    DOM.isInsideCommentLayerNode = function (rootNode, node) {
        return $(node).parentsUntil(rootNode, DOM.COMMENTLAYER_NODE_SELECTOR).length > 0;
    };

    /**
     * The class name representing an application content node.
     */
    DOM.APPCONTENT_NODE_CLASS = 'app-content';

    /**
     * A jQuery selector that matches elements representing an application content node.
     */
    DOM.APPCONTENT_NODE_SELECTOR = 'div.app-content';

    /**
     * Returns whether the passed node is an application content node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an application content node.
     */
    DOM.isAppContentNode = function (node) {
        return $(node).hasClass(DOM.APPCONTENT_NODE_CLASS);
    };

    /**
     * The class name representing a page break node.
     */
    DOM.PAGEBREAK_NODE_CLASS = 'page-break';

    /**
     * A jQuery selector that matches elements representing a page break node.
     */
    DOM.PAGEBREAK_NODE_SELECTOR = 'div.page-break';

    /**
     * Returns whether the passed node is a page break node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a page break node.
     */
    DOM.isPageBreakNode = function (node) {
        return $(node).hasClass(DOM.PAGEBREAK_NODE_CLASS);
    };

    /**
     * Returns whether the passed node contains at least one page break node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains a page break node.
     */
    DOM.hasPageBreakNode = function (node) {
        return $(node).has(DOM.PAGEBREAK_NODE_SELECTOR).length > 0;
    };

    // paragraphs and tables ==================================================

    /**
     * The class name representing a paragraph.
     */
    DOM.PARAGRAPH_NODE_CLASS_NAME = 'p';

    /**
     * A jQuery selector that matches elements representing a paragraph.
     */
    DOM.PARAGRAPH_NODE_SELECTOR = 'div.p';

    /**
     * A class assigned to an empty paragraph in a list (presentation only).
     */
    DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS = 'emptylistparagraph';

    /**
     * A class assigned to an empty paragraph in a list, that contains the selection (presentation only).
     */
    DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS = 'emptyselectedlistparagraph';

    /**
     * A class assigned to first paragraphs in lists (text only).
     */
    DOM.PARAGRAPH_NODE_IS_FIRST_IN_LIST = 'isfirstinlist';

    /**
     * A class assigned to list paragraphs (text only).
     */
    DOM.LIST_PARAGRAPH_MARKER = 'listparagraph';

    /**
     * The name of the attribute, in which the list style ID is saved.
     */
    DOM.LIST_PARAGRAPH_ATTRIBUTE = 'data-list-id';

    /**
     * A jQuery selector that matches elements representing a border for lists.
     * The '.drawing' handles also comments.
     */
    DOM.LIST_BORDER_NODE_SELECTOR = '.page, .drawing, .header, .footer';

    /**
     * Returns whether the passed node is a paragraph element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph element.
     */
    DOM.isParagraphNode = function (node) {
        return $(node).hasClass(DOM.PARAGRAPH_NODE_CLASS_NAME);
    };

    /**
     * Returns whether the passed node is part of a paragraph element.
     * Can be text span, text span container node, drawing frame, comment placeholder, range marker, complex field, hard break node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a part of paragraph element.
     */
    DOM.isPartOfParagraph = function (node) {
        return DOM.isTextSpan(node) || DOM.isTextSpanContainerNode(node) || DOM.isHardBreakNode(node) || DOM.isComplexFieldNode(node) || DrawingFrame.isDrawingFrame(node) || DOM.isCommentPlaceHolderNode(node) || DOM.isRangeMarkerNode(node) || DOM.isBookmarkNode(node);
    };

    /**
     * Returns whether the passed node a list paragraph element. For this it is checked, if the first
     * element inside the paragraph is a list label node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a list paragraph element.
     */
    DOM.isListParagraphNode = function (node) {
        return DOM.isParagraphNode(node) && $(node).children(DOM.LIST_LABEL_NODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is a list paragraph element that is the first paragraph
     * in its list and in its level. For every list level, one paragraph has the marker class.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a list paragraph element, that is the first paragraph in its
     *  list.
     */
    DOM.isFirstParagraphInList = function (node) {
        return DOM.isListParagraphNode(node) && $(node).hasClass(DOM.PARAGRAPH_NODE_IS_FIRST_IN_LIST);
    };

    /**
     * Returns whether the passed node is an empty paragraph element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an empty paragraph element. It is empty, if
     *  it only contains an empty text span and a 'br' element.
     */
    DOM.isEmptyParagraph = function (node) {
        var allChildren = $(node).children();
        return (allChildren.length === 2 && DOM.isEmptySpan(allChildren.first()) && allChildren.last().is('br'));
    };

    /**
     * Returns whether the passed node is an empty paragraph with list label node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an empty list paragraph element.
     */
    DOM.isEmptyListParagraph = function (node) {

        var // a collector for the text span children inside the paragraph
            allTextSpans = null;

        if (DOM.isListParagraphNode(node)) {
            allTextSpans = $(node).children('span');
            return allTextSpans.length === 1 && allTextSpans.text() === '';
        }

        return false;
    };

    /**
     * Returns whether the passed node is an empty cell element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an empty cell element.
     */
    DOM.isEmptyCell = function (node) {
        return ($(node).children().length === 1 && DOM.isEmptyParagraph($(node).children().first()));
    };

    /**
     * Creates a new paragraph element.
     *
     * @returns {jQuery}
     *  A paragraph element, as jQuery object.
     */
    DOM.createParagraphNode = function () {
        return $('<div class="p">');
    };

    /**
     * Creates a new paragraph marked as 'implicit'. This is used in
     * empty documents or table cells to allow text insertion in browser.
     *
     * @returns {jQuery}
     *  An implicit paragraph node, as jQuery object.
     */
    DOM.createImplicitParagraphNode = function () {
        return DOM.createParagraphNode().data('implicit', true);
    };

    /**
     * Flag a paragraph node with data flag 'implicit'. This paragraphs are
     * used in empty documents or table cells to allow text insertion in browser.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be flagged.
     */
    DOM.makeParagraphNodeImplicit = function (node) {
        $(node).data('implicit', true);
    };

    /**
     * Returns whether the passed paragraph node is an implicit paragraph
     * node that is used to allow text insertion into document (into empty
     * document or cells). This paragraph nodes were created without sending
     * an operation to the server.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an implicit paragraph node.
     */
    DOM.isImplicitParagraphNode = function (node) {
        return $(node).data('implicit') === true;
    };

    /**
     * Returns whether the passed paragraph node is a paragraph without
     * neighbours (other paragraphs or tables).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph without neighbours.
     */
    DOM.isParagraphWithoutNeighbour = function (node) {
        return $(node).parent().children(DOM.CONTENT_NODE_SELECTOR).length === 1;
    };

    /**
     * Returns whether the passed paragraph node is a paragraph that is a last
     * paragraph directly behind a table.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph following a table and with no
     *  following sibling.
     */
    DOM.isFinalParagraphBehindTable = function (node) {
        return ((DOM.isTableNode($(node).prev())) && ($(node).nextAll(DOM.CONTENT_NODE_SELECTOR).length === 0));
    };

    /**
     * Returns whether the passed paragraph node is a paragraph that is a last
     * paragraph directly behind a table.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph following a table and with no
     *  following sibling.
     */
    DOM.isFinalParagraphBehindTableInCell = function (node) {
        return DOM.isCellContentNode($(node).parent()) && DOM.isFinalParagraphBehindTable(node);
    };

    /**
     * Returns whether the passed paragraph node is a paragraph that is the first
     * content node (paragraph or table) inside a table cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph that is the first content node
     *  (paragraph or table) inside a table cell.
     */
    DOM.isFirstParagraphInTableInCell = function (node) {
        return DOM.isCellContentNode($(node).parent()) && $(node).prevAll(DOM.CONTENT_NODE_SELECTOR).length === 0;
    };

    /**
     * Returns whether the passed paragraph node is a paragraph that is a last
     * paragraph directly behind a table and that is empty. This paragraph contains
     * only an empty text span and a 'br' element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an empty paragraph following a table and with no
     *  following sibling.
     */
    DOM.isFinalEmptyParagraphBehindTable = function (node) {
        return DOM.isFinalParagraphBehindTable(node) && DOM.isEmptyParagraph(node);
    };

    /**
     * Returns whether the passed paragraph node is an empty paragraph that is a last
     * paragraph directly behind a table and that is empty and that is positioned inside
     * a table cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an empty paragraph following a table and with no
     *  following sibling. Furthermore the paragraph is positioned inside a table cell.
     */
    DOM.isFinalEmptyParagraphBehindTableInCell = function (node) {
        return DOM.isCellContentNode($(node).parent()) && DOM.isFinalEmptyParagraphBehindTable(node);
    };

    /**
     * Returns whether the passed paragraph node is a paragraph inside a table cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph inside a table cell.
     */
    DOM.isParagraphInTableCell = function (node) {
        return DOM.isCellContentNode($(node).parent());
    };

    /**
     * Returns whether the height of the passed paragraph node can be increased and decreased
     * dynamically. This can happen for implicit paragraph nodes that are behind tables or for
     * non-implicit paragraph nodes, that are empty, behind a table, with no following sibling
     * and inside a table cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph, whose height need to be set dynamically.
     */
    DOM.isIncreasableParagraph = function (node) {
        return ((DOM.isImplicitParagraphNode(node)) && (DOM.isFinalParagraphBehindTable(node))) || DOM.isFinalEmptyParagraphBehindTableInCell(node);
    };

    /**
     * Returns whether the passed paragraph node is a paragraph that can be merged
     * with a following paragraph.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a paragraph that is followed by a further paragraph.
     */
    DOM.isMergeableParagraph = function (node) {
        if ($(node).next().hasClass('page-break')) {
            return (($(node).next().next().length !== 0) && (DOM.isParagraphNode($(node).next().next())));
        } else {
            return (($(node).next().length !== 0) && (DOM.isParagraphNode($(node).next())));
        }
    };

    /**
     * Collecting all nodes below a specified root node that fullfill a specified
     * selector.
     * Additionally the selection can optionally be reduced by a reference node. It
     * is possible to reduce the selection to the nodes before or after the reference
     * node.
     *
     * @param {Node|jQuery} rootNode
     *  The root node, below that the descendant nodes are searched.
     *
     * @param {String} selector
     *  A selector string to filter the descendant nodes.
     *
     * @param {Node|jQuery} [referenceNode]
     *  An optional reference node that can be used to reduce the found node collection
     *  to the range before or after this reference node.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.followingNodes=true]
     *      Optional parameter in conjunction with a specified reference node. If set to
     *      true, the selection is reduced to nodes following the reference node. Otherwise
     *      only nodes before the reference node will become part of the selection.
     *  @param {Boolean} [options.includeReference=true]
     *      Optional parameter in conjunction with a specified reference node. If set to
     *      true, the selection will include the reference node itself. Otherwise it will
     *      not be part of the selection.
     *
     * @returns {jQuery}
     *  A selection that contain all descendants relative to the specified root node,
     *  the selector and optionally a reference node. If not descendant was found, the
     *  jQuery list will be empty.
     */
    DOM.getNodeSelection = function (rootNode, selector, referenceNode, options) {

        // the jQuery object with all nodes with valid selector
        var allNodes = $(rootNode).find(selector);
        // the index of the referenceNode in the jQuery selection
        var index = 0;
        // whether the reference node shall be included into the searched selection
        var includeReference = true;

        if (referenceNode) {

            index = allNodes.index(referenceNode);

            if (index > -1) {

                includeReference = Utils.getBooleanOption(options, 'includeReference', true);

                if (Utils.getBooleanOption(options, 'followingNodes', true)) {
                    if (!includeReference) { index++; }
                    allNodes = allNodes.slice(index);
                } else {
                    if (includeReference) { index++; }
                    allNodes = allNodes.slice(0, index);
                }
            }
        }

        return allNodes;
    };

    /**
     * A jQuery selector that matches elements representing a table.
     */
    DOM.TABLE_NODE_SELECTOR = 'table';

    /**
     * A jQuery selector that matches elements representing a table with exceeded size.
     */
    DOM.TABLE_SIZE_EXCEEDED_NODE_SELECTOR = 'table.size-exceeded';

    /**
     * A jQuery selector that matches placeholder div elements included into an exceeded size table.
     */
    DOM.PLACEHOLDER_SELECTOR = 'div.placeholder';

    /**
     * A jQuery selector that matches elements representing a table. This can be a table node itself
     * or a drawing of type 'table'.
     */
    DOM.TABLE_NODE_AND_DRAWING_SELECTOR = DOM.TABLE_NODE_SELECTOR + ', ' + DrawingFrame.TABLE_TEXTFRAME_NODE_SELECTOR;

    /**
     * Returns whether the passed node is a table element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table element.
     */
    DOM.isTableNode = function (node) {
        return $(node).is(DOM.TABLE_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table element that is located inside
     * a drawing of type table (in this case it is not counted for the logical
     * position).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table element that is located inside
     *  a drawing of type table.
     */
    DOM.isTableNodeInTableDrawing = function (node) {
        return DOM.isTableNode(node) && $(node).hasClass(DrawingFrame.TABLE_NODE_IN_TABLE_DRAWING_CLASS);
    };

    /**
     * Returns whether the passed table node is a replacement table for
     * non-editable oversized tables. This table node is created without
     * further operations and takes care of its presentation.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a replacement table for oversized tables.
     */
    DOM.isExceededSizeTableNode = function (node) {
        return DOM.isTableNode(node) && $(node).hasClass('size-exceeded');
    };

    /**
     * Makes a exceeded size table placeholder from a provided empty table node.
     *
     * @param {Node|jQuery|Null} [tableNode]
     */
    DOM.makeExceededSizeTable = function (tableNode, overflowElement, value, maxValue) {
        var placeHolder,
            cell = $('<td>').append(
                placeHolder = $('<div>').addClass('placeholder').append(
                    $('<div>').addClass('abs background-icon').append(Forms.createIconMarkup('fa-table')),
                    $('<p>').text(gt('Table')),
                    $('<p>').text(gt('This table is too large to be displayed here.'))
                )),
            labels = DOM.getTableSizeExceededTextLabels(overflowElement, value, maxValue);

        if (labels.countLabel && labels.limitLabel) {
            placeHolder.append($('<p>').text(labels.countLabel + ' ' + labels.limitLabel));
        }

        $(tableNode).addClass('size-exceeded')
            .attr('contenteditable', false)
            .append($('<tr>').append(cell));
    };

    /**
    * Provides the correct localized labels for a table exceeds size case.
    *
    * @param {String} [type]
    *  Specifies which part of a table exceeded the size. Currently the parts
    *  "rows", "cols", "cells".
    *
    * @param {Integer} [value]
    *  Specifies the current value of the table part which exceeds the limit.
    *
    * @param {Integer} [maxValue]
    *  Specifies the maximal allowed value of the table part which exceeds the
    *  limit.
    */
    DOM.getTableSizeExceededTextLabels = function (type, value, maxValue) {

        var result = { countLabel: null, limitLabel: null };

        switch (type) {
            case 'rows':
                //#. %1$d is the number of rows in an oversized text table
                //#, c-format
                result.countLabel = gt.ngettext('The table contains %1$d row.', 'This table contains %1$d rows.', value, _.noI18n(value));
                //#. %1$d is the maximum allowed number of rows in a text table
                //#, c-format
                result.limitLabel = gt.ngettext('Tables are limited to %1$d row.', 'Tables are limited to %1$d rows.', maxValue, _.noI18n(maxValue));
                break;
            case 'cols':
                //#. %1$d is the number of columns in an oversized text table
                //#, c-format
                result.countLabel = gt.ngettext('The table contains %1$d column.', 'This table contains %1$d columns.', value, _.noI18n(value));
                //#. %1$d is the maximum allowed number of columns in a text table
                //#, c-format
                result.limitLabel = gt.ngettext('Tables are limited to %1$d column.', 'Tables are limited to %1$d columns.', maxValue, _.noI18n(maxValue));
                break;
            case 'cells':
                //#. %1$d is the number of cells in an oversized text table
                //#, c-format
                result.countLabel = gt.ngettext('The table contains %1$d cell.', 'This table contains %1$d cells.', value, _.noI18n(value));
                //#. %1$d is the maximum allowed number of cells in a text table
                //#, c-format
                result.limitLabel = gt.ngettext('Tables are limited to %1$d cell.', 'Tables are limited to %1$d cells.', maxValue, _.noI18n(maxValue));
        }

        return result;
    };

    /**
     * Returns whether the passed node is a table node containing the splitting
     * class.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a replacement table for oversized tables.
     */
    DOM.isSplittedTableNode = function (node) {
        return DOM.isTableNode(node) && $(node).hasClass('tb-split-nb');
    };

    /**
     * Returns whether the passed node is a div node with class 'placeholder' inside a table with
     * an exceeded size.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is an div node with class 'placeholder' inside a replacement table
     *  for oversized tables.
     */
    DOM.isPlaceholderNode = function (node) {
        return $(node).is(DOM.PLACEHOLDER_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table node that is located in
     * another table.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node inside another table.
     */
    DOM.isTableInTableNode = function (node) {
        return DOM.isTableNode(node) && DOM.isCellContentNode($(node).parent());
    };

    /**
     * Returns whether the passed node is a table node that is located in
     * another node (table, text frame, ...) so that it is NOT a top level
     * node inside the page content node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node located inside another node
     *  that is not the page content.
     */
    DOM.isChildTableNode = function (node) {
        return DOM.isTableNode(node) && !$(node).parent().is(DOM.PAGECONTENT_NODE_SELECTOR);
    };

    /**
     * A jQuery selector that matches elements representing a table row.
     */
    DOM.TABLE_ROWNODE_SELECTOR = 'tr';

    /**
     * Returns whether the passed node is a table row element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table row element.
     */
    DOM.isTableRowNode = function (node) {
        return $(node).is(DOM.TABLE_ROWNODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table row element inside a first
     * level table element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table row element in a first level table
     *  element.
     */
    DOM.isTopLevelRowNode = function (node) {
        return DOM.isTableRowNode(node) && $(node).parents(DOM.TABLE_ROWNODE_SELECTOR).length < 1;
    };

    /**
     * A class name that matches elements representing a table page break row.
     */
    DOM.TABLE_PAGEBREAK_ROWNODE_CLASSNAME = 'pb-row';

    /**
     * A jQuery selector that matches elements representing a table page break row.
     */
    DOM.TABLE_PAGEBREAK_ROWNODE_SELECTOR = '.pb-row';

    /**
     * A class name that matches elements representing a repeated, read-only table row.
     */
    DOM.TABLE_REPEATED_ROWNODE_CLASSNAME = 'repeated-row';

    /**
     * A jQuery selector that matches elements representing a repeated, read-only table row.
     */
    DOM.TABLE_REPEATED_ROWNODE_SELECTOR = '.repeated-row';

    /**
     * A class name that matches elements representing a repeated, read-only table row.
     */
    DOM.TABLE_REPEAT_ROWNODE_CLASSNAME = 'repeat-row';

    /**
     * A jQuery selector that matches elements representing a repeated, read-only table row.
     */
    DOM.TABLE_REPEAT_ROWNODE_SELECTOR = '.repeat-row';

    /**
     * Returns whether the passed node is a table page break row element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table page break row element.
     */
    DOM.isTablePageBreakRowNode = function (node) {
        return $(node).hasClass(DOM.TABLE_PAGEBREAK_ROWNODE_CLASSNAME);
    };

    /**
     * Returns whether the passed node is repeated table row element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is repeated table row element.
     */
    DOM.isRepeatedRowNode = function (node) {
        return $(node).hasClass(DOM.TABLE_REPEATED_ROWNODE_CLASSNAME);
    };

    /**
     * Info: this function is not the same as DOM.isRepeatedRowNode!
     * Returns whether the passed node is table row element to be repeated across table split pages.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is repeated table row element.
     */
    DOM.isRepeatRowNode = function (node) {
        return $(node).hasClass(DOM.TABLE_REPEAT_ROWNODE_CLASSNAME);
    };

    /**
     * Returns whether the passed node is repeated table row element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is repeated table row element.
     */
    DOM.isFirstRowInRepeatingRowsTable = function (node) {
        return DOM.isRepeatRowNode(node) && !$(node).prev().length; // check if table is having repeating rows property
    };

    /**
     * Returns whether the passed node is a table node containing a page break
     * row element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node containing a page break row element.
     */
    DOM.isTableWithPageBreakRowNode = function (node) {
        return DOM.isTableNode(node) && ($(node).find(DOM.TABLE_PAGEBREAK_ROWNODE_SELECTOR).length > 0);
    };

    /**
     * Returns the collection of table rows from the passed table element.
     *
     * @param {HTMLTableElement|jQuery} tableNode
     *  The table DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  A jQuery collection containing all rows of the specified table.
     */
    DOM.getTableRows = function (tableNode) {

        if (DOM.isTableNode(tableNode)) {
            return $(tableNode).children('tbody').children('tr').not(DOM.TABLE_PAGEBREAK_ROWNODE_SELECTOR);
        } else if (DrawingFrame.isTableDrawingFrame(tableNode)) {
            return $(tableNode).children('.content:not(.copy)').find('tr');
        } else {
            return $();
        }
    };

    /**
     * A jQuery selector that matches elements representing a table cell.
     */
    DOM.TABLE_CELLNODE_SELECTOR = 'td';

    /**
     * Returns whether the passed node is a table cell element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table cell element.
     */
    DOM.isTableCellNode = function (node) {
        return $(node).is(DOM.TABLE_CELLNODE_SELECTOR);
    };

    /**
     * Creates a new table cell element.
     *
     * @param {jQuery} paragraph
     *  A paragraph node that is inserted into the cellcontent div of the new
     *  table cell.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.plain=false]
     *      If set to true, no resize nodes will be inserted into the cell, and
     *      other special formatting will be omitted (for example, the
     *      'contenteditable' attributes).
     *
     * @returns {jQuery}
     *  A table cell element, as jQuery object.
     */
    DOM.createTableCellNode = function (paragraph, options) {

        var cellContentNode = $('<div>').addClass('cellcontent'),
            cellChildNode = $('<div>').addClass('cell'),
            cellNode = $('<td>'),
            plain = Utils.getBooleanOption(options, 'plain', false);

        if (!plain) {
            cellChildNode.append($('<div>').addClass('resize bottom'), $('<div>').addClass('resize right'));
            cellChildNode.children().toggleClass('touch', Utils.TOUCHDEVICE);
        }
        cellNode.append(cellChildNode.append(cellContentNode.append(paragraph)));

        if (!plain) {
            // -> setting attribute only for IE.
            //    this leads to a yellow border around div.p in Chrome
            // -> Chrome Bug is fixed with: "outline: none !important;"
            Utils.handleReadOnly(cellChildNode);
            cellContentNode.attr({ contenteditable: true, 'data-focus-role': 'cell' });
        }

        return cellNode;
    };

    /**
     * A jQuery selector that matches elements representing a cell content node.
     */
    DOM.CELLCONTENT_NODE_SELECTOR = 'div.cellcontent';

    /**
     * A jQuery selector that matches elements representing a cell node inside a table cell.
     */
    DOM.CELL_NODE_SELECTOR = 'div.cell';

    /**
     * Returns whether the passed node is a table cell content element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table cell content element.
     */
    DOM.isCellContentNode = function (node) {
        return $(node).is(DOM.CELLCONTENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table cell node element inside a table cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table cell node element.
     */
    DOM.isCellNode = function (node) {
        return $(node).is(DOM.CELL_NODE_SELECTOR);
    };

    /**
     * Returns the container node of a table cell that contains all top-level
     * content nodes (paragraphs and tables).
     *
     * @param {HTMLTableCellElement|jQuery} cellNode
     *  The table cell DOM node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed table cell that contains all
     *  top-level content nodes (paragraphs and tables).
     */
    DOM.getCellContentNode = function (cellNode) {
        return $(cellNode).find('> * > ' + DOM.CELLCONTENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a table cell node element that is
     * a start cell of a vertically merged cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table cell node element that is
     * a start cell of a vertically merged cell.
     */
    DOM.isVerticalStartCellNode = function (node) {
        return DOM.isCellNode(node) && $(node).attr('mergeVert') === 'restart';
    };

    /**
     * Returns whether the passed node is a table cell node element that is
     * a 'continue' cell of a vertically merged cell. So it is not visible.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table cell node element that is
     *  a 'continue' cell of a vertically merged cell, so that it is not
     *  visible.
     */
    DOM.isVerticalContinueCellNode = function (node) {
        return DOM.isCellNode(node) && $(node).attr('mergeVert') === 'continue';
    };

    /**
     * A jQuery selector that matches elements representing a resize node.
     */
    DOM.RESIZE_NODE_SELECTOR = 'div.resize';

    /**
     * A jQuery selector that matches elements representing an active resize
     * node. This means the tracking is enabled on the node after a mouse down
     * or touchstart event.
     */
    DOM.ACTIVE_RESIZE_NODE_SELECTOR = 'div.resizeActive';

    /**
     * Returns whether the passed node is a table resize element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table resize element.
     */
    DOM.isResizeNode = function (node) {
        return $(node).is(DOM.RESIZE_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an active table resize element. This
     * means the tracking is enabled on the node after a mouse down or
     * touchstart event.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an active table resize element.
     */
    DOM.isActiveResizeNode = function (node) {
        return (DOM.isResizeNode(node) && $(node).is(DOM.ACTIVE_RESIZE_NODE_SELECTOR));
    };

    /**
     * Returns all cell nodes of the passed table. Does not include cells from
     * tables embedded in the passed table.
     *
     * @param {HTMLTableElement|jQuery} tableNode
     *  The table DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  A jQuery collection containing all cells of the passed table.
     */
    DOM.getTableCells = function (tableNode) {

        if (DOM.isTableNode(tableNode)) {
            return $(tableNode).find('> tbody > tr > td');
        } else if (DrawingFrame.isTableDrawingFrame(tableNode)) {
            return $(tableNode).children('.content:not(.copy)').find('tbody > tr > td');
        } else {
            return $();
        }
    };

    /**
     * A jQuery selector that matches elements representing a table cell
     * or a table row.
     */
    DOM.TABLE_CELL_ROW_NODE_SELECTOR = 'tr, td';

    /**
     * Returns whether the passed node is a table row or a table cell element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table row or a table cell element.
     */
    DOM.isTableCellOrRowNode = function (node) {
        return $(node).is(DOM.TABLE_CELL_ROW_NODE_SELECTOR);
    };

    /**
     * Check, whether the browser selection containing a focus node and an
     * anchor node describes a cell selection.
     *
     * @param {Node|jQuery|Null} focusNode
     *  The focus node of the browser selection.
     *
     * @param {Node|jQuery|Null} anchorNode
     *  The anchor node of the browser selection.
     *
     * @returns {Boolean}
     *  Whether the browser selection containing a focus node and an
     *  anchor node describes a cell selection.
     */
    DOM.isCellRangeSelected = function (focusNode, anchorNode) {

        var // the row containing the anchor node
            anchorRowNode = null;

        // focus node must be a row
        if (DOM.isTableRowNode(focusNode)) {
            // both nodes are rows in the same table
            if (DOM.isTableRowNode(anchorNode) && (Utils.getDomNode(anchorNode).parentNode === Utils.getDomNode(focusNode).parentNode)) { return true; }

            // anchor node can also be a cell content node (only two neighboring cells selected)
            if (DOM.isCellContentNode(anchorNode)) {
                anchorRowNode = $(anchorNode).closest(DOM.TABLE_ROWNODE_SELECTOR);
                if (anchorRowNode.length > 0 && Utils.getDomNode(anchorRowNode).parentNode === Utils.getDomNode(focusNode).parentNode) { return true; }
            }
        }

        return false;
    };

    // text spans, text nodes, text components ================================

    // text spans -------------------------------------------------------------

    /**
     * Returns a new empty text span element with a single child text node.
     *
     * @returns {jQuery}
     *  The empty text span element, as jQuery object.
     */
    DOM.createTextSpan = function () {
        var textSpan = $('<span>');
        textSpan[0].appendChild(document.createTextNode(''));
        return textSpan;
    };

    /**
     * This function checks, if a specified node is a text span node. If the text node is missing
     * it adds an empty text node into the span. This is necessary, because jQuerys text-function
     * creates no text node, if it is given an empty string.
     *
     * @param {Node|jQuery|Null} node
     *  The text span element, as jQuery object or html node.
     */
    DOM.ensureExistingTextNode = function (node) {
        var domNode = node && Utils.getDomNode(node);
        if (DOM.isTextSpanWithoutTextNode(domNode)) { domNode.appendChild(document.createTextNode('')); }
    };

    /**
     * This function checks, if a specified node is a text span node inside simple field in ODF.
     * If the text node is missing it adds an empty text node into the span.
     * This is necessary, because jQuerys text-function creates no text node, if it is given an empty string.
     *
     * @param {Node|jQuery|Null} node
     *  The text span element, as jQuery object or html node.
     */
    DOM.ensureExistingTextNodeInField = function (node) {
        var domNode = node && Utils.getDomNode(node);
        if (DOM.isTextSpanInFieldWithoutTextNode(domNode)) { domNode.appendChild(document.createTextNode('')); }
    };

    /**
     * This function checks, if a specified node is a span node inside a field marked with the class
     * 'special-field'. This is for example used in headers for page numbers. If the text node is
     * missing it adds an empty text node into the span.
     *
     * @param {Node|jQuery|Null} node
     *  The text span element, as jQuery object or html node.
     */
    DOM.ensureExistingTextNodeInSpecialField = function (node) {
        var domNode = node && Utils.getDomNode(node);
        if (DOM.isSpanInSpecialFieldWithoutTextNode(domNode)) { domNode.appendChild(document.createTextNode('')); }
    };

    /**
     * Returns whether the passed node is a <span> element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element.
     */
    DOM.isSpan = function (node) {
        return ($(node).is('span'));
    };

    /**
     * Returns whether the passed node is a <span> element containing a single
     * text node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element with a text node.
     */
    DOM.isTextSpan = function (node) {
        var contents = $(node).contents();
        return ($(node).is('span') && (contents.length === 1) && (contents[0].nodeType === 3));
    };

    /**
     * Returns whether the passed node is a text span element that extends to more
     * than one line in the containing paragraph.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element with a text node that extends
     *  to more than one line.
     */
    DOM.isMultiLineTextSpan = function (node) {
        return DOM.isTextSpan(node) && $(node).height() > Utils.getElementCssLength(node, 'line-height');
    };

    /**
     * Returns whether the passed node is a <span> whose parent is a
     * div with class 'hardbreak'
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a hard break span element.
     */
    DOM.isHardBreakSpan = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return $(domNode).is('span') && domNode.parentNode && DOM.isHardBreakNode(domNode.parentNode);
    };

    /**
     * Returns whether the passed node is a <span> element containing an empty
     * text node
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element with an empty text node.
     */
    DOM.isEmptySpan = function (node) {
        return (DOM.isTextSpan(node) && ($(node).text().length === 0));
    };

    /**
     * Returns whether the passed node is a <span> element representing a text
     * portion (a child element of a paragraph node).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text portion span element.
     */
    DOM.isPortionSpan = function (node) {
        return DOM.isTextSpan(node) && DOM.isParagraphNode(Utils.getDomNode(node).parentNode);
    };

    /**
     * Internet Explorer removes empty text nodes from text spans inside
     * table cells.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element without a text node.
     */
    DOM.isTextSpanWithoutTextNode = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return $(domNode).is('span') && ($(domNode).contents().length === 0) && DOM.isParagraphNode(domNode.parentNode);
    };

    /**
     * Returns whether span has empty text nodes inside ODF empty field.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span element without a text node.
     */
    DOM.isTextSpanInFieldWithoutTextNode = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return $(domNode).is('span') && ($(domNode).contents().length === 0) && DOM.isFieldNode(domNode.parentNode);
    };

    /**
     * Returns whether the specified node is a span without a text node inside
     * a special field (for example page number in header).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a span without a text node inside a special field.
     */
    DOM.isSpanInSpecialFieldWithoutTextNode = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return $(domNode).is('span') && ($(domNode).contents().length === 0) && DOM.isSpecialField(domNode.parentNode);
    };

    /**
     * Returns whether the passed node is a text node embedded in a text
     * portion span (see DOM.isPortionSpan() method).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text node in a portion span element.
     */
    DOM.isTextNodeInPortionSpan = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return domNode && (domNode.nodeType === 3) && DOM.isPortionSpan(domNode.parentNode);
    };

    /**
     * Returns the first text portion span of the specified paragraph node.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph node whose first text portion span will be returned. If
     *  this object is a jQuery collection, uses the first DOM node it
     *  contains.
     *
     * @returns {HTMLElement}
     *  The first text portion span of the paragraph.
     */
    DOM.findFirstPortionSpan = function (paragraph) {
        return Utils.findDescendantNode(paragraph, function () { return DOM.isTextSpan(this); }, { children: true });
    };

    /**
     * Returns the last text portion span of the specified paragraph node.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph node whose last text portion span will be returned. If
     *  this object is a jQuery collection, uses the first DOM node it
     *  contains.
     *
     * @returns {HTMLElement}
     *  The last text portion span of the paragraph.
     */
    DOM.findLastPortionSpan = function (paragraph) {
        return Utils.findDescendantNode(paragraph, function () { return DOM.isTextSpan(this); }, { children: true, reverse: true });
    };

    // change tracking ------------------------------------------

    /**
     * A jQuery selector that matches elements representing a node with
     * a changes attribute.
     */
    DOM.CHANGETRACK_NODE_SELECTOR = '[data-change-track-inserted="true"], [data-change-track-removed="true"], [data-change-track-modified="true"]';

    /**
     * A jQuery selector that matches elements representing a node with
     * the changes attribute 'inserted'.
     */
    DOM.CHANGETRACK_INSERTED_NODE_SELECTOR = '[data-change-track-inserted="true"]';

    /**
     * A jQuery selector that matches elements representing a node with
     * the changes attribute 'removed'.
     */
    DOM.CHANGETRACK_REMOVED_NODE_SELECTOR = '[data-change-track-removed="true"]';

    /**
     * A jQuery selector that matches elements representing a node with
     * the changes attribute 'modified'.
     */
    DOM.CHANGETRACK_MODIFIED_NODE_SELECTOR = '[data-change-track-modified="true"]';

    /**
     * A jQuery selector that matches table rows with the changes attribute 'inserted'.
     */
    DOM.CHANGETRACK_INSERTED_ROW_SELECTOR = 'tr[data-change-track-inserted="true"]';

    /**
     * A jQuery selector that matches table rows with the changes attribute 'removed'.
     */
    DOM.CHANGETRACK_REMOVED_ROW_SELECTOR = 'tr[data-change-track-removed="true"]';

    /**
     * A jQuery selector that matches table cells with the changes attribute 'removed'.
     */
    DOM.CHANGETRACK_REMOVED_CELL_SELECTOR = 'td[data-change-track-removed="true"]';

    /**
     * Returns whether the passed node is a node with a changes
     * attribute.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a node with change tracking attribute.
     */
    DOM.isChangeTrackNode = function (node) {
        return $(node).is(DOM.CHANGETRACK_NODE_SELECTOR) || DOM.isChangeTrackInlineNode(node);
    };

    /**
     * Returns whether the passed node is a node with changes 'inserted'
     * attribute.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a node with changes 'inserted'
     *  attribute.
     */
    DOM.isChangeTrackInsertNode = function (node) {
        return $(node).is(DOM.CHANGETRACK_INSERTED_NODE_SELECTOR) || DOM.isChangeTrackInsertInlineNode(node);
    };

    /**
     * Returns whether the passed node is a node with changes 'removed'
     * attribute.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a node  with changes 'removed'
     *  attribute.
     */
    DOM.isChangeTrackRemoveNode = function (node) {
        return $(node).is(DOM.CHANGETRACK_REMOVED_NODE_SELECTOR) || DOM.isChangeTrackRemoveInlineNode(node);
    };

    /**
     * Returns whether the passed node is a node with changes 'modified'
     * attribute.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a node with changes 'modified' attribute.
     */
    DOM.isChangeTrackModifyNode = function (node) {
        return $(node).is(DOM.CHANGETRACK_MODIFIED_NODE_SELECTOR) || DOM.isChangeTrackModifyInlineNode(node);
    };

    /**
     * Returns whether the passed node is an inline node with changes attribute
     * at its span (tab, hard break, field, ...).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline node with changes attribute set to its span.
     */
    DOM.isChangeTrackInlineNode = function (node) {
        return DOM.isInlineComponentNode(node) && $(Utils.getDomNode(node).firstChild).is(DOM.CHANGETRACK_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an inline node with changes 'inserted'
     * attribute at its span (tab, hard break, field, ...).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline node with changes 'inserted' attribute
     *  set to its span.
     */
    DOM.isChangeTrackInsertInlineNode = function (node) {
        return DOM.isInlineComponentNode(node) && $(Utils.getDomNode(node).firstChild).is(DOM.CHANGETRACK_INSERTED_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an inline node with changes 'removed'
     * attribute at its span (tab, hard break, field, ...).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline node with changes 'removed' attribute
     *  set to its span.
     */
    DOM.isChangeTrackRemoveInlineNode = function (node) {
        return DOM.isInlineComponentNode(node) && $(Utils.getDomNode(node).firstChild).is(DOM.CHANGETRACK_REMOVED_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is an inline node with changes 'modified'
     * attribute at its span (tab, hard break, field, ...).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline node with changes 'modified' attribute
     *  set to its span.
     */
    DOM.isChangeTrackModifyInlineNode = function (node) {
        return DOM.isInlineComponentNode(node) && $(Utils.getDomNode(node).firstChild).is(DOM.CHANGETRACK_MODIFIED_NODE_SELECTOR);
    };

    // text components: fields, tabs ------------------------------------------

    /**
     * A jQuery selector that matches elements representing an inline component
     * inside a paragraph (text components or inline drawing objects).
     */
    DOM.INLINE_COMPONENT_NODE_SELECTOR = 'div.inline';

    /**
     * Returns whether the passed node is an editable inline component node in
     * a paragraph. Inline components include text components (e.g. fields,
     * tabs) and inline drawing objects.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline component node in a paragraph.
     */
    DOM.isInlineComponentNode = function (node) {
        return $(node).is(DOM.INLINE_COMPONENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a <div> container with embedded text
     * spans, used as root elements for special text components in a paragraph.
     * Does NOT return true for helper nodes that do not represent editable
     * contents of a paragraph (e.g. numbering labels). To check for all helper
     * nodes that contain text spans (also non-editable elements such as
     * numbering labels), use the method DOM.isTextSpanContainerNode() instead.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a <div> element representing an editable
     *  text component in a paragraph.
     */
    DOM.isTextComponentNode = function (node) {
        // text component nodes are all inline nodes, except drawings.
        // This includes tabs, fields and hard breaks
        return DOM.isInlineComponentNode(node) && !DOM.isDrawingFrame(node);
    };

    /**
     * Returns whether the passed node is a <div> container with embedded text
     * spans, used as root elements for special text components in a paragraph.
     * Does NOT return true for text nodes contained in helper nodes that do
     * not represent editable contents of a paragraph (e.g. numbering labels).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a DOM text node contained in a <div> element
     *  representing an editable text component in a paragraph.
     */
    DOM.isTextNodeInTextComponent = function (node) {
        node = node ? Utils.getDomNode(node) : null;
        return node && node.parentNode && DOM.isTextComponentNode(node.parentNode.parentNode);
    };

    /**
     * Returns whether the passed node is a <div> container that is an inline component
     * node, but is not a placeholder for a drawing or a comment and no range marker
     * and no complex field node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an inline component node, that is not a placeholder
     *  for a drawing or a comment.
     */
    DOM.isEditableInlineComponentNode = function (node) {
        return DOM.isInlineComponentNode(node) && !DOM.isSubstitutionPlaceHolderNode(node) && !DOM.isRangeMarkerNode(node) && !DOM.isComplexFieldNode(node);
    };

    /**
     * The class name representing a text field.
     */
    DOM.FIELD_NODE_CLASS_NAME = 'field';

    /**
     * A jQuery selector that matches elements representing a text field.
     */
    DOM.FIELD_NODE_SELECTOR = 'div.field';

    /**
     * CSS class name that matches elements representing special text field:
     * page number type, only in header/footer.
     */
    DOM.SPECIAL_FIELD_NODE_CLASSNAME = 'special-field';

    /**
     * A jQuery selector that matches elements representing special text field:
     * page number type, only in header/footer.
     */
    DOM.SPECIAL_FIELD_NODE_SELECTOR = '.special-field';

    /**
     * A jQuery selector that matches elements representing a text field with type page-count.
     */
    DOM.PAGECOUNT_FIELD_SELECTOR = '.field.field-page-count, .field.field-NUMPAGES';

    /**
     * A jQuery selector that matches elements representing a text field with type page-number.
     */
    DOM.PAGENUMBER_FIELD_SELECTOR = '.field.field-page-number';

    /**
     * Returns whether the passed node is an element representing a text field.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing a text field.
     */
    DOM.isFieldNode = function (node) {
        return $(node).hasClass(DOM.FIELD_NODE_CLASS_NAME);
    };

    /**
     * Returns whether the passed node is element representing complex text field,
     * in header/footer, and with type page number.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing a special text field.
     */
    DOM.isSpecialField = function (node) {
        return $(node).hasClass(DOM.SPECIAL_FIELD_NODE_CLASSNAME);
    };

    /**
     * Returns a new empty text field element.
     *
     * @returns {jQuery}
     *  A new empty text field element, as jQuery object.
     */
    DOM.createFieldNode = function () {
        return $('<div>', { contenteditable: false }).addClass('inline field');
    };

    /**
     * A jQuery selector that matches elements representing a tab.
     */
    DOM.TAB_NODE_SELECTOR = 'div.tab';

    /**
     * Returns whether the passed node is a tab element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a is a tab element.
     */
    DOM.isTabNode = function (node) {
        return $(node).is(DOM.TAB_NODE_SELECTOR);
    };

    /**
     * Returns a new tab element.
     *
     * @returns {jQuery}
     *  A new tab element, as jQuery object.
     */
    DOM.createTabNode = function () {
        return $('<div>', { contenteditable: false }).addClass('inline tab');
    };

    /**
     * Returns whether the passed node is a <span> whose parent is a
     * div with class 'tab'
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a tab span element.
     */
    DOM.isTabSpan = function (node) {
        var domNode = node && Utils.getDomNode(node);
        return $(domNode).is('span') && domNode.parentNode && DOM.isTabNode(domNode.parentNode);
    };

    /**
     * A class name representing a hard-break.
     */
    DOM.HARDBREAK_NODE_CLASS_NAME = 'hardbreak';

    /**
     * A jQuery selector that matches elements representing a hard-break.
     */
    DOM.HARDBREAK_NODE_SELECTOR = 'div.hardbreak';

    /**
     * Returns whether the passed node is a hard-break element
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text node in a text field element.
     */
    DOM.isHardBreakNode = function (node) {
        return $(node).hasClass(DOM.HARDBREAK_NODE_CLASS_NAME);
    };

    /**
     * Returns a new hard-break element.
     *
     * @param {String} type
     *  The type of the hard-break element. Supported values are
     *  'page', 'column' or 'textWrapping'.
     *
     * @returns {jQuery}
     *  A new hard-break element, as jQuery object.
     */
    DOM.createHardBreakNode = function (type) {
        return $('<div>', { contenteditable: false }).addClass('inline hardbreak').data('type', type);
    };

    // paragraph helper nodes -------------------------------------------------

    /**
     * Returns whether the passed node is a <div> container with embedded text
     * spans, used as root elements for special text elements in a paragraph.
     * Returns also true for helper nodes that do NOT represent editable
     * contents of a paragraph (e.g. numbering labels). To check for container
     * nodes that represent editable components in a paragraph only, use the
     * method DOM.isTextComponentNode() instead.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a <div> element containing text spans.
     */
    DOM.isTextSpanContainerNode = function (node) {
        node = node ? Utils.getDomNode(node) : null;
        return $(node).is('div') && DOM.isParagraphNode(node.parentNode) && DOM.isTextSpan(node.firstChild);
    };

    /**
     * A jQuery selector that matches elements representing a list label.
     */
    DOM.LIST_LABEL_NODE_SELECTOR = 'div.list-label';

    /**
     * Returns whether the passed node is an element representing a list label.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a list label element.
     */
    DOM.isListLabelNode = function (node) {
        return $(node).is(DOM.LIST_LABEL_NODE_SELECTOR);
    };

    /**
     * Creates a new element representing a list label.
     *
     * @param {String} [text]
     *  The text contents of the list label node.
     *
     * @returns {jQuery}
     *  A new list label node, as jQuery object.
     */
    DOM.createListLabelNode = function (text) {
        var node = $('<div>').addClass('helper list-label').append(text ? DOM.createTextSpan().text(text) : DOM.createTextSpan());

        // Bug 27902: prevent placing text cursor into list label node
        // Bug 26568: IE9 and IE10 show a drawing selection frame when 'contenteditable' is set to false
        // Bug 29546 (still open): this leads to a wrong mouse pointer (move) for the list labels
        if (!_.browser.IE || (_.browser.IE >= 11)) { node.attr('contenteditable', false); }
        return node;
    };

    /**
     * Returns whether the passed node is a dummy text node that is used in
     * empty paragraphs to preserve an initial element height according to the
     * current font size.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a dummy text node.
     */
    DOM.isDummyTextNode = function (node) {
        return $(node).data('dummy') === true;
    };

    /**
     * Creates a dummy text node that is used in empty paragraphs to preserve
     * an initial element height according to the current font size.
     *
     * @returns {jQuery}
     *  A dummy text node, as jQuery object.
     */
    DOM.createDummyTextNode = function () {
        // TODO: create correct element for current browser
        return $('<br>').data('dummy', true);
    };

    // drawing nodes ----------------------------------------------------------

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DOM.isDrawingFrame = DrawingFrame.isDrawingFrame;

    /**
     * A jQuery selector that matches nodes representing a drawing frame.
     *
     * @constant
     */
    DOM.DRAWING_NODE_SELECTOR = '.' + DrawingFrame.NODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing a text frame content
     * element inside a drawing node. This is reassigned here for convenience
     * reasons to reduce required imports.
     *
     * @constant
     */
    DOM.TEXTFRAMECONTENT_NODE_SELECTOR = DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR;

    /**
     * A jQuery selector that matches elements representing a floating node
     * inside a paragraph (editable component nodes and helper nodes).
     */
    DOM.FLOATING_NODE_SELECTOR = 'div.float';

    /**
     * Returns whether the passed node is a floating node in a paragraph.
     * Floating nodes include floating drawing objects, and their associated
     * helper nodes.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a floating node in a paragraph.
     */
    DOM.isFloatingNode = function (node) {
        return $(node).is(DOM.FLOATING_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a <div> element wrapping a drawing
     * in inline mode.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing and is
     *  rendered inlined.
     */
    DOM.isInlineDrawingNode = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DOM.isInlineComponentNode(node);
    };

    /**
     * Returns whether the passed node is a <div> element wrapping a drawing
     * in floating mode.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing and is
     *  rendered floated.
     */
    DOM.isFloatingDrawingNode = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DOM.isFloatingNode(node);
    };

    /**
     * Returns whether the passed node (typically a paragraph) contains a floated
     * drawing node as direct child.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains a floated drawing node as direct child.
     */
    DOM.containsFloatingDrawingNode = function (node) {
        return $(node).children(DOM.FLOATING_NODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is a <div> element wrapping a drawing
     * of type 'image'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing of type
     *  'image'.
     */
    DOM.isImageDrawingNode = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR) && ($(node).data('type') === 'image');
    };

    /**
     * Returns whether the passed node is a <div> element wrapping a drawing
     * that is not of type 'image'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing that is not
     *  of type 'image'.
     */
    DOM.isNonImageDrawingNode = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR) && ($(node).data('type') !== 'image');
    };

    /**
     * Returns whether the passed node is a drawing node that cannot be restored
     * after deletion.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing node that cannot be restored after
     *  deletion.
     */
    DOM.isUnrestorableDrawingNode = function (node) {

        if (!DrawingFrame.isDrawingFrame(node)) { return false; }

        // unrestorable are all drawing nodes, that are not images, not text frames. Groups are restorable,
        // if they contain only images or text frames.
        if (DrawingFrame.isGroupDrawingFrame(node)) {
            // checking the children of the group
            return _.find($(node).find(DrawingFrame.NODE_SELECTOR), function (drawingNode) {
                return (DOM.isNonImageDrawingNode(drawingNode) && !DrawingFrame.isShapeDrawingFrame(drawingNode) && !DrawingFrame.isGroupDrawingFrame(drawingNode)) || DrawingFrame.isWatermarkDrawingNode(drawingNode);
            });
        } else {
            return (DOM.isNonImageDrawingNode(node) && !DrawingFrame.isShapeDrawingFrame(node) && !DOM.isListLabelNode($(node).parent())) || DrawingFrame.isWatermarkDrawingNode(node);
        }
    };

    /**
     * Returns whether the passed node is inside a drawing text frame node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside a text frame drawing node.
     */
    DOM.isNodeInsideTextFrame = function (node) {
        return node && $(node).length > 0 && $(node).closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is inside an auto resizing drawing text frame node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside an auto resizing text frame drawing node.
     */
    DOM.isNodeInsideAutoResizeHightTextFrame = function (node) {
        return node && $(node).length > 0 && $(node).closest(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR).length > 0;
    };

    /**
     * Returns whether the specified node is a text frame that contains no text
     * content and that has no (explicit) border specified.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} [attributes]
     *  The (explicit) attributes of the drawing node. If they are not specified,
     *  the explicit attributes at the specified drawing node are evaluated.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.ignoreTemplateText=false]
     *      If set to true, a drawing that contains the template text will be handled
     *      as empty drawing. Otherwise a drawing that contains a template text
     *      is NOT empty. The latter is the default.
     *
     * @returns {Boolean}
     *  Whether the specified node is a text frame that contains no text content
     *  and that has no (explicit) border specified.
     */
    DOM.isEmptyTextframeWithoutBorder = function (node, attributes, options) {

        var // the explicit attributes at the drawing
            attrs = attributes || AttributeUtils.getExplicitAttributes(node),
            // whether an explicit line attribute is set at this drawing
            hasBorder = attrs && attrs.line && attrs.line.type !== 'none';

        return DOM.isEmptyTextframe(node, options) && !hasBorder;
    };

    /**
     * Returns whether the specified node is a text frame that contains no text
     * content.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.ignoreTemplateText=false]
     *      If set to true, a drawing that contains the template text will be handled
     *      as empty drawing. Otherwise a drawing that contains a template text
     *      is NOT empty. The latter is the default.
     *
     * @returns {Boolean}
     *  Whether the specified node is a text frame that contains no text content.
     */
    DOM.isEmptyTextframe = function (node, options) {

        var // whether the text frame contains no content
            isEmptyTextFrame = false,
            // all paragraphs and all text spans inside the text frame
            allParagraphs = null, allTextSpans = null,
            // whether a template text shall be ignored
            ignoreTemplateText = Utils.getBooleanOption(options, 'ignoreTemplateText', false);

        if (DrawingFrame.isTextFrameShapeDrawingFrame(node)) {
            if (DrawingFrame.isModifyingDrawingActive(node)) { // there might be a second content node in the drawing
                allParagraphs = $(node).children('.content:not(.copy)').find(DOM.PARAGRAPH_NODE_SELECTOR);
            } else {
                allParagraphs = $(node).find(DOM.PARAGRAPH_NODE_SELECTOR);
            }

            if (allParagraphs.length === 1) {
                allTextSpans = allParagraphs.children('span');
                isEmptyTextFrame = (!allTextSpans.text() && allTextSpans.length === 1) || (ignoreTemplateText && allTextSpans.hasClass(DOM.TEMPLATE_TEXT_CLASS));
                // Info: This check allows more than one text span with class 'templatetext' although this should never happen (spellchecking not allowed)
                // Info: jQuery 'hasClass' returns true, if one of the text spans has class 'templatetext' (a mixture of spans must never happen).
            }
        }

        return isEmptyTextFrame;
    };

    /**
     * Returns whether the passed node is a drawing node that has a content node that
     * has the class 'DrawingFrame.EMPTYTEXTFRAME_CLASS' assigned (fast detection) or
     * that contains only an implicit paragraph.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing node that has a content node that
     *  has the class 'DrawingFrame.EMPTYTEXTFRAME_CLASS' assigned or that contains
     *  only an implicit paragraph.
     */
    DOM.isDrawingWithOnlyImplicitParagraph = function (node) {

        var // whether the specified node contains only an implicit paragraph
            isOnlyImplicitParagraph = false,
            // the content node inside the drawing
            contentNode = null,
            // all paragraphs and all text spans inside the text frame
            allParagraphs = null;

        if (DrawingFrame.isDrawingFrame(node)) {
            contentNode = DrawingFrame.getContentNode(node);
            if (contentNode.hasClass(DrawingFrame.EMPTYTEXTFRAME_CLASS)) { return true; }
            // checking for paragraphs
            allParagraphs = contentNode.find(DOM.PARAGRAPH_NODE_SELECTOR);

            if (allParagraphs.length === 1 && DOM.isImplicitParagraphNode(allParagraphs)) { return true; }
        }

        return isOnlyImplicitParagraph;
    };

    /**
     * Returns whether the passed node is the first direct child inside a
     * text frame. This can be a paragraph or a table.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the first child inside a text frame.
     */
    DOM.isFirstContentNodeInTextframe = function (node) {

        var // the dom node of the specified node
            domNode = Utils.getDomNode(node);

        return domNode && DOM.isContentNode(domNode) && !domNode.previousSibling && DrawingFrame.isTextFrameNode(domNode.parentNode);
    };

    /**
     * Returns whether the passed node is the first text span inside a text
     * frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the first text span inside a text frame.
     */
    DOM.isFirstTextSpanInTextframe = function (node) {

        var // a text frame containing the specified node
            textFrame = null,
            // the first paragraph in the text frame
            firstParagraph = null;

        // Check: node is text span and node is inside text frame and node is the first text span in the text frame
        if (DOM.isTextSpan(node)) {
            textFrame = DrawingFrame.getClosestTextFrameDrawingNode(node);

            if (textFrame) {
                firstParagraph = textFrame.find(DOM.PARAGRAPH_NODE_SELECTOR).first();
                return DOM.findFirstPortionSpan(firstParagraph) === Utils.getDomNode(node);
            }
        }

        return false;
    };

    /**
     * Returns whether the passed node is the last text span inside a text
     * frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the last text span inside a text frame.
     */
    DOM.isLastTextSpanInTextframe = function (node) {

        var // a text frame containing the specified node
            textFrame = null,
            // the first paragraph in the text frame
            lastParagraph = null;

        // Check: node is text span and node is inside text frame and node is the last text span in the text frame
        if (DOM.isTextSpan(node)) {
            textFrame = DrawingFrame.getClosestTextFrameDrawingNode(node);

            if (textFrame) {
                lastParagraph = textFrame.find(DOM.PARAGRAPH_NODE_SELECTOR).last();
                return DOM.findLastPortionSpan(lastParagraph) === Utils.getDomNode(node);
            }
        }

        return false;
    };

    /**
     * The class name representing a text span that contains template text.
     */
    DOM.TEMPLATE_TEXT_CLASS = 'templatetext';

    /**
     * Css selector representing a text span containing template text.
     */
    DOM.TEMPLATE_TEXT_SELECTOR = '.templatetext';

    /**
     * Returns whether the passed node is a text node with default template text.
     *
     * @param {Node} node
     *  The node that will be checked.
     *
     * @returns {Boolean}
     *  Whether the specified node is a text node and the parent has the specified class
     *  to recognize the template text.
     */
    DOM.isTextFrameTemplateTextNode = function (node) {
        return node && node.nodeType === 3 && node.parentNode && $(node.parentNode).hasClass(DOM.TEMPLATE_TEXT_CLASS);
    };

    /**
     * Returns whether the passed node is a span with default template text.
     *
     * @param {Node} node
     *  The node that will be checked.
     *
     * @returns {Boolean}
     *  Whether the specified node is a text span node and has the specified class
     *  to recognize the template text.
     */
    DOM.isTextFrameTemplateTextSpan = function (node) {
        return $(node).hasClass(DOM.TEMPLATE_TEXT_CLASS);
    };

    /**
     * Returns whether the passed node is a paragraph with default template text. This
     * template text span is not necessarily the only child. It is possible, that there
     * is also a list label in the paragraph.
     *
     * @param {Node} node
     *  The node that will be checked.
     *
     * @returns {Boolean}
     *  Whether the specified node is a paragraph node that contains a text span
     *  that has the specified class to recognize the template text.
     */
    DOM.isTextFrameTemplateTextParagraph = function (node) {
        return DOM.isParagraphNode(node) && $(node).children(DOM.TEMPLATE_TEXT_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is a text span and the first element without previous node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text span and the first element inside its parent node.
     */
    DOM.isFirstTextSpanInParagraph = function (node) {
        return DOM.isTextSpan(node) && $(node).prev().length === 0;
    };

    /**
     * Returns whether the passed node is a <div> element wrapping an image.
     * Simple 'find' for 'img' does not work, because drawing groups can contain several
     * image drawings and images, but are no image drawings.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping an image.
     */
    DOM.isImageNode = function (node) {
        // drawing div contains another div (class content) that contains an image
        return DrawingFrame.isDrawingFrame(node) && ($(node).data('type') === 'image');
    };

    /**
     * Returns true if the source of the passed image node points to a mail inline image.
     * See: http://oxpedia.org/wiki/index.php?title=HTTP_API#Requesting_an_image
     * The node can be a <div> wrapping an image or an <img> itself.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed image node points to a mail inline image.
     */
    DOM.isMailInlineImageNode = function (node) {

        var img = $(node).is('img') ? $(node) : $(node).find('img');
        var src = img.attr('src');

        if (!src) { return false; }

        // check for URL parameters
        return _.every(['/mail/picture', 'id', 'folder', 'uid'], function (param) {
            return src.indexOf(param) !== -1;
        });
    };

    /**
     * Returns true if the source of the passed image node points to a server document.
     * The node can be a <div> wrapping an image or an <img> itself.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed image node points to a server document.
     */
    DOM.isDocumentImageNode = function (node) {

        var img = $(node).is('img') ? $(node) : $(node).find('img'),
            src = img.attr('src');

        if (!src) { return false; }  // replacement data is shown

        // check for base64 data-URL first
        if (src.substring(0, 10) === 'data:image') {
            return false;
        }

        // check for URL parameters
        return _.every(['id', 'folder_id', 'action', 'get_filename'], function (param) {
            return src.indexOf(param) !== -1;
        });
    };

    /**
     * Returns true if the source of the passed image node contains a base64 data-URL.
     * The node can be a <div> wrapping an image or an <img> itself.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed image node contains a base64 data-URL.
     */
    DOM.isBase64ImageNode = function (node) {

        var img = $(node).is('img') ? $(node) : $(node).find('img'),
            src = img.attr('src');

        return src.substring(0, 10) === 'data:image';
    };

    /**
     * Returns the value of the given URL param of the image src attribute
     *
     * @param {Node|jQuery} node
     *  The image node to get the URL param from.
     *  If this object is a jQuery collection, uses the first DOM node it contains.
     *
     * @param {String} [param]
     *  The URL param
     *
     * @returns {String|Null}
     *  The value of the given URL param or null
     */
    DOM.getUrlParamFromImageNode = function (node, param) {

        var img = $(node).is('img') ? $(node) : $(node).find('img'),
            src = img.attr('src'),
            reg,
            match = null;

        if (src && param && param.length) {
            reg = new RegExp('.*[?&]' + param + '=([^&]+)(&|$)');
            match = src.match(reg);
            return (_.isArray(match) ? match[1] : null);
        }

        return null;
    };

    /**
     * Returns the base64 encoded data-URL of the given image node
     *
     * @param {Node|jQuery} node
     *  The image node to get the base64 data-URL from.
     *  If this object is a jQuery collection, uses the first DOM node it contains.
     *
     *  @param {String} [mimeType='image/png']
     *   The mime type of the base64 encoded data-URL.
     *   If not given or not supported by the browser 'image/png' is used.
     *
     *  @returns {String}
     *  The base64 encoded data-URL
     */
    DOM.getBase64FromImageNode = function (node, mimeType) {

        var // the image DOM node
            img = Utils.getDomNode($(node).is('img') ? $(node) : $(node).find('img')),
            // the mime type
            mime = (_.isString(mimeType) && mimeType.length) ? mimeType : 'image/png',
            // create an empty canvas element
            canvas = document.createElement('canvas'),
            // the drawing context
            ctx,
            // the base64 data URL
            base64;

        try {
            // set canvas to natural image size
            canvas.width = img.naturalWidth || img.width;
            canvas.height = img.naturalHeight || img.height;

            // copy the image contents to the canvas
            ctx = canvas.getContext('2d');
            ctx.drawImage(img, 0, 0);

            // return the data-URL formatted image
            base64 = canvas.toDataURL(mime);

        } catch (ex) {
            Utils.warn('DOM.getBase64FromImageNode(): ' + ((ex && ex.message) ? ex.message : 'failed'));
            base64 = 'data:,';
        }

        return base64;
    };

    /**
     * A jQuery selector that matches elements representing a drawing offset
     * helper node.
     */
    DOM.OFFSET_NODE_SELECTOR = 'div.float.offset';

    /**
     * A jQuery selector that matches elements representing an absolutely
     * positioned drawing.
     */
    DOM.ABSOLUTE_DRAWING_SELECTOR = 'div.drawing.absolute';

    /**
     * Returns whether the passed node is a <div> element for positioning
     * a drawing with a vertical or horizontal offset.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div element wrapping a drawing.
     */
    DOM.isOffsetNode = function (node) {
        return $(node).is(DOM.OFFSET_NODE_SELECTOR);
    };

    /**
     * The CSS class used to mark drawing placeholder nodes. This place holder
     * nodes are located in the page content node and safe the position for
     * an absolutely positioned drawing in the drawing layer.
     *
     * @constant
     */
    DOM.DRAWINGPLACEHOLDER_CLASS = 'drawingplaceholder';

    /**
     * The data property name for registering the drawing place holder at the absolutely
     * positioned drawing. This works in both directions, because the drawing place
     * holder has a data attribute with this name to the absolutely positioned drawing
     * and vice versa.
     *
     * @constant
     */
    DOM.DRAWINGPLACEHOLDER_LINK = 'drawingplaceholderlink';

    /**
     * The data property name for registering the margins (in px) at the absolutely positioned
     * drawings.
     *
     * @constant
     */
    DOM.DRAWING_MARGINS_PIXEL = 'pixelMargins';

    /**
     * The data property name for registering the margins (in hmm) at the absolutely positioned
     * drawings.
     *
     * @constant
     */
    DOM.DRAWING_MARGINS_HMM = 'hmmMargins';

    /**
     * A jQuery selector that matches elements representing an image place holder.
     */
    DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR = 'div.' + DOM.DRAWINGPLACEHOLDER_CLASS;

    /**
     * Returns whether the passed node is an element representing an image
     * place holder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing an image placeholder
     *  node.
     */
    DOM.isDrawingPlaceHolderNode = function (node) {
        return $(node).hasClass(DOM.DRAWINGPLACEHOLDER_CLASS);
    };

    /**
     * Returns whether the passed node (a paragraph) has at least one child
     * image place holder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node has at least one child representing an image
     *  place holder node.
     */
    DOM.hasDrawingPlaceHolderNode = function (node) {
        return $(node).children(DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR).length > 0;
    };

    /**
     * Searching the place holder node for a specific drawing node. This place
     * holder node exists in the page content for all drawings inside the
     * drawing layer node. This also works vice versa: Searching the drawing
     * node in the drawing layer for a specified place holder. In both cases
     * the 'data' attribute is named DOM.DRAWINGPLACEHOLDER_LINK.
     *
     * @param {Node|jQuery|Null} [node]
     *  The drawing node.
     *
     * @returns {Node}
     *  The place holder node for the drawing node in the drawing layer node
     *  or the drawing in the drawing layer node itself.
     */
    DOM.getDrawingPlaceHolderNode = function (node) {

        var // the node saved in the data object
            searchedNode = $(node).data(DOM.DRAWINGPLACEHOLDER_LINK),
            // an optional header or footer node
            marginalRoot = null,
            // the ID of the drawing in the drawing layer
            drawingID = 0,
            // a string to find the related placeholder node
            selectorString = null,
            // the nodes with selector string inside header or footer
            foundNodes = null,
            // a selector for the searched dom node
            searchSelector = null;

        if (searchedNode) { return searchedNode; }

        // Handling headers and footers, where the data object is not used
        if (DOM.isDrawingInsideMarginalDrawingLayerNode(node)) {
            marginalRoot = $(node).parent().parent(); // this is a drawing in the text drawing layer
            searchSelector = DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR;
        // } else if (DOM.isMarginalNode($(node).parent())) {
        } else if (DOM.isParagraphNode($(node).parent())) {
            marginalRoot = DOM.getClosestMarginalTargetNode(node); // this is a placeholder node
            searchSelector = DOM.ABSOLUTE_DRAWING_SELECTOR;
        }

        if (marginalRoot && marginalRoot.length > 0) {
            drawingID = $(node).attr('data-drawingID');
            selectorString = '[data-drawingID=' + drawingID + ']';
            // finding the place holder and optionally the space maker node, that are located inside the page content node
            foundNodes = $(marginalRoot).find(selectorString);

            if (foundNodes && foundNodes.length > 1) {  // at least drawing in drawing layer, place holder and optionally spacemaker
                searchedNode = _.find(foundNodes, function (node) {
                    return $(node).is(searchSelector);
                });

                if (!searchedNode) { Utils.warn('Warning: failed to find marginal node with selector: ' + searchSelector); }

            } else {
                Utils.warn('Warning: Expected at least two marginal nodes, but found: ' + foundNodes.length);
            }
        } else {
            // outside of header or footer the node must be found from data object
            if (!searchedNode) { Utils.warn('Warning: Failed to find place holder node.'); }
        }

        return searchedNode;
    };

    /**
     * Returns whether the passed node is an element representing a drawing that is
     * absolutely positioned and located inside a paragraph (not in drawing layer).
     * This are the drawings, that are anchored to paragraphs, not to pages.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an absolutely positioned drawing inside a paragraph.
     */
    DOM.isAbsoluteParagraphDrawing = function (node) {
        return DOM.isParagraphNode($(node).parent()) && $(node).is(DOM.ABSOLUTE_DRAWING_SELECTOR);
    };

    /**
     * Returns whether the passed node (typically a paragraph) contains at least one
     * absolutely positioned drawing anchored to the paragraph.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains at least one child representing an absolutely
     *  positioned drawing anchored to a paragraph.
     */
    DOM.hasAbsoluteParagraphDrawing = function (node) {
        return DOM.isParagraphNode(node) && ($(node).has(DOM.ABSOLUTE_DRAWING_SELECTOR).length > 0);
    };

    /**
     * The CSS class used to mark drawing space maker nodes.x
     *
     * @constant
     */
    DOM.DRAWING_SPACEMAKER_CLASS = 'drawingspacemaker';

    /**
     * The data property name for registering the drawing space maker at the absolutely
     * positioned drawing.
     *
     * @constant
     */
    DOM.DRAWING_SPACEMAKER_LINK = 'drawingspacemakerlink';

    /**
     * A jQuery selector that matches elements representing a drawing space maker node.
     */
    DOM.DRAWING_SPACEMAKER_NODE_SELECTOR = 'div.' + DOM.DRAWING_SPACEMAKER_CLASS;

    /**
     * The data property name for registering the increase of height in pixel into a footer
     * node, after inserting the drawing space maker node.
     *
     * @constant
     */
    DOM.DRAWING_SPACEMAKER_SHIFT = 'drawingSpaceMakerShift';

    /**
     * Returns whether the passed node is an element representing a drawing space
     * maker node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing a drawing space maker
     *  node.
     */
    DOM.isDrawingSpaceMakerNode = function (node) {
        return $(node).hasClass(DOM.DRAWING_SPACEMAKER_CLASS);
    };

    /**
     * Searching the space maker node for a specific drawing node. This space
     * maker node exists in the page content for all drawings inside the
     * drawing layer node, that require space 'below' the drawing.
     *
     * @param {Node|jQuery|Null} [node]
     *  The drawing node.
     *
     * @returns {Node}
     *  The space maker node for the drawing node in the drawing layer node.
     */
    DOM.getDrawingSpaceMakerNode = function (node) {

        var // the node saved in the data object
            searchedNode = $(node).data(DOM.DRAWING_SPACEMAKER_LINK),
            // an optional header or footer node
            marginalRoot = null,
            // the ID of the drawing in the drawing layer
            drawingID = 0,
            // a string to find the related placeholder node
            selectorString = null,
            // the nodes with selector string inside header or footer
            foundNodes = null,
            // the number of nodes that need to be found
            minValue = 2;

        if (searchedNode) { return searchedNode; }

        // Handling headers and footers, where the data object is not used
        if (DOM.isDrawingInsideMarginalDrawingLayerNode(node)) {
            marginalRoot = $(node).parent().parent(); // this is a drawing in the text drawing layer in header or footer
        }

        // also handling absolute paragraph drawings
        if (!marginalRoot && DOM.isAbsoluteParagraphDrawing(node) && DOM.isMarginalNode($(node).parent())) {
            marginalRoot = DOM.getClosestMarginalTargetNode(node);
            minValue = 1; // only drawing and space maker, no place holder
        }

        if (marginalRoot && marginalRoot.length > 0) {
            drawingID = $(node).attr('data-drawingID');
            selectorString = '[data-drawingID=' + drawingID + ']';
            // finding the optional space maker node
            foundNodes = $(marginalRoot).find(selectorString);

            if (foundNodes && foundNodes.length > minValue) {  // at least drawing in drawing layer, place holder and optionally spacemaker
                searchedNode = _.find(foundNodes, function (node) {
                    return DOM.isDrawingSpaceMakerNode(node);
                });
            }
        }

        return searchedNode;
    };

    /**
     * Returns whether the passed node (a paragraph) has at least one child
     * representing a drawing space maker node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node has at least one child representing a drawing
     *  space maker node.
     */
    DOM.hasDrawingSpaceMakerNode = function (node) {
        return $(node).children(DOM.DRAWING_SPACEMAKER_NODE_SELECTOR).length > 0;
    };

    /**
     * The CSS class used to mark the drawing layer node.
     *
     * @constant
     */
    DOM.DRAWINGLAYER_CLASS = 'textdrawinglayer';

    /**
     * A jQuery selector that matches the drawing layer node.
     */
    DOM.DRAWINGLAYER_NODE_SELECTOR = 'div.' + DOM.DRAWINGLAYER_CLASS;

    /**
     * Returns whether the passed node is the drawing layer node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the drawing layer node.
     */
    DOM.isDrawingLayerNode = function (node) {
        return $(node).hasClass(DOM.DRAWINGLAYER_CLASS);
    };

    /**
     * Returns whether the passed node is a drawing layer node inside a header
     * or footer.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing layer node inside header or footer.
     */
    DOM.isMarginalDrawingLayerNode = function (node) {
        return DOM.isDrawingLayerNode(node) && DOM.isHeaderOrFooter($(node).parent());
    };

    /**
     * Returns the container node of absolute positioned drawings.
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node for all absolute positioned drawings.
     */
    DOM.getTextDrawingLayerNode = function (pageNode) {
        return $(pageNode).children(DOM.DRAWINGLAYER_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is inside the drawing layer node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is located inside the drawing layer node.
     */
    DOM.isInsideDrawingLayerNode = function (node) {
        return $(node).parents(DOM.DRAWINGLAYER_NODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is a drawing inside the drawing layer node
     * inside a header or footer.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing node inside the drawing layer node
     *  inside header or footer.
     */
    DOM.isDrawingInsideMarginalDrawingLayerNode = function (node) {
        return DOM.isMarginalDrawingLayerNode($(node).parent());
    };

    /**
     * Returns the drawing node that is directly located below the drawing
     * layer. The specified node needs to be located inside this drawing.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Returns the top level drawing inside the drawing layer.
     */
    DOM.getTopLevelDrawingInDrawingLayerNode = function (node) {
        return $(node).parentsUntil(DOM.DRAWINGLAYER_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR).last();
    };

    // manipulate and iterate text spans --------------------------------------

    /**
     * Splits the passed text span element into two text span elements. Clones
     * all formatting to the new span element.
     *
     * @param {HTMLSpanElement|jQuery} span
     *  The text span to be split. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Number} offset
     *  The character position where the text span will be split. If this
     *  position is at the start or end of the text in the span, an empty text
     *  span will be inserted.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.append=false]
     *      If set to true, the right part of the text will be inserted after
     *      the passed text span; otherwise the left part of the text will be
     *      inserted before the passed text span. The position of the new text
     *      span may be important when iterating and manipulating a range of
     *      DOM nodes.
     *
     * @returns {jQuery}
     *  The newly created text span element, as jQuery object. Will be located
     *  before or after the passed text span, depending on the 'options.append'
     *  option.
     */
    DOM.splitTextSpan = function (span, offset, options) {

        var // the new span for the split text portion, as jQuery object
            newSpan = $(span).clone(true),
            // the existing text node (must not be invalidated, e.g. by using jQuery.text())
            textNode = Utils.getDomNode(span).firstChild,
            // text for the left span
            leftText = textNode.nodeValue.substr(0, offset),
            // text for the right span
            rightText = textNode.nodeValue.substr(offset);

        // insert the span and update the text nodes
        if (Utils.getBooleanOption(options, 'append', false)) {
            newSpan.insertAfter(span);
            textNode.nodeValue = leftText;
            newSpan.text(rightText);
            if (!rightText) { DOM.ensureExistingTextNode(newSpan); }
        } else {
            newSpan.insertBefore(span);
            newSpan.text(leftText);
            if (!leftText) { DOM.ensureExistingTextNode(newSpan); }
            textNode.nodeValue = rightText;
        }

        // return the new text span
        return newSpan;
    };

    /**
     * Calls the passed iterator function for all descendant text span elements
     * in a the passed node. As a special case, if the passed node is a text
     * span by itself, it will be visited directly. Text spans can be direct
     * children of a paragraph node (regular editable text portions), or
     * children of other nodes such as text fields or list label nodes.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM node whose descendant text spans will be visited (or which will
     *  be visited by itself if it is a text span). If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Function} iterator
     *  The iterator function that will be called for every text span. Receives
     *  the DOM span element as first parameter. If the iterator returns the
     *  Utils.BREAK object, the iteration process will be stopped immediately.
     *
     * @param {Object} [context]
     *  If specified, the iterator will be called with this context (the symbol
     *  'this' will be bound to the context inside the iterator function).
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    DOM.iterateTextSpans = function (node, iterator, context) {

        // visit passed text span directly
        if (DOM.isTextSpan(node)) {
            return iterator.call(context, Utils.getDomNode(node));
        }

        // do not iterate into drawings, they may contain their own paragraphs
        // Bug 30794: also omit list-label nodes as they are especially handled
        // in updateList function in the editor.
        if (!DrawingFrame.isDrawingFrame(node) && !DOM.isListLabelNode(node)) {
            return Utils.iterateSelectedDescendantNodes(node, function () { return DOM.isTextSpan(this); }, iterator, context);
        }
    };

    // generic container and content nodes ====================================

    /**
     * A jQuery selector that matches elements representing a top-level content
     * node (e.g. paragraphs or tables).
     */
    DOM.CONTENT_NODE_SELECTOR = DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR;

    /**
     * Returns whether the passed node is a top-level content node (e.g.
     * paragraphs or tables).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a top-level content node.
     */
    DOM.isContentNode = function (node) {
        return $(node).is(DOM.CONTENT_NODE_SELECTOR);
    };

    /**
     * Returns the correct node of a container node that contains its content
     * nodes as direct children. Container nodes include the document page,
     * table cells, or drawing objects containing text.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The element that contains the child components of the passed nodes as
     *  direct children, as jQuery object.
     */
    DOM.getChildContainerNode = function (node) {

        // convert to a jQuery object
        node = $(node);

        // page nodes
        if (DOM.isPageNode(node)) {
            return DOM.getPageContentNode(node);
        }

        // comment nodes
        if (DOM.isCommentNode(node)) {
            return DrawingFrame.getTextFrameNode(node);
        }

        // table cell nodes
        if (node.is('td')) {
            return DOM.getCellContentNode(node);
        }

        // drawing nodes
        if (DrawingFrame.isDrawingFrame(node)) {

            if (DrawingFrame.isTextFrameShapeDrawingFrame(node)) {
                return DrawingFrame.getTextFrameNode(node);
            } else {
                return DrawingFrame.getContentNode(node);
            }

        }

        // header/footer node - it doesn't have content node, paragraphs are direct children!
        if (DOM.isHeaderOrFooter(node)) {
            return DOM.getMarginalContentNode(node);
        }

        return $();
    };

    /**
     * Finding the allowed neighboring node taking into account all nodes
     * that are used for calculating the logical positions. For example as
     * top level nodes only paragraphs and tables are allowed, page-break
     * nodes need to be skipped.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} options
     *  A map with additional options controlling the search of the neighbor.
     *  The following options are supported:
     *  @param {Boolean} [options.next=true]
     *      The search direction. If not specified, the next neighboring node
     *      will be searched.
     *
     * @returns {Node|null}
     *  The search element of the specified node. If it cannot be determined
     *  null is returned.
     */
    DOM.getAllowedNeighboringNode = function (node, options) {

        var // determining the direction for searching the neighbor
            next = Utils.getBooleanOption(options, 'next', true),
            // the dom node of the specified node
            domNode = Utils.getDomNode(node),
            // the iterator function
            iterator = next ? 'nextSibling' : 'previousSibling',
            // the neighboring node
            neighbor = domNode[iterator] || null;

        while (neighbor && DOM.isInvalidNode(neighbor)) {
            neighbor = neighbor[iterator];
        }

        return neighbor;
    };

    /**
     * Checking, whether a specified node is part of the calculation of
     * logical positions.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified node is part of the calculation of
     *  logical positions.
     */
    DOM.isInvalidNode = function (node) {
        return DOM.isPageBreakNode(node) || DOM.isListLabelNode(node) || DOM.isCommentPlaceHolderNode(node);  // TODO: List needs to be expanded
    };

    /**
     * Creates a white list of allowed DOM nodes (paragraphs and tables currently),
     * and ignores pagebreaks and clearfloating divs
     *
     * @param {HTMLElement|jQuery} childNodes
     *  List of nodes that we want to filter, to contain only allowed ones, such as top level nodes paragraphs and tables
     *
     * @returns {HTMLElement|jQuery}
     *  list of nodes after filtering
     */
    DOM.allowedChildNodes = function (childNodes) {
        return $(childNodes).filter(DOM.PARAGRAPH_NODE_SELECTOR + ',' + DOM.TABLE_NODE_SELECTOR);
    };

    /**
     * jQuery selector for skipping drawing, repeated rows and page break nodes.
     *
     */
    DOM.SKIP_DRAWINGS_AND_P_BREAKS_SELECTOR = '.drawing, .page-break, .pb-row';

    // page break nodes ========================================================

    /**
     * jQuery selector that matches page break div, used to separate pages in page layout.
     */
    DOM.PAGE_BREAK_SELECTOR = '.page-break';

    /**
     * CSS class name that matches page break div, used to separate pages in page layout.
     */
    DOM.PAGE_BREAK_CLASS_NAME = 'page-break';

    /**
     * jQuery selector that matches manualy inserted page break by user
     */
    DOM.MANUAL_PAGE_BREAK_SELECTOR = '.manual-page-break';

    /**
     * CSS class name that matches manualy inserted page break by user
     */
    DOM.MANUAL_PAGE_BREAK_CLASS_NAME = 'manual-page-break';

    /**
     * jQuery selector that matches inner line in page break that marks begining of new page
     */
    DOM.PAGE_BREAK_LINE_SELECTOR = '.inner-pb-line';

    /**
     * CSS class name that matches line inside page break, that marks begining of new page
     */
    DOM.PAGE_BREAK_LINE_CLASS_NAME = 'inner-pb-line';

    /**
     * jQuery selector name that matches hardbreak with type page, from Microsoft only
     */
    DOM.MS_HARDBREAK_TYPE_PAGE_SELECTOR = '.ms-hardbreak-page';

    /**
     * CSS class name that matches hardbreak with type page, from Microsoft only
     */
    DOM.MS_HARDBREAK_TYPE_PAGE_CLASS_NAME = 'ms-hardbreak-page';

    /**
     * CSS class name that matches element which contains (is split by) page break.
     */
    DOM.CONTAINS_PAGEBREAK_CLASS_NAME = 'contains-pagebreak';

    /**
     * Returns whether the passed node is a page break element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a page break element.
     */
    DOM.isPageBreakNode = function (node) {
        return $(node).hasClass(DOM.PAGE_BREAK_CLASS_NAME);
    };

    /**
     * Returns whether the passed node is a page break element inserted by user.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a manual page break element.
     */
    DOM.isManualPageBreakNode = function (node) {
        return $(node).hasClass(DOM.MANUAL_PAGE_BREAK_CLASS_NAME);
    };

    /**
     * Returns whether the passed node is a hardbreak element with type page,
     * inserted in MS Word.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a MS Word page hardbreak element.
     */
    DOM.isMSPageHardbreakNode = function (node) {
        return $(node).hasClass(DOM.MS_HARDBREAK_TYPE_PAGE_CLASS_NAME);
    };

    /**
     * Returns whether the passed node contains a hardbreak element with type page,
     * inserted in MS Word.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains at least one MS Word page hardbreak element.
     */
    DOM.hasMSPageHardbreakNode = function (node) {
        return $(node).find(DOM.MS_HARDBREAK_TYPE_PAGE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is a table element that contains a page break
     * in its first paragraph in the upper left cell.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node with manual page break element.
     */
    DOM.isTableWithManualPageBreak = function (node) {
        return DOM.isTableNode(node) && $(node).find('.p').first().hasClass(DOM.MANUAL_PAGE_BREAK_CLASS_NAME);
    };

    /**
     * Returns whether node contains manual page break after element attribute,
     * set as a class name.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a table node with manual page break after element.
     * */
    DOM.isNodeWithPageBreakAfterAttribute = function (node) {
        return $(node).hasClass('manual-pb-after');
    };

    /**
     * Returns whether node contains page break inside itself.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains page break, or not.
     * */
    DOM.nodeContainsPageBreak = function (node) {
        return $(node).hasClass(DOM.CONTAINS_PAGEBREAK_CLASS_NAME);
    };

    // header/footer ==========================================================

    DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR = '.header-footer-placeholder';

    DOM.HEADER_FOOTER_PLACEHOLDER_CLASSNAME = 'header-footer-placeholder';

    DOM.HEADER_WRAPPER_SELECTOR = '.header-wrapper';

    DOM.HEADER_WRAPPER_CLASSNAME = 'header-wrapper';

    DOM.FOOTER_WRAPPER_SELECTOR = '.footer-wrapper';

    DOM.FOOTER_WRAPPER_CLASSNAME = 'footer-wrapper';

    DOM.MARGINAL_NODE_SELECTOR = '.marginal';

    DOM.MARGINAL_NODE_CLASSNAME = 'marginal';

    DOM.MARGINALCONTENT_NODE_SELECTOR = '.marginalcontent';

    DOM.MARGINALCONTENT_NODE_CLASSNAME = 'marginalcontent';

    /**
     * The class name used to specify header nodes.
     */
    DOM.HEADER_CLASSNAME = 'header';

    /**
     * The selector used to specify header nodes.
     */
    DOM.HEADER_SELECTOR = '.' + DOM.HEADER_CLASSNAME;

    /**
     * The class name used to specify footer nodes.
     */
    DOM.FOOTER_CLASSNAME = 'footer';

    /**
     * The selector used to specify footer nodes.
     */
    DOM.FOOTER_SELECTOR = '.' + DOM.FOOTER_CLASSNAME;

    /**
     * Selector used to specify content nodes that are direct children of page node.
     *
     */
    DOM.PAGE_CHILDREN_CONTENT_SELECTOR = DOM.HEADER_WRAPPER_SELECTOR + ', ' + DOM.PAGECONTENT_NODE_SELECTOR + ', ' + DOM.FOOTER_WRAPPER_SELECTOR + ', ' + DOM.DRAWINGLAYER_NODE_SELECTOR;

    /**
     * Checks if passed node is a header element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a header element.
     */
    DOM.isHeaderNode = function (node) {
        return $(node).hasClass(DOM.HEADER_CLASSNAME);
    };

    /**
     * Checks if passed node is a footer element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a footer element.
     */
    DOM.isFooterNode = function (node) {
        return $(node).hasClass(DOM.FOOTER_CLASSNAME);
    };

    /**
     * Checks if passed node is header or footer element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a header or footer element.
     */
    DOM.isHeaderOrFooter = function (node) {
        return DOM.isHeaderNode(node) || DOM.isFooterNode(node);
    };

    /**
     * Checks if passed node is content node inside header or footer element.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a header or footer marginal content element.
     */
    DOM.isMarginalContentNode = function (node) {
        return $(node).hasClass(DOM.MARGINALCONTENT_NODE_CLASSNAME);
    };

    /**
     * Returns content node of passed header or footer element.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Node|jQuery}
     *  If found, returns content node of passed header or footer element.
     */
    DOM.getMarginalContentNode = function (node) {
        return $(node).children(DOM.MARGINALCONTENT_NODE_SELECTOR);
    };

    /**
     * Returns the template folder for header and footer.
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node for the header and footer placeholders.
     */
    DOM.getMarginalTemplateNode = function (pageNode) {
        return $(pageNode).children(DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR);
    };

    /**
     * Checks if passed node is a header wrapper element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a header wrapper element.
     */
    DOM.isHeaderWrapper = function (node) {
        return $(node).hasClass(DOM.HEADER_WRAPPER_CLASSNAME);
    };

    /**
     * Returns the header wrapper node, that is a child of the page node.
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  The header wrapper node as a child of the page node.
     */
    DOM.getHeaderWrapperNode = function (pageNode) {
        return $(pageNode).children(DOM.HEADER_WRAPPER_SELECTOR);
    };

    /**
     * Checks if passed node is a footer wrapper element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a footer wrapper element.
     */
    DOM.isFooterWrapper = function (node) {
        return $(node).hasClass(DOM.FOOTER_WRAPPER_CLASSNAME);
    };

    /**
     * Returns the footer wrapper node, that is a child of the page node.
     *
     * @param {HTMLElement|jQuery} pageNode
     *  The page DOM node. If this object is a jQuery collection, uses the
     *  first DOM node it contains.
     *
     * @returns {jQuery}
     *  The footer wrapper node as a child of the page node.
     */
    DOM.getFooterWrapperNode = function (pageNode) {
        return $(pageNode).children(DOM.FOOTER_WRAPPER_SELECTOR);
    };

    /**
     * Checks if passed node is a header or footer wrapper element.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a header or footer wrapper element.
     */
    DOM.isHeaderOrFooterWrapper = function (node) {
        return DOM.isHeaderWrapper(node) || DOM.isFooterWrapper(node);
    };

    /**
     * Returns whether the passed node is an element representing a marginal
     * (header&footer) place holder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing a marginal placeholder
     *  node.
     */
    DOM.isMarginalPlaceHolderNode = function (node) {
        return $(node).hasClass(DOM.HEADER_FOOTER_PLACEHOLDER_CLASSNAME);
    };

    /**
     * Checks if passed node is positioned relative.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is 'relative' positioned.
     */
    DOM.isRelativePositionedNode = function (node) {
        return $(node).css('position') === 'relative';
    };

    /**
     * Returning the header or footer node that is wrapping a specified node.
     *
     * @param {Node|jQuery} rootNode
     *  The root node until which the search of header or footer is executed.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     *  Optional parameters:
     *  @param {Boolean} [options.footer=false]
     *    If set to true, the specified node is searched in the footer.
     *  @param {Boolean} [options.header=false]
     *    If set to true, the specified node is searched in the header.
     *
     * @returns {jQuery}
     *  A jQuery element, that contains the header or footer node or no node
     *  if the specified node is not inside a header or footer.
     */
    DOM.getMarginalTargetNode = function (rootNode, node, options) {

        var // finding the current node inside a footer
            findFooter = Utils.getBooleanOption(options, 'footer', false),
            // finding the current node inside a footer
            findHeader = Utils.getBooleanOption(options, 'header', false),
            // the selector to find the marginal node
            marginSelector = findFooter ? 'div.footer' : (findHeader ? 'div.header' : 'div.footer, div.header');

        return $(node).parentsUntil(rootNode, marginSelector);
    };

    /**
     * Returning the closest header or footer node that is wrapping a specified node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     *  Optional parameters:
     *  @param {Boolean} [options.footer=false]
     *    If set to true, the specified node is searched in the footer.
     *  @param {Boolean} [options.header=false]
     *    If set to true, the specified node is searched in the header.
     *
     * @returns {jQuery}
     *  A jQuery element, that contains the closest header or footer node or no
     *  node, if the specified node is not inside a header or footer.
     */
    DOM.getClosestMarginalTargetNode = function (node, options) {

        var // finding the current node inside a footer
            findFooter = Utils.getBooleanOption(options, 'footer', false),
            // finding the current node inside a footer
            findHeader = Utils.getBooleanOption(options, 'header', false),
            // the selector to find the marginal node
            marginSelector = findFooter ? 'div.footer' : (findHeader ? 'div.header' : 'div.footer, div.header');

        return $(node).closest(marginSelector);
    };

    /**
     * Returning whether a specified node is inside the header/footer
     * and has the class 'marginal'.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {jQuery}
     *  Whether the specified node is located inside header/footer (because
     *  it has class 'marginal'.
     */
    DOM.isMarginalNode = function (node) {
        return $(node).hasClass(DOM.MARGINAL_NODE_CLASSNAME);
    };

    /**
     * Returning whether specified node is marginal context menu goto or close.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked.
     * @returns {Boolean}
     *  If node is marginal context item or not.
     *
     */
    DOM.isMarginalContextItem = function (node) {
        return $(node).hasClass('marginal-menu-goto') || $(node).hasClass('marginal-menu-close');
    };

    /**
     * Returning whether a specified node is inside the header/footer
     * template folder (div.header-footer-placeholder).
     *
     * @param {Node|jQuery} rootNode
     *  The root node until which the search of header or footer is executed.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {jQuery}
     *  Whether the specified node is located inside header/footer
     *  template folder.
     */
    DOM.isInsideHeaderFooterTemplateNode = function (rootNode, node) {
        return $(node).parentsUntil(rootNode, DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR).length > 0;
    };

    /**
     * Getter for the current target ID of a specified node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The target of the current active root node as string. Returns an empty
     *  string, if there is no target attribute set.
     */
    DOM.getTargetContainerId = function (node) {
        return $(node).attr('data-container-id') || '';
    };

    /**
     * setter for the current target ID of a specified node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to which the container id will be assigned.
     *
     * @param {String}
     *  The container id, that will be assigned.
     */
    DOM.setTargetContainerId = function (node, id) {
        $(node).attr('data-container-id', id);
    };

    /**
     * Checks whether node of its type is first occurance in document.
     * Compares passed node with first in document with same target, and checks equality.
     *
     * @param {Node|jQuery} rootNode
     *  Parent root node of passed node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {String} [target]
     *  The target of the passed node as string. If missing calculate one from data attributes.
     *
     * @returns {Boolean}
     *
     */
    DOM.isFirstMarginalInDocument = function (rootNode, node, target) {

        target = target || DOM.getTargetContainerId(node);
        var $firstNode = rootNode.find('[data-container-id="' + target + '"]').first();

        return $(node)[0] === $firstNode[0];
    };

    /**
     * Getter the container id of the container node for a specified node located
     * inside a container. If it is not inside a container, null is returned.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The target id of the container node as string. Returns null, if no container
     *  node is found.
     */
    DOM.getNodeContainerId = function (rootNode, node) {

        var // the header or footer node, in which this node is located
            containerNode = DOM.getMarginalTargetNode(rootNode, node);

        return (containerNode.length > 0) ? DOM.getTargetContainerId(containerNode[0]) : null;
    };

    /**
     * Checks if container comment node of passed node has same id as given one.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {String} activeNodeId
     *  The target id to be compared, passed as string.
     *
     * @returns {Boolean}
     */
    DOM.isSurroundingCommentNodeActive = function (node, activeNodeId) {
        var surroundingNode = DOM.getSurroundingCommentNode(node);
        if (!surroundingNode || !surroundingNode.length) {
            return false;
        }
        return DOM.getTargetContainerId(surroundingNode) === activeNodeId;
    };

    // comments ==================================================

    /**
     * The CSS class used to mark comment nodes.
     *
     * @constant
     */
    DOM.COMMENTNODE_CLASS = 'comment';

    /**
     * The CSS class used to mark comment closer nodes.
     *
     * @constant
     */
    DOM.COMMENTCLOSERNODE_CLASS = 'commentcloser';

    /**
     * The CSS class used to mark comment reply nodes.
     *
     * @constant
     */
    DOM.COMMENTREPLYNODE_CLASS = 'commentreply';

    /**
     * The CSS class used to mark comment placeholder nodes. This place holder
     * nodes are located in the page content node and safe the position for
     * a comment in the comment layer.
     *
     * @constant
     */
    DOM.COMMENTPLACEHOLDER_CLASS = 'commentplaceholder';

    /**
     * The CSS class used to mark child comment nodes.
     *
     * @constant
     */
    DOM.CHILDCOMMENTNODE_CLASS = 'childcomment';

    /**
     * The CSS class used to mark parent comment nodes.
     *
     * @constant
     */
    DOM.PARENTCOMMENTNODE_CLASS = 'parentcomment';

    /**
     * The CSS class used to mark comment nodes with target.
     *
     * @constant
     */
    DOM.TARGETCOMMENTNODE_CLASS = 'targetcomment';

    /**
     * The CSS class used to mark comment thread nodes.
     *
     * @constant
     */
    DOM.COMMENTTHREADNODE_CLASS = 'commentthread';

    /**
     * The CSS class used to mark comment bubble nodes.
     *
     * @constant
     */
    DOM.COMMENTBUBBLENODE_CLASS = 'commentbubble';

    /**
     * The CSS class used to mark comment meta info nodes.
     *
     * @constant
     */
    DOM.COMMENTMETAINFO_CLASS = 'commentmetainfo';

    /**
     * The CSS class used to mark the page node, if it has a margin for the comment node.
     *
     * @constant
     */
    DOM.COMMENTMARGIN_CLASS = 'commentmargin';

    /**
     * A jQuery selector that matches nodes representing a comment.
     *
     * @constant
     */
    DOM.COMMENTNODE_SELECTOR = '.' + DOM.COMMENTNODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing a child comment.
     *
     * @constant
     */
    DOM.CHILDCOMMENTNODE_SELECTOR = '.' + DOM.CHILDCOMMENTNODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing a parent comment.
     *
     * @constant
     */
    DOM.PARENTCOMMENTNODE_SELECTOR = '.' + DOM.PARENTCOMMENTNODE_CLASS;

    /**
     * A jQuery selector that matches comment nodes inside header or footer.
     *
     * @constant
     */
    DOM.TARGETCOMMENTNODE_SELECTOR = '.' + DOM.TARGETCOMMENTNODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing a comment thread.
     *
     * @constant
     */
    DOM.COMMENTTHREADNODE_SELECTOR = '.' + DOM.COMMENTTHREADNODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing a comment bubble node.
     *
     * @constant
     */
    DOM.COMMENTBUBBLENODE_SELECTOR = '.' + DOM.COMMENTBUBBLENODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing a comment meta info node.
     *
     * @constant
     */
    DOM.COMMENTMETAINFO_SELECTOR = '.' + DOM.COMMENTMETAINFO_CLASS;

    /**
     * A jQuery selector that matches nodes representing a comment closer.
     *
     * @constant
     */
    DOM.COMMENTCLOSERNODE_SELECTOR = '.' + DOM.COMMENTCLOSERNODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing a comment reply node.
     *
     * @constant
     */
    DOM.COMMENTREPLYNODE_SELECTOR = '.' + DOM.COMMENTREPLYNODE_CLASS;

    /**
     * The data property name for registering the comment place holder at the absolutely
     * positioned comment. This works in both directions, because the comment place
     * holder has a data attribute with this name to the absolutely positioned comment
     * and vice versa.
     *
     * @constant
     */
    DOM.COMMENTPLACEHOLDER_LINK = 'commentplaceholderlink';

    /**
     * A jQuery selector that matches elements representing a comment place holder.
     */
    DOM.COMMENTPLACEHOLDER_NODE_SELECTOR = 'div.' + DOM.COMMENTPLACEHOLDER_CLASS;

    /**
     * The CSS class used to mark fields that are inside of comment nodes.
     *
     * @constant
     */
    DOM.FIELD_IN_COMMENT_CLASS = 'fieldInComment';

    /**
     * Returns whether the passed node is a comment node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a comment node.
     */
    DOM.isCommentNode = function (node) {
        return $(node).hasClass(DOM.COMMENTNODE_CLASS);
    };

    /**
     * Returns whether the passed node is a comment thread node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a comment thread node.
     */
    DOM.isCommentThreadNode = function (node) {
        return $(node).hasClass(DOM.COMMENTTHREADNODE_CLASS);
    };

    /**
     * Returns whether the passed node is a comment bubble node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a comment bubble node.
     */
    DOM.isCommentBubbleNode = function (node) {
        return $(node).hasClass(DOM.COMMENTBUBBLENODE_CLASS);
    };

    /**
     * Returns whether the passed node is a parent comment node in a comment thread.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a parent comment node in a comment thread.
     */
    DOM.isParentCommentNode = function (node) {
        return $(node).hasClass(DOM.PARENTCOMMENTNODE_CLASS);
    };

    /**
     * Returns whether the passed node is a comment node with a specified target.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a comment node with a specified target. Marked in
     *  this way are only comments in header or footer (odt only).
     */
    DOM.isTargetCommentNode = function (node) {
        return $(node).hasClass(DOM.TARGETCOMMENTNODE_CLASS);
    };

    /**
     * Returns whether the passed node is a child comment node in a comment thread.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a child comment node in a comment thread.
     */
    DOM.isChildCommentNode = function (node) {
        return $(node).hasClass(DOM.CHILDCOMMENTNODE_CLASS);
    };

    /**
     * Returns whether the passed node is a comment closer node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a comment closer node.
     */
    DOM.isCommentCloserNode = function (node) {
        return $(node).hasClass(DOM.COMMENTCLOSERNODE_CLASS);
    };

    /**
     * Returns whether the passed node is a comment reply node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a comment reply node.
     */
    DOM.isCommentReplyNode = function (node) {
        return $(node).hasClass(DOM.COMMENTREPLYNODE_CLASS);
    };

    /**
     * Returns the meta info node inside the specified node. To find a valid
     * meta info node, the specified node must be the comment node itself.
     * Otherwise null is returned.
     *
     * @param {Node|jQuery|Null} [node]
     *  The comment node.
     *
     * @returns {Node|Null}
     *  The comment meta info node inside the specified comment node. Or null,
     *  if the specified node is not a comment node.
     */
    DOM.getCommentMetaInfoNode = function (node) {
        return DOM.isCommentNode(node) ? $(node).children(DOM.COMMENTMETAINFO_SELECTOR) : null;
    };

    /**
     * Searching the place holder node for a specific comment node. This place
     * holder node exists in the page content for all comments inside the
     * comment layer node. This also works vice versa: Searching the comment
     * node in the comment layer for a specified place holder. In both cases
     * the 'data' attribute is named DOM.COMMENTPLACEHOLDER_LINK.
     *
     * @param {Node|jQuery|Null} [node]
     *  The comment node.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.ignoreMissing=false]
     *      If set to true, it is valid, if the comment node for a place holder
     *      not cannot be found (special handling during removal of marginal
     *      comments).
     *
     * @returns {Node}
     *  The place holder node for the comment node in the comment layer node
     *  or the comment in the comment layer node itself.
     */
    DOM.getCommentPlaceHolderNode = function (node, options) {

        var // the node saved in the data object
            searchedNode = $(node).data(DOM.COMMENTPLACEHOLDER_LINK),
            // the common parent of comment layer and header or footer node
            searchRootNode = null,
            // the ID of the drawing in the drawing layer
            commentID = 0,
            // a string to find the related placeholder node
            selectorString = null,
            // the nodes with selector string inside header or footer
            foundNodes = null,
            // the root node in header or footer
            marginalRoot = null,
            // whether it is valid, if a comment node cannot be found
            ignoreMissing = false,
            // the target of the marginal container
            targetString = '',
            // a selector for the searched dom node
            searchSelector = null;

        if (searchedNode && !_.isString(searchedNode)) { return searchedNode; }

        // comments are not supported in header and footer yet, but for 'delete all comments', it
        // is necessary to find the correct position of the place holder and the range marker nodes.

        if (_.isString(searchedNode)) {

            targetString = searchedNode;  // saving the target
            searchSelector = DOM.COMMENTPLACEHOLDER_NODE_SELECTOR;  // -> searching for the comment place holder
            searchRootNode = $(node).closest(DOM.PAGE_NODE_SELECTOR);
            // selecting the first header or footer with correct id as root node
            searchRootNode = searchRootNode.find('.header, .footer').filter('[data-container-id="' + searchedNode + '"]').first();

        } else if (DOM.isCommentPlaceHolderNode(node)) {

            ignoreMissing = Utils.getBooleanOption(options, 'ignoreMissing', false);

            // check, if the place holder is inside header or footer
            marginalRoot = DOM.getClosestMarginalTargetNode(node);

            // only expand search, if the place holder is inside header or footer
            if (marginalRoot && marginalRoot.length > 0) {
                searchSelector = DOM.COMMENTNODE_SELECTOR;  // -> searching for the comment
                searchRootNode = $(node).closest(DOM.PAGE_NODE_SELECTOR); // this is a comment place holder node (in header or footer)
                searchRootNode = DOM.getCommentLayerNode(searchRootNode);  // restricting search to comment layer
            }
        }

        if (searchRootNode && searchRootNode.length > 0) {

            commentID = DOM.getTargetContainerId(node);
            selectorString = '[data-container-id=' + commentID + ']';
            foundNodes = searchRootNode.find(selectorString);

            if (foundNodes && foundNodes.length > 0) {
                searchedNode = _.find(foundNodes, function (node) {
                    return $(node).is(searchSelector);
                });

                if (searchedNode && $(searchedNode).length > 0) {
                    // saving the target string at the place holder node for later usage
                    if (targetString) { $(searchedNode).data(DOM.DATA_TARGETSTRING_NAME, targetString); }
                } else {
                    Utils.warn('Warning: failed to find comment or comment place holder node with selector: ' + searchSelector);
                }
            } else {
                if (!ignoreMissing) { Utils.warn('Warning: Failed to find comment or comment place holder node with selector: ' + selectorString); }
            }

        } else {
            // outside of header or footer the node must be found from data object
            if (!searchedNode && !ignoreMissing) { Utils.warn('Warning: Failed to find comment or comment place holder node.'); }
        }

        return searchedNode;
    };

    /**
     * Returns whether the passed node is an element representing a comment
     * place holder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing a comment placeholder
     *  node.
     */
    DOM.isCommentPlaceHolderNode = function (node) {
        return $(node).hasClass(DOM.COMMENTPLACEHOLDER_CLASS);
    };

    /**
     * Returns whether the passed node is an element representing a comment
     * place holder node or a drawing place holder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element representing a comment placeholder
     *  node or a drawing place holder node.
     */
    DOM.isSubstitutionPlaceHolderNode = function (node) {
        return DOM.isCommentPlaceHolderNode(node) || DOM.isDrawingPlaceHolderNode(node);
    };

    /**
     * Returns whether the passed node is an element representing a top level node
     * like a paragraph or a table inside a comment node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a top level node inside a comment node.
     */
    DOM.isTopLevelNodeInComment = function (node) {
        return node && DOM.isCommentNode(Utils.getDomNode(node).parentNode.parentNode.parentNode);
    };

    /**
     * Returns whether the passed node (a paragraph) has at least one child
     * that is a comment place holder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node has at least one child representing a comment
     *  place holder node.
     */
    DOM.hasCommentPlaceHolderNode = function (node) {
        return $(node).children(DOM.COMMENTPLACEHOLDER_NODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed node is inside a comment node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside a comment node.
     */
    DOM.isNodeInsideComment = function (node) {
        return node && $(node).length > 0 && $(node).closest(DOM.COMMENTNODE_SELECTOR).length > 0;
    };

    /**
     * Returns whether the passed field node is inside a comment node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside a comment node.
     */
    DOM.isFieldInsideComment = function (node) {
        return node && $(node).hasClass(DOM.FIELD_IN_COMMENT_CLASS);
    };

    /**
     * Returns the comment node, that is that includes a specified node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The comment node, that includes the specified node. Or an empty jQuery
     *  object, if the specified node is not included in a comment node.
     */
    DOM.getSurroundingCommentNode = function (node) {
        return node && $(node).length > 0 && $(node).closest(DOM.COMMENTNODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is inside a comment meta info node.
     * This is the header of a comment node, that contains the comment fuctionality.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside a comment meta info node.
     */
    DOM.isNodeInsideCommentMetaInfo = function (node) {
        return node && $(node).length > 0 && $(node).closest(DOM.COMMENTMETAINFO_SELECTOR).length > 0;
    };

    /**
     * Returns the author for a specified comment node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The name of the comment author.
     */
    DOM.getCommentAuthor = function (node) {
        return (node && $(node).attr('data-container-author')) || '';
    };

    /**
     * Returns the author's uid for a specified comment node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The uid of the comment author.
     */
    DOM.getCommentAuthorUid = function (node) {
        return (node && $(node).attr('data-container-uid')) || '';
    };

    /**
     * Returns the date for a specified comment node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The date of the comment node.
     */
    DOM.getCommentDate = function (node) {
        return (node && $(node).attr('data-container-date')) || '';
    };

    // range markers ==================================================

    /**
     * The CSS class used to mark range marker nodes.
     *
     * @constant
     */
    DOM.RANGEMARKERNODE_CLASS = 'rangemarker';

    /**
     * A jQuery selector that matches nodes representing a range marker node.
     *
     * @constant
     */
    DOM.RANGEMARKERNODE_SELECTOR = '.' + DOM.RANGEMARKERNODE_CLASS;

    /**
     * The CSS class used to mark range marker nodes at the beginning of the range.
     *
     * @constant
     */
    DOM.RANGEMARKER_STARTTYPE_CLASS = 'rangestart';

    /**
     * A jQuery selector that matches nodes representing a range marker start node.
     *
     * @constant
     */
    DOM.RANGEMARKER_STARTTYPE_SELECTOR = '.' + DOM.RANGEMARKER_STARTTYPE_CLASS;

    /**
     * The CSS class used to mark range marker nodes at the end of the range.
     *
     * @constant
     */
    DOM.RANGEMARKER_ENDTYPE_CLASS = 'rangeend';

    /**
     * A jQuery selector that matches nodes representing a range marker end node.
     *
     * @constant
     */
    DOM.RANGEMARKER_ENDTYPE_SELECTOR = '.' + DOM.RANGEMARKER_ENDTYPE_CLASS;

    /**
     * The CSS class used to mark the range overlay node.
     *
     * @constant
     */
    DOM.RANGEMARKEROVERLAYNODE_CLASS = 'rangemarkeroverlay';

    /**
     * The key name for the data object that is assigned to the comment place holder,
     * if it is located in header or footer. This assignment happens in the function
     * DOM.getCommentPlaceHolderNode(). The value is the string with the target name
     * of the header or footer node. This value can be used to obtain the target node
     * after using DOM.getCommentPlaceHolderNode().
     *
     * @constant
     */
    DOM.DATA_TARGETSTRING_NAME = 'data_targetstring';

    /**
     * Returns whether the passed node is a range marker node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is a range marker node.
     */
    DOM.isRangeMarkerNode = function (node) {
        return $(node).hasClass(DOM.RANGEMARKERNODE_CLASS);
    };

    /**
     * Returns whether the passed node is a range marker start node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is a range marker start node.
     */
    DOM.isRangeMarkerStartNode = function (node) {
        return DOM.isRangeMarkerNode(node) && $(node).hasClass(DOM.RANGEMARKER_STARTTYPE_CLASS);
    };

    /**
     * Returns whether the passed node is a range marker end node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is a range marker end node.
     */
    DOM.isRangeMarkerEndNode = function (node) {
        return DOM.isRangeMarkerNode(node) && $(node).hasClass(DOM.RANGEMARKER_ENDTYPE_CLASS);
    };

    /**
     * Returns whether the passed node contains at least one range marker node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains a range marker node.
     */
    DOM.hasRangeMarkerNode = function (node) {
        return $(node).has(DOM.RANGEMARKERNODE_SELECTOR).length > 0;
    };

    /**
     * Getter for the type a specified range node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The type of the specified range node. Returns an empty
     *  string, if there is no type specified.
     */
    DOM.getRangeMarkerType = function (node) {
        return $(node).attr('data-range-type') || '';
    };

    /**
     * Getter for the current unique ID of a specified range node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The unique id of the specified range node. Returns an empty
     *  string, if there is no unique id.
     */
    DOM.getRangeMarkerId = function (node) {
        return $(node).attr('data-range-id') || '';
    };

    /**
     * Returns whether the passed node is a range marker node of type 'comment'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a range marker node of type comment.
     */
    DOM.isCommentRangeMarkerNode = function (node) {
        return DOM.isRangeMarkerNode(node) && DOM.getRangeMarkerType() === 'comment';
    };

    /**
     * Checks if the passed node contains a range marker node with checkBox field character attribute.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains a range marker node with checkBox field character attribute.
     */
    DOM.isRangeStartWithCheckboxAttr = function (node) {
        var characterAttrs = node && AttributeUtils.getExplicitAttributes(node, { family: 'character', direct: true });
        var fieldAttrs = characterAttrs && characterAttrs.field;

        return DOM.isRangeMarkerStartNode(node) && fieldAttrs && fieldAttrs.formFieldType === 'checkBox';
    };

    /**
     * Checks if the passed node contains a range marker node with checkBox field character attribute,
     * and checkbox has state enabled (checked).
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains a range marker node with checkBox field character attribute and checked is true.
     */
    DOM.isRangeStartWithCheckboxEnabledState = function (node) {
        var characterAttrs = node && AttributeUtils.getExplicitAttributes(node, { family: 'character', direct: true });
        var fieldAttrs = characterAttrs && characterAttrs.field;

        return DOM.isRangeMarkerStartNode(node) && fieldAttrs && fieldAttrs.formFieldType === 'checkBox' && fieldAttrs.checked === true;
    };

    /**
     * Returns whether the passed node with the passed offset is the last
     * text position in front of an range marker end node.
     *
     * @param {Node|Null} node
     *  The DOM node to be checked. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node with the passed offset is the last text
     *  position before a range marker end node.
     */
    DOM.isLastTextPositionBeforeRangeEnd = function (node, offset) {
        return node.nodeType === 3 && (node.length === offset) && node.parentNode.nextSibling && DOM.isRangeMarkerEndNode(node.parentNode.nextSibling);
    };

    /**
     * Returns whether the passed node with the passed offset is the first
     * text position behind a range marker start node or behind a complex field
     * node.
     *
     * TODO: Handling for comments in ODT, where the range marker start node is the comment node itself
     *
     * @param {Node|Null} node
     *  The DOM node to be checked. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node with the passed offset is the first text
     *  position after a range marker start node or a complex field node.
     */
    DOM.isFirstTextPositionBehindRangeStart = function (node, offset) {
        return (offset === 0) && (node.nodeType === 3) && node.parentNode.previousSibling && (DOM.isRangeMarkerStartNode(node.parentNode.previousSibling) || DOM.isComplexFieldNode(node.parentNode.previousSibling));
    };

    // simple fields ===================================================

    /**
     * Getter for the instruction of a specified simple field node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The instruction for the specified simple field node. Returns an empty
     *  string, if there is no instruction.
     */
    DOM.getFieldInstruction = function (node) {
        return $(node).data('type') || '';
    };

    /**
     * Getter for the date/time format of a specified simple field node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The format for the specified simple field node. Returns an empty
     *  string, if there is no format.
     */
    DOM.getFieldDateTimeFormat = function (node) {
        return $(node).data('dateFormat') || '';
    };

    /**
     * Getter for the PAGE NUMBER and PAGE COUNT format of a specified simple field node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The format for the specified simple field node. Returns an empty
     *  string, if there is no format.
     */
    DOM.getFieldPageFormat = function (node) {
        return $(node).data('pageNumFormat') || '';
    };

    /**
     * Checks if passed field node has fixed property,
     * that prevents automatic date value update.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  If the field has fixed date property or not
     */
    DOM.isAutomaticDateField = function (node) {
        return $(node).attr('data-auto-date') === 'true';
    };

    /**
     * Checks if passed ODF or simple ooxml field node has fixed property,
     * that prevents automatic date value update.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Boolean} isODF
     *  If document format is odf, or ooxml.
     *
     * @returns {Boolean}
     *  If the field has fixed date property or not
     */
    DOM.isFixedSimpleField = function (node, isODF) {
        var isFixed = false;
        var nodeAttrs;

        if (isODF) {
            nodeAttrs = (node && _.isObject($(node).data().attributes)) ? $(node).data().attributes : false;
            isFixed = $(node).data('text:fixed') === 'true' || (nodeAttrs && nodeAttrs.character && nodeAttrs.character.field && nodeAttrs.character.field['text:fixed'] === 'true');
        } else {
            isFixed = !DOM.isAutomaticDateField(node);
        }

        return isFixed;
    };

    /**
     * Getter for the current unique ID of a specified simple field node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The unique id of the specified simple field node. Returns an empty
     *  string, if there is no unique id.
     */
    DOM.getSimpleFieldId = function (node) {
        return $(node).data('sFieldId') || '';
    };

    /**
     * Getter for the current unique ID of a specified Slide field node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The unique id of the specified simple field node. Returns an empty
     *  string, if there is no unique id.
     */
    DOM.getSlideFieldId = function (node) {
        return $(node).attr('data-fid') || '';
    };

    /**
     * Returns whether the passed node contains at least one simple field node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains at least one simple field node.
     */
    DOM.hasSimpleFieldNode = function (node) {
        return $(node).has(DOM.FIELD_NODE_SELECTOR).length > 0;
    };

    // complex fields ==================================================

    /**
     * The CSS class used to mark complex field nodes.
     *
     * @constant
     */
    DOM.COMPLEXFIELDNODE_CLASS = 'complexfield';

    /**
     * The CSS class used to mark all nodes that are between a range start node
     * for a complex field and a range end node for a complex field.
     *
     * @constant
     */
    DOM.COMPLEXFIELDMEMBERNODE_CLASS = 'complex-field';

    /**
     * A jQuery selector that matches nodes representing a complex field node.
     *
     * @constant
     */
    DOM.COMPLEXFIELDNODE_SELECTOR = '.' + DOM.COMPLEXFIELDNODE_CLASS;

    /**
     * A jQuery selector that matches nodes that are located between a start range
     * node and an end range node of a complex field.
     *
     * @constant
     */
    DOM.COMPLEXFIELDMEMBERNODE_SELECTOR = '.' + DOM.COMPLEXFIELDMEMBERNODE_CLASS;

    /**
     * Returns whether the passed node is a complex field node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is a complex field node.
     */
    DOM.isComplexFieldNode = function (node) {
        return $(node).hasClass(DOM.COMPLEXFIELDNODE_CLASS);
    };

    /**
     * Returns whether the passed node is inside a complex field range.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside a complex field range.
     */
    DOM.isInsideComplexFieldRange = function (node) {
        return $(node).hasClass(DOM.COMPLEXFIELDMEMBERNODE_CLASS);
    };

    /**
     * Returns whether the passed node is inside a complex field range
     * of type 'PLACEHOLDER'.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is inside a placeholder complex field range.
     */
    DOM.isInsidePlaceHolderComplexField = function (node) {

        var // the complex field node
            complexFieldNode = null,
            // the instruction for the complex field
            fieldInstruction = null;

        if (DOM.isInsideComplexFieldRange(node)) {

            complexFieldNode = DOM.getPreviousComplexFieldNode(node);

            if (complexFieldNode) {
                fieldInstruction = DOM.getComplexFieldInstruction(complexFieldNode);

                if (fieldInstruction && Utils.trimString(fieldInstruction).indexOf('PLACEHOLDER ') === 0) {
                    return true;
                }
            }
        }

        return false;
    };

    /**
     * Returns the complex field node the preceeds the passed node as sibling.
     *
     * @param {Node|jQuery} node
     *  The DOM node whose previous matching sibling will be returned. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @returns {Node|Null}
     *  The previous matching complex field sibling node of the passed node;
     *  or null, if no previous sibling node has been found.
     */
    DOM.getPreviousComplexFieldNode = function (node) {
        return Utils.findPreviousSiblingNode(node, DOM.COMPLEXFIELDNODE_SELECTOR);
    };

    /**
     * Returns whether the passed node contains at least one complex field node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains at least one complex field node.
     */
    DOM.hasComplexFieldNode = function (node) {
        return $(node).has(DOM.COMPLEXFIELDNODE_SELECTOR).length > 0;
    };

    /**
     * Getter for the current unique ID of a specified complex field node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The unique id of the specified complex field node. Returns an empty
     *  string, if there is no unique id.
     */
    DOM.getComplexFieldId = function (node) {
        return $(node).attr('data-field-id') || '';
    };

    /**
     * Getter for the instruction of a specified complex field node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The instruction for the specified complex field node. Returns an empty
     *  string, if there is no instruction.
     */
    DOM.getComplexFieldInstruction = function (node) {
        return $(node).data('fieldInstruction') || '';
    };

    /**
     * Getter for the complex field id of a member node of a complex field range.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String}
     *  The id of the complex field, that contain the specified node between its
     *  start and its end range. Or an empty string, if the node is not member
     *  of a complex field range.
     */
    DOM.getComplexFieldMemberId = function (node) {
        return $(node).data('fieldId') || '';
    };

    // bookmark =======================================================

    /**
     * The CSS class used to mark bookmark node.
     *
     * @constant
     */
    DOM.BOOKMARKNODE_CLASS = 'bookmark';

    /**
     * A jQuery selector that matches nodes representing a bookmark node.
     *
     * @constant
     */
    DOM.BOOKMARKNODE_SELECTOR = '.' + DOM.BOOKMARKNODE_CLASS;

    /**
     * Returns whether the passed node is a bookmark node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is a bookmark node.
     */
    DOM.isBookmarkNode = function (node) {
        return $(node).hasClass(DOM.BOOKMARKNODE_CLASS);
    };

    /**
     * Returns id of bookmark.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {String}
     */
    DOM.getBookmarkId = function (node) {
        return $(node).attr('bmId');
    };

    /**
     * Returns anchor name of the given bookmark node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {String}
     */
    DOM.getBookmarkAnchor = function (node) {
        return $(node).attr('anchor');
    };

    /**
     * Returns position type of given bookmark node. Can be 'start' or 'end'.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {String}
     */
    DOM.getBookmarkPosition = function (node) {
        return $(node).attr('bmPos');
    };

    /**
     * Returns whether given bookmark node has position 'start'.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     */
    DOM.isBookmarkStartMarker = function (node) {
        return DOM.getBookmarkPosition(node) === 'start';
    };

    /**
     * Returns whether given bookmark node has position 'end'.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     */
    DOM.isBookmarkEndMarker = function (node) {
        return DOM.getBookmarkPosition(node) === 'end';
    };

    // spell checker ==================================================

    /**
     * The CSS class used to mark spell error nodes.
     *
     * @constant
     */
    DOM.SPELLERRORNODE_CLASS = 'spellerror';

    /**
     * A jQuery selector that matches nodes representing a spell error node.
     *
     * @constant
     */
    DOM.SPELLERRORNODE_SELECTOR = '.' + DOM.SPELLERRORNODE_CLASS;

    /**
     * Returns whether the passed node is a spell error node.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is a spell error node.
     */
    DOM.isSpellerrorNode = function (node) {
        return $(node).is(DOM.SPELLERRORNODE_SELECTOR);
    };

    // misc =================================================================

    /**
     * Returns, if the specified node requires that an artifical cursor is drawn.
     * This happens in Chrome for empty text spans at the end of the paragraph
     * behind selected nodes (26292).
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether this is a node, behind that an artifical cursor needs to be drawn.
     */
    DOM.isCursorSimulationNode = function (node) {
        return DOM.isInlineComponentNode(node) && (DrawingFrame.isDrawingFrame(node) || DOM.isSubstitutionPlaceHolderNode(node) || DOM.isRangeMarkerNode(node));
    };

    /**
     * Returns, whether the specified node has a length of 0 for the calculation
     * of the logical positions.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether this is a node, that does not influence the logical position.
     */
    DOM.isZeroLengthNode = function (node) {
        return DOM.isListLabelNode(node) || DOM.isPageBreakNode(node) || DOM.isDrawingSpaceMakerNode(node);
    };

    /**
     * Returns, whether the specified node has a length of 1 for the calculation
     * of the logical positions.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether this is a node, that does has a length of 1 for calculating the logical position.
     */
    DOM.isOneLengthNode = function (node) {
        return DOM.isDrawingFrame(node) || DOM.isTabNode(node) || DOM.isFieldNode(node) || DOM.isSubstitutionPlaceHolderNode(node) || DOM.isRangeMarkerNode(node);
    };

    /**
     * Checking, if the specified point describes a first position inside a text span that directly
     * follows the specified node. So the point is the first position behind the node.
     *
     * @param {DOM.Point} point
     *  The point, that will be checked, if it is directly following the specified node. It needs
     *  the properties 'offset' and 'node'.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the point describes the first position behind the node.
     */
    DOM.isFirstTextPositionBehindNode = function (point, node) {
        return point && point.offset === 0 && point.node.nodeType === 3 && point.node.parentNode && (point.node.parentNode.previousSibling === Utils.getDomNode(node));
    };

    /**
     * Returns, whether the specified node can be included inside a word, without
     * creating a word boundary.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether this is a node, that can be included inside one word, without creating
     *  a word boundary.
     */
    DOM.isValidNodeInsideWord = function (node) {
        return DOM.isSubstitutionPlaceHolderNode(node) || DOM.isRangeMarkerNode(node) || DOM.isDrawingSpaceMakerNode(node);
    };

    /**
     * An attribute string to replace 'src'
     * it is saved so in local storage
     * and will be replace with 'src' when load again
     *
     * @constant
     */
    DOM.LOCALSTORAGE_SRC_OVERRIDE = '__source__';

    /**
     * Returns the parents of the passed node until the endNodes validators
     * stop it or the parent node will be undefined.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Array} [options.endNodes='DOM.APPCONTENT_NODE_SELECTOR']
     *      Validators for iterating the parent nodes.
     *      By default it's the app-content node selector.
     *
     * @returns {Array}
     *  The array with all reached parents of the passed node
     */
    DOM.getDomTree = function (node, options) {
        var endNodes = Utils.getArrayOption(options, 'endNodes', [DOM.APPCONTENT_NODE_SELECTOR]),
            parent = $(node),
            returnArr = [];

        function check(ele) {
            var returnVal = true;

            _.each(endNodes, function (item) {
                if (_.isFunction(item)) {
                    if (item.call(this, ele) === true) {
                        returnVal = false;
                    }

                } else {
                    if ($.inArray(ele, $(item)) !== -1) {
                        returnVal = false;
                    }
                }

            });
            return returnVal;
        }

        while (check(parent.first()[0]) && !_.isUndefined(parent.first()[0])) {
            returnArr.push(parent.first()[0]);
            parent = parent.parent();
        }
        returnArr.push(parent.first()[0]);

        return returnArr;
    };

    // presentation specific functions ==================================================

    /**
     * The CSS class used to mark slide container nodes.
     *
     * @constant
     */
    DOM.SLIDE_CLASS = 'slide';

    /**
     * A jQuery selector that matches nodes representing slide container classes.
     *
     * @constant
     */
    DOM.SLIDE_SELECTOR = '.' + DOM.SLIDE_CLASS;

    /**
     * The CSS class used to mark slide container nodes.
     *
     * @constant
     */
    DOM.SLIDECONTAINER_CLASS = 'slidecontainer';

    /**
     * A jQuery selector that matches nodes representing slide container classes.
     *
     * @constant
     */
    DOM.SLIDECONTAINER_SELECTOR = '.' + DOM.SLIDECONTAINER_CLASS;

    /**
     * The class name representing the background node.
     *
     * @constant
     */
    DOM.BACKGROUNDSLIDE_CLASS = 'backgroundslide';

    /**
     * The class name representing the image inside background node.
     *
     * @constant
     */
    DOM.BACKGROUNDIMAGE_CLASS = 'backgroundimage';

    /**
     * The class name representing the layout slide layer node.
     *
     * @constant
     */
    DOM.LAYOUTSLIDELAYER_CLASS = 'layoutslidelayer';

    /**
     * A jQuery selector that matches nodes representing the layout slide layer node.
     *
     * @constant
     */
    DOM.LAYOUTSLIDELAYER_SELECTOR = '.' + DOM.LAYOUTSLIDELAYER_CLASS;

    /**
     * The class name representing the master slide layer node.
     *
     * @constant
     */
    DOM.MASTERSLIDELAYER_CLASS = 'masterslidelayer';

    /**
     * A jQuery selector that matches nodes representing the master slide layer node.
     *
     * @constant
     */
    DOM.MASTERSLIDELAYER_SELECTOR = '.' + DOM.MASTERSLIDELAYER_CLASS;

    /**
     * A jQuery selector that matches the layout slide layer and the master slide layer node.
     */
    DOM.LAYOUT_MASTER_LAYER_SELECTOR = DOM.LAYOUTSLIDELAYER_SELECTOR + ', ' + DOM.MASTERSLIDELAYER_SELECTOR;

    /**
     * The class name representing a the template buttons in empty place holder drawings.
     */
    DOM.PLACEHOLDER_TEMPLATE_BUTTON_CLASS = 'placeholdertemplatebutton';

    /**
     * The class name representing an image template buttons in empty place holder drawings.
     */
    DOM.IMAGE_TEMPLATE_BUTTON_CLASS = 'imagetemplatebutton';

    /**
     * The selector for image template buttons in empty place holder drawings.
     */
    DOM.IMAGE_TEMPLATE_BUTTON_SELECTOR = '.' + DOM.IMAGE_TEMPLATE_BUTTON_CLASS;

    /**
     * The class name representing a table template button in empty place holder drawings.
     */
    DOM.TABLE_TEMPLATE_BUTTON_CLASS = 'tabletemplatebutton';

    /**
     * The selector for table template buttons in empty place holder drawings.
     */
    DOM.TABLE_TEMPLATE_BUTTON_SELECTOR = '.' + DOM.TABLE_TEMPLATE_BUTTON_CLASS;

    /**
     * The class name representing a table template button in empty place holder drawings.
     */
    DOM.LIVE_PREVIEW_SLIDE_BACKGROUND_CLASS = 'live-preview-background';

    /**
     * The selector for table template buttons in empty place holder drawings.
     */
    DOM.LIVE_PREVIEW_SLIDE_BACKGROUND_SELECTOR = '.' + DOM.LIVE_PREVIEW_SLIDE_BACKGROUND_CLASS;

    /**
     * Creating a new empty slide node (with a paragraph as template)
     *
     * @returns {jQuery}
     *  A slide element, as jQuery object.
     */
    DOM.createSlideNode = function () {
        return DOM.createParagraphNode().addClass(DOM.SLIDE_CLASS);
    };

    /**
     * Returns, whether the specified node is a slide node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether this is a slide node.
     */
    DOM.isSlideNode = function (node) {
        return $(node).is(DOM.SLIDE_SELECTOR);
    };

    /**
     * Returns, whether the specified node is a container for a slide node.
     *
     * @param {Node|jQuery|Null} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether this is a slide container node.
     */
    DOM.isSlideContainerNode = function (node) {
        return $(node).is(DOM.SLIDECONTAINER_SELECTOR);
    };

    /**
     * Returns whether the passed node is within a slide container node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is within a comment layer node.
     */
    DOM.isInsideSlideContainerNode = function (node) {
        return $(node).closest(DOM.SLIDECONTAINER_SELECTOR).length > 0;
    };

    /**
     * Returns temporary slide node for background preview, while dialog is open.
     * @param  {jQuery} slideNode current slide node
     * @return {jQuery} if found, temporary slide node for background preview.
     */
    DOM.getTempSlideNodeForSlide = function (slideNode) {
        return $(slideNode).parent().children('.temp-preview-slide');
    };

    /**
     * Returns whether the passed node is a template button in an empty place holder drawing.
     *
     * @param {Node} node
     *  The node that will be checked.
     *
     * @returns {Boolean}
     *  Whether the specified node is a template button.
     */
    DOM.isPlaceholderTemplateButton = function (node) {
        return $(node).hasClass(DOM.PLACEHOLDER_TEMPLATE_BUTTON_CLASS);
    };

    /**
     * Returns whether the passed node is a template button for inserting images
     * in an empty place holder drawing.
     *
     * @param {Node} node
     *  The node that will be checked.
     *
     * @returns {Boolean}
     *  Whether the specified node is a template button for inserting images.
     */
    DOM.isImagePlaceholderTemplateButton = function (node) {
        return $(node).hasClass(DOM.IMAGE_TEMPLATE_BUTTON_CLASS);
    };

    /**
     * Returns whether the passed node is a template button for inserting tables
     * in an empty place holder drawing.
     *
     * @param {Node} node
     *  The node that will be checked.
     *
     * @returns {Boolean}
     *  Whether the specified node is a template button for inserting tables.
     */
    DOM.isTablePlaceholderTemplateButton = function (node) {
        return $(node).hasClass(DOM.TABLE_TEMPLATE_BUTTON_CLASS);
    };

    /**
     * Returns whether the passed node is set as hyperlink.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is set as hyperlink.
     */
    DOM.isHyperlinkNode = function (node) {
        var characterAttrs = node && AttributeUtils.getExplicitAttributes(node, { family: 'character', direct: true });

        return !!(characterAttrs && _.isString(characterAttrs.url));
    };

    /**
     *
     */
    DOM.getComputedElementStyles = createGetComputedElementStyles(window, _);

    /**
     *
     */
    DOM.getComputedElementCssText = function (elm, styleNameFilter) {
        return DOM.getComputedElementStyles(elm, styleNameFilter).join(';');
    };

    // private ----------------------------------------------------------------
    function createGetComputedElementStyles(global, _) {
        var
            getComputedElementStyles = function () { return []; };

        if (global.getComputedStyle) { // State Of The Art DOM

            getComputedElementStyles = function (elm, styleNameFilter) {
                var
                    cssStyleDeclarationCollection = global.getComputedStyle(elm, ''),
                    styleNameList                 = _.toArray(cssStyleDeclarationCollection);

                if (_.isFunction(styleNameFilter)) {
                    styleNameList                 = _.filter(styleNameList, styleNameFilter);
                }
                return _.map(styleNameList, function (styleName) {
                    return [

                        styleName,
                        ':',
                        cssStyleDeclarationCollection.getPropertyValue(styleName)

                    ].join('');
                });
            };
        }
        return getComputedElementStyles;
    }

    // exports ================================================================

    return DOM;
});
