/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/textframework/selection/remoteselection', [
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/utils/drawingutils'
], function (BaseObject, Config, Utils, Position, AttributeUtils, DrawingFrame, DrawingUtils) {

    'use strict';

    // class RemoteSelection ==================================================

    /**
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {TextBaseModel} docModel
     *  The document model instance.
     */
    function RemoteSelection(docModel) {

        var // the local selection instance
            selection = null,

            // the application instance
            app = docModel.getApp(),

            // the root node of the document
            rootNode = docModel.getNode(),

            // slide mode for spreadsheet/presentation
            slideMode = docModel.useSlideMode(),

            // the display timer of the collaborative overlay
            overlayTimer = null,

            // list with all drawings nodes with mouseenter and mouseleave handlers with the namespace 'remoteselection'
            drawingNodesWithHandlers = [],

            self = this;

        // base constructor ---------------------------------------------------

        BaseObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Calculate physical positions from logical selection.
         *
         * @param {DOM.Point} anchorPoint
         *  the selection start point
         *
         * @param {DOM.Point} focusPoint
         *  the selection end point.
         *
         * @returns {Object | null}
         *  An selection object containing physical start and end coordinates.
         *  - {Number} Object.anchor - selection start absolute coordinate in pixels
         *  - {Number} Object.focus -  selection end absolute coordinate in pixels
         */
        function calculatePhysicalPosition(anchorPoint, focusPoint) {

            // quit if no logical selection is available
            if (!anchorPoint || !focusPoint) { return null; }

            var zoom = app.getView().getZoomFactor();

            function getPhysicalPosition(point1, point2, start) {
                return Utils.getCSSRectangleFromPoints(point1, point2, rootNode, zoom, start);
            }

            return { anchor: getPhysicalPosition(anchorPoint, null, true), focus: getPhysicalPosition(focusPoint), range: getPhysicalPosition(anchorPoint, focusPoint) };
        }

        /**
         * Draw overlays for a specific remote user's selection debounced.
         */
        var renderCollaborativeSelectionDebounced = this.createDebouncedMethod('RemoteSelection.renderCollaborativeSelectionDebounced', null, function () {
            if (Config.SHOW_REMOTE_SELECTIONS) {

                // first, clear all old selection nodes from the DOM
                rootNode.find('.collaborative-overlay').children().remove();

                removeDrawingNodesHandler();

                // render all remote selections
                _.each(app.getActiveClients(), renderSelection);
            }
        }, { delay: 300 });

        /**
         * Create a Overlay with the username to show the remote username for a selection.
         * @param {type} username name of the user
         * @param {type} className classname for the overlay
         * @param {String} clientId Id of the client
         * @param {jQuery} UserName Overlay
         */
        function createUserNameOverlay(username, className, clientId) {
            var usernameOverlay = $(document.createElement('div'));
            // draw user name
            usernameOverlay.addClass('collaborative-username ' + className);
            usernameOverlay.attr('data-clientid', clientId);
            usernameOverlay.text(username);

            return usernameOverlay;
        }

        /**
         * Draw overlays for a specific remote user's selection.
         *
         * @param {Object} client
         *  Descriptor of a client editing this document.
         */
        function renderSelection(client) {

            var // the selection of the client
                userSelections = client.userData.selections,
                userTargetSelection = client.userData.target,
                userTargetIndex = client.userData.targetIndex;

            // helper function to determine if selection is on active slide
            function isSelectionOnActiveSlide(userSelection) {
                var state = false;
                var activeSlideId = docModel.getActiveSlideId();

                if (docModel.isLayoutOrMasterId(activeSlideId)) {
                    state = activeSlideId === client.userData.target;
                } else {
                    state = docModel.getActiveSlideIndex() === _.first(userSelection.start);
                }
                return state;
            }

            // draw nothing and quit if:
            // - user is the current user himself
            // - no user data/selection is available.
            if (!client.remote || !userSelections || userSelections.length === 0) {
                return;
            }

            var username = client.userName,
                overlay = rootNode.find('.collaborative-overlay'),
                className = 'user-' + client.colorIndex,
                zoom = app.getView().getZoomFactor(),
                // skip the render of drawing selections if one drawing is not on the active slide
                skipDrawing = false,
                usernameOverlay;

            // loop through the selections of the user
            _.each(userSelections, function (userSelection) {

                // special handling if selection is a top level cursor position on a slide
                // -> not drawing the remote cursor on the slide.
                if (skipDrawing || (slideMode && selection.isTopLevelTextCursor(userSelection))) {
                    // TODO: Draw remote selection (on slide pane?)
                    return;
                }

                // Don't draw selection if is on inactive slide, #46237
                if (slideMode && !isSelectionOnActiveSlide(userSelection)) { return; }

                var activeRootNode = docModel.getRootNode(userTargetSelection, userTargetIndex),
                    selectionAnchorPoint = Position.getDOMPosition(activeRootNode, userSelection.start),
                    selectionFocusPoint = Position.getDOMPosition(activeRootNode, userSelection.end);

                if (!selectionAnchorPoint || !selectionFocusPoint) { return; }

                var selectionOverlayGroup = $('<div>').addClass('collaborative-selection-group');

                var drawingNode;
                var rotation = 0;
                var drawingAttrs;
                var setUsernameRotation = false;
                var flipH = false;
                var flipV = false;
                // if the drawing is rotatet only select the frame and not the text
                if (slideMode && userSelection.type !== 'drawing') {
                    drawingNode = DrawingFrame.getDrawingNode($(Position.getDOMPosition(activeRootNode, userSelection.start, true).node));
                    if (drawingNode) {
                        drawingAttrs = AttributeUtils.getExplicitAttributes(drawingNode);
                        rotation = drawingAttrs.drawing ? Utils.getNumberOption(drawingAttrs.drawing, 'rotation', 0) : 0;
                    }
                }

                // render the selection for drawings
                if (userSelection.type === 'drawing' || rotation > 0) {
                    // Test if the drawing is on the aktive slide (Presentation), if not skip the rendering.
                    if (slideMode) {
                        var slideID = docModel.getActiveSlideId();
                        var layoutOrMaster = docModel.isMasterView();
                        if (layoutOrMaster) {
                            if (slideID !== client.userData.target) {
                                skipDrawing = true;
                            }
                            // TODO select slidePane if the target is layoutOrMaster too
                        } else {
                            if (client.userData.target || userSelection.start[0] !== docModel.getActiveSlideIndex()) {
                                skipDrawing = true;
                            }
                            // TODO select slidePane it the target is not layoutOrMaster
                        }
                        if (skipDrawing) {
                            return;
                        }
                    }

                    if (!drawingNode) {
                        drawingNode = $(Position.getDOMPosition(activeRootNode, userSelection.start, true).node);
                    }

                    if (drawingNode.length === 0) {
                        return;
                    }
                    drawingAttrs = drawingAttrs ? drawingAttrs : AttributeUtils.getExplicitAttributes(drawingNode);
                    rotation = 0;

                    if (drawingAttrs && drawingAttrs.drawing) {
                        flipH = Utils.getBooleanOption(drawingAttrs.drawing, 'flipH', false);
                        flipV = Utils.getBooleanOption(drawingAttrs.drawing, 'flipV', false);
                        rotation = Utils.getNumberOption(drawingAttrs.drawing, 'rotation', 0);
                    }

                    var drawingOverlay = document.createElement('div'),
                        top = slideMode ? parseFloat(drawingNode.css('top')) : (drawingNode.offset().top - rootNode.offset().top) / zoom,
                        left = slideMode ? parseFloat(drawingNode.css('left')) : (drawingNode.offset().left - rootNode.offset().left) / zoom;

                    if (rotation !== 0 && !slideMode) {
                        var rect = DrawingUtils.getRotatedDrawingPoints({ width: drawingNode.width(), height: drawingNode.height(), top: 0, left: 0 }, -rotation);
                        top -= rect.top;
                        left -= rect.left;
                    }

                    usernameOverlay = createUserNameOverlay(username, className, client.userId);

                    // Set the selection frame position and size attributes
                    $(selectionOverlayGroup).css({
                        top: top - 2,
                        left: left - 2,
                        width: parseFloat(drawingNode.css('width')) + 4,
                        height: parseFloat(drawingNode.css('height')) + 4,
                        transform: DrawingUtils.getCssTransform(rotation, flipH, flipV)
                    });

                    selectionOverlayGroup.append(drawingOverlay);
                    selectionOverlayGroup.append(usernameOverlay);

                    // Add 'hover' handler with namespace to show the Username Overlay.
                    // The namespace is to remove the event handler without the function.
                    drawingNode.on({
                        'mouseenter.remoteselection': function () {
                            usernameOverlay.show();
                        },
                        'mouseleave.remoteselection': function () {
                            usernameOverlay.hide();
                        }
                    });
                    usernameOverlay.show();
                    usernameOverlay.css('display', 'inherit');
                    drawingNodesWithHandlers.push(drawingNode);
                    drawingOverlay.setAttribute('class', 'selection-overlay drawing ' + className);
                    setUsernameRotation = true;
                } else {
                    var caretHandle = $('<div>').addClass('collaborative-cursor-handle'),
                        caretOverlay = $('<div>').addClass('collaborative-cursor'),
                        startNode = $(selectionAnchorPoint.node.parentNode),
                        endNode = $(selectionFocusPoint.node.parentNode),
                        physicalPosition = calculatePhysicalPosition(selectionAnchorPoint, selectionFocusPoint),
                        isRangeSelection = physicalPosition.anchor.top !== physicalPosition.focus.top || physicalPosition.anchor.left !== physicalPosition.focus.left;

                    usernameOverlay = createUserNameOverlay(username, className, client.userId);

                    // draw user cursors and names
                    usernameOverlay.css('top', physicalPosition.focus.height);
                    caretOverlay.css({ top: physicalPosition.focus.top, left: physicalPosition.focus.left, height: physicalPosition.focus.height }).addClass(className);
                    caretOverlay.append(usernameOverlay, caretHandle);
                    caretHandle.hover(function () { $(usernameOverlay).show(); }, function () { usernameOverlay.hide(); });
                    usernameOverlay.show();
                    usernameOverlay.css('display', 'inherit');
                    overlay.append(caretOverlay);

                    // if remote selection is in header/footer, fix z-index collision.
                    if (client.editor) {
                        if (userTargetSelection) {
                            overlay.addClass('target-overlay');
                        } else {
                            overlay.removeClass('target-overlay');
                        }
                    }

                    if (isRangeSelection) {

                        // draw collaborative selection area highlighting if we have an 'range' selection
                        var rootNodePos = rootNode.offset(), //both
                            startNodeCell = startNode.closest('td'), // table
                            endNodeCell = endNode.closest('td'), // table
                            isTableSelection = startNodeCell.length && endNodeCell.length && !startNodeCell.is(endNodeCell);

                        // special handling of pure table selection
                        if (isTableSelection) {

                            // handle special mega cool firefox cell selection
                            if (userSelection.type === 'cell') {

                                var ovStartPos = startNodeCell.offset(),
                                    ovEndPos = endNodeCell.offset(),
                                    ovWidth = (ovEndPos.left  - ovStartPos.left) / zoom + endNodeCell.outerWidth(),
                                    ovHeight = (ovEndPos.top  - ovStartPos.top) / zoom + endNodeCell.outerHeight(),
                                    ov = document.createElement('div');

                                $(ov).css({
                                    top: (ovStartPos.top - rootNodePos.top) / zoom,
                                    left: (ovStartPos.left - rootNodePos.left) / zoom,
                                    width: ovWidth,
                                    height: ovHeight
                                });

                                selectionOverlayGroup.append(ov);

                            } else { // normal table selection (Chrome, IE): iterate cells between start and end

                                var cells = $(Utils.findFarthest(rootNode, endNodeCell, 'table')).find('td'),
                                    cellsToHighlight = cells.slice(cells.index(startNodeCell), cells.index(endNodeCell) + 1);

                                cellsToHighlight.each(function () {

                                    var cellOv = document.createElement('div'),
                                        offset = $(this).offset();

                                    $(cellOv).css({
                                        top: (offset.top - rootNodePos.top) / zoom,
                                        left: (offset.left - rootNodePos.left) / zoom,
                                        width: $(this).outerWidth(),
                                        height: $(this).outerHeight()
                                    });

                                    selectionOverlayGroup.append(cellOv);

                                });
                            }
                        } else { // paragraph / mixed selection
                            var startTop = physicalPosition.anchor.top,
                                startHeight = physicalPosition.anchor.height,
                                endTop = physicalPosition.focus.top,
                                endHeight = physicalPosition.focus.height,
                                startBottom = startTop + startHeight,
                                endBottom = endTop + endHeight,
                                isSingleLineSelection = startBottom === endBottom || (endTop >= startTop && endBottom <= startBottom) || (startTop >= endTop && startBottom <= endBottom),
                                caretHeight;

                            // selection area is in a line
                            if (isSingleLineSelection) {
                                caretHeight = physicalPosition.range.height;

                                var selectionOverlay = document.createElement('div');
                                $(selectionOverlay).css({ top: physicalPosition.range.top, left: physicalPosition.range.left, width: physicalPosition.range.width, height: caretHeight });
                                selectionOverlayGroup.append(selectionOverlay);

                                usernameOverlay.css('top', caretHeight);
                                caretOverlay.css({ top: physicalPosition.range.top, left: physicalPosition.focus.left, height: caretHeight }).addClass(className);
                            } else { // multi line selection area

                                var anchorSelection = physicalPosition.anchor;
                                var focusSelection = physicalPosition.focus;
                                var rangeSelection = physicalPosition.range;

                                var height = (anchorSelection.top + anchorSelection.height) - rangeSelection.top;
                                var headOverlay = document.createElement('div');
                                $(headOverlay).css({
                                    top: rangeSelection.top,
                                    left: anchorSelection.left,
                                    width: anchorSelection.left === rangeSelection.left ?  rangeSelection.width : (rangeSelection.left + rangeSelection.width) - anchorSelection.left,
                                    height: height
                                });

                                var tailOverlay = document.createElement('div');
                                var tailWidth = focusSelection.left - rangeSelection.left;
                                $(tailOverlay).css({
                                    top: focusSelection.top,
                                    left: rangeSelection.left,
                                    width: tailWidth,
                                    height: focusSelection.height
                                });

                                var bodyOverlay = document.createElement('div');
                                var bodyTop = rangeSelection.top + height;
                                var bodyHeight = focusSelection.top - bodyTop;
                                $(bodyOverlay).css({
                                    top: bodyTop,
                                    left: rangeSelection.left,
                                    width: rangeSelection.width,
                                    height: bodyHeight
                                });

                                caretHeight = focusSelection.height;

                                usernameOverlay.css('top', caretHeight);
                                caretOverlay.css({ top: focusSelection.top, left: rangeSelection.left + tailWidth, height: caretHeight }).addClass(className);

                                selectionOverlayGroup.append(headOverlay, bodyOverlay, tailOverlay);
                            }
                        }

                        _.each(selectionOverlayGroup.children(), function (overlayElement) {
                            overlayElement.setAttribute('class', 'selection-overlay ' + className);
                        });
                    }
                }

                if (!_.isBoolean(client.selectionChanged)) {
                    usernameOverlay.show(); // 36800
                }
                usernameOverlay.show();
                usernameOverlay.css('display', 'inherit');
                overlay.append(selectionOverlayGroup);
                if (setUsernameRotation) {
                    var userHeight = usernameOverlay.height(); //clientSize ? clientSize.height : 0;
                    var userWidth = usernameOverlay.width(); //clientSize ? clientSize.width : 0;
                    var usernameCSS;
                    if (rotation >= 316 || rotation <= 45) {
                        //top
                        usernameCSS = {
                            top: 0,
                            left: 0,
                            transform: DrawingUtils.getCssTransform(0, flipH, flipV)
                        };
                    } else if (rotation >= 136 && rotation <= 225) {
                        // bottom
                        usernameCSS = {
                            top: Math.ceil(parseFloat(drawingNode.css('height')) - userHeight),
                            left: Math.ceil(parseFloat(drawingNode.css('width')) - userWidth - 12),
                            transform: DrawingUtils.getCssTransform(0, !flipH, !flipV)
                        };
                    } else {
                        var isLeft = rotation >= 46 && rotation <= 135 && !flipH && !flipV;
                        var isRight = rotation >= 226 && rotation <= 315;
                        if (isLeft || (isRight && (flipV || flipH))) {
                            usernameCSS = {
                                top: Math.ceil(parseFloat(drawingNode.css('height')) - userWidth / 2 - userHeight + 2),
                                left: Math.ceil(-userWidth / 2 + 2),
                                transform: DrawingUtils.getCssTransform(90, !flipH || (flipH && flipV), !flipV || (flipH && flipV))
                            };
                        } else {
                            // right
                            usernameCSS = {
                                top:  Math.ceil(userWidth / 2 - userHeight / 2 + 6),
                                left: Math.ceil(-userWidth / 2 + 2 + parseFloat(drawingNode.css('width')) - userHeight),
                                transform: DrawingUtils.getCssTransform(90, flipH && !flipV, flipV && !flipH)
                            };
                        }
                    }
                    $(usernameOverlay).css(usernameCSS);
                }
            });

            // show user name for a second
            if (overlayTimer) { overlayTimer.abort(); }

            overlayTimer = self.executeDelayed(function () {
                overlay.find('.collaborative-username').fadeOut('fast');
            }, 'RemoteSelection.overlayTimer', 3000);
        }

        /**
         * Remove the 'Hover' event handlers to show the username tooltip from the selected drawing layer.
         */
        function removeDrawingNodesHandler() {
            // remove the handlers from the drawing nodes
            drawingNodesWithHandlers.forEach(function (drawingNode) {
                drawingNode.off('mouseenter.remoteselection mouseleave.remoteselection');
            });

            drawingNodesWithHandlers = [];
        }

        // public methods -----------------------------------------------------

        /**
         * Draw overlays for the selections of all remote clients.
         *
         * @returns {RemoteSelection}
         *  A reference to this instance.
         */
        this.renderCollaborativeSelections = function () {
            renderCollaborativeSelectionDebounced();
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members when application is alive
        app.onInit(function () {
            removeDrawingNodesHandler();
            selection = docModel.getSelection();
            // render selection if the user change the slide to see the current selection of the new slide
            self.listenTo(docModel, 'change:slide', _.bind(self.renderCollaborativeSelections, self));
            // render selection if drawing attributes will be changed
            self.listenTo(docModel, 'change:drawing', _.bind(self.renderCollaborativeSelections, self));
            // render selection if the list of active clients has changed
            self.listenTo(app, 'docs:users:selection', _.bind(self.renderCollaborativeSelections, self));
            docModel.listenTo(app, 'update:absoluteElements', _.bind(self.renderCollaborativeSelections, self));
            docModel.listenTo(app,  'drawingHeight:update', _.bind(self.renderCollaborativeSelections, self));
        });

        // destroy all class members
        this.registerDestructor(function () {
            docModel = app = rootNode = selection = drawingNodesWithHandlers = null;
        });

    } // class RemoteSelection

    // exports ================================================================

    return BaseObject.extend({ constructor: RemoteSelection });

});
