/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/keyhandlingmixin', [
    'io.ox/office/tk/keycodes',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/textframework/components/table/table',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/keycodeutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/listutils',
    'io.ox/office/textframework/utils/textutils'
], function (KeyCodes, DrawingFrame, AttributeUtils, Hyperlink, Table, DOM, KeyCodeUtils, Operations, Position, ListUtils, Utils) {

    'use strict';

    // mix-in class KeyHandlingMixin ======================================

    /**
     * A mix-in class for the hard break handling in the document model.
     * And additionally for the manual page break handling.
     *
     * @constructor
     */
    function KeyHandlingMixin(app) {

        var // self reference for local functions
            self = this,
            // the period of time, that a text input is deferred, waiting for a further text input (in ms).
            inputTextTimeout = 2,
            // the maximum number of characters that are buffered before an insertText operation is created.
            maxTextInputChars = 3,
            // the content, that will be printed in the current active text input process
            activeInputText = null,
            // a deferred that is resolved, after the text is inserted. This is required for the undo group and must be global.
            insertTextDef = null;

        // private methods ----------------------------------------------------

        /**
         * modify current listlevel - if any - depending on the increase flag
         */
        function implModifyListLevel(currentlistLevel, increase, minLevel) {

            self.getUndoManager().enterUndoGroup(function () {

                var levelKey = null;

                if (self.useSlideMode()) {
                    self.changeListIndent({ increase: increase });
                } else {

                    levelKey = 'listLevel';
                    if (increase && currentlistLevel < 8) {
                        currentlistLevel += 1;
                        self.setAttribute('paragraph', levelKey, currentlistLevel);
                    } else if (!increase && currentlistLevel > minLevel) {
                        currentlistLevel -= 1;
                        self.setAttribute('paragraph', levelKey, currentlistLevel);
                        if (currentlistLevel < 0 && !self.useSlideMode()) {
                            self.setAttribute('paragraph', 'listStyleId', null);
                        }
                    }
                }
            });
        }

        /**
         * Removes pageBreakBefore paragraph attribute from passed table element. It is called when
         * cursor is in first position inside first table paragraph and BACKSPACE is pressed, or
         * when DELETE is pressed on cursor placed in paragraph's last position right above table
         *
         * @param {jQuery|DOM} table - table element that has paragraph with pageBreakBefore attribute,
         *  and has to be removed.
         *
         */
        function removePageBreakBeforeAttribute(table) {

            var paragraphWithManualPageBreak,
                paragraphPos,
                operation,
                // the page node
                editdiv = self.getNode();

            paragraphWithManualPageBreak = $(table).find(DOM.MANUAL_PAGE_BREAK_SELECTOR);

            if (paragraphWithManualPageBreak.length > 0) {
                paragraphPos = Position.getOxoPosition(editdiv, paragraphWithManualPageBreak);

                operation = { name: Operations.SET_ATTRIBUTES, attrs: { paragraph: { pageBreakBefore: false } }, start: paragraphPos };
                self.applyOperations(operation);
            } else {
                if (app.isODF()) {
                    paragraphPos = Position.getFirstPositionInParagraph(editdiv, Position.getOxoPosition(editdiv, table));
                    operation = { name: Operations.SET_ATTRIBUTES, attrs: { paragraph: { pageBreakBefore: null } }, start: paragraphPos };
                    self.applyOperations(operation);
                } else {
                    // #35358 - if we delete paragraph, we cannot remove attr from it, so just clean up table marker for displaying
                    $(table).removeClass(DOM.MANUAL_PAGE_BREAK_CLASS_NAME);
                }
            }
        }

        /**
         * Handler for the keyDown event.
         *
         * @param {jQuery.Event} event
         *  A jQuery keyboard event object.
         */
        function pageProcessKeyDownHandler(event) {

            var readOnly = !app.isEditable(),
                isCellSelection = false,
                currentBrowserSelection = null,
                checkPosition = null,
                returnObj = null,
                startPosition,
                endPosition,
                paraLen,
                activeRootNode = self.getCurrentRootNode(),
                // the selection object
                selection = self.getSelection(),
                // the page layout object
                pageLayout = self.getPageLayout(),
                // the range marker object
                rangeMarker = self.getRangeMarker(),
                // keys combination used to jump into header/footer on diff OS
                keyCombo = null,
                // is keycode is left or right arrow
                leftRightArrow = event.keyCode === KeyCodes.LEFT_ARROW || event.keyCode === KeyCodes.RIGHT_ARROW,
                // control varible used for bugfix #58296 in Crome for rulerpane
                rulerPaneTempHidden = false;

            if (self.isFormatPainterActive()) {
                self.setFormatPainterState(false);
            }

            if (selection.isCropMode()) {
                // TODO check keys ESC, block typing...
                if (event.keyCode === KeyCodes.ESCAPE) {
                    self.exitCropMode();
                    return;
                }
                if (!DOM.isCursorKey(event.keyCode)) {
                    event.preventDefault();
                    return;
                }
            }

            // disable text input into slide, if slide mode is active (no preventDefault -> this breaks pasting)
            if (self.useSlideMode() && selection.isTopLevelTextCursor() && event.keyCode !== KeyCodes.ESCAPE && !KeyCodeUtils.isF6AcessibilityKeyEvent(event) && !KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                if (DOM.isIgnorableKey(event.keyCode) || KeyCodeUtils.isBrowserShortcutKeyEvent(event)) { return; } // 50901
                if (DOM.isCursorKey(event.keyCode) && self.setActiveSlideByCursorEvent) { self.setActiveSlideByCursorEvent(event); }
                if (!KeyCodeUtils.isPasteKeyEventKeyDown(event)) { event.preventDefault(); } // still enabling pasting into slide root
                return;
            }

            // disabling all kind of text modifications in master view in place holder drawings of ODF documents
            if (self.isODFReadOnyPlaceHolderDrawingProcessing && !DOM.isCursorKey(event.keyCode) && self.isODFReadOnyPlaceHolderDrawingProcessing()) {
                event.preventDefault();
                return;
            }

            // prevent browser's navigation history alt+arrow shortcut, if it is used later as drawing rotation
            if (self.useSlideMode() && (selection.isDrawingSelection() || selection.isMultiSelection()) && (event.altKey && leftRightArrow)) {
                event.preventDefault();
            }

            // saving the keyDown event for later usage in processKeyPressed
            self.setLastKeyDownEvent(event);

            // registering MacOS marker for multiple directly following keydown events (46659)
            if (_.browser.MacOS && !DOM.isCursorKey(event.keyCode)) {
                if (_.isNumber(self.getLastKeyDownKeyCode()) && self.getLastKeyDownKeyCode() === event.keyCode) {
                    self.setRepeatedKeyDown(true);
                }
                self.setLastKeyDownKeyCode(event.keyCode);
                self.setLastKeyDownModifiers(event.ctrlKey, event.shiftKey, event.altKey, event.metaKey);
            }

            // open the change track popup, when available, with 'ALT' + 'DOWN_ARROW' combination
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true })) {
                app.getView().showChangeTrackPopup();
                return;
            }

            // close the change track popup, when available, with 'ALT' + 'UP_ARROW' combination
            if (KeyCodes.matchKeyCode(event, 'UP_ARROW', { alt: true })) {
                var changeTrackPopup = app.getView().getChangeTrackPopup();
                if (changeTrackPopup) { changeTrackPopup.hide(); }
                return;
            }

            if (self.getRoIOSParagraph()) {
                if (app.isEditable()) {
                    $(self.getRoIOSParagraph()).removeAttr('contenteditable');
                }
                self.setRoIOSParagraph(null);
            }

            // Fix for 32910: Client reports internal error if user types as soon as possible after pasting lengthly content with lists
            if (self.getBlockKeyboardEvent()) {
                event.preventDefault();
                return false;
            }

            // special handling for IME input on IE
            // having a browser selection that spans up ak range,
            // the IME input session start event (compositionstart) is omitted.
            // Normally we have to call deleteSelected() to remove the selection within
            // compositionstart. Unfortunately IE is NOT able to work correctly if the
            // browser selection is changed during a IME session. Therefore this code
            // cancels the IME session whenever we detect a selected range to preserve
            // a consistent document state.
            // TODO: We have to find a better solution to support IME and selections.
            if (_.browser.IE && (event.keyCode === KeyCodes.IME_INPUT) && selection.hasRange()) {
                event.preventDefault();
                return false;
            }

            if (DOM.isIgnorableKey(event.keyCode) || KeyCodeUtils.isBrowserShortcutKeyEvent(event)) {
                return;
            }

            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                // We need the keyboard events for the popup which normally contains
                // an external URL which must be opened by the browser.
                return;
            }

            self.dumpEventObject(event);

            // 'escape' converts a cell selection into a text selection
            if ((event.keyCode === KeyCodes.ESCAPE) && (selection.getSelectionType() === 'cell')) {
                Table.resolveCellSelection(selection, activeRootNode);
                event.preventDefault();
                return false;
            }

            // resetting values for table cell selection, if no 'shift' is pressed
            if ((selection.getAnchorCellRange() !== null) && !event.shiftKey) {
                selection.setAnchorCellRange(null);
            }

            if (DOM.isCursorKey(event.keyCode)) {

                if (app.isTextApp() && _.browser.Chrome && app.getView().getRulerPane().isVisible() && (event.keyCode === KeyCodes.PAGE_UP || event.keyCode === KeyCodes.PAGE_DOWN)) { //#58296
                    app.getView().getRulerPane().getNode().addClass('temp-hide-ruler');
                    rulerPaneTempHidden = true;
                }

                // deferring all cursor position calculations, because insertText is also deferred (39358)
                self.executeDelayed(function () {
                    var isTextFrameSelectionInSlideMode = selection.isAdditionalTextframeSelection() && self.useSlideMode();

                    isCellSelection = (selection.getSelectionType() === 'cell');

                    if (isCellSelection) {
                        currentBrowserSelection = selection.getBrowserSelection();
                    }

                    // Setting cursor up and down in tables on MS IE or Webkit
                    if ((event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.DOWN_ARROW) && (!event.shiftKey) && (!event.ctrlKey)) {
                        // Setting cursor up and down in tables on MS IE or Webkit
                        if ((_.browser.IE || _.browser.WebKit) && self.isPositionInTable()) {
                            if (Table.updownCursorTravel(event, selection, activeRootNode, { up: event.keyCode === KeyCodes.UP_ARROW, down: event.keyCode === KeyCodes.DOWN_ARROW, readonly: readOnly })) {
                                return;
                            }
                        }
                        // Setting cursor up and down before and behind exceeded-size tables in Firefox (27395)
                        if (_.browser.Firefox) {
                            if (Table.updownExceededSizeTable(event, selection, activeRootNode, { up: event.keyCode === KeyCodes.UP_ARROW, down: event.keyCode === KeyCodes.DOWN_ARROW })) {
                                return;
                            }
                        }
                    }

                    // Changing table cell selections in FireFox when shift is used with arrow keys.
                    // Switching from text selection to cell selection already happened in 'selection.moveTextCursor()'.
                    if (_.browser.Firefox && DOM.isArrowCursorKey(event.keyCode) && event.shiftKey && isCellSelection && self.isPositionInTable()) {

                        event.preventDefault();

                        if ((event.keyCode === KeyCodes.UP_ARROW) || (event.keyCode === KeyCodes.DOWN_ARROW)) {
                            Table.changeCellSelectionVert(app, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.UP_ARROW });
                        }

                        if ((event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW)) {
                            Table.changeCellSelectionHorz(app, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.LEFT_ARROW });
                        }

                        return;
                    }

                    if (_.browser.IE && DOM.isArrowCursorKey(event.keyCode) && event.shiftKey &&
                        (self.isPositionInTable() || Position.isPositionInTable(activeRootNode, selection.getEndPosition()) || Position.isPositionInTable(activeRootNode, selection.getStartPosition()))) {

                        if ((event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW)) {
                            if (Table.changeCellSelectionHorzIE(app, activeRootNode, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.LEFT_ARROW })) {
                                event.preventDefault();
                                return;
                            }
                        }

                    }
                    // jumping to header/footer on Mac and other OS
                    keyCombo = _.browser.MacOS ? { ctrl: true, meta: true } : { ctrl: true, alt: true };

                    if (KeyCodes.matchKeyCode(event, 'PAGE_UP', keyCombo)) { // Go to header (if exists, if not, create it first) on the page where cursor is
                        event.preventDefault();
                        if (!self.isHeaderFooterEditState() && !readOnly) {
                            pageLayout.jumpToHeaderOnCurrentPage();
                            return;
                        }
                    } else if (KeyCodes.matchKeyCode(event, 'PAGE_DOWN', keyCombo)) { // Go to footer (if exists, if not, create it first) on the page where cursor is
                        event.preventDefault();
                        if (!self.isHeaderFooterEditState() && !readOnly) {
                            pageLayout.jumpToFooterOnCurrentPage();
                            return;
                        }
                    } else if (KeyCodes.matchKeyCode(event, 'HOME', { ctrlOrMeta: true }) && isTextFrameSelectionInSlideMode) {
                        event.preventDefault();
                        selection.setTextSelection(Position.getFirstTextPositionInTextFrame(activeRootNode, selection.getAnyTextFrameDrawing()));
                        return;
                    } else if (KeyCodes.matchKeyCode(event, 'HOME', { ctrlOrMeta: true, shift: true }) && isTextFrameSelectionInSlideMode) {
                        event.preventDefault();
                        selection.setTextSelection(Position.getFirstTextPositionInTextFrame(activeRootNode, selection.getAnyTextFrameDrawing()), selection.getEndPosition());
                        return;
                    } else if (KeyCodes.matchKeyCode(event, 'END', { ctrlOrMeta: true }) && isTextFrameSelectionInSlideMode) {
                        event.preventDefault();
                        selection.setTextSelection(Position.getLastTextPositionInTextFrame(activeRootNode, selection.getAnyTextFrameDrawing()));
                        return;
                    } else if (KeyCodes.matchKeyCode(event, 'END', { ctrlOrMeta: true, shift: true }) && isTextFrameSelectionInSlideMode) {
                        event.preventDefault();
                        selection.setTextSelection(selection.getStartPosition(), Position.getLastTextPositionInTextFrame(activeRootNode, selection.getAnyTextFrameDrawing()));
                        return;
                    }

                    // handling vertical cursor key events inside text frames is done in selection class
                    // in checkVerticalPositions, it's corrected after browser tries to set selection

                    // any navigation key: change drawing selection to text selection before or behind the drawing (OX Text)
                    if (!self.useSlideMode() && selection.isDrawingSelection()) { selection.selectDrawingAsText(); }

                    // moving selected drawing(s) in slide mode
                    if (self.useSlideMode() && (selection.isDrawingSelection() || selection.isMultiSelection()) && self.handleDrawingOperations(event)) { return; }

                    // changing the slide always with page up and page down, independent from the current selection
                    if (self.handleSlideChangeByPageUpDown && self.handleSlideChangeByPageUpDown(event)) { return; }

                    // let browser process the key event (this may create a drawing selection)
                    selection.processBrowserEvent(event, { readonly: readOnly }); // processing is dependent from read-only mode

                    // #58296
                    if (rulerPaneTempHidden) {
                        app.getView().getRulerPane().getNode().removeClass('temp-hide-ruler');
                        rulerPaneTempHidden = false;
                    }
                }, 'KeyHandlingMixin.processKeyDown', inputTextTimeout);

                return;
            }

            // invoke copy-method in IE manually when we have a TextCursor within a TextFrame (shape) and pressing "copy", cause it won't be done automatically
            if (_.browser.IE && self.useSlideMode() && KeyCodeUtils.isCopyKeyEventKeyDown(event) && selection.isTextCursor() && selection.isAdditionalTextframeSelection()) {
                self.copy(event);
            }

            // handle just cursor, copy, search, selectall and the global F6 accessibility events if in read only mode
            if (readOnly && !KeyCodeUtils.isCopyKeyEventKeyDown(event) && !KeyCodeUtils.isF6AcessibilityKeyEvent(event) && !KeyCodes.matchKeyCode(event, 'TAB', { shift: null }) && !KeyCodes.matchKeyCode(event, 'ESCAPE', { shift: null }) && !KeyCodeUtils.isSearchKeyEvent(event) && !KeyCodeUtils.isSelectAllEvent(event)) {
                if (!self.isImportFinished()) {
                    app.rejectEditAttempt('loadingInProgress');
                } else {
                    app.rejectEditAttempt();
                }
                event.preventDefault();
                return;
            }

            if (event.keyCode === KeyCodes.DELETE && !event.shiftKey) {

                event.preventDefault();

                // if there is an active 'inputTextPromise', it has to be set to null now, so that
                // following text input is included into a new deferred
                self.setInputTextPromise(null);

                // Executing the code for 'DELETE' deferred. This is necessary because of deferred input of text. Defer time is 'inputTextTimeout'
                self.executeDelayed(function () {

                    var helperNode,
                        // the change track object
                        changeTrack = self.getChangeTrack(),
                        // a local helper for lastOperationEnd
                        lastOperationEndLocal = null;

                    startPosition = selection.getStartPosition();

                    if (event.ctrlKey) {
                        var storedEndPos = Position.getRangeForDeleteWithCtrl(self.getCurrentRootNode(), startPosition, false);

                        selection.setTextSelection(startPosition, storedEndPos, { simpleTextSelection: true });
                    }

                    if (selection.hasRange()) {
                        if (app.isODF()) { self.setKeepPreselectedAttributes(true); } // 52649
                        self.deleteSelected({ deleteKey: true })
                            .done(function () {
                                startPosition = selection.getStartPosition(); // refresh required after deleteSelected()
                                selection.setTextSelection((self.getLastOperationEnd() && (startPosition.length === self.getLastOperationEnd().length)) ? startPosition : self.getLastOperationEnd(), null, { simpleTextSelection: true });
                                if (app.isODF()) { self.setKeepPreselectedAttributes(false); } // 52649
                            });
                    } else {
                        endPosition = selection.getEndPosition();

                        returnObj = Position.skipDrawingsAndTables(_.clone(startPosition), _.clone(endPosition), activeRootNode, { backwards: false });
                        paraLen = returnObj.paraLen;
                        startPosition = returnObj.start;
                        endPosition = returnObj.end;

                        if (returnObj && returnObj.selectComplexField) {
                            helperNode = rangeMarker.getEndMarker(DOM.getRangeMarkerId(returnObj.node));
                            if (helperNode) {
                                startPosition = selection.getStartPosition();
                                endPosition = Position.getOxoPosition(activeRootNode, helperNode);
                                selection.setTextSelection(startPosition, endPosition);
                            } else {
                                Utils.error('editor.processKeyDown(): handling DELETE before complex field failed to fetch valid node!');
                            }
                            return;
                        }

                        // handling for grouped drawings, only ODF (58986, 58987) ->  not generating invalid operation
                        if (app.isODF() && app.isPresentationApp() && Position.isGroupedDrawingPosition(self.getCurrentRootNode(), startPosition)) { return; }

                        if (startPosition[startPosition.length - 1] < paraLen) {
                            self.deleteRange(startPosition, endPosition, { setTextSelection: false, handleUnrestorableContent: true })
                                .always(function () {
                                    // If change tracking is active, lastOperationEnd is set inside deleteRange
                                    if (changeTrack.isActiveChangeTracking() && _.isArray(self.getLastOperationEnd()) && !_.isEqual(self.getLastOperationEnd(), startPosition)) {
                                        selection.setTextSelection(self.getLastOperationEnd(), null, { simpleTextSelection: true });
                                    } else {
                                        selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                                    }

                                    // list update in presentation app necessary, if the last character in the paragraph was removed
                                    if (self.useSlideMode() && paraLen === 1 && _.last(startPosition) === 0 && _.isEqual(startPosition, endPosition)) {
                                        self.handleListItemVisibility(); // last character removed without changing selection
                                    }
                                });

                        } else {
                            var mergeselection = _.clone(startPosition),
                                characterPos = mergeselection.pop(),
                                nextParagraphPosition = _.copy(mergeselection);

                            nextParagraphPosition[nextParagraphPosition.length - 1] += 1;

                            var domPos = Position.getDOMPosition(activeRootNode, nextParagraphPosition),
                                nextIsTable = false,
                                isLastParagraph = false;

                            if (domPos) {
                                if (DOM.isTableNode(domPos.node)) {
                                    nextIsTable = true;
                                }
                            } else {
                                nextParagraphPosition[nextParagraphPosition.length - 1] -= 1;
                                isLastParagraph = true;
                            }

                            if (DOM.isMergeableParagraph(Position.getParagraphElement(activeRootNode, mergeselection))) {
                                if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(domPos.node)) {  // not really deleting paragraph, but marking for deletion
                                    self.applyOperations({
                                        name: Operations.SET_ATTRIBUTES,
                                        start: nextParagraphPosition,  // the latter paragraph needs to be marked for removal
                                        attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } }
                                    });
                                    // Setting the cursor to the correct position, if no character was deleted
                                    lastOperationEndLocal = _.clone(mergeselection);
                                    lastOperationEndLocal[lastOperationEndLocal.length - 1] += 1;
                                    lastOperationEndLocal = Position.appendNewIndex(lastOperationEndLocal);
                                    self.setLastOperationEnd(lastOperationEndLocal);

                                } else {
                                    if (paraLen === 0 && !DOM.isManualPageBreakNode(domPos.node)) {  // Simply remove an empty paragraph
                                        self.deleteRange(_.initial(_.clone(startPosition)));
                                        // updating lists, if required. This is the case, if the deleted paragraph is part of a list
                                        self.handleTriggeringListUpdate(domPos.node.previousSibling);
                                    } else { // Merging two paragraphs
                                        self.mergeParagraph(mergeselection);
                                    }
                                }
                            }

                            if (nextIsTable) {
                                if (characterPos === 0) {
                                    // removing empty paragraph
                                    var localPos = _.clone(startPosition);
                                    localPos.pop();
                                    if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(domPos.node)) {  // not really deleting paragraph, but marking for deletion
                                        self.applyOperations({
                                            name: Operations.SET_ATTRIBUTES,
                                            start: localPos,  // the latter paragraph needs to be marked for removal
                                            attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } }
                                        });
                                        // Setting the cursor to the correct position, if no character was deleted
                                        lastOperationEndLocal = _.clone(localPos);
                                        lastOperationEndLocal[lastOperationEndLocal.length - 1] += 1;
                                        lastOperationEndLocal = Position.getFirstPositionInParagraph(activeRootNode, lastOperationEndLocal);
                                        self.setLastOperationEnd(lastOperationEndLocal);

                                    } else {
                                        self.getUndoManager().enterUndoGroup(function () {

                                            var // the current paragraph
                                                currentParagraph = Position.getContentNodeElement(activeRootNode, startPosition.slice(0, -1)),
                                                // the previous allowed neighbour of the paragraph
                                                prevNeighbour = DOM.getAllowedNeighboringNode(currentParagraph, { next: false }),
                                                // the next allowed neighbour of the paragraph
                                                nextNeighbour = DOM.getAllowedNeighboringNode(currentParagraph),
                                                // the logical position of the previous neighbour node
                                                prevNeighbourPos = null;

                                            // remove paragraph explicitely
                                            if (!DOM.isImplicitParagraphNode(currentParagraph)) { // first handle if implicit paragraph at that position
                                                self.deleteRange(localPos);
                                                nextParagraphPosition[nextParagraphPosition.length - 1] -= 1;
                                            } else {
                                                $(currentParagraph).remove();
                                            }

                                            if (nextNeighbour && DOM.isTableNode(nextNeighbour)) {

                                                // check if table has paragraph with manual page break set, and remove it
                                                if (DOM.isManualPageBreakNode(nextNeighbour)) {
                                                    removePageBreakBeforeAttribute(nextNeighbour);
                                                }

                                                // if both neighbours are tables, and it's possible, merge them
                                                if (prevNeighbour && DOM.isTableNode(prevNeighbour) && Table.mergeableTables(prevNeighbour, nextNeighbour) && !app.isODF()) {
                                                    prevNeighbourPos = Position.getOxoPosition(activeRootNode, prevNeighbour);
                                                    self.mergeTable(prevNeighbourPos, { next: true });
                                                }
                                            }

                                            return $.when();
                                        }, this); // end of enterUndoGroup()
                                    }
                                }
                                startPosition = Position.getFirstPositionInParagraph(activeRootNode, nextParagraphPosition);
                            } else if (isLastParagraph) {

                                if (Position.isPositionInTable(activeRootNode, nextParagraphPosition)) {

                                    returnObj = Position.getFirstPositionInNextCell(activeRootNode, nextParagraphPosition);
                                    startPosition = returnObj.position;
                                    var endOfTable = returnObj.endOfTable;
                                    if (endOfTable && startPosition) {
                                        startPosition[startPosition.length - 1] += 1;
                                        startPosition = Position.getFirstPositionInParagraph(activeRootNode, startPosition);
                                    }
                                }
                            }

                            if (self.getChangeTrack().isActiveChangeTracking() && _.isArray(self.getLastOperationEnd()) && !_.isEqual(self.getLastOperationEnd(), startPosition)) {
                                selection.setTextSelection(self.getLastOperationEnd(), self.getLastOperationEnd(), { simpleTextSelection: true });
                            } else if (startPosition) {
                                selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                            }
                        }
                    }

                }, 'KeyHandlingMixin.processKeyDown', inputTextTimeout);

            } else if (event.keyCode === KeyCodes.BACKSPACE && !event.altKey) {

                if (Utils.IOS && self.isImeActive()) {
                    //Bug 42457
                    //IPAD directly hangs up an reloads the page :(
                    self.setInputTextPromise(null);
                    return;
                }

                var imeSyncActive = self.isImeActive();
                event.preventDefault();

                // if there is an active 'inputTextPromise', it has to be set to null now, so that
                // following text input is included into a new deferred
                self.setInputTextPromise(null);

                // Executing the code for 'BACKSPACE' deferred. This is necessary because of deferred input of text. Defer time is 'inputTextTimeout'
                var handleBack = function () {
                    var helperNode;

                    if (self.isImeActive() && !imeSyncActive) {
                        // #51390: Never process BACKSPACE key asynchronously if we are within a composition
                        return;
                    }

                    startPosition = selection.getStartPosition();

                    if (event.ctrlKey) {
                        var storedEndPos = Position.getRangeForDeleteWithCtrl(self.getCurrentRootNode(), startPosition, true);

                        selection.setTextSelection(storedEndPos, startPosition, { simpleTextSelection: true });
                        startPosition = storedEndPos;
                    }

                    if (selection.hasRange()) {
                        if (app.isODF()) { self.setKeepPreselectedAttributes(true); } // 52649
                        self.deleteSelected({ deleteKey: true })
                            .done(function () {
                                startPosition = selection.getStartPosition(); // refresh required after deleteSelected()
                                selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                                if (app.isODF()) { self.setKeepPreselectedAttributes(false); } // 52649
                            });
                    } else {
                        returnObj = Position.skipDrawingsAndTables(_.clone(startPosition), _.clone(selection.getEndPosition()), activeRootNode, { backwards: true });
                        if (returnObj && returnObj.selectComplexField) {
                            helperNode = rangeMarker.getStartMarker(DOM.getRangeMarkerId(returnObj.node));
                            if (helperNode) {
                                startPosition = Position.getOxoPosition(activeRootNode, helperNode);
                                endPosition = selection.getStartPosition();
                                selection.setTextSelection(startPosition, endPosition);
                            } else {
                                Utils.error('editor.processKeyDown(): handling BACKSPACE behind complex field failed to fetch valid node!');
                            }
                            return;
                        }

                        if (returnObj.start === null || returnObj.end === null) { // #33154 backspace on table with position [0, 0] and exceeded size limit
                            return;
                        }

                        startPosition = returnObj.start;
                        endPosition = returnObj.end;

                        // handling for grouped drawings, only ODF (58986, 58987) ->  not generating invalid operation
                        if (app.isODF() && app.isPresentationApp() && Position.isGroupedDrawingPosition(self.getCurrentRootNode(), startPosition)) { return; }

                        if (startPosition[startPosition.length - 1] > 0) {

                            startPosition[startPosition.length - 1] -= 1;
                            endPosition[endPosition.length - 1] -= 1;
                            if (app.isODF()) { self.setKeepPreselectedAttributes(true); } // 52649
                            self.deleteRange(startPosition, endPosition, { setTextSelection: false, handleUnrestorableContent: true })
                                .fail(function () {
                                    // keeping the current selection
                                    startPosition[startPosition.length - 1] += 1;
                                })
                                .always(function () {
                                    selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                                    if (app.isODF()) { self.setKeepPreselectedAttributes(false); }
                                });

                        } else if (startPosition[startPosition.length - 2] >= 0) {

                            var paragraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, startPosition, DOM.PARAGRAPH_NODE_SELECTOR),
                                listLevel = -1,
                                atParaStart = startPosition[startPosition.length - 1] === 0,
                                styleId = null,
                                elementAttributes = null,
                                styleAttributes = null,
                                paraAttributes = null;

                            if (!_(startPosition).all(function (value) { return (value === 0); })) {

                                startPosition[startPosition.length - 2] -= 1;
                                startPosition.pop();  // -> removing last value from startPosition !

                                var length = Position.getParagraphLength(activeRootNode, startPosition),
                                    domPos = Position.getDOMPosition(activeRootNode, startPosition),
                                    prevIsTable = false,
                                    paraBehindTable = null,
                                    localStartPos = null,
                                    fromStyle = false,
                                    prevPara;

                                if ((domPos) && (DOM.isTableNode(domPos.node))) {
                                    prevIsTable = true;
                                }

                                if (atParaStart) {

                                    elementAttributes = self.getParagraphStyles().getElementAttributes(paragraph);
                                    paraAttributes = elementAttributes.paragraph;

                                    if (self.useSlideMode()) {
                                        if (_.isNumber(paraAttributes.level) && paraAttributes.level > -1) {
                                            // removing a bullet, if it exists
                                            if (paraAttributes.bullet && paraAttributes.bullet.type && paraAttributes.bullet.type !== 'none') {
                                                self.removeListAttributes();
                                                self.handleTriggeringListUpdate(paragraph);
                                                return;
                                            }
                                        }
                                    } else {

                                        styleId = elementAttributes.styleId;
                                        listLevel = paraAttributes.listLevel;
                                        styleAttributes = self.getParagraphStyles().getStyleAttributeSet(styleId).paragraph;

                                        if (paraAttributes.listStyleId !== '') {
                                            fromStyle = paraAttributes.listStyleId === styleAttributes.listStyleId;
                                        }

                                        if (fromStyle) {
                                            // works in OOX as listStyle 0 is a non-numbering style
                                            // will not work in OpenDocument
                                            self.setAttribute('paragraph', 'listStyleId', 'L0');
                                            return;
                                        } else if (listLevel >= 0) {
                                            implModifyListLevel(listLevel, false, -1);
                                            return;
                                        } else if (styleId === self.getDefaultUIParagraphListStylesheet()) {
                                            self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet() });
                                            // invalidate previous paragraph to recalculate bottom distance
                                            prevPara = Utils.findPreviousNode(activeRootNode, paragraph, DOM.PARAGRAPH_NODE_SELECTOR);
                                            if (prevPara) {
                                                self.getParagraphStyles().updateElementFormatting(prevPara);
                                            }
                                            return;
                                        }
                                    }
                                }

                                if (startPosition[startPosition.length - 1] >= 0) {
                                    if (!prevIsTable) {
                                        if (self.getChangeTrack().isActiveChangeTracking() && !self.getChangeTrack().isInsertNodeByCurrentAuthor(paragraph)) {
                                            localStartPos = _.clone(startPosition);
                                            localStartPos[localStartPos.length - 1] += 1;
                                            self.applyOperations({
                                                name: Operations.SET_ATTRIBUTES,
                                                start: localStartPos,  // the latter paragraph is marked for removal
                                                attrs: { changes: { removed: self.getChangeTrack().getChangeTrackInfo() } }
                                            });
                                        } else {
                                            if (Position.getParagraphLength(activeRootNode, startPosition) === 0 && !DOM.isManualPageBreakNode(paragraph)) {
                                                // simply remove first paragraph, if it is empty
                                                self.deleteRange(startPosition);
                                            } else {
                                                // merge two following paragraphs, if both are not empty
                                                self.mergeParagraph(startPosition);
                                            }
                                        }
                                    } else {

                                        checkPosition = _.clone(startPosition);
                                        checkPosition[checkPosition.length - 1] += 1; // increasing paragraph again
                                        //checkPosition.push(0);

                                        // if it is not an implicit paragraph now, it can be removed via operation (but not a final paragraph behind a table in a cell)
                                        paraBehindTable = Position.getParagraphElement(activeRootNode, checkPosition);
                                        if ((Position.getParagraphNodeLength(paraBehindTable) === 0) && !DOM.isImplicitParagraphNode(paraBehindTable) && !DOM.isFinalParagraphBehindTableInCell(paraBehindTable)) {
                                            // remove paragraph explicitely or mark it as removed
                                            if (self.getChangeTrack().isActiveChangeTracking() && !self.getChangeTrack().isInsertNodeByCurrentAuthor(paragraph)) {
                                                self.applyOperations({
                                                    name: Operations.SET_ATTRIBUTES,
                                                    start: checkPosition,  // the latter paragraph is marked for removal
                                                    attrs: { changes: { removed: self.getChangeTrack().getChangeTrackInfo() } }
                                                });
                                            } else {
                                                self.getUndoManager().enterUndoGroup(function () {
                                                    var prevNeighbour = DOM.getAllowedNeighboringNode(paraBehindTable, { next: false }),
                                                        nextNeighbour = DOM.getAllowedNeighboringNode(paraBehindTable),
                                                        prevNeighbourPos = Position.getOxoPosition(activeRootNode, prevNeighbour);
                                                    // remove paragraph explicitely
                                                    self.applyOperations({ name: Operations.DELETE, start: checkPosition });
                                                    // check if table has paragraph with manual page break set, and remove it
                                                    if (DOM.isTableNode(nextNeighbour) && DOM.isManualPageBreakNode(nextNeighbour)) {
                                                        removePageBreakBeforeAttribute(nextNeighbour);
                                                    }
                                                    // if both neighbours are tables, and it's possible, merge them
                                                    if (DOM.isTableNode(prevNeighbour) && DOM.isTableNode(nextNeighbour) && Table.mergeableTables(prevNeighbour, nextNeighbour) && !app.isODF()) {
                                                        self.mergeTable(prevNeighbourPos, { next: true });
                                                    }
                                                    return $.when();
                                                }, this); // end of enterUndoGroup()
                                            }
                                        }
                                    }
                                }

                                if (prevIsTable) {
                                    startPosition = Position.getLastPositionInParagraph(activeRootNode, startPosition, { ignoreImplicitParagraphs: true });
                                } else {
                                    var isFirstPosition = startPosition[startPosition.length - 1] < 0;
                                    if (isFirstPosition) {
                                        if (DOM.isFirstContentNodeInTextframe(paragraph)) {
                                            return; // simply ignore 'backspace' at start position of text frame
                                        } else if (Position.isPositionInTable(activeRootNode, startPosition)) {

                                            returnObj = Position.getLastPositionInPrevCell(activeRootNode, startPosition, { ignoreImplicitParagraphs: true });
                                            startPosition = returnObj.position;
                                            var beginOfTable = returnObj.beginOfTable;
                                            if (beginOfTable) {
                                                var table = Position.getContentNodeElement(activeRootNode, startPosition);
                                                if (!table && Position.isGroupedDrawingPosition(activeRootNode, startPosition)) { return; } // table not found, if it is a drawing table (59803)
                                                if (DOM.isFirstContentNodeInTextframe(table)) {
                                                    return; // simply ignore 'backspace' at start position of table
                                                }
                                                // check if table has paragraph with manual page break set, and remove it
                                                if (DOM.isManualPageBreakNode(table)) {
                                                    removePageBreakBeforeAttribute(table);
                                                }
                                                // update position of cursor
                                                startPosition[startPosition.length - 1] -= 1;
                                                startPosition = Position.getLastPositionInParagraph(activeRootNode, startPosition, { ignoreImplicitParagraphs: true });
                                            }
                                        } else {
                                            startPosition.push(length);
                                        }
                                    } else {
                                        startPosition.push(length);
                                    }
                                }
                            } else {
                                // The position contains only '0's -> Backspace at the beginning of the document -> handle paragraphs with list style.
                                elementAttributes = self.getParagraphStyles().getElementAttributes(paragraph);
                                styleId = elementAttributes.styleId;
                                paraAttributes = elementAttributes.paragraph;
                                listLevel = paraAttributes.listLevel;
                                styleAttributes = self.getParagraphStyles().getStyleAttributeSet(styleId).paragraph;

                                if (self.useSlideMode()) {
                                    if (_.isNumber(paraAttributes.level) && paraAttributes.level > -1) {
                                        // removing a bullet, if it exists
                                        if (paraAttributes.bullet && paraAttributes.bullet.type && paraAttributes.bullet.type !== 'none') {
                                            self.removeListAttributes();
                                            self.handleTriggeringListUpdate(paragraph);
                                            return;
                                        }
                                    }
                                } else {
                                    if ((paraAttributes.listStyleId !== '') && (paraAttributes.listStyleId === styleAttributes.listStyleId)) {
                                        // works in OOX as listStyle 0 is a non-numbering style will not work in OpenDocument
                                        self.setAttribute('paragraph', 'listStyleId', 'L0');
                                    } else if (listLevel >= 0) {
                                        implModifyListLevel(listLevel, false, -1);
                                    } else if (styleId === self.getDefaultUIParagraphListStylesheet()) {
                                        self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet() });
                                    }
                                }
                            }

                            selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                        }
                    }

                };
                if (_.browser.Android) {
                    //it has to be syncron on android, otherwise it can overjump some letters
                    handleBack();
                } else {
                    self.executeDelayed(handleBack, 'KeyHandlingMixin.processKeyDown', inputTextTimeout);
                }

            } else if (Utils.boolXor(event.metaKey, event.ctrlKey) && !event.altKey) {

                if (event.keyCode === KeyCodes.ENTER) {
                    // insert manual page break, but not inside text frames
                    if (selection.isAnyDrawingSelection()) {
                        event.preventDefault();
                        return;
                    }
                    self.insertManualPageBreak();
                }

                // prevent browser from evaluating the key event, but allow cut, copy and paste events
                if (!KeyCodeUtils.isPasteKeyEventKeyDown(event) && !KeyCodeUtils.isCopyKeyEventKeyDown(event) && !KeyCodeUtils.isCutKeyEventKeyDown(event)) {
                    event.preventDefault();
                }

                //self.clearPreselectedAttributes();
            } else if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {

                event.preventDefault();

                if (self.isPositionInTable()) {

                    // if there is an active 'inputTextPromise', it has to be set to null now, so that
                    // following text input is included into a new deferred
                    self.setInputTextPromise(null);

                    // Executing the code for 'TAB' deferred. This is necessary because of deferred input of text.
                    // Defer time is 'inputTextTimeout'
                    self.executeDelayed(function () {

                        if (event.shiftKey) {
                            // Jump into first position of previous cell. Do not jump, if there is no previous cell
                            returnObj = Position.getFirstPositionInPreviousCell(activeRootNode, selection.getStartPosition());
                            if (!returnObj.beginOfTable) {
                                selection.setTextSelection(returnObj.position);
                            }
                        } else {
                            // Jumping into the next table cell or, if this is already the last cell of a table, insert a new row.
                            returnObj = Position.getFirstPositionInNextCell(activeRootNode, selection.getStartPosition());

                            if (returnObj.endOfTable) {

                                if (readOnly) {
                                    app.rejectEditAttempt();
                                    event.preventDefault();
                                    return;
                                }

                                if (self.isRowAddable()) {
                                    self.insertRow();
                                    selection.setTextSelection(self.getLastOperationEnd());
                                } else {
                                    app.getView().rejectEditTextAttempt('tablesizerow'); // checking table size (26809)
                                    return;
                                }
                            } else {
                                selection.setTextSelection(returnObj.position);
                            }
                        }

                    }, 'KeyHandlingMixin.processKeyDown', inputTextTimeout);

                // select next/prev drawing
                } else if (self.isDrawingSelected() || (self.useSlideMode() && selection.isTopLevelTextCursor())) {
                    app.getView().executeControllerItem('document/selectDrawing', { backwards: KeyCodes.matchKeyCode(event, 'TAB', { shift: true }) });
                    return;

                } else {

                    if (readOnly) {
                        app.rejectEditAttempt();
                        event.preventDefault();
                        return;
                    }

                    // (shift)Tab: Change list indent (if in list) when selection is at first position in paragraph
                    var paragraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, selection.getStartPosition(), DOM.PARAGRAPH_NODE_SELECTOR),
                        mustInsertTab = true;  // always true, independent from shift-key

                    if (!selection.hasRange() && paragraph && _.last(selection.getStartPosition()) === Position.getFirstTextNodePositionInParagraph(paragraph)) {

                        var elementAttributes = self.getParagraphStyles().getElementAttributes(paragraph),
                            styleId = elementAttributes.styleId,
                            paraAttributes = elementAttributes.paragraph,
                            listLevel = paraAttributes.listLevel,
                            styleAttributes = self.getParagraphStyles().getStyleAttributeSet(styleId).paragraph;

                        if (paraAttributes.listStyleId) {
                            mustInsertTab = false;
                            var fromStyle = listLevel === -1 || paraAttributes.listStyleId === styleAttributes.listStyleId;
                            if (listLevel === -1) {
                                listLevel = paraAttributes.listLevel;
                            }
                            if (listLevel !== -1) {

                                if (!fromStyle) {
                                    implModifyListLevel(listLevel, !event.shiftKey, 0);
                                } else {
                                    // numbering via paragraph style (e.g. outline numbering)
                                    var newStyleId = self.getListCollection().findPrevNextStyle(paraAttributes.listStyleId, styleId, event.shiftKey);
                                    if (newStyleId) {
                                        self.setAttributes('paragraph', { styleId: newStyleId });
                                    }
                                }
                            }
                        } else if (self.useSlideMode() && self.isListParagraph(paraAttributes)) {
                            mustInsertTab = false;
                            implModifyListLevel(paraAttributes.level, !event.shiftKey, 0, true);
                        }

                        event.stopPropagation(); // otherwise the drawing will change (in OX Spreadsheet)
                    }
                    if (mustInsertTab && paragraph) {
                        // if there is an active 'inputTextPromise', it has to be set to null now, so that
                        // following text input is included into a new deferred
                        self.setInputTextPromise(null);

                        // Executing the code for 'TAB' deferred. This is necessary because of deferred input of text.
                        // Defer time is 'inputTextTimeout'
                        self.executeDelayed(self.insertTab, 'KeyHandlingMixin.processKeyDown', inputTextTimeout);

                        event.stopPropagation(); // otherwise the drawing will change (in OX Spreadsheet)
                    }
                }
            } else if (event.keyCode === KeyCodes.ESCAPE) {

                var selectionBox = _.isFunction(self.getSelectionBox) ? self.getSelectionBox() : null;
                // cancelling an active selection box
                if (selectionBox && (selectionBox.isCancelModeActive() || selectionBox.isSelectionBoxActive())) {
                    selectionBox.cancelSelectionBox();
                    return false;
                }

                // deselect drawing node before the search bar
                if (self.isDrawingSelected()) {
                    selection.setTextSelection(selection.getEndPosition());
                    return false;
                }

                // deselect the paragraph in the text frame. Set the selection to the text frame itself.
                if (selection.isAdditionalTextframeSelection()) {
                    var start = Position.getOxoPosition(self.getCurrentRootNode(), selection.getSelectedTextFrameDrawing(), 0),
                        end = Position.increaseLastIndex(start);

                    // select drawing
                    selection.setTextSelection(start, end);
                    return false;
                }

                if (self.isHeaderFooterEditState()) {
                    // leave edit state
                    pageLayout.leaveHeaderFooterAndSetCursor(activeRootNode, activeRootNode.parent());
                    return false;
                }
                // else: let ESCAPE key bubble up the DOM tree
            } else if (KeyCodes.matchKeyCode(event, 'ENTER') && selection.getSelectionType() === 'drawing') {

                if (DrawingFrame.isTextFrameShapeDrawingFrame(selection.getSelectedDrawing())) {
                    event.preventDefault();
                    app.getView().executeControllerItem('document/setCursorIntoTextframe');
                }
            } else if (KeyCodes.matchKeyCode(event, 'F9')) {
                if (selection.hasRange()) {
                    self.getFieldManager().updateHighlightedFieldSelection();
                } else if (self.getFieldManager().isHighlightState()) {
                    self.getFieldManager().updateHighlightedField();
                }
            }

        }

        /**
         * Handler for the keyPress event, following the keyDown event.
         *
         * Info: Do not use comparisons with event.keyCode inside pageProcessKeyPressHandler!
         *  The event from processKeyDown, that was executed before this pageProcessKeyPressHandler
         *  is stored in the variable 'lastKeyDownEvent'. Please use this, if you want
         *  to make comparisons with keyCodes.
         *
         * Addition: Try to use this event ONLY for information about character
         *  that is beeing pressed on keyboard. For getting information about physical key being pressed,
         *  please use function processKeyDown! For combination of modifier keys and characters,
         *  use lastKeyDownEvent for modifier, and event for chars!
         *
         * @param {jQuery.Event} event
         *  A jQuery keyboard event object.
         */
        function pageProcessKeyPressHandler(event) {

            var // Editor mode
                readOnly = !app.isEditable(),
                // the char element
                c = null,
                hyperlinkSelection = null,
                // currently active root node
                activeRootNode = self.getCurrentRootNode(),
                // the last keydown event
                lastKeyDownEvent = self.getLastKeyDownEvent(),
                // the selection object
                selection = self.getSelection(),
                // whether an additional operation for removal of hyper links is required.
                removeHyperLinkAttributes = false;

            // Helper function for collecting and inserting characters. This function is typically executed synchronously,
            // but in special cases with a selection containing unrestorable content, it is executed deferred.
            function doInsertCharacter() {

                var // the string to be inserted with insert text operation
                    insertText = null,
                    // the logical start position of the selection
                    startPosition = null;

                // merging input text, if there is already a promise and there are not already 5 characters deferred
                if ((self.getInputTextPromise()) && (activeInputText) && (activeInputText.length < maxTextInputChars)) {
                    // deleting existing deferred for insert text and creating a new deferred
                    // containing the new and the old text
                    self.getInputTextPromise().abort();
                    activeInputText += c;
                } else {
                    // creating a completely new deferred for insert text
                    self.setInputTextPromise(null);
                    activeInputText = c;
                }

                insertText = activeInputText;
                // Calling insertText deferred. Restarting, if there is further text input in the next 'inputTextTimeout' ms.
                self.setInputTextPromise(self.executeDelayed(function () {
                    startPosition = selection.getStartPosition();

                    var // whether the text can be inserted at the specified position
                        validPosition = true;

                    // check if a modified selection is not a slide selection and the the drawing is already selected (happens on IE)
                    if (self.useSlideMode() && (startPosition.length === 2 || !selection.isAdditionalTextframeSelection())) { validPosition = false; }

                    if (validPosition) {

                        self.insertText(insertText, startPosition, self.getPreselectedAttributes());

                        // set cursor behind character
                        selection.setTextSelection(Position.increaseLastIndex(startPosition, insertText.length), null, { simpleTextSelection: !removeHyperLinkAttributes, insertOperation: true });
                    }

                    // Setting to null, so that new timeouts can be started.
                    // If there is already a running new inputTextPromise (fast typing), then this is also set to null, so that it cannot
                    // add further characters. But this is not problematic. Example: '123 456' -> After typing space, the inputTextPromise
                    // is set to null. Then it can happen that '4' and '5' are typed, and then this deferred function is executed. In this
                    // case the inputTextPromise for the '45' is terminated, even though there would be sufficient time to add the '6' too.
                    // But this is not important, because this concatenation is only required for faster type feeling.
                    self.setInputTextPromise(null);
                    // Resolving the insertTextDef for the undoGroup. This is very important, because otherwise there could be
                    // pending actions that will never be sent to the server (33226). Every new character in 'processKeyPressed'
                    // needs the same 'insertTextDef', because this is registered for the undoGroup with the return value of
                    // enterUndoGroup. A first character registers the promise of the 'insertTextDef' and a second very fast
                    // following character must use the same (already created) insertTextDef. It is finally resolved
                    // during the execution of this deferred function. What happens, if there is already a new inputTextPromise?
                    // Again the example '123 456'. After typing '123 ' the inputTextPromise is set to null. Then '456' is typed
                    // and collected. Then this deferred function is executed for '123 ', deletes the inputTextPromise (so that
                    // '456' is also completed) and resolves the insertTextDef, so that it will be set to null (via the 'always'
                    // function). This makes also the '456' a complete block with resolved undoGroup. If later this function is
                    // executed for '456', the insertTextDef might already be set to null. But in this case it simply does not
                    // need to be resolved, because this already happened at the previous run of the function with '123 '.
                    // The correct undoGroups are anyhow created by merging following insertText operations (mergeUndoActionHandler).
                    if (insertTextDef) {  // -> the insertTextDef might be already resolved in previous call of this function
                        insertTextDef.resolve();  // resolving the deferred for the undo group
                    }
                }, 'KeyHandlingMixin.insertText', inputTextTimeout));

                // After a space, always a new text input promise is required -> text will be written.
                // Setting inputTextPromise to null, makes it unreachable for abort() .
                if (c === ' ') { self.setInputTextPromise(null); }
            }

            // resetting MacOS marker for multiple directly following keydown events (46659)
            if (_.browser.MacOS) {
                self.setLastKeyDownKeyCode(null);
                self.setLastKeyDownModifiers(null);
                self.setRepeatedKeyDown(false);
                if (_.browser.Safari && event.key && (event.key.length > 1) && (event.charCode === event.key.charCodeAt(0))) {
                    // # 51454 - Text input on Mac/Safari doesn't work as expected using composing keys ´, `, ^
                    // BE CAREFUL: This workaround is based on behavior and NOT on specification. That Safari put
                    // two characters in the property 'key' is also strange. Therefore this workaround is doomed
                    // to fail, if somone in WebKit/Safari changes the behavior.
                    event.charCode = event.key.charCodeAt(1);
                }
            }

            // disable text input into slide, if slide mode is active (no preventDefault -> this breaks pasting)
            if (self.useSlideMode() && selection.isTopLevelTextCursor()) { return; }

            // disabling all kind of text modifications in master view in place holder drawings of ODF documents
            if (self.isODFReadOnyPlaceHolderDrawingProcessing && self.isODFReadOnyPlaceHolderDrawingProcessing()) {
                event.preventDefault();
                return;
            }

            // Android Chrome Code
            if (_.browser.Android && event.keyCode === 0) { return; }
            if (event.keyCode === KeyCodes.ENTER && _.browser.Android) { event.preventDefault(); }

            self.dumpEventObject(event);

            if (lastKeyDownEvent && DOM.isIgnorableKey(lastKeyDownEvent.keyCode)) {
                return;
            }

            if (Utils.IOS && lastKeyDownEvent && lastKeyDownEvent.keyCode === 0 && Utils.isHangulText(event.originalEvent.data)) {
                // #42667 Korean keyboard has no feedback for IME-Mode
                event.preventDefault();
                return;
            }

            // Fix for 32910: Client reports internal error if user types as soon as possible after pasting lengthly content with lists
            if (self.getBlockKeyboardEvent()) {
                event.preventDefault();
                return false;
            }

            if (lastKeyDownEvent && DOM.isCursorKey(lastKeyDownEvent.keyCode)) {
                // Fix for 32008: CTRL+POS1/END create special characters on Mac/Chrome
                if (_.browser.WebKit && _.browser.MacOS && _.browser.Chrome && lastKeyDownEvent.ctrlKey) {
                    if ((lastKeyDownEvent.keyCode === 35) || (lastKeyDownEvent.keyCode === 36)) {
                        event.preventDefault();
                    }
                }
                return;
            }

            // needs to be checked on both keydown and keypress events
            if (KeyCodeUtils.isBrowserShortcutKeyEvent(event)) {
                if (!_.device('macos') && event.metaKey && !event.ctrlKey) { event.preventDefault(); } // 51465, allowing metaKey only on Mac
                return;
            }

            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                // We need the keyboard events for the popup which normally contains
                // an external URL which must be opened by the browser.
                return;
            }

            // TODO: Check if unnecessary! In most browsers returned already at processKeyDown!
            // handle just cursor, copy, escape, search and the global F6 accessibility events if in read only mode
            if (readOnly && (
                !lastKeyDownEvent || (
                    !KeyCodeUtils.isCopyKeyEventKeyDown(lastKeyDownEvent) &&
                    !KeyCodeUtils.isF6AcessibilityKeyEvent(lastKeyDownEvent) &&
                    !KeyCodes.matchKeyCode(lastKeyDownEvent, 'TAB', { shift: null }) &&
                    !KeyCodes.matchKeyCode(lastKeyDownEvent, 'ESCAPE', { shift: null })
                )
            ) && !KeyCodeUtils.isSearchKeyEvent(event)
            ) {
                if (!self.isImportFinished()) {
                    app.rejectEditAttempt('loadingInProgress');
                } else {
                    app.rejectEditAttempt();
                }
                event.preventDefault();
                return;
            }

            // Special behavior for the iPad! Due to strange event order in mobile Safari we have to
            // pass the keyPressed event to the browser. We use the textInput event to cancel the DOM
            // modification instead. The composition system needs the keyPressed event to start a
            // a new composition.
            if (!Utils.IOS) {
                // prevent browser from evaluating the key event, but allow cut, copy and paste events
                if (!lastKeyDownEvent || (
                    !KeyCodeUtils.isPasteKeyEventKeyDown(lastKeyDownEvent) &&
                    !KeyCodeUtils.isCopyKeyEventKeyDown(lastKeyDownEvent) &&
                    !KeyCodeUtils.isCutKeyEventKeyDown(lastKeyDownEvent)
                )
                ) {
                    event.preventDefault();
                }
            } else {
                // We still get keydown/keypressed events during a composition session in mobile Safari.
                // According to the W3C specification this shouldn't happen. Therefore we have to prevent
                // to call our normal keyPressed code.
                if (self.isImeActive()) {
                    return;
                }
            }

            c = KeyCodeUtils.getPrintableChar(event);

            // TODO
            // For now (the prototype), only accept single chars, but let the browser process, so we don't need to care about DOM stuff
            // TODO: But we at least need to check if there is a selection!!!

            if (lastKeyDownEvent && (!lastKeyDownEvent.ctrlKey || (lastKeyDownEvent.ctrlKey && lastKeyDownEvent.altKey && !lastKeyDownEvent.shiftKey)) && !lastKeyDownEvent.metaKey && (c.length === 1)) {

                // do nothing in the case of a drawing multi selection or in the case of a slide selection
                if ((selection.isMultiSelectionSupported() && selection.isMultiSelection()) || (self.useSlideMode() && selection.isTopLevelTextCursor())) {
                    return;
                }

                self.getUndoManager().enterUndoGroup(function () {

                    var // whether the insertTextDef was created
                        createdInsertTextDef = false,
                        // an optional modified selection
                        modifiedSelection = null,
                        // a logical position
                        startPosition = null;

                    // Setting insertTextDef, if it is still null.
                    // Because of the undoGroup this insertTextDef must be global (33226). It cannot be local and be created
                    // in every run inside this 'undoManager.enterUndoGroup'.

                    if (!insertTextDef) {
                        createdInsertTextDef = true;
                        insertTextDef = self.createDeferred('KeyHandlingMixin.pageProcessKeyHandler insertText')
                            .always(function () {
                                insertTextDef = null;
                            });
                    }

                    if (selection.hasRange()) {

                        // Special handling for selected drawing frames
                        if (selection.isDrawingFrameSelection()) {

                            // not deleting the drawing -> adding the content into or before the drawing instead of deleting it
                            modifiedSelection = selection.getValidTextPositionAccordingToDrawingFrame();

                            if (modifiedSelection) {
                                // setting the new selection
                                selection.setTextSelection(modifiedSelection);
                                // inserting the character into the text frame
                                doInsertCharacter();
                            } else {
                                // did not find valid selection inside the text frame (there might be no paragraph inside)
                                createdInsertTextDef = false;
                                insertTextDef = null;
                            }

                        } else {

                            // saving predefined character attributes at the start position
                            self.setKeepPreselectedAttributes(true);

                            self.deleteSelected({ saveStartAttrs: true })
                                .done(function () {
                                    self.setKeepPreselectedAttributes(false);
                                    // inserting the character with taking care of existing selection range
                                    // -> this might lead to a deferred character insertion
                                    doInsertCharacter();
                                })
                                .fail(function () {
                                    if (insertTextDef) {
                                        insertTextDef.resolve();
                                    }
                                    self.setKeepPreselectedAttributes(false);
                                });
                        }
                    } else {
                        // check left text to support hyperlink auto correction
                        if (event.charCode === KeyCodes.SPACE) {
                            startPosition = selection.getStartPosition();
                            hyperlinkSelection = Hyperlink.checkForHyperlinkText(self, selection.getEnclosingParagraph(), startPosition);
                            if (hyperlinkSelection !== null) {
                                removeHyperLinkAttributes = true;
                            }
                        }

                        // inserting the character without taking care of existing selection range
                        doInsertCharacter();
                    }

                    // returning a promise for the undo group
                    return createdInsertTextDef ? insertTextDef.promise() : $.when();

                }) // enterUndoGroup
                    .done(function () {

                        if (hyperlinkSelection !== null) {
                            // perform auto stuff after undo group, so that first undo only undoes auto stuff
                            self.executeDelayed(function () {
                                self.getUndoManager().enterUndoGroup(function () {
                                    if (!hyperlinkSelection.isStartOfParagraph) {
                                        Hyperlink.insertHyperlink(
                                            self,
                                            hyperlinkSelection.start,
                                            hyperlinkSelection.end,
                                            (hyperlinkSelection.url === null) ? hyperlinkSelection.text : hyperlinkSelection.url,
                                            self.getActiveTarget(),
                                            { removeHyperLinkAttributes: removeHyperLinkAttributes }
                                        );
                                    }
                                }, null, { preventSelectionChange: true });
                            }, 'KeyHandlingMixin.handleHyperLink', inputTextTimeout);
                        }

                    });

            } else if (c.length > 1) {
                // TODO?
            } else {

                if (lastKeyDownEvent && KeyCodes.matchKeyCode(lastKeyDownEvent, 'ENTER', { shift: null })) {

                    // if there is an active 'inputTextPromise', it has to be set to null now, so that
                    // following text input is included into a new deferred
                    self.setInputTextPromise(null);

                    // 'Enter' key is possible with and without shift-key. With shift-key a hard break is inserted.
                    if (!lastKeyDownEvent.shiftKey) {

                        // Executing the code for 'ENTER' deferred. This is necessary because of deferred input of text.
                        // Defer time is 'inputTextTimeout'
                        self.executeDelayed(function () {

                            var // whether the selection has a range (saving state, because the selection will be removed by deleteSelected())
                                hasSelection = selection.hasRange();

                            // Helper function for executing the 'Enter' key event. This function is typically executed synchronously,
                            // but in special cases with a selection containing unrestorable content, it is executed deferred.
                            function doHandleEnterKey() {

                                var // the logical start position of the selection
                                    startPosition = selection.getStartPosition(),
                                    // the index of the last value in the logical start position
                                    lastValue = startPosition.length - 1,
                                    // a new logical position
                                    newPosition = _.clone(startPosition),
                                    // an object containing the start and end position of a hyperlink
                                    hyperlinkSelection = null,
                                    // a logical table position
                                    localTablePos = null,
                                    // whether the current position is the start position in a paragraph
                                    isFirstPositionInParagraph = (startPosition[lastValue] === 0),
                                    // whether the current position is the start position of the first paragraph in the document or table cell
                                    isFirstPositionOfFirstParagraph = (isFirstPositionInParagraph && (startPosition[lastValue - 1] === 0)),
                                    // the paragraph element addressed by the passed logical position
                                    paragraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, startPosition, DOM.PARAGRAPH_NODE_SELECTOR),
                                    // Performance: Whether a simplified selection can be used
                                    isSimpleTextSelection = true, isSplitOperation = true,
                                    // the paragraph attributes
                                    paragraphAttrs = null,
                                    listLevel,
                                    paragraphLength,
                                    endOfParagraph,
                                    split,
                                    numAutoCorrect,
                                    paraText,
                                    labelText;

                                // Local helper function, to check if cursor is currently placed inside complex field node,
                                // so the list formatting can be ignored if true
                                function isCursorInsideCxField() {
                                    var domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition()),
                                        node = null;
                                    if (domPos && domPos.node) {
                                        node = domPos.node;
                                        if (node.nodeType === 3) {
                                            node = node.parentNode;
                                        }
                                    }
                                    return DOM.isInsideComplexFieldRange(node);
                                }

                                // Local helper for checking if first non-empty paragraph child contains - and is not complex field, #42999
                                function isCxFieldFirstInPar() {
                                    var fieldRangeMarker = $(paragraph).children(DOM.RANGEMARKER_STARTTYPE_SELECTOR).first();
                                    var rangeMarkerStartPos = null;
                                    if (fieldRangeMarker.length && DOM.getRangeMarkerType(fieldRangeMarker) === 'field') {
                                        rangeMarkerStartPos = Position.getOxoPosition(activeRootNode, fieldRangeMarker);
                                        if (rangeMarkerStartPos && _.last(rangeMarkerStartPos) === 0) {
                                            return true;
                                        }
                                    }
                                    return false;
                                }

                                if (self.useSlideMode() && _.isFunction(self.isForcedHardBreakPosition) && self.isForcedHardBreakPosition(paragraph)) {
                                    hyperlinkSelection = Hyperlink.checkForHyperlinkText(self, paragraph, startPosition); // 48216
                                    // at some positions hard break must be inserted instead of splitting the paragraph
                                    var ret = self.insertHardBreak();
                                    if (hyperlinkSelection !== null) {
                                        self.getUndoManager().enterUndoGroup(function () {
                                            Hyperlink.insertHyperlink(self, hyperlinkSelection.start, hyperlinkSelection.end, hyperlinkSelection.url, self.getActiveTarget());
                                        }, null, { preventSelectionChange: true });
                                    }
                                    return ret;
                                }

                                // check for a possible hyperlink text
                                hyperlinkSelection = Hyperlink.checkForHyperlinkText(self, paragraph, startPosition);
                                numAutoCorrect = {};

                                if (isFirstPositionInParagraph &&
                                    isFirstPositionOfFirstParagraph &&
                                    (lastValue >= 4) &&
                                    Position.isPositionInTable(activeRootNode, [0]) &&
                                    startPosition.every(function (value) { return (value === 0); })
                                ) {
                                    //at first check if a paragraph has to be inserted before the current table
                                    self.insertParagraph([0]);
                                    // Setting attributes to new paragraph immediately (task 25670)
                                    self.getParagraphStyles().updateElementFormatting(Position.getParagraphElement(activeRootNode, [0]));
                                    newPosition = [0, 0];
                                } else if (
                                    isFirstPositionInParagraph &&
                                    isFirstPositionOfFirstParagraph &&
                                    (lastValue >= 4) &&
                                    (
                                        (localTablePos = Position.getTableBehindTablePosition(activeRootNode, startPosition)) || // Inserting an empty paragraph between to tables
                                        (localTablePos = Position.getTableAtCellBeginning(activeRootNode, startPosition)) // Inserting an empty paragraph at beginning of table cell
                                    )
                                ) {
                                    // Inserting an empty paragraph between to tables
                                    self.insertParagraph(localTablePos);
                                    // Setting attributes to new paragraph immediately (task 25670)
                                    self.getParagraphStyles().updateElementFormatting(Position.getParagraphElement(activeRootNode, localTablePos));
                                    newPosition = Position.appendNewIndex(localTablePos);
                                } else {
                                    // demote or end numbering instead of creating a new paragraph
                                    paragraphAttrs = self.getParagraphStyles().getElementAttributes(paragraph).paragraph;
                                    listLevel = paragraphAttrs.listLevel;
                                    paragraphLength = Position.getParagraphNodeLength(paragraph);
                                    endOfParagraph = paragraphLength === _.last(startPosition);
                                    split = true;

                                    if (!hasSelection && listLevel >= 0 && paragraphLength === 0) {
                                        listLevel--;
                                        self.getUndoManager().enterUndoGroup(function () {
                                            if (listLevel < 0) {
                                                //remove list label and update paragraph
                                                $(paragraph).children(DOM.LIST_LABEL_NODE_SELECTOR).remove();
                                                self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet(), paragraph: { listStyleId: null, listLevel: -1 } });
                                                self.implParagraphChanged(paragraph);
                                            } else {
                                                self.setAttribute('paragraph', 'listLevel', listLevel);
                                            }
                                        });
                                        split = false;
                                    }

                                    if (paragraph && self.useSlideMode() && DOM.isSlideNode(paragraph)) {
                                        split = false; // #61158 - prevent split on a slide paragraphs
                                    }

                                    if (split === true) {

                                        if (!hasSelection && paragraphLength > 2 && !self.isListParagraph(paragraphAttrs) && !isCursorInsideCxField()) {
                                            // detect Numbering/Bullet labels at paragraph start

                                            if (paragraph !== undefined && !isCxFieldFirstInPar() && self.isAutoDetectionPosition(paragraph)) {
                                                self.validateParagraphNode(paragraph); // 52264, avoiding check of non-breakable space
                                                paraText = paragraph.textContent;
                                                // Task 30826 -> No automatic list generation, if split happens inside the first '. ' substring
                                                if (self.isListAutoDetectionString(startPosition, paragraphLength, paraText)) {
                                                    // Fix for 29508: Not detecting '123. ' or ' xmv. ' as list
                                                    // Fix for 29732: Detecting '- ' and '* ' automatically as list
                                                    labelText = paraText.split(' ')[0];
                                                    numAutoCorrect.listDetection = ListUtils.detectListSymbol(labelText);
                                                    if (numAutoCorrect.listDetection.numberFormat !== undefined) {
                                                        numAutoCorrect.startPosition = _.clone(startPosition);
                                                        numAutoCorrect.startPosition[numAutoCorrect.startPosition.length - 1] = 0;
                                                        numAutoCorrect.endPosition = selection.getEndPosition();
                                                        numAutoCorrect.endPosition[numAutoCorrect.endPosition.length - 1] = labelText.length;
                                                    }
                                                }
                                            }
                                        }

                                        newPosition[lastValue - 1] += 1;
                                        newPosition[lastValue] = 0;

                                        self.getUndoManager().enterUndoGroup(function () {

                                            var localParaPos = _.clone(startPosition),
                                                charStyles = self.getCharacterStyles().getElementAttributes(paragraph.lastChild).character,
                                                attrs = {},
                                                newParagraph = null,
                                                paraAttrs = null,
                                                charAttrs = null,
                                                paraStyleId = null,
                                                isListParagraph = false,
                                                styleAttributes;

                                            if (paragraphLength > 0 && endOfParagraph && charStyles && charStyles.url) {

                                                // Special handling: Inserting new paragraph after return at end of paragraph after hyperlink (task 30742)
                                                paraAttrs = AttributeUtils.getExplicitAttributes(paragraph);
                                                charAttrs = AttributeUtils.getExplicitAttributes(paragraph.lastChild);
                                                paraStyleId = AttributeUtils.getElementStyleId(paragraph);
                                                isListParagraph = self.isListStyleParagraph(null, paraAttrs);

                                                if (charAttrs && charAttrs.character && charAttrs.character.url) { delete charAttrs.character.url; }
                                                if (paraStyleId) { attrs.styleId = paraStyleId; }
                                                if (paraAttrs.paragraph && !_.isEmpty(paraAttrs.paragraph)) { attrs.paragraph = paraAttrs.paragraph; }
                                                if (charAttrs.character && !_.isEmpty(charAttrs.character)) { attrs.character = charAttrs.character; }
                                                // use only paragraph attributes in list paragraphs as character attributes, otherwise merge character attributes, task 30794
                                                if (paraAttrs.character && !_.isEmpty(paraAttrs.character)) {
                                                    if (isListParagraph) {
                                                        attrs.character = paraAttrs.character;
                                                    } else {
                                                        attrs.character = _.extend(paraAttrs.character, attrs.character || {});
                                                    }
                                                }

                                                // checking for 'nextStyleId'
                                                styleAttributes = self.getParagraphStyles().getStyleSheetAttributeMap(paraStyleId);
                                                if (styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId) { attrs.styleId = styleAttributes.paragraph.nextStyleId; }

                                                // modifying the attributes, if changeTracking is activated
                                                if (self.getChangeTrack().isActiveChangeTracking()) {
                                                    attrs.changes = { inserted: self.getChangeTrack().getChangeTrackInfo() };
                                                }

                                                self.applyOperations({ name: Operations.PARA_INSERT, start: newPosition.slice(0, -1), attrs: attrs });

                                                newParagraph = Position.getParagraphElement(activeRootNode, newPosition.slice(0, -1));
                                                self.implParagraphChangedSync($(newParagraph));

                                                selection.setTextSelection(newPosition, null, { simpleTextSelection: false, splitOperation: false });
                                                isSplitOperation = false;

                                                // updating lists, if required
                                                self.handleTriggeringListUpdate(paragraph);

                                                // in Internet Explorer it is necessary to add new empty text nodes in paragraphs again
                                                // newly, also in other browsers (new jQuery version?)
                                                self.repairEmptyTextNodes(newParagraph);

                                            } else {

                                                self.setKeepPreselectedAttributes(true);  // preselected attributes must also be valid in following paragraph (26459)
                                                self.setUseParagraphCache(true);
                                                self.setParagraphCache(paragraph);  // Performance: Caching the currently used paragraph for temporary usage
                                                self.doCheckImplicitParagraph(startPosition);
                                                self.setGUITriggeredOperation(true); // Fix for 30597
                                                self.splitParagraph(startPosition, paragraph);
                                                self.setGUITriggeredOperation(false);
                                                self.setParagraphCache(null);  // Performance: Deleting the currently used paragraph cache
                                                self.setUseParagraphCache(false);

                                                // Special behaviour for splitting an empty paragraph
                                                if (_.last(localParaPos) === 0) { //  || (iPad)) {
                                                    // Fix for 28568: Splitted empty paragraph needs immediately content -> calling validateParagraphNode
                                                    self.validateParagraphNode(paragraph);
                                                }

                                                // Avoiding cursor jumping after 'Enter' at the end of a paragraph, because new paragraph is empty.
                                                if (paragraphLength > 0 && endOfParagraph && Position.getParagraphNodeLength(paragraph.nextSibling) === 0) {
                                                    self.implParagraphChangedSync($(paragraph.nextSibling));
                                                }

                                                // modifying the attributes, if changeTracking is activated
                                                if (self.getChangeTrack().isActiveChangeTracking()) {
                                                    self.applyOperations({
                                                        name: Operations.SET_ATTRIBUTES,
                                                        start: newPosition.slice(0, -1),
                                                        attrs: { changes: { inserted: self.getChangeTrack().getChangeTrackInfo(), removed: null } }
                                                    });
                                                } else {
                                                    if (DOM.isChangeTrackNode(paragraph)) {
                                                        self.applyOperations({
                                                            name: Operations.SET_ATTRIBUTES,
                                                            start: newPosition.slice(0, -1),
                                                            attrs: { changes: { inserted: null, removed: null, modified: null } }
                                                        });
                                                    }
                                                }

                                                // checking 'nextStyleId' at paragraphs
                                                if (endOfParagraph) {

                                                    var styleId = AttributeUtils.getElementStyleId(paragraph),
                                                        styleName = self.getParagraphStyles().getName(styleId);

                                                    styleAttributes = self.getParagraphStyles().getStyleSheetAttributeMap(styleId);
                                                    if (styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId) {
                                                        var nextStyleId = styleAttributes.paragraph.nextStyleId;

                                                        if (nextStyleId === styleId || nextStyleId === styleName) {
                                                            //next style is same style
                                                        } else {
                                                            //listStyleId = null because header want normal text after them
                                                            self.applyOperations({
                                                                name: Operations.SET_ATTRIBUTES,
                                                                start: newPosition.slice(0, -1),
                                                                attrs: { styleId: nextStyleId, paragraph: { listStyleId: null, listLevel: -1 } }
                                                            });
                                                        }
                                                    }
                                                }
                                            }
                                        });

                                    }
                                }

                                // set hyperlink style and url attribute
                                if (hyperlinkSelection !== null) {
                                    self.getUndoManager().enterUndoGroup(function () {
                                        Hyperlink.insertHyperlink(self, hyperlinkSelection.start, hyperlinkSelection.end, hyperlinkSelection.url, self.getActiveTarget());
                                    }, null, { preventSelectionChange: true });
                                }

                                // now apply 'AutoCorrection'
                                if (numAutoCorrect.listDetection && numAutoCorrect.listDetection.numberFormat !== undefined) {
                                    self.getUndoManager().enterUndoGroup(function () {
                                        self.deleteRange(numAutoCorrect.startPosition, numAutoCorrect.endPosition);
                                        self.createList((numAutoCorrect.listDetection.numberFormat === 'bullet') ? 'bullet' : 'numbering', {
                                            listStartValue: numAutoCorrect.listDetection.listStartValue,
                                            left: numAutoCorrect.listDetection.leftString,
                                            right: numAutoCorrect.listDetection.rightString,
                                            symbol: numAutoCorrect.listDetection.symbol,
                                            startPosition: numAutoCorrect.startPosition,
                                            numberFormat: numAutoCorrect.listDetection.numberFormat
                                        });
                                    }, null, { preventSelectionChange: true });
                                }

                                selection.setTextSelection(newPosition, null, { simpleTextSelection: isSimpleTextSelection, splitOperation: isSplitOperation });

                                self.setKeepPreselectedAttributes(false);

                            }  // function doHandleEnterKey

                            var // an optionally modified selection
                                modifiedSelection = null;

                            if (selection.hasRange()) {
                                // a deferred object required for the undo group
                                var enterDef = self.createDeferred('KeyHandlingMixin.pageProcessKeyHandler enterDef');

                                self.getUndoManager().enterUndoGroup(function () {
                                    // Special handling for selected drawing frames
                                    if (selection.isDrawingFrameSelection()) {
                                        // not deleting the drawing -> adding the content into or before the drawing instead of deleting it
                                        modifiedSelection = selection.getValidTextPositionAccordingToDrawingFrame();

                                        if (modifiedSelection) {
                                            // setting the new selection
                                            selection.setTextSelection(modifiedSelection);
                                            // inserting the return before the drawing character into the text frame
                                            doHandleEnterKey();
                                        }

                                    } else {

                                        self.deleteSelected()
                                            .done(function () {
                                                doHandleEnterKey();
                                            })
                                            .always(function () {
                                                enterDef.resolve();   // closing the undo group
                                            });
                                        return enterDef.promise();  // for the undo group
                                    }
                                });

                            } else {
                                doHandleEnterKey();
                            }

                            return $.when();

                        }, 'KeyHandlingMixin.handleEnterKey', inputTextTimeout);

                    } else if (lastKeyDownEvent.shiftKey) {
                        // insert a hard break
                        self.insertHardBreak();
                    }
                }
            } // end of else
        }

        // public methods -----------------------------------------------------

        /**
         * Getting the handler function that is used for processing keydown
         * events on the page node.
         *
         * @returns {Function}
         *  The handler function for the 'keydown' event.
         */
        this.getPageProcessKeyDownHandler = function () {
            return pageProcessKeyDownHandler;
        };

        /**
         * Getting the handler function that is used for processing keydown
         * events on the clipboard node.
         *
         * @returns {Function}
         *  The handler function for the 'keydown' event at clipboard node.
         */
        this.getPageProcessKeyDownHandlerClipboardNode = function () {
            return function (event) {
                if (!app.getView().hasSlidepaneVirtualFocus()) {
                    pageProcessKeyDownHandler.call(this, event);
                }
            };
        };

        /**
         * Getting the handler function that is used for processing keypressed
         * events on the page node.
         *
         * @returns {Function}
         *  The handler function for the 'keypressed' event.
         */
        this.getPageProcessKeyPressHandler = function () {
            return pageProcessKeyPressHandler;
        };

        /**
         * Getting the handler function that is used for processing keypressed
         * events on the clipboard node.
         *
         * @returns {Function}
         *  The handler function for the 'keydown' event at clipboard node.
         */
        this.getPageProcessKeyPressHandlerClipboardNode = function () {
            return function (event) {
                if (!app.getView().hasSlidepaneVirtualFocus()) {
                    pageProcessKeyPressHandler.call(this, event);
                }
            };
        };

        /**
         * Setting the content of 'activeInputText', that is the
         * collector for the inserted characters.
         *
         * @param {String} value
         *  The new value of the character collector
         */
        this.setActiveInputText = function (value) {
            activeInputText = value;
        };

        /**
         * Getting the content of 'activeInputText', that is the
         * collector for the inserted characters.
         *
         * @returns {String}
         */
        this.getActiveInputText = function () {
            return activeInputText;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = insertTextDef = null;
        });

    } // class KeyHandlingMixin

    // exports ================================================================

    return KeyHandlingMixin;

});
