/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/view/rulerpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/baseframework/view/toolpane',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/settings/units',
    'io.ox/office/text/view/dialog/paragraphdialog',
    'gettext!io.ox/office/text/main'
], function (Utils, Tracking, Forms, KeyCodes, Canvas, ToolPane, ToolBar, AttributeUtils, Operations, Position, Units, ParagraphDialog, gt) {

    'use strict';

    // class RulerPane =======================================================

    /**
     * Represents the ruler in OX Documents edit applications.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditView} docView
     *  The document view instance containing this ruler pane.
     */
    var RulerPane = ToolPane.extend({ constructor: function (docView) {
        var self = this;
        // application instance
        var app = docView.getApp();
        // the model
        var docModel = docView.getDocModel();
        var selection = docModel.getSelection();
        var rulerBar = new ToolBar(docView);
        // container node of the ruler
        var rulerPaneNode = null;
        // app content root node (also known as scrolling node of the app)
        var scrollingNode = null;
        // page specific values in Pixels
        var pageWidth = 0;
        var minPageWidth = 0;
        var innerPageWidth = 0;
        var marginLeft = 0, marginRight = 0;
        // jQuery nodes used for building ruler
        var leadingNode = null, centerNode = null, trailingNode = null;
        var mainInnerNode = null;
        var mainInnerRuler = null;
        var leftInnerRuler = null;
        var marginLeftNode = null, marginRightNode = null;
        var controlsContainer = null;
        var leftIndentCtrl = null, rightIndentCtrl = null, firstIndentCtrl = null;
        // unit based on locale of appsuite
        var localeDocUnit = Units.getStandardUnit();
        var quarterUnit = Utils.convertLength(0.25, (localeDocUnit === 'mm' ? 'cm' : localeDocUnit), 'px');
        // predefined minimum value for paragraph width
        var minParaWidthHmm = 500;
        // saved value of the last paragraph position (used for performance, to compare if paragraph selection has changed)
        var storedParaPos = [];
        var storedTarget = '';
        // global Ruler Pane arguments stored during call of debounced method
        var globalRpEventType = null, globalRpOtions = null;
        // global scroll left value for ruler pane (usually on zoom in)
        var globalLeftScroll = 0;
        // whether the ruler pane is visible after the document is loaded
        var isVisibleRulerPaneAfterLoad = false;
        // all tracking event names except tracking start
        var TRACKING_EVENT_NAMES = 'tracking:move tracking:end tracking:cancel';
        // canvas image for indent controls
        var indentCtrlsImg = null;

        // base constructor ---------------------------------------------------

        ToolPane.call(this, docView, { position: 'top', classes: 'standard-design ruler-pane' });

        // private methods ----------------------------------------------------

        // debounced method for repainting ruler pane
        var paintRulerPanesDebounced = docModel.createDebouncedMethod('RulerPane.paintRulerPanesDebounced', storeRulerPaneData, paintRulerPane, { delay: 200, animFrame: true });

        /**
         * Adds quarter, half and full unit values to the given ruler node.
         * Quarter is 1/4 of the vertical line, half is 1/2 of the vertical line,
         * and full unit is represented as digit.
         *
         * @param {jQuery} node
         *  Ruler node (main or left) to be filled with values.
         *  (Left ruler has mirrored numbering)
         * @param {Number} width
         *  Width of the node in px.
         * @param {Number} zoomFactor
         *  Zoom factor of the document.
         * @param {Object} [options]
         *   @param {Boolean} [options.reverse=false]
         *      If node should be filled with reversed values (like left ruler), or not.
         */
        function fillRulerWithValues(node, width, zoomFactor, options) {
            var distancePx = quarterUnit * zoomFactor; // 1/4 of the unit
            var reverse = Utils.getBooleanOption(options, 'reverse', false);
            var count = (reverse) ? Utils.roundDown(width / distancePx, 4)  : width / distancePx; // make it divisable by 4, as left ruler (reverse) shows only full units
            var nodesArr = [];

            for (var i = 0; i < count; i++) {
                var extraClass = (i % 2 === 0) ? (i % 4 === 0 ? 'fullUnit' : 'halfUnit') : 'quarterUnit';
                var delimiterNode = $('<div class="delimiter ' + extraClass + '" style="width:' + (distancePx) + 'px">');
                nodesArr.push(delimiterNode);
            }
            if (reverse) {
                leftInnerRuler.css('counterReset', 'left ' + (count / 4 + 1)); // +1 as it's 0 based
            }
            node.empty().append(nodesArr);
        }

        /**
         * Generates indent controls from canvas and exports image
         * Same image is used for left, right and first indend,
         * except that for first css scale transform is used to flip image.
         *
         * @returns{CanvasDataURL}
         */
        function createIndentCtrlsImg() {
            var canvas = new Canvas(docModel);

            canvas.initialize({ width: 10, height: 10 });
            canvas.clear().render(function (ctx) {
                ctx.setFillStyle('#fff');
                ctx.setLineStyle({ style: '#000', width: 0.6 });
                var path = ctx.createPath().pushPolygon(0.5, 9, 9.5, 9, 9.5, 6.5, 5, 0, 0.5, 6.5);
                ctx.drawPath(path, 'all');
            });

            return canvas.getDataURL();
        }

        /**
         * Creates new positions for firstLine, left and right indent markers in ruler.
         */
        function repositionIndentCtrls() {
            var zoomFactor = docView.getZoomFactor();
            var startPos = selection.getStartPosition();
            var paraPos = _.initial(startPos);
            var paraOffset = null;
            var indentLeft, indentRight, indentFirstLine;
            var paragraph = Position.getParagraphElement(selection.getRootNode(), paraPos);
            var attrs = paragraph && docModel.getParagraphStyles().getElementAttributes(paragraph);
            var paraAttrs = (attrs && attrs.paragraph) || null;
            var explAttrs = AttributeUtils.getExplicitAttributes(paragraph);
            var explParaAttrs = (explAttrs && explAttrs.paragraph) || null;
            var explIndentLeft = null;
            var explIndentRight = null;
            var explIndentFirst = null;
            var listIndentLeft = null;
            var listIndentRight = null;
            var listIndentFirst = null;

            if (paraAttrs) {
                if (explParaAttrs) {
                    explIndentLeft = explParaAttrs.indentLeft;
                    explIndentRight = explParaAttrs.indentRight;
                    explIndentFirst = explParaAttrs.indentFirstLine;
                }
                if (paraAttrs.listStyleId) {
                    var listAttrs = docModel.getListCollection().getListLevel(paraAttrs.listStyleId, paraAttrs.listLevel || 0);
                    if (listAttrs) {
                        listIndentLeft = listAttrs.indentLeft;
                        listIndentRight = listAttrs.indentRight;
                        listIndentFirst = listAttrs.indentFirstLine;
                    }
                }
                // priority order: explicit attribute > list style > paragraph style
                indentLeft = _.isFinite(explIndentLeft) ? explIndentLeft : (_.isFinite(listIndentLeft) ? listIndentLeft : paraAttrs.indentLeft) || 0;
                indentRight = _.isFinite(explIndentRight) ? explIndentRight : (_.isFinite(listIndentRight) ? listIndentRight : paraAttrs.indentRight) || 0;
                indentFirstLine = indentLeft + (_.isFinite(explIndentFirst) ? explIndentFirst : (_.isFinite(listIndentFirst) ? listIndentFirst : paraAttrs.indentFirstLine) || 0);
                // converting hmm -> px
                indentLeft = Utils.convertHmmToLength(indentLeft, 'px', 0.001) * zoomFactor;
                indentRight = Utils.convertHmmToLength(indentRight, 'px', 0.001) * zoomFactor;
                indentFirstLine = Utils.convertHmmToLength(indentFirstLine, 'px', 0.001) * zoomFactor;

                if (docModel.isPositionInTable()) {
                    paraOffset = ($(paragraph).parent().offset().left - $(selection.getEnclosingTable()).offset().left);
                    indentLeft += paraOffset;
                    indentFirstLine += paraOffset;
                    indentRight = innerPageWidth - indentLeft - $(paragraph).width() * zoomFactor;
                } else if (selection.isAdditionalTextframeSelection()) {
                    var drawingTextframe = $(selection.getSelectedDrawings()[0]).find('.textframe');
                    paraOffset = ($(paragraph).parent().offset().left - drawingTextframe.offset().left);
                    indentLeft += paraOffset;
                    indentFirstLine += paraOffset;
                    indentRight = innerPageWidth - indentLeft - $(paragraph).width() * zoomFactor;
                }

                leftIndentCtrl.css('left', indentLeft);
                rightIndentCtrl.css('left', innerPageWidth - indentRight);
                firstIndentCtrl.css('left', indentFirstLine);
            } else {
                leftIndentCtrl.css('left', 0);
                rightIndentCtrl.css('left', innerPageWidth);
                firstIndentCtrl.css('left', 0);
            }
        }

        /**
         * Direct function callback for method paintRulerPanesDebounced.
         * Stores parameters used for debounced function paintRulerPane.
         *
         * @param {String} eventType
         * @param {Object} [options]
         *  @param {Number} [options.leftAddition=0]
         *  @param {Number} [options.rightAddition=0]
         */
        function storeRulerPaneData(eventType, options) {
            globalRpEventType = eventType;
            globalRpOtions = options;
        }

        /**
         * Debounced function callback for method paintRulerPanesDebounced.
         */
        function paintRulerPane() {
            var pageAttributes = docModel.getPageStyles().getElementAttributes(docModel.getNode());
            var zoomFactor = docView.getZoomFactor();
            var options = globalRpOtions;
            var eventType = globalRpEventType;
            var preLeftOffset = Utils.getNumberOption(options, 'leftAddition', 0);
            var preRightOffset = Utils.getNumberOption(options, 'rightAddition', 0);

            // re-initialize values because of possible zoom or page settings change
            minPageWidth = Utils.convertHmmToLength(minParaWidthHmm, 'px', 0.001) * zoomFactor; // preset min width value 0,5cm
            if (docModel.isDraftMode()) {
                var pageNode = docModel.getNode();
                var pageNodeOuterWidth = pageNode.outerWidth();
                var pageNodeInnerWidth = pageNode.width();
                marginLeft = (pageNodeOuterWidth - pageNodeInnerWidth) / 2 + preLeftOffset;
                marginRight = (pageNodeOuterWidth - pageNodeInnerWidth) / 2;
                pageWidth = pageNodeOuterWidth;
            } else {
                marginLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 0.001) * zoomFactor + preLeftOffset;
                marginRight = Utils.convertHmmToLength(pageAttributes.page.marginRight, 'px', 0.001) * zoomFactor;
                pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 0.001) * zoomFactor;
            }
            innerPageWidth = pageWidth - marginLeft - marginRight;
            // predefined right offset - tables and drawings
            if (preRightOffset) {
                innerPageWidth = preRightOffset;
                marginRight = pageWidth - marginLeft - innerPageWidth;
            }

            var leftRulerWidth = pageWidth - marginRight - minPageWidth;
            var fillSpace = docModel.getNode().offset().left + globalLeftScroll; // page node left offset

            if (globalLeftScroll) { rulerPaneNode.css('left', -globalLeftScroll); }
            leadingNode.css('width', fillSpace);
            trailingNode.css('width', fillSpace);
            // initial indent ctrls positioning
            controlsContainer.css('left', fillSpace + marginLeft - 5); // minus half width of indent control arrow
            repositionIndentCtrls();

            // for performance reasons call fillRuler methods only when really needed
            if ((eventType !== 'pagination:finished' && eventType !== 'pane:layout') || !mainInnerRuler.children().length) {
                mainInnerRuler.css('width', pageWidth);
                mainInnerNode.css('width', innerPageWidth);
                marginLeftNode.css('width', marginLeft);
                marginRightNode.css('width', marginRight);

                centerNode.toggleClass('small-zoom', zoomFactor < 1 && localeDocUnit !== 'in');

                fillRulerWithValues(leftInnerRuler, leftRulerWidth, zoomFactor, { reverse: true });
                fillRulerWithValues(mainInnerRuler, pageWidth, zoomFactor);
            }
            globalRpEventType = null;
            globalRpOtions = null;
        }

        /**
         * Generates specific left and right addition values,
         * when ruler needs to be painted for tables, nested tables or shapes.
         */
        function generateTableDrawingOptions() {
            var zoomFactor = docView.getZoomFactor();
            var options = null;
            var leftAddition, rightAddition;
            if (docModel.isPositionInTable()) {
                var table = $(selection.getEnclosingTable());
                leftAddition = table.offset().left - docModel.getNode().children('.pagecontent').offset().left;
                rightAddition = (leftAddition !== 0) ? table.width() * zoomFactor : 0;

                options = { leftAddition: leftAddition, rightAddition: rightAddition };
            } else if (selection.isAdditionalTextframeSelection()) {
                var drawingTextframe = $(selection.getSelectedDrawings()[0]).find('.textframe');
                leftAddition = drawingTextframe.offset().left - docModel.getNode().children('.pagecontent').offset().left;
                rightAddition = drawingTextframe.width() * zoomFactor;

                options = { leftAddition: leftAddition, rightAddition: rightAddition };
            }
            return options;
        }

        // -------------------------------HANDLERS-------------------------------

        function paneLayoutHandler(event) {
            if (!self.isVisible() || !scrollingNode) { return; }

            paintRulerPanesDebounced(event.type, generateTableDrawingOptions());
        }

        function paneHideHandler() {
            mainInnerRuler.empty();
        }

        function togglePaneHandler(event) {
            // removing border-top from app-pane node when ruler is visible
            docView.getAppPaneNode().toggleClass('ruler-active', (event.type === 'pane:show'));
        }

        function mouseUpHandler() {
            selection.restoreBrowserSelection();
        }

        function selectionChangeHandler() {
            if (!self.isVisible() || !scrollingNode) { return; }
            var startPos = selection.getStartPosition();
            var paraPos = _.initial(startPos);
            var target = docModel.getActiveTarget();

            if (Utils.equalArrays(paraPos, storedParaPos) && storedTarget === target) {
                return;
            } else {
                storedParaPos = paraPos;
                storedTarget = target;
            }

            paintRulerPanesDebounced('ruler:initialize', generateTableDrawingOptions());
        }

        function trackingStartHandler(event) {
            $(event.target).off(TRACKING_EVENT_NAMES);
            if (app.isEditable()) {
                $(event.target).on(TRACKING_EVENT_NAMES, indentMarkerHandler);
                indentMarkerHandler.call(this, event);
            }
        }

        function enableIndentMarkersHandler() {
            leftIndentCtrl.removeClass('disabled');
            firstIndentCtrl.removeClass('disabled');
            rightIndentCtrl.removeClass('disabled');
        }

        function disableIndentMarkersHandler() {
            leftIndentCtrl.addClass('disabled');
            firstIndentCtrl.addClass('disabled');
            rightIndentCtrl.addClass('disabled');
        }

        /**
         * Handler for tracking events of indent markers on ruler pane.
         */
        var indentMarkerHandler = (function () {
            var markerNode = null;
            var originalLeftPos = null;
            var complLeftPos = null;
            var createdOp = false;
            var isLeftMarker = false;
            var isRightMarker = false;
            var resizeLine = $('<div>').addClass('resizeline');
            var zoomFactor = null;
            var diffPos = 0;

            function startMoveMarker(event) {
                zoomFactor = docView.getZoomFactor();
                var activePageHeight = (docModel.getNode().height() + 60) * zoomFactor; // 60 for margins and extra space

                markerNode = $(event.target);
                isRightMarker = markerNode.hasClass('right-ind-ctrl');
                isLeftMarker = markerNode.hasClass('left-ind-ctrl');
                originalLeftPos = parseFloat(markerNode.css('left'));
                complLeftPos = isRightMarker ? Math.max(parseFloat(leftIndentCtrl.css('left')), parseFloat(firstIndentCtrl.css('left'))) : parseFloat(rightIndentCtrl.css('left'));
                diffPos = (isLeftMarker) ? originalLeftPos - parseFloat(firstIndentCtrl.css('left')) : 0;
                if (isLeftMarker && diffPos < 0) { complLeftPos += diffPos; }

                resizeLine.css({ width: '1px', height: activePageHeight, left: (event.startX + globalLeftScroll), top: '0px', zIndex: '99' });
                scrollingNode.append(resizeLine);
            }

            function updateMoveMarker(event) {
                var updatePos = (originalLeftPos + event.offsetX);
                var minBoundary = isRightMarker ? (minPageWidth + complLeftPos) : ((isLeftMarker && diffPos > 0) ? -marginLeft + diffPos : -marginLeft);
                var maxBoundary = isRightMarker ? (innerPageWidth + marginRight) : (complLeftPos - minPageWidth);

                updatePos = Utils.roundUp(updatePos, quarterUnit * zoomFactor); // first round up to full quarter unit value
                updatePos = Utils.minMax(updatePos, minBoundary, maxBoundary); // then check boundaries

                markerNode.css('left', updatePos);
                if (isLeftMarker) { firstIndentCtrl.css('left', updatePos - diffPos); }
                resizeLine.css('left', markerNode.offset().left + globalLeftScroll + 4);
            }

            function endMoveMarker() {
                var generator = docModel.createOperationGenerator();
                var target = docModel.getActiveTarget();
                var paraPos = _.initial(selection.getStartPosition());
                var operationProperties = { attrs: { paragraph: {} } };
                var tableAddition = 0;
                var localInnerWidth = innerPageWidth;
                var rightIndPosHmm, leftIndPosHmm, firstIndPosHmm;

                if (Math.abs(originalLeftPos - parseFloat(markerNode.css('left'))) < 1) { return; } // no ops generation if marker didn't move

                if (docModel.isPositionInTable()) {
                    var paragraph = Position.getParagraphElement(selection.getRootNode(), paraPos);
                    var cellcontent = $(paragraph).parent(); // cellcontent is a reference for indentation
                    var paraOffset = (cellcontent.offset().left - $(selection.getEnclosingTable()).offset().left);

                    tableAddition = paraOffset;
                    localInnerWidth = paraOffset + cellcontent.width() * zoomFactor;
                }

                if (isRightMarker) {
                    var rightIndPos = (localInnerWidth - parseFloat(rightIndentCtrl.css('left'))) / zoomFactor;
                    rightIndPosHmm = Utils.convertLengthToHmm(rightIndPos, 'px');
                    operationProperties.attrs.paragraph.indentRight = rightIndPosHmm;
                } else if (isLeftMarker) {
                    var leftIndPos = (parseFloat(leftIndentCtrl.css('left')) - tableAddition) / zoomFactor;
                    leftIndPosHmm = Utils.convertLengthToHmm(leftIndPos, 'px');
                    operationProperties.attrs.paragraph.indentLeft = leftIndPosHmm;
                } else { // first indent marker
                    var firstIndPos = (parseFloat(firstIndentCtrl.css('left')) - parseFloat(leftIndentCtrl.css('left'))) / zoomFactor;
                    firstIndPosHmm = Utils.convertLengthToHmm(firstIndPos, 'px');
                    operationProperties.attrs.paragraph.indentFirstLine = firstIndPosHmm;
                }

                selection.iterateContentNodes(function (paragraph, position) {
                    var paraWidthHmm = Utils.convertLengthToHmm($(paragraph).parent().width(), 'px');
                    var attrs = docModel.getParagraphStyles().getElementAttributes(paragraph);
                    var paraAttrs = (attrs && attrs.paragraph) || null;
                    var totalIndentation = 0;

                    if (paraAttrs) {
                        var paraTxtPos = _.clone(position);
                        paraTxtPos.push(0);
                        docModel.doCheckImplicitParagraph(paraTxtPos); // check if paragraph is implicit

                        var opProperties = _.copy(operationProperties, true);

                        if (isRightMarker) {
                            totalIndentation = rightIndPosHmm + paraAttrs.indentLeft + minParaWidthHmm;
                            if (paraWidthHmm <= totalIndentation) {
                                opProperties.attrs.paragraph.indentRight = rightIndPosHmm - (totalIndentation - paraWidthHmm + minParaWidthHmm);
                            }
                        } else if (isLeftMarker) {
                            totalIndentation = leftIndPosHmm + paraAttrs.indentRight + minParaWidthHmm;
                            if (paraWidthHmm <= totalIndentation) {
                                opProperties.attrs.paragraph.indentLeft = leftIndPosHmm - (totalIndentation - paraWidthHmm + minParaWidthHmm);
                            }
                        } else { // first indent marker
                            totalIndentation = firstIndPosHmm + paraAttrs.indentRight + minParaWidthHmm;
                            if (paraWidthHmm <= totalIndentation) {
                                opProperties.attrs.paragraph.indentFirstLine = firstIndPosHmm - (totalIndentation - paraWidthHmm + minParaWidthHmm);
                            }
                        }

                        opProperties.start = position;
                        if (target) { opProperties.target = target; }
                        if (docModel.getChangeTrack().isActiveChangeTracking()) { opProperties.attrs.changes = { modified: docModel.getChangeTrack().getChangeTrackInfo() }; }
                        generator.generateOperation(Operations.SET_ATTRIBUTES, opProperties);
                    }
                });

                docModel.applyOperations(generator);
                docView.scrollToSelection();
                createdOp = true;
            }

            function finalizeMoveMarker() {
                if (!createdOp) { // restore start values for markers if tracking is canceled (no operation created)
                    markerNode.css('left', originalLeftPos);
                    if (isLeftMarker) { firstIndentCtrl.css('left', originalLeftPos - diffPos); }
                }

                resizeLine.remove();
                // reset values
                createdOp = false;
                markerNode = null;
                originalLeftPos = null;
                complLeftPos = null;
                isLeftMarker = false;
                isRightMarker = false;
                zoomFactor = null;
                diffPos = 0;
            }

            // return the actual indentMarkerHandler() function
            return function (event) {
                event.preventDefault();
                switch (event.type) {
                    case 'tracking:start':
                        startMoveMarker(event);
                        break;
                    case 'tracking:move':
                        updateMoveMarker(event);
                        break;
                    case 'tracking:end':
                        endMoveMarker(event);
                        finalizeMoveMarker(event);
                        break;
                    case 'tracking:cancel':
                        finalizeMoveMarker(event);
                        break;
                }
                event.stopPropagation();
            };
        }());

        // initialization -----------------------------------------------------

        // initially not visible ruler pane - shown only after successful import (but preparing space to avoid moving of document)
        isVisibleRulerPaneAfterLoad = !Utils.SMALL_DEVICE && app.getUserSettingsValue('showRulerPane', true);
        this.toggle(isVisibleRulerPaneAfterLoad);
        if (isVisibleRulerPaneAfterLoad) { docView.getAppPaneNode().addClass('ruler-active'); } // hiding border-top of app-pane node

        app.waitForImportSuccess(function () {
            rulerPaneNode   = self.getNode();
            centerNode      = rulerPaneNode.children('.center').children('.area-container');
            leadingNode     = rulerPaneNode.children('.leading');
            trailingNode    = rulerPaneNode.children('.trailing');

            leftInnerRuler      = $('<div class="left-inner-ruler">');
            marginLeftNode      = $('<div class="margin-left-node">').append(leftInnerRuler);
            mainInnerRuler      = $('<div class="main-inner-ruler">');
            mainInnerNode       = $('<div class="inner-node">').append(mainInnerRuler);
            marginRightNode     = $('<div class="margin-right-node">');

            controlsContainer   = $('<div class="controls-container">');
            leftIndentCtrl      = $('<div class="left-ind-ctrl">');
            rightIndentCtrl     = $('<div class="right-ind-ctrl">');
            firstIndentCtrl     = $('<div class="first-ind-ctrl">');

            Forms.setToolTip(leftIndentCtrl, gt('Left indent'));
            Forms.setToolTip(rightIndentCtrl, gt('Right indent'));
            Forms.setToolTip(firstIndentCtrl, gt('First line indent'));

            if (!indentCtrlsImg) { indentCtrlsImg = createIndentCtrlsImg(); }
            _.each([firstIndentCtrl, leftIndentCtrl, rightIndentCtrl], function (indCtrl) {
                var img = $('<img src="' + indentCtrlsImg + '" width="10" height="10">');
                $(indCtrl).append(img);
            });

            controlsContainer.append(firstIndentCtrl, leftIndentCtrl, rightIndentCtrl);
            rulerPaneNode.append(controlsContainer);
            centerNode.append(marginLeftNode, mainInnerNode, marginRightNode);

            // visibility
            self.toggle(app.getUserSettingsValue('showRulerPane', true));
            docView.getAppPaneNode().toggleClass('ruler-active', (app.getUserSettingsValue('showRulerPane', true)));

            self.addViewComponent(rulerBar);

            scrollingNode = app.getWindowNode().find('.scrolling');

            // call ruler pane initially
            paneLayoutHandler({ type: 'ruler:initialize' });
            if (!app.isEditable()) { disableIndentMarkersHandler(); }

            // event listeners
            self.listenTo(self, 'pane:layout', paneLayoutHandler);
            self.listenTo(self, 'pane:hide', paneHideHandler);
            self.listenTo(self, 'pane:show pane:hide', togglePaneHandler);
            self.listenTo(docView, 'change:zoom', paneLayoutHandler);
            self.listenTo(docModel, 'change:pageSettings pagination:finished ruler:initialize', paneLayoutHandler);
            self.listenTo(selection, 'change', selectionChangeHandler);

            self.listenTo(rulerPaneNode, 'mouseup', mouseUpHandler);

            _.each([firstIndentCtrl, leftIndentCtrl, rightIndentCtrl], function (ctrl) {
                self.listenTo(ctrl, 'dblclick', function () {
                    if (app.isEditable()) {
                        return new ParagraphDialog(app.getView()).show(); // open dialog with paragraph properties
                    }
                });
            });

            self.listenTo(app, 'docs:editmode:leave', disableIndentMarkersHandler);
            self.listenTo(app, 'docs:editmode:enter', enableIndentMarkersHandler);

            // scroll ruler with scrolling node horizontally
            self.listenTo(scrollingNode, 'scroll', function () {
                globalLeftScroll = scrollingNode.scrollLeft();
                rulerPaneNode.css('left', -globalLeftScroll);
            });

            // tracking for the indentation markers
            Tracking.enableTracking(firstIndentCtrl);
            docModel.listenTo(firstIndentCtrl, 'tracking:start', trackingStartHandler);
            Tracking.enableTracking(leftIndentCtrl);
            docModel.listenTo(leftIndentCtrl, 'tracking:start', trackingStartHandler);
            Tracking.enableTracking(rightIndentCtrl);
            docModel.listenTo(rightIndentCtrl, 'tracking:start', trackingStartHandler);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = app = docModel = docView = selection = rulerBar = null;
            rulerPaneNode = leadingNode = centerNode = trailingNode = leftInnerRuler = null;
            controlsContainer = firstIndentCtrl = leftIndentCtrl = rightIndentCtrl = marginLeftNode = mainInnerNode = marginRightNode = null;
            indentCtrlsImg = null;
        });

    } }); // class RulerPane

    // exports ================================================================

    return RulerPane;

});
