/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/view/dialog/paragraphdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/dialog/basedialog',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/operations',
    'gettext!io.ox/office/text/main',
    'less!io.ox/office/text/view/dialog/paragraphdialog'
], function (Utils, Forms, BaseDialog, AttributeUtils, EditControls, Position, Operations, gt) {

    'use strict';

    // Global constants and variables
    var paraDialogTitle = /*#. Change paragraph properties from dialog menu*/ gt('Paragraph');

    var dialogWidth = Utils.SMALL_DEVICE ? 320 : 420;

    // class ParagraphDialog ==================================================

    /**
     * The paragraph modal dialog.
     * Provides possibility to set paragraph indentation values (left, right, first line, hanging).
     *
     * The changes will be applied to the current document after clicking the 'Ok' button.
     *
     * The dialog itself is shown by calling this.execute. The caller can react on successful
     * finalization of the dialog within an appropriate done-Handler, chained to the
     * this.execute function call.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {TextView} docView
     *  The view instance containing this editor instance.
     */
    var ParagraphDialog = BaseDialog.extend(function (docView) {

        var self = this;
        var docModel = docView.getDocModel();
        var selection = docModel.getSelection();
        var controlRootNode = $('<div class="para-dialog-control">');
        var paraIndAttrs = [];
        // page specific values in hmm
        var pageWidth = 0;
        var marginLeft = 0;
        var marginRight = 0;
        var minPageWidth = 0;
        var minParaWidthHmm = 500;

        // base constructor ---------------------------------------------------

        BaseDialog.call(this, docView, { title: paraDialogTitle, width: dialogWidth });

        // private methods ----------------------------------------------------

        /**
         * Spinner wrapper.
         * @param {Number} min Value
         * @param {Number} max Value
         * @param {String} label String for the label
         * @param {String} tooltip
         */
        var Spinner = EditControls.SpinField.extend({ constructor: function (min, max, dataValue, label, tooltip) {
            var spinnerValue,
                spinnerNode,
                cleanLabel  = label ? Utils.trimAndCleanString(label.toLowerCase().replace(' ', '')) : false,
                labelNode = Forms.createElementMarkup('label', { attributes: { class: 'spin-field-label', for: cleanLabel }, content: label });

            EditControls.UnitField.call(this, docView, {
                width: 70,
                tooltip: tooltip,
                min: min,
                max: max,
                smallerVersion: { hide: true }
            });

            this.setValue = function (value) {
                var minVal, maxVal;
                var indentLeft = paraIndAttrs.indentLeft || 0;
                var indentRight = paraIndAttrs.indentRight || 0;
                var indentFirstLine = paraIndAttrs.indentFirstLine || 0;

                // re-validate min/max boundaries after setting values
                if (dataValue === 'indentLeft') {
                    minVal = -marginLeft;
                    maxVal = pageWidth - (marginLeft + indentFirstLine) - (marginRight + indentRight) - minPageWidth;
                } else if (dataValue === 'indentRight') {
                    minVal = -marginRight;
                    maxVal = pageWidth - marginRight - (marginLeft + Math.max(indentLeft, indentLeft + indentFirstLine)) - minPageWidth;
                } else {
                    minVal = -marginLeft;
                    maxVal = pageWidth - (marginLeft + indentLeft) - (marginRight + indentRight) - minPageWidth;
                }
                value = Utils.minMax(value, minVal, maxVal);
                spinnerValue = value;
                this.setFieldValue(spinnerValue);
                this.getInputNode().trigger('input');
                paraIndAttrs[dataValue] = value;
            };

            this.triggerChange = function (value, options) {
                if (!_.isNull(value)) {
                    this.setValue(value);
                    if (options.sourceEvent.keyCode === 13) {
                        self.invoke('ok');
                    }
                } else {
                    this.setValue(0);
                }
            };

            /**
             * Returns the node of the spinner
             * @returns {jQuery} the spinner node.
             */
            this.node = function () {
                return spinnerNode || (spinnerNode = $(this.getNode()));
            };

            /**
             * Returns the label for the spinner.
             * @returns {String} markup of the label.
             */
            this.label = function () {
                return labelNode;
            };
            /**
             * Returns the value of the spinner.
             * @returns {String} markup of the label.
             */
            this.getValue = function () {
                return spinnerValue;
            };
        } });

        /**
         * Initialize controls of the dialog.
         */
        function initControls() {
            var indentHeadingNode = $('<h4 class="para-dialog-heading">').text(gt('Indentation'));
            var startPos = selection.getStartPosition();
            var paraPos = _.initial(startPos);
            var indentLeft, indentRight, indentFirstLine;
            var paragraph = Position.getParagraphElement(selection.getRootNode(), paraPos);
            var attrs = paragraph && docModel.getParagraphStyles().getElementAttributes(paragraph);
            var paraAttrs = (attrs && attrs.paragraph) || null;
            var explAttrs = AttributeUtils.getExplicitAttributes(paragraph);
            var explParaAttrs = (explAttrs && explAttrs.paragraph) || null;
            var explIndentLeft, explIndentRight, explIndentFirst;
            var listIndentLeft, listIndentRight, listIndentFirst;
            var minBoundary = -10000;
            var maxBoundary = 50000;
            // page attribute values
            var pageAttributes = docModel.getPageStyles().getElementAttributes(docModel.getNode());
            minPageWidth = minParaWidthHmm;
            marginLeft = pageAttributes.page.marginLeft;
            marginRight = pageAttributes.page.marginRight;
            pageWidth = pageAttributes.page.width;

            if (paraAttrs) {
                if (explParaAttrs) {
                    explIndentLeft = explParaAttrs.indentLeft;
                    explIndentRight = explParaAttrs.indentRight;
                    explIndentFirst = explParaAttrs.indentFirstLine;
                }
                if (paraAttrs.listStyleId) {
                    var listAttrs = docModel.getListCollection().getListLevel(paraAttrs.listStyleId, paraAttrs.listLevel || 0);
                    if (listAttrs) {
                        listIndentLeft = listAttrs.indentLeft;
                        listIndentRight = listAttrs.indentRight;
                        listIndentFirst = listAttrs.indentFirstLine;
                    }
                }
                // priority order: explicit attribute > list style > paragraph style
                indentLeft = _.isFinite(explIndentLeft) ? explIndentLeft : (_.isFinite(listIndentLeft) ? listIndentLeft : paraAttrs.indentLeft) || 0;
                indentRight = _.isFinite(explIndentRight) ? explIndentRight : (_.isFinite(listIndentRight) ? listIndentRight : paraAttrs.indentRight) || 0;
                indentFirstLine = _.isFinite(explIndentFirst) ? explIndentFirst : (_.isFinite(listIndentFirst) ? listIndentFirst : paraAttrs.indentFirstLine) || 0;

                // spinner fields
                // -> left indentation
                var leftIndSpinner = new Spinner(minBoundary, maxBoundary, 'indentLeft', gt('Left'), gt('Set left indent'));
                var leftIndNode = $('<div class="ind-wrapper left-ind">').append(leftIndSpinner.label(), leftIndSpinner.node());
                // -> right indentation
                var rightIndSpinner = new Spinner(minBoundary, maxBoundary, 'indentRight', gt('Right'), gt('Set right indent'));
                var rightIndNode = $('<div class="ind-wrapper right-ind">').append(rightIndSpinner.label(), rightIndSpinner.node());
                // -> first line indentation
                var firstIndSpinner = new Spinner(minBoundary, maxBoundary, 'indentFirstLine', gt('First line'), gt('Set paragraph\'s first line indent'));
                var firstIndNode = $('<div class="ind-wrapper first-ind">').append(firstIndSpinner.label(), firstIndSpinner.node());

                leftIndSpinner.setValue(indentLeft);
                rightIndSpinner.setValue(indentRight);
                firstIndSpinner.setValue(indentFirstLine);
                controlRootNode.append(indentHeadingNode, leftIndNode, rightIndNode, firstIndNode);
            }
        }

        // initialization -----------------------------------------------------

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // initialize the body element of the dialog
        this.getBody()
            .addClass('io-ox-office-text-paragraph-dialog')
            .toggleClass('mobile', !!Utils.SMALL_DEVICE)
            .append(controlRootNode);

        // create the layout of the dialog
        initControls();

        // set initial control focus
        this.setFocusNode(controlRootNode);

        // handler for the OK button
        this.setOkHandler(function () {
            var generator = docModel.createOperationGenerator();
            var target = docModel.getActiveTarget();
            var zoomFactor = docView.getZoomFactor();

            selection.iterateContentNodes(function (paragraph, position) {
                var opProperties = { attrs: { paragraph: {} } };
                var paraTxtPos = _.clone(position);
                var paraIndRight = paraIndAttrs.indentRight;
                var paraIndLeft = paraIndAttrs.indentLeft;
                var paraIndFirst = paraIndAttrs.indentFirstLine;
                paraTxtPos.push(0);
                docModel.doCheckImplicitParagraph(paraTxtPos); // check if paragraph is implicit

                // Re-adjust ind values if they are overlapped
                var paraWidthHmm = Utils.convertLengthToHmm(($(paragraph).parent().width() * zoomFactor), 'px');

                if (paraWidthHmm < minParaWidthHmm + paraIndRight + Math.max(paraIndLeft, paraIndFirst + paraIndLeft)) {
                    if (paraIndLeft > paraIndRight) {
                        paraIndLeft = paraWidthHmm - minParaWidthHmm - (paraIndFirst > 0 ? paraIndFirst : 0);
                        paraIndRight = 0;
                    } else {
                        paraIndLeft = 0;
                        paraIndRight = paraWidthHmm - minParaWidthHmm - (paraIndFirst > 0 ? paraIndFirst : 0);
                    }
                }

                opProperties.attrs.paragraph.indentRight = paraIndRight;
                opProperties.attrs.paragraph.indentLeft = paraIndLeft;
                opProperties.attrs.paragraph.indentFirstLine = paraIndFirst;
                opProperties.start = position;
                if (target) { opProperties.target = target; }
                if (docModel.getChangeTrack().isActiveChangeTracking()) { opProperties.attrs.changes = { modified: docModel.getChangeTrack().getChangeTrackInfo() }; }
                generator.generateOperation(Operations.SET_ATTRIBUTES, opProperties);
            });

            docModel.applyOperations(generator);
            docView.scrollToSelection();
        });

        // block keyboard input during applying of operations
        this.on('show', function () {
            docModel.setBlockKeyboardEvent(true);
        });

        this.registerDestructor(function () {
            docModel.setBlockKeyboardEvent(false);
            self = docModel = docView = selection = null;
            controlRootNode = null;
        });

    }); // class ParagraphDialog

    // exports ================================================================

    return ParagraphDialog;

});
