/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/model/model', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/model/editor',
    'io.ox/office/text/model/modelattributesmixin',
    'io.ox/office/text/format/listcollection',
    'io.ox/office/text/format/stylesheetmixin',
    'io.ox/office/text/model/pagehandlermixin',
    'io.ox/office/text/model/updatedocumentmixin',
    'io.ox/office/text/model/listhandlermixin',
    'io.ox/office/text/model/updatelistsmixin',
    'io.ox/office/text/components/drawing/drawingresize',
    'io.ox/office/textframework/components/field/fieldmanager',
    'io.ox/office/textframework/model/numberformatter',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/view/selectionbox',
    'io.ox/office/drawinglayer/utils/drawingutils'
], function (DrawingFrame, DrawingLabels, AttributeUtils, Editor, ModelAttributesMixin, ListCollection, StylesheetMixin, PageHandlerMixin, UpdateDocumentMixin, ListHandlerMixin, UpdateListsMixin, DrawingResize, FieldManager, NumberFormatter, Utils, DOM, Position, Operations, SelectionBox, DrawingUtils) {

    'use strict';

    // class TextModel ========================================================

    /**
     * Represents the document model of a text application.
     *
     * @constructor
     *
     * @extends Editor
     * @extends ModelAttributesMixin
     * @extends StylesheetMixin
     * @extends PageHandlerMixin
     * @extends UpdateDocumentMixin
     * @extends ListHandlerMixin
     * @extends UpdateListsMixin
     *
     * @param {TextApplication} app
     *  The application containing this document model.
     */
    function TextModel(app) {

        var // self reference for local functions
            self = this,
            // the root element for the document contents
            pagediv = null,
            // the selection box object
            selectionBox = null;

        // base constructor ---------------------------------------------------

        Editor.call(this, app, {
            selectionStateHandler: selectionStateHandler,
            updateSelectionInUndo: true,
            spellingSupported: true
        });

        ModelAttributesMixin.call(this);
        StylesheetMixin.call(this, app);
        PageHandlerMixin.call(this, app);
        UpdateDocumentMixin.call(this, app);
        ListHandlerMixin.call(this);
        UpdateListsMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Callback function to save and restore the document selection during
         * undo and redo operations.
         */
        function selectionStateHandler(/*selectionState*/) {

            var selection = null;

            // no argument passed: return current selection state
            if (arguments.length === 0) {
                selection = self.getSelection();
                return { start: selection.getStartPosition(), end: selection.getEndPosition(), target: selection.getRootNodeTarget() };
            }

            // restoring the selection state needs to be done in undo-redo-after handler
        }

        /**
         * Listener function for the event 'document:reloaded:after', that is triggered, if the user cancels a long
         * running action.
         *
         * @param {jQuery.Event} event
         *  The 'document:reloaded:after' event
         *
         * @param {Object} snapShot
         *  The snapshot object.
         */
        function documentReloadAfterHandler(/*event, snapShot*/) {
            self.getNode().children('.drawingselection-overlay').empty(); // clearing old selection frames (58570)
        }

        // public methods -----------------------------------------------------

        /**
         * Getting the default text for an empty text frame drawing.
         * This function is application specific.
         * This is not implemented for OX Text yet.
         *
         * @param {HTMLElement|jQuery} drawing
         *  The drawing node.
         *
         * @param {Object} [options]
         *  Some additional options.
         *
         * @returns {String|Null}
         *  The default text for a place holder drawing. Or null, if it cannot be determined.
         */
        this.getDefaultTextForTextFrame = function (/*drawing, options*/) {
            return null;
        };

        /**
         * Handling the property 'autoResizeHeight' of a selected text frame node.
         *
         * @param {Boolean} state
         *  Whether the property 'autoResizeHeight' of a selected text frame shall be
         *  enabled or disabled.
         */
        this.handleTextFrameAutoFit = function (state) {

            var selection = self.getSelection();
            // the operations generator
            var generator = self.createOperationGenerator();
            // the options for the setAttributes operation
            var operationOptions = {};
            // a selected text frame node or the text frame containing the selection
            var textFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true });

            // collecting the attributes for the operation
            operationOptions.attrs =  {};
            operationOptions.attrs.shape = { autoResizeHeight: state, noAutoResize: !state }; // also noAutoResize property has to be set, see #50645

            // if auto fit is disabled, the current height must be set explicitely
            if (!state) {
                if (textFrame && textFrame.length > 0) {
                    operationOptions.attrs.drawing = { height: Utils.convertLengthToHmm(textFrame.height(), 'px') };
                }
            }

            if (selection.isAdditionalTextframeSelection()) {
                operationOptions.start = Position.getOxoPosition(self.getCurrentRootNode(), textFrame, 0);
            } else {
                operationOptions.start = selection.getStartPosition();
            }

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            this.applyOperations(generator);
        };

        /**
         * Inserting a shape with a default size.
         *
         * @param {String} presetShapeId
         *  The identifier for the shape.
         *
         * @param {Object} [options]
         *  For Optional parameters see insertShape.
         *
         * @returns {undefined}
         *  The return value of 'self.insertShape'.
         */
        this.insertShapeWithDefaultBox = function (presetShapeId) {
            return self.insertShape(presetShapeId, DrawingUtils.createDefaultSizeForShapeId(presetShapeId, TextModel.DEFAULT_SHAPE_SIZE.width, TextModel.DEFAULT_SHAPE_SIZE.height));
        };

        /**
         * Inserting a text frame drawing node of type 'shape' into the document.
         *
         * @param {String} presetShapeId
         *  The string specifying the shape to be inserted.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.selectDrawingOnly=false]
         *      Whether the shape shall be selected after inserting it. Default is
         *      that the paragraph inside the shape is selected.
         *  @param {Number[]} [options.position=null]
         *      A logical position that at which the shape shall be inserted.
         *  @param {Object} [options.pargraphOffset=null]
         *      An offset object containing the properties 'left' and 'top' that describe
         *      the distance of the shape to a paragraph in hmm.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog asking the user, if
         *  he wants to delete the selected content (if any) has been closed
         *  with the default action; or rejected, if the dialog has been
         *  canceled. If no dialog is shown, the promise is resolved
         *  immediately.
         */
        this.insertShape = function (presetShapeId, size, options) {

            // the undo manager returns the return value of the callback function
            return self.getUndoManager().enterUndoGroup(function () {

                var // the selection object
                    selection = self.getSelection();

                // getting a valid position for inserting a shape, if it is inserted via keyboard.
                // Otherwise the position was already determined by the selection box.
                // -> shapes can only be inserted in top-level paragraphs or paragraphs in tables.
                function getValidInsertShapePosition() {

                    var validPos = selection.getStartPosition();
                    var leadingAbsoluteDrawings = 0;
                    var paragraphPoint = null;

                    if (selection.isAdditionalTextframeSelection() || (Utils.TOUCHDEVICE && _.browser.Android && validPos.length > 2)) {
                        validPos = [_.first(validPos)]; // the position of the top level element (table or paragraph)
                        validPos = Position.getFirstPositionInParagraph(self.getCurrentRootNode(), validPos);
                        // inserting behind already existing absolute drawings are place holder drawings in this paragraph
                        paragraphPoint = Position.getDOMPosition(self.getCurrentRootNode(), _.initial(validPos), true);
                        if (paragraphPoint && paragraphPoint.node) {
                            leadingAbsoluteDrawings = Position.getLeadingAbsoluteDrawingOrDrawingPlaceHolderCount(paragraphPoint.node);
                            if (leadingAbsoluteDrawings > 0) { validPos = Position.increaseLastIndex(validPos, leadingAbsoluteDrawings); }
                        }
                    }

                    return validPos;
                }

                // helper function, that generates the operations
                function doInsertShape() {

                    // the operations generator
                    var generator = self.createOperationGenerator();
                    // whether the shape shall be selected after insertion
                    var selectDrawingOnly = Utils.getBooleanOption(options, 'selectDrawingOnly', false) || Utils.TOUCHDEVICE;
                    // an optional position for inserting the shape
                    var optPosition = Utils.getArrayOption(options, 'position', null);
                    // an offset relative to a paragraph in pixel
                    var pargraphOffset = Utils.getObjectOption(options, 'pargraphOffset', null);
                    // the width of the parent paragraph
                    var paraWidth = 0;
                    // the current cursor (start) position or a specified position for inserting the drawing
                    var start = optPosition ? optPosition : getValidInsertShapePosition();
                    // the position of the first paragraph inside the text frame
                    var paraPos = _.clone(start);
                    // the paragraph node, in which the text frame will be inserted
                    var paraNode = null;
                    // target for operation - if exists, it's for ex. header or footer
                    var target = self.getActiveTarget();
                    // the options for the insert drawing operation
                    var operationOptions = {};
                    // the attribute set with default formatting
                    var attrSet = DrawingUtils.createDefaultShapeAttributeSet(self, presetShapeId, options);
                    // the character attribute set
                    var charAttrs = attrSet.character;
                    // whether the shapes can contain text (in ODF: connectors can contain text too)
                    var hasText = app.isODF() || !DrawingLabels.isNoTextShape(presetShapeId);
                    // the options for the setAttributes operation
                    var setAttrsOptions = null;
                    // the drawing attributes for the vertical order
                    var drawingOrderAttrs = null;
                    // the options for the insertParagraph opertion
                    var paraInsertOptions = null;

                    self.doCheckImplicitParagraph(start);
                    paraPos.push(0);

                    // the shape family object for the insertDrawing operation
                    attrSet.shape = { anchor: 'centered', anchorCentered: false, paddingLeft: 100, paddingRight: 100, paddingTop: 100, paddingBottom: 100, wordWrap: true, horzOverflow: 'overflow', vertOverflow: 'overflow' };
                    // the attributes of the drawing family
                    AttributeUtils.insertAttribute(attrSet, 'drawing', 'name', presetShapeId);

                    if (!size) { size = TextModel.DEFAULT_SHAPE_SIZE; }
                    _.extend(attrSet.drawing, size);
                    attrSet.shape.paddingLeft = attrSet.shape.paddingRight = Math.round(size.width / 10);
                    attrSet.shape.paddingTop = attrSet.shape.paddingBottom = Math.round(size.height / 10);

                    operationOptions.attrs = attrSet;
                    operationOptions.start = start;
                    operationOptions.type = DrawingLabels.getPresetShape(presetShapeId).type;

                    // modifying the attributes, if changeTracking is activated
                    if (self.getChangeTrack().isActiveChangeTracking()) {
                        operationOptions.attrs.changes = { inserted: self.getChangeTrack().getChangeTrackInfo(), removed: null };
                    }

                    // checking the width of the parent paragraph to adapt the shape size
                    paraNode = Position.getParagraphElement(pagediv, _.initial(start));

                    if (paraNode) {
                        paraWidth = Utils.convertLengthToHmm($(paraNode).width(), 'px');
                        if (paraWidth && paraWidth < operationOptions.attrs.drawing.width) {
                            operationOptions.attrs.drawing.width = paraWidth;
                            operationOptions.attrs.drawing.height = paraWidth; // keeping the drawing quadratic
                        }
                    }

                    if (pargraphOffset) {
                        // adding attributes to the insertDrawing operation, so that the drawing is absolutely positioned in paragraph
                        _.extend(operationOptions.attrs.drawing, { anchorHorBase: 'column', anchorHorAlign: 'offset', anchorVertBase: 'paragraph', anchorVertAlign: 'offset', inline: false, textWrapMode: 'none' });
                        operationOptions.attrs.drawing.anchorHorOffset = pargraphOffset.left;
                        operationOptions.attrs.drawing.anchorVertOffset = pargraphOffset.top;
                    }

                    // not assigning character attributes to the drawing (51104) -> will be inherited to following text spans
                    if (operationOptions.attrs.character) { delete operationOptions.attrs.character; }

                    self.extendPropertiesWithTarget(operationOptions, target);
                    generator.generateOperation(Operations.INSERT_DRAWING, operationOptions);

                    setAttrsOptions = { attrs: { shape: { anchor: 'centered' } }, start: start };

                    // setting autoResizeHeight explicitely to false in odt files (56351)
                    if (app.isODF()) { setAttrsOptions.attrs.shape.autoResizeHeight = false; }

                    // bringing the shape always to the 'front' (54735)
                    drawingOrderAttrs = self.getDrawingLayer().getDrawingOrderAttributes('front', $(paraNode));
                    if (drawingOrderAttrs) { setAttrsOptions.attrs.drawing = drawingOrderAttrs; }

                    self.extendPropertiesWithTarget(setAttrsOptions, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, setAttrsOptions);

                    if (hasText) {
                        paraInsertOptions = { start: paraPos, attrs: _.extend({ character: charAttrs }, DrawingUtils.getParagraphAttrsForDefaultShape(self)) };
                        self.extendPropertiesWithTarget(paraInsertOptions, target);
                        generator.generateOperation(Operations.PARA_INSERT, paraInsertOptions);
                    }

                    self.applyOperations(generator);

                    if (!hasText || selectDrawingOnly) {
                        self.getSelection().setTextSelection(start, Position.increaseLastIndex(start)); // select the drawing
                    } else {
                        self.getSelection().setTextSelection(Position.appendNewIndex(_.clone(paraPos), 0)); // setting cursor into text frame
                    }

                }

                // Inserting a shape never deletes an existing selection
                // if (selection.hasRange()) {
                //     return self.deleteSelected()
                //         .done(function () {
                //             doInsertShape();
                //         });
                //     }

                doInsertShape();
                return $.when();

            }, this); // enterUndoGroup()
        };

        /**
         * Preparations for inserting a shape. A selection box can be opened
         * so that the user can determine the position of the shape.
         *
         * @param {String} presetShapeId
         *  The identifier for the shape.
         *
         * @returns {Boolean}
         *  Whether a selection box could be activated.
         */
        this.insertShapePreparation = function (presetShapeId) {

            var contentRootNode = app.getView().getContentRootNode();
            var contentHeight = contentRootNode.children(DOM.APPCONTENT_NODE_SELECTOR).outerHeight() + 100;
            var contentRootOffset = contentRootNode.offset();
            var selectionBox = self.getSelectionBox();
            var shapeMode = 'shape';
            var canvasDiv = $('<div style="position:absolute;z-index:99">');
            var vertStartLine = $('<div class="guide-line v"></div>').height(contentHeight);
            var horzStartLine = $('<div class="guide-line h"></div>');
            var vertEndLine = $('<div class="guide-line v"></div>').height(contentHeight);
            var horzEndLine = $('<div class="guide-line h"></div>');
            // special behavior for line/connector shapes
            var twoPointMode = DrawingLabels.isPresetConnector(presetShapeId);
            // custom aspect ratio for special shapes
            var aspectRatio = DrawingLabels.getPresetAspectRatio(presetShapeId);

            function initGuidelinesHandler(event) {
                var eventPos = selectionBox.getEventPosition(event, (event.type === 'touchmove') ? 'touch' : 'mouse');
                var startX = eventPos.x + contentRootNode.scrollLeft() - contentRootOffset.left;
                var startY = eventPos.y + contentRootNode.scrollTop() - contentRootOffset.top;
                horzStartLine.css('top', startY);
                vertStartLine.css('left', startX);
            }

            function clearGuidelines() {
                contentRootNode.children('.guide-line').remove();
            }

            // returns the flipping options for connector/line shapes
            function getFlippingOptions(frameRect) {
                return twoPointMode ? { flipH: frameRect.reverseX, flipV: frameRect.reverseY } : null;
            }

            function clearGuidelinesHandler() {
                clearGuidelines();
            }

            function moveHandler(frameRect) {
                canvasDiv.css({ left: frameRect.left, top: frameRect.top });
                horzStartLine.css('top', frameRect.top);
                vertStartLine.css('left', frameRect.left);
                vertEndLine.css('left', frameRect.right());
                horzEndLine.css('top', frameRect.bottom());
                var options = _.extend({ transparent: true }, getFlippingOptions(frameRect));
                DrawingFrame.drawPreviewShape(app, canvasDiv, frameRect, presetShapeId, options);
                Utils.clearBrowserSelection(); // avoiding visible browser selection during inserting shape
            }

            function startHandler(frameRect) {
                contentRootNode.append(canvasDiv, vertEndLine, horzEndLine);
                self.stopListeningTo(contentRootNode, 'mousemove touchmove', initGuidelinesHandler);
                moveHandler(frameRect);
            }

            function stopHandler(frameRect) {

                // the options for the insertShape operation
                var insertShapeOptions = getFlippingOptions(frameRect) || {};
                // the current zoom level
                var zoomLevel = app.getView().getZoomFactor() * 100;
                // an object with the paragraph offset
                var paraOffset = null;
                // the page info object returned from the pixel API
                var pageInfo = null;
                // the last paragraph in the document
                var lastParagraph = null;
                // the number of leading absolute positioned drawings in the paragraph
                var leadingAbsoluteDrawings = 0;
                // defaultBox for shapes
                var defaultBox;

                // inserting drawing with default size after click without creating a box
                if (frameRect.area() === 0) {

                    // box.width = Utils.convertHmmToLength(TextModel.DEFAULT_SHAPE_SIZE.width, 'px', 1);
                    // box.height = Utils.convertHmmToLength(TextModel.DEFAULT_SHAPE_SIZE.height, 'px', 1);
                    defaultBox = DrawingUtils.createDefaultSizeForShapeId(presetShapeId, Utils.convertHmmToLength(TextModel.DEFAULT_SHAPE_SIZE.width, 'px', 1), Utils.convertHmmToLength(TextModel.DEFAULT_SHAPE_SIZE.height, 'px', 1));
                    frameRect.width = defaultBox.width;
                    frameRect.height = defaultBox.height;
                }

                // converting the pixel relative to app content root node relative to the page
                frameRect = Utils.convertAppContentBoxToPageBox(app, frameRect);

                if (!self.isHeaderFooterEditState()) { // in header/footer edit state the pixel API is not precise enough
                    pageInfo = Position.getPositionFromPagePixelPosition(pagediv, frameRect.left, frameRect.top, zoomLevel, { onlyParagraphPosition: true });

                    if (pageInfo && pageInfo.pos) {
                        leadingAbsoluteDrawings = Position.getLeadingAbsoluteDrawingOrDrawingPlaceHolderCount(pageInfo.point.node);
                        insertShapeOptions.position = Position.appendNewIndex(pageInfo.pos, leadingAbsoluteDrawings);

                        // calculating pixel position relative to the paragraph
                        paraOffset = Position.getPixelPositionToRootNodeOffset(pagediv, pageInfo.point.node, zoomLevel);
                        paraOffset = { left: Utils.convertLengthToHmm(frameRect.left - paraOffset.x, 'px'), top: Utils.convertLengthToHmm(frameRect.top - paraOffset.y, 'px') };

                        insertShapeOptions.pargraphOffset = paraOffset;
                        insertShapeOptions.absolute = true;
                    } else {
                        // check if the position is below the last paragraph
                        lastParagraph = DOM.getPageContentNode(pagediv).children(DOM.PARAGRAPH_NODE_SELECTOR).last();
                        paraOffset = Position.getPixelPositionToRootNodeOffset(pagediv, lastParagraph, zoomLevel);
                        if (frameRect.top > paraOffset.y) {
                            leadingAbsoluteDrawings = Position.getLeadingAbsoluteDrawingOrDrawingPlaceHolderCount(lastParagraph);
                            paraOffset = { left: Utils.convertLengthToHmm(frameRect.left - paraOffset.x, 'px'), top: Utils.convertLengthToHmm(frameRect.top - paraOffset.y, 'px') };
                            insertShapeOptions.pargraphOffset = paraOffset;
                            insertShapeOptions.absolute = true;
                            insertShapeOptions.position = Position.appendNewIndex(Position.getOxoPosition(pagediv, lastParagraph, 0), leadingAbsoluteDrawings);
                        }
                    }
                }

                // setting marker for operations triggered from selection box
                // -> this marker can be used to cancel an active selection box, if there are operations not generated
                //    from the selection box (the user might have clicked on another button in the top bar).
                selectionBox.setSelectionBoxOperation(true);

                // setting the size of the drawing
                var size = { width: Utils.convertLengthToHmm(frameRect.width, 'px'), height: Utils.convertLengthToHmm(frameRect.height, 'px') };
                // assigning the drawing to the paragraph of the upper left position
                self.insertShape(presetShapeId, size, insertShapeOptions);

                // resetting marker for operations triggered from selection box
                selectionBox.setSelectionBoxOperation(false);
            }

            function finalizeHandler() {
                self.executeDelayed(function () {
                    canvasDiv.remove();
                    clearGuidelines();
                }, 'TextModel.insertShapePreparation', 50);
                // reset to the previous selection box mode
                selectionBox.setPreviousMode();
            }

            if (self.isInsertShapeActive()) {
                selectionBox.cancelSelectionBox();
                clearGuidelines();
                return false;
            }

            contentRootNode.append(vertStartLine, horzStartLine);
            self.listenTo(contentRootNode, 'mousemove touchmove', initGuidelinesHandler);

            // always registering the mode for text shape insertion
            selectionBox.registerSelectionBoxMode(shapeMode, {
                contentRootClass: 'textframemode',
                preventScrolling: true,
                leaveModeOnCancel: true,
                excludeFilter: TextModel.EXCLUDE_SELECTION_BOX_FILTER,
                trackingOptions: { twoPointMode: twoPointMode, aspectRatio: aspectRatio },
                stopHandler: stopHandler,
                startHandler: startHandler,
                moveHandler: moveHandler,
                finalizeHandler: finalizeHandler,
                clearGuidelinesHandler: clearGuidelinesHandler
            });

            // activating the mode for text frame insertion
            selectionBox.setActiveMode(shapeMode);

            return true;
        };

        /**
         * Getter for the selection box object, that can be used to create a specified
         * rectangle on the application content root node.
         *
         * @returns {Object}
         *  The selection box object.
         */
        this.getSelectionBox = function () {
            return selectionBox;
        };

        /**
         * Getting the default shape size with the properties 'width' and 'height'.
         *
         * @returns {Object}
         *  An object containing the properties 'width' and 'height' for the default shape size.
         */
        this.getDefaultShapeSize = function () {
            return TextModel.DEFAULT_SHAPE_SIZE;
        };

        /**
         * Whether the 'shape' mode is the active mode.
         *
         * @returns {Boolean}
         *  Whether the 'shape' mode is the active mode.
         */
        this.isInsertShapeActive = function () {
            return self.getSelectionBox() && self.getSelectionBox().getActiveMode() === 'shape';
        };

        /**
         * Saving the attributes that are explicitely set at an implicit paragraph. In this case
         * the new paragraph that replaces the implicit paragraph, shall also use this attributes.
         * This is for example the case for paragraphs in empty shapes.
         *
         * @param {Node|jQuery} para
         *  The implicit paragraph that will be replaced via operation.
         *
         * @returns {Object|Null}
         *  The attributes that are required for the insertParagraph operation.
         */
        this.getReplaceImplicitParagraphAttrs = function (para) {

            var // the explicit attributes at the implicit paragraph
                attrs = AttributeUtils.getExplicitAttributes(para);

            return attrs;
        };

        /**
         * Rotates and/or flips the selected drawing obejcts.
         *
         * @param {Number} angle
         *  The rotation angle to be added to the current rotation angle of the
         *  drawing objects.
         *
         * @param {Boolean} flipH
         *  Whether to flip the drawing objects horizontally.
         *
         * @param {Boolean} flipV
         *  Whether to flip the drawing objects vertically.
         */
        this.transformDrawings = function (angle, flipH, flipV) {

            // the drawing style sheet collection of the document
            var drawingStyles = self.getDrawingStyles();
            // the selection object
            var selection = self.getSelection();
            // if drawing or text inisde is selected
            var isAdditionalTextSel = selection.isAdditionalTextframeSelection();
            // root node
            var rootNode = selection.getRootNode();
            // drawing node
            var drawing = isAdditionalTextSel ? selection.getSelectedTextFrameDrawing() : selection.getSelectedDrawing();

            // create the new drawing attributes
            var drawingAttrs = drawingStyles.getElementAttributes(drawing).drawing;
            drawingAttrs = DrawingUtils.transformAttributeSet(drawingAttrs, angle, flipH, flipV, app.isODF());

            // create the new 'setAttributes' operation
            if (drawingAttrs) {

                // the operations generator
                var generator = self.createOperationGenerator();
                // drawing start position
                var startPos = isAdditionalTextSel ? Position.getOxoPosition(rootNode, drawing, 0) : selection.getStartPosition();
                // the properties for the new setAttributes operation
                var properties = { start: startPos, attrs: { drawing: drawingAttrs } };

                if (selection.isCropMode()) { self.exitCropMode(); }

                // create and apply the operations (undo group is created automatically)
                self.extendPropertiesWithTarget(properties, self.getActiveTarget());
                generator.generateOperation(Operations.SET_ATTRIBUTES, properties);
                self.applyOperations(generator);

                // after operations are applied, update resize mouse poiters
                if ('rotation' in drawingAttrs) {
                    DrawingFrame.updateResizersMousePointers(drawing, drawingAttrs.rotation);
                }
            }
        };

        /**
         * After undo/redo or pasting internal clipboard is applied, it is necessary that empty shapes,
         * that were not generated inside OX Text, receive an implicit paragraph, so that text can be
         * inserted inside OX Text (53983).
         * After loading the document this is already handled within updateDocumentFormatting (call of
         * DrawingFrame.checkEmptyTextShape).
         *
         * @param {Array<Object>} operations
         *  An array containing the undo or redo operations.
         */
        this.checkInsertedEmptyTextShapes = function (operations) {

            // all insert drawing operations from undo/redo
            var insertDrawingOps = _.filter(operations, function (op) { return op.name === Operations.INSERT_DRAWING; });

            _.each(insertDrawingOps, function (op) {
                // the root node for the logical position in the operation
                var rootNode = self.getRootNode(op.target);
                // the drawing nodes and its offset
                var drawingPoint = Position.getDOMPosition(rootNode, op.start, true);
                // whether the drawing node was really modified
                var drawingModified = false;

                if (drawingPoint && drawingPoint.node) {
                    drawingModified = DrawingFrame.checkEmptyTextShape(self, drawingPoint.node);
                    if (drawingModified) { self.getDrawingStyles().updateElementFormatting(drawingPoint.node); }
                }
            });

        };

        /**
         * Check, whether at the current position, a list can be inserted. This is in ODT not always the case (58663).
         *
         * @returns {Boolean}
         *   Whether a list can be inserted at the current position.
         */
        this.checkTextListAvailability = function () {
            return !app.isODF() || !self.getSelection().isAnyDrawingSelection() || self.isFullOdfTextframeFunctionality();
        };

        // initialization -----------------------------------------------------

        pagediv = self.getNode();

        // setting the handler function for moving and resizing drawings and for updating the selection in the overlay node
        self.getSelection().setDrawingResizeHandler(DrawingResize.drawDrawingSelection);
        self.getSelection().setUpdateOverlaySelectionHandler(DrawingResize.updateOverlaySelection);

        // setting the handler function for updating lists (this is used during 'updateDocumentFormatting')
        self.setUpdateLists(self.getUpdateListsHandler());

        // setting handler for handling fields in presentation
        self.setFieldManager(new FieldManager(app));

        // setting handler for handling fields in presentation
        self.setNumberFormatter(new NumberFormatter(this));

        // setting the list collection object
        self.setListCollection(new ListCollection(this));

        // setting the handler function for updating lists after the document is imported successfully
        self.waitForImportSuccess(function () {
            self.setUpdateListsDebounced(self.getDebouncedUpdateListsHandler());
            if (!self.getSlideTouchMode()) {
                selectionBox = new SelectionBox(app, app.getView().getContentRootNode());
                self.on('operations:before', function () {
                    if (self.isInsertShapeActive() && !selectionBox.isSelectionBoxOperation()) { selectionBox.cancelSelectionBox(); } // for example: The user clicked another button
                });
            }
        });

        // registering the handler for document reload after event. This happens, if the user cancels a long running action
        self.listenTo(self, 'document:reloaded:after', documentReloadAfterHandler);

        // registering process mouse down handler at the page
        pagediv.on({ 'mousedown touchstart': self.getPageProcessMouseDownHandler() });
        if (self.getListenerList()) { self.getListenerList()['mousedown touchstart'] = self.getPageProcessMouseDownHandler(); }

        // complex fields
        pagediv.on('mouseenter', '.complex-field', self.getFieldManager().createHoverHighlight);
        pagediv.on('mouseleave', '.complex-field', self.getFieldManager().removeHoverHighlight);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            if (selectionBox) { selectionBox.deRegisterSelectionBox(); }
            self = selectionBox = pagediv = null;
        });

    } // class TextModel

    // constants --------------------------------------------------------------

    /**
     * The attributes that are used for new inserted drawings.
     */
    TextModel.DEFAULT_SHAPE_SIZE = {
        width: 5000,
        height: 5000
    };

    /**
     * A CSS filter to specify not allowed event targets for the selection box.
     */
    TextModel.EXCLUDE_SELECTION_BOX_FILTER = DOM.APPCONTENT_NODE_SELECTOR + ', ' + DOM.APP_CONTENT_ROOT_SELECTOR;

    // exports ================================================================

    // derive this class from class Editor
    return Editor.extend({ constructor: TextModel });

});
