/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/nameitem', [
    'io.ox/office/tk/control/group',
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/viewobjectmixin'
], function (Group, Button, ViewObjectMixin) {

    'use strict';

    // class NameItem =========================================================

    /**
     * A item displaying properties of a defined name.
     *
     * @constructor
     *
     * @extends Group
     * @extends ViewObjectMixin
     */
    var NameItem = Group.extend({ constructor: function (docView, nameModel) {

        // the document model
        var docModel = docView.getDocModel();

        // the DOM container node for the label and formula expression spans
        var nameNode = $('<div class="name">');

        // the DOM node with the label of the defined name
        var labelNode = $('<span class="text">').appendTo(nameNode);

        // the DOM node with the formula expression of the defined name
        var formulaNode = $('<span class="formula">').appendTo(nameNode);

        // the DOM container node with additional action buttons
        var actionsNode = $('<div class="actions">');

        // the edit button that will open the 'Edit Name' dialog
        var editButton = new Button(docView, { icon: 'fa-pencil-square-o' });

        // the delete button to remove a defined name from the document
        var deleteButton = new Button(docView, { icon: 'fa-trash-o' });

        // the token array of the defined name model
        var tokenArray = nameModel.getTokenArray();

        // unique identifier of the current range highlighting mode for mouse-hover
        var highlightUid = null;

        // base constructor ---------------------------------------------------

        Group.call(this, docView);
        ViewObjectMixin.call(this, docView);

        // private methods ----------------------------------------------------

        function setLabel() {
            var label = nameModel.getLabel();
            labelNode.attr('title', label).text(_.noI18n(label));
        }

        function setFormula() {
            var formula = nameModel.getFormula('ui', docView.getActiveCell());
            formulaNode.attr('title', formula).text(_.noI18n(formula));
        }

        function getExtractRangeOptions() {
            return {
                refSheet: docView.getActiveSheet(),
                targetAddress: docView.getActiveCell(),
                resolveNames: true
            };
        }

        function startHoverHighlighting() {
            highlightUid = docView.startRangeHighlighting(tokenArray, getExtractRangeOptions());
        }

        function endHoverHighlighting() {
            docView.endRangeHighlighting(highlightUid);
        }

        function selectListItem() {

            // do not leave cell edit mode (but drawing edit mode can be left safely)
            if (docView.isTextEditMode('cell')) { return; }

            // all ranges contained in the defined name
            var rangeInfos = tokenArray.extractRanges(getExtractRangeOptions());
            // select a single range in the token array
            var range = (rangeInfos.length === 1) ? rangeInfos[0].range : null;

            // nothing to do for names without range address
            if (!range) { return; }

            // switch to the first visible sheet referred by the range
            if (!range.containsSheet(docModel.getActiveSheet())) {
                var sheet = docModel.findVisibleSheet(range.sheet1, 'next');
                if (range.containsSheet(sheet)) {
                    docModel.setActiveSheet(sheet);
                }
            }

            // scroll to start cell of the range (do not change the active grid pane in frozen split mode)
            if (range.containsSheet(docModel.getActiveSheet())) {
                docView.scrollToCell(range.start, { updateFocus: false });
            }
        }

        function updateActionButtons() {
            editButton.enable(docView.isControllerItemEnabled('name/edit/dialog'));
            deleteButton.enable(docView.isControllerItemEnabled('name/delete'));
        }

        // initialization -----------------------------------------------------

        // initialization of the root node of this badge
        this.getNode().addClass('named-range').attr('role', 'dialog');

        // insert the additional nodes into the group
        this.addChildNodes(nameNode, actionsNode);

        // set current label and formula expression
        setLabel();
        setFormula();

        // add the action buttons to the DOM
        actionsNode.append(editButton.getNode(), deleteButton.getNode());
        updateActionButtons();

        // show name dialog when clicking the edit button
        editButton.on('group:change', function () {
            docView.executeControllerItem('name/edit/dialog', nameModel);
        });

        // delete the name when clicking the delete button
        deleteButton.on('group:change', function () {
            docView.executeControllerItem('name/delete', nameModel);
        });

        // add event listeners to highlight the cell ranges
        this.getNode().on({
            mouseenter: startHoverHighlighting,
            mouseleave: endHoverHighlighting,
            click: selectListItem
        });

        // update the relative references in the formula expression according to the selection
        this.listenTo(docView, 'change:selection', this.createDebouncedMethod('NameItem.changeSelectionListener', null, setFormula, { delay: 500 }));

        // update the enabled state of the additional action buttons when controller state changes
        this.handleChangedControllerItems(updateActionButtons);

        // destroy all class members
        this.registerDestructor(function () {
            endHoverHighlighting();
            editButton.destroy();
            deleteButton.destroy();
            docView = docModel = nameModel = nameNode = formulaNode = actionsNode = editButton = deleteButton = null;
        });

    } }); // class NameItem

    // exports ================================================================

    return NameItem;

});
