/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/activesheetgroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/control/radiogroup',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/spreadsheet/view/popup/sheetcontextmenu',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Forms, Iterator, Tracking, RadioGroup, AppObjectMixin, SheetContextMenu, gt) {

    'use strict';

    // class ActiveSheetGroup =================================================

    /**
     * The selector for the current active sheet, as radio group (all sheet
     * tabs are visible at the same time).
     *
     * @constructor
     *
     * @extends RadioGroup
     * @extends AppObjectMixin
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    var ActiveSheetGroup = RadioGroup.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the document model
        var docModel = docView.getDocModel();

        // the context menu with additional sheet actions
        var contextMenu = null;

        // the scroll button nodes
        var prevButton = $(Forms.createButtonMarkup({ icon: 'fa-angle-left' })).addClass('scroll-button');
        var nextButton = $(Forms.createButtonMarkup({ icon: 'fa-angle-right' })).addClass('scroll-button');

        // keep track of active sheet index (scroll to active after change)
        var activeIndex = -1;

        // index of the first and last sheet tab button currently visible
        var firstIndex = 0;
        var lastIndex = 0;
        var scrollable = false;

        // the sheet tab button currently dragged
        var dndButtonNode = null;

        // tracker node shown while dragging a sheet button
        var dndTrackerNode = $('<div id="io-ox-office-spreadsheet-sheet-dnd-tracker"><div class="caret-icon"></div></div>');

        // helper node inside the tracker node containing the sheet name
        var dndSheetNode = $('<span class="sheet-name">').appendTo(dndTrackerNode);

        // index of the sheet tab button is currently dragged, and the target button index
        var dndStartIndex = -1;
        var dndTargetIndex = -1;

        // whether the tracking point has been moved at all (bug 50653)
        var dndMoved = false;

        // base constructors --------------------------------------------------

        RadioGroup.call(this, docView, { classes: 'active-sheet-group', role: 'tablist' });
        AppObjectMixin.call(this, docView.getApp());

        // private methods ----------------------------------------------------

        /**
         * Shows all sheet buttons and removes explicit width, and hides the
         * scroll buttons.
         */
        function showAllSheetButtons() {
            var buttonNodes = self.getOptionButtons();
            Forms.showNodes(buttonNodes, true);
            buttonNodes.css('width', '');
            Forms.showNodes(prevButton, false);
            Forms.showNodes(nextButton, false);
            nextButton.css({ position: '', right: '' });
            scrollable = false;
        }

        /**
         * Shows the sheet buttons starting at the start sheet index currently
         * cached.
         */
        function renderSheetButtons() {

            // the root node of this group
            var groupNode = self.getNode();
            // the inner width of the group (space available for buttons)
            var groupWidth = groupNode.width();

            // all sheet tab buttons; do nothing if there are no sheets available (yet)
            var sheetButtons = self.getOptionButtons();
            if (sheetButtons.length === 0) { return; }

            // initialize all sheet buttons, hide scroll buttons
            showAllSheetButtons();

            // get total width of all buttons (bug 35869: round down, browsers may report slightly oversized width due to text nodes)
            var buttonsWidth = Math.floor(sheetButtons.last().offset().left + sheetButtons.last().width() - sheetButtons.first().offset().left);
            if (buttonsWidth <= groupWidth) { return; }

            // collect position information of all buttons (performance: once for all buttons, before manipulating DOM again)
            var leadingOffset = sheetButtons.first().offset().left;
            var buttonPositions = sheetButtons.get().map(function (buttonNode) {
                buttonNode = $(buttonNode);
                var buttonOffset = buttonNode.offset().left - leadingOffset;
                return {
                    offset: buttonOffset,
                    leadingWidth: buttonOffset + buttonNode.width(),
                    trailingWidth: buttonsWidth - buttonOffset
                };
            });

            // total width of all buttons, including scroll buttons
            var totalWidth = 0;
            // remaining width available in the group node for sheet buttons
            var remainingWidth = 0;
            // maximum index of button that can be shown at the left border (without gap to right border)
            var maxIndex = 0;

            // do not show scroll buttons, if there is only one visible sheet
            if (sheetButtons.length > 1) {

                // show scroll buttons
                Forms.showNodes(prevButton, true);
                Forms.showNodes(nextButton, true);
                scrollable = true;

                // get total width of all buttons, including scroll buttons, and inner remaining width
                totalWidth = nextButton.offset().left + nextButton.width() - prevButton.offset().left;
                remainingWidth = groupWidth - (totalWidth - buttonsWidth);

                // hide leading sheet tabs
                maxIndex = Utils.findFirstIndex(buttonPositions, function (position) { return position.trailingWidth <= remainingWidth; }, { sorted: true });
                firstIndex = Utils.minMax(firstIndex, 0, Math.max(0, maxIndex));
                Forms.showNodes(sheetButtons.slice(0, firstIndex), false);

                // adjust button positions relative to first visible button
                var relOffset = buttonPositions[firstIndex].offset;
                buttonPositions.forEach(function (position) {
                    position.offset -= relOffset;
                    position.leadingWidth -= relOffset;
                });

                // hide trailing sheet tabs
                maxIndex = Utils.findFirstIndex(buttonPositions, function (position) { return position.leadingWidth > remainingWidth; }, { sorted: true });
                lastIndex = ((maxIndex < 0) ? sheetButtons.length : maxIndex) - 1;
                Forms.showNodes(sheetButtons.slice(lastIndex + 1), false);

                // enable/disable scroll buttons
                Forms.enableNodes(prevButton, firstIndex > 0);
                Forms.enableNodes(nextButton, lastIndex + 1 < sheetButtons.length);

            } else {
                // single sheet: initialize indexes
                firstIndex = lastIndex = 0;
                totalWidth = buttonsWidth;
                remainingWidth = groupWidth;
            }

            // set fixed position for the right scroll button
            nextButton.css({ position: 'absolute', right: 0 });

            // shrink single visible sheet button, if still oversized
            var sheetButton = sheetButtons.eq(firstIndex);
            if ((firstIndex === lastIndex) && (remainingWidth < sheetButton.width())) {
                sheetButton.css({ width: remainingWidth });
                return;
            }

            // add the first hidden sheet button after the last visible, if there is some space left
            if (lastIndex + 1 === buttonPositions.length) { return; }
            remainingWidth -= buttonPositions[lastIndex + 1].offset;
            if (remainingWidth >= 45) {
                sheetButton = sheetButtons.eq(lastIndex + 1);
                Forms.showNodes(sheetButton, true);
                sheetButton.css({ width: remainingWidth });
            }
        }

        /**
         * Called every time the value of the control (the index of the active
         * sheet) has been changed.
         */
        function updateHandler(index) {

            // scroll backwards, if the active sheet is located before the first visible;
            // or scroll forwards, if the active sheet is located after the last visible
            if ((activeIndex !== index) && ((index < firstIndex) || (index > lastIndex))) {
                firstIndex = index;
                renderSheetButtons();
            }
            activeIndex = index;
        }

        /**
         * Handles double click events on sheet tab buttons and initiates
         * renaming the sheet.
         */
        function doubleClickHandler() {
            docView.executeControllerItem('sheet/rename/dialog');
        }

        /**
         * Scrolls to the left to make at least one preceding sheet tab button
         * visible.
         */
        function scrollToPrev() {

            // do not try to scroll, if all buttons are visible
            if (!scrollable) { return; }

            // index of the old last visible sheet
            var oldLastIndex = lastIndex;

            // move to previous sheets until the index of the last sheet has changed
            while ((firstIndex > 0) && (oldLastIndex === lastIndex)) {
                firstIndex -= 1;
                renderSheetButtons();
            }
        }

        /**
         * Scrolls to the right to make at least one following sheet tab button
         * visible.
         */
        function scrollToNext() {

            // do not try to scroll, if all buttons are visible
            if (!scrollable) { return; }

            // all sheet tab buttons
            var sheetButtons = self.getOptionButtons();
            // index of the old last visible sheet
            var oldLastIndex = lastIndex;

            // move to next sheets until the index of the last sheet has changed
            while ((lastIndex >= 0) && (lastIndex + 1 < sheetButtons.length) && (oldLastIndex === lastIndex)) {
                firstIndex += 1;
                renderSheetButtons();
            }
        }

        /**
         * Handles tracking on the scroll-back button. Stops event propagation
         * (to not interfere with sheet move tracking).
         */
        function prevButtonTrackingHandler(event) {
            scrollToPrev();
            event.stopPropagation();
        }

        /**
         * Handles tracking on the scroll-forward button. Stops event
         * propagation (to not interfere with sheet move tracking).
         */
        function nextButtonTrackingHandler(event) {
            scrollToNext();
            event.stopPropagation();
        }

        /**
         * Handles end of tracking on the scroll buttons. Moves the browser
         * focus back to the application, and stops event propagation (to not
         * interfere with sheet move tracking).
         */
        function buttonTrackingEndHandler(event) {
            event.stopPropagation();
            self.triggerCancel({ sourceEvent: event });
        }

        /**
         * Returns whether the dragged sheet tab button will currently drop to
         * a new position.
         *
         * @returns {Boolean}
         *  Whether the dragged sheet tab button will currently drop to a new
         *  position.
         */
        function isValidDndTargetIndex() {
            // dragging directly behind the start button will not move the sheet
            return (dndTargetIndex < dndStartIndex) || (dndStartIndex + 1 < dndTargetIndex);
        }

        /**
         * Starts drag&drop tracking of a sheet tab button.
         */
        function dndTrackingStartHandler(event) {

            // do nothing in read-only mode
            if (!docView.isEditable()) {
                event.cancelTracking();
                return;
            }

            // the sheet tab button clicked for tracking
            dndButtonNode = $(event.target).closest(Forms.OPTION_BUTTON_SELECTOR);

            // remember button index as long as tracking lasts
            dndStartIndex = dndTargetIndex = self.getOptionButtons().index(dndButtonNode);

            // prepare the tracking node shown at the current insertion position
            dndSheetNode.text(Forms.getCaptionText(dndButtonNode));
            $(document.body).append(dndTrackerNode.hide());
            dndButtonNode.addClass('tracking-active');

            // bug 50653: check whether the tracking point has been moved at all
            dndMoved = false;
        }

        /**
         * Updates drag&drop tracking of a sheet tab button.
         */
        function dndTrackingMoveHandler(event) {

            // all sheet tab buttons
            var buttonNodes = self.getOptionButtons();
            // all visible sheet tab buttons
            var visibleNodes = Forms.filterVisibleNodes(buttonNodes);
            // absolute position and size of all visible sheet buttons
            var buttonRects = visibleNodes.get().map(Utils.getNodePositionInPage);
            // search rectangle of hovered sheet tab button in array of visible button rectangles
            var rectIndex = Utils.findLastIndex(buttonRects, function (rect) {
                return (rect.left + rect.width / 2) <= event.pageX;
            }, { sorted: true }) + 1;

            // calculate new insertion index (add number of leading hidden buttons)
            dndTargetIndex = buttonNodes.index(visibleNodes[0]) + rectIndex;
            // bug 50653: check whether the tracking point has been moved at all
            dndMoved = true;

            // hide tracking icon on invalid position (next to start button)
            if (!isValidDndTargetIndex()) {
                dndTrackerNode.hide();
                return;
            }

            // whether to show the tracking position after the last visible button
            var trailing = rectIndex >= buttonRects.length;
            // position of the sheet tab button following the tracking position
            var buttonRect = trailing ? _.last(buttonRects) : buttonRects[rectIndex];
            // exact insertion position (center of the tracker node), in pixels
            var offsetX = buttonRect.left + (trailing ? buttonRect.width : 0);

            // update position of the tracking icon
            dndTrackerNode.show().css({
                left: Math.round(offsetX - dndTrackerNode.width() / 2),
                top: self.getNode().offset().top - dndTrackerNode.height() + 2
            });
        }

        /**
         * Automatically scrolls the buttons while dragging a sheet tab button.
         */
        function dndTrackingScrollHandler(event) {

            // scroll buttons into the specified direction if possible
            if (event.scrollX < 0) {
                scrollToPrev();
            } else if (event.scrollX > 0) {
                scrollToNext();
            }

            // update position of the tracker node
            dndTrackingMoveHandler(event);
        }

        /**
         * Finalizes drag&drop tracking of a sheet tab button. Moves the sheet
         * associated to the dragged button in the document, and moves the
         * browser focus back to the application.
         */
        function dndTrackingEndHandler() {

            // leave text edit mode before manipulating the sheet order
            var promise = docView.leaveTextEditMode().done(function () {

                // source sheet index for document operation
                var fromSheet = Forms.getButtonValue(dndButtonNode);

                // move sheet only if its position will really change
                if (isValidDndTargetIndex()) {

                    // adjust drop position, if target position follows source position
                    if (dndStartIndex < dndTargetIndex) { dndTargetIndex -= 1; }

                    // the sheet tab button nodes
                    var buttonNodes = self.getOptionButtons();
                    // target sheet index for document operation
                    var toSheet = Forms.getButtonValue(buttonNodes[dndTargetIndex]);

                    // change sheet order in the document
                    if (fromSheet !== toSheet) {
                        docView.executeControllerItem('document/movesheet', { from: fromSheet, to: toSheet });
                    }

                } else if (dndMoved) {
                    // activate the sheet that has been tracked but will not be moved
                    self.triggerChange(fromSheet);
                }
            });

            // remove tracking icon, focus back to document
            promise.always(dndTrackingCancelHandler);
        }

        /**
         * Cancels drag&drop tracking of a sheet tab button. Moves the browser
         * focus back to the application, without actually moving a sheet.
         */
        function dndTrackingCancelHandler() {
            dndTrackerNode.remove();
            if (dndButtonNode) {
                dndButtonNode.removeClass('tracking-active');
                dndButtonNode = null;
            }
            docView.grabFocus();
        }

        /**
         * Recreates all sheet tabs in this radio group.
         */
        var fillList = this.createDebouncedMethod('ActiveSheetGroup.fillList', null, function () {

            // remove all old option buttons
            self.clearOptionButtons();

            // create buttons for all visible sheets
            var iterator = docModel.createSheetIterator({ supported: true, visible: true });
            Iterator.forEach(iterator, function (sheetModel, iterResult) {
                var sheetLabel = _.noI18n(iterResult.name);
                self.createOptionButton(iterResult.sheet, {
                    icon: sheetModel.isLocked() ? 'fa-lock' : null,
                    iconClasses: 'small-icon',
                    label: sheetLabel,
                    labelStyle: 'white-space:pre;',
                    tooltip: sheetLabel,
                    attributes: { role: 'tab' }
                });
            });

            // move trailing scroll button to the end
            self.addChildNodes(nextButton);

            // explicitly notify listeners, needed for reliable layout
            self.trigger('change:sheets');
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to automatic width, shows all sheet tab buttons,
         * and hides the scroll buttons.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setFullWidth = function () {
            this.getNode().css('width', '');
            showAllSheetButtons();
            return this;
        };

        /**
         * Sets this control to a fixed width. Performs further initialization
         * according to the available space: shows/hides the scroll buttons,
         * and draws the available sheet tabs according to the last cached
         * scroll position.
         *
         * @param {Number} width
         *  The new outer width of this control, in pixels.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().width(width);
            renderSheetButtons();
            return this;
        };

        // initialization -----------------------------------------------------

        // set tool tips for scroll buttons
        Forms.setToolTip(prevButton, gt('Scroll to the previous sheet'));
        Forms.setToolTip(nextButton, gt('Scroll to the next sheet'));

        // cancel button drag&drop when losing edit rights
        this.listenTo(docView.getApp(), 'docs:editmode:leave', function () {
            if (dndButtonNode) { Tracking.cancelTracking(); }
        });

        // register handler early for fast preview during import
        this.listenTo(docView.getApp(), 'docs:preview:activesheet', fillList);

        // initialization after import
        this.waitForImportSuccess(function () {
            // build the list of available sheets once after import
            fillList();
            // rebuild the list of available sheets after it has been changed
            this.listenTo(docModel, 'transform:sheet rename:sheet change:sheet:attributes', fillList);
        }, this);

        // create the context menu
        contextMenu = new SheetContextMenu(docView, this.getNode());

        // the update handler scrolls to the active sheet, after it has been changed
        this.registerUpdateHandler(updateHandler);

        // add scroll buttons to the group node
        this.addChildNodes(prevButton, nextButton);

        // double click shows rename dialog (TODO: in-place editing)
        this.getNode().on('dblclick', Forms.OPTION_BUTTON_SELECTOR, doubleClickHandler);

        // handle tracking events of scroll buttons
        Tracking.enableTracking(prevButton, { autoRepeat: true })
            .on('tracking:start tracking:repeat', prevButtonTrackingHandler)
            .on('tracking:move', function (event) { event.stopPropagation(); })
            .on('tracking:end tracking:cancel', buttonTrackingEndHandler);
        Tracking.enableTracking(nextButton, { autoRepeat: true })
            .on('tracking:start tracking:repeat', nextButtonTrackingHandler)
            .on('tracking:move', function (event) { event.stopPropagation(); })
            .on('tracking:end tracking:cancel', buttonTrackingEndHandler);

        // drag sheet buttons to a new position (not on touch devices)
        if (!Utils.TOUCHDEVICE) {
            Tracking.enableTracking(this.getNode(), { selector: Forms.OPTION_BUTTON_SELECTOR, cursor: 'ew-resize', autoScroll: 'horizontal', scrollInterval: 500, borderMargin: -30 })
                .on('tracking:start', dndTrackingStartHandler)
                .on('tracking:move', dndTrackingMoveHandler)
                .on('tracking:scroll', dndTrackingScrollHandler)
                .on('tracking:end', dndTrackingEndHandler)
                .on('tracking:cancel', dndTrackingCancelHandler);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            self = docModel = docView = contextMenu = null;
            prevButton = nextButton = dndTrackerNode = dndSheetNode = null;
        });

    } }); // class ActiveSheetGroup

    // exports ================================================================

    return ActiveSheetGroup;

});
