/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/range3d', [
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/range'
], function (Address, Range) {

    'use strict';

    // convenience shortcuts
    var RangeProto = Range.prototype;

    // class Range3D ==========================================================

    /**
     * A cell range address with two sheet index properties, that represents a
     * cell range in a specific single sheet, or in a range of consecutive
     * sheets of a spreadsheet document.
     *
     * @constructor
     *
     * @extends Range
     *
     * @property {Number} sheet1
     *  The zero-based index of the first sheet of this cell range address.
     *
     * @property {Number} sheet2
     *  The zero-based index of the second sheet of this cell range address.
     */
    var Range3D = Range.extend(function (sheet1, sheet2, start, end) {

        Range.call(this, start, end);

        this.sheet1 = sheet1;
        this.sheet2 = sheet2;

    }); // class Range3D

    // static methods ---------------------------------------------------------

    /**
     * Creates a cell range address from the passed column, row, and sheet
     * indexes. The column, row, and sheet indexes in the resulting cell range
     * address will be ordered automatically.
     *
     * @param {Number} col1
     *  The column index of one of the borders in the cell range.
     *
     * @param {Number} row1
     *  The row index of one of the borders in the cell range.
     *
     * @param {Number} col2
     *  The column index of the other border in the cell range.
     *
     * @param {Number} row2
     *  The row index of the other border in the cell range.
     *
     * @param {Number} sheet1
     *  The zero-based index of the first sheet of the new cell range address.
     *
     * @param {Number} [sheet2]
     *  The zero-based index of the second sheet of the new cell range address.
     *  If omitted, the new range will refer to a single sheet.
     *
     * @returns {Range3D}
     *  The cell range address with adjusted column and row indexes.
     */
    Range3D.create = function (col1, row1, col2, row2, sheet1, sheet2) {
        var start = new Address(Math.min(col1, col2), Math.min(row1, row2)),
            end = new Address(Math.max(col1, col2), Math.max(row1, row2));
        if (typeof sheet2 !== 'number') { sheet2 = sheet1; }
        return new Range3D(Math.min(sheet1, sheet2), Math.max(sheet1, sheet2), start, end);
    };

    /**
     * Creates a cell range address with sheet indexes from the passed simple
     * cell address. The sheet indexes in the resulting cell range address will
     * be ordered automatically.
     *
     * @param {Address} address
     *  A simple cell address without sheet indexes.
     *
     * @param {Number} sheet1
     *  The zero-based index of the first sheet of the new cell range address.
     *
     * @param {Number} [sheet2]
     *  The zero-based index of the second sheet of the new cell range address.
     *  If omitted, the new range will refer to a single sheet.
     *
     * @returns {Range3D}
     *  The cell range address created from the passed parameters.
     */
    Range3D.createFromAddress = function (address, sheet1, sheet2) {
        return Range3D.create(address[0], address[1], address[0], address[1], sheet1, sheet2);
    };

    /**
     * Creates a cell range address with sheet indexes from the passed cell
     * addresses. The columns, rows, and sheets in the resulting cell range
     * address will be ordered automatically. The passed cell addresses will be
     * deeply cloned when creating the new range address.
     *
     * @param {Address} address1
     *  The first cell address used to create the new cell range address.
     *
     * @param {Address} address2
     *  The second cell address used to create the new cell range address.
     *
     * @param {Number} sheet1
     *  The zero-based index of the first sheet of the new cell range address.
     *
     * @param {Number} [sheet2]
     *  The zero-based index of the second sheet of the new cell range address.
     *  If omitted, the new range will refer to a single sheet.
     *
     * @returns {Range3D}
     *  The cell range address created from the passed cell addresses.
     */
    Range3D.createFromAddresses = function (address1, address2, sheet1, sheet2) {
        return Range3D.create(address1[0], address1[1], address2[0], address2[1], sheet1, sheet2);
    };

    /**
     * Creates a cell range address with sheet indexes from the passed simple
     * cell range address. The sheet indexes in the resulting cell range
     * address will be ordered automatically. The start and end address of the
     * original cell range address will be deeply cloned when creating the new
     * range address.
     *
     * @param {Range} range
     *  A simple cell range address without sheet indexes.
     *
     * @param {Number} sheet1
     *  The zero-based index of the first sheet of the new cell range address.
     *
     * @param {Number} [sheet2]
     *  The zero-based index of the second sheet of the new cell range address.
     *  If omitted, the new range will refer to a single sheet.
     *
     * @returns {Range3D}
     *  The cell range address created from the passed parameters.
     */
    Range3D.createFromRange = function (range, sheet1, sheet2) {
        return Range3D.create(range.start[0], range.start[1], range.end[0], range.end[1], sheet1, sheet2);
    };

    /**
     * Creates a cell range address with sheet indexes from the passed column
     * and row interval. The sheet indexes in the resulting cell range address
     * will be ordered automatically.
     *
     * @param {Interval} colInterval
     *  The column interval of the new cell range address.
     *
     * @param {Interval} rowInterval
     *  The row interval of the new cell range address.
     *
     * @param {Number} sheet1
     *  The zero-based index of the first sheet of the new cell range address.
     *
     * @param {Number} [sheet2]
     *  The zero-based index of the second sheet of the new cell range address.
     *  If omitted, the new range will refer to a single sheet.
     *
     * @returns {Range3D}
     *  The cell range address created from the passed index intervals.
     */
    Range3D.createFromIntervals = function (colInterval, rowInterval, sheet1, sheet2) {
        return Range3D.create(colInterval.first, rowInterval.first, colInterval.last, rowInterval.last, sheet1, sheet2);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a unique string key for this cell range address that can be used
     * for example as key in an associative map. This method is faster than the
     * method Range3D.toString().
     *
     * @returns {String}
     *  A unique string key for this cell range address.
     */
    Range3D.prototype.key = function () {
        return this.sheet1 + ':' + this.sheet2 + '!' + RangeProto.key.call(this);
    };

    /**
     * Creates a deep clone of this cell range address.
     *
     * @returns {Range3D}
     *  A deep clone of this range address.
     */
    Range3D.prototype.clone = function () {
        return new Range3D(this.sheet1, this.sheet2, this.start.clone(), this.end.clone());
    };

    /**
     * Returns whether the passed cell range address is equal to this cell
     * range address.
     *
     * @param {Range3D} range
     *  The other cell range address to be compared to this cell range address.
     *
     * @returns {Boolean}
     *  Whether both cell range addresses contain the same start and end cell
     *  addresses.
     */
    Range3D.prototype.equals = function (range) {
        return (this.sheet1 === range.sheet1) && (this.sheet2 === range.sheet2) && RangeProto.equals.call(this, range);
    };

    /**
     * Returns whether the passed cell range address differs from this cell
     * range address.
     *
     * @param {Range3D} range
     *  The other cell range address to be compared to this cell range address.
     *
     * @returns {Boolean}
     *  Whether the passed cell range address differs from this cell range
     *  address.
     */
    Range3D.prototype.differs = function (range) {
        return (this.sheet1 !== range.sheet1) || (this.sheet2 !== range.sheet2) || RangeProto.differs.call(this, range);
    };

    /**
     * Returns the number of sheets covered by this cell range address.
     *
     * @returns {Number}
     *  The number of sheets covered by this cell range address.
     */
    Range3D.prototype.sheets = function () {
        return this.sheet2 - this.sheet1 + 1;
    };

    /**
     * Returns the number of cells covered by this cell range address.
     *
     * @returns {Number}
     *  The number of cells covered by this cell range address.
     */
    Range3D.prototype.cells = function () {
        return this.sheets() * RangeProto.cells.call(this);
    };

    /**
     * Returns whether this cell range address covers a single sheet.
     *
     * @returns {Boolean}
     *  Whether this cell range address covers a single sheet.
     */
    Range3D.prototype.singleSheet = function () {
        return this.sheet1 === this.sheet2;
    };

    /**
     * Returns whether this cell range address refers exactly to the specified
     * sheet index (with both sheet indexes).
     *
     * @param {Number} sheet
     *  The zero-based sheet index to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address refers to the specified sheet index.
     */
    Range3D.prototype.isSheet = function (sheet) {
        return (this.sheet1 === sheet) && (this.sheet2 === sheet);
    };

    /**
     * Returns whether this cell range address contains the specified sheet
     * index.
     *
     * @param {Number} sheet
     *  The zero-based sheet index to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address contains the specified sheet index.
     */
    Range3D.prototype.containsSheet = function (sheet) {
        return (this.sheet1 <= sheet) && (sheet <= this.sheet2);
    };

    /**
     * Returns whether this cell range address contains the passed cell range
     * address completely, including their sheet ranges.
     *
     * @param {Range3D} range
     *  The other cell range address to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address contains the passed cell range address.
     */
    Range3D.prototype.contains = function (range) {
        return (this.sheet1 <= range.sheet1) && (range.sheet2 <= this.sheet2) && RangeProto.contains.call(this, range);
    };

    /**
     * Returns whether this cell range address overlaps with the passed cell
     * range address by at least one cell, including their sheet ranges.
     *
     * @param {Range3D} range
     *  The other cell range address to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address overlaps with the passed cell range
     *  address by at least one cell.
     */
    Range3D.prototype.overlaps = function (range) {
        return (this.sheet1 <= range.sheet2) && (range.sheet1 <= this.sheet2) && RangeProto.overlaps.call(this, range);
    };

    /**
     * Returns the address of the specified columns in this cell range address.
     *
     * @param {Number} index
     *  The relative column index (the value zero will return the address of
     *  the leading column in this range).
     *
     * @param {Number} [count=1]
     *  The number of columns covered by the resulting range.
     *
     * @returns {Range3D}
     *  The address of the specified columns in this cell range address.
     */
    Range3D.prototype.colRange = function (index, count) {
        return Range3D.createFromRange(RangeProto.colRange.call(this, index, count), this.sheet1, this.sheet2);
    };

    /**
     * Returns the address of the specified rows in this cell range address.
     *
     * @param {Number} index
     *  The relative row index (the value zero will return the address of the
     *  top row in this range).
     *
     * @param {Number} [count=1]
     *  The number of rows covered by the resulting range.
     *
     * @returns {Range3D}
     *  The address of the specified rows in this cell range address.
     */
    Range3D.prototype.rowRange = function (index, count) {
        return Range3D.createFromRange(RangeProto.rowRange.call(this, index, count), this.sheet1, this.sheet2);
    };

    /**
     * Returns the address of the specified columns or rows in this cell range
     * address.
     *
     * @param {Boolean} columns
     *  Whether to return a column range (true), or a row range (false) of this
     *  cell range address.
     *
     * @param {Number} index
     *  The relative column/row index (the value zero will return the address
     *  of the leading column or top row in this range).
     *
     * @param {Number} [count=1]
     *  The number of columns or rows covered by the resulting range.
     *
     * @returns {Range3D}
     *  The address of the specified columns/rows in this cell range address.
     */
    Range3D.prototype.lineRange = function (columns, index, count) {
        return columns ? this.colRange(index, count) : this.rowRange(index, count);
    };

    /**
     * Returns the address of the bounding cell range of this cell range, and
     * the passed cell range address (the smallest range that contains both
     * ranges).
     *
     * @param {Range3D} range
     *  The other cell range address to create the bounding range from.
     *
     * @returns {Range3D}
     *  The address of the bounding cell range of this cell range, and the
     *  passed cell range address
     */
    Range3D.prototype.boundary = function (range) {
        return Range3D.create(
            Math.min(this.start[0], range.start[0]),
            Math.min(this.start[1], range.start[1]),
            Math.max(this.end[0], range.end[0]),
            Math.max(this.end[1], range.end[1]),
            Math.min(this.sheet1, range.sheet1),
            Math.max(this.sheet2, range.sheet2)
        );
    };

    /**
     * Returns the intersecting cell range address between this cell range
     * address, and the passed cell range address.
     *
     * @param {Range3D} range
     *  The other cell range address.
     *
     * @returns {Range3D|Null}
     *  The address of the cell range covered by both cell range addresses, if
     *  existing; otherwise null.
     */
    Range3D.prototype.intersect = function (range) {
        return this.overlaps(range) ? Range3D.create(
            Math.max(this.start[0], range.start[0]),
            Math.max(this.start[1], range.start[1]),
            Math.min(this.end[0], range.end[0]),
            Math.min(this.end[1], range.end[1]),
            Math.max(this.sheet1, range.sheet1),
            Math.min(this.sheet2, range.sheet2)
        ) : null;
    };

    /**
     * Converts this cell range address to an instance of the class Range
     * without sheet indexes.
     *
     * @returns {Range}
     *  An instance of the class Range containing start and end cell address of
     *  this range, but no sheet indexes.
     */
    Range3D.prototype.toRange = function () {
        return new Range(this.start, this.end);
    };

    /**
     * Returns the string representation of this cell range address, in
     * upper-case A1 notation. The sheet indexes will be inserted as numbers.
     *
     * @returns {String}
     *  The string representation of this cell range address, in upper-case A1
     *  notation.
     */
    Range3D.prototype.toString = function () {
        return this.sheet1 + ':' + this.sheet2 + '!' + RangeProto.toString.call(this);
    };

    /**
     * Returns the JSON representation of this cell range address.
     *
     * @returns {Object}
     *  The JSON representation of this cell range address (a plain JavaScript
     *  object with the properties 'start' and 'end' being JSON representations
     *  of the start and end cell address, and the number properties 'sheet1'
     *  and 'sheet2').
     */
    Range3D.prototype.toJSON = function () {
        var json = RangeProto.toJSON.call(this);
        json.sheet1 = this.sheet1;
        json.sheet2 = this.sheet2;
        return json;
    };

    // exports ================================================================

    return Range3D;

});
