/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/mathfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/utils/statutils'
], function (ErrorCode, FormulaUtils, StatUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all mathematical spreadsheet functions.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // convenience shortcuts
    var MathUtils = FormulaUtils.Math;
    var FilterType = FormulaUtils.FilterType;

    // mathematical constants
    var PI = Math.PI;
    var PI_2 = PI / 2;
    var PI_180 = PI / 180;

    // shortcuts to mathematical functions
    var floor = Math.floor;
    var ceil = Math.ceil;
    var round = Math.round;
    var trunc = MathUtils.trunc;
    var abs = Math.abs;
    var div = MathUtils.div;
    var pow10 = MathUtils.pow10;
    var sqrt = Math.sqrt;
    var exp = Math.exp;
    var log = Math.log;
    var log10 = MathUtils.log10;
    var sin = Math.sin;
    var sinh = MathUtils.sinh;
    var cos = Math.cos;
    var cosh = MathUtils.cosh;
    var tan = Math.tan;
    var tanh = MathUtils.tanh;
    var atan = Math.atan;

    // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
    var SKIP_MAT_SKIP_REF_SKIP_EMPTY = { // id: CSS4
        valMode: 'convert', // value operands: convert strings and booleans to numbers
        matMode: 'skip', // matrix operands: skip strings and booleans
        refMode: 'skip', // reference operands: skip strings and booleans
        // emptyParam: false, // skip all empty parameters
        complexRef: true // accept multi-range and multi-sheet references
    };

    // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
    var RCONVERT_ALL_SKIP_EMPTY = { // id: RRR0
        valMode: 'rconvert', // value operands: convert strings to numbers (but not booleans)
        matMode: 'rconvert', // matrix operands: convert strings to numbers (but not booleans)
        refMode: 'rconvert' // reference operands: convert strings to numbers (but not booleans)
        // emptyParam: false, // skip all empty parameters
        // complexRef: false // do not accept multi-range and multi-sheet references
    };

    // private global functions ===============================================

    /**
     * Returns the greatest common divisor of the passed integer numbers.
     *
     * @param {Number} int1
     *  The first integer number.
     *
     * @param {Number} int2
     *  The second integer number.
     *
     * @returns {Number}
     *  The greatest common divisor of the passed numbers.
     */
    function gcd(int1, int2) {
        while (int2 > 0) {
            var tmp = int2;
            int2 = round(int1 % int2);
            int1 = tmp;
        }
        return int1;
    }

    /**
     * @param {Number} value
     *  the value to be rounded
     *
     * @param {Number} mult
     *  multiple to round to
     *
     * @returns {Number}
     *  the value rounded to a multiple of mult
     *
     * @throws {ErrorCode}
     *  if signs of value and mult are different
     */
    function mround(value, mult) {
        if (value * mult < 0) {
            throw ErrorCode.NUM;
        }
        var mod = abs(value % mult),
            result = abs(value),
            amult = abs(mult);
        if (mod) {
            if (mod > amult / 2) {
                result += amult - mod;
            } else {
                result -= mod;
            }
        }
        return value < 0 ? -result : result;
    }

    // exports ================================================================

    return {

        ABS: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: abs
        },

        ACOS: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: Math.acos
        },

        ACOSH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: MathUtils.acosh
        },

        ACOT: {
            category: 'math',
            name: { ooxml: '_xlfn.ACOT' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return PI_2 - atan(number);
            }
        },

        ACOTH: {
            category: 'math',
            name: { ooxml: '_xlfn.ACOTH' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: MathUtils.acoth
        },

        AGGREGATE: {
            category: 'math',
            name: { ooxml: '_xlfn.AGGREGATE', odf: 'COM.MICROSOFT.AGGREGATE' },
            minParams: 3,
            type: 'val:num',
            recalc: 'always', // function result depends on hidden state of rows
            signature: 'val:int val:int any any ref',
            resolve: (function () {

                // all function implementations used by the AGGREGATE function
                var AGGREGATE_FUNCTIONS_INFO = [
                    { func: StatUtils.groupAverage,       type: 'any' },
                    { func: StatUtils.groupCount,         type: 'any' },
                    { func: StatUtils.groupCountA,        type: 'any' },
                    { func: StatUtils.groupMax,           type: 'any' },
                    { func: StatUtils.groupMin,           type: 'any' },
                    { func: StatUtils.groupProduct,       type: 'any' },
                    { func: StatUtils.groupStdDevS,       type: 'any' },
                    { func: StatUtils.groupStdDevP,       type: 'any' },
                    { func: StatUtils.groupSum,           type: 'any' },
                    { func: StatUtils.groupVarS,          type: 'any' },
                    { func: StatUtils.groupVarP,          type: 'any' },
                    { func: StatUtils.groupMedian,        type: 'any' },
                    { func: StatUtils.groupModeSngl,      type: 'any' },
                    { func: StatUtils.groupLarge,         type: 'val:int' },
                    { func: StatUtils.groupSmall,         type: 'val:int' },
                    { func: StatUtils.groupPercentileInc, type: 'val:num' },
                    { func: StatUtils.groupQuartileInc,   type: 'val:int' },
                    { func: StatUtils.groupPercentileExc, type: 'val:num' },
                    { func: StatUtils.groupQuartileExc,   type: 'val:int' }
                ];

                // the implementation of the AGGREGATE function
                return function (func, flags) {

                    // get the function implementation (one-based to zero-based index!)
                    var funcInfo = AGGREGATE_FUNCTIONS_INFO[func - 1];
                    if (!funcInfo) { throw ErrorCode.VALUE; }

                    // initialize the options according to the passed flags:
                    var options = {};
                    // bit 0: whether to visit visible rows only (hidden columns will always be visited!)
                    if (flags & 1) { options.skipHiddenRows = true; }
                    // bit 1: whether to skip error codes in source data (by default, they will be thrown immediately)
                    if (flags & 2) { options.skipErrors = true; }

                    // 'any' functions: invoke the function implementation with the remaining operands
                    if (funcInfo.type === 'any') {
                        return funcInfo.func(this, this.getOperands(2), options);
                    }

                    // parameter count of functions with numeric fourth parameter (e.g. LARGE/SMALL) are restricted
                    if (this.getOperandCount() !== 4) { throw ErrorCode.VALUE; }
                    return funcInfo.func(this, this.getOperand(2), this.getOperand(3, funcInfo.type), options);
                };
            }())
        },

        ASIN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: Math.asin
        },

        ASINH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: MathUtils.asinh
        },

        ATAN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: atan
        },

        ATAN2: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: MathUtils.atan2 // parameter order is (x,y), Math.atan2 expects (y,x)
        },

        ATANH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: MathUtils.atanh
        },

        CEILING: {
            category: 'math',
            name: { odf: 'COM.MICROSOFT.CEILING' },
            // TODO: change to fixed-2-parameters
            minParams: { ooxml: 2, odf: 1 },
            maxParams: { ooxml: 2, odf: 3 },
            type: 'val:num',
            signature: 'val:num val:num val:int',
            resolve: 'CEILING.MATH'
        },

        'CEILING.MATH': {
            category: 'math',
            name: { ooxml: '_xlfn.CEILING.MATH', odf: 'COM.MICROSOFT.CEILING.MATH' },
            minParams: 1,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:int',
            resolve: function (number, significance, mode) {
                var result, modulo, minBorder = 1e-16;
                if (_.isUndefined(significance)) {
                    significance = 1;
                    //TODO: empty cell of significance must be interpreted as 0
                }

                if (_.isUndefined(mode)) {
                    //TODO: odf correct?!
                    mode = 0;//this.ODF ? 0 : 1;
                }

                if (significance === 0) {
                    throw ErrorCode.VALUE;
                } else if (significance < 0 && number > 0) {
                    throw ErrorCode.NUM;
                }
                modulo = abs(significance - (number % significance));
                if (number === 0 || modulo < minBorder) {
                    result = number;
                } else {
                    if (mode !== 0 && number < 0) {
                        result = (floor(number / significance)) * significance;
                    } else {
                        result = (floor(number / significance) + 1) * significance;
                    }
                }

                return result;
            }
        },

        'CEILING.ODF': {
            category: 'math',
            name: { ooxml: null, odf: 'CEILING' },
            minParams: 1,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:int',
            resolve: 'CEILING.MATH'
        },

        'CEILING.PRECISE': {
            category: 'math',
            name: { ooxml: '_xlfn.CEILING.PRECISE', odf: 'COM.MICROSOFT.CEILING.PRECISE' },
            hidden: true, // replaced by CEILING.MATH in XL2013
            minParams: 1,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: 'CEILING.MATH'
        },

        COMBIN: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:int val:int',
            resolve: MathUtils.binomial
        },

        COMBINA: {
            category: 'math',
            name: { ooxml: '_xlfn.COMBINA' },
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:int val:int',
            resolve: function (number1, number2) {
                return MathUtils.binomial(number1 + number2 - 1, number2);
            }
        },

        COS: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: cos
        },

        COSH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: cosh
        },

        COT: {
            category: 'math',
            name: { ooxml: '_xlfn.COT' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return div(1, tan(number));
            }
        },

        COTH: {
            category: 'math',
            name: { ooxml: '_xlfn.COTH' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return div(1, tanh(number));
            }
        },

        CSC: {
            category: 'math',
            name: { ooxml: '_xlfn.CSC' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return div(1, sin(number));
            }
        },

        CSCH: {
            category: 'math',
            name: { ooxml: '_xlfn.CSCH' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return div(2 * exp(-number), 1 - exp(-2 * number));
            }
        },

        DEGREES: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return number / PI_180;
            }
        },

        EVEN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                var result = 2 * ceil(abs(number) / 2);
                return (number < 0) ? -result : result;
            }
        },

        EXP: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: exp
        },

        FACT: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:int',
            resolve: MathUtils.factorial
        },

        FACTDOUBLE: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:int',
            resolve: MathUtils.factorial2
        },

        FLOOR: {
            category: 'math',
            name: { odf: 'COM.MICROSOFT.FLOOR' },
            // TODO: change to fixed-2-parameters
            minParams: { ooxml: 2, odf: 1 },
            maxParams: { ooxml: 2, odf: 3 },
            type: 'val:num',
            signature: 'val:num val:num val:int',
            resolve: 'FLOOR.MATH'
        },

        'FLOOR.MATH': {
            category: 'math',
            name: { ooxml: '_xlfn.FLOOR.MATH', odf: 'COM.MICROSOFT.FLOOR.MATH' },
            minParams: 1,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:int',
            resolve: function (number, significance, mode) {
                var result, modulo, minBorder = 1e-16;

                if (_.isUndefined(significance)) {
                    significance = 1;
                    //TODO: empty cell of significance must be interpreted as 0
                }

                if (_.isUndefined(mode)) {
                    //TODO: odf correct?!
                    mode = 0;//this.ODF ? 0 : 1;
                }
                if (significance === 0 || (number > 0 && significance < 0)) {
                    throw ErrorCode.VALUE;
                }
                modulo = abs(significance - (number % significance));
                if (number === 0 || modulo < minBorder) {
                    result = number;
                } else {
                    if (mode !== 0 && number < 0) {
                        result = (floor(number / significance) + 1) * significance;
                    } else {
                        result = (floor(number / significance)) * significance;
                    }
                }
                return result;
            }
        },

        'FLOOR.ODF': {
            category: 'math',
            name: { ooxml: null, odf: 'FLOOR' },
            minParams: 1,
            maxParams: 3,
            type: 'val:num',
            signature: 'val:num val:num val:int',
            resolve: 'FLOOR.MATH'
        },

        'FLOOR.PRECISE': {
            category: 'math',
            name: { ooxml: '_xlfn.FLOOR.PRECISE', odf: 'COM.MICROSOFT.FLOOR.PRECISE' },
            hidden: true, // replaced by FLOOR.MATH in XL2013
            minParams: 1,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: 'FLOOR.MATH'
        },

        GCD: {
            category: 'math',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: (function () {

                // returns the GCD of the result and the new number
                function aggregate(result, number) {
                    // round down to integers, fail on negative numbers
                    number = floor(number);
                    if (number < 0) { throw ErrorCode.NUM; }
                    return gcd(result, number);
                }

                // throw #VALUE! without any number (reference to empty cells)
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCode.VALUE; }
                    return result;
                }

                return function () {
                    // start with 0, needed to force calling aggregate() for all function
                    // parameters for integer conversion and check for negative numbers
                    // Booleans lead to error: GCD(6,TRUE) = #VALUE!
                    // empty parameters are ignored: GCD(6,) = 6
                    return this.aggregateNumbers(arguments, 0, aggregate, finalize, RCONVERT_ALL_SKIP_EMPTY);
                };
            }())
        },

        INT: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:int',
            resolve: _.identity // always rounds down, e.g. INT(-1.5) = -2
        },

        'ISO.CEILING': {
            category: 'math',
            name: { odf: 'COM.MICROSOFT.ISO.CEILING' },
            hidden: true, // replaced by CEILING.MATH in XL2013
            minParams: 1,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: 'CEILING.PRECISE'
        },

        LCM: {
            category: 'math',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: (function () {

                // updates the LCM in the passed result according to the passed number
                function aggregate(result, number) {
                    // round down to integers, fail on negative numbers
                    number = floor(number);
                    if (number < 0) { throw ErrorCode.NUM; }
                    return ((result === 0) || (number === 0)) ? 0 : (result * number / gcd(number, result));
                }

                // throw #VALUE! without any number (reference to empty cells)
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCode.VALUE; }
                    return result;
                }

                // start with 1, needed to force calling aggregate() for all function
                // parameters for integer conversion and check for negative numbers
                // Booleans lead to error: LCM(6,TRUE) = #VALUE!
                // empty parameters are ignored: LCM(6,) = 6 while LCM(6,0) = 0
                return function () {
                    return this.aggregateNumbers(arguments, 1, aggregate, finalize, RCONVERT_ALL_SKIP_EMPTY);
                };
            }())
        },

        LN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: log
        },

        LOG: {
            category: 'math',
            minParams: 1,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: function (number, base) {
                return this.isMissingOperand(1) ? log10(number) : (log(number) / log(base));
            }
        },

        LOG10: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: log10
        },

        MOD: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: MathUtils.mod
        },

        MROUND: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: mround
        },

        MULTINOMIAL: {
            category: 'math',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: (function () {

                // round down all numbers to integers
                var MULTINOMIAL_OPTIONS = _.extend({}, RCONVERT_ALL_SKIP_EMPTY, { floor: true });

                // use the same simple implementation as Excel: if the sum of all numbers exceeds
                // the value 170, getting factorial in the numerator will fail with a #NUM! error,
                // although the final result of the function would be small enough
                function finalize(numbers, sum) {
                    if (numbers.length === 0) { throw ErrorCode.VALUE; }
                    var denom = numbers.reduce(function (accu, number) {
                        return accu * MathUtils.factorial(number);
                    }, 1);
                    return MathUtils.factorial(sum) / denom;
                }

                // the implementation of the MULTINOMIAL function
                return function () {
                    return this.aggregateNumbersAsArray(arguments, finalize, MULTINOMIAL_OPTIONS);
                };
            }())
        },

        N: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val',
            resolve: function (value) {
                return _.isNumber(value) ? value : (value === true) ? 1 : 0;
            }
        },

        NEG: {
            category: 'math',
            name: { ooxml: null },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) { return -number; }
        },

        ODD: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                var result = 2 * ceil((abs(number) + 1) / 2) - 1;
                return (number < 0) ? -result : result;
            }
        },

        PI: {
            category: 'math',
            minParams: 0,
            maxParams: 0,
            type: 'val:num',
            resolve: _.constant(PI)
        },

        POWER: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: MathUtils.POW
        },

        PRODUCT: {
            category: 'math',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () { return StatUtils.groupProduct(this, arguments); }
        },

        QUOTIENT: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:num',
            resolve: function (number1, number2) {
                return trunc(div(number1, number2));
            }
        },

        RADIANS: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return number * PI_180;
            }
        },

        RAND: {
            category: 'math',
            minParams: 0,
            maxParams: 0,
            type: 'val:num',
            recalc: 'always',
            resolve: Math.random
        },

        RANDBETWEEN: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            recalc: 'always',
            signature: 'val:num val:num',
            resolve: function (min, max) {
                min = ceil(min);
                max = floor(max);
                if (max < min) { throw ErrorCode.NUM; }
                return floor(Math.random() * (max - min + 1)) + min;
            }
        },

        ROUND: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow10(-digits);
                return mround(number, number < 0 ? -mult : mult);
            }
        },

        ROUNDDOWN: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow10(-digits),
                    smult = (number < 0) ? -mult : mult,
                    result = mround(number, smult);
                return ((result !== number) && (abs(result) > abs(number))) ? (result - smult) : result;
            }
        },

        ROUNDUP: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow10(-digits),
                    smult = (number < 0) ? -mult : mult,
                    result = mround(number, smult);
                return ((result !== number) && (abs(result) < abs(number))) ? (result + smult) : result;
            }
        },

        SEC: {
            category: 'math',
            name: { ooxml: '_xlfn.SEC' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return div(1, cos(number));
            }
        },

        SECH: {
            category: 'math',
            name: { ooxml: '_xlfn.SECH' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return div(2 * exp(-number), 1 + exp(-2 * number));
            }
        },

        SERIESSUM: {
            category: 'math',
            minParams: 4,
            maxParams: 4,
            type: 'val:num',
            signature: 'val:num val:num val:num any|mat:pass',
            resolve: (function () {

                var AGGREGATE_OPTIONS = {
                    valMode: 'rconvert', // value operands: convert strings to numbers (but not booleans)
                    matMode: 'exact', // matrix operands: do not accept strings or booleans
                    refMode: 'exact' // reference operands: do not accept strings or booleans
                    // emptyParam: false, // skip empty parameter
                    // complexRef: false // do not accept multi-range and multi-sheet references
                };

                // the finalizer callback: throw #N/A error code, if no coefficients are available
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCode.NA; }
                    return result;
                }

                // the implementation of the SERIESSUM function
                return function (x, n, m, c) {
                    return this.aggregateNumbers([c], 0, function (result, a, i) {
                        return result + a * Math.pow(x, n + i * m);
                    }, finalize, AGGREGATE_OPTIONS);
                };
            }())
        },

        SIGN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return (number < 0) ? -1 : (number > 0) ? 1 : 0;
            }
        },

        SIN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: sin
        },

        SINH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: sinh
        },

        SQRT: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: sqrt
        },

        SQRTPI: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: function (number) {
                return sqrt(PI * number);
            }
        },

        SUBTOTAL: {
            category: 'math',
            minParams: 2,
            type: 'val:num',
            recalc: 'always', // function result depends on hidden state of rows
            signature: 'val:int any|mat:pass',
            resolve: (function () {

                // all function implementations used by the SUBTOTAL function
                var SUBTOTAL_FUNCTIONS = [
                    StatUtils.groupAverage,
                    StatUtils.groupCount,
                    StatUtils.groupCountA,
                    StatUtils.groupMax,
                    StatUtils.groupMin,
                    StatUtils.groupProduct,
                    StatUtils.groupStdDevS,
                    StatUtils.groupStdDevP,
                    StatUtils.groupSum,
                    StatUtils.groupVarS,
                    StatUtils.groupVarP
                ];

                // the implementation of the SUBTOTAL function
                return function (func) {

                    // get the function implementation (one-based to zero-based index!)
                    if ((func < 1) || (func > 199)) { throw ErrorCode.VALUE; }
                    var funcImpl = SUBTOTAL_FUNCTIONS[func % 100 - 1];
                    if (!funcImpl) { throw ErrorCode.VALUE; }

                    // Function indexes above 100 are used to skip manually hidden rows.
                    // Bug 49375: If a filter is active (auto-filter or table), ALL hidden rows will
                    // ALWAYS be skipped (not only filtered rows, but also all manually hidden rows).
                    return funcImpl(this, this.getOperands(1), {
                        skipHiddenRows: func >= 100,
                        skipFilteredRows: true
                    });
                };
            }())
        },

        SUM: {
            category: 'math',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {
                return StatUtils.groupSum(this, arguments, { collectFormats: true });
            }
        },

        SUMIF: {
            category: 'math',
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:val val:any ref:val',
            resolve: function () {
                return this.aggregateFilteredCells(FilterType.SINGLE, 0, MathUtils.add);
            }
        },

        SUMIFS: {
            category: 'math',
            minParams: 3,
            repeatParams: 2,
            type: 'val:num',
            signature: 'ref:val ref:val val:any',
            resolve: function () {
                return this.aggregateFilteredCells(FilterType.MULTI, 0, MathUtils.add);
            }
        },

        SUMSQ: {
            category: 'math',
            minParams: 1,
            type: 'val:num',
            signature: 'any|mat:pass',
            resolve: function () {
                function aggregate(result, number) { return result + number * number; }
                return this.aggregateNumbers(arguments, 0, aggregate, _.identity, SKIP_MAT_SKIP_REF_SKIP_EMPTY);
            }
        },

        TAN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: tan
        },

        TANH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:num',
            resolve: tanh
        },

        TRUNC: {
            category: 'math',
            minParams: 1,
            maxParams: 2,
            type: 'val:num',
            signature: 'val:num val:int',
            resolve: function (number, digits) {

                // nothing to do for zeros
                if (number === 0) { return 0; }

                // default value for missing digits
                digits = digits || 0;
                // simpler implementation for integer truncation
                if (digits === 0) { return trunc(number); }

                // absolute number
                var absNum = abs(number);
                // exponent of the passed number
                var exp10 = floor(log10(absNum));
                // mantissa of the passed number
                var mant = absNum / pow10(exp10);

                // ignore the exponent when truncating the mantissa to the specified digits
                digits += exp10;
                // truncating mantissa (which is always less than 10) to entire tens results in zero
                if (digits < 0) { return 0; }
                // truncating to more than 14 decimal places does not change the passed number
                if (digits >= 14) { return number; }
                // truncate and construct the result
                return ((number < 0) ? -1 : 1) * floor(mant * pow10(digits)) * pow10(exp10 - digits);
            }
        }
    };

});
