/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chart/titlemodel', [
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/drawing/sourcelinkmixin'
], function (AttributedModel, ChartStyleUtil, Operations, SourceLinkMixin) {

    'use strict';

    // class TitleModel =======================================================

    /**
     * A title in a chart object. Can be used as main title of the chart, or as
     * title of a chart axis.
     *
     * @constructor
     *
     * @extends AttributedModel
     * @extends SourceLinkMixin
     *
     * @param {SheetChartModel} chartModel
     *  The parent chart model containing this title object.
     *
     * @param {String} axisId
     *  The identifier of this title object. Can be null for the main title
     *  of the chart, or the id for the respective axis title.
     */
    var TitleModel = AttributedModel.extend({ constructor: function (chartModel, axisId, data, textKey, dataKey) {

        var self = this;
        var sheetModel = chartModel.getSheetModel();
        var docModel = chartModel.getDocModel();

        // base constructors --------------------------------------------------

        AttributedModel.call(this, docModel, null, { families: 'text character' });
        SourceLinkMixin.call(this, sheetModel, 'text.link');

        // Fix for Bug 54278 - Chart - title with source link - change title name
        self.listenTo(self, 'refresh:formulas', function () {
            chartModel.refreshChartDebounced();
        });

        // public methods -----------------------------------------------------

        this.refreshInfo = function (chartTitle) {

            var display = this.resolveText('text.link') || chartTitle;
            var textLink = this.resolveText('text.link');
            var seriesTitle = chartTitle;

            if (textLink) {
                display = textLink;
            } else if (seriesTitle) {
                display = seriesTitle;
            }

            // If the ChartTitle is an automatic title and if the chart contains more than one dataseries, hide the auto title.
            if (!axisId && !textLink && seriesTitle && chartModel.getModelData().series.length > 1) {
                display = '';
            }

            var change = data[textKey] !== display;
            data[textKey] = display;

            if (display) {
                var attrs = self.getMergedAttributeSet(true);
                ChartStyleUtil.handleCharacterProps(chartModel, attrs.character, data, dataKey);
            }
            // Fix for Bug 54278 - Chart - title with source link - change title name
            if (change) {
                chartModel.trigger('change:drawing', 'series');
            }
            return change;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore this title object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @returns {TitleModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position) {
            if (this.hasExplicitAttributes()) {
                var properties = { attrs: this.getExplicitAttributeSet() };
                if (axisId !== null) {
                    properties.axis = axisId;
                }
                generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, properties, { undo: true });
            }
            return this;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source link of this title object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @param {Object} updateDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on its 'type' property. See
         *  method TokenArray.resolveOperation() for more details.
         *
         * @returns {TitleModel}
         *  A reference to this instance.
         */
        this.generateUpdateFormulaOperations = function (generator, position, updateDesc) {
            // mix-in class generates the necessary change operations
            var properties = {};
            if (axisId !== null) {
                properties.axis = axisId;
            }
            return this.implGenerateUpdateFormulaOperations(generator, Operations.SET_CHART_TITLE_ATTRIBUTES, position, updateDesc, properties);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = sheetModel = chartModel = null;
        });

    } }); // class TitleModel

    // exports ================================================================

    return TitleModel;

});
