/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellvaluecache', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/scalarset'
], function (Iterator, ValueMap, TriggerObject, SheetUtils, ScalarSet) {

    'use strict';

    // convenience shortcuts
    var RangeArray = SheetUtils.RangeArray;

    // class CellValueCache ===================================================

    /**
     * A cache for the values, and specific subtotal results, of the cells
     * covered by specific cell ranges in a sheet.
     *
     * The cache will collect all nessecary information lazily on first access.
     * It will register as a change listener at the cell collection, and will
     * invalidate itself whenever the values in the covered cell ranges have
     * been changed.
     *
     * Triggers the following events:
     *  - 'invalidate'
     *      After the value of at least one cell covered by this cache has been
     *      changed in the document. This cache has invalidated itself before
     *      triggering this event.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing the cells to be cached by this instance.
     *
     * @param {RangeArray|Range} [targetRanges]
     *  The addresses of the cell ranges to be cached by this instance. If
     *  omitted, the cache will not cover any ranges. New target ranges can be
     *  set later with the method CellValueCache.setRanges().
     */
    var CellValueCache = TriggerObject.extend({ constructor: function (sheetModel, targetRanges) {

        // self reference
        var self = this;

        // the cell collection of the sheet model
        var cellCollection = sheetModel.getCellCollection();

        // a map used as cache for various results
        var cacheMap = new ValueMap();

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, sheetModel.getDocModel());

        // private methods ----------------------------------------------------

        function collectSettings() {
            return cacheMap.getOrCreate('settings', function () {

                // initialization
                var valueMap = {};
                var valueCount = 0;
                var numbers = [];
                var sum = 0;
                var product = 1;
                var min = Number.POSITIVE_INFINITY;
                var max = Number.NEGATIVE_INFINITY;

                // collect the numbers of all covered cells
                var iterator = cellCollection.createAddressIterator(targetRanges, { type: 'value', covered: true });
                Iterator.forEach(iterator, function (address) {
                    var value = cellCollection.getValue(address);
                    var key = ScalarSet.key(value);
                    valueMap[key] = (valueMap[key] || 0) + 1;
                    valueCount += 1;
                    if (typeof value === 'number') {
                        numbers.push(value);
                        sum += value;
                        product *= value;
                        min = Math.min(min, value);
                        max = Math.max(max, value);
                    }
                });

                return {
                    valueMap: valueMap,
                    valueCount: valueCount,
                    blankCount: targetRanges.cells() - valueCount,
                    numbers: numbers,
                    sum: sum,
                    product: product,
                    min: min,
                    max: max,
                    mean: (numbers.length > 0)  ? (sum / numbers.length) : Number.NaN
                };
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the number of non-blank cells covered by the target ranges.
         *
         * @returns {Number}
         *  The number of non-blank cells covered by the target ranges.
         */
        this.getValueCount = function () {
            return collectSettings().valueCount;
        };

        /**
         * Returns the number of blank cells covered by the target ranges.
         *
         * @returns {Number}
         *  The number of blank cells covered by the target ranges.
         */
        this.getBlankCount = function () {
            return collectSettings().blankCount;
        };

        /**
         * Returns the number of occurrences of the specified scalar value.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The scalar value to be counted.
         *
         * @returns {Number}
         *  The number of occurrences of the specified scalar value.
         */
        this.countScalar = function (value) {
            var settings = collectSettings();
            return (value === null) ? settings.blankCount : (settings.valueMap[ScalarSet.key(value)] || 0);
        };

        /**
         * Returns the numbers from all cells covered by this cache as array.
         *
         * @returns {Array<Number>}
         *  The numbers from all cells covered by this cache as array.
         */
        this.getNumbers = function () {
            return collectSettings().numbers;
        };

        /**
         * Returns all numbers covered by this model as sorted array.
         *
         * @returns {Array<Number>}
         *  All numbers covered by this model as sorted array.
         */
        this.getSortedNumbers = function () {
            return cacheMap.getOrCreate('sorted', function () {
                return _.sortBy(this.getNumbers());
            }, this);
        };

        /**
         * Returns all numbers covered by this model as unified and sorted
         * array.
         *
         * @returns {Array<Number>}
         *  All numbers covered by this model as unified and sorted array.
         */
        this.getUniqueNumbers = function () {
            return cacheMap.getOrCreate('unified', function () {
                return _.unique(this.getSortedNumbers(), true);
            }, this);
        };

        /**
         * Returns the sum of all covered numbers.
         *
         * @returns {Number}
         *  The sum of all covered numbers.
         */
        this.getSum = function () {
            return collectSettings().sum;
        };

        /**
         * Returns the product of all covered numbers.
         *
         * @returns {Number}
         *  The product of all covered numbers.
         */
        this.getProduct = function () {
            return collectSettings().product;
        };

        /**
         * Returns the smallest of the covered numbers.
         *
         * @returns {Number}
         *  The smallest of the covered numbers; or POSITIVE_INFINITY if the
         *  covered cells do not contain any numbers.
         */
        this.getMin = function () {
            return collectSettings().min;
        };

        /**
         * Returns the largest of the covered numbers.
         *
         * @returns {Number}
         *  The largest of the covered numbers; or NEGATIVE_INFINITY if the
         *  covered cells do not contain any numbers.
         */
        this.getMax = function () {
            return collectSettings().max;
        };

        /**
         * Returns the arithmetic mean of all covered numbers.
         *
         * @returns {Number}
         *  The arithmetic mean of all covered numbers; or NaN if the covered
         *  cells do not contain any numbers.
         */
        this.getMean = function () {
            return collectSettings().mean;
        };

        /**
         * Returns the standard deviation of all covered numbers.
         *
         * @returns {Number}
         *  The standard deviation of all covered numbers; or NaN if the
         *  covered cells do not contain any numbers.
         */
        this.getDeviation = function () {
            return cacheMap.getOrCreate('deviation', function () {
                var settings = collectSettings();
                var numbers = settings.numbers;
                if (numbers.length === 0) { return Number.NaN; }
                var mean = settings.mean;
                return numbers.reduce(function (sum, num) {
                    var diff = num - mean;
                    return sum + diff * diff;
                }, 0) / numbers.length;
            });
        };

        /**
         * Returns a custom cache entry. If the cache entry does not exist yet,
         * it will be created by invoking the passed callback function.
         *
         * @param {String} key
         *  A unique key for the custom cache entry.
         *
         * @param {Function} generator
         *  A callback function that will be invoked if this cache does not
         *  contain an entry with the specified custom key. Receives the
         *  following parameters:
         *  (1) {Object} iterator
         *      A cell address iterator that will generate the addresses of all
         *      (non-blank) value cells of the target ranges covered by this
         *      value cache.
         *  The return value of this function will be inserted into this cache.
         *
         * @param {Object} [context]
         *  The calling context for the generator callback function.
         *
         * @returns {Any}
         *  The cached entry for the specified custom key.
         */
        this.getCustom = function (key, generator, context) {
            return cacheMap.getOrCreate('custom:' + key, function () {
                var iterator = cellCollection.createAddressIterator(targetRanges, { type: 'value', covered: true });
                return generator.call(context, iterator);
            });
        };

        /**
         * Invalidates this cache. The next invocation of an accessor method
         * will collect the covered cell values again
         *
         * @returns {CellValueCache}
         *  A reference to this instance.
         */
        this.invalidate = function () {
            cacheMap.clear();
            return this.trigger('invalidate');
        };

        /**
         * Changes the addresses of the cell ranges covered by this instance,
         * and invalidates the cache.
         *
         * @param {RangeArray|Range|Null} [ranges]
         *  The new target ranges for this cache. Can be omitted or set to null
         *  to remove the current target ranges.
         *
         * @returns {CellValueCache}
         *  A reference to this instance.
         */
        this.setRanges = function (ranges) {
            targetRanges = ranges ? RangeArray.get(ranges).clone(true) : new RangeArray();
            return this.invalidate();
        };

        // initialization -----------------------------------------------------

        // create a clone of the passed cell range addresses
        targetRanges = targetRanges ? RangeArray.get(targetRanges).clone(true) : new RangeArray();

        // clear the cached subtotals when the values of the covered cells have changed
        this.listenTo(cellCollection, 'change:cells', function (event, changeDesc) {

            // nothing to do, if the cache is empty (do not waste time to evaluate the passed addresses),
            // or if no cell values have been changed (e.g. formatting only)
            if (targetRanges.empty() || cacheMap.empty() || changeDesc.valueCells.empty()) { return; }

            // check if the cells covered by this formatting model have changed their values (ignore formatting)
            if (changeDesc.getValueRanges().overlaps(targetRanges)) {
                self.invalidate();
            }
        });

        // invalidate the cache, if cells are moved around
        this.listenTo(cellCollection, 'move:cells', function (event, moveDesc) {
            if (targetRanges.overlaps(moveDesc.dirtyRange)) {
                self.invalidate();
            }
        });

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = sheetModel = cellCollection = targetRanges = cacheMap = null;
        });

    } }); // CellValueCache

    // exports ================================================================

    return CellValueCache;

});
