/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/sortmixin', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/dialogs',
    'gettext!io.ox/office/spreadsheet/main'
], function (SheetUtils, Dialogs, gt) {

    'use strict';

    // convenience shortcuts
    var SortDirection = SheetUtils.SortDirection;

    // mix-in class SortMixin =================================================

    /**
     * Implementations of all controller items for sorting selected ranges,
     * intended to be mixed into a document controller instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the current sheet
     *  selection.
     */
    function SortMixin(docView) {

        // self reference (the document controller)
        var self = this;

        // the model and collections of the active sheet
        var sheetModel = null;
        var colCollection = null;
        var rowCollection = null;
        var cellCollection = null;
        var mergeCollection = null;
        var tableCollection = null;

        // sort order to be used for next "toggle" request
        var nextDescending = false;

        // private methods ----------------------------------------------------

        /**
         * Returns the table range located at the active cell of the selection.
         *
         * @returns {TableModel|Null}
         *  The model of the table range that is covered by the active cell,
         *  otherwise null.
         */
        function getSelectedTableModel() {
            var tableModel = tableCollection.findTable(docView.getActiveCell());
            return (tableModel && !tableModel.isAutoFilter()) ? tableModel : null;
        }

        /**
         * Returns whether sorting is currently available according to the
         * selection in the active sheet.
         *
         * @returns {Boolean}
         *  Whether sorting is currently available.
         */
        function isSortEnabled() {

            // only one single range can be sorted
            var selectedRanges = docView.getSelectedRanges();
            if (selectedRanges.length !== 1) { return false; }
            var selectedRange = selectedRanges[0];

            // if the selection is completely inside a table range, it will be sorted
            var tableModel = getSelectedTableModel();
            if (tableModel) { return tableModel.getRange().contains(selectedRange); }

            // otherwise, the selection must not overlap with any table range
            return !tableCollection.overlapsTables(selectedRange);
        }

        /**
         * Checks the size of the passed sorting range.
         */
        function hasValidSortRangeSize(sortRange, sortDir) {

            // check the maximum supported sort size (columns)
            var vertical = sortDir === SortDirection.VERTICAL;
            var cols = vertical ? sortRange.cols() : colCollection.countVisibleIndexes(sortRange.colInterval());
            var maxCols = vertical ? SheetUtils.MAX_SORT_LINE_LENGTH : SheetUtils.MAX_SORT_LINES_COUNT;
            if (cols > maxCols) { return false; }

            // check the maximum supported sort size (rows)
            var horizontal = sortDir === SortDirection.HORIZONTAL;
            var rows = horizontal ? sortRange.rows() : rowCollection.countVisibleIndexes(sortRange.rowInterval());
            var maxRows = horizontal ? SheetUtils.MAX_SORT_LINE_LENGTH : SheetUtils.MAX_SORT_LINES_COUNT;
            if (rows > maxRows) { return false; }

            return true;
        }

        /**
         * Checks the passed sorting range for validity. Rejects the returned
         * promise, if the range is too large, or if it contains or overlaps
         * with matrix formulas or merged cell ranges.
         *
         * @returns
         *  A promise that will fulfil if the passed cell range can be used for
         *  sorting.
         */
        function checkSortRange(sortRange) {

            // check that the sort range does not contain matrix formulas
            return docView.ensureUnlockedRanges(sortRange, { lockMatrixes: 'full' }).then(function () {

                // check the maximum supported sort size
                if (!hasValidSortRangeSize(sortRange)) {
                    return SheetUtils.makeRejected('sort:overflow');
                }

                // bug 56313: check for merged ranges before generating the operations
                if (mergeCollection.coversAnyMergedRange(sortRange)) {
                    return SheetUtils.makeRejected('sort:merge:overlap');
                }

                return sortRange;
            });
        }

        /**
         * Returns the address of the effective cell range to be sorted. For
         * table range, returns its entire data range. For all other cell
         * selections, asks whether to expand the range to the entire content
         * range if necessary.
         *
         * @returns
         *  A promise that will fulfil with an object with the properties
         *  "sortRange" (instance of class Range, cell range to be sorted, with
         *  header cells), "tableModel" (instance of class TableModel, or
         *  null), and "hasHeader" (boolean value, either from table model or
         *  detected from sorting range contents).
         */
        function getSortRange() {
            // this function will be called from controller item setters, therefore the view
            // contains a valid single cell range selection according to isSortEnabled()

            // resolve the table range containing the cell selection
            var tableModel = getSelectedTableModel();

            // the selected range to be sorted
            var sortRange = tableModel ? tableModel.getDataRange() : docView.getActiveRange();
            if (!sortRange) { return SheetUtils.makeRejected('sort:blank'); }

            // selection must not be blank (nothing to sort)
            if (!tableModel && cellCollection.areRangesBlank(sortRange)) {
                return SheetUtils.makeRejected('sort:blank');
            }

            // check validity of the sort range (no matrix formulas, not oversized, etc.)
            var promise = checkSortRange(sortRange);

            // try to expand the sorting range (promise will return the new sorting range)
            promise = promise.then(function () {

                // resolve the promise with the table range; or with a 2D range (without expansion)
                if (tableModel || (!sortRange.singleCol() && !sortRange.singleRow())) {
                    return sortRange;
                }

                // column/row vector, or single cell: try to expand to the enclosing content range
                var contentRange = cellCollection.findContentRange(sortRange);
                if (contentRange.equals(sortRange)) { return sortRange; }

                // column/row vector: ask whether to expand (always expand from single cell selection)
                var promise2 = sortRange.single() ? self.createResolvedPromise(true) :
                    docView.showQueryDialog(gt('Sorting'), gt('Expand the selection?'), { cancel: true });

                // dialog answered with "Yes": expand sort range to content range
                return promise2.then(function (response) {
                    return response ? checkSortRange(contentRange) : sortRange;
                });
            });

            // resolve the promise with the effective sort range and table model
            return promise.then(function (effSortRange) {
                docView.selectRange(effSortRange, { active: docView.getActiveCell() });
                var hasHeader = tableModel ? tableModel.hasHeaderRow() : cellCollection.hasHeaderRow(effSortRange);
                return { sortRange: effSortRange, tableModel: tableModel, hasHeader: hasHeader };
            });
        }

        /**
         * Returns the identifier of the sort order to be applied for the next
         * "cell/sort" controller command.
         *
         * @returns {string}
         *  The next sort order (either "ascending", or "descending").
         */
        function getNextSortOrder() {
            return nextDescending ? 'descending' : 'ascending';
        }

        /**
         * Sorts the specified cell range according to the sorting rules and
         * other settings.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the selected cell range has
         *  been sorted.
         */
        function sortCellRange(sortRange, tableModel, sortDir, sortRules, hasHeader) {

            // check the maximum supported sort size according to sorting direction
            var validSize = hasValidSortRangeSize(sortRange, sortDir);
            var promise = validSize ? self.createResolvedPromise() : SheetUtils.makeRejected('sort:overflow');

            // generate and apply the operations to sort a table range or a custom cell range
            promise = promise.then(function () {
                return sheetModel.createAndApplyOperations(function (generator) {

                    // sort table range via its model
                    if (tableModel) {
                        return tableModel.generateSortOperations(generator, sortRules);
                    }

                    // reduce sort range by header row/column
                    if (hasHeader) {
                        sortRange = sortRange.clone();
                        sortRange.start.move(1, sortDir === SortDirection.HORIZONTAL);
                    }

                    // generate sort operations via cell collection
                    return cellCollection.generateSortOperations(generator, sortRange, sortDir, sortRules);

                }, { storeSelection: true });
            });

            return promise;
        }

        /**
         * Sorts the selected cell range immediately (without showing the
         * custom sort dialog) by the column containing the active cell.
         *
         * @param {string} sortOrder
         *  The order to be used for sorting. Possible values are "ascending",
         *  "descending", and "toggle" (use result of getNextSortOrder()).
         *
         * @returns {Promise<void>}
         *  A promise that will be resolved after the selected cell range has
         *  been sorted.
         */
        function sortBySingleColumn(sortOrder) {

            // the effective sorting order (before calling getSortRange() which resets "nextDescending")
            var descending = (sortOrder === 'toggle') ? nextDescending : (sortOrder === 'descending');
            // the sort rule to be used (always sort in the column of the active cell)
            var sortRule = { index: docView.getActiveCell()[0], descending: descending };

            // resolve the effective sorting range, and the involved table model
            var promise = getSortRange();

            // sort the cell range by the column containing the active cell
            promise = promise.then(function (settings) {
                return sortCellRange(settings.sortRange, settings.tableModel, SortDirection.VERTICAL, [sortRule], settings.hasHeader);
            });

            // update cached sorting order for toggle mode
            promise = promise.then(function () { nextDescending = !descending; });

            // notify user about any problems
            return docView.yellOnFailure(promise);
        }

        /**
         * Shows the custom sorting dialog box, and sorts the selected cell
         * range according to the result of the dialog.
         *
         * @returns {Promise<void>}
         *  A promise that will be resolved after the selected cell range has
         *  been sorted.
         */
        function showSortDialog() {

            // resolve the effective sorting range, and the involved table model
            var promise = getSortRange();

            // show the custom sort dialog box, sort the cell range according to the result of the dialog
            promise = promise.then(function (settings) {
                var dialog = new Dialogs.CustomSortDialog(docView, settings.sortRange, settings.tableModel, settings.hasHeader);
                return dialog.show().then(function (result) {
                    return sortCellRange(settings.sortRange, settings.tableModel, result.sortDir, result.sortRules, result.hasHeader);
                });
            });

            // reset cached sorting order for toggle mode
            promise = promise.then(function () { nextDescending = false; });

            // notify user about any problems
            return docView.yellOnFailure(promise);
        }

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // active sheet must be unlocked (regardless of locked state of selection)
            'cell/sort/enabled': {
                parent: 'sheet/operation/unlocked/cell',
                enable: isSortEnabled
            },

            'cell/sort': {
                parent: 'cell/sort/enabled',
                get: getNextSortOrder,
                set: sortBySingleColumn
            },

            'cell/sort/dialog': {
                parent: 'cell/sort/enabled',
                set: showSortDialog,
                preserveFocus: true
            }
        });

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function () {
            sheetModel = docView.getSheetModel();
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            cellCollection = sheetModel.getCellCollection();
            mergeCollection = sheetModel.getMergeCollection();
            tableCollection = sheetModel.getTableCollection();
        });

        // reset the cached sort order for toggling, if the selection changes
        this.listenTo(docView, 'change:selection', function () {
            nextDescending = false;
            self.update();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = null;
            sheetModel = cellCollection = colCollection = rowCollection = null;
            mergeCollection = tableCollection = null;
        });

    } // class SortMixin

    // exports ================================================================

    return SortMixin;

});
