/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */

/* eslint no-new: 0 */

define('io.ox/office/presenter/actions', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/actions',
    'io.ox/core/extPatterns/links',
    'gettext!io.ox/office/presenter/main'
], function (Ext, ActionsPattern, LinksPattern, gt) {

    'use strict';

    var TOOLBAR_ID          = 'io.ox/office/presenter/toolbar';
    var TOOLBAR_LINKS_ID    = TOOLBAR_ID + '/links';
    var PRESENTER_ACTION_ID = 'io.ox/office/presenter/actions';

    var Action              = ActionsPattern.Action;
    var ActionLink          = LinksPattern.ActionLink;

    // helper function for local presentation ---------------------------------

    function canStartLocal(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app        = baton.context.app;
        var localModel = app.localModel;
        var userId     = app.rtConnection.getRTUuid();

        return (localModel.canStart(userId));
    }

    function isPresenterLocal(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app        = baton.context.app;
        var localModel = app.localModel;
        var userId     = app.rtConnection.getRTUuid();

        return (localModel.isPresenter(userId));
    }

    function canPauseLocal(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app        = baton.context.app;
        var localModel = app.localModel;
        var userId     = app.rtConnection.getRTUuid();

        return (localModel.canPause(userId));
    }

    function canContinueLocal(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app        = baton.context.app;
        var localModel = app.localModel;
        var userId     = app.rtConnection.getRTUuid();

        return (localModel.canContinue(userId));
    }

    // helper function for remote presentation --------------------------------

    function canStartRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app       = baton.context.app;
        var encrypted = app.fileModel.isEncrypted();
        var rtModel   = app.rtModel;
        var userId    = app.rtConnection.getRTUuid();

        return (!encrypted && rtModel.canStart(userId));
    }

    function isPresenterRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app     = baton.context.app;
        var rtModel = app.rtModel;
        var userId  = app.rtConnection.getRTUuid();

        return (rtModel.isPresenter(userId));
    }

    function hasPresenterRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app     = baton.context.app;
        var rtModel = app.rtModel;

        return (rtModel.hasPresenter());
    }

    function isJoinedRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app     = baton.context.app;
        var rtModel = app.rtModel;
        var userId  = app.rtConnection.getRTUuid();

        return (rtModel.isJoined(userId));
    }

    function canPauseRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app     = baton.context.app;
        var rtModel = app.rtModel;
        var userId  = app.rtConnection.getRTUuid();

        return (rtModel.canPause(userId));
    }

    function isPausedRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var rtModel = baton.context.app.rtModel;

        return (rtModel.isPaused());
    }

    function canContinueRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app     = baton.context.app;
        var rtModel = app.rtModel;
        var userId  = app.rtConnection.getRTUuid();

        return (rtModel.canContinue(userId));
    }

    function canJoinRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app     = baton.context.app;
        var rtModel = app.rtModel;
        var userId  = app.rtConnection.getRTUuid();

        return (rtModel.canJoin(userId));
    }

    function canLeaveRemote(baton) {
        if (!baton.context || !baton.context.app) { return false; }

        var app     = baton.context.app;
        var rtModel = app.rtModel;
        var userId  = app.rtConnection.getRTUuid();

        return (rtModel.canLeave(userId));
    }

    // start presentation drop-down
    new ActionLink(TOOLBAR_LINKS_ID + '/dropdown/start-presentation', {
        index: 100,
        id: 'startlocal',
        //#. 'start presentation' dropdown menu entry to start a local only presentation where no remote participants would be able to join.
        label: gt('Start local presentation'),
        description: gt('View the presentation on your device.'),
        ref: PRESENTER_ACTION_ID + '/start/local'
    });

    new ActionLink(TOOLBAR_LINKS_ID + '/dropdown/start-presentation', {
        index: 200,
        id: 'startremote',
        //#. 'start presentation' dropdown menu entry to start a remote presentation where remote participants would be able to join.
        label: gt('Start remote presentation'),
        description: gt('Broadcast your presentation over the Web.'),
        ref: PRESENTER_ACTION_ID + '/start/remote'
    });

    new Action(PRESENTER_ACTION_ID + '/start/local', {
        requires: function (e) {
            return (canStartLocal(e.baton) && !isPresenterRemote(e.baton) && !isJoinedRemote(e.baton));
        },
        action: function (baton) {
            var app         = baton.context.app;
            var localModel  = app.localModel;
            var userId      = app.rtConnection.getRTUuid();

            localModel.startPresentation(userId);
        }
    });

    new Action(PRESENTER_ACTION_ID + '/start/remote', {
        // starting a remote presentation requires the 'remote_presenter' capability
        capabilities: 'remote_presenter',
        requires: function (e) {
            return (!isPresenterLocal(e.baton) && canStartRemote(e.baton));
        },
        action: function (baton) {
            var app     = baton.context.app;
            var slideId = app.mainView.getActiveSlideIndex();

            app.rtConnection.startPresentation({ activeSlide: slideId });
        }
    });

    new Action(PRESENTER_ACTION_ID + '/end', {
        requires: function (e) {
            return (isPresenterLocal(e.baton) || isPresenterRemote(e.baton));
        },
        action: function (baton) {
            var app         = baton.context.app;
            var rtModel     = app.rtModel;
            var localModel  = app.localModel;
            var userId      = app.rtConnection.getRTUuid();

            if (localModel.isPresenter(userId)) {
                localModel.endPresentation(userId);

            } else if (rtModel.isPresenter(userId)) {
                app.rtConnection.endPresentation();
            }
        }
    });

    new Action(PRESENTER_ACTION_ID + '/pause', {
        requires: function (e) {
            return (canPauseLocal(e.baton) || canPauseRemote(e.baton));
        },
        action: function (baton) {
            var app         = baton.context.app;
            var rtModel     = app.rtModel;
            var localModel  = app.localModel;
            var userId      = app.rtConnection.getRTUuid();

            if (localModel.canPause(userId)) {
                localModel.pausePresentation(userId);

            } else if (rtModel.canPause(userId)) {
                app.rtConnection.pausePresentation();
                app.mainView.toggleFullscreen(false);
            }
        }
    });

    new Action(PRESENTER_ACTION_ID + '/continue', {
        requires: function (e) {
            return (canContinueLocal(e.baton) || canContinueRemote(e.baton));
        },
        action: function (baton) {
            var app         = baton.context.app;
            var rtModel = app.rtModel;
            var localModel = app.localModel;
            var userId = app.rtConnection.getRTUuid();

            if (localModel.canContinue(userId)) {
                localModel.continuePresentation(userId);

            } else if (rtModel.canContinue(userId)) {
                baton.context.app.rtConnection.continuePresentation();
            }
        }
    });

    new Action(PRESENTER_ACTION_ID + '/join', {
        requires: function (e) {
            return (!isPresenterLocal(e.baton) && canJoinRemote(e.baton));
        },
        action: function (baton) {
            baton.context.app.mainView.joinPresentation();
        }
    });

    new Action(PRESENTER_ACTION_ID + '/leave', {
        requires: function (e) {
            return (canLeaveRemote(e.baton));
        },
        action: function (baton) {
            baton.context.app.rtConnection.leavePresentation();
        }
    });

    new Action(PRESENTER_ACTION_ID + '/fullscreen', {
        requires: function (e) {
            // iOS doesn't support full-screen
            if (!e.baton.context || _.device('iOS')) { return false; }

            return (isPresenterLocal(e.baton) || isPresenterRemote(e.baton) || isJoinedRemote(e.baton));
        },
        action: function (baton) {
            baton.context.app.mainView.toggleFullscreen();
        }
    });

    new Action(PRESENTER_ACTION_ID + '/togglesidebar', {
        requires: function (e) {
            return (!isPresenterLocal(e.baton) && hasPresenterRemote(e.baton) && (!isPresenterRemote(e.baton) || isPausedRemote(e.baton)));
        },
        action: function (baton) {
            baton.context.app.mainView.onToggleSidebar();
        }
    });

    new Action(PRESENTER_ACTION_ID + '/zoomin', {
        requires: function (e) {
            return !isPresenterRemote(e.baton);
        },
        action: function (baton) {
            baton.context.app.mainView.presenterEvents.trigger('presenter:zoomin');
        }
    });

    new Action(PRESENTER_ACTION_ID + '/zoomout', {
        requires: function (e) {
            return !isPresenterRemote(e.baton);
        },
        action: function (baton) {
            baton.context.app.mainView.presenterEvents.trigger('presenter:zoomout');
        }
    });

    new Action(PRESENTER_ACTION_ID + '/close', {
        requires: function (e) {
            return _.isObject(e.baton.context);
        },
        action: function (baton) {
            baton.context.app.mainView.presenterEvents.trigger('presenter:close');
        }
    });
});
