/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/view/view', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/view/popup/attributestooltip',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/view/view',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/textframework/view/dialogs',
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/presentation/view/dialog/slidebackgrounddialog',
    'io.ox/office/presentation/view/popup/presentationcontextmenu',
    'io.ox/office/presentation/view/controls',
    'io.ox/office/presentation/view/toolbars',
    'io.ox/office/presentation/view/slidepreviewmixin',
    'io.ox/office/presentation/view/slidepane',
    'io.ox/office/presentation/view/statuspane',
    'gettext!io.ox/office/presentation/main',
    'less!io.ox/office/presentation/view/style'
], function (Utils, Forms, DrawingFrame, AttributeUtils, AttributesToolTip, Config, Position, DOM, TextBaseView, Labels, Dialogs, PresentationUtils, SlideBackgroundDialog, PresentationContextMenu, Controls, ToolBars, SlidePreviewMixin, SlidePane, StatusPane, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CheckBox = Controls.CheckBox;
    var TextField = Controls.TextField;
    var RadioGroup = Controls.RadioGroup;
    var CompoundButton = Controls.CompoundButton;

    // predefined zoom factors
    var ZOOM_FACTORS = Utils.SMALL_DEVICE ? [100, 150, 200] : [35, 50, 75, 100, 150, 200, 300, 400, 600, 800];

    // class PresentationView =================================================

    /**
     * The presentation view.
     *
     * Triggers the events supported by the base class EditView, and the
     * following additional events:
     * - 'change:zoom': When the zoom level for the displayed document was changed.
     *
     * @constructor
     *
     * @extends TextBaseView
     *
     * @param {TextApplication} app
     *  The application containing this view instance.
     *
     * @param {TextModel} docModel
     *  The document model created by the passed application.
     */
    var PresentationView = TextBaseView.extend({ constructor: function (app, docModel) {

        var // self reference
            self = this,

            // the root node of the entire application pane
            appPaneNode = null,

            // the scrollable document content node
            contentRootNode = null,

            // the app content node (child of contentRootNode)
            appContentNode = null,

            // the page node
            pageNode = null,

            //page content node
            pageContentNode = null,

            // the 'slide pane' reference
            slidepane = null,

            // scroll position of the application pane
            scrollPosition = { left: 0, top: 0 },

            // the vertical position of the application pane
            verticalScrollPosition = 0,

            // debug operations for replay
            replayOperations = null,

            // maximum osn for debug operations for replay
            replayOperationsOSN = null,

            // current zoom type (percentage or keyword)
            zoomType = (Utils.COMPACT_DEVICE) ? 'slide' : 100,

            // the current effective zoom factor, in percent
            zoomFactor = 100,

            // outer width of page node, used for zoom calculations
            pageNodeWidth = 0,

            // outer height of page node, used for zoom calculations
            pageNodeHeight = 0,

            // the generic content menu for all types of contents
            contextMenu = null,

            // whether the vertical scrolling happens with pressed mouse button (on the scroll bar)
            isMouseButtonPressedScrolling = false,

            // whether the slide changed during scrolling with pressed mouse button. If yes, the
            // slide position needs to be adapted when mouse button is released.
            slideChanged = false,

            // whether the vertical scroll position was really modified, when mouse button was pressed.
            // This marker is required to distinguish this scrolling from moving of drawings.
            verticalScrollChanged = false,

            // the handler for the vertical scroll bar events
            refreshVerticalScrollBar = $.noop,

            // the handler for leaving the vertical scrolling (on the scroll bar)
            stopScrollBarScrolling = $.noop,

            // the debounced handler for refresh:layout events
            refreshLayoutDebounced = $.noop,

            // whether there is a scroll event triggered by scrolling into selection
            scrollingToPageRectangle = false,

            // storage for last know touch position
            // (to open context menu on the correct position)
            touchX = null, touchY = null,

            // A virtual focus to indicate whether the slidepane or the document
            // has the focus. The browser focus can't be used for this because of
            // clashing with the clipboard.This is needed for keyboard handling and shortcuts
            slidepaneHasFocus = false,

            initOptions = {
                initHandler: initHandler,                 // modified by `SlidePreviewMixin` via function composition.
                initDebugHandler: initDebugHandler,
                initGuiHandler: initGuiHandler,
                initDebugGuiHandler: initDebugGuiHandler,
                grabFocusHandler: grabFocusHandler,
                contentScrollable: true,
                contentMargin: (Utils.SMALL_DEVICE ? 0 : 30),
                enablePageSettings: true,
                enablePresenter: true,
                showOwnCollaboratorColor: true
            };

        // base constructor ---------------------------------------------------

        TextBaseView.call(this, app, docModel, initOptions);

        // private methods ----------------------------------------------------

        function handleSlideSizeChange(/*evt, data*/) {
            self.setZoomType('slide');
        }

        /**
         * Handler for slide changes.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.newSlideId='']
         *      The id of the standard slide that will be activated.
         *  @param {String} [options.newLayoutId='']
         *      The id of the layout slide that will be activated.
         *  @param {String} [options.newMasterId='']
         *      The id of the master slide that will be activated.
         *  @param {String} [options.oldSlideId='']
         *      The id of the standard slide that will be deactivated.
         *  @param {String} [options.oldLayoutId='']
         *      The id of the layout slide that will be deactivated.
         *  @param {String} [options.oldMasterId='']
         *      The id of the master slide that will be deactivated.
         *  @param {Boolean} [options.isMasterChange=false]
         *      Whether only the master slide for a layout slide was changed. In this case the
         *      'options.newLayoutId' is not set, because this is also possible for inactive
         *      slides.
         */
        function handleSlideChange(event, options) {

            var // the id of the standard slide that will be activated
                slideId = Utils.getStringOption(options, 'newSlideId', ''),
                // the id of the layout slide that will be activated
                layoutId = Utils.getStringOption(options, 'newLayoutId', ''),
                // the id of the master slide that will be activated
                masterId = Utils.getStringOption(options, 'newMasterId', ''),
                // the id of the standard slide that will be deactivated
                oldSlideId = Utils.getStringOption(options, 'oldSlideId', ''),
                // the id of the layout slide that will be deactivated
                oldLayoutId = Utils.getStringOption(options, 'oldLayoutId', ''),
                // the id of the master slide that will be deactivated
                oldMasterId = Utils.getStringOption(options, 'oldMasterId', ''),
                // whether drawings layout slide shall be visible
                showLayoutSlide = slideId ? docModel.isFollowMasterShapeSlide(slideId) : true,
                // whether drawings of master slide shall be visible (can only be visible, if showLayoutSlide is true)
                showMasterSlide = showLayoutSlide ? (layoutId ? docModel.isFollowMasterShapeSlide(layoutId) : true) : false,
                // whether a (not active) layout slide got a new master slide assigned (multi selection in master view side pane)
                // -> in this case the layout ID is not defined, but the master must stay not selectable.
                isMasterChange = Utils.getBooleanOption(options, 'isMasterChange', false);

            // deactivating the old slide (together with master and layout slide)
            if (oldSlideId) { docModel.getSlideById(oldSlideId).addClass('invisibleslide'); }
            if (oldLayoutId) { docModel.getSlideById(oldLayoutId).addClass('invisibleslide'); }
            if (oldMasterId) { docModel.getSlideById(oldMasterId).addClass('invisibleslide'); }

            // activating one slide again (together with master and layout slide)
            if (slideId) { docModel.getSlideById(slideId).removeClass('invisibleslide'); }
            if (layoutId) { docModel.getSlideById(layoutId).removeClass('invisibleslide'); }
            if (masterId) { docModel.getSlideById(masterId).removeClass('invisibleslide'); }

            // making layout and master slide non-selectable, if required
            if (layoutId) { docModel.getSlideById(layoutId).toggleClass('notselectable', slideId !== ''); }
            if (masterId) { docModel.getSlideById(masterId).toggleClass('notselectable', layoutId !== '' || isMasterChange); }

            // taking care of empty place holder drawings that need border and template text
            handleEmptyPlaceHolderVisibility(event, slideId || layoutId || masterId);

            // handling the visibility of the drawings of the layout and the master slide
            // -> taking care, that a layout slide (slideId is empty), does not hide itself.
            handleLayoutMasterVisibility(slideId, layoutId, masterId, showLayoutSlide, showMasterSlide);

            // handling the visibility of the slide backgrounds of slide, layout and master slide
            handleSlideBackgroundVisibility(slideId, layoutId, masterId);
        }

        /**
         * Handler for the slide background visibility - background union from master, layout and standard slides
         * @param  {String} slideId
         * @param  {String} layoutId
         * @param  {String} masterId
         * @param  {String} [previewFillAttrs] optional
         */
        function handleSlideBackgroundVisibility(slideId, layoutId, masterId, previewFillAttrs) {
            var hideBkg;

            if (docModel.isActiveSlideId(slideId)) {
                hideBkg = previewFillAttrs ? ((previewFillAttrs.type && (previewFillAttrs.type !== 'none')) || false) : docModel.isSlideWithExplicitBackground(slideId);

                if (layoutId) {
                    docModel.getSlideById(layoutId).toggleClass('invisiblebackground', hideBkg);

                    if (!hideBkg) { // if standard slide has no bg set, check if layout has bg, to hide master
                        hideBkg = docModel.isSlideWithExplicitBackground(layoutId);
                    }
                }
                if (masterId) {
                    docModel.getSlideById(masterId).toggleClass('invisiblebackground', hideBkg);
                }
            } else if (docModel.isActiveSlideId(layoutId)) {
                hideBkg = previewFillAttrs ? ((previewFillAttrs.type && (previewFillAttrs.type !== 'none')) || false) : docModel.isSlideWithExplicitBackground(layoutId);

                docModel.getSlideById(layoutId).removeClass('invisiblebackground');
                if (masterId) { docModel.getSlideById(masterId).toggleClass('invisiblebackground', hideBkg); }
            } else if (docModel.isActiveSlideId(masterId)) {
                docModel.getSlideById(masterId).removeClass('invisiblebackground');
            }
        }

        /**
         * Handler for all slide attribute changes.
         */
        function handleSlideAttributeChange(event, dataObj) {
            // build an inheritance chain of ids, to check if changed slide is related to any of them
            function checkSlideBackgoundVisibility() {
                var activeSlideId = docModel.getActiveSlideId();
                var idChain;
                var slideId, layoutId, masterId;

                if (docModel.isMasterSlideId(activeSlideId)) {
                    slideId = '';
                    layoutId = '';
                    masterId = activeSlideId;
                } else if (docModel.isLayoutSlideId(activeSlideId)) {
                    slideId = '';
                    layoutId = activeSlideId;
                    masterId = docModel.getMasterSlideId(layoutId);
                } else {
                    slideId = activeSlideId;
                    layoutId = docModel.getLayoutSlideId(slideId);
                    masterId = docModel.getMasterSlideId(layoutId);
                }
                idChain = [slideId, layoutId, masterId];

                if (_.isString(dataObj.slideId) && idChain.indexOf(dataObj.slideId) > -1) {
                    self.handleSlideBackgroundVisibility(activeSlideId);
                }
            }
            // updating a possible change of attribute 'followMasterPage', and possible slide background attribute
            if (docModel.isImportFinished()) {
                updateFollowMasterPageChange();
                checkSlideBackgoundVisibility();
            }
        }

        /**
         * After changes of slide attributes, it is required to react to changes
         * of 'followMasterShapes' attributes of slides.
         */
        function updateFollowMasterPageChange() {

            var // the id of the active slide
                slideId = docModel.getActiveSlideId(),
                // the id of the layout slide
                layoutId = null,
                // the id of the master slide
                masterId = null,
                // whether drawings of layout slide shall be visible
                showLayoutSlide = true,
                // whether drawings of master slide shall be visible
                showMasterSlide = true;

            if (!slideId) { return; } // this can happen during loading

            if (docModel.isStandardSlideId(slideId)) {
                layoutId = docModel.getLayoutSlideId(slideId);
                showLayoutSlide = docModel.isFollowMasterShapeSlide(slideId);
                if (layoutId) {
                    masterId = docModel.getMasterSlideId(layoutId);
                    // master slide can only be visible, if layout slide is visible, too
                    showMasterSlide = showLayoutSlide ? docModel.isFollowMasterShapeSlide(layoutId) : false;
                }
            } else if (docModel.isLayoutSlideId(slideId)) {
                masterId = docModel.getMasterSlideId(slideId);
                showMasterSlide = slideId ? docModel.isFollowMasterShapeSlide(slideId) : true;
            }

            // handling the visibility of the drawings of the layout and the master slide
            handleLayoutMasterVisibility(slideId, layoutId, masterId, showLayoutSlide, showMasterSlide);
        }

        /**
         * Handler for a removal of the active slide. In this it might be necessary, to hide also
         * the underlying master and layout slides.
         *
         * Info: The options object supports the properties 'id', 'layoutId' and 'masterId'. This
         *       is generated by the public model class function 'getSlideIdSet'.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.id='']
         *      The id of the standard slide that is removed.
         *  @param {String} [options.layoutId='']
         *      The id of the layout slide that will be removed or needs to be made invisible.
         *  @param {String} [options.masterId='']
         *      The id of the layout slide that will be removed or needs to be made invisible.
         */
        function hideActiveSlideCompletely(event, options) {

            var // the id of the standard slide that will be activated
                slideId = Utils.getStringOption(options, 'id', ''),
                // the id of the layout slide that will be activated
                layoutId = Utils.getStringOption(options, 'layoutId', ''),
                // the id of the master slide that will be activated
                masterId = Utils.getStringOption(options, 'masterId', '');

            // deactivating the old slide (together with master and layout slide)
            if (slideId && docModel.getSlideById(slideId)) { docModel.getSlideById(slideId).addClass('invisibleslide'); }
            if (layoutId && docModel.getSlideById(layoutId)) { docModel.getSlideById(layoutId).addClass('invisibleslide'); }
            if (masterId && docModel.getSlideById(masterId)) { docModel.getSlideById(masterId).addClass('invisibleslide'); }

            handleLayoutMasterVisibility(slideId, layoutId, masterId, false, false);
        }

        /**
         * Helper function to handle the visibility of the layout and the master slide. This is
         * required, if the slide attribute 'followMasterShapes' is modified. It is important, that
         * not the complete slide gets 'display: none' but only the drawing children. Otherwise
         * the background would also vanish. But that would not be the correct behavior.
         *
         * @param {String} [layoutId]
         *  The id of the layout slide.
         *
         * @param {String} [masterId]
         *  The id of the master slide.
         *
         * @param {Boolean} showLayoutSlide
         *  Whether the specified layout slide shall be made visible or not.
         *
         * @param {Boolean} showMasterSlide
         *  Whether the specified master slide shall be made visible or not.
         */
        function handleLayoutMasterVisibility(slideId, layoutId, masterId, showLayoutSlide, showMasterSlide) {

            var // the layout slide
                layoutSlide = null;

            if (layoutId) { docModel.getSlideById(layoutId).toggleClass('invisibledrawings', !showLayoutSlide); }
            if (masterId) { docModel.getSlideById(masterId).toggleClass('invisibledrawings', !showMasterSlide); }

            // in ODP format the visibility of the footer slides can be set
            // -> the footer place holder drawings of the layout slide must become visible on document slides,
            //    although they are place holder drawings. Therefore the class 'forceplaceholdervisibility' to
            //    overwrite all existing rules for hiding place holder drawings.
            if (app.isODF()) {
                if (docModel.isMasterView() && layoutId) {
                    layoutSlide = docModel.getSlideById(layoutId);
                    layoutSlide.children('[placeholdertype=sldNum]').toggleClass('forceplaceholdervisibility', false);
                    layoutSlide.children('[placeholdertype=ftr]').toggleClass('forceplaceholdervisibility', false);
                    layoutSlide.children('[placeholdertype=dt]').toggleClass('forceplaceholdervisibility', false);
                } else if (!docModel.isMasterView() && layoutId && slideId) {
                    layoutSlide = docModel.getSlideById(layoutId);
                    layoutSlide.children('[placeholdertype=sldNum]').toggleClass('forceplaceholdervisibility', docModel.isSlideNumberDrawingSlide(slideId));
                    layoutSlide.children('[placeholdertype=ftr]').toggleClass('forceplaceholdervisibility', docModel.isFooterDrawingSlide(slideId));
                    layoutSlide.children('[placeholdertype=dt]').toggleClass('forceplaceholdervisibility', docModel.isDateDrawingSlide(slideId));
                }
            }
        }

        /**
         * Handling the visibility of empty place holder drawings, before a slide is made visible. If
         * it was inserted before, it might be possible that the class 'emptyplaceholdercheck' is still
         * set. This functionality just before showing the slide is especially important for remote clients,
         * where it is otherwise not possible to detect, when this handling of empty place holder needs
         * to be done.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {String} id
         *  The slide ID.
         */
        function handleEmptyPlaceHolderVisibility(event, id) {

            var // the jQuerified slide node
                slide = null;

            if (id && event.type === 'change:slide:before') {

                slide = docModel.getSlideById(id);

                if (slide.hasClass('emptyplaceholdercheck')) {
                    docModel.getDrawingStyles().handleAllEmptyPlaceHolderDrawingsOnSlide(slide);
                    slide.removeClass('emptyplaceholdercheck');
                }
            }
        }

        /**
         * Getting the larger of the two values 'content root node height' and the 'height of a slide' (in pixel).
         *
         * @returns {Number}
         *  The larger of the two values 'content root node height' and the 'height of a slide' (in pixel).
         */
        function getScrollStepHeight() {
            return Math.max(Utils.round((pageNode.outerHeight() * zoomFactor) / 100, 1), contentRootNode.height());
        }

        /**
         * Handling the change of the active view. It is necessary to set the height
         * of the app content node, so that the vertical scrollbar can be adapted.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.sliceCount=1]
         *      The number of slides in the activated view.
         */
        function handleActiveViewChange(event, options) {

            var // the number of slides in the activated view
                slideCount = Utils.getNumberOption(options, 'slideCount', 1);

            // setting the height of the app content node, so that the vertical scroll bar is adapted.
            appContentNode.height(Utils.round(getScrollStepHeight() * slideCount, 1));
        }

        /**
         * Hander for the event 'update:verticalscrollbar'. This is triggered
         * after a change of the active slide.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.pos=0]
         *      The index of the slide inside its container.
         *  @param {Boolean} [options.keepScrollPosition=false]
         *      Whether the current scroll position shall not be modified, although the vertical scroll bar
         *      was modified. This is the case if a drawing is moved above a slide.
         */
        function handleVerticalScrollbarUpdate(event, options) {

            var // the index of the slide in its container
                index = Utils.getNumberOption(options, 'pos', 0),
                // the old margin-top caused by the slide position
                oldMarginTop = pageNode.data('scrollTopMargin') || 0,
                // the new margin-top caused by the slide position
                newMarginTop = Utils.getElementCssLength(pageNode, 'margin-top') - oldMarginTop + (getScrollStepHeight() * index),
                // whether the scroll position shall not be modified (this is the case if a drawing is moved above a slide)
                keepScrollPosition = Utils.getBooleanOption(options, 'keepScrollPosition', false),
                // a collector for all visible drawings on the slide.
                // The top margin must be as large as the distance to the upmost drawing on the slide (47692)
                // -> this can also be a drawing on a layout or master slide!
                allDrawings = app.getModel().getAllVisibleDrawingsOnSlide(),
                // the maximum top distance of a drawing above the slide to the slide
                maxTopDrawingDistance = 0;

            if (isDrawingAboveSlide(allDrawings)) { // check, if at least one drawing is above the slide
                maxTopDrawingDistance = getDrawingDistanceAboveSlide(allDrawings);
                // increasing the top margin because of the drawing above the slide
                if (maxTopDrawingDistance > 0) { newMarginTop += maxTopDrawingDistance; }
            }

            pageNode.css('margin-top', newMarginTop + 'px');
            pageNode.data('scrollTopMargin', newMarginTop);

            if (!keepScrollPosition && !isMouseButtonPressedScrolling) {
                contentRootNode.scrollTop(newMarginTop); // setting the vertical scroll bar position
                verticalScrollPosition = Utils.round(newMarginTop, 1);
            }

            // saving that a slide change happened when scrolling with pressed mouse button
            if (isMouseButtonPressedScrolling) { slideChanged = true; }
        }

        /**
         * Moves the browser focus focus to the editor root node.
         */
        function grabFocusHandler(options) {
            if (Utils.TOUCHDEVICE) {
                var changeEvent = Utils.getOption(options, 'changeEvent', { sourceEventType: 'click' });
                var softKeyboardOpen = Utils.isSoftKeyboardOpen();
                var sourceType = changeEvent.sourceEventType;
                var node = docModel.getNode();

                if (sourceType === 'tap' && !softKeyboardOpen) {
                    Utils.focusHiddenNode();
                } else {
                    Utils.trigger('change:focus', node, node, null);
                    Utils.setFocus(node);
                }
            } else {
                Utils.setFocus(docModel.getNode());
            }
        }

        /**
         * Caches the current scroll position of the application pane.
         */
        function saveScrollPosition() {
            scrollPosition = { top: contentRootNode.scrollTop(), left: contentRootNode.scrollLeft() };
        }

        /**
         * Restores the scroll position of the application pane according to
         * the cached scroll position.
         */
        function restoreScrollPosition() {
            contentRootNode.scrollTop(scrollPosition.top).scrollLeft(scrollPosition.left);
        }

        /**
         * Scrolls the application pane to the focus position of the selection.
         * If this is a slide selection, there is no need to scroll.
         */
        function scrollToSelection() {

            // the selection object
            var selection = docModel.getSelection();
            // the boundaries of the text cursor
            var boundRect = selection.isSlideSelection() ? null : selection.getFocusPositionBoundRect();

            // make cursor position visible in application pane
            if (boundRect) {
                self.scrollToPageRectangle(boundRect);
            }
        }

        /**
         * Helper function that checks, if there is at least one drawing, that is above the active slide. This
         * function is used for the vertical scroll bar calculations.
         *
         * @param {jQuery} [allVisibleDrawings]
         *  A container with all visible drawings on the slide. If it is not specified, it is calculated within
         *  this function.
         *
         * @returns {Boolean}
         *  Whether there is at least one visible drawing, whose upper border is above the active slide.
         */
        function isDrawingAboveSlide(allVisibleDrawings) {

            var // the drawings on the slide
                allDrawings = allVisibleDrawings || docModel.getAllVisibleDrawingsOnSlide(),
                // the top offset of the page node
                pageNodeTopOffset = pageNode.offset().top,
                // a drawing that is positioned above the slide
                drawingAboveSlide = _.find(allDrawings, function (drawing) {
                    return (drawing.getBoundingClientRect().top < pageNodeTopOffset);
                });

            return !_.isUndefined(drawingAboveSlide);
        }

        /**
         * Helper function that checks, if there is at least one drawing, that is below the active slide. This
         * function is used for scrolling of the last slide. If there is one drawing at least 1mm below the last
         * slide, it must not be scrolled upwards to the upper border of the slide (47514).
         *
         * @param {Number} pageHeight
         *  The effective height of the visible page in px. This includes already the zoom level.
         *
         * @param {jQuery} [allVisibleDrawings]
         *  A container with all visible drawings on the slide. If it is not specified, it is calculated within
         *  this function.
         *
         * @returns {Boolean}
         *  Whether there is at least one visible drawing, whose upper border is above the active slide.
         */
        function isDrawingBelowSlide(pageHeight, allVisibleDrawings) {

            var // the visible drawings on the slide
                allDrawings = allVisibleDrawings || docModel.getAllVisibleDrawingsOnSlide(),
                // the top offset of the page node
                pageNodeTopOffset = pageNode.offset().top,
                // a drawing that is not yet visible in scroll direction
                drawingBelowSlide;

            // checking the bottom positions of drawings on the slide
            drawingBelowSlide = _.find(allDrawings, function (drawing) {
                return (drawing.getBoundingClientRect().bottom > (pageNodeTopOffset + pageHeight));
            });

            return !_.isUndefined(drawingBelowSlide);
        }

        /**
         * Calculating the distance of the most top drawing on the active slide to the slide itself for all
         * drawings, whose upper border is above the slide. If no drawing is above the slide, 0 is returned.
         *
         * @param {jQuery} [allVisibleDrawings]
         *  A container with all visible drawings on the slide. If it is not specified, it is calculated within
         *  this function.
         *
         * @returns {Number}
         *  The distance in px of the most top drawing above the slide to the slide itself. If no drawing is
         *  located above the slide, 0 is returned.
         */
        function getDrawingDistanceAboveSlide(allVisibleDrawings) {

            var // the drawings on the slide
                allDrawings = allVisibleDrawings || docModel.getAllVisibleDrawingsOnSlide(),
                // the top offset of the page node
                pageNodeTopOffset = pageNode.offset().top,
                // the maximum distance of a visible drawing above the slide
                maxTopDistance = 0;

            // iterating over all visible drawings on the slide
            _.each(allDrawings, function (drawing) {
                var distance = pageNodeTopOffset - drawing.getBoundingClientRect().top;
                if (distance > maxTopDistance) {
                    maxTopDistance = distance;
                }
            });

            return maxTopDistance;
        }

        /**
         * Handler for scroll events at the content root node.
         */
        function scrollInContentRootNode() {

            var // the current scrollPosition
                scrollY = contentRootNode.scrollTop(),
                // the scroll direction
                downwards = scrollY > verticalScrollPosition,
                // the additional top margin that is required because of the scroll bar
                marginTop = 0,
                // the height of the visible page
                realPageHeight = 0,
                // whether the top border of the page is visible
                topBorderVisible = false,
                // whether the top border of the page is visible
                bottomBorderVisible = false,
                // the height of the content root node
                contentRootNodeHeight = 0,
                // the top offset of the content root node
                contentRootNodeTopOffset = 0,
                // the top offset of the page node
                pageNodeTopOffset = 0,
                // if drawing is currently selected
                selectedDrawing = null,
                // whether the slide is completely visible (top and bottom border)
                fullSlideVisible = false,
                // the zoom factor divided by 100
                localZoomFactor = zoomFactor / 100,
                // the selection object
                selection = docModel.getSelection(),
                // a collector for all visible drawings on the slide (should be filled as late as possible (performance))
                allVisibleDrawings = null,
                // allow visibility of the border before switchting to other slide (only if slide is not completely visible)
                offset = 50;

            /**
             * helper function that checks, if there is at least one drawing, that is only partly visible
             * in the scroll direction. In this case, it must not be switched to the neighboring slide.
             */
            function allDrawingsVisible(allDrawings) {

                var // a drawing that is not yet visible in scroll direction
                    partlyVisibleDrawing;

                // Info: Using an optional offset, that is necessary to access for example rotation selector, if drawing is outside the slide.
                // -> but trying to avoid usage of offset, because slides must not flicker, if they are completely visible.

                if (downwards) {
                    // checking the bottom positions of drawings on the slide
                    partlyVisibleDrawing = _.find(allDrawings, function (drawing) {
                        var drawingBottomPos = drawing.getBoundingClientRect().bottom,
                            useOffset = (drawingBottomPos > (pageNodeTopOffset + realPageHeight)); // drawing is outside the slide
                        return ((drawingBottomPos + (useOffset ? offset : 0)) > (contentRootNodeTopOffset + contentRootNodeHeight));
                    });
                } else {
                    // checking the top positions of drawings on the slide
                    partlyVisibleDrawing = _.find(allDrawings, function (drawing) {
                        var drawingTopPos = drawing.getBoundingClientRect().top,
                            useOffset = (drawingTopPos < pageNodeTopOffset); // drawing is outside the slide
                        return ((drawingTopPos - (useOffset ? (2 * offset) : 0)) < contentRootNodeTopOffset);  // twice the offset to enable rotation
                    });
                }

                return _.isUndefined(partlyVisibleDrawing);
            }

            // do nothing in Safari, if a popup-container is open (61423)
            if (_.browser.Safari && appContentNode.children('.popup-container').length > 0) { return; }

            // no change of slide, if vertical scroll position did not change or did change only one pixel or less.
            // -> allowing 1px difference increases the resilience of the vertical scrolling, because this difference
            //    might be caused by rounding errors. This can lead for example to an upward slide change directly
            //    following a downward slide change making a slide change via scroll wheel impossible (doc 44961).
            if (Math.abs(scrollY - verticalScrollPosition) <= 1) { return; }

            verticalScrollPosition = scrollY; // saving the value for the next scroll event. This must be done before any of the following 'return' statements!

            // scroll event triggered by selection of a drawing -> never change slide
            if (scrollingToPageRectangle) {
                scrollingToPageRectangle = false;
                return;
            }

            // setting a marker, if the vertical scrollbar was really modified with pressed mouse button
            if (isMouseButtonPressedScrolling) {
                // never handle selection box or drawing moves within this function
                if (docModel.getSelectionBox() && docModel.getSelectionBox().isSelectionBoxActive()) { return; }
                verticalScrollChanged = true;
            }

            // don't scroll if modifying of drawing or group of drawings is currently active - early exit on move, resize, rotate
            if (selection.isMultiSelection()) {
                selectedDrawing = selection.getFirstSelectedDrawingNode();
            } else if (selection.isAnyDrawingSelection()) {
                selectedDrawing = selection.getAnyDrawingSelection();
            }

            if (selectedDrawing && DrawingFrame.isModifyingDrawingActive(selectedDrawing)) { return; }

            // calculating change of vertical scroll bar
            realPageHeight = Utils.round(pageNode.outerHeight() * localZoomFactor, 1);
            pageNodeTopOffset = pageNode.offset().top;
            contentRootNodeHeight = contentRootNode.height();
            contentRootNodeTopOffset = contentRootNode.offset().top;

            topBorderVisible = pageNodeTopOffset > contentRootNodeTopOffset;
            bottomBorderVisible = (pageNodeTopOffset + realPageHeight) < (contentRootNodeTopOffset + contentRootNodeHeight);

            fullSlideVisible = topBorderVisible && bottomBorderVisible; // not using offset, slides must not move up or downwards

            // switching to the next or previous slide
            if (downwards) {
                if (bottomBorderVisible) {

                    // avoid using the offset to make the bottom border visible, if the slide is completely visible
                    if (!fullSlideVisible && ((contentRootNodeTopOffset + contentRootNodeHeight - (pageNodeTopOffset + realPageHeight)) < offset)) { return; }

                    allVisibleDrawings = docModel.getAllVisibleDrawingsOnSlide(); // filling the container with visible drawings
                    if (!allDrawingsVisible(allVisibleDrawings)) { return; }

                    if (!docModel.changeToNeighbourSlide()) {
                        // restoring the previous position (scroll down on last slide)
                        if ((contentRootNodeHeight > realPageHeight) && !isDrawingBelowSlide(realPageHeight, allVisibleDrawings)) {
                            // only jump upwards, if top and bottom border is visible and no drawing is below the last slide (47514)
                            marginTop = pageNode.data('scrollTopMargin') || 0;
                            contentRootNode.scrollTop(marginTop); // setting the vertical scroll bar position
                            verticalScrollPosition = Utils.round(marginTop, 1);
                        } else {
                            contentRootNode.scrollTop(verticalScrollPosition);
                        }
                    }
                } else {
                    // saving vertical scroll position (required for the last slide)
                    verticalScrollPosition = contentRootNode.scrollTop();
                }
            } else if (!downwards && topBorderVisible) {
                allVisibleDrawings = docModel.getAllVisibleDrawingsOnSlide();
                if (!allDrawingsVisible(allVisibleDrawings)) { return; }
                docModel.changeToNeighbourSlide({ next: false });
            }

        }

        /**
         * Performance: Register in the selection, whether the current operation is an insertText
         * operation. In this case the debounced 'scrollToSelection' function can use the saved
         * DOM node instead of using Position.getDOMPosition in 'getPointForTextPosition'. This
         * makes the calculation of scrolling requirements faster.
         */
        function registerInsertText(options) {
            docModel.getSelection().registerInsertText(options);

            //fixing the problem with an upcoming soft-keyboard
            if (Utils.TOUCHDEVICE) { scrollToSelection(); }
        }

        /**
         * Handles the 'contextmenu' event for the complete browser.
         */
        function globalContextMenuHandler(event) {
            if (_.isUndefined(event)) { return false; }

            var // was the event triggered by keyboard?
                isGlobalTrigger = Utils.isContextEventTriggeredByKeyboard(event),
                // node, on which the new event should be triggered
                triggerNode     = null,
                // search inside this node for the allowed targets
                windowNode      = app.getWindowNode(),
                // parent elements on which 'contextmenu' is allowed (inside the app node)
                winTargetAreas  = ['.commentlayer', '.app-content', '.app-content-root', '.page', '.textdrawinglayer', '.popup-content'],
                // global target areas
                globalTargetAreas  = ['.popup-container > .popup-content'],
                // was one of these targets clicked
                targetClicked   = false,
                // the slide pane container as a jQuery object
                slidePaneContainer = self.getSlidePane().getSlidePaneContainer(),
                // the selection object
                selection = docModel.getSelection();

            // set virtual focus to the document when the the context menu was not invoked in the slide pane and not by keyboard (e.g. right mouse click in the slide)
            if (!isGlobalTrigger && Utils.containsNode(slidePaneContainer, event.target, { allowEqual: true }) === false) { self.setVirtualFocusSlidepane(false); }

            /* context menu detection for the slide pane */

            // Bugfix for IE 11 only (can be deleted otherwise): In IE 11 the focus is the clicked element,
            // while in all other browsers it's the element with a tab index (here '.slide-pane-container').
            // So we need to set the focus to the '.slide-pane-container' when a child element from it was clicked.
            if (_.browser.IE < 12) {
                if (Utils.containsNode(self.getSlidePane().getNode(), event.target) === true) {
                    Utils.setFocus(slidePaneContainer);
                }
            }

            // when the focus is on the slidepane, trigger the contextmenu on the slide pane, when the following cases are NOT fulfilled:
            //if (Utils.isElementNode(Utils.getActiveElement(), '.slide-pane-container') || self.hasSlidepaneVirtualFocus()) {
            if (self.hasSlidepaneVirtualFocus()) {

                // On COMPACT_DEVICES a different method in the slidepane.js is used to trigger the context menu.
                // That means taphold is not used on the slide pane to trigger the context menu, because this interaction is used for drag and drop on COMPACT_DEVICES.
                if (Utils.COMPACT_DEVICE) {
                    return false;
                }

                // // when event.target is NOT inside the slide pane and when it's also NOT triggered by keyboard, don't open it // TODO check comment and if
                // if (Utils.containsNode(slidePaneContainer, event.target, { allowEqual: true }) === false && !isGlobalTrigger) {
                //     return false;
                // }

                // trigger an event on the slide pane. that invokes a context menu on the activeSlide. when it's triggered by keyboard open the menu at the side
                slidePaneContainer.trigger(new $.Event('documents:contextmenu', { sourceEvent: event, stickToSlide: isGlobalTrigger }));
                // we have a detected a context menu for the slide pane, therefore return false here
                return false;
            }

            /* context menu detection for the windowNode */

            // prevent context menu on key to open again
            if (Utils.isElementNode(Utils.getActiveElement(), '.popup-content')) {
                return false;
            }

            // check if the event target is inside of one of the allowed containers
            _.each(winTargetAreas, function (selector) {
                var parentNode = windowNode.find(selector);

                if (parentNode.length > 0) {
                    if (Utils.containsNode(parentNode, event.target) === true || Utils.isElementNode(event.target, selector)) {
                        targetClicked = true;
                    }
                }
            });
            _.each(globalTargetAreas, function (selector) {
                var parentNode = $('body > ' + selector);

                if (parentNode.length > 0) {
                    if (Utils.containsNode(parentNode, event.target) === true || Utils.isElementNode(event.target, selector)) {
                        targetClicked = true;
                    }
                }
            });

            // get out here, if no allowed target was clicked, or the global trigger
            // was fired (via keyboard)
            if (!targetClicked && !isGlobalTrigger) { return false; }

            // if a drawing is selected
            if (docModel.isDrawingSelected()) {
                if ($(event.target).is('.tracker')) { return false; }
                // get the (first) selected drawing as trigger node
                triggerNode = (selection.isMultiSelectionSupported() && selection.isMultiSelection()) ? selection.getFirstSelectedDrawingNode() : $(selection.getSelectedDrawing()[0]);
            } else if (selection.isSlideSelection()) {
                if (_.browser.IE < 12 && !selection.hasRange() && $(event.target).is('div.p:not(.slide)') && (_.isNull(triggerNode) || $(triggerNode.is('.clipboard')))) {
                    // bug 57416: select drawing only when no range selected
                    triggerNode = $(event.target).closest(DrawingFrame.NODE_SELECTOR);
                    if (triggerNode.length > 0) { selection.selectOneSpecificDrawing(triggerNode); } // 45355
                } else {
                    triggerNode = docModel.getNode();
                }
            } else {
                var windowSelection = window.getSelection();

                // get the current focus node to trigger on it
                _.each($('.popup-content'), function (popupContentNode) {
                    if (Utils.containsNode(popupContentNode, event.target)) { triggerNode = $(event.target); }
                });

                if (_.browser.IE < 12 && !selection.hasRange() && $(event.target).is('div.p:not(.slide), div.p:not(.slide) > span') && (_.isNull(triggerNode) || $(triggerNode.is('.clipboard')))) {
                    // bug 57416: select drawing only when no range selected
                    triggerNode = $(event.target).closest(DrawingFrame.NODE_SELECTOR);
                    if (triggerNode.length > 0) { selection.selectOneSpecificDrawing(triggerNode, { textSelection: $(event.target).is('span') }); } // 45355
                }

                if (_.isNull(triggerNode) && windowSelection.type !== 'None') {
                    triggerNode = $(window.getSelection().focusNode.parentNode);
                }

                if (_.isNull(triggerNode)) {
                    Utils.warn('TextView.globalContextMenuHandler(): No "triggerNode" found. Contextmenu will not be open!');
                    return;
                }
            }

            // if we have a last known touch position, use it
            if (touchX && touchY) {
                event.pageX = touchX;
                event.pageY = touchY;

            // in case of global trigger (via keyboard), or x/y are both "0"
            // locate a meaningful position
            } else if (isGlobalTrigger || (event.pageX === 0 && event.pageY === 0) || (_.isUndefined(event.pageX) && _.isUndefined(event.pageY))) {
                // find the position of the trigger node
                var xy = Position.getPixelPositionToRootNodeOffset($('html'), triggerNode);
                if (isGlobalTrigger && selection.isSlideSelection()) {
                    event.pageX = xy.x;
                    event.pageY = xy.y;
                    event.target = triggerNode.parent();
                } else {
                    // and set pageX/pageY manually
                    event.pageX = (xy.x + (triggerNode.width() / 2));
                    event.pageY = (xy.y + (triggerNode.height() / 2));
                }

            }

            // trigger our own contextmennu event on the found trigger node
            triggerNode.trigger(new $.Event('documents:contextmenu', { sourceEvent: event }));

            return false;
        }

        /**
         * Helper function to set the correct cursor in overlapping drawings. If the top level
         * drawing is not responsible for the mouse move event at the event position (for example
         * a line or a circle without filling), it might be possible, that an underlying drawing
         * has text input at the event position, so that the cursor must be the text cursor. If
         * no underlying drawing is responsible, the pointer for the slide selection must be
         * activated (docs-977).
         */
        function getDebouncedMouseMoveHandler() {

            // the last event node
            var lastEvent = null;
            // the last modified drawing node
            var lastDrawingNode = null;

            // saving the last mouse move event in direct callback
            function saveMouseMoveEvent(event) {
                lastEvent = event;
            }

            // deferred callback to determine the cursor type
            function checkCursorAtPosition() {

                // clean up old forced cursor types
                if (lastDrawingNode) {
                    lastDrawingNode.removeClass('forcepointercursor forcetextcursor forcecopycursor');
                    lastDrawingNode = null;
                }

                if (lastEvent && app.isEditable()) {

                    var targetNode = $(lastEvent.target);

                    if (targetNode.is('.content') || targetNode.is('.canvasexpansion')) {

                        var drawingNode = targetNode.parent();

                        if (drawingNode.length > 0) {
                            if (!PresentationUtils.checkEventPosition(app, lastEvent, drawingNode)) {
                                // trying to find a valid drawing node behind the drawing that is target of the mouse move event
                                var drawingNodeObj = PresentationUtils.findValidDrawingNodeBehindDrawing(app, lastEvent, drawingNode);

                                if (drawingNodeObj && drawingNodeObj.textFrameNode) {
                                    // drawing with text frame found to handle event
                                    lastDrawingNode = drawingNode;
                                    drawingNode.addClass('forcetextcursor');
                                } else if (drawingNodeObj && drawingNodeObj.drawing) {
                                    // drawing found to handle this event
                                    // -> typically do nothing, because move cursor is already active,
                                    //    but if Ctrl-Key is pressed, the copy cursor needs to be activated (for an underlying drawing)
                                    if (docModel.getNode().hasClass('drag-to-copy')) { // ctrl-key is pressed
                                        lastDrawingNode = drawingNode;
                                        drawingNode.addClass('forcecopycursor');
                                    }
                                } else {
                                    // no drawing found to handle event
                                    lastDrawingNode = drawingNode;
                                    drawingNode.addClass('forcepointercursor');
                                }
                            } else if (docModel.getNode().hasClass('drag-to-copy')) { // ctrl-key is pressed
                                lastDrawingNode = drawingNode;
                                drawingNode.addClass('forcecopycursor');
                            }
                        }
                    }
                }

                lastEvent = null;
                return;
            }

            // create and return the deferred method
            return self.createDebouncedMethod('PresentationModel.getDebouncedMouseMoveHandler', saveMouseMoveEvent, checkCursorAtPosition, { delay: 50 });
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            // deferred and debounced scroll handler, registering insertText operations immediately
            var scrollToSelectionDebounced = self.createDebouncedMethod('PresentationView.scrollToSelectionDebounced', registerInsertText, scrollToSelection, { delay: 50, maxDelay: 500 });

            // register the global event handler for 'contextmenu' event
            if (Utils.COMPACT_DEVICE) {
                //app.registerGlobalEventHandler($(document), 'taphold', globalContextMenuHandler);
                app.registerGlobalEventHandler($('body'), 'contextmenu', function (evt) {
                    if (evt && /^(INPUT|TEXTAREA)/.test(evt.target.tagName)) { return true; }
                    if (evt) { evt.preventDefault(); }
                    return false;
                });
            } else {
                app.registerGlobalEventHandler($('body'), 'contextmenu', globalContextMenuHandler);
            }

            // to save the position from 'touchstart', we have to register some global eventhandler
            // we need this position to open the contextmenu on the correct location
            if (Utils.IOS || _.browser.Android) {
                app.registerGlobalEventHandler($('body'), 'touchstart', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = event.originalEvent.changedTouches[0].pageX;
                    touchY = event.originalEvent.changedTouches[0].pageY;
                });
                app.registerGlobalEventHandler($('body'), 'touchend', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = touchY = null;
                });
            }

            // initialize other instance fields
            appPaneNode = self.getAppPaneNode();
            contentRootNode = self.getContentRootNode();
            appContentNode = self.getAppContentNode();

            // add the SlidePane
            self.addPane(slidepane = new SlidePane(self));

            // Add slidepane keyhandler to the clipboard too. This is needed,
            // because the browser focus must be partly in the clipboard for copy&paste.
            // Without this handler shortcuts for the slidepane would not work.
            docModel.getSelection().getClipboardNode().on({
                keydown: slidepane.getSlidepaneKeyDownHandler()
            });

            // insert the editor root node into the application pane
            self.insertContentNode(docModel.getNode());

            // Page node, used for zooming
            pageNode = docModel.getNode();

            //page content node, used for zooming on small devices
            pageContentNode = $(pageNode).children('div.pagecontent');

            // handle scroll position when hiding/showing the entire application
            self.listenTo(app.getWindow(), {
                beforehide: function () {
                    saveScrollPosition();
                    slidepane.saveSlidePaneScrollPosition();
                    appPaneNode.css('opacity', 0);
                },
                show: function () {
                    restoreScrollPosition();
                    slidepane.restoreSlidePaneScrollPosition();
                    // restore visibility deferred, needed to restore IE scroll position first
                    self.executeDelayed(function () { appPaneNode.css('opacity', 1); }, 'PresentationView.initHandler');
                }
            });

            // handle selection change events
            self.listenTo(docModel, 'selection', function (event, selection, options) {
                // disable internal browser edit handles (browser may re-enable them at any time)
                self.disableBrowserEditControls();
                // scroll to text cursor (but not, if only a remote selection was updated (39904))
                if (!Utils.getBooleanOption(options, 'updatingRemoteSelection', false)) { scrollToSelectionDebounced(options); }
                var slideID = docModel.getActiveSlideId();
                var layoutOrMaster = docModel.isLayoutOrMasterId(slideID);
                var userData = {
                    selections: selection.getRemoteSelectionsObject(),
                    target: layoutOrMaster ? slideID : undefined
                };
                if (docModel.isHeaderFooterEditState()) {
                    userData.targetIndex = docModel.getPageLayout().getTargetsIndexInDocument(docModel.getCurrentRootNode(), docModel.getActiveTarget());
                }
                // send user data to server
                app.updateUserData(userData);
                // exit from crop image mode, if currently active
                if (selection.isCropMode()) { docModel.exitCropMode(); }
            });

            // slide size has changed - e.g. orientation change.
            self.listenTo(docModel, 'slidesize:change', handleSlideSizeChange);

            // disable drag&drop for the content root node to prevent dropping images which are loaded by the browser
            self.listenTo(self.getContentRootNode(), 'drop dragstart dragenter dragexit dragover dragleave', false);
            // Handle switch of slide inside standard view or master/layout view
            self.listenTo(docModel, 'change:slide:before', handleSlideChange);
            // Handle the removal of the active slide
            self.listenTo(docModel, 'removed:activeslide', hideActiveSlideCompletely);
            // Handle the change of a layout slide for a document slide
            self.listenTo(docModel, 'change:layoutslide', handleSlideChange);
            // Handle switch of followMasterShapes attribute
            self.listenTo(docModel, 'change:slideAttributes', handleSlideAttributeChange);

            // Registering a debounced hander function for setting correct cursor in underlying drawings
            self.listenTo(self.getContentRootNode(), 'mousemove', getDebouncedMouseMoveHandler());

            // store the values of page width and height with paddings, after loading is finished
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();
        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            var // pending operations
                pendingOperations = [],
                // delayed recording operations active
                delayedRecordingActive = false,
                // checking the operation length, so that operations removed from the
                // 'optimizeOperationHandler' will not collected twice in the recorder.
                oplCounter = 1;

            // collect operations
            function collectOperations(event, operations, external) {
                if (docModel.isRecordingOperations()) {
                    pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                        return { operation: operation, external: external };
                    }));

                    if (!delayedRecordingActive) {
                        recordOperations();
                    }
                }
            }

            // record operations
            function recordOperations() {

                if (delayedRecordingActive) { return; }

                delayedRecordingActive = true;
                self.executeDelayed(function () {
                    var // full JSON string
                        fullJSONString = replayOperations.getFieldValue(),
                        // json part string to be appended
                        jsonPartString = '',
                        // current part to be processed
                        operations = pendingOperations.splice(0, 20),
                        // first entry
                        firstEntry = (fullJSONString.length === 0);

                    _(operations).each(function (entry) {

                        // skipping operations, that were 'optimized' in 'sendActions' in the
                        // optimizeOperationHandler. Otherwise 'merged operations' are still
                        // followed by the operations, that will not be sent to
                        // the server, because they were removed by the 'optimizeOperationHandler'.
                        var skipOperation = false,
                            opl = ('opl' in entry.operation) ? entry.operation.opl : '';

                        // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                        // only happen for internal operations (task 29601)
                        if (!entry.external) {
                            if (_.isNumber(opl) && (opl > 1)) {
                                oplCounter = opl;
                            } else if (oplCounter > 1) {
                                oplCounter--;
                                skipOperation = true;
                            }
                        }

                        if (!skipOperation) {
                            if (!firstEntry) {
                                jsonPartString += ',';
                            }
                            jsonPartString += JSON.stringify(entry.operation);
                            firstEntry = false;
                        }
                    });

                    if (fullJSONString.length >= 2) {
                        if (fullJSONString.charAt(0) === '[') {
                            fullJSONString = fullJSONString.substr(1, fullJSONString.length - 2);
                        }
                    }
                    fullJSONString = '[' + fullJSONString + jsonPartString + ']';
                    replayOperations.setValue(fullJSONString);

                    delayedRecordingActive = false;
                    if (pendingOperations.length > 0) {
                        recordOperations();
                    }
                }, 'PresentationView.recordOperations', 50);
            }

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoHeader('selection', 'Current Selection')
                .addDebugInfoNode('selection', 'type', 'Type of the selection')
                .addDebugInfoNode('selection', 'start', 'Start position')
                .addDebugInfoNode('selection', 'end', 'End position')
                .addDebugInfoNode('selection', 'target', 'Target position')
                .addDebugInfoNode('selection', 'dir', 'Direction')
                .addDebugInfoNode('selection', 'attrs1', 'Explicit attributes of cursor position')
                .addDebugInfoNode('selection', 'attrs2', 'Explicit attributes (parent level)')
                .addDebugInfoNode('selection', 'style', 'Style sheet attributes');

            // log all selection events, and new formatting after operations
            self.listenTo(docModel, 'selection operations:success', function () {

                function debugAttrs(position, ignoreDepth, target) {
                    if (!operationsPane.isVisible()) { return; }

                    var markup = '';

                    position = _.initial(position, ignoreDepth);
                    var element = (position && position.length) ? Position.getDOMPosition(docModel.getCurrentRootNode(), position, true) : null;
                    if (element && element.node) {
                        var attrs = AttributeUtils.getExplicitAttributes(element.node);
                        markup = '<span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>' + Utils.escapeHTML(Utils.stringifyForDebug(attrs)) + '</span>';
                    }

                    operationsPane.setDebugInfoMarkup('selection', target, markup);
                }

                function debugStyle(family, target) {
                    if (!operationsPane.isVisible()) { return; }

                    var markup = '';

                    var styleId = docModel.getAttributes(family).styleId;
                    if (styleId) {
                        var attrs = docModel.getStyleCollection(family).getStyleAttributeSet(styleId);
                        markup = '<span>family=' + family + '</span> <span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>id=' + Utils.escapeHTML(styleId) + '</span>';
                    }

                    operationsPane.setDebugInfoMarkup('selection', target, markup);
                }

                var selection = docModel.getSelection();
                operationsPane
                    .setDebugInfoText('selection', 'type', selection.getSelectionType() + (selection.isMultiSelection() ? ' multi selection (' + selection.getMultiSelection().length + ')' : ''))
                    .setDebugInfoText('selection', 'start', selection.isMultiSelection() ? selection.getListOfPositions() : selection.getStartPosition().join(', '))
                    .setDebugInfoText('selection', 'end', selection.isMultiSelection() ? selection.getListOfPositions({ start: false }) : selection.getEndPosition().join(', '))
                    .setDebugInfoText('selection', 'target', selection.getRootNodeTarget())
                    .setDebugInfoText('selection', 'dir', selection.getDirection());

                debugAttrs(selection.getStartPosition(), 0, 'attrs1');
                debugAttrs(selection.getStartPosition(), 1, 'attrs2');
                debugStyle('paragraph', 'style');
            });

            // operations to replay
            replayOperations = new TextField(self, { tooltip: _.noI18n('Paste operations here'), select: true });

            // operations to replay
            replayOperationsOSN = new TextField(self, { tooltip: _.noI18n('Define OSN for last executed operation (optional)'), width: 50 });

            // record operations after importing the document
            self.waitForImport(function () {
                self.listenTo(docModel, 'operations:after', collectOperations);
            });

            // process debug events triggered by the model
            self.listenTo(docModel, 'debug:clipboard', function (event, content) {
                clipboardPane.setClipboardContent(content);
            });
        }

        /**
         * Debounced handler function for the 'refresh:layout' event.
         */
        function refreshLayoutHandler() {

            var // the remote selection object
                remoteSelection = docModel.getRemoteSelection(),
                // temporary page node width value
                tempPageNodeWidth = pageNode.outerWidth();

            // fix for #33829 - if padding is reduced, dependent variables have to be updated
            pageNodeWidth = tempPageNodeWidth;
            pageNodeHeight = pageNode.outerHeight();

            if (zoomType === 'slide') {
                self.executeDelayed(function () { self.setZoomType('slide'); }, 'PresentationView.refreshLayoutHandler');
            }

            // it might be necessary to update the slide position at the vertical scroll bar
            refreshVerticalScrollBar(null, { keepScrollPosition: true });

            // clean and redraw remote selections
            remoteSelection.renderCollaborativeSelections();
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         *
         * @param {String} tabId
         *  The id of the tool bar tab that is activated. Or the identifier for
         *  generating the tool bar tabs ('toolbartabs') or for the view menu in
         *  the top bar ('viewmenu').
         *
         * @param {CompoundButton} viewMenuGroup
         *  The 'View' drop-down menu group.
         */
        function initGuiHandler(tabId, viewMenuGroup) {

            var insertToolBar           = null,
                insertDrawingBar        = null,
                slideToolBar            = null,
                tableToolBar            = null,
                tableBorderToolBar      = null,
                tableStyleToolBar       = null,
                tablePositionToolBar    = null,
                drawingToolBar          = null,
                drawingAnchorToolBar    = null,
                drawingCropToolBar      = null,
                drawingFillToolBar      = null,
                drawingLineToolBar      = null,
                drawingLineEndsToolBar  = null,
                drawingGroupToolBar     = null,
                drawingOptionsToolBar   = null,
                reviewToolBar           = null,

                pickTableSize           = null,
                pickLanguage            = null,
                insertField             = null,

                btnInsertImage          = null,
                btnInsertHyperlink      = null,
                btnInsertTab            = null,
                btnInsertFooter         = null,

                btnInsertRow            = null,
                btnInsertColumn         = null,
                btnDeleteRow            = null,
                btnDeleteColumn         = null,

                btnSpelling             = null;

            switch (tabId) {

                case 'toolbartabs':
                    // TABS: prepare all tabs (for normal or combined panes)
                    if (!Config.COMBINED_TOOL_PANES) {
                        self.createToolBarTab('format',  { label: Labels.FORMAT_HEADER_LABEL, visibleKey: 'document/editable/text/format',                  priority: 2,  startVisible: true });
                        self.createToolBarTab('insert',  { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable',                              priority: 2,  startVisible: true });
                        self.createToolBarTab('slide',   { label: Labels.SLIDE_HEADER_LABEL,  visibleKey: 'document/editable',                              priority: 2,  startVisible: true });
                        self.createToolBarTab('review',  { label: Labels.REVIEW_HEADER_LABEL, visibleKey: 'document/editable',  sticky: true,               priority: 20, startVisible: true });
                        self.createToolBarTab('table',   { label: Labels.TABLE_HEADER_LABEL,  visibleKey: 'document/editable/table',                        priority: 1,  startVisible: false });
                        self.createToolBarTab('drawing', { labelKey: 'drawing/type/label',    visibleKey: 'document/editable/anydrawingexceptsingletable',  priority: 1,  startVisible: true });
                    } else {
                        self.createToolBarTab('format',    { label: Labels.FONT_HEADER_LABEL,   visibleKey: 'document/editable/text',                       priority: 2,  startVisible: true });
                        self.createToolBarTab('paragraph', { label: Labels.PARAGRAPH_LABEL,     visibleKey: 'document/editable/text',                       priority: 2,  startVisible: true });
                        self.createToolBarTab('insert',    { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable',                            priority: 2,  startVisible: true });
                        self.createToolBarTab('slide',     { label: Labels.SLIDE_HEADER_LABEL,  visibleKey: 'document/editable',                            priority: 2,  startVisible: true });
                        self.createToolBarTab('table',     { label: Labels.TABLE_HEADER_LABEL,  visibleKey: 'document/editable/table',                      priority: 1,  startVisible: false });
                        self.createToolBarTab('drawing',   { labelKey: 'drawing/type/label',    visibleKey: 'document/editable/anydrawingexceptsingletable', priority: 1,  startVisible: true });
                        if (Config.SPELLING_ENABLED) {
                            self.createToolBarTab('review', { label: Labels.REVIEW_HEADER_LABEL, visibleKey: 'document/editable', priority: 20 });
                        }
                    }

                    self.addPane(new StatusPane(self));

                    // Note: The delay should be higher than resizeWindowHandlerDebounced() in slidepane.js to prevent flickering while resizing with 'fit to width'.
                    refreshLayoutDebounced = self.createDebouncedMethod('PresentationView.refreshLayoutDebounced', null, refreshLayoutHandler, { delay: 100 });
                    self.on('refresh:layout', refreshLayoutDebounced);
                    break;

                case 'viewmenu':
                    // the 'View' drop-down menu
                    viewMenuGroup
                        .addSectionLabel(Labels.ZOOM_LABEL)
                        .addGroup('view/zoom/dec', new Button(self, Labels.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                        .addGroup('view/zoom/inc', new Button(self, Labels.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                        .addGroup('view/zoom', new Controls.PercentageLabel(self), { inline: true })
                        .addGroup('view/zoom/type', new Button(self, { label: Labels.ZOOM_SCREEN_WIDTH_LABEL, value: 'slide' }))

                        .addSectionLabel(gt('Views'))
                        .addGroup('slide/activemasterslide', new CheckBox(self, { label: gt('View master'), tooltip: gt('View master & layout slides') }))

                        .addSectionLabel(Labels.OPTIONS_LABEL)
                        // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                        .addGroup('view/toolbars/show',    new CheckBox(self, Labels.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                        .addGroup('view/toolbars/sidebar', new CheckBox(self, Labels.SHOW_SLIDEBAR_CHECKBOX_OPTIONS))
                        .addGroup('document/users',        new CheckBox(self, Labels.SHOW_COLLABORATORS_CHECKBOX_OPTIONS));
                    break;

                case 'format':
                    self.addToolBar('format', new ToolBars.SlideToolBar(self),        { priority: 1 });
                    self.addToolBar('format', new ToolBars.FontFamilyToolBar(self),   { priority: 1 });
                    self.addToolBar('format', new ToolBars.FontStyleToolBar(self),    { priority: 2 });
                    self.addToolBar('format', new ToolBars.FontColorToolBar(self),    { priority: 3 });
                    self.addToolBar('format', new ToolBars.FormatPainterToolBar(self), { priority: 3 });

                    if (!Config.COMBINED_TOOL_PANES) {
                        self.addToolBar('format', new ToolBars.BoxAlignmentToolBar(self), { priority: 4 });
                        self.addToolBar('format', new ToolBars.ListStyleToolBar(self),    { priority: 5 });
                    }
                    break;

                case 'paragraph':

                    if (Config.COMBINED_TOOL_PANES) {
                        self.addToolBar('paragraph', new ToolBars.BoxAlignmentToolBar(self), { priority: 4 });
                        self.addToolBar('paragraph', new ToolBars.ListStyleToolBar(self),    { priority: 5 });
                    }
                    break;

                case 'insert':

                    self.addToolBar('insert', new ToolBars.SlideToolBar(self), { priority: 1 });
                    insertToolBar           = self.createToolBar('insert');
                    insertDrawingBar        = self.createToolBar('insert',  { priority: 1, shrinkToMenu: { label: gt('Insert'), tooltip: gt('Insert Drawing') } });

                    // controls
                    pickTableSize = new Controls.TableSizePicker(self, { label: /*#. a table in a text document */ gt.pgettext('text-doc', 'Table'), tooltip: /*#. insert a table in a text document */ gt.pgettext('text-doc', 'Insert a table'), maxCols: Config.MAX_TABLE_COLUMNS, maxRows: Config.MAX_TABLE_ROWS, smallerVersion: { css: { width: 50 } } });
                    insertField = new Controls.PresentationDateNumberPicker(self);
                    btnInsertImage = new Controls.ImagePicker(self);
                    btnInsertHyperlink = new Button(self, Labels.HYPERLINK_BUTTON_OPTIONS);
                    btnInsertTab = new Button(self, { icon: 'docs-insert-tab', label: gt('Tab stop'), tooltip: gt('Insert tab stop'), smallerVersion: { hideLabel: true } });
                    btnInsertFooter = new Button(self, Labels.FOOTER_BUTTON_OPTIONS);

                    if (Config.COMBINED_TOOL_PANES) {
                        insertDrawingBar
                            .addGroup('placeholder/insert', new Controls.PlaceholderTypePicker(self))
                            .addSeparator()
                            .addGroup('shape/insert', new Controls.ShapeTypePicker(self))
                            .addSeparator()
                            .addGroup('textframe/insert', new Controls.InsertTextFrameButton(self, { toggle: !docModel.getSlideTouchMode() })) // toggle should be false in SlideTouchMode, otherwise true
                            .addSeparator()
                            .addGroup('table/insert', pickTableSize, { visibleKey: 'table/insert/available' })
                            .addSeparator()
                            .addGroup('image/insert/dialog', btnInsertImage)
                            .addSeparator()
                            .addGroup('character/hyperlink/dialog', btnInsertHyperlink)
                            .addGroup('character/insert/tab', btnInsertTab)
                            .addGroup('document/insertfooter', btnInsertFooter)
                            .addGroup('document/insertfield', insertField);
                    } else {
                        insertToolBar
                            .addGroup('placeholder/insert', new Controls.PlaceholderTypePicker(self))
                            .addSeparator()
                            .addGroup('table/insert', pickTableSize, { visibleKey: 'table/insert/available' })
                            .addSeparator()
                            .addGroup('image/insert/dialog', btnInsertImage)
                            .addSeparator()
                            .addGroup('textframe/insert', new Controls.InsertTextFrameButton(self, { toggle: !docModel.getSlideTouchMode() })) // toggle should be false in SlideTouchMode, otherwise true
                            .addSeparator()
                            .addGroup('shape/insert', new Controls.ShapeTypePicker(self))
                            .addSeparator()
                            .addGroup('character/hyperlink/dialog', btnInsertHyperlink)
                            .addGroup('character/insert/tab', btnInsertTab)
                            .addGroup('document/insertfooter', btnInsertFooter)
                            .addGroup('document/insertfield', insertField);
                    }

                    break;

                case 'slide':

                    if (!Config.COMBINED_TOOL_PANES) { self.addToolBar('slide', new ToolBars.SlideToolBar(self), { priority: 1 }); }
                    slideToolBar = self.createToolBar('slide');

                    // controls
                    slideToolBar
                        .addGroup('slide/duplicateslides',   new Button(self, { icon: 'fa-files-o', label: gt('Duplicate'), tooltip: gt('Duplicate selected slides'), smallerVersion: { hideLabel: true } }))
                        .addGroup('slide/deleteslide',       new Button(self, { icon: 'fa-trash-o', label: gt('Delete'), tooltip: gt('Delete slide'), smallerVersion: { hideLabel: true } }))
                        .addGroup('debug/hiddenslide',       new Button(self, { icon: 'fa-eye-slash', toggle: true, label: gt('Hide'), tooltip: gt('Hide slide'), smallerVersion: { hideLabel: true } }))
                        .addSeparator()
                        .addGroup('slide/setbackground',     new Button(self, { icon: 'docs-slide-background', label: gt('Background'), tooltip: gt('Set slide background'), smallerVersion: { hideLabel: true } }))
                        .addSeparator()
                        .addGroup('slide/masternormalslide', new Controls.SwitchSlideEditModeButton(self));

                    break;

                case 'table':

                    tableToolBar            = self.createToolBar('table');
                    tableBorderToolBar      = self.createToolBar('table', { priority: 1, hideable: true, classes: 'noSeparator' });
                    tableStyleToolBar       = self.createToolBar('table', { priority: 2, hideable: true });
                    tablePositionToolBar    = self.createToolBar('table', { priority: 3, hideable: true });

                    // controls
                    btnInsertRow    = new Button(self, { icon: 'docs-table-insert-row', tooltip: gt('Insert row') });
                    btnInsertColumn = new Button(self, { icon: 'docs-table-insert-column', tooltip: gt('Insert column') });
                    btnDeleteRow    = new Button(self, { icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') });
                    btnDeleteColumn = new Button(self, { icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') });

                    tableToolBar
                        .addGroup('drawing/delete', new Button(self, Labels.DELETE_DRAWING_BUTTON_OPTIONS))
                        .addSeparator()
                        .addGroup('table/insert/row', btnInsertRow)
                        .addGroup('table/delete/row', btnDeleteRow)
                        .addGap()
                        .addGroup('table/insert/column', btnInsertColumn)
                        .addGroup('table/delete/column', btnDeleteColumn);

                    if (!Config.COMBINED_TOOL_PANES) {

                        var pickFillColorTable  = new Controls.ColorPicker(self, null, { icon: 'docs-table-fill-color', tooltip: gt('Cell fill color') }),
                            pickBorderModeTable = new Controls.BorderModePicker(self, { tooltip: Labels.CELL_BORDERS_LABEL, showInsideHor: true, showInsideVert: true }),
                            pickBorderWidth     = new Controls.BorderWidthPicker(self, { tooltip: gt('Cell border width') }),
                            pickTableStyle      = new Controls.TableStylePicker(self, self.getActiveTableFlags.bind(self)),
                            alignmentGroup      = new Controls.TableAlignmentPicker(self);

                        tableToolBar
                            .addSeparator()
                            .addGroup('table/alignment/menu', alignmentGroup)
                            .addSeparator()
                            .addGroup('table/fillcolor', pickFillColorTable)
                            .addGap()
                            .addGroup('table/cellborder', pickBorderModeTable);
                        tableBorderToolBar
                            .addGroup('table/borderwidth', pickBorderWidth);
                        tableStyleToolBar
                            .addGroup('table/stylesheet', pickTableStyle);
                    }

                    tablePositionToolBar.addGroup('drawing/order', new Controls.DrawingArrangementPicker(self));
                    tablePositionToolBar.addGroup('drawing/align', new Controls.DrawingAlignmentPicker(self));

                    break;

                case 'drawing':

                    drawingToolBar          = self.createToolBar('drawing');
                    drawingLineToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/line' });
                    drawingLineEndsToolBar  = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/lineendings' });
                    drawingFillToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/fill', hideable: true });
                    drawingCropToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/singledrawingselection', hideable: true });
                    self.addToolBar('drawing', new ToolBars.FormatPainterToolBar(self), { priority: 1 });
                    drawingAnchorToolBar    = self.createToolBar('drawing', { visibleKey: 'document/editable/anydrawing' });
                    drawingGroupToolBar     = self.createToolBar('drawing', { visibleKey: 'drawing/notsingledrawing', hideable: true });
                    drawingOptionsToolBar   = self.createToolBar('drawing', { visibleKey: 'document/editable/anydrawing' });

                    // controls
                    drawingToolBar.addGroup('shape/insert', new Controls.ShapeTypePicker(self, { label: null }), { visibleKey: 'drawing/type/shape' });
                    drawingToolBar.addGroup('drawing/delete', new Button(self, Labels.DELETE_DRAWING_BUTTON_OPTIONS), { visibleKey: 'drawing/delete' });

                    var radioAutoFit              = new RadioGroup(self),
                        pickBorderPresetStyle     = new Controls.BorderPresetStylePicker(self, { label: Labels.BORDER_STYLE_LABEL }),
                        pickLinePresetStyle       = new Controls.BorderPresetStylePicker(self, { label: Labels.LINE_STYLE_LABEL }),
                        pickArrowPresetStyle      = new Controls.ArrowPresetStylePicker(self),
                        pickBorderColor           = new Controls.BorderColorPicker(self, { label: gt('Border color'), smallerVersion: { hideLabel: true } }),
                        pickLineColor             = new Controls.BorderColorPicker(self, { label: gt('Line color'), smallerVersion: { hideLabel: true } }),
                        pickFillColorDrawing      = new Controls.FillColorPicker(self, { label: gt('Background color'),  icon: 'docs-drawing-fill-color', smallerVersion: { hideLabel: true } }),
                        drawingOrder              = new Controls.DrawingArrangementPicker(self),
                        cBoxToggleLockRatio       = new CheckBox(self, { label: gt('Lock ratio'), boxed: true, tooltip: gt('Lock or unlock the drawing aspect ratio'), ambiguous: true }),
                        drawingAlignment          = new Controls.DrawingAlignmentPicker(self);

                    radioAutoFit.createOptionButton('noautofit',      { label: gt('Off'),           tooltip: gt('Turn automatic resizing of text frames off') });
                    if (!app.isODF()) { radioAutoFit.createOptionButton('autotextheight', { label: gt('Shrink text'),   tooltip: gt('Turn automatic shrink text on overflow on') }); }
                    radioAutoFit.createOptionButton('autofit',        { label: gt('Resize frame'),  tooltip: gt('Turn automatic height resizing of text frames on') });

                    drawingAnchorToolBar
                        .addGroup('drawing/order', drawingOrder)
                        .addGap()
                        .addGroup('drawing/align', drawingAlignment);

                    drawingGroupToolBar
                        .addGroup('drawing/group', new Button(self, { icon: 'fa-object-group', toggle: true, tooltip: Labels.GROUP_DRAWING_TOOLTIP }))
                        .addGroup('drawing/ungroup', new Button(self, { icon: 'fa-object-ungroup', toggle: true, tooltip: Labels.UNGROUP_DRAWING_TOOLTIP }));

                    drawingOptionsToolBar
                        .addGroup(null,
                            new CompoundButton(self, {
                                label: gt('Options'),
                                tooltip: gt('More options'),
                                dropDownVersion: { visible: false }
                            })
                                .addSectionLabel(gt('Autofit'))
                                .addGroup('drawing/textframeautofit', radioAutoFit)
                                .addSectionLabel(gt('Resize'))
                                .addGroup('drawing/lockratio', cBoxToggleLockRatio)
                        );

                    drawingLineToolBar
                        .addGroup('drawing/border/style', pickBorderPresetStyle, { visibleKey: 'drawing/border/style/support' })
                        .addGroup('drawing/border/style', pickLinePresetStyle, { visibleKey: 'drawing/line/style/support' })
                        .addGroup('drawing/border/color', pickBorderColor, { visibleKey: 'drawing/border/style/support' })
                        .addGroup('drawing/border/color', pickLineColor, { visibleKey: 'drawing/line/style/support' });

                    drawingLineEndsToolBar
                        .addGroup('drawing/line/endings', pickArrowPresetStyle);

                    drawingFillToolBar
                        .addGroup('drawing/fill/color', pickFillColorDrawing, { visibleKey: 'drawing/fill/style/support' });

                    drawingCropToolBar
                        .addGroup('drawing/crop', new Controls.ImageCropPosition(self), { visibleKey: 'drawing/crop' });

                    break;

                case 'review':

                    if (!Config.COMBINED_TOOL_PANES) {
                        reviewToolBar = self.createToolBar('review', {
                            priority: 2,
                            visibleKey: 'document/spelling/available',
                            shrinkTomenu: {
                                icon: 'docs-online-spelling',
                                splitKey: 'document/onlinespelling',
                                toggle: true,
                                tooltip: gt('Check spelling'),
                                //#. tooltip for a drop-down menu with functions for reviewing the document (spelling, change tracking)
                                caretTooltip: gt('More review actions')
                            }
                        });
                    } else {
                        reviewToolBar = self.createToolBar('review', { visibleKey: 'document/spelling/available' });
                    }

                    // controls
                    if (Config.SPELLING_ENABLED) {
                        btnSpelling = new Button(self, { icon: 'docs-online-spelling', tooltip: gt('Check spelling permanently'), toggle: true, dropDownVersion: { label: gt('Check spelling') } });
                        pickLanguage = new Controls.LanguagePicker(self, { width: 200 });

                        reviewToolBar
                            .addGroup('document/onlinespelling', btnSpelling)
                            .addSeparator()
                            .addGroup('character/language', pickLanguage);
                    }

                    break;

                case 'file':
                    self.addToFilebar('document/present', new Button(self, { label: gt('Present'), tooltip: gt('Present'), icon: 'fa-play-circle-o', classes: 'link-style', smallerVersion: { hideLabel: true } }));
                    break;

                default:
            }
        }

        /**
         * Additional initialization of debug GUI after importing the document.
         *
         * @param {String} tabId
         *  The id of the tool bar tab that is activated. Or the identifier for generating the tool bar
         *  tabs ('toolbartabs') or for the view menu in the top bar ('viewmenu').
         *
         * @param {CompoundButton} viewMenuGroup
         *  The 'View' drop-down menu group.
         */
        function initDebugGuiHandler(tabId, viewMenuGroup) {

            switch (tabId) {
                case 'viewmenu':
                    viewMenuGroup
                        .addGroup('debug/uselocalstorage',   new Button(self, { icon: 'fa-rocket', label: _.noI18n('Use local storage'), toggle: true }))
                        .addGroup('debug/useFastLoad', new Button(self, { icon: 'fa-fighter-jet', label: _.noI18n('Use fast load'), toggle: true }));
                    break;
                case 'debug':
                    self.createToolBar('debug')
                        .addGroup('document/cut',            new Button(self, { icon: 'fa-cut',   tooltip: _.noI18n('Cut to clipboard') }))
                        .addGroup('document/copy',           new Button(self, { icon: 'fa-copy',  tooltip: _.noI18n('Copy to clipboard') }))
                        .addGroup('document/paste/internal', new Button(self, { icon: 'fa-paste', tooltip: _.noI18n('Paste from clipboard') }));

                    self.createToolBar('debug')
                        .addGroup('debug/recordoperations', new Button(self, { icon: 'fa-dot-circle-o', iconStyle: 'color:red;', toggle: true, tooltip: _.noI18n('Record operations') }))
                        .addGroup('debug/replayoperations', new Button(self, { icon: 'fa-play-circle-o', tooltip: _.noI18n('Replay operations') }))
                        .addGroup(null, replayOperations)
                        .addGroup(null, replayOperationsOSN)
                        .addSeparator()
                        .addGroup('slide/resetbackground',    new Button(self, { icon: 'fa-eraser', label: gt('Reset background'), tooltip: gt('Reset slide background to default value'), smallerVersion: { hideLabel: true } }))
                        .addSeparator()
                        .addGroup('debug/followmastershapes', new Button(self, { icon: 'fa-file-image-o', toggle: true, label: gt('Follow master'), tooltip: gt('Shapes follow master shapes') }));
                    break;
                default:
            }
        }

        /**
         * calculate the optimal document scale for the slide.
         * It considers width and hight so that the whole slide fits to the visible screen.
         */
        function calculateOptimalDocumentScale() {

            var // the slide in presentation
                documentWidth = pageNode.outerWidth(),
                documentHeight = pageNode.outerHeight(),

                parentWidth = appPaneNode.width(),
                parentHeight = appPaneNode.height(),

                // the visible area, more or less...
                appContent = appPaneNode.find('.app-content'),
                scaledWidth = null,
                scaledHeight = null,

                // "3" because of the rounding issues
                outerDocumentMarginX = 3 + Utils.getElementCssLength(appContent, 'margin-left') + Utils.getElementCssLength(appContent, 'margin-right'),
                outerDocumentMarginY = 3 + Utils.getElementCssLength(appContent, 'margin-top') + Utils.getElementCssLength(appContent, 'margin-bottom'),

                // the scale ratio
                ratio;

            if (_.browser.Android) {
                //fix for Bug 46375 ignore height
                parentHeight = 50000;
            }

            //landscape
            if (documentWidth > documentHeight) {

                ratio = ((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMarginX)));
                scaledHeight = (documentHeight * ratio);

                // when image is still not fitting stage - then we need to resize it by smaller dimension
                if (scaledHeight > (parentHeight - outerDocumentMarginY)) {
                    ratio = ((parentHeight * parentHeight) / (documentHeight * (parentHeight + outerDocumentMarginY)));
                }
            //portrait
            } else {

                ratio = ((parentHeight * parentHeight) / (documentHeight * (parentHeight + outerDocumentMarginY)));
                scaledWidth = (documentWidth * ratio);

                // when image is still not fitting stage - then we need to resize it by smaller dimension
                if (scaledWidth > (parentWidth - Utils.SCROLLBAR_WIDTH - 4 - outerDocumentMarginX)) {
                    ratio = ((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMarginX)));
                }
            }

            Utils.log('calculateOptimalDocumentScale', pageNode.outerWidth());
            Utils.log('ratio', ratio);
            return Math.floor(ratio * 100);
        }

        function setAdditiveMargin() {
            if (docModel.getSlideTouchMode()) {
                //extra special code for softkeyboard behavior, we nee more space to scroll to the end of the textfile
                var dispHeight = Math.max(screen.height, screen.width);

                pageNode.css('margin-bottom', (Utils.getElementCssLength(pageNode, 'margin-bottom') + dispHeight / 3) + 'px');

                //workaround for clipped top of the first page (20)
                pageNode.css('margin-top', (Utils.getElementCssLength(pageNode, 'margin-top') + 20) + 'px');
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the position of the passed node in the entire scroll area of
         * the application pane.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element whose dimensions will be calculated. Must be
         *  contained somewhere in the application pane. If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @returns {Object}
         *  An object with numeric attributes representing the position and
         *  size of the passed node relative to the entire scroll area of the
         *  application pane. See method Utils.getChildNodePositionInNode() for
         *  details.
         */
        this.getChildNodePositionInNode = function (node) {
            return Utils.getChildNodePositionInNode(contentRootNode, node);
        };

        /**
         * Scrolls the application pane to make the passed node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element that will be made visible by scrolling the
         *  application pane. Must be contained somewhere in the application
         *  pane. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Object} options
         *  @param {Boolean} [options.forceToTop=false]
         *   Whether the specified node shall be at the top border of the
         *   visible area of the scrolling node.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (node, options) {
            if (Utils.getBooleanOption(options, 'regardSoftkeyboard', false)) {
                //fix for Bug 39979 on Ipad text is sometimes under the keyboard
                if (!Utils.IOS || !Utils.isSoftKeyboardOpen()) { delete options.regardSoftkeyboard; }
            }
            scrollingToPageRectangle = true;
            Utils.scrollToChildNode(contentRootNode, node, _.extend(options || {}, { padding: 15 }));
            return this;
        };

        /**
         * Scrolls the application pane to make the passed document page
         * rectangle visible.
         *
         * @param {Object} pageRect
         *  The page rectangle that will be made visible by scrolling the
         *  application pane. Must provide the properties 'left', 'top',
         *  'width', and 'height' in pixels. The properties 'left' and 'top'
         *  are interpreted relatively to the entire document page.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToPageRectangle = function (pageRect) {
            var options = { padding: 15 };

            scrollingToPageRectangle = true;
            Utils.scrollToPageRectangle(contentRootNode, pageRect, options);

            return this;
        };

        /**
         * Returns the slide pane.
         *
         * @returns {SlidePane}
         *  The slidepane pane of this application.
         */
        this.getSlidePane = function () {
            return slidepane;
        };

        /**
         * Provides the debug operations as string for the debugging feature
         * replay operations.
         *
         * @returns {String}
         *  The content of the debug operations field, or null if debug mode is
         *  not active.
         */
        this.getDebugOperationsContent = function () {
            var // the operations in the required json format
                json = null;

            if (replayOperations) {
                json = replayOperations.getFieldValue();

                try {
                    json = json.length ? JSON.parse(json) : null;
                    if (_.isObject(json) && json.operations) {
                        json = json.operations;
                    }
                    return json;
                } catch (e) {
                    // error handling outside this function
                }
            }
            return null;
        };

        /**
         * Setting the maximum OSN for the operations from the operation player, that will
         * be executed.
         *
         * @returns {Number|Null}
         *  The OSN of the last operation that will be executed from the operation player.
         */
        this.getDebugOperationsOSN = function () {

            // the operations in the required json format
            var number = null;

            if (replayOperationsOSN) {
                number = parseInt(replayOperationsOSN.getFieldValue(), 10);
                if (!Utils.isFiniteNumber(number)) {
                    number = null;
                    replayOperationsOSN.setValue('');
                }
            }

            return number;
        };

        /**
         * Returns the current zoom type.
         *
         * @returns {Number|String}
         *  The current zoom type, either as fixed percentage, or as one of the
         *  keywords 'slide' (slide size is aligned to width and hight of the visible
         *  area), or 'page' (page size is aligned to visible area).
         */
        this.getZoomType = function () {
            return zoomType;
        };

        /**
         * Returns the current effective zoom factor in range between 0.35 and 8.
         *
         * @returns {Number}
         *  The current zoom factor in range [0.35, 8].
         */
        this.getZoomFactor = function () {
            return zoomFactor / 100;
        };

        /**
         * Returns the minimum zoom factor in percent.
         *
         * @returns {Number}
         *  The minimum zoom factor in percent.
         */
        this.getMinZoomFactor = function () {
            return ZOOM_FACTORS[0];
        };

        /**
         * Returns the maximum zoom factor in percent.
         *
         * @returns {Number}
         *  The maximum zoom factor in percent.
         */
        this.getMaxZoomFactor = function () {
            return _.last(ZOOM_FACTORS);
        };

        /**
         * Changes the current zoom settings.
         *
         * @param {Number|String} newZoomType
         *  The new zoom type. Either a fixed percentage, or one of the
         *  keywords 'slide' (slide size is aligned to width and hight of the visible
         *  area), or 'page' (page size is aligned to visible area).
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.setZoomType = function (newZoomType) {
            Utils.log('setZoomType');
            var optimalLevel = calculateOptimalDocumentScale();

            if (zoomType !== newZoomType && _.isNumber(newZoomType)) {
                if (zoomType < newZoomType) {
                    this.increaseZoomLevel(newZoomType);
                } else {
                    this.decreaseZoomLevel(newZoomType);
                }
            } else if (newZoomType === 'slide') {
                if (docModel.isDraftMode()) {
                    optimalLevel = 100;
                }
                if (Math.abs(zoomFactor - optimalLevel) > 1.1) {
                    this.increaseZoomLevel(optimalLevel);
                }
            }
            zoomType = newZoomType;
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and decreases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.decreaseZoomLevel = function (newZoomFactor) {

            var // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoomFactor = Utils.findLast(ZOOM_FACTORS, function (factor) { return factor < zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                // the parent node of the page
                appContentRoot = $(appPaneNode).find('.app-content'),
                // an additional padding that is required to display comment layer, if overflow hidden is set
                commentLayerWidth = 0;

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(prevZoomFactor) ? prevZoomFactor : this.getMinZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if (docModel.isDraftMode()) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                if (zoomInsertion < 100 && docModel.isDraftMode()) {
                    pageContentNode.css({ width: '100%' });
                } else {
                    pageContentNode.css({ width: ((100 / zoomInsertion) * 100) + '%', backgroundColor: '#fff' });
                }
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({ opacity: (zoomInsertion > 100) ? 0.99 : 1 });
                }

                self.recalculateDocumentMargin();

                if (zoomInsertion <= 99) {
                    // there needs to be a padding, so that comments will still be visible
                    if (!docModel.getCommentLayer().isEmpty()) {
                        commentLayerWidth = docModel.getCommentLayer().getAppContentPaddingWidth();
                        appContentRoot.css({ paddingLeft: commentLayerWidth, paddingRight: commentLayerWidth });
                    }
                } else {
                    appContentRoot.css({ paddingLeft: '', paddingRight: '' });
                }

                // Use zoom as soon as all browsers support it.
                // pageNode.css({
                //    'zoom': zoomInsertion / 100
                // });

                zoomType = zoomFactor;
                scrollToSelection();
            }
            self.trigger('change:zoom');
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and increases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @param {Boolean} option.clean
         *  Switching between draft and normal mode requires zoom cleanup
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.increaseZoomLevel = function (newZoomFactor, clean) {

            var // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoomFactor = _(ZOOM_FACTORS).find(function (factor) { return factor > zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                appContentRoot = $(appPaneNode).find('.app-content'),
                // an additional padding that is required to display comment layer, if overflow hidden is set
                commentLayerWidth = 0;

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(nextZoomFactor) ? nextZoomFactor : this.getMaxZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if ((docModel.isDraftMode()) && !clean) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' + (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                pageContentNode.css({ width: ((100 / zoomInsertion) * 100) + '%', backgroundColor: '#fff' });
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({ opacity: (zoomInsertion > 100) ? 0.99 : 1 });
                }

                self.recalculateDocumentMargin();

                if (zoomInsertion <= 99) {
                    // there needs to be a padding, so that comments will still be visible
                    if (!docModel.getCommentLayer().isEmpty()) {
                        commentLayerWidth = docModel.getCommentLayer().getAppContentPaddingWidth();
                        appContentRoot.css({ paddingLeft: commentLayerWidth, paddingRight: commentLayerWidth });
                    }
                } else {
                    appContentRoot.css({ paddingLeft: '', paddingRight: '' });
                }

                // Use zoom as soon as all browsers support it.
                //  pageNode.css({
                //      'zoom': zoomInsertion / 100
                //  });

                zoomType = zoomFactor;
                scrollToSelection();

            }
            self.trigger('change:zoom');
            return this;
        };

        /**
         * Creates necessary margin values when using transform scale css property
         *
         */
        this.recalculateDocumentMargin = function (options) {

            // must be refreshed before 'recalculatedMargin' is calculated,
            // otherwise the margin is wrong at the document loading
            if (zoomFactor !== 100) {
                // #34735 page width and height values have to be updated
                pageNodeWidth = pageNode.outerWidth();
                pageNodeHeight = pageNode.outerHeight();
            }

            var
                marginAddition = zoomFactor < 99 ? 3 : 0,
                recalculatedMargin = (pageNodeWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition,
                // whether the (vertical) scroll position needs to be restored
                keepScrollPosition = Utils.getBooleanOption(options, 'keepScrollPosition', false),
                // an additional right margin used for comments that might be set at the page node
                commentMargin = pageNode.hasClass(DOM.COMMENTMARGIN_CLASS) ? pageNode.data(DOM.COMMENTMARGIN_CLASS) : 0;

            if (_.browser.Chrome && zoomFactor > 100) {
                pageNode.css({
                    margin:  marginAddition + 'px ' + (recalculatedMargin + commentMargin) + 'px ' + (30 / (zoomFactor / 100)) + 'px ' + recalculatedMargin + 'px'
                });
            } else {
                pageNode.css({
                    margin:  marginAddition + 'px ' + (recalculatedMargin + commentMargin) + 'px ' + (pageNodeHeight * (zoomFactor / 100 - 1) + marginAddition) + 'px ' + recalculatedMargin + 'px'
                });
            }

            setAdditiveMargin();

            // in the end scroll to cursor position/selection (but not after event 'refresh:layout', that uses the option 'keepScrollPosition')
            if (!keepScrollPosition) { scrollToSelection(); }
        };

        /**
         * Rejects an attempt to edit the text document (e.g. due to reaching
         * the table size limits).
         *
         * @param {String} cause
         *  The cause of the rejection.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.rejectEditTextAttempt = function (cause) {

            switch (cause) {
                case 'tablesizerow':
                    this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further rows.') });
                    break;

                case 'tablesizecolumn':
                    this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further columns.') });
                    break;
            }

            return this;
        };

        this.showPageSettingsDialog = function () {
            return new Dialogs.PageSettingsDialog(this).show();
        };

        /**
         * Function, that is only required, because it might be called from field manager.
         * This is not required in presentation app.
         */
        this.hideFieldFormatPopup = $.noop;

        /**
         * Function, that is only required, because it might be called in undo:after.
         * This is not required in presentation app.
         */
        this.getChangeTrackPopup = function () {
            return { hide: $.noop };
        };

        /**
         * Change track function, that is not (yet) required in presentation app.
         */
        this.showChangeTrackPopup = $.noop;

        /**
         *
         * This is not  yet required in presentation app.
         */
        this.showFieldFormatPopup = function () {
            return { hide: $.noop };
        };

        /**
         *
         * This is not  yet required in presentation app.
         */
        this.hideFieldFormatPopup = function () {
            return { hide: $.noop };
        };

        /**
         * Shows an insert image dialog for a slide background.
         *
         * @param {String} dialogType
         *  The dialog type. See EditView.showInsertImageDialog() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the dialog has
         *  been closed and the image has been inserted into the document.
         */
        this.showInsertImageDialogForSlideBackground = function (dialogType) {
            return this.showInsertImageDialog(dialogType).then(function (imageFragment) {

                var activeSlideId = docModel.getActiveSlideId();
                var $slide = docModel.getSlideById(activeSlideId);
                var attrs = { type: 'bitmap', bitmap: { imageUrl: imageFragment.url } };
                var oldAttrs = $slide.data('previewAttrs');

                if (oldAttrs && oldAttrs.fill && oldAttrs.fill.gradient) { attrs.gradient = null; }

                return docModel.getImageSize(imageFragment.url).then(function (size) {
                    var // the ratio of width and height of the slide
                        slideRatio = docModel.getSlideRatio(),
                        // the ratio of width and height of the drawing
                        drawingRatio = Utils.round(size.width / size.height, 0.01),
                        // stretching ratio
                        stretchingRatio = 1;

                    if (slideRatio > drawingRatio) {
                    // the drawing needs to be cropped at top and bottom -> height need to be increased
                        stretchingRatio = Math.round((1 - slideRatio / drawingRatio) * 50);
                        attrs.bitmap.stretching = { top: stretchingRatio, bottom: stretchingRatio, left: 0, right: 0 };
                    } else {
                        // the drawing needs to be cropped at left and right sides -> width need to be increased
                        stretchingRatio = Math.round((1 - drawingRatio / slideRatio) * 50);
                        attrs.bitmap.stretching = { top: 0, bottom: 0, left: stretchingRatio, right: stretchingRatio };
                    }

                    return docModel.getSlideStyles().updateBackground(DOM.getTempSlideNodeForSlide($slide), { fill: attrs }, activeSlideId, null, { imageLoad: true, preview: true }).then(function () {
                        app.getView().handleSlideBackgroundVisibility(activeSlideId, attrs);
                        // store values for apply
                        $slide.data('previewAttrs', { fill: attrs });
                    });
                })
                    .fail(function () {
                        app.rejectEditAttempt('image');
                    });
            })
                .always(function () {
                    if (Utils.SMALL_DEVICE) {
                        $('.picker-pc-container').remove(); // remove unnecessary container on small devices on filepicker close
                    }
                });
        };

        /**
         * Opens slide background dialog.
         *
         * @returns {SlideBackgroundDialog}
         */
        this.openSlideBackgroundDialog = function () {
            return new SlideBackgroundDialog(self, docModel.getSlideAttributesByFamily(docModel.getActiveSlideId(), 'fill')).show(); // open slide background dialog
        };

        /**
         * Public method for handling slide background visibility.
         *
         * @param {String} activeSlideId  id of the active slide
         * @param {Object} [previewFillAttrs] optional set of attributes on the slide for preview
         */
        this.handleSlideBackgroundVisibility = function (activeSlideId, previewFillAttrs) {
            var slideId, layoutId, masterId;

            if (docModel.isMasterSlideId(activeSlideId)) {
                slideId = null;
                layoutId = null;
                masterId = activeSlideId;
            } else if (docModel.isLayoutSlideId(activeSlideId)) {
                slideId = null;
                layoutId = activeSlideId;
                masterId = docModel.getMasterSlideId(layoutId);
            } else {
                slideId = activeSlideId;
                layoutId = docModel.getLayoutSlideId(slideId);
                masterId = docModel.getMasterSlideId(layoutId);
            }
            handleSlideBackgroundVisibility(slideId, layoutId, masterId, previewFillAttrs);
        };

        /**
         * Public method for handling follow master page change.
         * Info: this method is only used as preview, while slide background dialog is open.
         * @param {Object} options
         *     @param {Boolean} options.followMasterShapes = false
         */
        this.previewFollowMasterPageChange = function (options) {
            var // the id of the active slide
                slideId = docModel.getActiveSlideId(),
                // the id of the layout slide
                layoutId = null,
                // the id of the master slide
                masterId = null,
                // whether drawings of layout slide shall be visible
                followMasterShapes = Utils.getBooleanOption(options, 'followMasterShapes', false),
                // whether drawings of layout slide shall be visible
                showLayoutSlide = true,
                // whether drawings of master slide shall be visible
                showMasterSlide = true;

            if (docModel.isStandardSlideId(slideId)) {
                layoutId = docModel.getLayoutSlideId(slideId);
                showLayoutSlide = followMasterShapes;
                if (layoutId) {
                    masterId = docModel.getMasterSlideId(layoutId);
                    // master slide can only be visible, if layout slide is visible, too
                    showMasterSlide = showLayoutSlide ? docModel.isFollowMasterShapeSlide(layoutId) : false;
                }
            } else if (docModel.isLayoutSlideId(slideId)) {
                masterId = docModel.getMasterSlideId(slideId);
                showMasterSlide = followMasterShapes;
            }

            // handling the visibility of the drawings of the layout and the master slide
            handleLayoutMasterVisibility(slideId, layoutId, masterId, showLayoutSlide, showMasterSlide);
        };

        /**
         * After successful import the scroll handler for the vertical scrolling can be registered.
         */
        this.enableVerticalScrollBar = function () {

            /**
             * After a change of the layout, the height of the content node need
             * to be updated and also the margin-top of the page node.
             */
            refreshVerticalScrollBar = function (event, options) {

                // do nothing, if the document was reloaded with a snapshot
                if (Utils.getBooleanOption(options, 'documentReloaded', false)) { return; }

                // setting the height of the app content node, so that the vertical scroll bar is adapted.
                handleActiveViewChange(null, { slideCount: app.getModel().getActiveViewCount() });

                // also updating the top margin at the page for the active slide
                handleVerticalScrollbarUpdate(null, { pos: app.getModel().getActiveSlideIndex(), keepScrollPosition: Utils.getBooleanOption(options, 'keepScrollPosition', false) });
            };

            /**
             * After releasing the mouse button when scrolling on the vertical scroll bar,
             * it might be necessary to adapt the slide position and to reset some global markers.
             */
            stopScrollBarScrolling = function () {

                isMouseButtonPressedScrolling = false; // must be set to false before final call of 'handleVerticalScrollbarUpdate'

                //  updating the top margin at the page for the active slide, if a slide change happened
                if (slideChanged && verticalScrollChanged) { handleVerticalScrollbarUpdate(null, { pos: app.getModel().getActiveSlideIndex() }); }
                // resetting global markers
                slideChanged = false;
                verticalScrollChanged = false;
            };

            // Registering vertical scroll handler
            self.listenTo(self.getContentRootNode(), 'scroll', scrollInContentRootNode);
            // Handle an update of a zoom change
            self.listenTo(self, 'change:zoom', function () {
                pageNode.removeData('scrollTopMargin'); // the saved value cannot be reused
                refreshVerticalScrollBar();
            });
            // Handle switch of active view
            self.listenTo(docModel, 'change:activeView', handleActiveViewChange);
            // Handle an update of the vertical scroll bar
            self.listenTo(docModel, 'update:verticalscrollbar', handleVerticalScrollbarUpdate);
            // inserting or deleting a slide requires also an update of the vertical scroll bar
            self.listenTo(docModel, 'inserted:slide removed:slide moved:slide', refreshVerticalScrollBar);

            // avoiding scrolling together with mouse down
            self.listenTo(self.getContentRootNode(), 'mousedown touchstart', function () { isMouseButtonPressedScrolling = true; });
            self.listenTo(self.getContentRootNode(), 'mouseup touchend', stopScrollBarScrolling);

            // refreshing once the vertical scroll bar after successful loading
            refreshVerticalScrollBar();
        };

        /**
         * Set the virtual focus to the slidepane or the document.
         *
         * @param {Boolean} focusToSlidepane
         *  Set focus to slidepane when 'true' or set it to the document when 'false'.
         */
        this.setVirtualFocusSlidepane = function (focusToSlidepane) {
            slidepaneHasFocus = !!focusToSlidepane;
        };

        /**
         * Returns whether the slidepane has the virtual focus.
         */
        this.hasSlidepaneVirtualFocus = function () {
            return slidepaneHasFocus;
        };

        this.getActiveTableFlags = function () {
            var flags = {
                    firstRow:   true,
                    lastRow:    true,
                    firstCol:   true,
                    lastCol:    true,
                    bandsHor:   true,
                    bandsVert:  true
                },
                drawing = docModel.getSelection().getAnyDrawingSelection(),
                tableAttrs =  drawing ? AttributeUtils.getExplicitAttributes(drawing).table : null;

            if (tableAttrs && tableAttrs.exclude && _.isArray(tableAttrs.exclude) && tableAttrs.exclude.length > 0) {
                tableAttrs.exclude.forEach(function (ex) {
                    flags[ex] = false;
                });
            }

            return flags;
        };

        // initialization -----------------------------------------------------

        // create the context menu
        contextMenu = new PresentationContextMenu(this, { delay: (_.browser.IE) ? 1 : 200 });

        // store the values of page width and height with paddings, after loading is finished
        this.waitForImportSuccess(function () {
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();

            if (Utils.COMPACT_DEVICE) {
                self.executeDelayed(function () {
                    var optimalDocWidth = calculateOptimalDocumentScale();
                    if (optimalDocWidth < 100) {
                        zoomType = 'slide';
                    }
                }, 'PresentationView.waitForImportSuccess');
            }

            if (docModel.getSlideTouchMode()) {
                var appContent = pageNode.parent();
                appContent.attr('contenteditable', true);

                if (Utils.IOS) { docModel.getSelection().getClipboardNode().attr('contenteditable', false); }
                var listenerList = docModel.getListenerList();
                pageNode.off(listenerList);
                appContent.on(listenerList);

                //we dont want the focus in the document on the beginning!
                Utils.focusHiddenNode();
            }

            // quick fix for bug 40053
            if (_.browser.Safari) { Forms.addDeviceMarkers(app.getWindowNode()); }

        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            app = self = replayOperations = replayOperationsOSN = docModel = null;
            appPaneNode = contentRootNode = pageNode = pageContentNode = contextMenu = null;
        });

        // presentation view mixins -------------------------------------------

        /**
         * Additional preview specific behavior for the presentation view.
         *
         * `SlidePreviewMixin` provides a set of specific functionality
         *  that is just for rendering the preview of any presentation
         *  slide that most recently has been changed.
         */
        SlidePreviewMixin.call(this, app, initOptions); // provide the presentations view context and inject `app` and `initOptions` each as additional state.

    } }); // class PresentationView

    // exports ================================================================

    return PresentationView;

});
