/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Jonas Regier <jonas.regier@open-xchange.com>
 */

define('io.ox/office/portal/view/contextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/popup/listmenu'
], function (Utils, ListMenu) {

    'use strict';

    // class ContextMenu ======================================================

    // global flag if a context menu is open
    var isContextMenuOpenFlag = false;

    /**
     * Creates a context menu
     *
     * @constructor
     *
     * @extends ListMenu
     *
     * @param {event} e
     *  The event which invoked the context menu.
     *  It's necessary to prevent the native context menu later
     *
     */
    var ContextMenu = ListMenu.extend({ constructor: function (event) {

        // self reference
        var self = this;

        // position were the ContextMenu should be opened
        var x = event.pageX;
        var y = event.pageY;

        // the identifier of the root application window
        var windowId = $(event.target).closest('.io-ox-office-portal-main').attr('id');

        // base constructor ---------------------------------------------------

        ListMenu.call(this, windowId, { itemDesign: 'list', anchorBorder: 'right', boundingBox: $('body') });

        // private methods ----------------------------------------------------

        // calculate context menu position and set the anchor
        function setPos() {

            // if the context menu width (x-axis) does not fit fully on screen
            // move the anchor to prevent shrinking
            var width = document.body.clientWidth - 15; // to get -15 pixel offset from left screen border
            if (x + self.getContentNode().innerWidth() >= width) {
                x = width - self.getContentNode().innerWidth();
            }

            // open the contextmenu above the finger on touch
            if (Utils.TOUCHDEVICE) {
                y = Math.max(0, y - self.getContentNode().innerHeight());
            }

            // set the anchor
            self.setAnchor({ left: x, top: y, width: 1, height: 1 });
        }

        // initialization -----------------------------------------------------

        // After closing an OX app there could still be an invisible text selection active.
        // Due to this, iOS detects an open softkeyboard and therefore would reduce the boundingBox.
        // This results in a self closing context menu, when opened at the bottom part from the screen,
        // because the popup is self closed in the the refreshNode loop.
        window.getSelection().removeAllRanges();

        // prevent the default context menu
        event.preventDefault();

        // add marker class for additional CSS formatting
        this.getNode().addClass('portal-context-menu');

        //  check and set the position from the contextmenu
        this.on('popup:show', setPos);

        // manually trigger destroy when the contextmenu is hidden
        this.on('popup:hide', self.destroy);

        // close the context menu on orientationchange
        this.listenOnceTo($(window), 'orientationchange', function () {
            self.hide();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = x = y = null;
        });

    } }); // class ContextMenu

    // static methods ---------------------------------------------------------

    // getter for the isContextMenuOpenFlag
    ContextMenu.isOpen = function () {
        return isContextMenuOpenFlag;
    };

    // setter for the isContextMenuOpenFlag
    ContextMenu.setOpen = function (state) {
        isContextMenuOpenFlag = state;
    };

    // exports ================================================================

    return ContextMenu;
});
