/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/operationspane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/baseframework/view/pane',
    'io.ox/office/editframework/utils/editconfig',
    'io.ox/office/editframework/utils/operationutils',
    'io.ox/office/editframework/view/popup/attributestooltip',
    'less!io.ox/office/editframework/view/operationspane'
], function (Utils, Forms, KeyCodes, Pane, EditConfig, OperationUtils, AttributesToolTip) {

    'use strict';

    // class OperationsPane ===================================================

    /**
     * A view pane attached to the bottom border of the application view that
     * contains the operations log, and other custom debug information defined
     * by the application.
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {EditView} docView
     *  The document view containing this pane instance.
     */
    var OperationsPane = Pane.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the application instance
        var app = docView.getApp();

        // the document model instance
        var docModel = docView.getDocModel();

        // a tool-tip for details about attribute sets shown in this pane
        var attrsToolTip = new AttributesToolTip(docView);

        // the <table> element of the operations log
        var opTableNode = null;

        // the <table> element of the debug info table
        var infoTableNode = null;

        // operations not yet logged
        var pendingOperations = [];

        // output elements for the debug info table (faster than DOM look-ups)
        var infoNodeMap = {};

        // base constructor ---------------------------------------------------

        Pane.call(this, docView, {
            position: 'bottom',
            classes: 'operations-pane noI18n',
            size: 300,
            resizable: true,
            minSize: 100,
            maxSize: 500
        });

        // private methods ----------------------------------------------------

        /**
         * Generates and inserts the HTML mark-up for all pending operations
         * contained in the 'pendingOperations' array.
         */
        var logPendingOperations = (function () {

            // number of operations already logged
            var operationsCount = 0;
            // running background loop for the HTML generator
            var timer = null;

            function logOperations() {

                // nothing to do with running timer
                if (timer || !self.isImportFinished()) { return; }

                // create a new background loop that runs as long as the array contains entries
                timer = self.repeatDelayed(function () {

                    // do not generate operations mark-up when the pane becomes hidden
                    if ((pendingOperations.length === 0) || !self.isVisible()) {
                        return Utils.BREAK;
                    }

                    // the number of operations applied successfully by the document model
                    var successCount = docModel.getOperationsCount();
                    // the generated mark-up
                    var markup = '';
                    // start timestamp
                    var t0 = Date.now();

                    while ((pendingOperations.length > 0) && (Date.now() - t0 < 200)) {

                        var entry = pendingOperations.shift();

                        var name = entry.operation.name || entry.operation.n || '';
                        var osn = OperationUtils.getOSN(entry.operation);
                        var opl = OperationUtils.getOPL(entry.operation);

                        var properties = _.clone(entry.operation);
                        delete properties.name;
                        delete properties.n;
                        delete properties.osn;
                        delete properties.opl;

                        operationsCount += 1;

                        var source = entry.external ? 'External' : 'Internal';
                        var classes = source.toLowerCase() + ((successCount < operationsCount) ? ' error' : '');
                        if (entry.removed) { classes += ' removed'; }

                        markup += '<tr class="' + classes + '">';
                        markup += '<td>' + operationsCount + '</td>';
                        markup += '<td title="' + source + ' operation">' + source[0] + '</td>';
                        markup += '<td style="text-align:right;">' + ((osn >= 0) ? osn : '') + '</td>';
                        markup += '<td style="text-align:right;">' + ((opl > 0) ? opl : '') + '</td>';
                        markup += '<td>' + Utils.escapeHTML(name) + '</td>';
                        markup += '<td title="' + Forms.createLoggingTooltipMarkup(properties) + '">' + Utils.escapeHTML(Utils.stringifyForDebug(properties)) + '</td>';
                        markup += '</tr>';
                    }

                    // the scrollable DOM node
                    var scrollNode = opTableNode[0].parentNode;
                    // the maximum vertical scroll position
                    var scrollMax = scrollNode.scrollHeight - scrollNode.clientHeight;
                    // whether the scroll node is scrolled to the bottom border
                    var autoScroll = scrollMax - scrollNode.scrollTop <= 2;

                    // append the mark-up
                    opTableNode.append(markup);
                    // scroll down to the last log entry
                    if (autoScroll) {
                        scrollNode.scrollTop = scrollNode.scrollHeight;
                    }

                }, 'OperationsPane.logOperations', { repeatDelay: 500 });

                // clean up after the loop is finished or has been aborted
                timer.always(function () { timer = null; });
            }

            return self.createDebouncedMethod('OperationsPane.logPendingOperations', null, logOperations, { delay: 500 });
        }());

        // public methods -----------------------------------------------------

        /**
         * Adds a new header row to the information window.
         *
         * @param {String} header
         *  The name of the header row, used to identify data rows shown under
         *  the new header row.
         *
         * @param {String} title
         *  The title label shown in the header row.
         *
         * @param {String} [tooltip]
         *  A tooltip for the entire header row.
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.addDebugInfoHeader = function (header, title, tooltip) {
            var tooltipMarkup = tooltip ? (' title="' + Utils.escapeHTML(tooltip) + '"') : '';
            var rowNode = $('<tr' + tooltipMarkup + '><th colspan="2">' + Utils.escapeHTML(title) + '</th></tr>');
            infoTableNode.append(rowNode);
            infoNodeMap[header] = rowNode;
            return this;
        };

        /**
         * Adds a new data row to the information window.
         *
         * @param {String} header
         *  The name of the header row to associate this data row to.
         *
         * @param {String} label
         *  The name of the data row, used to identify the row when inserting
         *  text contents, and as label text.
         *
         * @param {String} [tooltip]
         *  A tooltip for the first cell of the data row (the row label).
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.addDebugInfoNode = function (header, label, tooltip) {
            var infoNode = $('<td>');
            var rowNode = $('<tr><td title="' + Utils.escapeHTML(tooltip) + '">' + label + '</td></tr>').append(infoNode);
            infoNodeMap[header].last().after(rowNode);
            infoNodeMap[header] = infoNodeMap[header].add(rowNode);
            infoNodeMap[header + ':' + label] = infoNode;
            return this;
        };

        /**
         * Changes the text shown in a specific data row of the information
         * window.
         *
         * @param {String} header
         *  The name of the header containing the data row to be changed.
         *
         * @param {String} label
         *  The name of the data row to be changed.
         *
         * @param {String} text
         *  The new text contents shown in the data row.
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.setDebugInfoText = function (header, label, text) {
            infoNodeMap[header + ':' + label].text(text);
            return this;
        };

        /**
         * Changes the HTML mark-up shown in a specific data row of the
         * information window.
         *
         * @param {String} header
         *  The name of the header containing the data row to be changed.
         *
         * @param {String} label
         *  The name of the data row to be changed.
         *
         * @param {String} markup
         *  The new HTML mark-up shown in the data row.
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.setDebugInfoMarkup = function (header, label, markup) {
            infoNodeMap[header + ':' + label].html(markup);
            return this;
        };

        /**
         * Registers a formatter callback function for specific attribute types
         * or values, shown in a tool-tip for attribute sets provided by this
         * instance. See method AttributesToolTip.registerFormatter() for more
         * details.
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.registerAttributeFormatter = function (selector, formatter) {
            attrsToolTip.registerFormatter(selector, formatter);
            return this;
        };

        // initialization -----------------------------------------------------

        // create the operations log table, and the debug info table
        opTableNode = $('<table class="debug-table"><tbody><tr><th></th><th title="Operation type">T</th><th title="Operation State Number" style="font-size:70%;">OSN</th><th title="Operation Length" style="font-size:70%;">OPL</th><th>Name</th><th>Properties</th></tr></tbody></table>');
        infoTableNode = $('<table class="debug-table" style="table-layout:fixed;"><colgroup><col width="50"></col><col></col></colgroup><tbody></tbody></table>');

        // add the operations table and the info table to the debug pane
        [{ node: opTableNode, cls: 'ops' }, { node: infoTableNode, cls: 'info' }].forEach(function (entry) {
            var scrollNode = $('<div tabindex="0">').append(entry.node);
            self.getNode().append($('<div class="output-panel ' + entry.cls + '">').append(scrollNode));
        });

        // create the output nodes in the debug pane
        this.addDebugInfoHeader('application', 'Application State')
            .addDebugInfoNode('application', 'state', 'Application state')
            .addDebugInfoNode('application', 'osn', 'Operation state number')
            .addDebugInfoNode('application', 'undo', 'Undo/redo stack state');

        // log the application state
        this.listenTo(app, 'docs:state', function (state) {
            self.setDebugInfoText('application', 'state', state);
        });

        // log the operation state number of the document
        this.listenTo(docModel, 'change:osn', function (event, osn) {
            self.setDebugInfoText('application', 'osn', osn);
        });

        // log size of undo/redo stacks
        this.handleChangedControllerItems(function () {
            var undoCount = docView.getControllerItemValue('document/undo');
            var redoCount = docView.getControllerItemValue('document/redo');
            self.setDebugInfoText('application', 'undo', 'undo=' + undoCount + ', redo=' + redoCount);
        });

        // collect and log all operations
        this.listenTo(docModel, 'operations:after', function (event, operations, external) {
            if (!app.isOperationsBlockActive()) {
                operations.forEach(function (operation) {
                    var localOp = _.copy(operation, true);
                    if (EditConfig.USE_SHORT_OPERATIONS && OperationUtils.isConversionDefined) { OperationUtils.handleMinifiedObject(localOp); }
                    pendingOperations.push({ operation: localOp, orig: operation, external: external });
                });
                logPendingOperations();
            }
        });

        // continue operations logging when making the operations pane visible
        this.on('pane:show', logPendingOperations);

        // start logging collected operations after the document is imported
        this.waitForImport(function () {
            this.executeDelayed(logPendingOperations, 'OperationsPane.waitForImport', 3000);
        }, this);

        // Mark the last operations in the operations table with orange background color. This means,
        // that they were not executed, not send to the server and they do not influence the OSN for
        // the following operations. This scenario happens for example, if a paste process is cancelled
        // and an existing selection was already removed before the pasting started.
        this.listenTo(app, 'docs:operations:remove', function (number) {

            // mark operations not yet logged
            var pendingCount = pendingOperations.length;
            for (var index = Math.max(0, pendingCount - number); index < pendingCount; index += 1) {
                pendingOperations[index].removed = true;
            }

            // mark operations already logged
            if (pendingCount < number) {
                opTableNode.find('tr').slice(number - pendingCount).addClass('removed');
            }
        });

        // keyboard shortcuts
        this.getNode().find('[tabindex]').on('keydown', function (event) {
            if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                var docRange = window.document.createRange();
                docRange.setStart(event.delegateTarget, 0);
                docRange.setEnd(event.delegateTarget, 1);
                window.getSelection().removeAllRanges();
                window.getSelection().addRange(docRange);
                return false;
            }
            if (event.keyCode === KeyCodes.ESCAPE) {
                docView.grabFocus();
                return false;
            }
        });

        // show the formatting attributes tool-tip
        this.getNode().on('click', AttributesToolTip.NODE_SELECTOR, function (event) {
            attrsToolTip.initialize(event.currentTarget).show();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            attrsToolTip.destroy();
            self = app = docView = docModel = attrsToolTip = null;
            opTableNode = infoTableNode = pendingOperations = infoNodeMap = null;
        });

    } }); // class OperationsPane

    // exports ================================================================

    return OperationsPane;

});
