/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/fontfamilypicker', [
    'io.ox/office/tk/control/combofield',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/editframework/utils/attributeutils',
    'gettext!io.ox/office/editframework/main'
], function (ComboField, AppObjectMixin, AttributeUtils, gt) {

    'use strict';

    // class FontNameValidator ================================================

    var FontNameValidator = ComboField.Validator.extend(function (docModel) {
        ComboField.Validator.call(this);
        this._docModel = docModel;
    });

    FontNameValidator.prototype.valueToText = function (value) {
        return this._docModel.resolveFontName(value);
    };

    // class FontFamilyPicker =================================================

    /**
     * A combo-box control used to select a font family.
     *
     * @constructor
     *
     * @extends ComboField
     * @extends AppObjectMixin
     *
     * @param {EditView} docView
     *  The document view instance containing this control.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class ComboField.
     */
    var FontFamilyPicker = ComboField.extend({ constructor: function (docView, initOptions) {

        // self reference
        var self = this;

        // the document model, and its containers
        var docModel = docView.getDocModel();
        var themeCollection = docModel.getThemeCollection();
        var fontCollection = docModel.getFontCollection();

        // base constructors --------------------------------------------------

        ComboField.call(this, docView, _.extend({
            width: 95,
            tooltip: gt('Font name')
        }, initOptions, {
            sortItems: true,
            typeAhead: true,
            validator: new FontNameValidator(docModel)
        }));

        AppObjectMixin.call(this, docView.getApp());

        // private methods ----------------------------------------------------

        /**
         * Creates a list item for the specified font.
         */
        function createFontEntry(section, fontKey, subLabel, sortIndex) {
            var fontName = self.valueToText(fontKey);
            var buttonNode = self.getMenu().createItemNode(fontKey, {
                section: section,
                label: _.noI18n(fontName),
                labelStyle: { fontFamily: docModel.getCssFontFamily(fontName), fontSize: '115%' },
                tooltip: _.noI18n(fontName + (subLabel ? (' ' + subLabel) : '')),
                sortIndex: sortIndex || fontName.toLowerCase()
            });
            if (subLabel) {
                buttonNode.append($('<span class="sub-label">').text(subLabel));
            }
        }

        /**
         * Fills the drop-down list with all known font names.
         */
        var fillList = this.createDebouncedMethod('FontFamilyPicker.fillList', null, function () {

            self.getMenu().clearContents();

            // create the menu entries for scheme fonts
            if (docView.getApp().isOOXML()) {
                self.getMenu().createSectionNode('scheme');
                //#. a label suffix for the default font of the document, used for "important text" such as headings (shown in font drop-down font name)
                createFontEntry('scheme', AttributeUtils.MAJOR_FONT_KEY, gt.pgettext('font', '(Headings)'), 1);
                //#. a label suffix for the default font of the document, used for "regular text" (shown in font drop-down after font name)
                createFontEntry('scheme', AttributeUtils.MINOR_FONT_KEY, gt.pgettext('font', '(Body)'), 2);
            }

            // create the menu entries for the regular fonts contained in the document font collection
            self.getMenu().createSectionNode('fonts');
            fontCollection.getFontNames().forEach(function (fontName) {
                createFontEntry('fonts', fontName);
            });
        });

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('font-family-picker');

        // add all known fonts after import and changed fonts
        this.waitForImportSuccess(fillList);
        this.listenTo(themeCollection, 'triggered', fillList);
        this.listenTo(fontCollection, 'triggered', fillList);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = docModel = themeCollection = fontCollection = null;
        });

    } });  // class FontFamilyPicker

    // exports ================================================================

    return FontFamilyPicker;

});
