/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/editframework/utils/toolpanemanager', [
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/editframework/utils/editconfig'
], function (TriggerObject, Config) {

    'use strict';

    // class ToolPaneManager ===========================================

    /**
     * An instance of this class represents the tool pane manager. This manager keeps
     * track of the content that is displayed in the tool pane. For performance reasons
     * the toolpane is not created completely when the document is loaded. Instead
     * the toolpanes are activated, if the corresponding top bar is activated the
     * first time.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {PresentationApplication} app
     *  The application instance.
     */
    function ToolPaneManager(app, guiHandler) {

        // self reference
        var self = this;
        // the document view
        var view = null;
        // the document model
        var model = null;
        // collection containing data for all registered tool bar tabs
        var toolBarTabs = null;
        // the list of all activated tabs
        var activatedTabs = {};
        // the list of all registered, but still not activated tabs
        var inactiveTabs = {};
        // a function that checks for idle time to update the tool pane in background
        var idleTimeChecker = $.noop;
        // the idle time (in ms), after that the tool pane can be prepared
        var idleTime = Config.AUTOTEST ? 10 : 2500;
        // a priority list for the tool bar tabs (first the large toolbar tabs)
        var tabPrioList = ['drawing', 'insert'];

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Activating the edit tool pane for the top pane with the specified id.
         *
         * @param {jQuery.Event} event
         *  A jQuery 'tab:activate' event object.
         *
         * @param {topPaneId} String
         *  The id of the activated top pane.
         */
        function activateToolPane(event, topPaneId) {
            if (topPaneId && !activatedTabs[topPaneId]) {
                guiHandler.call(view, topPaneId);
                activatedTabs[topPaneId] = 1;
                // app.getController().forceOneCompleteControllerUpdate(); // requires a full update of the GUI elements (56005)
                app.getController().update(true); // requires a full update of the GUI elements (56005)
                if (inactiveTabs[topPaneId]) {
                    delete inactiveTabs[topPaneId];
                    if (_.isEmpty(inactiveTabs)) {
                        // deactivating the listener for 'tab:activate' events
                        self.stopListeningTo(toolBarTabs, 'tab:activate', activateToolPane);
                    }
                    if (tabPrioList && _.contains(tabPrioList, topPaneId)) {
                        tabPrioList = _.without(tabPrioList, topPaneId);
                        if (_.isEmpty(tabPrioList)) { tabPrioList = null; }
                    }
                }
            }
        }

        /**
         * Registering a new generated tool bar tab.
         *
         * @param {jQuery.Event} event
         *  A jQuery 'tab:create' event object.
         *
         * @param {topPaneId} String
         *  The id of the created tool bar tab element.
         */
        function registerToolPane(event, topPaneId) {

            // whether this is the first created tool bar tab
            var isFirst = false;

            if (!inactiveTabs[topPaneId]) {
                isFirst = _.isEmpty(inactiveTabs);
                inactiveTabs[topPaneId] = 1;

                if (isFirst) {
                    // activating the listener for 'tab:activate' events
                    self.listenTo(toolBarTabs, 'tab:activate', activateToolPane);
                }
            }
        }

        /**
         * After the import has finished the listener for the view menu group can be registered.
         */
        function importFinishedHandler() {

            // allow, that the first tool bar tab can be shown. This must not happen before, because there
            // might be changes between toolbars before the document is loaded. This must be avoided.
            if (app.getUserSettingsValue('showToolbars', true)) { toolBarTabs.allowFirstTab(); }

            self.listenOnceTo(view.getTopPane().getViewMenuGroup().getMenu(), 'popup:beforeshow', function (event) {
                activateToolPane(event, 'viewmenu');
            });

            // starting the process to prepare the tool pane, if there is no user input for a specified time
            registerIdleTimeChecker();
            idleTimeChecker(); // starting process
        }

        /**
         * Registering the handler to find idle time for preparing the tool pane.
         *
         * @returns {undefined}
         */
        function registerIdleTimeChecker() {
            idleTimeChecker = self.createDebouncedMethod('ToolPaneManager.idleTimeChecker', null, idleToolPaneHandler, { delay: idleTime, background: true });
            model.getNode().on('keydown mousedown touchstart', idleTimeChecker);
        }

        /**
         * Deregistering the handler to find idle time for preparing the tool pane.
         * This can be done, if there are no more open tasks.
         */
        function deregisterIdleTimeChecker() {
            model.getNode().off('keydown mousedown touchstart', idleTimeChecker);
            idleTimeChecker = _.noop;
        }

        /**
         * Handler to prepare the tool bar, if there was no user input for a specified time.
         */
        function idleToolPaneHandler() {

            if (!model || model.destroyed) { return; } // handling closing of application

            if (_.isEmpty(inactiveTabs)) {
                deregisterIdleTimeChecker();
            } else {
                var tabId = tabPrioList ? _.first(tabPrioList) : _.first(_.keys(inactiveTabs));
                activateToolPane(null, tabId); // activating the selected tab
                idleTimeChecker(); // keeping process running
            }
        }

        // methods ------------------------------------------------------------

        // initialization -----------------------------------------------------

        app.onInit(function () {
            view = app.getView();
            model = app.getModel();
            toolBarTabs = view.getToolBarTabs();

            self.listenTo(toolBarTabs, 'tab:create', registerToolPane);
            model.waitForImport(importFinishedHandler);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            deregisterIdleTimeChecker();
            self = model = view = activatedTabs = inactiveTabs = null;
        });

    } // class ToolPaneManager

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: ToolPaneManager });
});
