/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/pattern', [
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/editframework/utils/color'
], function (ValueMap, Canvas, Color) {

    'use strict';

    // the bitmaps of all predefined patterns
    var PRESET_PATTERN_MAP = {
        cross:      [0xff, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80],
        dashDnDiag: [0x88, 0x44, 0x22, 0x11, 0x00, 0x00, 0x00, 0x00],
        dashHorz:   [0xf0, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00],
        dashUpDiag: [0x11, 0x22, 0x44, 0x88, 0x00, 0x00, 0x00, 0x00],
        dashVert:   [0x80, 0x80, 0x80, 0x80, 0x08, 0x08, 0x08, 0x08],
        diagBrick:  [0x01, 0x02, 0x04, 0x08, 0x18, 0x24, 0x42, 0x81],
        diagCross:  [0xc1, 0x63, 0x36, 0x1c, 0x1c, 0x36, 0x63, 0xc1],
        divot:      [0x40, 0x80, 0x40, 0x00, 0x08, 0x04, 0x08, 0x00],
        dkDnDiag:   [0xcc, 0x66, 0x33, 0x99, 0xcc, 0x66, 0x33, 0x99],
        dkHorz:     [0xff, 0xff, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00],
        dkUpDiag:   [0x33, 0x66, 0xcc, 0x99, 0x33, 0x66, 0xcc, 0x99],
        dkVert:     [0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc],
        dnDiag:     [0xc0, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x03, 0x81],
        dotDmnd:    [0x80, 0x00, 0x22, 0x00, 0x08, 0x00, 0x22, 0x00],
        dotGrid:    [0xaa, 0x00, 0x80, 0x00, 0x80, 0x00, 0x80, 0x00],
        horz:       [0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
        horzBrick:  [0xff, 0x80, 0x80, 0x80, 0xff, 0x08, 0x08, 0x08],
        lgCheck:    [0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f],
        lgConfetti: [0xc0, 0xd8, 0x1b, 0x03, 0x30, 0xb1, 0x8d, 0x0c],
        lgGrid:     [0xff, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80],
        ltDnDiag:   [0x88, 0x44, 0x22, 0x11, 0x88, 0x44, 0x22, 0x11],
        ltHorz:     [0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00],
        ltUpDiag:   [0x11, 0x22, 0x44, 0x88, 0x11, 0x22, 0x44, 0x88],
        ltVert:     [0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88],
        narHorz:    [0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00],
        narVert:    [0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa],
        none:       [0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
        openDmnd:   [0x80, 0x41, 0x22, 0x14, 0x08, 0x14, 0x22, 0x41],
        pct10:      [0x80, 0x00, 0x08, 0x00, 0x80, 0x00, 0x08, 0x00],
        pct20:      [0x88, 0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00],
        pct25:      [0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22],
        pct30:      [0x88, 0x55, 0x22, 0x55, 0x88, 0x55, 0x22, 0x55],
        pct40:      [0x8a, 0x55, 0xaa, 0x55, 0xa8, 0x55, 0xaa, 0x55],
        pct5:       [0x80, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00],
        pct50:      [0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55],
        pct60:      [0xee, 0x55, 0xbb, 0x55, 0xee, 0x55, 0xbb, 0x55],
        pct70:      [0xee, 0xbb, 0xee, 0xbb, 0xee, 0xbb, 0xee, 0xbb],
        pct75:      [0xee, 0xff, 0xbb, 0xff, 0xee, 0xff, 0xbb, 0xff],
        pct80:      [0xfe, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xef, 0xff],
        pct90:      [0xfe, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff],
        plaid:      [0xf0, 0xf0, 0xf0, 0xf0, 0xaa, 0x55, 0xaa, 0x55],
        shingle:    [0x81, 0x42, 0x24, 0x18, 0x06, 0x01, 0x80, 0x80],
        smCheck:    [0xcc, 0xcc, 0x33, 0x33, 0xcc, 0xcc, 0x33, 0x33],
        smConfetti: [0x80, 0x10, 0x02, 0x40, 0x04, 0x20, 0x01, 0x08],
        smGrid:     [0xff, 0x88, 0x88, 0x88, 0xff, 0x88, 0x88, 0x88],
        solid:      [0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff],
        solidDmnd:  [0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00],
        sphere:     [0x77, 0x98, 0xf8, 0xf8, 0x77, 0x89, 0x8f, 0x8f],
        trellis:    [0xff, 0xcc, 0xff, 0x33, 0xff, 0xcc, 0xff, 0x33],
        upDiag:     [0x81, 0x03, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0],
        vert:       [0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80],
        wave:       [0x0c, 0x92, 0x60, 0x00, 0x0c, 0x92, 0x60, 0x00],
        wdDnDiag:   [0xc1, 0xe0, 0x70, 0x38, 0x1c, 0x0e, 0x07, 0x83],
        wdUpDiag:   [0x83, 0x07, 0x0e, 0x1c, 0x38, 0x70, 0xe0, 0xc1],
        weave:      [0x88, 0x41, 0x22, 0x15, 0x88, 0x45, 0x22, 0x54],
        zigZag:     [0x81, 0x42, 0x24, 0x18, 0x81, 0x42, 0x24, 0x18]
    };

    // a helper canvas element used to generate the fill patterns
    var canvas = new Canvas(Canvas.SINGLETON).initialize({ width: 8, height: 8 });

    // a cache with all generated fill patterns
    var patternCache = new ValueMap();

    // private global functions ===============================================

    /**
     * Creates all data needed to render the specified predefined fill pattern.
     * Internally, all generated patterns will be cached, and will be returned
     * immediately on subsequent calls.
     *
     * @param {String} presetId
     *  The identifier of a preset pattern to generate the rendering data for.
     *  If this value is not a supported pattern identifier, this method
     *  generates a pattern consisting of the background color only.
     *
     * @param {Object} backColor
     *  The background color (or 'fill color'), as JSON color descriptor as
     *  used in document operations.
     *
     * @param {Object} foreColor
     *  The foreground color (or 'pattern color'), as JSON color descriptor as
     *  used in document operations.
     *
     * @param {ThemeModel} themeModel
     *  The model of a document theme used to map scheme color names to color
     *  values.
     *
     * @returns {Object}
     *  A pattern descriptor with the following properties:
     *  - {String} url
     *      The data URL of a fill bitmap.
     *  - {CanvasPattern} pattern
     *      A pattern object to be used in canvas elements.
     */
    function generatePattern(patternId, backColor, foreColor, themeModel) {

        // validate the pattern identifier
        if (!(patternId in PRESET_PATTERN_MAP)) { patternId = 'none'; }

        // parse the passed JSON colors, and resolve to color descriptors
        backColor = Color.parseJSON(backColor).resolve('fill', themeModel);
        foreColor = Color.parseJSON(foreColor).resolve('line', themeModel);

        // the cache key consists of the pattern identifier, and the resulting CSS colors (with alpha)
        var cacheKey = patternId + ':' + backColor.css + ':' + foreColor.css;

        // return an existing pattern, or create a new pattern
        return patternCache.getOrCreate(cacheKey, function () {

            // fill with background color, draw the foreground pixels of the pattern
            canvas.renderPixels(function (context) {
                context.setColor(backColor.css);
                context.fillRect(0, 0, 8, 8);
                context.setColor(foreColor.css);
                PRESET_PATTERN_MAP[patternId].forEach(function (bits, y) {
                    for (var x = 0, mask = 128; x < 8; x += 1, mask /= 2) {
                        if (bits & mask) { context.drawPixel(x, y); }
                    }
                });
            });

            // create a fill pattern from the canvas contents
            return canvas.render(function (context) {
                return {
                    url: canvas.getDataURL(),
                    pattern: context.createPattern(canvas)
                };
            });
        });
    }

    // static class Pattern ===================================================

    var Pattern = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates the data URL of a fill bitmap for the specified predefined fill
     * pattern.
     *
     * @param {String} presetId
     *  The identifier of a preset pattern to generate the fill bitmap for. If
     *  this value is not a supported pattern identifier, this method generates
     *  a pattern consisting of the background color only.
     *
     * @param {Object} backColor
     *  The background color (or 'fill color'), as JSON color descriptor as
     *  used in document operations.
     *
     * @param {Object} foreColor
     *  The foreground color (or 'pattern color'), as JSON color descriptor as
     *  used in document operations.
     *
     * @param {ThemeModel} themeModel
     *  The model of a document theme used to map scheme color names to color
     *  values.
     *
     * @returns {String}
     *  The data URL of a fill bitmap.
     */
    Pattern.createDataURL = function (patternId, backColor, foreColor, themeModel) {
        return generatePattern(patternId, backColor, foreColor, themeModel).url;
    };

    /**
     * Creates a fill pattern structure for canvas elements for the specified
     * predefined fill pattern.
     *
     * @param {String} presetId
     *  The identifier of a preset pattern to generate the canvas pattern for.
     *  If this value is not a supported pattern identifier, this method
     *  generates a pattern consisting of the background color only.
     *
     * @param {Object} backColor
     *  The background color (or 'fill color'), as JSON color descriptor as
     *  used in document operations.
     *
     * @param {Object} foreColor
     *  The foreground color (or 'pattern color'), as JSON color descriptor as
     *  used in document operations.
     *
     * @param {ThemeModel} themeModel
     *  The model of a document theme used to map scheme color names to color
     *  values.
     *
     * @returns {CanvasPattern}
     *  The canvas pattern object. Can be used directly as value for the
     *  appropriate properties of a CanvasRenderingContext2D, or as value for
     *  the methods of the ContextWrapper of a toolkit Canvas instance.
     */
    Pattern.createCanvasPattern = function (patternId, backColor, foreColor, themeModel) {
        return generatePattern(patternId, backColor, foreColor, themeModel).pattern;
    };

    // exports ================================================================

    return Pattern;

});
