/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/border', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color'
], function (Utils, Color) {

    'use strict';

    // maximum line width for borders in GUI preview elements, in pixels
    var MAX_PREVIEW_LINEWIDTH = 2;

    // minimum line width for borders with style 'double', in pixels
    var MIN_DOUBLE_LINEWIDTH = 3;

    // maps operation line styles to CSS line styles (anything else maps to 'solid')
    var CSS_LINE_STYLES = {
        none: 'none',
        double: 'double',
        triple: 'double',
        dashed: 'dashed',
        dashSmallGap: 'dashed',
        dotted: 'dotted',
        dotDash: 'dotted',
        dotDotDash: 'dotted',
        dashDotStroked: 'dotted'
    };

    // maps predefined border width names to line width in 1/100 mm
    var BORDER_WIDTH_HMM_MAP = {
        hair: Utils.convertLengthToHmm(0.5, 'px'),
        thin: Utils.convertLengthToHmm(1, 'px'),
        medium: Utils.convertLengthToHmm(2, 'px'),
        thick: Utils.convertLengthToHmm(3, 'px')
    };

    // maps border attribute styles to dash patterns
    var BORDER_PATTERNS = {
        dashed: [6, 2],
        dotted: [2, 2],
        dashDot: [6, 2, 2, 2],
        dashDotDot: [6, 2, 2, 2, 2, 2]
    };

    // cache for scaled border patterns, mapped by line width
    var borderPatternCache = { 1: BORDER_PATTERNS };

    // static class Border ====================================================

    var Border = {};

    // constants --------------------------------------------------------------

    /**
     * Predefined border style for 'no border'.
     *
     * @constant
     */
    Border.NONE = { style: 'none' };

    Border.HAIR_WIDTH_HMM = BORDER_WIDTH_HMM_MAP.hair;
    Border.THIN_WIDTH_HMM = BORDER_WIDTH_HMM_MAP.thin;
    Border.MEDIUM_WIDTH_HMM = BORDER_WIDTH_HMM_MAP.medium;
    Border.THICK_WIDTH_HMM = BORDER_WIDTH_HMM_MAP.thick;

    // static methods ---------------------------------------------------------

    /**
     * Returns the best CSS border style that can represent the passed line
     * style used in border attributes.
     *
     * @param {String} style
     *  The border line style, as used in formatting attributes of document
     *  operations.
     *
     * @returns {String}
     *  The best matching CSS border style.
     */
    Border.getCssBorderStyle = function (style) {
        return CSS_LINE_STYLES[style] || 'solid';
    };

    /**
     * Returns the border dash pattern for the passed line style and width,
     * intended to be used in a canvas renderer (see toolkit class Canvas).
     *
     * @param {String} style
     *  The border line style, as used in formatting attributes of document
     *  operations.
     *
     * @param {Number} width
     *  The line width, in pixels.
     *
     * @returns {Array<Number>|Null}
     *  The dash pattern for the passed line style, or null for solid lines.
     */
    Border.getBorderPattern = function (style, width) {

        // the pattern map for the passed line style
        var patternMap = null;

        // get pattern map for integral line width
        width = Math.max(1, Math.round(width));
        patternMap = borderPatternCache[width];

        // build missing pattern map for new line width
        if (!patternMap) {
            borderPatternCache[width] = patternMap = {};
            _.each(BORDER_PATTERNS, function (pattern, key) {
                patternMap[key] = _.map(pattern, function (length) {
                    return Math.floor(length * (width / 2 + 0.5));
                });
            });
        }

        // return pattern for passed line style
        return patternMap[style] || null;
    };

    /**
     * Returns the effective CSS attributes for the passed border value.
     *
     * @param {Object} border
     *  The border object as used in operations.
     *
     * @param {ThemeModel} themeModel
     *  The model of the theme used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.preview=false]
     *      If set to true, the border will be rendered for a preview element
     *      in the GUI. The border width will be restricted to two pixels in
     *      that case.
     *  @param {Object} [options.autoColor]
     *      If specified, a replacement color to be used in case the border
     *      object contains the automatic color.
     *  @param {Boolean} [options.transparentHair=false]
     *      If set to true, border lines with a resulting width less than one
     *      pixel (hair lines) will be drawn half-transparent.
     *
     * @returns {Object}
     *  A map with CSS border attributes. Contains the property 'style' with
     *  the effective CSS border style as string; the property 'width' with the
     *  effective border width in pixels (as number, rounded to entire pixels);
     *  and the property 'color' with the effective CSS color (as string with
     *  leading hash sign).
     */
    Border.getCssBorderAttributes = function (border, themeModel, options) {

        var style = Utils.getStringOption(border, 'style', 'none');
        var width = Utils.getIntegerOption(border, 'width', 0);
        var color = Utils.getObjectOption(border, 'color', Color.AUTO);
        var preview = Utils.getBooleanOption(options, 'preview', false);
        var autoColor = Utils.getObjectOption(options, 'autoColor');

        // convert operation line styles to CSS styles
        style = Border.getCssBorderStyle(style);

        // convert 1/100mm to pixels (round to one digit)
        width = (style === 'none') ? 0 : Utils.convertHmmToLength(width, 'px');

        // minimum width for borders with style 'double', maximum width for preview borders
        if ((style === 'double') && (width > 0)) {
            width = preview ? MIN_DOUBLE_LINEWIDTH : Math.max(width, MIN_DOUBLE_LINEWIDTH);
        } else if (preview) {
            width = Math.min(width, MAX_PREVIEW_LINEWIDTH);
        }

        // create real color instances
        color = Color.parseJSON(color);
        if (autoColor) { autoColor = Color.parseJSON(autoColor); }

        // add color transparency for hair lines
        if ((width > 0) && (width < 1) && Utils.getBooleanOption(options, 'transparentHair', false)) {
            color.transform('alphaMod', 50000);
        }

        // round line width to entire pixels
        width = (width > 0) ? Math.max(Math.round(width), 1) : 0;

        // resolve color object to CSS color value
        return { style: style, width: width, color: color.resolve(autoColor || 'line', themeModel).css };
    };

    /**
     * Converts the passed border attribute object to a CSS border value.
     *
     * @param {Object} border
     *  The border object as used in operations.
     *
     * @param {ThemeModel} themeModel
     *  The model of the theme used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Border.getCssBorderAttributes(). Additionally, the following options
     *  are supported:
     *  @param {Boolean} [options.clearNone=false]
     *      If set to true, the return value for invisible borders will be the
     *      empty string instead of the keyword 'none'.
     *
     * @returns {String}
     *  The CSS border value converted from the passed border object.
     */
    Border.getCssBorder = function (border, themeModel, options) {

        var visible = Border.isVisibleBorder(border);
        var attributes = visible ? Border.getCssBorderAttributes(border, themeModel, options) : null;

        return visible ? (attributes.style + ' ' + attributes.width + 'px ' + attributes.color) :
            Utils.getBooleanOption(options, 'clearNone', false) ? '' : 'none';
    };

    /**
     * Returns whether the passed value is a valid and visible border style
     * (line style different to 'none', and line width greater than 0).
     *
     * @param {Object|Null} border
     *  The border attribute value as used in operations. May be a falsy value
     *  (null or undefined) which will be interpreted as invisible border.
     *
     * @returns {Boolean}
     *  Whether the passed border attribute value is visible.
     */
    Border.isVisibleBorder = function (border) {
        return _.isObject(border) && _.isString(border.style) && (border.style !== 'none') && _.isNumber(border.width) && (border.width > 0);
    };

    /**
     * Returns whether the passed borders are equal.
     *
     * @param {Border|Null} border1
     * The first border to compare. A falsy value (null or undefined) will be
     * considered to be equal to an invisible border.
     *
     * @param {Border|Null} border2
     * The second border to compare. A falsy value (null or undefined) will be
     * considered to be equal to an invisible border.
     *
     * @returns {Boolean}
     *  Whether the two borders are equal.
     */
    Border.isEqual = function (border1, border2) {
        var visible1 = Border.isVisibleBorder(border1);
        var visible2 = Border.isVisibleBorder(border2);
        return (!visible1 && !visible2) || (
            visible1 && visible2 &&
            ((!('style' in border1) && (!('style' in border2))) || (('style' in border1) && ('style' in border2) && (border1.style === border2.style))) &&
            ((!('width' in border1) && (!('width' in border2))) || (('width' in border1) && ('width' in border2) && (border1.width === border2.width))) &&
            ((!('color' in border1) && (!('color' in border2))) || (('color' in border1) && ('color' in border2) && (Color.isEqual(border1.color, border2.color))))
        );
    };

    /**
     * Returns a predefined border width identifier for the passed line width.
     *
     * @param {Number} width
     *  The line width, in 1/100 of millimeters.
     *
     * @returns {String}
     *  The predefined line width identifier: one of 'hair', 'thin', 'medium',
     *  or 'thick'.
     */
    Border.getPresetForWidth = function (width) {
        var pixels = Utils.convertHmmToLength(width, 'px');
        return (pixels >= 2.5) ? 'thick' : (pixels >= 1.5) ? 'medium' : (pixels >= 0.75) ? 'thin' : 'hair';
    };

    /**
     * Returns the line width for a predefined border width identifier.
     *
     * @param {String} preset
     *  A predefined line width identifier: one of 'hair', 'thin', 'medium', or
     *  'thick'.
     *
     * @returns {Number}
     *  The resulting line width, in 1/100 of millimeters.
     */
    Border.getWidthForPreset = function (preset) {
        return BORDER_WIDTH_HMM_MAP[preset] || 0;
    };

    // exports ================================================================

    return Border;

});
