/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/attributeutils', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // static class AttributeUtils ============================================

    var AttributeUtils = {};

    // constants --------------------------------------------------------------

    /**
     * The internal identifier of the major scheme font in OOXML documents.
     *
     * @constant
     * @type String
     */
    AttributeUtils.MAJOR_FONT_KEY = '+mj-lt';

    /**
     * The internal identifier of the minor scheme font in OOXML documents.
     *
     * @constant
     * @type String
     */
    AttributeUtils.MINOR_FONT_KEY = '+mn-lt';

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed (incomplete) attribute set contains the
     * specified attribute, regardless of its value.
     *
     * @param {Object} attributeSet
     *  An (incomplete) attribute set to be checked for an attribute.
     *
     * @param {String} family
     *  The family of the attribute to be checked.
     *
     * @param {String} name
     *  The name of the attribute to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed attribute set contains the specified attribute.
     */
    AttributeUtils.hasAttribute = function (attributeSet, family, name) {
        return _.isObject(attributeSet[family]) && (name in attributeSet[family]);
    };

    /**
     * Returns whether the values of all attributes in the second attribute set
     * are equal to the values of the first attribute set.
     *
     * @param {Object} attributeSet1
     *  The first attribute set. It may contain more attributes than the second
     *  attribute set passed to this method.
     *
     * @param {Object} attributeSet2
     *  An (incomplete) attribute set. Each formatting attribute contained in
     *  this set must exist in the first attribute set, and their values must
     *  be equal.
     *
     * @returns {Boolean}
     *  Whether the values of all attributes in the second attribute set are
     *  equal to the values of the first attribute set.
     */
    AttributeUtils.matchesAttributesSet = function (attributeSet1, attributeSet2) {
        return _.every(attributeSet2, function (attributes2, family) {
            var attributes1 = attributeSet1[family];
            return _.isObject(attributes1) && _.isObject(attributes2) && _.every(attributes2, function (value, name) {
                return _.isEqual(value, attributes1[name]);
            });
        });
    };

    /**
     * Inserts a new attribute, or updates an existing attribute in the passed
     * (incomplete) attribute set.
     *
     * @param {Object} attributeSet
     *  An (incomplete) attribute set to be changed.
     *
     * @param {String} family
     *  The family of the attribute to be changed.
     *
     * @param {String} name
     *  The name of the attribute to be changed.
     *
     * @param {Any} value
     *  The new value of the attribute. The value null will be inserted too,
     *  see method AttributeUtils.deleteAttribute() to delete an attribute from
     *  the attribute set.
     *
     * @param {Boolean} [missing=false]
     *  If set to true, only missing attributes will be inserted; existing
     *  attributes will not be changed.
     */
    AttributeUtils.insertAttribute = function (attributeSet, family, name, value, missing) {
        var attributes = _.isObject(attributeSet[family]) ? attributeSet[family] : (attributeSet[family] = {});
        if (!missing || !(name in attributes)) { attributes[name] = value; }
    };

    /**
     * Deletes an attribute from the passed (incomplete) attribute set.
     *
     * @param {Object} attributeSet
     *  An (incomplete) attribute set to be changed.
     *
     * @param {String} family
     *  The family of the attribute to be deleted.
     *
     * @param {String} name
     *  The name of the attribute to be deleted.
     */
    AttributeUtils.deleteAttribute = function (attributeSet, family, name) {
        if (_.isObject(attributeSet[family])) {
            delete attributeSet[family][name];
        }
    };

    /**
     * Adds or removes an attribute value in the specified attribute map.
     *
     * @param {Object} attributes
     *  A simple map with formatting attributes to be updated (this MUST NOT be
     *  a structured attribute set with sub objects mapped by family!).
     *
     * @param {String} name
     *  The name of the formatting attribute to be changed.
     *
     * @param {Any} value
     *  The new value of the formatting attribute. If set to the value null,
     *  the specified attribute will be removed from the map.
     *
     * @param {Object} [autoClearDefinition]
     *  The definition descriptor of the attribute. If specified, and if the
     *  passed attribute value is equal to the default value according to the
     *  property 'def' of the definition, the attribute will be removed from
     *  the attribute map. Additionally, child attributes described in the
     *  property 'childAttrs' of the attribute definition will be removed if
     *  they do not match with the current value of the attribute changed
     *  initially. The array property 'childAttrs' of an attribute definition
     *  has been assembled from the 'parent' attribute descriptors passed to
     *  the method AttributePool.registerAttributes().
     */
    AttributeUtils.updateAttribute = function (attributes, name, value, autoClearDefinition) {

        // whether this attribute has changed
        var changed = false;

        // remove the attribute from the map, if the passed value is null or equal to the default value
        if (_.isNull(value) || (_.isObject(autoClearDefinition) && _.isEqual(value, autoClearDefinition.def))) {
            if (name in attributes) {
                delete attributes[name];
                changed = true;
            }
        } else {
            if (!_.isEqual(attributes[name], value)) {
                attributes[name] = _.copy(value, true);
                changed = true;
            }
        }

        // delete child attributes according to current attribute value
        if (changed && _.isObject(autoClearDefinition) && _.isArray(autoClearDefinition.childAttrs)) {

            // fall-back to the default value of the attribute, if null has been passed
            if (_.isNull(value)) { value = autoClearDefinition.def; }

            // remove all child attributes from the map that do not match with the current attribute
            _.each(autoClearDefinition.childAttrs, function (childAttr) {

                // the expected value of the primary attribute changed above
                var expectedValue = childAttr.value;
                // whether the current value of the primary attribute matches the expected value
                var valueMatches = _.isFunction(expectedValue) ? expectedValue(value) : _.isEqual(value, expectedValue);

                // delete secondary attribute if expected value does not match
                if (!valueMatches) { delete attributes[childAttr.name]; }
            });
        }

        return changed;
    };

    /**
     * Returns the attribute set containing all explicit attributes stored in
     * the passed DOM element.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose explicit attribute set will be returned. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.family]
     *      If specified, extracts the attributes of a specific attribute
     *      family from the attribute set. Otherwise, returns the entire
     *      attribute set object with all attributes mapped by their family.
     *  @param {Boolean} [options.direct=false]
     *      If set to true, the returned attribute set will be a reference to
     *      the original map stored in the passed DOM node, which MUST NOT be
     *      modified! By default, a deep clone of the attribute set will be
     *      returned that can be freely modified.
     *
     * @returns {Object}
     *  The (incomplete) attribute set if existing, otherwise an empty object.
     */
    AttributeUtils.getExplicitAttributes = function (node, options) {

        var // the original and complete attribute map
            attributes = $(node).data('attributes'),
            // the attribute family to be extracted from the complete map
            family = Utils.getStringOption(options, 'family'),
            // whether to return the resulting object without cloning
            direct = Utils.getBooleanOption(options, 'direct', false);

        // reduce to selected family
        if (_.isObject(attributes) && _.isString(family)) {
            attributes = (family in attributes) ? attributes[family] : undefined;
        }

        // return attributes directly or as a deep clone
        return _.isObject(attributes) ? (direct ? attributes : _.copy(attributes, true)) : {};
    };

    /**
     * Stoes the passed explicit attribute set into the DOM node.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose explicit attribute set will be changed. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {Object} attributeSet
     *  The explicit attribute set to be stored into the passed DOM node.
     */
    AttributeUtils.setExplicitAttributes = function (node, attributeSet) {
        $(node).data('attributes', attributeSet);
    };

    /**
     * Returns the identifier of the style sheet referred by the passed
     * element.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose style sheet identifier will be returned. If this
     *  object is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {String|Null}
     *  The style sheet identifier at the passed element.
     */
    AttributeUtils.getElementStyleId = function (node) {
        var styleId = AttributeUtils.getExplicitAttributes(node, { direct: true }).styleId;
        return _.isString(styleId) ? styleId : null;
    };

    /**
     * Returns whether the passed elements contain equal formatting attributes.
     *
     * @param {HTMLElement|jQuery} node1
     *  The first DOM element whose formatting attributes will be compared with
     *  the attributes of the other passed DOM element. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {HTMLElement|jQuery} node2
     *  The second DOM element whose formatting attributes will be compared
     *  with the attributes of the other passed DOM element. If this object is
     *  a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether both DOM elements contain equal explicit formatting attributes.
     */
    AttributeUtils.hasEqualElementAttributes = function (node1, node2) {
        return _.isEqual(
            AttributeUtils.getExplicitAttributes(node1, { direct: true }),
            AttributeUtils.getExplicitAttributes(node2, { direct: true })
        );
    };

    /**
     * Returns the value of the CSS property 'text-decoration' for the passed
     * character attributes.
     *
     * @param {Object} charAttrs
     *  A map of character attributes. Uses the attributes 'underline' and
     *  'strike' to generate the CSS property value.
     *
     * @returns {String}
     *  The resulting value of the CSS property 'text-decoration'.
     */
    AttributeUtils.getCssTextDecoration = function (charAttrs) {

        var result = '';

        function addToken(token) {
            if (result) { result += ' '; }
            result += token;
        }

        if (charAttrs.underline === true) { addToken('underline'); }
        if ((typeof charAttrs.strike === 'string') && (charAttrs.strike !== 'none')) { addToken('line-through'); }

        return result || 'none';
    };

    /**
     * checks if assigned character attribute have theme-relevant parameters
     * -character.fontName is minor or major font
     */
    AttributeUtils.isCharacterFontThemed = function (charAttrs) {
        var fontName = Utils.getStringOption(charAttrs, 'fontName', null);
        return (fontName === AttributeUtils.MAJOR_FONT_KEY) || (fontName === AttributeUtils.MINOR_FONT_KEY);
    };

    /**
     * checks if assigned fill attribute have theme-relevant parameters
     * -fill.color.type is scheme
     * -fill.type is gradient
     */
    AttributeUtils.isFillThemed = function (fill) {
        if (fill) { return AttributeUtils.isColorThemed(fill.color) || (fill.type && fill.type === 'gradient'); }
        return false;
    };

    /**
     * checks if assigned line attribute have theme-relevant parameters
     * -line.color.type is scheme
     */
    AttributeUtils.isLineThemed = function (line) {
        if (line) { return AttributeUtils.isColorThemed(line.color); }
        return false;
    };

    /**
     * checks if assigned color attribute have theme-relevant parameters
     * -color.type is scheme
     */
    AttributeUtils.isColorThemed = function (color) {
        if (color) { return color.type === 'scheme'; }
        return false;
    };

    // exports ================================================================

    return AttributeUtils;

});
