/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/basecontroller', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/utils/shareutils',
    'io.ox/office/tk/utils/domconsole',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/utils/baseconfig',
    'io.ox/office/baseframework/app/appobjectmixin'
], function (Utils, KeyCodes, Dialogs, DriveUtils, ShareUtils, DOMConsole, ValueMap, BaseObject, TriggerObject, Config, AppObjectMixin) {

    'use strict';

    // class ItemState ========================================================

    /**
     * A descriptor for the complete state of a controller item.
     *
     * @constructor
     *
     * @property {String} key
     *  The key of the controller item.
     *
     * @property {Boolean} enabled
     *  The enabled state of the controller item.
     *
     * @property {Any} value
     *  The current value (the result of the getter callback) of the controller
     *  item.
     *
     * @property {Array<ItemState>} parentStates
     *  The item states of all parent items of the controller item, in order of
     *  the parent keys registered for the item. If the item does not contain
     *  any parent items, this property will be an empty array.
     *
     * @property {Array<Any>} parentValues
     *  The current values (results of all getter callbacks) of the parent
     *  items, in order of the parent keys registered for the item. This
     *  property is provided for convenience (the parent values are contained
     *  in the property 'parentStates' of this instance).
     *
     * @property {Boolean} parentsEnabled
     *  The accumulated enabled state of the parent items of the controller
     *  item. Will be true, if all parent items are enabled.
     */
    function ItemState(key, parentStates) {

        this.key = key;
        this.enabled = false;
        this.value = undefined;

        // parent states and values
        this.parentStates = parentStates;
        this.parentValues = _.pluck(parentStates, 'value');

        // all parents must be enabled for 'parentsEnabled' being true
        this.parentsEnabled = parentStates.every(function (state) { return state.enabled; });

    } // class ItemState

    // singletons -------------------------------------------------------------

    var DUMMY_STATE = new ItemState('\x00', []);

    // class ControllerItem ===================================================

    var ControllerItem = BaseObject.extend({ constructor: function (controller, key, definition, stateCache) {

        // the controller containing this item
        this.controller = controller;
        // the key of this controller item
        this.key = key;
        // parent items whose values/states are needed to resolve the own value/state
        this.parentKeys = Utils.getOption(definition, 'parent');
        // handler for enabled state (default to function that returns true)
        this.enableHandler = Utils.getFunctionOption(definition, 'enable', _.constant(true));
        // handler for value getter (default to identity to forward first parent value)
        this.getHandler = Utils.getFunctionOption(definition, 'get', _.identity);
        // handler for value setter
        this.setHandler = Utils.getFunctionOption(definition, 'set', null);
        // whether an asynchronous setter is cancelable
        this.cancelable = Utils.getBooleanOption(definition, 'cancel', false);
        // whether an asynchronous setter does not block the controller
        this.nonblocking = Utils.getBooleanOption(definition, 'nonblocking', false);
        // whether the setter does not cuase a controller update
        this.silent = Utils.getBooleanOption(definition, 'silent', false);
        // behavior for returning browser focus to application
        this.preserveFocus = Utils.getBooleanOption(definition, 'preserveFocus', false);
        // custom focus target node or callback handler
        this.focusTarget = Utils.getOption(definition, 'focusTarget');
        // the shared cache of the controller for the states of all registered items
        this.stateCache = stateCache;

        // base constructor ---------------------------------------------------

        BaseObject.call(this, controller);

        // initialization -----------------------------------------------------

        // always use an array of parent keys (as strings)
        this.parentKeys =
            _.isArray(this.parentKeys) ? this.parentKeys.filter(_.isString) :
            _.isString(this.parentKeys) ? [this.parentKeys] : [];

    } }); // class ControllerItem

    // private methods  -------------------------------------------------------

    ControllerItem.prototype._updateController = function () {
        if (!this.silent) {
            this.controller.update();
        }
    };

    /**
     * Returns the state descriptor of the specified parent item.
     */
    ControllerItem.prototype._getParentState = function (parentKey) {

        // get the specified parent item
        var parentItem = this.controller._getItem(parentKey);
        if (!parentItem) {
            Utils.warn('ControllerItem._getParentState(): item "' + this.key + '" refers to unknown parent "' + parentKey + '"');
        }

        // resolve the state of the parent item
        return parentItem ? parentItem.getState() : DUMMY_STATE;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a descriptor for the current state of this item.
     *
     * @returns {ItemState}
     *  The state descriptor for this controller item.
     */
    ControllerItem.prototype.getState = function () {

        // return cached state
        var itemState = this.stateCache.get(this.key, null);
        if (itemState) { return itemState; }

        // insert the state descriptor of this item into the cache BEFORE resolving enabled state and value
        itemState = this.stateCache.insert(this.key, new ItemState(this.key, this.parentKeys.map(this._getParentState, this)));

        // resolve and store the value first (enable callback may want to query the own value)
        itemState.value = this.getHandler.apply(this, itemState.parentValues);

        // resolve and store the enabled state (pass parent values to the resolver functions)
        itemState.enabled = itemState.parentsEnabled && (this.enableHandler.apply(this, itemState.parentValues) === true);

        return itemState;
    };

    /**
     * Returns whether all parent items are enabled.
     *
     * @returns {Boolean}
     *  False, if at least one parent item is disabled, otherwise true.
     */
    ControllerItem.prototype.areParentsEnabled = function () {
        return this.getState().parentsEnabled;
    };

    /**
     * Returns whether this item is effectively enabled.
     *
     * @returns {Boolean}
     *  False, if at least one parent item is disabled, or if the
     *  enable handler of this item has returned false; otherwise true.
     */
    ControllerItem.prototype.isEnabled = function () {
        return this.getState().enabled;
    };

    /**
     * Returns the current value of the specified parent item.
     *
     * @param {Number} [index=0]
     *  The index of the parent item (can be used, if multiple parent items
     *  have been specified for this item).
     *
     * @returns {Any}
     *  The current value of the specified parent item.
     */
    ControllerItem.prototype.getParentValue = function (index) {
        var parentState = this.getState().parentStates[index || 0];
        return parentState ? parentState.value : undefined;
    };

    /**
     * Returns the current value of this item.
     *
     * @returns {Any}
     *  The current value of this item.
     */
    ControllerItem.prototype.getValue = function () {
        return this.getState().value;
    };

    /**
     * Executes the setter function of this item (passing in the new value),
     * and moves the browser focus back to the application pane.
     *
     * @param {Any} value
     *  The new value of the item.
     *
     * @param {Object} [options]
     *  Optional parameters. See method BaseController.executeItem() for
     *  details.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved or rejected according to the result of
     *  the item setter function. If the item is disabled, a rejected promise
     *  will be returned. If the item setter returns a promise, that promise
     *  will be returned. Otherwise, the return value of the item setter
     *  function will be wrapped and returned in a resolved promise.
     */
    ControllerItem.prototype.execute = function (value, options) {

        // the controller containing this item
        var controller = this.controller;
        // the document view (for busy mode)
        var docView = controller.getApp().getView();
        // the current state and value of the item
        var state = this.getState();
        // the resulting promise
        var promise = null;
        // whether the setter runs asynchronous code
        var pending = false;
        // focus options after executing the item
        var focusOptions = _.extend({}, options);

        // mix item focus options and passed focus options
        focusOptions.preserveFocus = this.preserveFocus || Utils.getBooleanOption(options, 'preserveFocus', false);
        focusOptions.focusTarget = Utils.getOption(options, 'focusTarget', this.focusTarget);

        // bind focus target callback function to the item instance
        if (_.isFunction(focusOptions.focusTarget)) {
            focusOptions.focusTarget = focusOptions.focusTarget.bind(this);
        }

        // do nothing if item is disabled
        if (state.enabled) {

            // execute the set handler, convert result of the setter callback to a promise
            promise = this.convertToPromise(this.setHandler ? this.setHandler(value, options) : null);

            // show busy screen if setter is running asynchronously
            // (but not for modal dialogs which have their own blocker overlay)
            pending = !this.nonblocking && (promise.state() === 'pending');
            if (pending && !Dialogs.isDialogOpen()) {
                var showCancel = this.cancelable && _.isFunction(promise.abort);
                docView.enterBusy({ cancelHandler: showCancel ? function () { promise.abort(); } : null });
            }
        } else {
            // item is disabled
            promise = this.createRejectedPromise();
        }

        // focus back to application
        if (!focusOptions.preserveFocus) {
            this.waitForAny(promise, function () {
                if (pending) {
                    // execute in a timeout, needed for dialogs which are closed *after* resolve/reject
                    controller.executeDelayed(function () { controller.grabFocus(focusOptions); }, 'ControllerItem.execute');
                } else {
                    controller.grabFocus(focusOptions);
                }
            });
        }

        // post-processing after the setter is finished
        Utils.logSelenium(this.key, promise.state());
        if (pending) {
            this.waitForAny(promise, function () {
                docView.leaveBusy();
                this._updateController();
                Utils.logSelenium(this.key, promise.state());
            }, this);
        } else {
            this._updateController();
        }

        return promise;
    };

    // class BaseController ===================================================

    /**
     * A controller contains a collection of items, consisting of unique key
     * and value, and providing arbitrary getter and setter methods for their
     * values.
     *
     * Triggers the following events:
     * - 'change:items'
     *      After the current state (value and/or enabled state) of at least
     *      one controller item has been changed, after the controller has
     *      updated itself or the method BaseController.update() has been
     *      invoked. Event listeners receive the descriptors of the changed
     *      items in a map with item identifiers as keys, and instances of the
     *      class ItemState as values.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends AppObjectMixin
     *
     * @param {BaseApplication} app
     *  The application that has created this controller instance.
     *
     * @param {BaseModel} docModel
     *  The document model created by the passed application.
     *
     * @param {BaseView} docView
     *  The document view created by the passed application.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Number} [initOptions.updateDelay=50]
     *      The delay for the debounced Controller.update() method, in
     *      milliseconds. Must not be less than 50.
     */
    var BaseController = TriggerObject.extend({ constructor: function (app, docModel, docView, initOptions) {

        // self reference
        var self = this;

        // all registered controller items, mapped by item key
        var itemMap = new ValueMap();

        // the delay time for the debounced BaseController.update() method
        // Bug 30479: must NOT be 0, to be able give some space for deferred
        // focus handling in form controls, for example in text fields which
        // want to commit their value before the controller will reset it.
        var updateDelay = Utils.getIntegerOption(initOptions, 'updateDelay', 50, 50);

        // cached item states, mapped by item keys
        var stateCache = new ValueMap();

        // cached values of all controller items as notified the last time
        var notifiedStates = new ValueMap();

        // all dirty keys (items that have been executed before the next update), as flag set
        var dirtyKeys = {};

        // number of item setters currently running (recursion counter)
        var runningSetters = 0;

        // shortcut definitions for 'keydown', mapped by key code (for performance)
        var keyShortcuts = {};

        // shortcut definitions for 'keypress', mapped by char code (for performance)
        var charShortcuts = {};

        // the internal deferred object representing the current update cycle
        var updateDeferred = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, app);
        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Creates a new deferred object representing the next update cycle.
         */
        function createUpdateDeferred() {

            // pending deferred object represents the current controller update cycle
            if (!updateDeferred || (updateDeferred.state() !== 'pending')) {
                updateDeferred = self.createDeferred('BaseController.updateDeferred');
            }
        }

        /**
         * Handles 'keydown' and 'keypress' events and calls the setter of this
         * item, if it contains a matching keyboard shortcut definition.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object. If a matching shortcut definition has been
         *  found, propagation of the event will be stopped, and the browser
         *  default action will be suppressed.
         */
        function keyHandler(event) {

            // whether to stop propagation and prevent the default action
            var stopPropagation = false;

            // executes the item setter defined in the passed shortcut
            function executeItemForShortcut(shortcut) {

                // the definition data of the shortcut
                var definition = shortcut.definition;

                // check whether to actually process the key event according to the event target node
                if (_.isString(definition.selector) && ($(event.target).closest(definition.selector, app.getRootNode()[0]).length === 0)) {
                    return;
                }

                // bug 30096: do not allow to execute items while a blocking operation is still running
                if (runningSetters === 0) {
                    // call value resolver function, or take constant value
                    var value = _.isFunction(definition.value) ? definition.value(self.getItemValue(shortcut.key)) : definition.value;
                    self.executeItem(shortcut.key, value, { sourceType: 'keyboard' });
                }

                // stop event propagation unless specified otherwise
                if (!Utils.getBooleanOption(definition, 'propagate', false)) {
                    stopPropagation = true;
                }
            }

            switch (event.type) {
                case 'keydown':
                    // process all shortcut definitions for the key code in the passed event
                    if (event.keyCode in keyShortcuts) {
                        _.each(keyShortcuts[event.keyCode], function (shortcut) {
                            // check if the additional modifier keys match the shortcut definition
                            if (KeyCodes.matchModifierKeys(event, shortcut.definition)) {
                                executeItemForShortcut(shortcut);
                            }
                        });
                    }
                    break;
                case 'keypress':
                    // process all shortcut definitions for the char code in the passed
                    // event, but ignore 'keypress' events in text fields
                    if ((event.charCode in charShortcuts) && !$(event.target).is('input,textarea')) {
                        _.each(charShortcuts[event.charCode], executeItemForShortcut);
                    }
                    break;
            }

            return stopPropagation ? false : undefined;
        }

        /**
         * Initializes the key handlers for all registered keyboard shortcuts.
         */
        function initializeKeyHandler() {

            var // the root node of the application window
                rootNode = app.getRootNode();

            // Bug 27528: IE does not allow to prevent default actions of some special key events
            // (especially CTRL+P which always opens the Print dialog, regardless whether the keydown
            // event has been canceled in the bubbling phase). The 'official' hack is to bind a
            // keydown event handler with IE's ancient 'attachEvent()' method, and to suppress the
            // default action by changing the key code in the event object (add smiley-with-big-eyes
            // here). Note that the 'addEventListener()' method does NOT work for this hack.
            // Bug 29544: IE 11 does not support the attachEvent() method anymore (hack for bug 27528 will fail).
            if (_.browser.IE && _.isFunction(rootNode[0].attachEvent)) {
                rootNode[0].attachEvent('onkeydown', function (event) {
                    var result = keyHandler(_.extend(event, { target: event.srcElement }));
                    // modify the key code to prevent the browser default action
                    if (result === false) { event.keyCode = 0; }
                    return result;
                });
            } else {
                rootNode.on('keydown', keyHandler);
            }

            // register 'keypress' event listener for shortcuts
            rootNode.on('keypress', keyHandler);

            // prevent that Ctrl+A shortcut selects the entire HTML document
            rootNode.on('keydown', function (event) {
                if (!$(event.target).is('input,textarea') && KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                    return false;
                }
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Updates this controller if the event source emits the specified
         * events.
         *
         * @param {Events|jQuery|HTMLElement} source
         *  The event source object. See method BaseObject.listenTo() for
         *  details.
         *
         * @param {String} [type]
         *  The type of the event to listen to. Can be a space-separated list
         *  of event type names. If the event source is an instance the mix-in
         *  class Events (e.g. a TriggerObject), this parameter can be omitted
         *  which will cause to update the controller on every event emitted by
         *  the event source (the controller will listen to the synthetic event
         *  'triggered' of the event source).
         *
         * @returns {BaseController}
         *  A reference to this instance.
         */
        this._updateOnEvent = function (source, type) {
            return this.listenTo(source, type || 'triggered', function () { self.update(); });
        };

        /**
         * Returns the specified controller item instance.
         *
         * @param {String} key
         *  The key of a controller item.
         *
         * @returns {ControllerItem|Null}
         *  The specified controller item instance; or null, if an item with
         *  the specified key does not exist.
         */
        this._getItem = function (key) {
            return itemMap.get(key, null);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the application instance that owns this document controller.
         *
         * @returns {BaseApplication}
         *  The application instance that owns this document controller.
         */
        this.getApp = function () {
            return app;
        };

        /**
         * Adds the definition for a new item to this controller.
         *
         * @param {String} key
         *  The key of the new item.
         *
         * @param {Object} definition
         *  A map with callback functions defining the behavior of the item.
         *  All callback functions will be executed in the context of the Item
         *  class instance. The following attributes are supported:
         *  @param {String|Array<String>} [definition.parent]
         *      The key of an item that will be used to calculate intermediate
         *      results for the getter function and enabler function (see
         *      below), or an array of item keys used as parents for this item.
         *      The key feature of parent items is that if a controller enables
         *      or updates multiple items at once, the getter or enabler of the
         *      same parent item registered at multiple items will be executed
         *      exactly once before the first item getter or enabler is called,
         *      and its result will be cached and passed to all item getters or
         *      enablers that are using this parent item.
         *  @param {Function} [definition.enable]
         *      Predicate function returning true if the item is enabled, and
         *      false otherwise. If one or more parent items have been
         *      specified (see above) and at least one of them returned false,
         *      the item is in disabled state already, and this function will
         *      not be called anymore. Otherwise, the cached return values of
         *      the parent getters (see option 'get' below) will be passed to
         *      this function, in the same order as specified. If the item does
         *      not specify an enable handler, it defaults to the value true,
         *      so the enabled state is dependent on the parent items only.
         *  @param {Function} [definition.get]
         *      Getter function returning the current value of the item. Can be
         *      omitted for one-way action items (actions without a return
         *      value). If one ore more parent items have been specified (see
         *      above), the cached return values of their getters will be
         *      passed to this getter, in the same order as specified. May
         *      return null to indicate an ambiguous state. Defaults to a
         *      function that returns undefined; or, if parent items have been
         *      registered, that returns the value of the first parent.
         *  @param {Function} [definition.set]
         *      Setter function changing the value of an item to the first
         *      parameter of the setter. Can be omitted for read-only items.
         *      Defaults to an empty function. May return a promise to indicate
         *      that the setter executes asynchronous code. In this case, the
         *      application window will be blocked and a busy indicator will be
         *      shown while the setter is running.
         *  @param {Boolean} [definition.nonblocking=false]
         *      If set to true, the promise returned by the setter function
         *      will be ignored. The busy blocker screen will NOT be shown.
         *  @param {Boolean} [definition.silent=false]
         *      If set to true, executing the item will not cause a controller
         *      update.
         *  @param {Boolean} [definition.cancel=false]
         *      If set to true, and if the setter function of the item has
         *      returned a pending abortable promise, the busy blocker screen
         *      will show a Cancel button that will abort that promise. It is
         *      up to the implementation of the setter function to react
         *      properly on aborting the promise. This option will be ignored,
         *      if the option 'nonblocking' is set to true.
         *  @param {Object|Array} [definition.shortcut]
         *      One or multiple keyboard shortcut definitions. If the window
         *      root node of the application receives a 'keydown' or 'keypress'
         *      event that matches a shortcut definition, the setter function
         *      of this item will be executed. Can be as single shortcut
         *      definition, or an array of shortcut definitions. Each
         *      definition object supports the following attributes:
         *      - {Number|String} [shortcut.charCode]
         *          If specified, the shortcut definition will be matched
         *          against 'keypress' events, and the value represents the
         *          numeric code point of a Unicode character, or the Unicode
         *          character as a string.
         *      - {Number|String} [shortcut.keyCode]
         *          If specified, the shortcut definition will be matched
         *          against 'keydown' events, and the value represents the
         *          numeric key code, or the upper-case name of a key code, as
         *          defined in the static class KeyCodes. Be careful with digit
         *          keys, for example, the number 9 matches the TAB key
         *          (KeyCodes.TAB), but the string '9' matches the digit '9'
         *          key (KeyCodes['9'] with the key code 57).
         *      - {Boolean|Null} [shortcut.shift=false]
         *          If set to true, the SHIFT key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the SHIFT key must not be pressed. If set to null, the
         *          current state of the SHIFT key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.alt=false]
         *          If set to true, the ALT key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the ALT key must not be pressed. If set to null, the
         *          current state of the ALT key will be ignored. Has no effect
         *          when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.ctrl=false]
         *          If set to true, the CTRL key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the CTRL key must not be pressed. If set to null, the
         *          current state of the CTRL key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.meta=false]
         *          If set to true, the META key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the META key must not be pressed. If set to null, the
         *          current state of the META key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean} [shortcut.altOrMeta=false]
         *          Convenience option that if set to true, matches if either
         *          the ALT key, or the META key are pressed. Has the same
         *          effect as defining two separate shortcuts, one with the
         *          'shortcut.alt' option set to true, and one with the
         *          'shortcut.meta' option set to true, while keeping the other
         *          option false. Must not be used in a shortcut definition
         *          where these options are set explicitly.
         *      - {Boolean} [shortcut.ctrlOrMeta=false]
         *          Convenience option that if set to true, matches if either
         *          the CTRL key, or the META key are pressed. Has the same
         *          effect as defining two separate shortcuts, one with the
         *          'shortcut.ctrl' option set to true, and one with the
         *          'shortcut.meta' option set to true, while keeping the other
         *          option false. Must not be used in a shortcut definition
         *          where these options are set explicitly.
         *      - {Any|Function} [shortcut.value]
         *          The value that will be passed to the setter function of
         *          this item. If multiple shortcuts are defined for an item,
         *          each shortcut definition may define its own value. If this
         *          option contains a function, it will receive the current
         *          value of the controller item as first parameter, and its
         *          return value will be passed to the setter function.
         *      - {Boolean} [shortcut.propagate=false]
         *          If set to true, the event will propagate up to the DOM root
         *          element, and the browser will execute its default action.
         *          If omitted or set to false, the event will be cancelled
         *          immediately after calling the setter function.
         *      - {String} [shortcut.selector]
         *          A CSS selector used to restrict the keyboard shortcut to
         *          specific DOM nodes in the application DOM. The target node
         *          of the keyboard event has to match the selector, or has to
         *          be a descendant node of a DOM node matching the selector.
         *          Otherwise, the event will be ignored completely, and the
         *          browser will execute its default action.
         *  @param {Boolean} [definition.preserveFocus=false]
         *      Whether to return the browser focus to the application pane
         *      after executing the setter function of this item (the default
         *      behavior). If set to true, the current browser focus will not
         *      be changed. Otherwise, the focus will return to the application
         *      pane after the setter function has finished (will wait for
         *      asynchronous setters before returning the focus).
         *  @param {HTMLElement|jQuery|Object|Function} [options.focusTarget]
         *      A DOM node that will receive the browser focus after the item
         *      has been executed, an object that provides a grabFocus()
         *      method, or a callback function that returns a focus target
         *      dynamically. Default focus target is the view. Has no effect,
         *      if the option 'preserveFocus' is set to true.
         *
         * @returns {BaseController}
         *  A reference to this controller instance.
         */
        this.registerDefinition = function (key, definition) {

            // destroy an existing item instance
            itemMap.with(key, function (item) { item.destroy(); });

            // create a new item instance
            itemMap.insert(key, new ControllerItem(this, key, definition, stateCache));

            // special preparations for global keyboard shortcuts
            if (_.isObject(definition.shortcut)) {
                _.getArray(definition.shortcut).forEach(function (shortcut) {

                    var keyCode = Utils.getOption(shortcut, 'keyCode');
                    var charCode = Utils.getOption(shortcut, 'charCode');

                    if (_.isString(keyCode)) {
                        keyCode = KeyCodes[keyCode] || 0;
                    }
                    if (_.isNumber(keyCode) && (keyCode > 0)) {
                        (keyShortcuts[keyCode] || (keyShortcuts[keyCode] = [])).push({ key: key, definition: shortcut });
                    }
                    if (_.isString(charCode)) {
                        charCode = charCode.charCodeAt(0);
                    }
                    if (_.isNumber(charCode) && (charCode > 0)) {
                        (charShortcuts[charCode] || (charShortcuts[charCode] = [])).push({ key: key, definition: shortcut });
                    }
                });
            }

            return this;
        };

        /**
         * Adds definitions for multiple items to this controller.
         *
         * @param {Object} definitions
         *  A map of definitions for all new items, mapped by item key. Each
         *  new controller item will be defined by calling the method
         *  BaseController.registerDefinition(). See this method for more
         *  details.
         *
         * @returns {BaseController}
         *  A reference to this controller instance.
         */
        this.registerDefinitions = function (definitions) {
            _.each(definitions, function (definition, key) {
                this.registerDefinition(key, definition);
            }, this);
            return this;
        };

        /**
         * Returns a promise that represents the current update cycle.
         *
         * @returns {jQuery.Promise}
         *  A promise that represents the current update cycle. If the promise
         *  is pending, the update cycle is not finished yet. The promise is
         *  (or will be) resolved with the states of all changed controller
         *  items, as passed with the 'change:items' event triggered by this
         *  instance.
         */
        this.getUpdatePromise = function () {
            return updateDeferred.promise();
        };

        /**
         * Refreshes the states and values of all registered items, and
         * triggers a 'change:items' event.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the controller has updated
         *  all registered items. The promise will be resolved with the states
         *  of all changed controller items, as passed with the 'change:items'
         *  event triggered by this instance.
         */
        this.update = (function () {

            // whether a full update is required without using cached states (56005)
            var fullUpdateRequired = false;

            // clear the entire cache state on every call of the update() method
            function invalidateStates(fullUpdate) {
                if (fullUpdate) { fullUpdateRequired = true; }
                stateCache.clear();
                createUpdateDeferred();
                return this.getUpdatePromise();
            }

            // after a delay, collect all item states, and notify change listeners
            function updateStates() {

                // collect all item states missing in the state cache

                // IE workaround: early exit after 3 seconds (only important for view update),
                // but freezing too long can kill RT connection and the browser itself
                if (_.browser.IE) {
                    var t0 = _.now(), i = 0;
                    itemMap.some(function (item) {
                        item.getState();
                        i = (i + 1) % 10;
                        if ((i === 0) && (_.now() - t0 > 3000)) {
                            Utils.error('BaseController.update(): performance bottleneck (early exit after 3s), please fix item getters!');
                            return true;
                        }
                    });
                } else {
                    itemMap.forEach(function (item) { item.getState(); });
                }

                /* test code for performance issues
                _.any(items, function (item, key) {
                    var now = _.now();
                    item.getState();
                    var time = _.now() - now;
                    if (time > 1) {
                        Utils.warn('BaseController updateState', key, (_.now() - now));
                    }
                });
                */

                // reduce to the properties that differ from previous state
                var changedStates = stateCache.omit(function (state, key) {
                    if (fullUpdateRequired || (key in dirtyKeys)) { return false; }
                    var prevState = notifiedStates.get(key, null);
                    return prevState && (prevState.enabled === state.enabled) && _.isEqual(prevState.value, state.value);
                });

                // return to partial update mode
                fullUpdateRequired = false;

                // bug 40770: store the values of the current cache for next update() call
                notifiedStates = stateCache.clone(function (state) {
                    return { enabled: state.enabled, value: state.value };
                });

                // notify all listeners (with locked GUI refresh for performance,
                // view elements are listening to controller items to change their visibility)
                if (!_.isEmpty(changedStates)) {
                    docView.lockPaneLayout(function () {
                        self.trigger('change:items', changedStates);
                        dirtyKeys = {};
                    });
                }

                updateDeferred.resolve(changedStates);
            }

            return self.createDebouncedMethod('BaseController.update', invalidateStates, updateStates, { delay: updateDelay });
        }());

        /**
         * Returns the current enabled state of the specified item.
         *
         * @param {String} key
         *  The key of the item.
         *
         * @returns {Boolean}
         *  Whether the specified item exists and is enabled.
         */
        this.isItemEnabled = function (key) {
            return itemMap.with(key, function (item) { return item.isEnabled(); }) || false;
        };

        /**
         * Returns the current value of the specified item.
         *
         * @param {String} key
         *  The key of the item.
         *
         * @returns {Any}
         *  The current value of the item, or undefined, if the item does not
         *  exist.
         */
        this.getItemValue = function (key) {
            return itemMap.with(key, function (item) { return item.getValue(); });
        };

        /**
         * Triggers a controller item manually. Executes the setter function of
         * the item associated to the specified key, passing in the new value.
         *
         * @param {String} key
         *  The key of the item to be changed.
         *
         * @param {Any} [value]
         *  The new value of the item.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.sourceType='custom']
         *      The cause of the item invocation. Can be one of the following
         *      types:
         *      - 'click': The item has been executed due to a button click, or
         *          another comparable GUI action such as a tap on a touch
         *          device.
         *      - 'keyboard': The item has been executed due to a standard
         *          keyboard event, e.g. pressing the ENTER key on buttons or
         *          text fields, pressing the SPACE key on buttons, or pressing
         *          the TAB key in modified text fields; or a keyboard shortcut
         *          as defined for the respective controller item.
         *      - 'custom': (default) Any other cause.
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the browser focus will not be modified after
         *      executing the item, regardless of the focus mode configured for
         *      that item.
         *  @param {HTMLElement|jQuery|Object|Function} [options.focusTarget]
         *      A DOM node that will receive the browser focus after the item
         *      has been executed, an object that provides a grabFocus()
         *      method, or a callback function that returns a focus target
         *      dynamically. Overrides the default focus target defined for the
         *      executed item (option 'focusTarget' in the item definition).
         *      Default focus target is the view. Has no effect, if the option
         *      'preserveFocus' is set to true.
         *  @param {jQuery.Event} [options.changeEvent]
         *      Set by the framework when executing a controller item after
         *      activating a GUI control element. Represents the event object
         *      of the 'group:change' event as triggered by instances of the
         *      Group class.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected according to the result
         *  of the item setter function. If the item is disabled or does not
         *  exist, a rejected promise will be returned. If the item setter
         *  returns a promise, that promise will be returned. Otherwise, the
         *  return value of the item setter function will be wrapped and
         *  returned in a resolved promise.
         */
        this.executeItem = function (key, value, options) {

            // execute setter of existing item
            var item = itemMap.get(key, null);
            if (item) {
                dirtyKeys[key] = true;
                runningSetters += 1;
                return item.execute(value, options).always(function () {
                    runningSetters -= 1;
                });
            }

            // item does not exist: return focus to application
            this.grabFocus(options);
            return $.Deferred().reject();
        };

        /**
         * Moves the browser focus back to the application pane, or to a custom
         * focus target node, if specified in the passed options.
         *
         * @param {Object} [options]
         *  Optional parameters. See method BaseController.executeItem() for
         *  details.
         *
         * @returns {BaseController}
         *  A reference to this instance.
         */
        this.grabFocus = function (options) {

            var preserveFocus = Utils.getBooleanOption(options, 'preserveFocus', false);
            var focusTarget = Utils.getOption(options, 'focusTarget');
            var changeEvent = Utils.getOption(options, 'changeEvent');
            var sourceNode = Utils.getOption(changeEvent, 'sourceNode');

            // keep current focus, if 'preserveFocus' is set
            // Bug 28214: Focus back to application if source GUI element is hidden
            // now. If the source GUI element is hidden now, IE has already changed
            // the 'document.activeElement' property, so it cannot be used to detect
            // visibility. Therefore, all group instances now insert the source DOM
            // node into the 'group:change' event.
            if (preserveFocus && (!sourceNode || $(sourceNode).is(':visible'))) {
                return this;
            }

            // resolve callback function to focus target
            if (_.isFunction(focusTarget)) {
                focusTarget = focusTarget.call(this, Utils.getStringOption(options, 'sourceType', 'custom'));
            }

            // move focus to specified target, or to the view
            if (_.isObject(focusTarget) && _.isFunction(focusTarget.grabFocus) && _.isFunction(focusTarget.isVisible) && focusTarget.isVisible()) {
                focusTarget.grabFocus(options);
            } else if (((focusTarget instanceof $) || (focusTarget instanceof HTMLElement)) && $(focusTarget).is(':visible')) {
                Utils.setFocus($(focusTarget));
            } else {
                docView.grabFocus(options);
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // create the initial update promise
        createUpdateDeferred();

        // register all controller items
        this.registerDefinitions({

            // quit/destroy the application (do not return the promise,
            // otherwise this controller would wait for its own destruction)
            'app/quit': {
                set: function () { _.defer(function () { app.quit(); }); },
                silent: true // bug 58409: no controller update when quitting
            },

            // disabled during import, enabled afterwards
            'app/imported': {
                enable: function () { return app.isImportFinished(); }
            },

            // disabled in stand-alone mode
            'app/bundled': {
                enable: function () { return !app.isStandalone(); }
            },

            // enabled if the document has NOT been stored encrypted
            'app/unencrypted': {
                enable: function () { return !app.isDocumentEncrypted(); }
            },

            'app/share': {
                parent: ['app/imported', 'app/bundled', 'app/unencrypted'],
                enable: function () { return DriveUtils.isShareable(app.getFileDescriptor()); }
            },

            // all supported operation systems have a file system where user can save in, but not on IOS
            'system/filesystem': {
                enable: function () { return !Utils.IOS; }
            },

            // download the document: enabled in read-only mode, disabled during import
            'document/download': {
                parent: ['app/imported', 'system/filesystem'],
                set: function () { return app.download(); }
            },

            // print the document (download as PDF): enabled in read-only mode, disabled during import
            'document/print': {
                parent: 'app/imported',
                enable: function () { return Config.CONVERTER_AVAILABLE; },
                set: function () { return app.print(); },
                shortcut: { keyCode: 'P', ctrlOrMeta: true }
            },

            // compose new mail with this document as attachment
            'document/sendmail': {
                parent: ['app/imported', 'app/bundled', 'app/unencrypted'],
                enable: function () { return Config.MAIL_AVAILABLE && !DriveUtils.isGuest(); },
                set: function () { return app.sendMail(); }
            },

            // compose new mail with this document's content being the mail-body's content:
            // enabled in read-only mode, disabled during import, disabled in stand-alone-mode
            'document/sendmail/inline-html': {
                parent: 'document/sendmail',
                enable: function () { return app.canSendMail(); },
                set: function (config) { return app.sendMail(config); },
                nonblocking: !!_.browser.Safari
            },

            'document/sendmail/pdf-attachment': {
                parent: 'document/sendmail',
                //enable: function () { return app.canSendMail(); },
                set: function (config) { return app.sendMail(config); }
            },

            'document/sendmail/menu': {
                parent: 'app/valid',
                enable: function () { return Config.MAIL_AVAILABLE && !DriveUtils.isGuest(); }
            },

            'document/share': {
                parent: 'app/share',
                enable: function () { return ShareUtils.isSharingSupported(); }
            },

            'document/share/getlink': {
                parent: 'document/share',
                enable: function () { return ShareUtils.canShareLink(); },
                set: function () { return ShareUtils.shareLink(app.getFileDescriptor());  }
            },

            'document/share/inviteuser': {
                parent: 'document/share',
                enable: function () { return ShareUtils.canInviteUser(); },
                set: function () { return ShareUtils.inviteUser(app.getFileDescriptor()); }
            },

            // debug mode

            'debug/enabled': {
                parent: 'app/valid',
                enable: function () { return Config.DEBUG; }
            },

            'debug/console': {
                parent: 'debug/enabled',
                get: function () { return DOMConsole.isVisible(); },
                set: function (state) { DOMConsole.toggle(state); }
            }
        });

        // update once after successful import
        this.waitForImportSuccess(function () {
            initializeKeyHandler();
            this._updateOnEvent(docView, 'refresh:layout');
            this._updateOnEvent(DOMConsole, 'change:state');
            this.update();
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            itemMap.destroyElements();
            self = docModel = docView = null;
            itemMap = stateCache = notifiedStates = null;
            keyShortcuts = charShortcuts = null;
        });

    } }); // class BaseController

    // exports ================================================================

    return BaseController;

});
