/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/model/formula/funcs/operators',
    'io.ox/office/spreadsheet/model/formula/funcs/complexfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/conversionfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/databasefuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/datetimefuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/engineeringfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/financialfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/informationfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/logicalfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/mathfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/matrixfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/referencefuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/statisticalfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/textfuncs',
    'io.ox/office/spreadsheet/model/formula/funcs/webfuncs'
], function (Labels, Operators) {

    'use strict';

    var functionModules = _.toArray(arguments).slice(2);
    var Functions = _.extend.apply(_, [{}].concat(functionModules));

    // private helpers ========================================================

    // identifiers of all supported file formats
    var FILE_FORMATS = ['ooxml', 'odf'];

    // regular expressions for valid property values
    var FILE_FORMAT_RE = new RegExp('^(' + FILE_FORMATS.join('|') + ')$');
    var CATEGORY_PATTERN = '(' + _.keys(Labels.FUNCTION_CATEGORY_LABELS).join('|') + ')';
    var CATEGORY_RE = new RegExp('^' + CATEGORY_PATTERN + '( ' + CATEGORY_PATTERN + ')*$');
    var SIG_PARAM_PATTERN = '(val(:(num|int|date|day|str|bool|comp|any))?|mat(:(num|str|bool|any))?|ref(:(sheet|single|multi|val))?|any(:lazy)?)(\\|deps:(skip|pass)|\\|mat:(force|pass|forward))*';
    var SIGNATURE_RE = new RegExp('^' + SIG_PARAM_PATTERN + '( ' + SIG_PARAM_PATTERN + ')*$');

    function isObject(value) {
        return _.isObject(value) && !_.isArray(value) && !_.isFunction(value);
    }

    function testFormatCategory(desc) {
        if (!('format' in desc)) { return; }
        expect(desc.type).to.match(/^(val|mat)/);
        expect(desc.format).to.match(/^(percent|currency|date|time|datetime|combine)$/);
    }

    function testPropertyNames(desc, re) {
        _.each(desc, function (value, key) {
            expect(key).to.match(re);
            if (isObject(value)) {
                _.each(value, function (v, format) {
                    expect(format).to.match(FILE_FORMAT_RE);
                });
            }
        });
    }

    function resolveDescriptor(desc, format) {
        var result = {};
        _.each(desc, function (value, key) {
            if (isObject(value)) {
                if (format in value) {
                    result[key] = value[format];
                }
            } else {
                result[key] = value;
            }
        });
        return result;
    }

    function testOperatorNameProperty(desc, key, format, outMap) {
        var names = desc.name;
        var outArray = outMap[format] || (outMap[format] = []);
        if (_.isString(names)) { names = [names]; }
        expect(names).to.be.an('array');
        expect(names.length).to.be.above(0);
        names.forEach(function (name) {
            expect(name).to.match(/^[^A-Z0-9._]{1,2}$/i);
            outArray.push(name);
        });
    }

    function testFunctionNameProperty(desc, key, format, outMap) {
        var outArray = outMap[format] || (outMap[format] = []);
        if ('name' in desc) {
            var names = desc.name;
            if (_.isNull(names)) { return; }
            if (_.isString(names)) { names = [names]; }
            expect(names).to.be.an('array');
            expect(names.length).to.be.above(0);
            names.forEach(function (name) {
                expect(name).to.match(/^(_xlfn\.)?[A-Z][A-Z0-9]*([._][A-Z0-9]+)*$/);
            });
            outArray = outArray.concat(names);
        } else {
            outArray.push(key);
        }
        outMap[format] = outArray;
    }

    function forEachOperator(callback) {
        FILE_FORMATS.forEach(function (format) {
            _.each(Operators, function (descriptor, key) {
                callback(descriptor, key, format);
            });
        });
    }

    function forEachFunction(callback) {
        FILE_FORMATS.forEach(function (format) {
            functionModules.forEach(function (module) {
                _.each(module, function (descriptor, key) {
                    callback(descriptor, key, format);
                });
            });
        });
    }

    // tests ==================================================================

    describe('Spreadsheet operator descriptor module', function () {
        it('should exist', function () {
            expect(Operators).to.be.an('object');
        });
        it('should be a map of descriptors', function () {
            _.each(Operators, function (descriptor) {
                expect(descriptor).to.be.an('object');
            });
        });
    });

    describe('Spreadsheet function descriptor modules', function () {
        it('should exist', function () {
            expect(functionModules).to.be.an('array');
        });
        it('should be maps of descriptors', function () {
            functionModules.forEach(function (descriptors) {
                _.each(descriptors, function (descriptor) {
                    expect(descriptor).to.be.an('object');
                });
            });
        });
    });

    describe('Spreadsheet operator descriptors', function () {
        var keys = _.keys(Operators);
        it('should have valid resource keys', function () {
            keys.forEach(function (key) {
                expect(key).to.match(/^[a-z]+$/);
            });
        });
        it('should have unique resource keys', function () {
            expect(keys.length).to.equal(_.unique(keys).length);
        });
    });

    describe('Spreadsheet function descriptors', function () {
        var keys = _.flatten(_.map(functionModules, _.keys), true);
        it('should have valid resource keys', function () {
            keys.forEach(function (key) {
                expect(key).to.match(/^[A-Z][A-Z0-9]*(\.[A-Z0-9]+)*$/);
            });
        });
        it('should have unique resource keys', function () {
            expect(keys.length).to.equal(_.unique(keys).length);
        });
    });

    describe('Spreadsheet operator descriptors', function () {
        it('should be objects', function () {
            FILE_FORMATS.forEach(function () {
                _.keys(Operators).forEach(function (key) {
                    expect(Operators).to.have.a.property(key).that.is.an('object');
                });
            });
        });
        it('should not contain unknown properties', function () {
            forEachOperator(function (descriptor) {
                testPropertyNames(descriptor, /^(name|(min|max)Params|type|format|recalc|signature|resolve)$/);
            });
        });
        var namesMap = {};
        it('should contain correct properties', function () {
            forEachOperator(function (descriptor, key, format) {
                var desc = resolveDescriptor(descriptor, format);
                testOperatorNameProperty(desc, key, format, namesMap);
                expect(desc).to.have.a.property('minParams').that.is.a('number').within(1, 2);
                expect(desc).to.have.a.property('maxParams').that.is.a('number').within(desc.minParams, 2);
                expect(desc).to.have.a.property('type').that.match(/^(val:(num|str|bool|any)|ref)$/);
                testFormatCategory(desc);
                if ('recalc' in desc) { expect(desc.recalc).to.match(/^(always|once)$/); }
                if ('signature' in desc) { expect(desc.signature).to.match(SIGNATURE_RE); }
                if ('resolve' in desc) { expect(_.isFunction(desc.resolve) || _.isString(desc.resolve)).to.equal(true); }
                if (_.isString(desc.resolve)) {
                    expect(desc.resolve).to.not.equal(key);
                    expect(Operators).to.contain.a.property(desc.resolve);
                }
            });
        });
        it('should have unique names', function () {
            _.forEach(namesMap, function (names) {
                expect(names.length).to.equal(_.unique(names).length + 2); // unary and binary plus/minus
            });
        });
    });

    describe('Spreadsheet function descriptors', function () {
        it('should be objects', function () {
            FILE_FORMATS.forEach(function () {
                functionModules.forEach(function (functionModule) {
                    _.keys(functionModule).forEach(function (key) {
                        expect(Functions).to.have.a.property(key).that.is.an('object');
                    });
                });
            });
        });
        it('should not contain unknown properties', function () {
            forEachFunction(function (descriptor) {
                testPropertyNames(descriptor, /^(category|name|hidden|(min|max|repeat)Params|type|format|recalc|signature|resolve|relColRef|relRowRef)$/);
            });
        });
        var namesMap = {};
        it('should contain correct properties', function () {
            forEachFunction(function (descriptor, key, format) {
                var desc = resolveDescriptor(descriptor, format);
                expect(desc).to.have.a.property('category').that.match(CATEGORY_RE);
                testFunctionNameProperty(desc, key, format, namesMap);
                if ('hidden' in desc) { expect(desc.hidden).to.be.a('boolean'); }
                expect(desc).to.have.a.property('minParams').that.is.a('number').at.least(0);
                if ('maxParams' in desc) { expect(desc.maxParams).to.be.a('number').at.least(desc.minParams); }
                if ('repeatParams' in desc) { expect(desc.repeatParams).to.be.a('number').at.least(1); }
                expect(!('maxParams' in desc) || !('repeatParams' in desc)).to.equal(true);
                expect(desc).to.have.a.property('type').that.match(/^(val:(num|str|bool|err|date|comp|any)|mat|ref|any)$/);
                testFormatCategory(desc);
                if ('recalc' in desc) { expect(desc.recalc).to.match(/^(always|once)$/); }
                if ('signature' in desc) { expect(desc.signature).to.match(SIGNATURE_RE); }
                if ('resolve' in desc) {
                    expect(_.isFunction(desc.resolve) || _.isString(desc.resolve)).to.equal(true);
                    if (desc.maxParams !== 0) { expect(desc).to.have.a.property('signature'); }
                }
                if (_.isString(desc.resolve)) {
                    expect(desc.resolve).to.not.equal(key);
                    expect(Functions).to.contain.a.property(desc.resolve);
                }
                if ('relColRef' in desc) { expect(desc.relColRef).to.be.a('function'); }
                if ('relRowRef' in desc) { expect(desc.relRowRef).to.be.a('function'); }
            });
        });
        it('should have unique names', function () {
            _.forEach(namesMap, function (names) {
                expect(names.length).to.equal(_.unique(names).length);
            });
        });
    });

    // ========================================================================
});
