/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/textfield', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/class',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/tk/control/group',
    'io.ox/office/tk/control/widthmixin'
], function (Utils, KeyCodes, Forms, Class, LocaleData, Parser, Formatter, Group, WidthMixin) {

    'use strict';

    // private global functions ===============================================

    /**
     * Returns a regular expression that matches numeric input strings, as used
     * by the NumberValidator class.
     *
     * @param {Boolean} neg
     *  Whether the regular expression will accept a leading minus sign.
     *
     * @param {Boolean} int
     *  Whether the regular expression will accept integers only.
     *
     * @returns {RegExp}
     *  A regular expression that matches numeric input strings.
     */
    var getNumberRE = (function () {

        // RE pattern for fractional part
        var FRAC_PATTERN = '(' + _.escapeRegExp(LocaleData.DEC) + '[0-9]*)?';
        // regular expression for non-negative integers (including empty string)
        var POS_INT_RE = /^[0-9]*$/;
        // regular expression for integers (including empty string)
        var NEG_INT_RE = /^-?[0-9]*$/;
        // regular expression for non-negative floating-point numbers (including empty string)
        var POS_NUM_RE = new RegExp('^[0-9]*' + FRAC_PATTERN + '$');
        // regular expression for floating-point numbers (including empty string)
        var NEG_NUM_RE = new RegExp('^-?[0-9]*' + FRAC_PATTERN + '$');

        return function (neg, int) {
            return int ? (neg ? NEG_INT_RE : POS_INT_RE) : (neg ? NEG_NUM_RE : POS_NUM_RE);
        };
    }());

    // class Validator ========================================================

    /**
     * Base class for text field validators used to convert between values and
     * field texts, and to validate the text field while editing. Provides a
     * default implementation of all methods that do not restrict editing.
     *
     * @constructor
     */
    var Validator = Class.extendable(function () {});

    // public methods ---------------------------------------------------------

    /**
     * Converts the passed value to a text string to be inserted into a text
     * field. Intended to be overwritten by derived classes. This default
     * implementation returns the passed value, if it is a string, otherwise an
     * empty string.
     *
     * @param {Any} value
     *  The value to be converted to a text.
     *
     * @returns {String}
     *  The text converted from the passed value, or an empty string, if the
     *  passed value cannot be converted to text.
     */
    Validator.prototype.valueToText = function (value) {
        return (typeof value === 'string') ? value : '';
    };

    /**
     * Converts the passed text to a value that will be passed to all event
     * listeners of a text field. Intended to be overwritten by derived
     * classes. This default implementation returns the unmodified text.
     *
     * @param {String} text
     *  The text to be converted to a value.
     *
     * @returns {Any}
     *  The value converted from the passed text. The value null indicates that
     *  the text cannot be converted to a valid value.
     */
    Validator.prototype.textToValue = function (text) {
        return text;
    };

    /**
     * Validates the passed text that has been changed while editing a text
     * field. It is possible to return a new string value that will be inserted
     * into the text field, or a boolean value indicating whether to restore
     * the old state of the text field. Intended to be overwritten by derived
     * classes. This default implementation does not change the text field.
     *
     * @param {String} text
     *  The current contents of the text field to be validated.
     *
     * @returns {String|Boolean}
     *  When returning a string, the text field will be updated to contain the
     *  returned value while restoring its selection (browsers may destroy the
     *  selection when changing the text). When returning the value false, the
     *  previous state of the text field (as it was after the last validation)
     *  will be restored. Otherwise, the value of the text field is considered
     *  to be valid and will not be modified.
     */
    Validator.prototype.validate = _.noop;

    // class TextValidator ====================================================

    /**
     * A validator for text fields that restricts the allowed values according
     * to the passed options.
     *
     * @constructor
     *
     * @extends Validator
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  - {Number} [initOptions.maxLength=0x7FFFFFFF]
     *      The maximum number of characters to be inserted into the text
     *      field. All attempts to insert more characters will be rejected.
     */
    var TextValidator = Validator.extend(function (initOptions) {

        // base constructor
        Validator.call(this);

        // maximum length
        this._maxLength = Utils.getIntegerOption(initOptions, 'maxLength', 0x7FFFFFFF, 0, 0x7FFFFFFF);

    }); // class TextValidator

    // public methods ---------------------------------------------------------

    /**
     * Converts the passed value to a text string that will be restricted to
     * the size passed to the constructor of this instance.
     *
     * @param {Any} value
     *  The value to be converted to a text.
     *
     * @returns {String}
     *  The text converted from the passed value (shortened to the maximum
     *  length), or an empty string, if the passed value cannot be converted to
     *  text.
     */
    TextValidator.prototype.valueToText = function (value) {
        return (typeof value === 'string') ? value.substr(0, this._maxLength) : '';
    };

    /**
     * Returns whether the length of the passed text does not exceed the
     * maximum text length of this instance.
     *
     * @param {String} text
     *  The current contents of the text field to be validated.
     *
     * @returns {Boolean}
     *  Whether the length of the passed text does not exceed the maximum text
     *  length of this instance.
     */
    TextValidator.prototype.validate = function (text) {
        return text.length <= this._maxLength;
    };

    // class NumberValidator ==================================================

    /**
     * A validator for text fields that restricts the allowed values to
     * floating-point numbers.
     *
     * @constructor
     *
     * @extends Validator
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  - {Number} [initOptions.min=-Number.MAX_VALUE]
     *      The minimum value allowed to enter.
     *  - {Number} [initOptions.max=Number.MAX_VALUE]
     *      The maximum value allowed to enter.
     *  - {Number} [initOptions.precision=1]
     *      The precision used to round the current number. The number will be
     *      rounded to multiples of this value. If omitted, the default
     *      precision 1 will round to integers. Must be positive.
     */
    var NumberValidator = Validator.extend(function (initOptions) {

        // base constructor
        Validator.call(this);

        // minimum, maximum, and precision
        this._min = Utils.getNumberOption(initOptions, 'min', -Number.MAX_VALUE, -Number.MAX_VALUE, Number.MAX_VALUE);
        this._max = Utils.getNumberOption(initOptions, 'max', Number.MAX_VALUE, this._min, Number.MAX_VALUE);
        this._prec = Utils.getNumberOption(initOptions, 'precision', 1, 0);
        this._regex = getNumberRE(this._min < 0, this._prec === Math.round(this._prec));

    }); // NumberValidator

    // public methods ---------------------------------------------------------

    /**
     * Converts the passed value to a text string to be inserted into a text
     * field.
     *
     * @param {Any} value
     *  The value to be converted to a text. If not a finite number, this
     *  method returns an empty string.
     *
     * @returns {String}
     *  The text converted from the passed number, or an empty string, if the
     *  passed value is not a finite number.
     */
    NumberValidator.prototype.valueToText = function (value) {
        return Utils.isFiniteNumber(value) ? Formatter.formatDecimal(Utils.round(value, this._prec)) : '';
    };

    /**
     * Tries to convert the passed text to a number.
     *
     * @param {String} text
     *  The text to be converted to a number.
     *
     * @returns {Number|Null}
     *  The number converted from the passed text; or null to indicate that the
     *  text cannot be converted to a number.
     */
    NumberValidator.prototype.textToValue = function (text) {
        var number = Parser.parseDecimal(text);
        return (isFinite(number) && (this._min <= number) && (number <= this._max)) ? Utils.round(number, this._prec) : null;
    };

    /**
     * Returns whether the passed text can be converted to a number.
     *
     * @param {String} text
     *  The current contents of the text field to be validated.
     *
     * @returns {Boolean}
     *  Whether the passed text is empty, or can be converted to a number.
     */
    NumberValidator.prototype.validate = function (text) {
        return this._regex.test(text);
    };

    /**
     * Returns the lower limit of this validator.
     *
     * @returns {Number}
     *  The lower limit of this validator.
     */
    NumberValidator.prototype.getMin = function () {
        return this._min;
    };

    /**
     * Returns the upper limit of this validator.
     *
     * @returns {Number}
     *  The upper limit of this validator.
     */
    NumberValidator.prototype.getMax = function () {
        return this._max;
    };

    /**
     * Returns the precision of this validator used to round numbers.
     *
     * @returns {Number}
     *  The precision of this validator used to round numbers.
     */
    NumberValidator.prototype.getPrecision = function () {
        return this._prec;
    };

    /**
     * Returns the passed number bound to the lower and upper limit.
     *
     * @param {Any} value
     *  The value to be converted. If not a finite number, it will be returned
     *  as passed.
     *
     * @returns {Any}
     *  The passed number bound to the lower and upper limit. Values of other
     *  types will be returned as passed.
     */
    NumberValidator.prototype.restrictValue = function (value) {
        return Utils.isFiniteNumber(value) ? Utils.minMax(value, this._min, this._max) : value;
    };

    // global singletons ======================================================

    // global instance of the default validator
    var defaultValidator = new Validator();

    // class TextField ========================================================

    /**
     * Creates a container element used to hold a text input field.
     *
     * @constructor
     *
     * @extends Group
     * @extends WidthMixin
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application owning the
     *  text field object, or an object with a method 'getWindowId' that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Group base class
     *  constructor, all formatting options for input elements supported by the
     *  method Forms.createInputMarkup(), and all options of the mix-in class
     *  WidthMixin. Additionally, the following options are supported:
     *  - {Boolean} [initOptions.select=false]
     *      If set to true, the entire text will be selected after the text
     *      field has been clicked, or if the method TextField.grabFocus() has
     *      been called. The text will always be selected (independently from
     *      this option), if the text field gets focus via keyboard navigation.
     *  - {Validator} [initOptions.validator]
     *      A text validator that will be used to convert the values from the
     *      control values (passed to the Group.setValue() method) to the text
     *      representation used in this text field, to validate the text while
     *      typing in the text field, and to convert the entered text to the
     *      control value (returned by the Group.getValue() method). If no
     *      validator has been specified, a default validator will be used that
     *      does not perform any conversions.
     *  - {Object} [initOptions.smallerVersion]
     *      If specified, the Textfield will known, how it should act, if there
     *      will be not enough free place to show normal view
     *  - {Boolean} [initOptions.showClearButton=false]
     *      If set to true, a clear button is visible to clear the text in the
     */
    function TextField(windowId, initOptions) {

        // self reference
        var self = this;

        // the <input> element
        var inputNode = $(Forms.createInputMarkup(initOptions));

        // create the wrapper node for the input field (IE renders input fields with padding wrong)
        var wrapperNode = $('<div class="input-wrapper">').append(inputNode);

        // whether to select the entire text on click
        var select = Utils.getBooleanOption(initOptions, 'select', false);

        // the validator used to convert and validate values
        var validator = Utils.getObjectOption(initOptions, 'validator', defaultValidator);

        // saved state of the text field, used to restore while validating
        var inputState = null;

        // initial value of text field when focused, needed for ESCAPE key handling
        var initialText = null;

        // whether the group is focused (needed to detect whether to restore selection after focus change inside the group)
        var groupFocused = false;

        // saves the (old) styles in case of switching to small version
        var savedStyles = {};

        // smaller version of group (to fit in smaller resolutions)
        var smallerVersion = Utils.getOption(initOptions, 'smallerVersion', false);

        // the clear button to clear the textbox
        var clearButtonNode = null;

        // base constructor ---------------------------------------------------

        Group.call(this, windowId, initOptions);
        WidthMixin.call(this, wrapperNode, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Saves the current value and selection of the text field node.
         */
        function saveInputState() {
            inputState = Forms.getInputSelection(inputNode);
            inputState.text = inputNode.val();
        }

        /**
         * Restores the current value and selection of the text field.
         */
        function restoreInputState() {
            inputNode.val(inputState.text);
            Forms.setInputSelection(inputNode, inputState.start, inputState.end);
        }

        /**
         * Selects the entire text of the input node.
         */
        function selectAll() {

            // Bug 43829: Workaround to select the whole text when 'touching' in the input field on iOS,
            // because select() doesn't work. It's an iOS problem.
            // see http://stackoverflow.com/questions/3272089/programmatically-selecting-text-in-an-input-field-on-ios-devices-mobile-safari
            if (Utils.IOS) {
                self.executeDelayed(function () {
                    inputNode[0].setSelectionRange(0, 9999);
                }, 'TextField.selectAll', 1);
                return;
            }

            // this is how selecting the whole text is supposed to work
            inputNode.select();
        }

        /**
         * The update handler for this text field.
         */
        function updateHandler(value) {
            value = (value === null) ? '' : validator.valueToText(value);
            // do not access the input element if the value does not change (browsers will destroy current selection)
            if (inputNode.val() !== value && initialText !== value) {
                inputNode.val(value);
                self.getNode().attr('data-value', value);
                saveInputState();
            }
        }

        /**
         * Handles all focus events of the text field.
         */
        function focusHandler(event) {

            switch (event.type) {
                case 'group:focus':
                    // save current value, if this group receives initial focus
                    initialText = inputState.text = inputNode.val();
                    // wait until the <input> element gets focused (see below)
                    groupFocused = false;
                    break;

                case 'beforedeactivate':
                    // Bug 27912: IE keeps text selection visible when losing focus (text
                    // is still highlighted although the control is not focused anymore).
                    // The 'beforedeactivate' event is triggered by IE only, before the
                    // focus leaves the text field. Do not interfere with focus handling
                    // in 'blur' events, e.g. Chrome gets confused when changing browser
                    // text selection while it currently changes the focus node.
                    if (_.browser.IE) {
                        saveInputState();
                        Forms.setInputSelection(inputNode, 0);
                    }
                    break;

                case 'blur':
                    // save field state to be able to restore selection (but not in IE where the
                    // selection has been cleared, see the 'beforedeactivate' event handler above)
                    if (!_.browser.IE) {
                        saveInputState();
                    }
                    break;

                case 'group:blur':
                    // Bug 27175: always commit value when losing focus
                    if (_.isString(initialText) && (initialText !== inputNode.val())) {
                        // pass preserveFocus option to not interfere with current focus handling
                        self.triggerChange(self.getFieldValue(), { sourceEvent: event, preserveFocus: true });
                    }
                    initialText = null;
                    groupFocused = false;
                    break;
            }
        }

        /**
         * Handles global 'change:focus' events triggered by the Utils module.
         * Updates the internal state of this instance after the <input> DOM
         * element has received or lost the browser focus. Selects the entire
         * text according to the focus cause (mouse versus keyboard).
         */
        function globalFocusHandler(event, focusNode, blurNode, keyEvent) {

            // the <input> element has received browser focus
            if (inputNode.is(focusNode)) {

                // <input> element focused again (e.g. from drop-down menu): restore selection
                if (groupFocused) {
                    restoreInputState();
                    return;
                }

                // on first focus for the <input> element: remember that this entire
                // group is focused, until the 'group:blur' event will be triggered
                groupFocused = true;

                // remove previous 'mouseup' handler (see below)
                inputNode.off('mouseup');

                // always select entire text when reaching the field with keyboard, or when specified via option
                if (keyEvent || select) {
                    selectAll();

                    // Bug 53025: Suppress next 'mouseup' event to prevent that the browser (especially MS Edge, see
                    // https://developer.microsoft.com/en-us/microsoft-edge/platform/issues/8229660/) changes the selection
                    // to a cursor. Simply preventing default browser action ('preventDefault') does not work though.
                    if (!keyEvent) {
                        inputNode.one('mouseup', selectAll);
                    }
                }

                // save current contents and selection
                saveInputState();
                return;
            }

            // the <input> element has lost browser focus
            if (inputNode.is(blurNode)) {
                // always remove the dummy 'mouseup' handler when losing focus (see above)
                inputNode.off('mouseup');
            }
        }

        /**
         * Handles enable/disable events of the group.
         */
        function enableHandler(event, state) {
            inputNode.attr('readonly', state ? null : 'readonly');
        }

        /**
         * Handles 'group:change' events triggered by any action, also by
         * additional controls from sub classes which are not under control of
         * this instance. Prevents triggering a second change event caused by
         * the 'group:blur' event.
         */
        function changeHandler() {
            // prevent another trigger from group:blur handler
            initialText = null;
        }

        /**
         * Handles keyboard events.
         */
        function keyHandler(event) {

            switch (event.keyCode) {
                case KeyCodes.ENTER:
                    if (event.type === 'keyup') {
                        self.triggerChange(self.getFieldValue(), { sourceEvent: event });
                    }
                    return false;
                case KeyCodes.ESCAPE:
                    if (event.type === 'keydown') {
                        inputNode.val(initialText);
                        inputHandler();
                    }
                    break;
                case KeyCodes.HOME:
                case KeyCodes.END:
                    // do not let these key codes bubble up, just do the default action
                    event.stopPropagation();
                    break;
            }
        }

        /**
         * Handles input events triggered when the text changes while typing.
         * Performs live validation with the current validator.
         */
        function inputHandler() {

            // current text in the input element
            var currText = inputNode.val();
            // previous text contents of the input element
            var prevText = inputState.text;

            if (clearButtonNode) {
                clearButtonNode.toggle(currText.length > 0);
            }

            // do not perform validation if nothing has changed
            if (currText === prevText) { return; }

            // validate the current field text
            var result = validator.validate(currText);

            // update the text field according to the validation result
            if (result === false) {
                // false: restore the old field state
                restoreInputState();
            } else if (_.isString(result) && (result !== currText)) {
                // insert the validation result and restore the old selection
                inputState.text = result;
                restoreInputState();
            }

            // trigger 'group:validate' event to listeners, pass old text
            self.trigger('group:validate', prevText);

            // update current state of the text field
            saveInputState();
        }

        /**
         * Activates/Deactivates a smaller version of the text field
         *
         * @param {Boolean} value
         *  decides whether the text field should be small or wide
         */
        function smallTextfield(value) {
            // show the small version
            if (value === true) {
                // set new css if exists
                if (_.isObject(smallerVersion.css)) {
                    if (_.isEmpty(savedStyles.css)) { savedStyles.css = wrapperNode.attr('style'); }
                    wrapperNode.css(smallerVersion.css);
                }

                // hide complete label
                if (smallerVersion.hide) { self.getNode().addClass('hidden'); }

            // show the default
            } else {
                // re-set the old (saved) styles
                if (_.has(savedStyles, 'css') && !_.isNull(savedStyles.css)) {
                    wrapperNode.removeAttr('style');
                    wrapperNode.attr('style', savedStyles.css);
                    savedStyles.css = null;
                }

                // hide complete label
                if (smallerVersion.hide) { self.getNode().removeClass('hidden'); }
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the text control element, as jQuery object.
         */
        this.getInputNode = function () {
            return inputNode;
        };

        /**
         * Returns the effective value represented by the current text in the
         * text input control.
         *
         * @returns {Any}
         *  The value represented by the current text, as returned by the
         *  validator of this control group.
         */
        this.getFieldValue = function () {
            return validator.textToValue(inputNode.val());
        };

        /**
         * Sets the effective value to be displayed in the text input control.
         *
         * @param {Any} value
         *  The value to be displayed. The input control will display the text
         *  for this value as returned by the validator of this control group.
         *
         * @returns {TextField}
         *  A reference to this instance.
         */
        this.setFieldValue = function (value) {
            inputNode.val(validator.valueToText(value));
            return this;
        };

        /**
         * Converts the passed value to a text using the validator of this
         * control group.
         */
        this.valueToText = function (value) {
            return validator.valueToText(value);
        };

        /**
         * Converts the passed text to a value using the validator of this
         * control group.
         */
        this.textToValue = function (text) {
            return validator.textToValue(text);
        };

        /**
         * Sets the browser focus into the text input element.
         *
         * @returns {TextField}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            Utils.setFocus(inputNode);
            return this;
        };

        /**
         * Overwrites the base-methods (group.js) to
         * activate/deactivate the small version of the button
         */
        this.activateSmallVersion = function () {
            if (_.isObject(smallerVersion)) {
                smallTextfield(true);
            }
        };
        this.deactivateSmallVersion = function () {
            if (_.isObject(smallerVersion)) {
                smallTextfield(false);
            }
        };

        // initialization -----------------------------------------------------

        if (Utils.getBooleanOption(initOptions, 'showClearButton', false)) {
            clearButtonNode = $(Forms.createIconMarkup('fa-times-circle', { classes: 'clear-button' }));
            inputNode.css('padding-right', '12px');
            clearButtonNode.hide();
            wrapperNode.append(clearButtonNode);

            clearButtonNode.on('click', function (event) {
                clearButtonNode.hide();
                inputNode.val(initialText);
                self.triggerChange('', { sourceEvent: event });
            });
        }

        Forms.setToolTip(this.getNode(), initOptions);

        // add special marker class used to adjust formatting
        this.getNode().addClass('text-field');

        // insert the text field into this group
        this.addChildNodes(wrapperNode);

        // update text contents in the input field
        this.registerUpdateHandler(updateHandler);

        // register event handlers
        this.on({
            'group:focus group:blur': focusHandler,
            'group:enable': enableHandler,
            'group:change': changeHandler
        });

        inputNode.on({
            'focus beforedeactivate blur': focusHandler,
            'keydown keypress keyup': keyHandler,
            input: inputHandler
        });

        // listen to global 'change:focus' events, to be able to distinguish between mouse and keyboard focus
        this.listenTo(Utils, 'change:focus', globalFocusHandler);

        // forward clicks on the wrapper node (left/right padding beside the input field)
        wrapperNode.on('click', function (event) {
            if (self.isEnabled() && wrapperNode.is(event.target)) {
                Utils.setFocus(inputNode);
            }
        });

        // initialize field state
        saveInputState();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = inputNode = wrapperNode = validator = clearButtonNode = null;
        });

    } // class TextField

    // exports ================================================================

    // export validators as child classes
    TextField.Validator = Validator;
    TextField.TextValidator = TextValidator;
    TextField.NumberValidator = NumberValidator;

    // derive this class from class Group
    return Group.extend({ constructor: TextField });

});
