/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/model/operationgenerator', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/operationgenerator',
    'io.ox/office/drawinglayer/utils/imageutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/textutils'
], function (Utils, AttributeUtils, OperationGenerator, ImageUtils, DrawingFrame, Operations, DOM, Position, TextUtils) {

    'use strict';

    // class TextOperationGenerator ===========================================

    /**
     * An instance of this class contains an operations array and provides
     * methods to generate operations for various element nodes.
     *
     * @constructor
     *
     * @extends OperationGenerator
     *
     * @param {TextModel} docModel
     *  The document model containing this generator.
     */
    var TextOperationGenerator = OperationGenerator.extend(function (docModel) {

        // base constructor
        OperationGenerator.call(this, docModel);

        // public properties
        this._slideMode = docModel.useSlideMode();
        this._attributePool = docModel.getAttributePool();
        this._paragraphStyles = docModel.getParagraphStyles();
        this._characterStyles = docModel.getCharacterStyles();
        this._drawingStyles = docModel.getDrawingStyles();

    }); // class TextOperationGenerator

    // private methods --------------------------------------------------------

    /**
     * Returns whether the passed DOM drawing frame is a placeholder that will
     * not be processed by this generator.
     *
     * @param {jQuery|HTMLElement} drawingFrame
     *  The DOM drawing frame to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed DOM drawing frame is a placeholder.
     */
    TextOperationGenerator.prototype._isPlaceHolderDrawing = function (drawingFrame) {
        return !!this._docModel.isPlaceHolderDrawing && this._docModel.isPlaceHolderDrawing(drawingFrame);
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates and appends a new operation with a target property to the
     * operations array.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {Object|Null} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options of the public method
     *  OperationGenerator.generateOperation(), and the following additional
     *  options:
     *  @param {String} [options.target]
     *      The identifier of the target to be added as 'target' property to
     *      the generated operation. If omitted or an empty string, no target
     *      property will be added to the operation.
     *
     * @returns {Object}
     *  The created JSON operation object.
     */
    TextOperationGenerator.prototype.generateOperationWithTarget = function (name, properties, options) {
        var target = Utils.getStringOption(options, 'target', null);
        if (target) { properties = _.extend({}, properties, { target: target }); }
        return this.generateOperation(name, properties, options);
    };

    /**
     * Creates and appends a new operation to the operations array. Adds
     * explicit attributes of the passed node to the 'attrs' option of the new
     * operation.
     *
     * @param {HTMLElement|jQuery} element
     *  The element node that may contain explicit formatting attributes. If
     *  this object is a jQuery collection, uses the first node it contains.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {Object|Null} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options of the public method
     *  OperationGenerator.generateOperationWithTarget(), and the following
     *  additional options:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute set of the generated operation.
     *  @param {Object} [options.drawingListStyles]
     *      If specified, this property will be object containing list style
     *      attributes of the drawing.
     *  @param {Object} [options.slideListStyles]
     *      If specified, this property will be object containing list style
     *      attributes of the slide.
     *  @param {Boolean} [options.getNewDrawingId=false]
     *      If set to true, new id for the drawing will be created.
     *
     * @returns {Object}
     *  The created operation JSON object.
     */
    TextOperationGenerator.prototype.generateOperationWithAttributes = function (element, name, properties, options) {

        // explicit attributes of the passed node
        var attributes = AttributeUtils.getExplicitAttributes(element);
        // a specified property, that shall be ignored in the setAttributes operation
        var ignoreFamily = Utils.getStringOption(options, 'ignoreFamily', '');
        // object containing list style attributes of the drawing
        var drawingListStyles = Utils.getObjectOption(options, 'drawingListStyles', {});
        // object containing the slide list style attributes
        var slideListStyles = Utils.getObjectOption(options, 'slideListStyles', null);
        // the active application
        var app = this._docModel.getApp();
        // restoring character attributes at paragraphs
        var characterAttributes = (!app.isODF() && Utils.getBooleanOption(options, 'isParagraph', false)) ? TextUtils.getCharacterAttributesFromEmptyTextSpan(element) : null;
        // Whether to use new drawing id, and overwrite existing one that is copied, to keep them unique
        var getNewDrawingId = Utils.getBooleanOption(options, 'getNewDrawingId', false);
        // the document file descriptor
        var fileDescriptor = app.getFileDescriptor();
        // receiving additional attributes from empty paragraph
        if (characterAttributes) { attributes.character = characterAttributes; }

        // deleting the specified families from the attributes
        if (attributes && ignoreFamily.length > 0) {
            _.each(ignoreFamily.split(' '), function (oneFamily) {
                if (attributes[oneFamily]) { delete attributes[oneFamily]; }
            });
        }

        // create a copy of the operation properties, ensure an existing object
        properties = _.extend({}, properties);

        // add the 'attrs' entry if there are attributes (but do not overwrite merged attributes, if they are already saved in properties)
        if (!_.isEmpty(attributes) && _.isEmpty(properties.attrs)) {
            properties.attrs = attributes;
        }

        if (!_.isEmpty(drawingListStyles)) {
            properties.attrs = properties.attrs || {};
            properties.attrs = _.extend(_.copy(drawingListStyles, true), properties.attrs); // always create a clone for drawing list styles
        }

        // adding the slide list styles, if available
        if (slideListStyles) {
            properties.attrs = properties.attrs || {};
            properties.attrs.listStyles = slideListStyles;
        }
        if (properties.attrs && getNewDrawingId) {
            // overwrite drawing id with the new one to keep them unique
            if (properties.attrs.drawing) {
                properties.attrs.drawing.id = this._docModel.getNewDrawingId();
            }
            // remove linked drawing ids, as they are not valid anymore
            if (properties.attrs.connector) {
                properties.attrs.connector.startId = null;
                properties.attrs.connector.startIndex = null;
                properties.attrs.connector.endId = null;
                properties.attrs.connector.endIndex = null;
            }
        }

        // add additinal attributes to identify whether copy&paste takes place between two documents or inside the same document
        if (properties.attrs && properties.attrs.fill && properties.attrs.fill.type === 'bitmap') {
            _.extend(properties.attrs.fill, { fileId: fileDescriptor.id, sessionId: ox.session });
        }

        // push the operation
        return this.generateOperationWithTarget(name, properties, options);
    };

    /**
     * Generates the 'setAttributes' operation needed to set the explicit
     * formatting attributes of the passed element node. If the passed node
     * does not contain any explicit attributes, no operation will be generated.
     *
     * @param {HTMLElement|jQuery} element
     *  The element node whose formatting attributes will be converted to an
     *  operation. If this object is a jQuery collection, uses the first node
     *  it contains.
     *
     * @param {Object|Null} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options of the public method
     *  OperationGenerator.generateOperationWithTarget(), and the following
     *  additional options:
     *  @param {String} [options.clearFamily]
     *      If specified, a style family for which additional formatting
     *      attributes with null values will be inserted into the operation.
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute set of the generated operation.
     *  @param {Boolean} [options.allAttributes=false]
     *      If true, merged attributes will be used instead of explicit
     *      attributes.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateSetAttributesOperation = function (element, properties, options) {

        // explicit or merged attributes of the passed node
        var elementAttributes = null;
        // the style families for generated null attributes
        var clearFamily = Utils.getStringOption(options, 'clearFamily', '');
        // a specified property, that shall be ignored in the setAttributes operation
        var ignoreFamily = Utils.getStringOption(options, 'ignoreFamily', '');
        // a specified property, that determines whether to use merged attributes instead of explicit attributes
        var allAttributes = Utils.getBooleanOption(options, 'allAttributes', false);

        if (allAttributes && this._isPlaceHolderDrawing(element)) {
            // merged attributes of the passed node
            elementAttributes = this._drawingStyles.getElementAttributes(element);

        } else if (allAttributes && this._isPlaceHolderDrawing($(element).parents(DrawingFrame.NODE_SELECTOR))) {
            // merged attributes of the passed node
            if (DOM.isParagraphNode(element)) {
                elementAttributes = this._paragraphStyles.getElementAttributes(element, { noDefaults: true });
            } else if (DOM.isTextSpan(element)) {
                elementAttributes = this._characterStyles.getElementAttributes(element, { noDefaults: true });
            }

        } else {
            // explicit attributes of the passed node
            elementAttributes = AttributeUtils.getExplicitAttributes(element);
        }

        // deleting the specified families from the attributes
        if (elementAttributes && ignoreFamily.length > 0) {
            _.each(ignoreFamily.split(' '), function (oneFamily) {
                if (elementAttributes[oneFamily]) { delete elementAttributes[oneFamily]; }
            });
        }

        // insert null values for all attributes registered for the specified style family
        var attributeSet = null;
        if (clearFamily.length > 0) {
            var styleCollection = this._docModel.getStyleCollection(clearFamily);
            attributeSet = styleCollection.buildNullAttributeSet({ style: true });
        } else {
            attributeSet = {};
        }

        // merge the explicit attributes of the passed element
        this._attributePool.extendAttributeSet(attributeSet, elementAttributes);

        // no attributes, no operation
        if (!_.isEmpty(attributeSet)) {
            properties = _.extend({}, properties, { attrs: attributeSet });
            this.generateOperationWithTarget(Operations.SET_ATTRIBUTES, properties, options);
        }

        return this;
    };

    /**
     * Generates all operations needed to recreate the child nodes of the
     * passed paragraph.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph element whose content nodes will be converted to
     *  operations. If this object is a jQuery collection, uses the first node
     *  it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed paragraph node. The generated
     *  operations will contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.start]
     *      The logical index of the first character to be included into the
     *      generated operations. By default, operations will include all
     *      contents from the beginning of the paragraph.
     *  @param {Number} [options.end]
     *      The logical index of the last character to be included into the
     *      generated operations (closed range). By default, operations will
     *      include all contents up to the end of the paragraph.
     *  @param {Boolean} [options.clear=false]
     *      If set to true, a 'setAttributes' operation will be generated for
     *      the first 'insertText' operation that clears all character
     *      attributes of the inserted text. This prevents that applying the
     *      operations at another place in the document clones the character
     *      formatting of the target position.
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {Number} [options.targetOffset]
     *      If set to a number, the logical positions in the operations
     *      generated for the child nodes will start at this offset. If
     *      omitted, the original node offset will be used in the logical
     *      positions.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *  @param {Boolean} [options.getDrawingListStyles=false]
     *      If specified, for drawings inside paragraph list styles will be
     *      fetched.
     *  @param {Boolean} [options.allAttributes=false]
     *      If true, merged attributes will be used instead of explicit
     *      attributes.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateParagraphChildOperations = function (paragraph, position, options) {

        // self reference
        var self = this;
        // start of text range to be included in the operations
        var rangeStart = Utils.getIntegerOption(options, 'start');
        // end of text range to be included in the operations
        var rangeEnd = Utils.getIntegerOption(options, 'end');
        // start position of text nodes in the generated operations
        var targetOffset = Utils.getIntegerOption(options, 'targetOffset');
        // used to merge several text portions into the same operation
        var lastTextOperation = null;
        // a specified property, that shall be ignored in the setAttributes operation
        var ignoreFamily = Utils.getStringOption(options, 'ignoreFamily', '');
        // a specified property, that shall send target string with operation
        var target = Utils.getStringOption(options, 'target', '');
        // fetch list style properties for drawings
        var getDrawingListStyles = Utils.getBooleanOption(options, 'getDrawingListStyles', false);
        // a specified property, that determines whether to use merged attributes instead of explicit attributes
        var allAttributes = Utils.getBooleanOption(options, 'allAttributes', false);
        // a specified property, that determines whether to use merged attributes instead of explicit attributes
        var clearAttributes = Utils.getBooleanOption(options, 'clear', false);
        // formatting ranges for text portions, must be applied after the contents
        var attributeRanges = [];
        // attributes passed to the first insert operation to clear all formatting
        var allClearAttributes = null;

        // adding character attributes that need to be removed
        function resetCharacterAttributes(span, insertOperation) {

            // not simply removing all character attributes, but only those, that are really set in the spans (55504).
            _.each(AttributeUtils.getExplicitAttributes(span, { family: 'character' }), function (value, oneAttribute) {
                insertOperation.attrs = insertOperation.attrs || {};
                insertOperation.attrs.character = insertOperation.attrs.character || {};
                insertOperation.attrs.character[oneAttribute] = null;
            });
        }

        // collecting those character attributes, that need to be cleared
        function getSpecificCharacterClearAttributes(span, isSpanStart) {

            // not simply removing all character attributes, but only those, that are really set in the span and optinally the previous span (55504).
            var requiredAttrs = _.keys(AttributeUtils.getExplicitAttributes(span, { family: 'character' }));

            if (isSpanStart && span.previousSibling) { // also remove character attributes from previous span (if it exists)
                requiredAttrs = requiredAttrs.concat(_.keys(AttributeUtils.getExplicitAttributes(span.previousSibling, { family: 'character' })));
            }

            var newCharacterAttributes = requiredAttrs.length > 0 ? {} :  null;
            _.each(requiredAttrs, function (oneAttribute) {
                newCharacterAttributes[oneAttribute] = null;
            });

            if (!allClearAttributes) { allClearAttributes = {}; }

            if (newCharacterAttributes) {
                allClearAttributes.character = newCharacterAttributes;
            } else {
                delete allClearAttributes.character;
            }

            return allClearAttributes;
        }

        // generates the specified operation, adds that attributes in clearAttributes on first call
        function generateOperationWithClearAttributes(name, properties, textSpan, isSpanStart) {

            // add the character attributes that will be cleared on first insertion operation
            if (clearAttributes) {
                properties.attrs = getSpecificCharacterClearAttributes(textSpan, isSpanStart);
                allClearAttributes = null;
            }

            return self.generateOperationWithTarget(name, properties, options);
        }

        // clear all attributes of the first inserted text span
        if (clearAttributes) {
            allClearAttributes = this._characterStyles.buildNullAttributeSet({ style: true });
        }

        // process all content nodes in the paragraph and create operations
        Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength, offsetStart, offsetLength) {

            // logical start index of the covered part of the child node
            var startIndex = _.isNumber(targetOffset) ? targetOffset : (nodeStart + offsetStart);
            // logical end index of the covered part of the child node (closed range)
            var endIndex = startIndex + offsetLength - 1;
            // logical start position of the covered part of the child node
            var startPosition = Position.appendNewIndex(position, startIndex);
            // logical end position of the covered part of the child node
            var endPosition = Position.appendNewIndex(position, endIndex);
            // text of a portion span
            var text = null;
            // type of the drawing or the hard break or the field or the range marker
            var type = null;
            // id and position of a text marker range
            var rangeId = null, rangePos = null;
            // whether the clearing of all attributes is required
            var isSpanStart = false;
            // the field attributes
            var attrs = {};

            // operation to create a (non-empty) generic text portion
            if (DOM.isTextSpan(node)) {

                // nothing to do for template texts in empty text frames
                if (DOM.isTextFrameTemplateTextSpan(node)) { return; }

                // extract the text covered by the specified range
                text = node.firstChild.nodeValue.substr(offsetStart, offsetLength);
                // append text portions to the last 'insertText' operation
                if (lastTextOperation) {
                    lastTextOperation.text += text;
                    if (clearAttributes) { resetCharacterAttributes(node, lastTextOperation); } // also removing attributes
                } else {
                    if (text.length > 0) {
                        isSpanStart = offsetStart === 0; // clearing of attributes only required for first character in text span
                        lastTextOperation = generateOperationWithClearAttributes(Operations.TEXT_INSERT, { start: startPosition, text: text }, node, isSpanStart);
                    }
                }
                attributeRanges.push({ node: node, position: _.extend({ start: startPosition }, (text.length > 1) ? { end: endPosition } : null) });

            } else {

                // anything else than plain text will be inserted, forget last text operation
                lastTextOperation = null;

                // operation to create a text field
                if (DOM.isFieldNode(node)) {
                    // extract text of all embedded spans representing the field
                    text = $(node).text();
                    attrs.character = {};
                    attrs.character.field = {};
                    $.each($(node).data(), function (name, value) {
                        if (name !== 'type') {
                            if (name !== 'sFieldId' && name !== 'attributes') { // #45332 - dont send unnecessary and invalid properties
                                attrs.character.field[name] = value;
                            }
                        } else {
                            type = value;
                        }
                    });
                    this.generateOperationWithTarget(Operations.FIELD_INSERT, { start: startPosition, representation: text, type: type, attrs: attrs }, options);
                    // attributes are contained in the embedded span elements
                    attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });

                // operation to create a tabulator
                } else if (DOM.isTabNode(node)) {
                    generateOperationWithClearAttributes(Operations.TAB_INSERT, { start: startPosition });
                    // attributes are contained in the embedded span elements
                    attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });

                // operation to create a hard break
                } else if (DOM.isHardBreakNode(node)) {
                    // reading type of hard break from the node
                    type = $(node).data('type');
                    generateOperationWithClearAttributes(Operations.HARDBREAK_INSERT, { start: startPosition, type: type });
                    // attributes are contained in the embedded span elements
                    attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });

                // operation to create a drawing (including its attributes)
                } else if (DrawingFrame.isDrawingFrame(node) || DOM.isDrawingPlaceHolderNode(node)) {
                    // switching to node in drawing layer
                    if (DOM.isDrawingPlaceHolderNode(node)) { node = DOM.getDrawingPlaceHolderNode(node); }
                    // skip drawing nodes that cannot be restored (inserting one space because of position counting inside paragraph)
                    // -> but never insert a text direct into a slide
                    if (DOM.isUnrestorableDrawingNode(node) && !this._slideMode) {
                        generateOperationWithClearAttributes(Operations.TEXT_INSERT, { start: startPosition, text: ' ' });
                        return;
                    }
                    // generate operations for the drawing
                    var drawingOptions = { ignoreFamily: ignoreFamily, getDrawingListStyles: getDrawingListStyles };
                    drawingOptions.target = Utils.getStringOption(options, 'target', null);
                    this.generateDrawingOperations(node, startPosition, drawingOptions);

                } else if (DOM.isCommentPlaceHolderNode(node)) {
                    // switching to node in comment layer
                    node = DOM.getCommentPlaceHolderNode(node);

                    // generate operations for the drawing
                    this.generateCommentOperations(node, startPosition, { ignoreFamily: ignoreFamily, target: target });

                } else if (DOM.isRangeMarkerNode(node)) {

                    rangeId = DOM.getRangeMarkerId(node);
                    rangePos = DOM.isRangeMarkerStartNode(node) ? 'start' : (DOM.isRangeMarkerEndNode(node) ? 'end' : undefined);
                    type = DOM.getRangeMarkerType(node);
                    this.generateOperationWithTarget(Operations.RANGE_INSERT, { start: startPosition, id: rangeId, type: type, position: rangePos }, options);
                    attributeRanges.push({ node: node, position: { start: startPosition } });

                } else if (DOM.isComplexFieldNode(node)) {
                    this.generateOperationWithTarget(Operations.COMPLEXFIELD_INSERT, { start: startPosition, instruction: DOM.getComplexFieldInstruction(node) }, options);
                    // special fields have content inside complex field div node, and lenght for oxo positioning is stored in data as "length"
                    if (DOM.isSpecialField(node)) {
                        var fieldTxtLen = $(node).data('length');
                        fieldTxtLen = (_.isNumber(fieldTxtLen) && fieldTxtLen > 0) ? (fieldTxtLen - 1) : 1;
                        this.generateOperationWithTarget(Operations.TEXT_INSERT, { start: Position.increaseLastIndex(startPosition, 1), text: Utils.repeatString('1', fieldTxtLen) }, options);
                    }
                    attributeRanges.push({ node: node, position: { start: startPosition } });
                } else if (DOM.isBookmarkNode(node)) {
                    this.generateOperationWithTarget(Operations.BOOKMARK_INSERT, { start: startPosition, id: DOM.getBookmarkId(node), anchorName: DOM.getBookmarkAnchor(node), position: DOM.getBookmarkPosition(node) }, options);
                } else {
                    Utils.error('TextOperationGenerator.generateParagraphChildOperations(): unknown content node');
                    return Utils.BREAK;
                }
            }

            // custom target offset: advance offset by covered node length
            if (_.isNumber(targetOffset)) {
                targetOffset += offsetLength;
            }

        }, this, { start: rangeStart, end: rangeEnd });

        // if it's presentation paragraph inside placeholder drawing and it has only template text, generate possible character attrs
        if (!DOM.isSlideNode(paragraph) && this._isPlaceHolderDrawing($(paragraph).parents(DrawingFrame.NODE_SELECTOR))) {
            var $templateTextSpan = $(paragraph).find(DOM.TEMPLATE_TEXT_SELECTOR);
            if ($templateTextSpan.length) {
                attributeRanges.push({ node: $templateTextSpan, position: { start: position } });
            }
        }

        // Generate 'setAttribute' operations after all contents have been
        // created via 'insertText', 'insertField', etc. Otherwise, these
        // operations would clone the attributes of the last text portion
        // instead of creating a clean text node as expected in this case.
        var attrOptions = { ignoreFamily: ignoreFamily, allAttributes: allAttributes };
        attrOptions.target = Utils.getStringOption(options, 'target', null);
        _(attributeRanges).each(function (range) {
            this.generateSetAttributesOperation(range.node, range.position, attrOptions);
        }, this);

        return this;
    };

    /**
     * Generates all operations needed to recreate the passed paragraph.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph element whose contents will be converted to operations.
     *  If this object is a jQuery collection, uses the first node it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed paragraph node. The generated
     *  operations will contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *  @param {Boolean} [options.allAttributes=false]
     *      If true, merged attributes will be used instead of explicit
     *      attributes.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateParagraphOperations = function (paragraph, position, options) {

        // a locally used options object
        var localOptions = null;
        // the options for the operation
        var operationOptions = { start: position };
        // the name of the operation
        var operationName = Operations.PARA_INSERT;
        // an object collecting data for the operation
        var operationInfo = null;
        // the element attributes
        var elementAttrs = null;
        // a specified property, that shall be ignored in the setAttributes operation
        var ignoreFamily = Utils.getStringOption(options, 'ignoreFamily', '');
        // a specified property, that determines whether to use merged attributes instead of explicit attributes
        var allAttributes = Utils.getBooleanOption(options, 'allAttributes', false);

        if (!options) { options = {}; }

        // operations to create the paragraph element and formatting
        if (DOM.isSlideNode(paragraph)) {
            // getting the target for the operation
            if ($(paragraph).data('target')) {
                localOptions = _.clone(options);
                localOptions.target = $(paragraph).data('target'); // only setting target for Operations.SLIDE_INSERT
            } else {
                localOptions = {};
            }
            if (_.isFunction(this._docModel.getInsertOperationNameForNode)) {
                operationInfo = this._docModel.getInsertOperationNameForNode(paragraph, localOptions);
                operationName = operationInfo.name;
                if (_.isNumber(operationInfo.index)) { operationOptions.start = operationInfo.index; }
                if (localOptions.id) { operationOptions.id = localOptions.id; } // reusing the node ID, if possible
            }
            this.generateOperationWithAttributes(paragraph, operationName, operationOptions, localOptions);
        } else {
            options.isParagraph = true;

            // explicit attributes do not represent the entire attribute set of paragraphs inside placeholder drawings, so use merged element attributes instead
            if (allAttributes && this._isPlaceHolderDrawing($(paragraph).parents(DrawingFrame.NODE_SELECTOR))) {
                elementAttrs = this._paragraphStyles.getElementAttributes(paragraph, { noDefaults: true });
                if (elementAttrs) {
                    // deleting the specified families from the attributes
                    if (ignoreFamily.length > 0) {
                        _.each(ignoreFamily.split(' '), function (oneFamily) {
                            if (elementAttrs[oneFamily]) { delete elementAttrs[oneFamily]; }
                        });
                    }

                    operationOptions = _.extend({ attrs: elementAttrs }, operationOptions);
                }
            }

            this.generateOperationWithAttributes(paragraph, operationName, operationOptions, options);
        }

        // process all content nodes in the paragraph and create operations
        return this.generateParagraphChildOperations(paragraph, position, options);
    };

    /**
     * Generates all operations needed to recreate the passed table cell.
     *
     * @param {HTMLTableCellElement|jQuery} cellNode
     *  The table cell element that will be converted to operations. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed table cell. The generated operations
     *  will contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateTableCellOperations = function (cellNode, position, options) {

        // operation to create the table cell element
        this.generateOperationWithAttributes(cellNode, Operations.CELLS_INSERT, { start: position, count: 1 }, options);

        // generate operations for the contents of the cell
        return this.generateContentOperations(cellNode, position, options);
    };

    /**
     * Generates all operations needed to recreate the passed table row.
     *
     * @param {HTMLTableRowElement|jQuery} rowNode
     *  The table row element that will be converted to operations. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed table row. The generated operations
     *  will contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateTableRowOperations = function (rowNode, position, options) {

        // operation to create the table row element
        this.generateOperationWithAttributes(rowNode, Operations.ROWS_INSERT, { start: position }, options);

        // generate operations for all cells
        position = Position.appendNewIndex(position);
        Utils.iterateSelectedDescendantNodes(rowNode, 'td', function (cellNode) {
            this.generateTableCellOperations(cellNode, position, options);
            position = Position.increaseLastIndex(position);
        }, this, { children: true });

        return this;
    };

    /**
     * Generates all operations needed to recreate the passed table.
     *
     * @param {HTMLTableElement|jQuery} tableNode
     *  The table element that will be converted to operations. If this object
     *  is a jQuery collection, uses the first node it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed table node. The generated operations
     *  will contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateTableOperations = function (tableNode, position, options) {

        // whether the table node is a table node inside a drawing of type 'table'
        var isTableDrawing = DOM.isTableNodeInTableDrawing(tableNode);

        // operation to create the table element (not necessary for drawings of type 'table')
        if (!isTableDrawing) {
            this.generateOperationWithAttributes(tableNode, Operations.TABLE_INSERT, { start: position }, options);
            position = Position.appendNewIndex(position);
        }

        // generate operations for all rows
        DOM.getTableRows(tableNode).get().forEach(function (node) {
            this.generateTableRowOperations(node, position, options);
            position = Position.increaseLastIndex(position);
        }, this);

        return this;
    };

    /**
     * Generates all operations needed to recreate the passed drawing group.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing element whose contents will be converted to operations. If
     *  this object is a jQuery collection, uses the first node it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed drawing node. The generated
     *  operations will contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *  @param {Boolean} [options.allAttributes=false]
     *      If true, merged attributes will be used instead of explicit
     *      attributes.
     *  @param {Boolean} [options.getDrawingListStyles=false]
     *      If specified, for drawings inside paragraph list styles will be
     *      fetched.
     *  @param {Boolean} [options.getNewDrawingId=false]
     *      If set to true, new id for the drawing will be created.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateDrawingOperations = function (drawingNode, position, options) {

        // the type of the drawing
        var type = DrawingFrame.getDrawingType(drawingNode);
        // the operation options
        var operationOptions = { start: position, type: type };
        // the element attributes
        var elementAttrs = null;
        // a specified property, that shall be ignored in the setAttributes operation
        var ignoreFamily = Utils.getStringOption(options, 'ignoreFamily', '');
        // a specified property, that determines whether to use merged attributes instead of explicit attributes
        var allAttributes = Utils.getBooleanOption(options, 'allAttributes', false);
        // fetch list style properties for drawings
        var getDrawingListStyles = Utils.getBooleanOption(options, 'getDrawingListStyles', false);

        if (type === 'image') {
            if (allAttributes && this._isPlaceHolderDrawing(drawingNode)) {
                elementAttrs = this._drawingStyles.getElementAttributes(drawingNode);
                if (elementAttrs) {
                    ignoreFamily += ' shape geometry'; // no 'shape' and 'geometry' attributes for images
                    // deleting the specified families from the attributes
                    if (ignoreFamily.length > 0) {
                        _.each(ignoreFamily.split(' '), function (oneFamily) {
                            if (elementAttrs[oneFamily]) { delete elementAttrs[oneFamily]; }
                        });
                    }
                    operationOptions = _.extend({ attrs: elementAttrs }, operationOptions);
                }
            }

            // special image handling: We need to take care that images are always readable for the target instance
            this.generateImageOperationWithAttributes(drawingNode, operationOptions, options);
            return this;
        }

        // explicit attributes do not represent the entire attribute set of placeholder drawings, so use merged element attributes instead
        if (allAttributes && (type === 'shape') && this._isPlaceHolderDrawing(drawingNode)) {
            elementAttrs = this._drawingStyles.getElementAttributes(drawingNode, { noDefaults: true });
            if (elementAttrs) {
                // deleting the specified families from the attributes
                if (ignoreFamily.length > 0) {
                    _.each(ignoreFamily.split(' '), function (oneFamily) {
                        if (elementAttrs[oneFamily]) { delete elementAttrs[oneFamily]; }
                    });
                }
                operationOptions = _.extend({ attrs: elementAttrs }, operationOptions);
            }
        }
        if (this._slideMode && getDrawingListStyles) {
            // fetch list styles of the drawing
            options.drawingListStyles = this._drawingStyles.getAllListStylesAtDrawing(drawingNode);
        }

        this.generateOperationWithAttributes(drawingNode, Operations.INSERT_DRAWING, operationOptions, options);

        // the drawing list styles shall not be used for generating the content of the drawing (for example for paragraphs)
        if (options.drawingListStyles) { delete options.drawingListStyles; }

        if (type === 'group') {
            // generate operations for the contents of the text frame
            this.generateDrawingGroupChildOperations(drawingNode, position, options);
        } else if (((type === 'shape' || type === 'connector') && DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode)) || type === 'table') {
            // generate operations for the contents of the text frame
            this.generateContentOperations(drawingNode, position, options);
        }

        return this;
    };

    /**
     * Generates all operations needed to recreate the passed comment.
     *
     * @param {HTMLElement|jQuery} commentNode
     *  The comment element whose contents will be converted to operations. If
     *  this object is a jQuery collection, uses the first node it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed drawing node. The generated
     *  operations will contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateCommentOperations = function (commentNode, position, options) {

        // restoring author, date and id
        var id = DOM.getTargetContainerId(commentNode);
        var author = DOM.getCommentAuthor(commentNode);
        var uid = DOM.getCommentAuthorUid(commentNode);
        var date = DOM.getCommentDate(commentNode);
        var operationOptions = { start: position, id: id, author: author };

        if (date) { operationOptions.date = date; } // date must be specified or ignored, but no empty string, 55221
        if (uid !== '') { operationOptions.uid = uid; }

        this.generateOperationWithAttributes(commentNode, Operations.COMMENT_INSERT, operationOptions, options);

        // using the target for the created operations
        var localOptions = options || {};
        localOptions.target = id;

        // generate operations for the contents of the text frame
        this.generateContentOperations(commentNode, [], localOptions);

        return this;
    };

    /**
     * Generates a usable image operation for an internal & external URL.
     *
     * @param {HTMLElement|jQuery} imageNode
     *  The image drawing element. If this object is a jQuery collection,
     *  uses the first node it contains.
     *
     * @param {Object} [operationOptions]
     *  Additional options that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *
     * @returns {Object}
     *  The created operation JSON object.
     */
    TextOperationGenerator.prototype.generateImageOperationWithAttributes = function (imageNode, operationOptions, options) {

        // the new operation
        var operation = this.generateOperationWithAttributes(imageNode, Operations.INSERT_DRAWING, _.extend({ type: 'image' }, operationOptions), options);
        // the image URL from the attributes map
        var imageUrl = operation.attrs && operation.attrs.image && operation.attrs.image.imageUrl;

        // set image attributes so they can be used in the same instance or between different instances
        if (imageUrl && DOM.isDocumentImageNode(imageNode)) {
            _.extend(operation.attrs.image, {
                imageData: DOM.getBase64FromImageNode(imageNode, ImageUtils.getMimeTypeFromImageUri(imageUrl)),
                sessionId: ox.session,
                fileId: DOM.getUrlParamFromImageNode(imageNode, 'id')
            });
        }

        return operation;
    };

    /**
     * Generates all operations needed to recreate the children of the passed
     * drawing group.
     *
     * @param {HTMLElement|jQuery} drawingGroupNode
     *  The drawing group element whose contents will be converted to
     *  operations. If this object is a jQuery collection, uses the first node
     *  it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed drawing group node. The generated
     *  operations will contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateDrawingGroupChildOperations = function (drawingGroupNode, position, options) {

        // generate operations for all drawings inside the drawing group
        position = Position.appendNewIndex(position);
        Utils.iterateSelectedDescendantNodes($(drawingGroupNode).children().first(), '.drawing', function (drawingNode) {
            this.generateDrawingOperations(drawingNode, position, options);
            position = Position.increaseLastIndex(position);
        }, this, { children: true });

        return this;
    };

    /**
     * Generates all operations needed to recreate the contents of the passed
     * root node. Root nodes are container elements for text paragraphs and
     * other first-level content nodes (e.g. tables). Examples for root nodes
     * are the entire document root node, table cells, or text shapes. Note
     * that the operation to create the root node itself will NOT be generated.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node containing the content nodes that will be converted to
     *  operations. The passed root node may contain an embedded node that
     *  serves as parent for all content nodes. If this object is a jQuery
     *  collection, uses the first node it contains.
     *
     * @param {Array<Number>} position
     *  The logical position of the passed node. The generated operations will
     *  contain positions starting with this address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.ignoreFamily]
     *      A space-separated list of families that will not be inserted into
     *      the attribute sets of the generated operations.
     *  @param {String} [options.target]
     *      If specified, this property will be send as target with operation.
     *
     * @returns {TextOperationGenerator}
     *  A reference to this instance.
     */
    TextOperationGenerator.prototype.generateContentOperations = function (rootNode, position, options) {

        // the container node (direct parent of the target content nodes)
        var containerNode = DOM.getChildContainerNode(rootNode);

        // iterate all child elements of the root node and create operations
        position = Position.appendNewIndex(position);
        Utils.iterateDescendantNodes(containerNode, function (node) {

            if (DOM.isParagraphNode(node)) {
                // skip implicit paragraph nodes without increasing the position
                if (DOM.isImplicitParagraphNode(node)) { return; }
                // operations to create a paragraph
                this.generateParagraphOperations(node, position, options);
            } else if (DOM.isTableNode(node)) {
                // skip table nodes with exceeded size without increasing the position
                if (DOM.isExceededSizeTableNode(node)) { return; }
                // operations to create a table with its structure and contents
                this.generateTableOperations(node, position, options);
            } else if (DOM.isMarginalDrawingLayerNode(node)) {
                // skip over text drawing layer in header/footer
            } else {
                Utils.error('TextOperationGenerator.generateContentOperations(): unexpected node "' + Utils.getNodeName(node) + '" at position ' + JSON.stringify(position) + '.');
                // continue with next child node (do not increase position)
                return;
            }

            // increase last element of the logical position (returns a clone)
            position = Position.increaseLastIndex(position);

        }, this, { children: true });

        return this;
    };

    // exports ================================================================

    return TextOperationGenerator;

});
