/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/textframework/components/drawing/imagecropframe', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/view/dialog/imagecropdialog',
    'io.ox/office/drawinglayer/view/drawingframe'
], function (Utils, Tracking, Canvas, Operations, ImageCropDialog, DrawingFrame) {

    'use strict';

    // static class ImageCropFrame ==============================================

    // the names of all tracking events but 'tracking:start'
    var TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';
    // dyn generated images for the resize handlers
    var resizersImgs = [];

    /**
     * Contains common static helper functions to handle drawing objects.
     */
    var ImageCropFrame = {};

    // methods ----------------------------------------------------------------

    function toggleTracking(node, state, scrollNode) {
        if (state) {
            Tracking.enableTracking(node, {
                autoScroll: true,
                borderNode: scrollNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 10,
                maxSpeed: 250,
                trackModifiers: true
            });
        } else {
            Tracking.disableTracking(node);
        }
    }

    // convert px to percentage
    function calcCropSize(frameSize, left, top) {
        return {
            left: Math.round((left * 100) / frameSize.width, 1),
            right: Math.round(((frameSize.width - frameSize.drawingW + left) * 100) / frameSize.width, 1),
            top: Math.round((top * 100) / frameSize.height, 1),
            bottom: Math.round(((frameSize.height - frameSize.drawingH + top) * 100) / frameSize.height, 1)
        };
    }

    function getResizersImgs(docModel) {
        var imgs = [];
        var canvas = new Canvas(docModel);
        var minPos = 1.5;
        var maxPos = 16.5;
        var oneThirdPos = (maxPos - minPos) / 3;

        canvas.initialize({ width: 17, height: 17 });
        canvas.render(function (ctx) {
            ctx.setLineStyle({ style: '#fff', width: 1 });
            ctx.setFillStyle('#000');
            var path = ctx.createPath().pushPolygon(minPos, minPos, minPos, oneThirdPos, maxPos, oneThirdPos, maxPos, minPos);
            ctx.drawPath(path, 'all');
        });
        imgs.push(canvas.getDataURL());

        canvas.clear().render(function (ctx) {
            ctx.setLineStyle({ style: '#fff', width: 1 });
            ctx.setFillStyle('#000');
            var path = ctx.createPath().pushPolygon(minPos, minPos, minPos, maxPos, maxPos, maxPos, maxPos, 11.5, oneThirdPos + minPos, 11.5, oneThirdPos + minPos, minPos);
            ctx.drawPath(path, 'all');
        });
        imgs.push(canvas.getDataURL());

        return imgs;

    }

    /**
     * Makes crop frame and all necessary selection nodes for cropping.
     *
     * @param {DocumentModel} docModel
     * @param {jQuery} drawing
     * @param {jQuery} cropNode
     * @param {Number} zoomFactor
     * @param {jQuery} scrollNode
     */
    ImageCropFrame.drawCropFrame = function (docModel, drawing, cropNode, zoomFactor, scrollNode) {
        var options = { movable: true, resizable: true, zoomValue: zoomFactor * 100, scaleHandles: 1.5 / zoomFactor };
        var selectionBox = DrawingFrame.drawSelection(cropNode, options);

        drawing.addClass(DrawingFrame.ACTIVE_CROPPING_CLASS);
        drawing.find('>.selection').addClass('crop-selection');
        var dataSelection = drawing.data('selection');
        var drawingSelectionNode = dataSelection || drawing;
        if (dataSelection) { dataSelection.addClass(DrawingFrame.ACTIVE_CROPPING_CLASS).find('>.selection').addClass('crop-selection'); }
        var drawingSelResizers = drawingSelectionNode.find('>.selection').find('>.resizers>[data-pos]');
        if (resizersImgs.length < 1) { resizersImgs = getResizersImgs(docModel); }
        _.each(drawingSelResizers, function (resizer, i) {
            var pos = i % 2; // only 2 original images
            var img = $('<img src="' + resizersImgs[pos] + '">');

            $(resizer).append(img);
        });
        cropNode.addClass('src-img');

        DrawingFrame.toggleTracking(cropNode, true);
        if (dataSelection) { DrawingFrame.toggleTracking(dataSelection.find('>.selection'), true); }
        toggleTracking(cropNode.find('.tracker'), true, scrollNode);
        toggleTracking(selectionBox.find('>.resizers>[data-pos]'), true, scrollNode);
        toggleTracking(selectionBox.find('>.borders>[data-pos]'), true, scrollNode);
    };

    /**
     * Clears and cleans up active drawing after crop.
     *
     * @param {jQuery} activeDrawing
     */
    ImageCropFrame.clearCropFrame = function (activeDrawing) {
        activeDrawing.removeClass(DrawingFrame.ACTIVE_CROPPING_CLASS).find('.crop').remove();
        activeDrawing.find('>.selection')
            .removeClass('crop-selection')
            .find('img').remove();
        var dataSelection = activeDrawing.data('selection');
        if (dataSelection) {
            dataSelection.removeClass(DrawingFrame.ACTIVE_CROPPING_CLASS)
                .find('>.selection').removeClass('crop-selection')
                .find('img').remove();
        }
    };

    /**
     * Repaints crop frame of the passed drawing
     *
     * @param {jQuery} drawing
     */
    ImageCropFrame.refreshCropFrame = function (drawing) {
        var cropNode = drawing.find('.content').not('.copy').find('.drawing.crop');
        var cropNodeImg = cropNode.find('img');
        var imgPosLeft = parseInt(cropNodeImg.css('left'), 10);
        var imgPosTop = parseInt(cropNodeImg.css('top'), 10);
        var imgWidth = parseInt(cropNodeImg.css('width'), 10);
        var imgHeight = parseInt(cropNodeImg.css('height'), 10);
        cropNode.css({ marginLeft: imgPosLeft, marginTop: imgPosTop, width: imgWidth, height: imgHeight });
        cropNodeImg.css({ left: 0, top: 0 });
    };

    /**
     * Returns type of crop handler for the given drawing node.
     *
     * @param {jQuery} node
     * @returns {String|Null}
     */
    ImageCropFrame.getCropHandlerType = function (node) {
        return node.is(DrawingFrame.ACTIVE_CROPPING_SELECTOR + ' .crop .resizers>[data-pos]') ? $(node).attr('data-pos') : null;
    };

    /**
     * Handles image crop events.
     *
     * @param {Application} app
     * @param {jQuery.Event} event
     * @param {jQuery.Node} drawingNode
     */
    ImageCropFrame.handleCropEvents = function (app, event, drawing) {
        var docModel = app.getModel();
        var docView = app.getView();
        var $target = $(event.target);

        var drawingCropMoveHandler = (function () {
            // the current position change (in px)
            var shiftX = 0, shiftY = 0;
            // the current scroll position
            var scrollX = 0, scrollY = 0;
            // the start scroll position
            var startScrollX = 0, startScrollY = 0;
            var startCropX = 0, startCropY = 0;
            var zoomFactor;
            var scrollNode, cropFrameClone, cropFrameCloneImg, originalCropFrame, origCropFrameImg;
            var normalizedShift;
            var flipH, flipV, rotation;

            function startCropMove() {
                zoomFactor = docView.getZoomFactor();
                scrollNode = docView.getContentRootNode();
                cropFrameClone = $target.closest('.drawing.crop');
                cropFrameCloneImg = cropFrameClone.find('img').first();
                originalCropFrame = drawing.find('.cropping-frame');
                origCropFrameImg = originalCropFrame.find('img').first();
                startCropX = parseInt(cropFrameClone.css('margin-left'), 10);
                startCropY = parseInt(cropFrameClone.css('margin-top'), 10);
                flipH = DrawingFrame.isFlippedHorz(drawing);
                flipV = DrawingFrame.isFlippedVert(drawing);
                rotation = DrawingFrame.getDrawingRotationAngle(docModel, drawing);
                cropFrameClone.addClass('active-crop-event');
                drawing.find('canvas').addClass(DrawingFrame.HIDE_CANVAS_ON_CROP_CLASS);
            }

            function updateCropMove(event) {
                shiftX = (event.pageX - event.startX + scrollX) / zoomFactor;
                shiftY = (event.pageY - event.startY + scrollY) / zoomFactor;

                // only move the moveBox, if the mouse was really moved
                if (_.isNumber(shiftX) && _.isNumber(shiftY) && ((shiftX !== 0) || (shiftY !== 0))) {
                    normalizedShift = DrawingFrame.getNormalizedMoveCoordinates(shiftX, shiftY, rotation, flipH, flipV);
                    shiftX = Math.round(normalizedShift.x + startCropX, 1);
                    shiftY = Math.round(normalizedShift.y + startCropY, 1);

                    cropFrameClone.css({ cursor: event.cursor, marginLeft: shiftX, marginTop: shiftY });
                    origCropFrameImg.css({ cursor: event.cursor, left: shiftX, top: shiftY });
                }
            }

            function updateCropScrollMove() {
                // update scrollPosition with suggestion from event
                if (event.scrollX) { scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX); }
                if (event.scrollY) { scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY); }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            function stopCropMove() {
                if (_.isNumber(shiftX) && _.isNumber(shiftY) && ((shiftX !== 0) || (shiftY !== 0))) {
                    var generator = docModel.createOperationGenerator();
                    var start = docModel.getSelection().getStartPosition();
                    var target = docModel.getActiveTarget();

                    var frameSize = { width: cropFrameClone.width(), height: cropFrameClone.height(), drawingW: drawing.width(), drawingH: drawing.height() };
                    var calcCropValue = calcCropSize(frameSize, shiftX, shiftY);
                    var operationProperties = {
                        start: start,
                        attrs: { image: {
                            cropLeft: -calcCropValue.left,
                            cropRight: calcCropValue.right,
                            cropTop: -calcCropValue.top,
                            cropBottom: calcCropValue.bottom
                        } }
                    };

                    docModel.extendPropertiesWithTarget(operationProperties, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

                    // apply the operations (undo group is created automatically)
                    docModel.applyOperations(generator);
                    var finalX = origCropFrameImg.css('left');
                    var finalY = origCropFrameImg.css('top');
                    cropFrameClone.css({ marginLeft: finalX, marginTop: finalY });
                    cropFrameCloneImg.css({ left: 0, top: 0 });
                }
                clearCrop();
            }

            function cancelCropMove() {
                cropFrameClone.css({ marginLeft: startCropX, marginTop: startCropY });
                origCropFrameImg.css({ left: startCropX, top: startCropY });
                clearCrop();
            }

            function clearCrop() {
                cropFrameClone.removeClass('active-crop-event');
                DrawingFrame.toggleTracking(drawing, false);
                drawing.find('canvas').removeClass(DrawingFrame.HIDE_CANVAS_ON_CROP_CLASS);
            }

            // return the actual drawingCropHandler() function
            return function (event) {
                event.preventDefault();
                //console.warn(event.target);

                switch (event.type) {
                    case 'tracking:start':
                        startCropMove(event);
                        break;
                    case 'tracking:move':
                        updateCropMove(event);
                        break;
                    case 'tracking:scroll':
                        updateCropScrollMove(event);
                        break;
                    case 'tracking:end':
                        stopCropMove(event);
                        break;
                    case 'tracking:cancel':
                        cancelCropMove(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();
            };
        }());

        var drawingCropResizeHandler = (function () {
            // the size of the resized drawing (in px)
            var finalWidth = 0, finalHeight = 0;
            // the current scroll position
            var scrollX = 0, scrollY = 0;
            // the start scroll position
            var startScrollX = 0, startScrollY = 0;
            // correction factor for resizing to the left/top
            var scaleX = 0, scaleY = 0;
            var scrollNode, cropFrameClone, origCropFrame;
            var cropFrameImg, origCropFrameImg;
            var cropFrame2Left, cropFrame2Top;
            var zoomFactor, rotationAngle;
            var useX, useY, useLeft, useTop;
            var startTop, startLeft, oldWidth, oldHeight;
            var topPos, leftPos, bottomPos, rightPos;
            var isFlippedHorz, isFlippedVert;

            function startCropResize() {
                var pos = $(event.target).attr('data-pos');

                // helper function to set position properties at move box(es)
                function setPositionAtMoveBox() {
                    scaleX = useLeft ? -1 : 1;
                    if (DrawingFrame.isFlippedHorz(drawing)) { scaleX *= -1; }

                    scaleY = useTop ? -1 : 1;
                    if (DrawingFrame.isFlippedVert(drawing)) { scaleY *= -1; }
                }
                cropFrameClone = $target.closest('.drawing.crop').addClass('active-crop-event');
                origCropFrame = drawing.find('.cropping-frame');
                cropFrameImg = cropFrameClone.find('img');
                origCropFrameImg = origCropFrame.find('img');

                zoomFactor = docView.getZoomFactor();
                scrollNode = docView.getContentRootNode();
                rotationAngle = DrawingFrame.getDrawingRotationAngle(docModel, drawing);
                isFlippedHorz = DrawingFrame.isFlippedHorz(drawing);
                isFlippedVert = DrawingFrame.isFlippedVert(drawing);
                oldWidth = cropFrameClone.width();
                oldHeight = cropFrameClone.height();
                startLeft = parseInt(cropFrameClone.css('margin-left'), 10);
                startTop = parseInt(cropFrameClone.css('margin-top'), 10);

                // collecting information about the handle node
                useX = /[lr]/.test(pos);
                useY = /[tb]/.test(pos);
                useLeft = /[l]/.test(pos);
                useTop = /[t]/.test(pos);

                setPositionAtMoveBox(cropFrameClone);
                drawing.find('canvas').addClass(DrawingFrame.HIDE_CANVAS_ON_CROP_CLASS);
            }

            function updateCropResize(event) {
                // the horizontal shift
                var deltaX = event.pageX / zoomFactor - event.startX / zoomFactor + scrollX;
                // the vertical shift
                var deltaY = event.pageY / zoomFactor - event.startY / zoomFactor + scrollY;
                // the scaling factor for the width
                var scaleWidth = 1;
                // the scaling factor for the height
                var scaleHeight = 1;
                // normalize and scale deltas of the dimensions
                var normalizedDeltas = DrawingFrame.getNormalizedResizeDeltas(deltaX, deltaY, useX, useY, scaleX, scaleY, rotationAngle);
                // changed width and height
                var sumWidth = oldWidth + normalizedDeltas.x;
                var sumHeight = oldHeight + normalizedDeltas.y;
                var signWidth = sumWidth > 0 ? 1 : -1;
                var signHeight = sumHeight > 0 ? 1 : -1;
                var flipH = sumWidth < 0;
                var flipV = sumHeight < 0;

                // use the same scaling factor for vertical and horizontal resizing, if both are enabled
                // -> the larger number wins
                if (useX && useY) { // aspect ratio is always locked
                    scaleWidth = Math.abs(sumWidth / oldWidth); // scale shouldn't be negative
                    scaleHeight = Math.abs(sumHeight / oldHeight);

                    if (scaleWidth > scaleHeight) {
                        sumHeight = scaleWidth * oldHeight * signHeight;  // return sign of current width/height after scale
                    } else {
                        sumWidth = scaleHeight * oldWidth * signWidth;
                    }
                }

                topPos = useTop ? 'auto' : Math.min(0, sumHeight);
                bottomPos = useTop ? Math.min(0, sumHeight) : 'auto';
                leftPos = useLeft ? 'auto' : Math.min(0, sumWidth);
                rightPos = useLeft ? Math.min(0, sumWidth) : 'auto';

                // update drawing size
                finalWidth = Math.round(Math.max(1, Math.abs(sumWidth)), 1);
                finalHeight = Math.round(Math.max(1, Math.abs(sumHeight)), 1);

                // reset drawing rotation and flip to get correct offset values
                var transformStr;
                if (rotationAngle || isFlippedHorz || isFlippedVert) {
                    transformStr = drawing.css('transform');
                    drawing.css('transform', '');
                }
                var origOffset = cropFrameClone.offset();
                var transfOffset = cropFrameImg.offset();
                if (rotationAngle || isFlippedHorz || isFlippedVert) {
                    drawing.css('transform', transformStr);
                }

                cropFrame2Left = Math.round(startLeft + (transfOffset.left - origOffset.left) / zoomFactor, 1);
                cropFrame2Top = Math.round(startTop + (transfOffset.top - origOffset.top) / zoomFactor, 1);

                var cssProps = { width: finalWidth, height: finalHeight };
                if (useLeft) {
                    if (flipH) {
                        cssProps.right = -(startLeft + cropFrameClone.width() - drawing.width()) - finalWidth;
                        cssProps.left = '';
                    } else {
                        cssProps.right = -(startLeft + cropFrameClone.width() - drawing.width());
                        cssProps.left = '';
                    }
                } else {
                    if (flipH) {
                        cssProps.right = drawing.width() - startLeft;
                        cssProps.left = '';
                    } else {
                        cssProps.left = cropFrame2Left;
                        cssProps.right = '';
                    }
                }
                if (useTop) {
                    if (flipV) {
                        cssProps.top = startTop + cropFrameClone.height();
                        cssProps.bottom = '';
                    } else {
                        cssProps.bottom = -(startTop + cropFrameClone.height() - drawing.height());
                        cssProps.top = '';
                    }
                } else {
                    if (flipV) {
                        cssProps.bottom = drawing.height() - startTop;
                        cssProps.top = '';
                    } else {
                        cssProps.top = cropFrame2Top;
                        cssProps.bottom = '';
                    }
                }

                origCropFrameImg.css(cssProps);
                cropFrameImg.css({ width: finalWidth, height: finalHeight, left: leftPos, top: topPos, bottom: bottomPos, right: rightPos });
            }

            function updateCropScrollResize() {
                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            function stopCropResize() {
                if (_.isNumber(finalWidth) && _.isNumber(finalHeight) && ((finalWidth !== 0) || (finalHeight !== 0))) {
                    var generator = docModel.createOperationGenerator();
                    var start = docModel.getSelection().getStartPosition();
                    var target = docModel.getActiveTarget();

                    var frameSize = { width: cropFrameImg.width(), height: cropFrameImg.height(), drawingW: drawing.width(), drawingH: drawing.height() };
                    var calcCropValue = calcCropSize(frameSize, cropFrame2Left, cropFrame2Top);
                    var operationProperties = {
                        start: start,
                        attrs: { image: {
                            cropLeft: -calcCropValue.left,
                            cropRight: calcCropValue.right,
                            cropTop: -calcCropValue.top,
                            cropBottom: calcCropValue.bottom
                        } }
                    };

                    docModel.extendPropertiesWithTarget(operationProperties, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

                    // apply the operations (undo group is created automatically)
                    docModel.applyOperations(generator);
                    var finalX = origCropFrameImg.css('left');
                    var finalY = origCropFrameImg.css('top');
                    cropFrameClone.css({ marginLeft: finalX, marginTop: finalY, width: cropFrameImg.width(), height: cropFrameImg.height() });
                    cropFrameImg.css({ left: 0, top: 0 });
                }
                clearCrop();
            }

            function cancelCropResize() {
                cropFrameClone.css({ marginLeft: startLeft, marginTop: startTop, width: oldWidth, height: oldHeight });
                cropFrameImg.css({ left: 0, top: 0, width: oldWidth, height: oldHeight });
                origCropFrameImg.css({ left: startLeft, top: startTop, width: oldWidth, height: oldHeight });
                clearCrop();
            }

            function clearCrop() {
                cropFrameClone.removeClass('active-crop-event');
                DrawingFrame.toggleTracking(drawing, false);
                drawing.find('canvas').removeClass(DrawingFrame.HIDE_CANVAS_ON_CROP_CLASS);
            }

            // return the actual drawingCropResizeHandler() function
            return function (event) {

                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startCropResize(event);
                        break;
                    case 'tracking:move':
                        updateCropResize(event);
                        break;
                    case 'tracking:scroll':
                        updateCropScrollResize(event);
                        break;
                    case 'tracking:end':
                        stopCropResize(event);
                        break;
                    case 'tracking:cancel':
                        cancelCropResize(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();
            };
        }());

        app.getView().grabFocus();
        if (_.isString(ImageCropFrame.getCropHandlerType($target))) {
            $target.off(TRACKING_EVENT_NAMES);
            $target.on(TRACKING_EVENT_NAMES, drawingCropResizeHandler);
            drawingCropResizeHandler.call(this, event);
        } else {
            $target.off(TRACKING_EVENT_NAMES);
            $target.on(TRACKING_EVENT_NAMES, drawingCropMoveHandler);
            drawingCropMoveHandler.call(this, event);
        }
    };

    /**
     * Calculate crop values for the operation during drawing move event.
     *
     * @param {jQuery} drawing
     * @param {Object} oldAttrs
     * @param {Object} newAttrs
     *
     * @returns {Object}
     */
    ImageCropFrame.getDrawingMoveCropOp = function (drawing, oldAttrs, newAttrs) {
        var cropFrame = drawing.find('.content').not('.copy').find('.drawing.crop');
        var cropFrameImg = cropFrame.find('img');
        var cropFrameLeft = parseInt(cropFrame.css('margin-left'), 10);
        var cropFrameTop = parseInt(cropFrame.css('margin-top'), 10);

        var topOffsetDiff = Utils.convertHmmToLength(oldAttrs.top - (!_.isUndefined(newAttrs.top) ? newAttrs.top : oldAttrs.top), 'px');
        var leftOffsetDiff = Utils.convertHmmToLength(oldAttrs.left - (!_.isUndefined(newAttrs.left) ? newAttrs.left : oldAttrs.left), 'px');
        if (!_.isUndefined(newAttrs.anchorVertOffset)) {
            var oldAnchorVertOff = oldAttrs.anchorVertOffset || 0;
            topOffsetDiff = Utils.convertHmmToLength(oldAnchorVertOff - newAttrs.anchorVertOffset, 'px');
        }
        if (!_.isUndefined(newAttrs.anchorHorOffset)) {
            var oldAnchorHorOff = oldAttrs.anchorHorOffset || 0;
            leftOffsetDiff = Utils.convertHmmToLength(oldAnchorHorOff - newAttrs.anchorHorOffset, 'px');
        }
        var drawingW = Utils.convertHmmToLength(newAttrs.width || oldAttrs.width, 'px');
        var drawingH = Utils.convertHmmToLength(newAttrs.height || oldAttrs.height, 'px');
        var scaleX = oldAttrs.flipH ? -1 : 1;
        var scaleY = oldAttrs.flipV ? -1 : 1;
        var normalizedDeltas = DrawingFrame.getNormalizedResizeDeltas(leftOffsetDiff, topOffsetDiff, true, true, scaleX, scaleY, oldAttrs.rotation);
        var frameSize = { width: cropFrameImg.width(), height: cropFrameImg.height(), drawingW: drawingW, drawingH: drawingH };
        var calcCropValue = calcCropSize(frameSize, cropFrameLeft + normalizedDeltas.x, cropFrameTop + normalizedDeltas.y);

        return {
            cropLeft: -calcCropValue.left,
            cropRight: calcCropValue.right,
            cropTop: -calcCropValue.top,
            cropBottom: calcCropValue.bottom
        };
    };

    /**
     * Calculate crop values for the operation during drawing resize event.
     *
     * @param {jQuery} drawing
     * @param {Object} oldAttrs
     * @param {Object} newAttrs
     * @param {Boolean} useLeft
     * @param {Boolean} useTop
     *
     * @returns {Object}
     */
    ImageCropFrame.getDrawingResizeCropOp = function (drawing, oldAttrs, newAttrs, useLeft, useTop) {
        var cropFrame = drawing.find('.content').not('.copy').find('.drawing.crop');
        var cropFrameImg = cropFrame.find('img');
        var cropFrameLeft = parseInt(cropFrame.css('margin-left'), 10);
        var cropFrameTop = parseInt(cropFrame.css('margin-top'), 10);
        var drawingW = Utils.convertHmmToLength(newAttrs.width || oldAttrs.width, 'px');
        var drawingH = Utils.convertHmmToLength(newAttrs.height || oldAttrs.height, 'px');
        var widthDiff = useLeft ? Utils.convertHmmToLength(newAttrs.width - oldAttrs.width, 'px') : 0;
        var heightDiff = useTop ? Utils.convertHmmToLength(newAttrs.height - oldAttrs.height, 'px') : 0;
        var frameSize = { width: cropFrameImg.width(), height: cropFrameImg.height(), drawingW: drawingW, drawingH: drawingH };

        var calcCropValue = calcCropSize(frameSize, cropFrameLeft + widthDiff, cropFrameTop + heightDiff);

        return {
            cropLeft: -calcCropValue.left,
            cropRight: calcCropValue.right,
            cropTop: -calcCropValue.top,
            cropBottom: calcCropValue.bottom
        };
    };

    /**
     * Calculates crop properties for set attribute operation.
     * Based on image crop Dialog settings!
     *
     * @param {Number} drawingWidth
     * @param {Number} drawingHeight
     * @param {Number} frameWidth
     * @param {Number} frameHeight
     * @param {Number} offsetX
     * @param {Number} offsetY
     *
     * @returns {Object}
     */
    ImageCropFrame.calculateCropOperation = function (drawingWidth, drawingHeight, frameWidth, frameHeight, offsetX, offsetY) {
        var leftPart = (frameWidth - drawingWidth) / 2 - offsetX;
        var rightPart = (frameWidth - drawingWidth) / 2 + offsetX;
        var topPart = (frameHeight - drawingHeight) / 2 - offsetY;
        var bottomPart = (frameHeight - drawingHeight) / 2 + offsetY;
         // convert to procentage
        var cropLeft = (100 * leftPart) / frameWidth;
        var cropRight = (100 * rightPart) / frameWidth;
        var cropTop = (100 * topPart) / frameHeight;
        var cropBottom = (100 * bottomPart) / frameHeight;

        return {
            cropLeft: Utils.round(cropLeft, 0.01),
            cropRight: Utils.round(cropRight, 0.01),
            cropTop: Utils.round(cropTop, 0.01),
            cropBottom: Utils.round(cropBottom, 0.01)
        };
    };

    /**
     * Create crop image properties used to crop image into bounding frame,
     * as fit or fill, depending on passed type.
     *
     * @param {DocumentModel} docModel
     * @param {Object} explAttrs
     * @param {String} type
     *
     * @returns {jQuery.Promise}
     */
    ImageCropFrame.cropFillFitSpace = function (docModel, explAttrs, type) {
        var imageAttrs = explAttrs.image;
        var opProps = {};
        var ratio;
        var frameWidth = explAttrs.drawing.width;
        var frameHeight = explAttrs.drawing.height;

        if (docModel.getSelection().isCropMode()) { docModel.exitCropMode(); }

        var promise = docModel.getImageSize(imageAttrs.imageUrl).then(function (size) {
            var imgRatio = size.width / size.height;
            var calcWidth, calcHeight;

            function adjustCropHeight() {
                opProps.cropLeft = null;
                opProps.cropRight = null;

                ratio = (calcHeight - frameHeight) / 2;
                ratio = Utils.round(100 * ratio / calcHeight, 0.01);
                opProps.cropTop = ratio;
                opProps.cropBottom = ratio;
            }

            function adjustCropWidth() {
                opProps.cropTop = null;
                opProps.cropBottom = null;

                ratio = (calcWidth - frameWidth) / 2;
                ratio = Utils.round(100 * ratio / calcWidth, 0.01);
                opProps.cropLeft = ratio;
                opProps.cropRight = ratio;
            }

            if (type === 'fill') {
                if (imgRatio < 1) { // width < height
                    // width = frameWidth
                    calcHeight = frameWidth / imgRatio;

                    if (calcHeight < frameHeight) {
                        calcWidth = frameHeight * imgRatio;
                        adjustCropWidth();
                    } else {
                        adjustCropHeight();
                    }
                } else {
                    // height = frameHeight
                    calcWidth = frameHeight * imgRatio;

                    if (calcWidth < frameWidth) {
                        calcHeight = frameWidth / imgRatio;
                        adjustCropHeight();
                    } else {
                        adjustCropWidth();
                    }
                }
            } else {
                if (imgRatio > 1) { // width > height
                    // width = frameWidth
                    calcHeight = frameWidth / imgRatio;

                    if (calcHeight > frameHeight) {
                        calcWidth = frameHeight * imgRatio;
                        adjustCropWidth();
                    } else {
                        adjustCropHeight();
                    }
                } else {
                    // height = frameHeight
                    calcWidth = frameHeight * imgRatio;

                    if (calcWidth > frameWidth) {
                        calcHeight = frameWidth / imgRatio;
                        adjustCropHeight();
                    } else {
                        adjustCropWidth();
                    }
                }
            }
            return opProps;
        })
        .fail(function () {
            docModel.getApp().rejectEditAttempt('image');
        });

        return promise;
    };

    /**
     * Calls open dialog for image cropping.
     *
     * @param {docView} view
     *  Document view instance.
     *
     * @returns {Dialog}
     *
     */
    ImageCropFrame.openCropDialog = function (view) {
        return new ImageCropDialog(view, ImageCropFrame).show();
    };

    // exports ================================================================

    return ImageCropFrame;
});
