/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/headerpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/trackingpane',
    'io.ox/office/spreadsheet/view/popup/headercontextmenu',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Forms, Iterator, Tracking, SheetUtils, PaneUtils, TrackingPane, HeaderContextMenu, RenderUtils, gt) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Interval = SheetUtils.Interval;

    // margin at trailing border of the entire sheet area, in pixels
    var SCROLL_AREA_MARGIN = 30;

    // size of the resizer nodes, in pixels
    var RESIZER_SIZE = 7;

    // tooltip labels for resizers
    var COL_RESIZER_VISIBLE = gt('Drag to change column width');
    var COL_RESIZER_HIDDEN = gt('Drag to show hidden column');
    var ROW_RESIZER_VISIBLE = gt('Drag to change row height');
    var ROW_RESIZER_HIDDEN = gt('Drag to show hidden row');

    // class HeaderPane =======================================================

    /**
     * Represents a single row or column header pane in the spreadsheet view.
     * If the view has been split or frozen at a specific cell position, the
     * view will consist of up to four header panes (left and right column
     * header panes, and top and bottom row header panes).
     *
     * Triggers the following events:
     * - 'change:scrollsize'
     *      After the size of the scrollable area in this header pane has been
     *      changed. Event handlers receive the new scroll area size.
     * - 'change:scrollpos'
     *      After the absolute scroll position of this header pane has been
     *      changed. Event handlers receive the new scroll position.
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the start column/row.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the index of the tracked column/row has changed while
     *      selection tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the current column/row.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the last column/row.
     *      (3) {Boolean} success
     *          True if selecting has finished successfully, false if selecting
     *          has been canceled.
     * - 'resize:start'
     *      After column/row resize tracking has been started. Receives the
     *      current offset and initial size of the column/row.
     * - 'resize:move'
     *      After the new column/row size has changed while resize tracking is
     *      active. Receives the current offset and size of the column/row.
     * - 'resize:end'
     *      After column/row resize tracking has been finished (either
     *      successfully or not).
     *
     * @constructor
     *
     * @extends TrackingPane
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this header pane.
     *
     * @param {String} paneSide
     *  The position of this header pane (one of 'left', 'right', 'top',
     *  'bottom', or 'corner').
     */
    var HeaderPane = TrackingPane.extend({ constructor: function (docView, paneSide) {

        // self reference
        var self = this;

        // the spreadsheet model and view
        var docModel = docView.getDocModel();

        // whether this header pane contains columns
        var columns = PaneUtils.isColumnSide(paneSide);

        // the model and collections of the active sheet
        var sheetModel = null;
        var colRowCollection = null;

        // whether this header pane precedes another header pane
        var leading = PaneUtils.isLeadingSide(paneSide);

        // the name of the scroll anchor sheet view attribute
        var anchorName = PaneUtils.getScrollAnchorName(paneSide);

        // the container node of this header pane (must be focusable for tracking support)
        var rootNode = $('<div class="header-pane" tabindex="-1" data-side="' + paneSide + '" data-orientation="' + (columns ? 'columns' : 'rows') + '">');

        // the content node with the header cell and resizer nodes
        var contentNode = $('<div class="header-content noI18n">').appendTo(rootNode);

        // the context menu for the header cells
        var contextMenu = null;

        // size of the trailing scroll bar shown in associated grid panes (trailing panes only)
        var scrollBarSize = leading ? 0 : columns ? Utils.SCROLLBAR_WIDTH : Utils.SCROLLBAR_HEIGHT;

        // the size of this header pane usable to show column/row header cells
        var paneSize = 0;

        // whether this header pane is frozen (not scrollable)
        var frozen = false;

        // distance between cell A1 and the top-left visible cell of this pane
        var hiddenSize = 0;

        // the column/row interval to be displayed in the DOM (null indicates hidden header pane)
        var interval = null;

        // the absolute position and size of the displayed column/row interval in the sheet
        var position = { offset: 0, size: 0 };

        // the absolute scroll position in the associated grid panes
        var scrollPos = 0;

        // the total size of the scroll area
        var scrollSize = 0;

        // base constructors --------------------------------------------------

        TrackingPane.call(this, docView, rootNode);

        // private methods ----------------------------------------------------

        /**
         * Initializes all sheet-dependent class members according to the
         * current active sheet.
         */
        function changeActiveSheetHandler() {
            sheetModel = docView.getSheetModel();
            colRowCollection = columns ? sheetModel.getColCollection() : sheetModel.getRowCollection();
        }

        /**
         * Refreshes the appearance of this header pane according to the
         * position of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            // which header pane is currently focused (while selection tracking is active)
            var focusPaneSide = sheetModel.getViewAttribute('activePaneSide');
            // which grid pane is currently focused
            var focusPanePos = sheetModel.getViewAttribute('activePane');
            // whether this grid pane appears focused
            var focused = false;

            if (sheetModel.hasFrozenSplit()) {
                // always highlight all header panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // header pane if it is active or in the opposite direction
                focused = (focusPaneSide === paneSide) || (PaneUtils.isColumnSide(focusPaneSide) !== columns);
            } else {
                // a single grid pane is focused
                focused = (PaneUtils.getColPaneSide(focusPanePos) === paneSide) || (PaneUtils.getRowPaneSide(focusPanePos) === paneSide);
            }

            rootNode.toggleClass(Forms.FOCUSED_CLASS, focused);
        }

        /**
         * Refreshes the position of the touch resizer handle node.
         *
         * @param {Number} [size]
         *  A custom size of the column/row the resizer node is attached to. If
         *  omitted, uses the current size of the column/row.
         */
        function updateTouchResizerPosition(size) {

            var resizerNode = contentNode.children('.touchresizer');
            var index = Utils.getElementAttributeAsInteger(resizerNode, 'data-index');
            var entry = colRowCollection.getEntry(index);
            var offset = entry.offset + (_.isNumber(size) ? size : entry.size) - position.offset;

            resizerNode.css(columns ? 'left' : 'top', offset + 'px');
        }

        /**
         * Redraws the selection highlighting in the header cells of this
         * header pane.
         */
        function renderSelection() {

            // the selected ranges
            var ranges = docView.getSelectedRanges();
            // the merged ranges of selected columns/rows
            var selectionIntervals = ranges.intervals(columns).merge();
            // current index into the intervals array
            var arrayIndex = 0;

            // find the first selection interval that ends at or before the passed index
            function findSelectionInterval(index) {
                while ((arrayIndex < selectionIntervals.length) && (selectionIntervals[arrayIndex].last < index)) { arrayIndex += 1; }
                return selectionIntervals[arrayIndex];
            }

            // find the first interval, and iterate all cell nodes
            findSelectionInterval(interval.first);
            Utils.iterateSelectedDescendantNodes(contentNode, '.cell', function (cellNode) {
                var index = Utils.getElementAttributeAsInteger(cellNode, 'data-index');
                var selectionInterval = findSelectionInterval(index);
                var selected = _.isObject(selectionInterval) && (selectionInterval.first <= index);
                Forms.selectNodes(cellNode, selected);
            }, undefined, { children: true });

            // on touch devices, update the resize handle for entire columns/rows
            if (Utils.TOUCHDEVICE) {
                if ((ranges.length === 1) && docModel.isFullRange(ranges.first(), columns)) {
                    contentNode.children('.touchresizer').show().attr('data-index', ranges.first().getEnd(columns));
                    updateTouchResizerPosition();
                } else {
                    contentNode.children('.touchresizer').hide();
                }
            }
        }

        /**
         * Updates the tool tips of all resizer nodes debounced.
         */
        var updateResizerToolTips = this.createDebouncedMethod('HeaderPane.updateResizerToolTips', _.noop, function () {
            Forms.setToolTip(contentNode.children('.resizer:not(.collapsed)'), columns ? COL_RESIZER_VISIBLE : ROW_RESIZER_VISIBLE);
            Forms.setToolTip(contentNode.children('.resizer.collapsed'), columns ? COL_RESIZER_HIDDEN : ROW_RESIZER_HIDDEN);
        }, { delay: 250 });

        /**
         * Renders all header cells in the current column/row interval.
         */
        function renderCells() {

            // the entry descriptors needed for rendering
            var entries = [];
            // the HTML mark-up for all header cells to be rendered
            var markup = '';
            // whether the previous entry was hidden
            var prevHidden = false;
            // name of specific CSS attribute
            var OFFSET_NAME = columns ? 'left' : 'top';
            var SIZE_NAME = columns ? 'width' : 'height';

            // returns the mark-up of an offset CSS property
            function getOffset(offset) {
                return OFFSET_NAME + ':' + offset + 'px;';
            }

            // returns the mark-up of a size CSS property
            function getSize(size) {
                return SIZE_NAME + ':' + size + 'px;';
            }

            // returns the mark-up of an offset and a size CSS property
            function getPosition(offset, size) {
                return getOffset(offset) + getSize(size);
            }

            // creates a new entry in the 'entries' array according to the passed column/row descriptor
            function createEntry(entryDesc) {

                // the descriptor for the current entry
                var entry = { index: entryDesc.index, size: entryDesc.size };

                // start offset of the cell node and resizer node
                entry.offset = entryDesc.offset - position.offset;
                entry.resizerOffset = entry.offset + entryDesc.size;

                // adjust resizer position according to hidden state
                if (entry.size > 0) {
                    entry.resizerOffset -= Math.ceil(RESIZER_SIZE / 2);
                } else if (entries.length > 0) {
                    _.last(entries).resizerOffset -= Math.ceil(RESIZER_SIZE / 2);
                }

                // store new descriptor in result array
                entries.push(entry);
            }

            // create a descriptor array first (needed to manipulate preceding entries before rendering)
            var entryIt = colRowCollection.createIterator(interval, { visible: true });
            Iterator.forEach(entryIt, function (entryDesc) {

                // if the current column/row follows a hidden column/row, create an extra data element for that
                if ((entryDesc.index > 0) && (entryDesc.index === entryDesc.uniqueInterval.first)) {
                    var prevEntryDesc = colRowCollection.getEntry(entryDesc.index - 1);
                    if (prevEntryDesc.size === 0) {
                        createEntry(prevEntryDesc);
                    }
                }

                // create the data object for the current column/row
                createEntry(entryDesc);
            });

            // add another entry for the trailing hidden columns/rows
            if ((entries.length === 0) || (_.last(entries).index < interval.last)) {
                createEntry(colRowCollection.getEntry(interval.last));
            }

            // set font size and line height (column headers only, at cell level for row headers) at content node
            contentNode.css('font-size', docView.getHeaderFontSize() + 'pt');
            if (columns) { contentNode.css('line-height', (rootNode.height() - 1) + 'px'); }

            // generate HTML mark-up text for all header cells
            entries.forEach(function (entry) {

                // whether the current entry is hidden and precedes a visible entry
                var hidden = entry.size === 0;
                // entry index as element attribute
                var dataAttr = ' data-index="' + entry.index + '"';

                // generate mark-up for the header cell node
                if (hidden) {
                    markup += '<div class="marker" style="' + getOffset(entry.offset) + '"><div><div></div></div></div>';
                } else {
                    markup += '<div class="cell' + (prevHidden ? ' prev-hidden' : '') + '"';
                    markup += ' style="' + getPosition(entry.offset, entry.size);
                    if (!columns) { markup += 'line-height:' + (entry.size - 1) + 'px;'; }
                    markup += '"' + dataAttr + '>' + Address.stringifyIndex(entry.index, columns) + '</div>';
                }

                // generate mark-up for the resizer node
                if (!Utils.TOUCHDEVICE) {
                    markup += '<div class="resizer' + (hidden ? ' collapsed' : '') + '"';
                    markup += ' style="' + getPosition(entry.resizerOffset, RESIZER_SIZE) + '"';
                    markup += dataAttr + '></div>';
                }

                // update the prevHidden flag
                prevHidden = hidden;
            });

            // generate mark-up for global resizer node for touch devices
            if (Utils.TOUCHDEVICE) {
                markup += '<div class="touchresizer">' + Forms.createIconMarkup(columns ? 'fa-arrows-h' : 'fa-arrows-v') + '</div>';
            }

            // insert entire HTML mark-up into the cell container node
            contentNode[0].innerHTML = markup;
            updateResizerToolTips();
        }

        /**
         * Returns the maximum possible size of the scrolling area, according
         * to the size of the entire sheet, ignoring the current used area.
         */
        function getMaxScrollSize() {
            // add fixed margin after the sheet, add scroll bar size in trailing header panes
            return colRowCollection.getTotalSize() - hiddenSize + SCROLL_AREA_MARGIN + scrollBarSize;
        }

        /**
         * Returns the scrolling position reduced to the valid scrolling range,
         * according to the size of the entire sheet, ignoring the current used
         * area.
         */
        function getValidScrollPos(pos) {
            return Math.max(0, Math.min(pos, getMaxScrollSize() - paneSize));
        }

        /**
         * Updates the size of the scrollable area according to the used area
         * in the sheet, and the current scroll position. Tries to reserve
         * additional space for the full size of this header pane at the
         * trailing border, to be able to scroll page-by-page into the unused
         * area of the sheet.
         */
        var updateScrollAreaSize = (function () {

            // calculates the new scroll area size, and triggers the 'change:scrollsize' event
            function calcScrollAreaSize(newScrollPos) {

                // the used area of the sheet (zero for empty sheets)
                var usedRange = sheetModel.getUsedRange();
                // number of used columns/rows in the sheet
                var usedCount = usedRange ? usedRange.size(columns) : 0;
                // the total size of the used area in the sheet, in pixels
                var usedSize = Math.max(0, colRowCollection.getEntry(usedCount).offset - hiddenSize);
                // the absolute size of the scrollable area
                var newScrollSize = newScrollPos + paneSize;

                // scrollable area includes the used area, and the passed scroll position,
                // expanded by the own client size to be able to scroll forward
                newScrollSize = Math.max(usedSize, newScrollSize) + paneSize;

                // restrict to the maximum scrolling size of the sheet
                newScrollSize = Math.min(newScrollSize, getMaxScrollSize());

                // trigger event, if the new scroll area size has changed
                if (scrollSize !== newScrollSize) {
                    scrollSize = newScrollSize;
                    self.trigger('change:scrollsize', scrollSize);
                }
            }

            // direct callback: updates the scroll size immediately, if the new scroll position would be outside
            function updateDirect(newScrollPos) {
                if (newScrollPos >= scrollSize - paneSize) {
                    calcScrollAreaSize(newScrollPos);
                }
            }

            // deferred callback: updates the scroll size according to the current scroll position (with trailing margin)
            function updateDeferred() {
                calcScrollAreaSize(scrollPos);
            }

            return self.createDebouncedMethod('HeaderPane.updateScrollAreaSize', updateDirect, updateDeferred, { delay: 500 });
        }());

        /**
         * Updates the relative scroll position and size of the content node,
         * and notifies listeners about the changed scroll position.
         */
        function updateScrollPosition(newScrollPos) {

            // restrict scroll position to the maximum possible scroll position
            // (prevent that header pane scrolls more than the grid panes can scroll)
            scrollPos = getValidScrollPos(newScrollPos);

            // the relative offset of the content node
            var contentOffset = position.offset - (scrollPos + hiddenSize);
            // maximum size (restrict to sheet size)
            var contentSize = Math.min(position.size, colRowCollection.getTotalSize() - position.offset);

            // set the position and size of the content node
            if (columns) {
                contentNode.css({ left: contentOffset, width: contentSize });
            } else {
                contentNode.css({ top: contentOffset, height: contentSize });
            }

            // notify listeners (do not check whether the scroll position really has
            // changed, depending grid panes may have a wrong scroll position initially
            // due to wrong size of the scroll area)
            self.trigger('change:scrollpos', scrollPos);
        }

        /**
         * Returns the complete column/row interval that can be displayed in
         * this header pane. If the active sheet does not contain a frozen
         * column/row split, this interval will contain all columns/rows of the
         * sheet, otherwise the interval will contain the available columns or
         * rows according to the position of this header pane, and the current
         * split settings.
         *
         * @returns {Interval}
         *  The complete column/row interval available in this header pane.
         */
        function getAvailableInterval() {

            // the frozen column/row interval
            var frozenInterval = null;

            // get frozen interval from sheet model
            if (sheetModel.hasFrozenSplit()) {
                frozenInterval = columns ? sheetModel.getSplitColInterval() : sheetModel.getSplitRowInterval();
            }

            // return entire column/row interval of the sheet in unfrozen views
            if (!frozenInterval) {
                return new Interval(0, docModel.getMaxIndex(columns));
            }

            // build the available interval according to the own position
            return leading ? frozenInterval : new Interval(frozenInterval.last + 1, docModel.getMaxIndex(columns));
        }

        /**
         * Updates this header pane after specific sheet view attributes have
         * been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // refresh focus display
            if (Utils.hasProperty(attributes, /^activePane/)) {
                updateFocusDisplay();
            }

            // selection changed: repaint the selected header cells, cancel resize tracking
            if ('selection' in attributes) {
                renderSelection();
                if (self.getTrackingType() === 'resize') {
                    Tracking.cancelTracking();
                }
            }
        }

        /**
         * Returns the layout data of the column/row matching the 'data-index'
         * attribute of the target DOM node in the passed 'tracking:start'
         * event.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event.
         *
         * @returns {Object}
         *  The layout data of the matching column/row, in the properties
         *  'index', 'offset', and 'size'.
         */
        function getTrackingStartEntryData(event) {
            var indexNode = $(event.target).closest('[data-index]');
            return colRowCollection.getEntry(Utils.getElementAttributeAsInteger(indexNode, 'data-index', -1));
        }

        /**
         * Returns the current absolute sheet position in the header pane for
         * the passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Number}
         *  The absolute sheet position, in pixels.
         */
        function getTrackingOffset(event) {
            var eventOffset = columns ? (event.pageX - contentNode.offset().left) : (event.pageY - contentNode.offset().top);
            return eventOffset + self.getRenderPosition().offset;
        }

        /**
         * Returns the column/row collection entry matching the position in the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Object}
         *  The column/row collection entry for the tracking position.
         */
        function getTrackingEntry(event, options) {
            return colRowCollection.getEntryByOffset(getTrackingOffset(event), Utils.extendOptions(options, { pixel: true }));
        }

        /**
         * Handles all tracking events while cell selection is active, and
         * triggers the appropriate selection events to the own listeners.
         */
        var selectionTrackingHandler = (function () {

            // current index (prevent updates while tracking over the same column/row)
            var currentIndex = 0;

            // initializes selection tracking
            function initializeTracking(event) {
                currentIndex = getTrackingStartEntryData(event).index;
                self.scrollToEntry(currentIndex);
                sheetModel.setViewAttribute('activePaneSide', paneSide);
                self.trigger('select:start', currentIndex, PaneUtils.getSelectionMode(event));
            }

            // updates the current index according to the passed tracking event
            function updateTracking(event) {
                var index = getTrackingEntry(event, { outerHidden: true }).index;
                if (index !== currentIndex) {
                    currentIndex = index;
                    self.trigger('select:move', currentIndex);
                }
            }

            // finalizes selection tracking
            function finalizeTracking(event, success) {
                self.scrollToEntry(currentIndex);
                sheetModel.setViewAttribute('activePaneSide', null);
                self.trigger('select:end', currentIndex, success);

                // fire 'contextmenu'-event
                if (Utils.TOUCHDEVICE &&  event.duration > 750) {
                    contentNode.find('.cell[data-index="' + currentIndex + '"]').trigger('contextmenu');
                }
            }

            // return the actual selectionTrackingHandler() method
            return function (event, endPromise) {
                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(columns ? event.scrollX : event.scrollY);
                        updateTracking(event);
                        break;
                    case 'tracking:end':
                    case 'tracking:cancel':
                        endPromise.always(function (success) {
                            finalizeTracking(event, success);
                        });
                        break;
                }
            };
        }()); // end of selectionTrackingHandler() local scope

        /**
         * Handles all tracking events while resizing a column/row is active.
         */
        var resizeTrackingHandler = (function () {

            // the layout data of the resized column/row
            var entryData = null;
            // difference between exact start position and trailing position of column/row
            var correction = 0;
            // the current size of the column/row while tracking, in pixels
            var currentSize = 0;
            // minimum/maximum scroll position allowed while resizing
            var minScrollPos = 0, maxScrollPos = 0;
            // maximum size of columns/rows
            var maxSize = 0;

            // initializes column/row resizing according to the passed tracking event
            function initializeTracking(event) {

                entryData = getTrackingStartEntryData(event);
                maxSize = Utils.convertHmmToLength(docModel.getMaxColRowSizeHmm(columns), 'px', 1);

                // calculate difference between exact tracking start position and end of column/row
                // (to prevent unwanted resize actions without moving the tracking point)
                var position = self.getVisiblePosition();
                var hiddenSize = position.offset - self.getScrollPos();
                correction = getTrackingOffset(event) - (entryData.offset + entryData.size);
                currentSize = entryData.size;
                minScrollPos = Math.max(0, entryData.offset - 20 - hiddenSize);
                maxScrollPos = Math.max(minScrollPos, entryData.offset + maxSize + 20 - position.size - hiddenSize);
                self.trigger('resize:start', entryData.offset, currentSize);
            }

            // updates column/row resizing according to the passed tracking event
            function updateTracking(event) {
                currentSize = Utils.minMax(getTrackingOffset(event) - correction - entryData.offset, 0, maxSize);
                if (Utils.TOUCHDEVICE) { updateTouchResizerPosition(currentSize); }
                self.trigger('resize:move', entryData.offset, currentSize);
            }

            // updates scroll position according to the passed tracking event
            function updateScroll(event) {
                var scrollPos = self.getScrollPos();
                var newScrollPos = scrollPos + (columns ? event.scrollX : event.scrollY);
                if ((newScrollPos < scrollPos) && (scrollPos > minScrollPos)) {
                    self.scrollTo(Math.max(newScrollPos, minScrollPos));
                } else if ((newScrollPos > scrollPos) && (scrollPos < maxScrollPos)) {
                    self.scrollTo(Math.min(newScrollPos, maxScrollPos));
                }
            }

            // finalizes the resize tracking
            function finalizeTracking() {
                if (Utils.TOUCHDEVICE) { updateTouchResizerPosition(); }
                self.trigger('resize:end');
            }

            // return the actual resizeTrackingHandler() method
            return function (event, endPromise) {
                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        break;
                    case 'tracking:scroll':
                        updateScroll(event);
                        updateTracking(event);
                        break;
                    case 'tracking:end':
                    case 'tracking:cancel':
                        endPromise.always(function (success) {
                            finalizeTracking(event);
                            if (success && (currentSize !== entryData.size)) {
                                // pass tracked column/row as target index, will cause to modify all columns/rows selected completely
                                var newSize = docView.convertPixelToHmm(currentSize);
                                docView[columns ? 'setColumnWidth' : 'setRowHeight'](newSize, { custom: true, target: entryData.index });
                            }
                        });
                        break;
                }
            };
        }()); // end of resizeTrackingHandler() local scope

        /**
         * Handles 'dblclick' events on resizer nodes and resizes the
         * column/row to the optimal width/height depending on cell content.
         */
        function doubleClickHandler(event) {

            // the column/row index
            var index = colRowCollection.getEntry(Utils.getElementAttributeAsInteger(event.target, 'data-index', -1)).index;

            if (columns) {
                docView.setOptimalColumnWidth(index);
            } else {
                docView.setOptimalRowHeight(index);
            }

            // return focus to active grid pane, after the event has been processed
            self.executeDelayed(function () { docView.grabFocus(); }, 'HeaderPane.doubleClickHandler');
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the root DOM node of this header pane.
         *
         * @returns {jQuery}
         *  The root node of this header pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns the content node of this header pane, containing the cell
         * nodes and resize tracking nodes.
         *
         * @returns {jQuery}
         *  The content node of this header pane, as jQuery object.
         */
        this.getContentNode = function () {
            return contentNode;
        };

        /**
         * Sets the browser focus into a visible grid pane associated to this
         * header pane, unless text edit mode is currently active in another
         * grid pane.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            var panePos = PaneUtils.getNextPanePos(sheetModel.getViewAttribute('activePane'), paneSide);
            docView.getVisibleGridPane(panePos).grabFocus();
            return this;
        };

        /**
         * Initializes the settings and layout of this header pane.
         *
         * @param {Object} settings
         *  The view settings of the pane side represented by this header pane.
         *  Supports the following properties:
         *  - {Number} settings.offset
         *      The absolute offset of the header pane in the view root node,
         *      in pixels.
         *  - {Number} settings.size
         *      The size of the header pane (width of column panes, height of
         *      row panes), in pixels. The value zero will have the effect to
         *      hide the entire header pane.
         *  - {Boolean} settings.frozen
         *      Whether the header pane and the associated grid panes are
         *      frozen (not scrollable in the main direction).
         *  - {Number} settings.hiddenSize
         *      The total size of the hidden columns/rows in front of the sheet
         *      in frozen view mode.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (settings) {

            // copy passed settings
            paneSize = settings.size;
            frozen = settings.frozen;
            hiddenSize = settings.hiddenSize;

            // initialize according to visibility
            if (paneSize > 0) {

                // show pane root node and initialize auto-scrolling
                rootNode.show();
                this.enableTracking({
                    autoScroll: frozen ? false : (columns ? 'horizontal' : 'vertical'),
                    borderNode: rootNode
                });

                // position and size
                rootNode.css(columns ?
                    { left: settings.offset, width: paneSize, height: docView.getHeaderHeight() } :
                    { top: settings.offset, width: docView.getHeaderWidth(), height: paneSize }
                );

                // update CSS classes at root node for focus display
                updateFocusDisplay();

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide();
                this.disableTracking();
            }

            return this;
        };

        /**
         * Recalculates the current row/column interval and its position and
         * size in the sheet, and renders all visible header cells and the
         * selection on demand.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.hide=false]
         *      If set to true, the layer interval will be explicitly hidden.
         *  - {Boolean} [options.force=false]
         *      If set to true, the layer interval will always be recalculated
         *      regardless of the current scroll position. This option will be
         *      ignored, if the option 'hide' has been set.
         *
         * @returns {HeaderBoundary|Null}
         *  A descriptor containing the new layer interval and position (if the
         *  properties in this object are null, the layer interval has been
         *  hidden); or null, if nothing has changed in this header pane.
         */
        this.updateLayerInterval = function (options) {

            // invalidates the layer interval and other members, and returns the descriptor for the caller
            function invalidateIntervalAndReturn() {

                // nothing to do, if the interval is already invalid
                if (!interval) { return null; }

                // reset members, return 'hide' flag
                interval = position = null;
                scrollPos = scrollSize = 0;
                return new PaneUtils.HeaderBoundary();
            }

            // invalidate existing layer interval (header pane hidden, or 'hide' flag passed)
            if ((paneSize === 0) || Utils.getBooleanOption(options, 'hide', false)) {
                return invalidateIntervalAndReturn();
            }

            // the total size of the sheet in pixels
            var totalSize = colRowCollection.getTotalSize();
            // the current scroll anchor from the sheet attributes
            var scrollAnchor = sheetModel.getViewAttribute(anchorName);
            // the current effective scroll position
            var newScrollPos = getValidScrollPos(colRowCollection.convertScrollAnchorToPixel(scrollAnchor) - hiddenSize);
            // the absolute position of the first visible pixel
            var visibleOffset = newScrollPos + hiddenSize;
            // the absolute offset behind the last visible pixel
            var visibleEndOffset = visibleOffset + paneSize;
            // the additional size before and after the visible area
            var additionalSize = frozen ? 0 : Math.ceil(paneSize * PaneUtils.ADDITIONAL_SIZE_RATIO);
            // whether updating the interval is required due to passed refresh settings
            var forceRefresh = Utils.getBooleanOption(options, 'force', false);

            // check if the layer interval does not need to be recalculated: layer interval exists, update not forced,
            // and the visible area of the header pane is inside the layer interval (with a small distance)
            if (interval && position && !forceRefresh && (frozen ||
                (((position.offset <= 0) || (visibleOffset - position.offset >= additionalSize / 3)) &&
                ((totalSize <= position.offset + position.size) || (position.offset + position.size - visibleEndOffset >= additionalSize / 3))
                ))) {
                updateScrollPosition(newScrollPos);
                return null;
            }

            // exact pixel size for the new interval with additional buffer size
            // (round up to multiples of 100 to reduce expensive resizing while rendering)
            var boundSize = Utils.roundUp(paneSize + 2 * additionalSize, 100);
            // exact pixel position for the new interval
            var boundOffset = visibleOffset - Math.round((boundSize - paneSize) / 2);
            // the available interval
            var availableInterval = getAvailableInterval();
            // the data of the new first column/row
            var firstEntryData = colRowCollection.getEntryByOffset(boundOffset, { pixel: true });
            // the data of the new last column/row
            var lastEntryData = colRowCollection.getEntryByOffset(boundOffset + boundSize - 1, { pixel: true });

            // calculate the new visible column/row interval
            interval = new Interval(
                Math.max(availableInterval.first, (firstEntryData.offset === 0) ? 0 : firstEntryData.index),
                Math.min(availableInterval.last, (lastEntryData.offset + lastEntryData.size >= totalSize) ? colRowCollection.getMaxIndex() : lastEntryData.index)
            );

            // set the new exact pixel position
            position = { offset: boundOffset, size: boundSize };

            // sanity check
            if (interval.last < interval.first) {
                Utils.warn('HeaderPane.updateLayerInterval(): invalid interval: side=' + paneSide + ' interval=' + interval.stringifyAs(columns));
                return invalidateIntervalAndReturn();
            }

            // update scroll size and position according to the new interval
            updateScrollAreaSize(newScrollPos);
            updateScrollPosition(newScrollPos);

            // render header cells and the selection
            renderCells();
            renderSelection();

            // return the new interval and position
            RenderUtils.log('HeaderPane.updateLayerInterval(): side=' + paneSide + ' size=' + paneSize + ' interval=' + interval.stringifyAs(columns) + ' position=' + JSON.stringify(position));
            return new PaneUtils.HeaderBoundary(interval, position);
        };

        /**
         * Returns whether this header pane is currently visible (depending on
         * its position, and the view split settings of the active sheet).
         *
         * @returns {Boolean}
         *  Whether this header pane is currently visible.
         */
        this.isVisible = function () {
            return (paneSize > 0) && _.isObject(interval);
        };

        /**
         * Registers an event handler at the specified event source object that
         * will only be invoked when this header pane is visible. See method
         * BaseObject.listenTo() for details about the parameters.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.listenToWhenVisible = function (source, type, handler) {
            return this.listenTo(source, type, function () {
                if (interval) { return handler.apply(self, arguments); }
            });
        };

        /**
         * Returns whether this header pane is frozen (cannot be scrolled).
         *
         * @returns {Boolean}
         *  Whether this header pane is frozen.
         */
        this.isFrozen = function () {
            return frozen;
        };

        /**
         * Returns the complete column/row interval that can be displayed in
         * this header pane. If the active sheet does not contain a frozen
         * column/row split, this interval will contain all columns/rows of the
         * sheet, otherwise the interval will contain the available columns or
         * rows according to the position of this header pane, and the current
         * split settings.
         *
         * @returns {Interval|Null}
         *  The complete column/row interval available in this header pane, in
         *  the zero-based index properties 'first' and 'last'; or null, if
         *  this header pane is currently hidden.
         */
        this.getAvailableInterval = function () {
            return this.isVisible() ? getAvailableInterval() : null;
        };

        /**
         * Returns the column/row interval currently shown in this header pane.
         *
         * @returns {Interval|Null}
         *  The column/row interval currently displayed in this header pane; or
         *  null, if this header pane is currently hidden.
         */
        this.getRenderInterval = function () {
            return interval ? interval.clone() : null;
        };

        /**
         * Returns the offset and size of the entire column/row interval
         * currently covered by this header pane, including the columns/rows
         * around the visible area.
         *
         * @returns {Object|Null}
         *  The offset and size of the displayed column/row interval in pixels;
         *  or null, if this header pane is currently hidden.
         */
        this.getRenderPosition = function () {
            return _.clone(position);
        };

        /**
         * Returns the offset and size of the visible area currently shown in
         * this header pane.
         *
         * @returns {Object}
         *  The offset and size of the visible area in this header pane, in
         *  pixels.
         */
        this.getVisiblePosition = function () {
            return { offset: hiddenSize + scrollPos, size: paneSize };
        };

        /**
         * Returns the current scroll position of this header pane.
         *
         * @returns {Number}
         *  The current scroll position, in pixels.
         */
        this.getScrollPos = function () {
            return scrollPos;
        };

        /**
         * Returns the size of the scrollable area according to the used area
         * in the sheet, and the current scroll position.
         *
         * @returns {Number}
         *  The size of the scrollable area, in pixels.
         */
        this.getScrollSize = function () {
            return scrollSize;
        };

        /**
         * Returns the zero-based index of the first visible column/row in this
         * header pane. The column/row is considered visible, it at least half
         * of its total size is really visible.
         *
         * @returns {Number}
         *  The zero-based index of the first visible column/row.
         */
        this.getFirstVisibleIndex = function () {

            // the current scroll anchor
            var scrollAnchor = sheetModel.getViewAttribute(anchorName);
            // the column/row index of the scroll anchor
            var scrollIndex = Math.floor(scrollAnchor);
            // the layout data of the next column/row
            var nextEntry = null;

            // check if the cell is visible less than half, go to next cell in that case
            if ((scrollAnchor % 1 > 0.5) || !colRowCollection.isEntryVisible(scrollIndex)) {
                nextEntry = colRowCollection.getNextVisibleEntry(scrollIndex + 1);
            }

            return nextEntry ? nextEntry.index : scrollIndex;
        };

        /**
         * Changes the scroll position of this header pane, and triggers a
         * 'change:scrollpos' event that causes all associated grid panes to
         * update their own scroll position, and optionally a debounced
         * 'change:scrollsize' event, if the total size of the scroll area has
         * been changed due to the new scroll position.
         *
         * @param {Number} pos
         *  The new scroll position in the sheet, in pixels (the value 0
         *  corresponds to the first available position; in frozen panes this
         *  value does NOT include the leading hidden part of the sheet).
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollTo = function (pos) {

            // nothing to do in frozen header panes
            if (frozen) { return this; }

            // the new scroll position (do nothing if scroll position does not change)
            var newScrollPos = getValidScrollPos(pos);
            if (scrollPos === newScrollPos) { return this; }

            // recalculate scroll area size (will be done immediately, if new scroll position is outside)
            updateScrollAreaSize(newScrollPos);

            // immediately update internal scroll position, this prevents unnecessary updates
            scrollPos = newScrollPos;

            // update scroll anchor view properties, this will trigger change
            // handler from the view that repositions the content node
            var scrollAnchor = colRowCollection.getScrollAnchorByOffset(newScrollPos + hiddenSize, { pixel: true });
            sheetModel.setViewAttribute(anchorName, scrollAnchor);

            return this;
        };

        /**
         * Changes the scroll position of this header pane relative to the
         * current scroll position.
         *
         * @param {Number} diff
         *  The difference for the scroll position, in pixels.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (diff) {
            return this.scrollTo(scrollPos + diff);
        };

        /**
         * Scrolls this header pane to make the specified area visible.
         *
         * @param {Number} offset
         *  The start offset of the area to be made visible, in pixels.
         *
         * @param {Number} size
         *  The size of the area to be made visible, in pixels.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.forceLeading=false]
         *      If set to true, the passed offset will be scrolled to the
         *      leading border of the header pane, also if the passed area is
         *      already visible.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollToPosition = function (offset, size, options) {

            // move the area offset to the leading border if specified
            if (Utils.getBooleanOption(options, 'forceLeading', false)) {
                return this.scrollTo(offset - hiddenSize);
            }

            // the end offset of the area (with little extra space to make selection border visible)
            var endOffset = offset + size + 2;
            // the absolute offset and size of the visible area
            var visibleOffset = this.getVisiblePosition().offset;
            // the inner pane side (without scroll bars in grid panes)
            var innerSize = paneSize - scrollBarSize;
            // whether the leading border is outside the visible area
            var leadingOutside = offset < visibleOffset;
            // whether the trailing border is outside the visible area
            var trailingOutside = endOffset > visibleOffset + innerSize;
            // new scrolling position
            var newScrollPos = scrollPos;

            if (leadingOutside && !trailingOutside) {
                newScrollPos = Math.max(offset, endOffset - innerSize) - hiddenSize;
            } else if (!leadingOutside && trailingOutside) {
                newScrollPos = Math.min(offset, endOffset - innerSize) - hiddenSize;
            }

            // update scroll position
            return this.scrollTo(newScrollPos);
        };

        /**
         * Scrolls this header pane to make the specified column/row visible.
         *
         * @param {Number} index
         *  The zero-based column/row index to be made visible.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.forceLeading=false]
         *      If set to true, the passed offset will be scrolled to the
         *      leading border of the header pane, also if the passed area is
         *      already visible.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollToEntry = function (index, options) {

            // the position and size of the target column/row
            var entryData = colRowCollection.getEntry(index);

            // update scroll position
            return this.scrollToPosition(entryData.offset, entryData.size, options);
        };

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Forms.addDeviceMarkers(rootNode);

        // create the context menu
        contextMenu = new HeaderContextMenu(this, columns);

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', changeActiveSheetHandler);

        // update scroll area size according to the size of the used area in the sheet
        this.listenTo(docView, 'change:usedrange', function () { updateScrollAreaSize(scrollPos); });

        // listen to view events and update own appearance (only if this pane is visible)
        this.listenToWhenVisible(docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // set new selection, or extend current selection with modifier keys
        this.registerTrackingHandler('select', '.cell', selectionTrackingHandler, { readOnly: true, keepTextEdit: true, preventScroll: true });

        // resize, hide, or show columns/rows
        this.registerTrackingHandler('resize', '.resizer,.touchresizer', resizeTrackingHandler, { preventScroll: true });

        // handle double click on resizer for columns/rows
        rootNode.on('dblclick', '.resizer', doubleClickHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            self = docModel = docView = contextMenu = null;
            sheetModel = colRowCollection = null;
        });

    } }); // class HeaderPane

    // exports ================================================================

    return HeaderPane;

});
