/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/edit/texteditorbase', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/object/triggerobject'
], function (Utils, Tracking, TriggerObject) {

    'use strict';

    // class TextEditorBase ===================================================

    /**
     * Base class for different implementations of in-place text edit modes in
     * a spreadsheet document.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetView} docView
     *  The document view that has created this instance.
     *
     * @param {String} editMode
     *  The identifier of the edit mode implemented by the subclass.
     *
     * @param {Object} [initOptions]
     *  Required and optional parameters:
     *  - {Function} initOptions.enterHandler
     *      The callback function that will be invoked from the public method
     *      TextEditorBase.enterEditMode(). Receives the active (focused) grid
     *      pane as first parameter, and the options passed to the public
     *      method as second parameter. Must return a boolean value or a
     *      promise to indicate whether the edit mode has been started
     *      successfully (true, or a promise that will be resolved); or whether
     *      the edit mode has not been started (false, or a promise that will
     *      be rejected).
     *  - {Function} [initOptions.leaveHandler]
     *      The callback function that will be invoked from the public method
     *      TextEditorBase.leaveEditMode(). Receives the options passed to the
     *      public method as first parameter. Must return a boolean value or a
     *      promise to indicate whether the edit mode has been left
     *      successfully (true, or a promise that will be resolved); or whether
     *      the edit mode has remains active (false, or a promise that will be
     *      rejected).
     *  - {Function} [initOptions.cancelHandler]
     *      The callback function that will be invoked from the public method
     *      TextEditorBase.cancelEditMode(). Must implement synchronous cleanup
     *      when leaving the edit mode.
     *  - {Function} initOptions.formatResolver
     *      The callback function that will be invoked to return the current
     *      formatting attributes.
     *  - {Function} initOptions.formatHandler
     *      The callback function that will be invoked to change the current
     *      formatting attributes. Will be called with an incomplete but
     *      non-empty attribute set as first parameter.
     *  - {Function} initOptions.focusResolver
     *      The callback fucntion that will be invoked to after the browser
     *      focus has been set to the text contents currently edited.
     */
    var TextEditorBase = TriggerObject.extend({ constructor: function (docView, editMode, initOptions) {

        // self reference (spreadsheet view instance)
        var self = this;

        // the spreadsheet application, model, and other model objects
        var docModel = docView.getDocModel();

        // callback handler that implements starting the edit mode (required)
        var enterHandler = initOptions.enterHandler;

        // callback handler that implements leaving the edit mode
        var leaveHandler = Utils.getFunctionOption(initOptions, 'leaveHandler', _.constant(true));

        // callback handler that implements canceling the edit mode
        var cancelHandler = Utils.getFunctionOption(initOptions, 'cancelHandler', _.noop);

        // callback handler that implements returning the formatting attributes (required)
        var formatResolver = initOptions.formatResolver;

        // callback handler that implements changing the formatting attributes (required)
        var formatHandler = initOptions.formatHandler;

        // callback handler that returns the browser focus target node (required)
        var focusResolver = initOptions.focusResolver;

        // current text edit mode (instance of enum EditMode, null if no edit mode is active)
        var editActive = false;

        // the active (focused) grid pane instance
        var editGridPane = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, docView);

        // protected methods --------------------------------------------------

        /**
         * Triggers the specified text edit mode event.
         *
         * @param {String} state
         *  The state notified by the events triggered by this method, e.g.
         *  'enter' when entering the text edit mode, or 'leave' when leaving
         *  the text edit mode.
         *
         * @returns {TextEditorBase}
         *  A reference to this instance.
         */
        this.implTriggerEvents = function (state) {
            var eventType = 'textedit:' + state;
            this.trigger(eventType, editMode).trigger(eventType + ':' + editMode);
            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the identifier of the text edit mode implemented by this
         * instance.
         *
         * @returns {String}
         *  The identifier of the text edit mode implemented by this instance.
         */
        this.getEditMode = function () {
            return editMode;
        };

        /**
         * Returns whether the text edit mode (either cell edit mode, or
         * drawing edit mode) is currently active.
         *
         * @returns {Boolean}
         *  Whether the text edit mode is currently active.
         */
        this.isActive = function () {
            return editActive;
        };

        /**
         * Returns the active grid pane where the current edit mode is running.
         *
         * @returns {GridPane|Null}
         *  The active grid pane where the current edit mode is running; or
         *  null if the text edit mode is currently not active.
         */
        this.getEditGridPane = function () {
            return editGridPane;
        };

        /**
         * Starts text edit mode in the active grid pane.
         *
         * @param {Object} [options]
         *  Optional parameters. All options will be passed to the callback
         *  function 'enterHandler' passed to the constructor. The supported
         *  options depend on the actual implementation.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the text edit mode has been
         *  started (or if the text edit mode is already running), or that will
         *  be rejected, if the text edit mode could not be started.
         */
        this.enterEditMode = function (options) {

            // check whether this editor is already active
            if (editActive) { return this.createResolvedPromise(); }

            // bug 47129: wait for a running document action (e.g. change multiple cells very quickly)
            var promise = docModel.waitForActionsProcessed();

            // check global document edit mode
            promise = promise.then(function () {
                if (!docView.requireEditMode()) {
                    return self.createRejectedPromise();
                }
            });

            // generic preparations for the text edit mode
            promise = promise.then(function () {

                // initialize class members
                editActive = true;
                editGridPane = docView.getActiveGridPane();

                // for safety, always cancel tracking mode when starting edit mode
                Tracking.cancelTracking();

                // invoke the subclass implementation, convert value false to rejected promise
                var result = enterHandler.call(self, editGridPane, options);
                return (result === false) ? self.createRejectedPromise() : result;
            });

            // notify listeners, if edit mode has been started successfully
            promise.done(function () {
                self.grabFocus();
                self.implTriggerEvents('enter');
            });

            // cleanup after callback handler has rejected edit mode
            promise.fail(function () {
                editActive = false;
                editGridPane = null;
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        };

        /**
         * Leaves the current text edit mode, and commits the pending changes
         * if necessary.
         *
         * @param {Object} [options]
         *  Optional parameters. All options will be passed to the callback
         *  function 'leaveHandler' passed to the constructor. The supported
         *  options depend on the actual implementation.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the current text edit mode has
         *  been left successfully; or rejected, if the implementation has
         *  decided to keep the current text edit mode active.
         */
        this.leaveEditMode = function (options) {

            // nothing to do, if text edit mode is not active
            if (!editActive) {
                return this.createResolvedPromise();
            }

            // ignore edit mode, if the document does not have edit rights anymore
            if (!docView.isEditable()) {
                this.cancelEditMode();
                return this.createResolvedPromise();
            }

            // invoke the subclass implementation, convert value false to rejected promise
            var promise = this.convertToPromise(leaveHandler.call(self, options), false);

            // deinitialize edit mode if callback handler indicates success
            promise.done(function () {
                self.cancelEditMode();
            });

            // immediately cancel active tracking cycle if edit mode cannot be left
            promise.fail(function () {
                Tracking.cancelTracking();
                // grab back focus (deferred needed in Chrome)
                _.defer(function () { self.grabFocus(); });
            });

            return promise;
        };

        /**
         * Cancels the current text edit mode immediately, without committing
         * any pending changes.
         *
         * @returns {TextEditorBase}
         *  A reference to this instance.
         */
        this.cancelEditMode = function () {

            // nothing to do, if edit mode is not active (anymore)
            if (!editActive) { return this; }
            editActive = false;

            // custom cleanup implemented in subclass
            cancelHandler.call(this);

            // clean up grid pane
            this.stopListeningTo(editGridPane);
            editGridPane = null;

            // remove custom status labels
            docView.setStatusLabel(null);

            // notify listeners
            this.implTriggerEvents('leave');

            // focus back to document view
            docView.grabFocus();

            return this;
        };

        /**
         * Returns the merged formatting attributes used in the current text
         * edit mode.
         *
         * @returns {Object|Null}
         *  The formatting attributes used in the current text edit mode; or
         *  null, if the text edit mode is not active currently. Depending on
         *  the actual edit mode, the attribute set may contain different
         *  attribute families.
         */
        this.getAttributeSet = function () {
            return editActive ? formatResolver.call(this) : null;
        };

        /**
         * Modifies formatting attributes while the text edit mode is active.
         *
         * @param {Object} [attributeSet]
         *  An incomplete attribute set with the new attributes to be applied
         *  to the current selection in the current text edit mode. If the
         *  parameter has been omitted, this method will do nothing.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the attributes have been set
         *  successfully.
         */
        this.setAttributeSet = function (attributeSet) {

            // no edit mode active
            if (!editActive) {
                return this.createRejectedPromise();
            }

            // nothing to do without attributes
            if (!_.isObject(attributeSet) || _.isEmpty(attributeSet)) {
                return this.createResolvedPromise();
            }

            // invoke the format handler, convert the result to a promise
            return this.convertToPromise(formatHandler.call(this, attributeSet));
        };

        /**
         * Returns the DOM node intended to contain the browser focus while
         * text edit mode is active.
         *
         * @returns {HTMLElement|Null}
         *  The DOM node intended to contain the browser focus while text edit
         *  mode is active; or null, if this text editor is not active.
         */
        this.getFocusNode = function () {
            return editActive ? focusResolver.call(this) : null;
        };

        /**
         * Sets the browser focus to the text contents currently edited.
         *
         * @returns {TextEditorBase}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            var focusNode = this.getFocusNode();
            if (focusNode) { Utils.setFocus(focusNode); }
            return this;
        };

        // initialization -----------------------------------------------------

        // cancel edit mode after the application loses edit rights
        this.listenTo(docModel.getApp(), 'docs:editmode:leave', function () {
            self.cancelEditMode();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            this.cancelEditMode();
            self = docView = docModel = initOptions = null;
            enterHandler = leaveHandler = cancelHandler = formatHandler = null;
        });

    } }); // class TextEditorBase

    // exports ================================================================

    return TextEditorBase;

});
