/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/scalarset', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode'
], function (Utils, ErrorCode) {

    'use strict';

    // class ScalarSet ========================================================

    /**
     * A set that stores scalar values used in spreadsheet cells (numbers,
     * strings, booleans, and error codes).
     *
     * @constructor
     *
     * @param {Object} [initOptions]
     *  Optional parameters.
     *  - {Boolean} [initOptions.withCase=false]
     *      If set to true, strings will be stored case-sensitively. Otherwise,
     *      the character case of the strings will be ignored.
     */
    function ScalarSet(initOptions) {

        this._set = {};
        this._case = Utils.getBooleanOption(initOptions, 'withCase', false);

    } // class ScalarSet

    // static methods ---------------------------------------------------------

    /**
     * Returns a unique string key for the passed scalar value. Internally, a
     * scalar set will use these string keys to store the scalar values.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  A scalar value to be converted to its string key.
     *
     * @param {Boolean} [withCase=false]
     *  If set to true, strings will be handled case-sensitively. Otherwise,
     *  the character case of the strings will be ignored when creating a key.
     *
     * @returns {String}
     *  The unique string key for the passed scalar value used by a scalar set.
     */
    ScalarSet.key = function (value, withCase) {

        if (value === null) { return 'null'; }
        if (value instanceof ErrorCode) { return 'e:' + value.key; }

        switch (typeof value) {
            case 'number': return 'n:' + value;
            case 'string': return 's:' + (withCase ? value : value.toLowerCase());
            case 'boolean': return 'b:' + value;
        }

        Utils.error('ScalarSet.key(): invalid value');
        return '';
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a unique string key for the passed scalar value. Internally,
     * this set will use these string keys to store the scalar values.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  A scalar value to be converted to its string key.
     *
     * @returns {String}
     *  The unique string key for the passed scalar value used by this set.
     */
    ScalarSet.prototype.key = function (value) {
        return ScalarSet.key(value, this._case);
    };

    /**
     * Returns whether the passed scalar value is contained in this set.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The scalar value to be tested.
     *
     * @returns {Boolean}
     *  Whether the passed scalar value is contained in this set.
     */
    ScalarSet.prototype.has = function (value) {
        return this.key(value) in this._set;
    };

    /**
     * Returns all scalar values in this set as a plain array.
     *
     * @returns {Array<Any>}
     *  All scalar values in this set as a plain array, in no specific order.
     *  If multiple strings have been added to this set that differ only in
     *  character case, only the first inserted string will be part of the
     *  returned array (unless the option 'withCase' has been passed to the
     *  constructor; in this case, all inserted strings will be returned).
     */
    ScalarSet.prototype.values = function () {
        return _.values(this._set);
    };

    /**
     * Adds a new scalar value to this set.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The scalar value to be added. If the value is already contained in this
     *  set, nothing will happen.
     *
     * @returns {Scalarset}
     *  A reference to this instance.
     */
    ScalarSet.prototype.add = function (value) {
        var key = this.key(value);
        // stick to the first inserted value, e.g. for case-insensitive strings
        if (key && !(key in this._set)) { this._set[key] = value; }
        return this;
    };

    /**
     * Removes a scalar value from this set.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The scalar value to be removed. If the value is not contained in this
     *  set, nothing will happen.
     *
     * @returns {Scalarset}
     *  A reference to this instance.
     */
    ScalarSet.prototype.remove = function (value) {
        var key = this.key(value);
        if (key) { delete this._set[key]; }
        return this;
    };

    // exports ================================================================

    return ScalarSet;

});
