/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/listcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/extarray',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, ExtArray, LocaleData, ModelObject) {

    'use strict';

    // class ListCollection ===================================================

    /**
     * Collects all defined lists of the spreadsheet document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model.
     */
    var ListCollection = ModelObject.extend({ constructor: function (docModel) {

        // the models of all defined lists
        var listArrays = new ExtArray();

        // a map with all values of all lists
        var valueMap = {};

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Adds a new list to the list-collection
         *
         * @param {Array} list
         *  The array that should be added to the list-collection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.toLowerCase=false]
         *      If set to true, the values of the given array will be modified to lowercase.
         *
         * @returns {Number}
         *  Returns the new length of the array.
         */
        this.addList = function (list, options) {
            if (!_.isArray(list)) { return false; }

            var toLower = Utils.getBooleanOption(options, 'toLowerCase', false),
                array   = toLower ? _.invoke(list, 'toLowerCase') : list;

            // fill the valueMap with new list values
            array.forEach(function (val) {
                if (val in valueMap) {
                    valueMap[val] += 1;
                } else {
                    valueMap[val] = 1;
                }
            });

            return listArrays.push(array);
        };

        /**
         * Returns all defined lists contained in this collection
         * as an array.
         *
         * @returns {Array<Array>}
         *  The array of all defined lists in this collection.
         */
        this.getLists = function () {
            return listArrays;
        };

        /**
         * Returns the array of the specified defined list.
         *
         * @param {Number} nr
         *  The index of the defined list to be returned.
         *
         * @returns {Array|Null}
         *  The array of the specified defined list, if existing; otherwise
         *  null.
         */
        this.getList = function (nr) {
            return (nr < listArrays.length) ? listArrays[nr] : null;
        };

        /**
         * Return whether any of the stored lists containing the given value
         *
         * @param  {String} value
         *  The value that should be checked
         *
         * @return {Boolean}
         *  Whether the value exists in any of the lists, or not.
         */
        this.listContains = function (value) {
            return value.toLowerCase() in valueMap;
        };

        /**
         * Iterates through all lists and invoke the callback
         *
         * @param  {Function} callback
         *  The function that should be invoked for each list.
         *
         * @return
         *  callback return value
         */
        this.each = function (callback) {
            return listArrays.forEach(callback);
        };

        /**
         * Iterates through the lists until predicate take effect
         *
         * @param {Function} predicate
         *  A predicate function to match something.
         *
         * @param {Object} [context]
         *  The calling context for the predicate function.
         *
         * @return {Boolean}
         *  Whether the value exists in any of the lists, or not.
         */
        this.some = function (predicate, context) {
            return listArrays.some(predicate, context);
        };

        // initialization -----------------------------------------------------

        // add default time-based lists (month-names, weekdays, ...)
        this.addList(LocaleData.LONG_MONTHS, { toLowerCase: true });
        this.addList(LocaleData.SHORT_MONTHS, { toLowerCase: true });
        this.addList(LocaleData.LONG_WEEKDAYS, { toLowerCase: true });
        this.addList(LocaleData.SHORT_WEEKDAYS, { toLowerCase: true });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            listArrays.clear();
            docModel = listArrays = valueMap = null;
        });

    } }); // class ListCollection

    // exports ================================================================

    return ListCollection;

});
