/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/referencefuncs', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Iterator, SheetUtils, FormulaUtils, TokenArray) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with cell range
     * references used as values (cell address arithmetics), and functions that
     * look up specific values in a range of cells.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, cell range references can be given literally,
     * or can be the result of functions or operators. The client-side formula
     * engine provides the function signature type 'ref', and the class
     * Reference whose instances represent a single cell range address, or a
     * list of several cell range addresses (e.g. as result of the reference
     * list operator).
     *
     *************************************************************************/

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range3D = SheetUtils.Range3D;
    var Scalar = FormulaUtils.Scalar;
    var Matrix = FormulaUtils.Matrix;
    var Dimension = FormulaUtils.Dimension;
    var CellRef = FormulaUtils.CellRef;
    var SheetRef = FormulaUtils.SheetRef;
    var FormulaType = FormulaUtils.FormulaType;

    // shortcuts to mathematical functions
    var floor = Math.floor;

    // private global variables ===============================================

    var SEARCH_ITER_OPTIONS = {
        emptyParam: true,
        emptyCells: false,
        acceptErrors: true
    };

    // private global functions ===============================================

    function splitOperand(context, operand, subsets) {
        var value = operand.getRawValue();
        if (operand.isReference()) {
            value = context.convertToValueRange(value);
        }
        var res = [];
        _.each(subsets, function (subset) {
            var nv = null;
            if (operand.isMatrix()) {
                nv = Matrix.generate(subset.rows, subset.cols, function (row, col) {
                    return value.get(subset.row + row, subset.col + col);
                });
            } else if (operand.isReference()) {
                nv = Range3D.create(value.start[0] + subset.col, value.start[1] + subset.row, value.start[0] + subset.col + subset.cols - 1, value.start[1] + subset.row + subset.rows - 1, value.sheet1, value.sheet2);
            } else {
                nv = value;
            }
            res.push(context.createOperand(nv));
        });

        return res;
    }

    function searchInOperand(context, lookup, inexact, searchOperand, resultOperand) {

        if (resultOperand.isScalar()) { throw ErrorCode.NA; }

        var index = searchIndexInOperand(context, lookup, inexact, searchOperand);
        var res = null;
        var iterator = context.createScalarIterator(resultOperand, SEARCH_ITER_OPTIONS);

        Iterator.some(iterator, function (value, result) {
            if (result.offset === index) {
                res = value;
                return true;
            }
        });
        return res;

    }

    function searchIndexInOperand(context, lookup, inexact, operand) {

        if (_.isUndefined(inexact)) { inexact = true; }
        if (_.isUndefined(lookup) || (lookup === null)) { throw ErrorCode.VALUE; }

        var regex = null;
        if (!inexact && _.isString(lookup)) { regex = context.convertPatternToRegExp(lookup, true); }

        var lookupType = Scalar.getType(lookup);
        var last = null;
        var lastIndex = null;
        var resultIndex = Number.NaN;
        var iterator = context.createScalarIterator(operand, SEARCH_ITER_OPTIONS);

        Iterator.some(iterator, function (value, result) {
            if (value === null) { return; }

            var index = result.offset;

            last = value;
            lastIndex = index;

            var currentType = Scalar.getType(value);
            if (currentType !== lookupType) { return; }

            var compare = context.compareScalars(value, lookup);
            if (compare === 0) {
                resultIndex = index;
                return true;
            }
            if (inexact && compare > 0) {
                if (index < 1) { throw ErrorCode.NA; }
                resultIndex = index - 1;
                return true;
            }
            if (regex && regex.test(value)) {
                resultIndex = index;
                return true;
            }
        });

        if (!_.isNaN(resultIndex)) { return resultIndex; }

        if (!_.isNaN(lastIndex) && inexact && Scalar.getType(last) === lookupType && context.compareScalars(last, lookup) < 0) {
            return lastIndex;
        }

        throw ErrorCode.NA;
    }

    function createNearestScalarResolver(context, operand) {

        // return scalar value, and an element of a matrix
        switch (operand.getType()) {

            case 'val':
                return _.constant({ value: operand.getScalar(), offset: 0 });

            case 'mat':
                var matrix = operand.getMatrix();
                return function matrixResolver(offset) {
                    return { value: matrix.getByIndex(offset), offset: offset };
                };

            case 'ref':
                var range3d = context.convertToValueRange(operand.getRanges());
                var sheet = range3d.sheet1;
                var range = range3d.toRange();
                return function rangeResolver(offset, min, max) {
                    return context.getNearestCellValue(sheet, range, offset, min, max);
                };
        }
    }

    // exports ================================================================

    return {

        ADDRESS: {
            category: 'reference',
            minParams: 2,
            maxParams: 5,
            type: 'val:str',
            signature: 'val:int val:int val:int val:bool val:str',
            resolve: function (row, col, absMode, a1Style, sheetName) {

                // the resulting address string
                var result = '';

                // default for missing or empty absolute mode (always defaults to 1)
                if (this.isMissingOrEmptyOperand(2)) {
                    absMode = 1;
                } else if ((absMode < 1) || (absMode > 4)) {
                    throw ErrorCode.VALUE;
                }

                // default for missing or empty A1 style (default value is TRUE)
                if (this.isMissingOrEmptyOperand(3)) {
                    a1Style = true;
                }

                // initialize the cell reference structure
                var cellRef = new CellRef(0, 0, (absMode % 2) > 0, absMode <= 2);
                var refAddress = new Address(0, 0);

                // get the effective column index (convert column offset in R1C1 notation to real column index)
                if (a1Style || cellRef.absCol) {
                    // absolute column index, or any column index in A1 notation (one-based)
                    cellRef.col = col - 1;
                } else if (col < 0) {
                    // negative column offset in R1C1 notation, e.g. R1C[-16383]
                    // (adjust reference address to be able to create large negative offsets)
                    var maxCol = this.docModel.getMaxCol();
                    cellRef.col = maxCol + col;
                    refAddress[0] = maxCol;
                } else {
                    // positive column offset in R1C1 notation, e.g. R1C[16383]
                    cellRef.col = col;
                    refAddress[0] = 0;
                }

                // get the effective row index (convert row offset in R1C1 notation to real row index)
                if (a1Style || cellRef.absRow) {
                    // absolute row index, or any row index in A1 notation (one-based)
                    cellRef.row = row - 1;
                } else if (row < 0) {
                    // negative row offset in R1C1 notation, e.g. R[-1048575]C1
                    // (adjust reference address to be able to create large negative offsets)
                    var maxRow = this.docModel.getMaxRow();
                    cellRef.row = maxRow + row;
                    refAddress[1] = maxRow;
                } else {
                    // positive row offset in R1C1 notation, e.g. R[1048575]C1
                    cellRef.row = row;
                    refAddress[1] = 0;
                }

                // validate the effective cell reference structure
                this.checkCellRef(cellRef);

                // start with encoded sheet name (enclosed in apostrophes if needed)
                // (not for missing or empty parameter, but for explicit empty string)
                if (_.isString(sheetName)) {

                    // enclose complex names in apostrophes
                    if ((sheetName.length > 0) && !this.formulaGrammar.isSimpleSheetName(this.docModel, sheetName, { range: true, external: true })) {
                        sheetName = SheetRef.encodeComplexSheetName(sheetName);
                    }

                    // use sheet name with an exclamation mark as separator
                    result = sheetName + '!';
                }

                // append the range address in A1 or R1C1 style
                var formulaGrammar = this.docModel.getFormulaGrammar('ui!' + (a1Style ? 'a1' : 'rc'));
                return result + formulaGrammar.formatReference(this.docModel, refAddress, null, null, cellRef, null);
            }
        },

        AREAS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'ref|deps:skip',
            resolve: function (ranges) { return ranges.length; }
        },

        CHOOSE: {
            category: 'reference',
            minParams: 2,
            type: 'any',
            signature: 'val:int|mat:pass any:lazy|deps:pass|mat:forward',
            resolve: function (index) {

                var context = this;
                var opCount = this.getOperandCount();

                // resolve an operand value according to the condition (called repeatedly in matrix context)
                function getOperand(opIndex, opType) {
                    if ((opIndex < 1) || (opIndex >= opCount)) { throw ErrorCode.VALUE; }
                    return context.getOperand(opIndex, opType);
                }

                // Under some circumstances, combine the result as matrix element-by-element from
                // the following parameters, otherwise pass through the result unmodified.
                //
                // Example (simple cell formulas):
                // =ISREF(CHOOSE(1,A1:B2)) results in TRUE (the reference A1:B2 directly)
                // =ISREF(CHOOSE({1},A1:B2)) results in FALSE (a matrix built from A1:B2)
                // =CHOOSE({1|2},{3|4},{5|6}) results in {3|6} (result matrix built element-by-element)
                //
                // Example (matrix formula with reference selector):
                // {=CHOOSE(A1:B2,{2|3},{4|5})} combines the matrixes according to A1:B2
                //
                if (
                    ((this.getContextType() !== 'val') && this.getOperand(0).isMatrix()) ||
                    ((this.getContextType() === 'mat') && !this.getOperand(0).isScalar())
                ) {

                    // the dimension of the result matrix is the boundary of all parameters
                    var matrixDim = this.getOperands(0).reduce(function (dim, operand) {
                        return Dimension.boundary(dim, operand.getDimension({ errorCode: ErrorCode.NA })); // #N/A thrown for complex references
                    }, null);

                    // build the matrix by evaluating all indexes separately
                    return this.aggregateMatrix(matrixDim, function () {
                        return getOperand(this.getOperand(0, 'val:int'), 'val:any');
                    });
                }

                // otherwise, return one of the unconverted operands
                return getOperand(index);
            }
        },

        COLUMN: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val:num',
            signature: 'ref:single|deps:skip',
            resolve: function (range) {
                // create a matrix with all column indexes in matrix context with explicit range
                if (range && (this.getContextType() === 'mat')) {
                    FormulaUtils.ensureMatrixDim(1, range.cols());
                    return Matrix.generate(1, range.cols(), function (row, col) {
                        return range.start[0] + col + 1;
                    });
                }
                // use target reference address, if parameter is missing
                var address = range ? this.getMatrixAddress(range) : this.getTargetAddress();
                // function returns one-based column index
                return address[0] + 1;
            },
            // result depends on reference address (behaves like a relative reference), if parameter is missing
            relColRef: function (count) { return count === 0; }
        },

        COLUMNS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'any|deps:skip',
            resolve: function (operand) {
                var value = operand.getRawValue();
                switch (operand.getType()) {
                    case 'val':
                        return (value instanceof ErrorCode) ? value : 1;
                    case 'mat':
                        return value.cols();
                    case 'ref':
                        return this.convertToRange(value).cols();
                    default:
                        FormulaUtils.throwInternal('ReferenceFuncs.COLUMNS.resolve(): unknown operand type');
                }
            }
        },

        FORMULATEXT: {
            category: 'reference',
            name: { ooxml: '_xlfn.FORMULATEXT', odf: 'FORMULA' },
            minParams: 1,
            maxParams: 1,
            type: 'val:str',
            recalc: 'always', // changing a formula may not change the result, thus no recalculation would be triggered
            signature: 'ref:single|deps:skip',
            resolve: function (range) {

                // always use the first cell in the range (TODO: matrix context)
                var address = range.start;

                // TODO: resolve self-reference while entering a cell or shared formula

                // getCellFormula() returns null for blank cells, and simple value cells
                var result = this.getCellFormula(range.sheet1, address);
                if (!result) { throw ErrorCode.NA; }

                // special handling for matrix formulas
                var formula = '=' + result.formula;
                return result.matrix ? ('{' + formula + '}') : formula;
            }
        },

        GETPIVOTDATA: {
            category: 'reference',
            hidden: true,
            minParams: 2,
            repeatParams: 2,
            type: 'val:any'
        },

        HLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any',
            signature: 'val any val:int val:bool',
            resolve: function (lookup, searchOperand, rowIndexParam, inexact) {
                var rowIndex = rowIndexParam - 1;
                var dim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references

                if (rowIndex < 0) { throw ErrorCode.VALUE; }
                if (rowIndex >= dim.rows) { throw ErrorCode.REF; }

                var ops = splitOperand(this, searchOperand, [{ col: 0, cols: dim.cols, row: 0, rows: 1 }, { col: 0, cols: dim.cols, row: rowIndex, rows: 1 }]);

                return searchInOperand(this, lookup, inexact, ops[0], ops[1]);
            }
        },

        INDEX: {
            category: 'reference',
            minParams: 2,
            maxParams: 4,
            type: 'any',
            recalc: 'always',
            signature: 'any|deps:skip val:int|mat:pass val:int|mat:pass val:int|mat:pass',
            resolve: function (source, row, col, area) {

                // defaults for missing operands
                if (this.isMissingOperand(2)) { col = 1; }
                if (this.isMissingOperand(3)) { area = 1; }

                // early exit for invalid parameters
                if ((row < 0) || (col < 0) || (area < 1)) { throw ErrorCode.REF; }

                // resolve "area" parameter (pick single range from range list)
                switch (source.getType()) {

                    case 'val':
                        // "row" or "col" can be zero for scalars
                        if ((area > 1) || (row > 1) || (col > 1)) { throw ErrorCode.REF; }
                        return source;

                    case 'mat':
                        if (area > 1) { throw ErrorCode.REF; }
                        var matrix = source.getRawValue();
                        if ((row > matrix.rows()) || (col > matrix.cols())) { throw ErrorCode.REF; }
                        if ((row === 0) && (col === 0)) { return matrix; }
                        if (row === 0) { return matrix.getColVector(col - 1); }
                        if (col === 0) { return matrix.getRowVector(row - 1); }
                        return matrix.get(row - 1, col - 1);

                    case 'ref':
                        // resolve single reference from reference list
                        var range = source.getRawValue()[area - 1];
                        if (!range) { throw ErrorCode.REF; }
                        if (!range.singleSheet()) { throw ErrorCode.VALUE; }
                        if ((row > range.rows()) || (col > range.cols())) { throw ErrorCode.REF; }
                        if (row > 0) { range = range.rowRange(row - 1); }
                        if (col > 0) { range = range.colRange(col - 1); }
                        // bug 55707: convert single cells to scalars in matrix mode
                        if (this.isMatrixMode() && range.single()) {
                            return this.getCellValue(range.sheet1, range.start);
                        }
                        return range;
                }

                FormulaUtils.throwInternal('INDEX(): invalid operand type');
            }
        },

        INDIRECT: {
            category: 'reference',
            minParams: 1,
            maxParams: 2,
            type: 'ref',
            recalc: 'always',
            signature: 'val:str val:bool',
            resolve: function (refText, a1Style) {

                // default for missing or empty A1 style (default value is TRUE)
                if (this.isMissingOrEmptyOperand(1)) {
                    a1Style = true;
                }

                var sheetModel = this.getRefSheetModel();
                var tokenArray = new TokenArray(sheetModel, FormulaType.LINK);
                var grammarId = 'ui!' + (a1Style ? 'a1' : 'rc');
                var refAddress = this.getTargetAddress();

                tokenArray.parseFormula(grammarId, refText, { refAddress: refAddress });

                var ranges = tokenArray.resolveRangeList({ resolveNames: 'simple', refAddress: refAddress, targetAddress: refAddress });
                if (ranges.length !== 1) { throw ErrorCode.REF; }

                return ranges.first();
            }
        },

        LOOKUP: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'any',
            signature: 'val any any',
            resolve: function (lookup, searchOperand, resultOperand) {

                if (this.isMissingOrEmptyOperand(2)) {
                    var dim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
                    var ops = [];
                    if (dim.rows < dim.cols) {
                        ops = splitOperand(this, searchOperand, [{ col: 0, cols: dim.cols, row: 0, rows: 1 }, { col: 0, cols: dim.cols, row: dim.rows - 1, rows: 1 }]);
                    } else {
                        ops = splitOperand(this, searchOperand, [{ col: 0, cols: 1, row: 0, rows: dim.rows }, { col: dim.cols - 1, cols: 1, row: 0, rows: dim.rows }]);
                    }

                    return searchInOperand(this, lookup, undefined, ops[0], ops[1]);
                } else {
                    if (resultOperand.isReference()) {
                        var sDim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
                        var rDim = resultOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
                        var resultRange = null;

                        if (rDim.cols < sDim.cols) {
                            resultRange = this.convertToValueRange(resultOperand.getRawValue());
                            resultRange.end[0] = resultRange.start[0] + sDim.cols;
                            resultOperand = this.createOperand(resultRange);
                        } else if (rDim.rows < sDim.rows) {
                            resultRange = this.convertToValueRange(resultOperand.getRawValue());
                            resultRange.end[1] = resultRange.start[1] + sDim.rows;
                            resultOperand = this.createOperand(resultRange);
                        }
                    }
                    return searchInOperand(this, lookup, undefined, searchOperand, resultOperand);
                }
            }
        },

        MATCH: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'val any|mat:pass val:num',
            resolve: function (lookupValue, lookupRange, matchType) {

                // check that search range is a vector (#VALUE! thrown for complex references)
                var dim = lookupRange.getDimension({ errorCode: ErrorCode.VALUE });
                if ((dim.rows > 1) && (dim.cols > 1)) { throw ErrorCode.NA; }

                // missing matchType: fall-back to ordered search
                if (_.isUndefined(matchType)) { matchType = 1; }

                // the result object, with (zero-based) "offset" property
                var result = (function (context) {

                    // exact match, unordered source list, pattern matching support
                    if (matchType === 0) {
                        var iterator = context.createScalarIterator(lookupRange, SEARCH_ITER_OPTIONS);
                        var exactMatcher = context.createScalarMatcher(lookupValue);
                        return Iterator.find(iterator, exactMatcher);
                    }

                    // a callback function that returns matrix elements, or the nearest non-blank cells
                    var scalarResolver = createNearestScalarResolver(context, lookupRange);
                    // a scalar matcher that compares the lookup value to a scalar value
                    var scalarMatcher = (matchType > 0) ?
                        function (value) { return context.compareScalars(value, lookupValue) <= 0; } :
                        function (value) { return context.compareScalars(value, lookupValue) >= 0; };

                    // the binary search interval
                    var minOffset = 0;
                    var maxOffset = dim.size() - 1;
                    // last matching result object with scalar value and offset
                    var prevResult = null;

                    // narrow down the offset interval
                    while (minOffset <= maxOffset) {

                        // the current offset in the binary search interval
                        var currOffset = floor((minOffset + maxOffset) / 2);
                        // fetch the scalar value from the current offset
                        var nextResult = scalarResolver(currOffset, minOffset, maxOffset);

                        // blank cells only: return previous result
                        if (!nextResult) { return prevResult; }

                        // if the scalar value matches, search in trailing interval
                        if (scalarMatcher(nextResult.value)) {
                            prevResult = nextResult;
                            minOffset = nextResult.offset + 1;
                        } else {
                            maxOffset = nextResult.offset - 1;
                        }
                    }

                    return prevResult;
                }(this));

                if (!result) { throw ErrorCode.NA; }
                return result.offset + 1;
            }
        },

        OFFSET: {
            category: 'reference',
            minParams: 3,
            maxParams: 5,
            type: 'ref',
            recalc: 'always',
            signature: 'ref|deps:skip val:int val:int val:int val:int',
            resolve: function (ranges, rows, cols, height, width) {

                // reference must contain a single range address
                var range = this.convertToRange(ranges, { errorCode: ErrorCode.VALUE });

                // apply offset
                range.start[0] += cols;
                range.start[1] += rows;
                range.end[0] += cols;
                range.end[1] += rows;

                // modify size (but not if 'height' or 'width' parameters exist but are empty)
                if (!this.isMissingOrEmptyOperand(4)) { range.end[0] = range.start[0] + width - 1; }
                if (!this.isMissingOrEmptyOperand(3)) { range.end[1] = range.start[1] + height - 1; }

                // check that the range does not left the sheet boundaries
                if (this.docModel.isValidRange(range)) { return range; }
                throw ErrorCode.REF;
            }
        },

        ROW: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val:num',
            signature: 'ref:single|deps:skip',
            resolve: function (range) {
                // create a matrix with all row indexes in matrix context with explicit range
                if (range && (this.getContextType() === 'mat')) {
                    FormulaUtils.ensureMatrixDim(range.rows(), 1);
                    return Matrix.generate(range.rows(), 1, function (row) {
                        return range.start[1] + row + 1;
                    });
                }
                // use target reference address, if parameter is missing
                var address = range ? this.getMatrixAddress(range) : this.getTargetAddress();
                // function returns one-based row index
                return address[1] + 1;
            },
            // result depends on reference address (behaves like a relative reference), if parameter is missing
            relRowRef: function (count) { return count === 0; }
        },

        ROWS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'any|deps:skip',
            resolve: function (operand) {
                var value = operand.getRawValue();
                switch (operand.getType()) {
                    case 'val':
                        return (value instanceof ErrorCode) ? value : 1;
                    case 'mat':
                        return value.rows();
                    case 'ref':
                        return this.convertToRange(value).rows();
                    default:
                        FormulaUtils.throwInternal('ReferenceFuncs.ROWS.resolve(): unknown operand type');
                }
            }
        },

        SHEET: {
            category: 'reference',
            name: { ooxml: '_xlfn.SHEET' },
            minParams: 0,
            maxParams: 1,
            type: 'val:num',
            recalc: 'always', // function result depends on index of a sheet (refresh on moving sheet)
            signature: 'any|deps:skip', // string (sheet name) or reference
            resolve: function (operand) {

                // use reference sheet, if parameter is missing
                if (!operand) { return this.getRefSheet() + 1; }

                var value = operand.getRawValue();

                // use sheet index of a reference
                if (operand.isReference()) {
                    return this.convertToRange(value).sheet1 + 1;
                }

                // convert values to strings (sheet name may look like a number or boolean)
                if (operand.isScalar()) {
                    var sheet = this.docModel.getSheetIndex(this.convertToString(value));
                    if (sheet < 0) { throw ErrorCode.NA; }
                    return sheet + 1;
                }

                // always fail for matrix operands, regardless of context
                throw ErrorCode.NA;
            }
        },

        SHEETS: {
            category: 'reference',
            name: { ooxml: '_xlfn.SHEETS' },
            minParams: 0,
            maxParams: 1,
            type: 'val:num',
            recalc: 'always', // function result depends on indexes of sheets (refresh on moving sheet)
            signature: 'ref:multi|deps:skip',
            resolve: function (range) {
                // return total number of sheets in document, if parameter is missing
                return range ? range.sheets() : this.docModel.getSheetCount();
            }
        },

        SWITCH: {
            category: 'reference',
            name: { ooxml: '_xlfn.SWITCH', odf: null },
            minParams: 3,
            // no repetition pattern, SWITCH supports a 'default value' (single parameter after a pair of parameters)!
            type: 'any',
            signature: 'val any:lazy|deps:pass any:lazy|deps:pass',
            resolve: function (selector) {

                // total number of operands
                var length = this.getOperandCount();
                // number of parameter pairs following the leading selector parameter
                var pairs = floor((length - 1) / 2);

                // try to find a parameter pair matching the selector
                for (var index = 0; index < pairs; index += 1) {
                    // immediately throw on error code
                    var value = this.getOperand(index * 2 + 1, 'val');
                    // do not convert data types while comparing (number 1 as selector DOES NOT select string '1')
                    if (this.compareScalars(selector, value) === 0) {
                        return this.getOperand(index * 2 + 2);
                    }
                }

                // return default value if provided, otherwise #N/A error code (thrown automatically)
                return this.getOperand(pairs * 2 + 1);
            }
        },

        VLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any',
            signature: 'val any val:int val:bool',
            resolve: function (lookup, searchOperand, colIndexParam, inexact) {
                var colIndex = colIndexParam - 1;
                var dim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references

                if (colIndex < 0) { throw ErrorCode.VALUE; }
                if (colIndex >= dim.cols) { throw ErrorCode.REF; }

                var ops = splitOperand(this, searchOperand, [{ col: 0, cols: 1, row: 0, rows: dim.rows }, { col: colIndex, cols: 1, row: 0, rows: dim.rows }]);

                return searchInOperand(this, lookup, inexact, ops[0], ops[1]);
            }
        }
    };

});
