/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/text/textframeutils', [
    'io.ox/office/drawinglayer/view/drawingframe'
], function (DrawingFrame) {

    'use strict';

    // static class TextFrameUtils ============================================

    /**
     * Static helper functions for handling of DOM text frame nodes embedded in
     * DOM drawing frames.
     */
    var TextFrameUtils = {};

    // static methods ---------------------------------------------------------

    /**
     * Returns the DOM text frame node embedded in the passed DOM drawing
     * frame.
     *
     * @param {jQuery|Null} drawingFrame
     *  The DOM drawing frame with an optional embedded text frame. May be
     *  null, in this case the method will return null.
     *
     * @returns {jQuery|Null}
     *  The DOM text frame embedded in the passed drawing frame; or null, if
     *  the drawing frame does not contain a text frame.
     */
    TextFrameUtils.getTextFrame = function (drawingFrame) {
        var textFrame = drawingFrame ? DrawingFrame.getTextFrameNode(drawingFrame) : null;
        return (textFrame && (textFrame.length === 1)) ? textFrame : null;
    };

    /**
     * Invokes the callback function with the DOM text frame node embedded in
     * the passed DOM drawing frame node.
     *
     * @param {jQuery|Null} drawingFrame
     *  The DOM drawing frame with an optional embedded text frame. May be
     *  null, in this case the method will do nothing, and will return null.
     *
     * @param {Function} callback
     *  The callback function that will be invoked if the passed drawing frame
     *  contains a text frame. Receives the text frame node as first parameter.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The return value of the callback function if invoked; otherwise null.
     */
    TextFrameUtils.withTextFrame = function (drawingFrame, callback, context) {
        var textFrame = TextFrameUtils.getTextFrame(drawingFrame);
        return textFrame ? callback.call(context, textFrame) : null;
    };

    /**
     * Creates a deep copy of the DOM text frame node in the passed source DOM
     * drawing frame, and inserts this copy into the target DOM drawing frame.
     * The current DOM text frame node of the target drawing frame will be
     * removed. If the source drawing frame does not contain a text frame, this
     * method will do nothing.
     *
     * @param {jQuery} sourceDrawingFrame
     *  The source DOM drawing frame with an optional embedded text frame.
     *
     * @param {jQuery} targetDrawingFrame
     *  The target DOM drawing frame that will receive the copied text frame.
     *
     * @returns {jQuery|Null}
     *  The copied DOM text frame that has been inserted into the target
     *  drawing frame; or null, if the source drawing frame does not contain a
     *  text frame.
     */
    TextFrameUtils.copyTextFrame = function (sourceDrawingFrame, targetDrawingFrame) {

        // remove the current text frame from the target drawing frame
        DrawingFrame.getTextFrameNode(targetDrawingFrame).remove();

        // the content node of the target drawing frame
        var targetContentNode = DrawingFrame.getContentNode(targetDrawingFrame);
        targetContentNode.removeClass(DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS);

        // copy the text frame from the source drawing frame to the target drawing frame
        return TextFrameUtils.withTextFrame(sourceDrawingFrame, function (sourceTextFrame) {
            targetContentNode.addClass(DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS);
            return sourceTextFrame.clone(true, true).appendTo(targetContentNode);
        });
    };

    // export =================================================================

    return TextFrameUtils;
});
