/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingcollection', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/model/indexedcollection',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/drawing/drawingcollectionmixin',
    'io.ox/office/spreadsheet/model/drawing/drawingmodel',
    'io.ox/office/spreadsheet/model/drawing/shapemodel',
    'io.ox/office/spreadsheet/model/drawing/connectormodel',
    'io.ox/office/spreadsheet/model/drawing/imagemodel',
    'io.ox/office/spreadsheet/model/drawing/chart/chartmodel',
    'io.ox/office/spreadsheet/model/drawing/groupmodel',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils'
], function (Iterator, DrawingUtils, IndexedCollection, Operations, SheetDrawingCollectionMixin, SheetDrawingModel, SheetShapeModel, SheetConnectorModel, SheetImageModel, SheetChartModel, SheetGroupModel, TextFrameUtils) {

    'use strict';

    // names of generic (cell independent) anchor attributes
    var GENERIC_ANCHOR_ATTR_NAMES = ['left', 'top', 'width', 'height'];

    // class SheetDrawingCollection ===========================================

    /**
     * Represents the drawing collection of a single sheet in a spreadsheet
     * document.
     *
     * @constructor
     *
     * @extends IndexedCollection
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    var SheetDrawingCollection = IndexedCollection.extend({ constructor: function (sheetModel) {

        // self reference
        var self = this;

        // the document model
        var docModel = sheetModel.getDocModel();

        // base constructors --------------------------------------------------

        IndexedCollection.call(this, docModel, modelFactory);
        SheetDrawingCollectionMixin.call(this, sheetModel);

        // private methods ----------------------------------------------------

        /**
         * Constructs a drawing model instance for the specified drawing type.
         *
         * @param {DrawingCollection} parentCollection
         *  The parent drawing collection that will contain the new drawing
         *  object.
         *
         * @param {String} drawingType
         *  The type of the drawing model to be created.
         *
         * @param {Object|Null} attributeSet
         *  Initial formatting attributes for the drawing model.
         *
         * @returns {DrawingModel|Null}
         *  The new drawing model instance, if the passed type is supported.
         */
        function modelFactory(parentCollection, drawingType, attributeSet) {

            // create an explicit model for supported drawing types
            switch (drawingType) {
                case 'shape':
                    return new SheetShapeModel(sheetModel, parentCollection, attributeSet);
                case 'connector':
                    return new SheetConnectorModel(sheetModel, parentCollection, attributeSet);
                case 'image':
                    return new SheetImageModel(sheetModel, parentCollection, attributeSet);
                case 'chart':
                    return new SheetChartModel(sheetModel, parentCollection, attributeSet);
                case 'group':
                    return new SheetGroupModel(sheetModel, parentCollection, attributeSet);
                case 'diagram':
                case 'ole':
                case 'undefined':
                    // create a placeholder model for known but unsupported drawing types
                    return new SheetDrawingModel(sheetModel, parentCollection, drawingType, attributeSet);
            }

            return null;
        }

        function findAllIntersections(sourceDrawingModel, compareDrawingModels, targetList) {
            var sourceRect = sourceDrawingModel.getRectangleHmm();

            _.each(compareDrawingModels, function (compareDrawingModel) {
                if (compareDrawingModel === sourceDrawingModel) { return; }

                if (sourceRect.overlaps(compareDrawingModel.getRectangleHmm())) {
                    targetList.push(compareDrawingModel);
                }
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * drawing objects from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {SheetDrawingCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // clone the contents of the source collection
            var iterator = collection.createModelIterator();
            Iterator.forEach(iterator, function (drawingModel, result) {

                // clone the source drawing model and insert it into this collection
                // (the embedded drawing models of group objects will be cloned automatically)
                var newDrawingModel = drawingModel.clone(sheetModel, this);
                this.implInsertModel([result.index], newDrawingModel);

                // clone the DOM text contents of the drawing model, and all embedded child models
                (function copyTextFrames(sourceModel) {

                    // copy the DOM text frame of the specified model
                    var sourceFrame = collection.getDrawingFrameForModel(sourceModel);
                    var targetFrame = self.getDrawingFrame(sourceModel.getPosition());
                    TextFrameUtils.copyTextFrame(sourceFrame, targetFrame);

                    // copy the DOM text frames of the embedded models
                    sourceModel.forEachChildModel(copyTextFrames);

                }(drawingModel));
            }, this);
        };

        /**
         * Callback handler for the document operation 'insertDrawing'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertDrawing' document operation.
         *
         * @param {Array<Number>} position
         *  The insertion position of the new drawing object in this collection
         *  (without leading sheet index).
         */
        this.applyInsertDrawingOperation = function (context, position) {
            var result = this.insertModel(position, context.getStr('type'), context.getOptObj('attrs'));
            context.ensure(result, 'cannot insert drawing');
        };

        /**
         * Callback handler for the document operation 'deleteDrawing'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteDrawing' document operation.
         *
         * @param {Array<Number>} position
         *  The position of the drawing object to be deleted in this collection
         *  (without leading sheet index).
         */
        this.applyDeleteDrawingOperation = function (context, position) {
            var result = this.deleteModel(position);
            context.ensure(result, 'cannot delete drawing');
        };

        /**
         * Callback handler for the document operation 'moveDrawing'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'moveDrawing' document operation.
         *
         * @param {Array<Number>} position
         *  The position of the drawing object to be moved in this collection
         *  (without leading sheet index).
         */
        this.applyMoveDrawingOperation = function (context, position) {

            // get the target position (must be located in the same sheet for now)
            var targetPos = context.getPos('to');
            context.ensure(targetPos[0] === sheetModel.getIndex(), 'cannot move drawing to another sheet');
            targetPos = targetPos.slice(1);

            // target position must be located in the same parent drawing object
            context.ensure(_.isEqual(position.slice(0, -1), targetPos.slice(0, -1)), 'invalid target position');
            var result = this.moveModel(position, _.last(targetPos));
            context.ensure(result, 'cannot move drawing');
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to insert new
         * drawing objects into this collection.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of the first new drawing object. All drawing objects
         *  will be inserted subsequently.
         *
         * @param {Array<Object>} drawingDescs
         *  An array of descriptors with the types and attribute sets for the
         *  drawing objects to be inserted. Each array element MUST be an
         *  object with the following properties:
         *  - {String} drawingDesc.type
         *      The type identifier of the drawing object.
         *  - {Object} drawingDesc.attrs
         *      Initial attributes for the drawing object, especially the
         *      anchor attributes specifying the physical position of the
         *      drawing object in the sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked every time after a new
         *  'insertDrawing' operation has been generated for a drawing object.
         *  Allows to create additional operations for the drawing objects.
         *  Receives the following parameters:
         *  (1) {SheetOperationGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation of the current drawing).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The effective document position of the current drawing object
         *      in the sheet, as inserted into the 'insertDrawing' operation.
         *  (4) {Object} data
         *      The data element from the array parameter 'drawingData' that is
         *      currently processed.
         *  The callback function may return a promise to defer processing of
         *  subsequent drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected on any error.
         */
        this.generateInsertDrawingOperations = function (generator, position, drawingDescs, callback) {

            // the index of the sheet containing this collection
            var sheet = sheetModel.getIndex();

            // generate all operations for the new drawing objects, and the undo operations
            return this.iterateArraySliced(drawingDescs, function (drawingDesc) {

                // generate the undo operations that delete the new drawings (in reversed order)
                generator.generateDrawingOperation(Operations.DELETE_DRAWING, position, null, { undo: true, prepend: true });

                // create a deep copy of the original attributes
                var attributeSet = _.copy(drawingDesc.attrs, true);

                // add the generic anchor attributes of top-level drawing objects to the attribute set
                if (position.length === 1) {

                    // the merged attribute set (with default values for all missing attributes)
                    var mergedAttributeSet = docModel.getDrawingStyles().getMergedAttributes(drawingDesc.attrs);
                    // the inactive anchor attributes
                    var anchorAttrs = self.getInactiveAnchorAttributes(mergedAttributeSet.drawing);
                    // the drawing attributes in the attribute set (create on demand)
                    var drawingAttrs = attributeSet.drawing || (attributeSet.drawing = {});

                    GENERIC_ANCHOR_ATTR_NAMES.forEach(function (attrName) {
                        if (attrName in anchorAttrs) {
                            drawingAttrs[attrName] = anchorAttrs[attrName];
                        }
                    });
                }

                // create the 'insertDrawing' operation at the current position
                generator.generateDrawingOperation(Operations.INSERT_DRAWING, position, { type: drawingDesc.type, attrs: attributeSet });
                // invoke callback function for more operations (may return a promise)
                var result = callback ? callback.call(self, generator, sheet, position.slice(), drawingDesc) : null;

                // adjust document position for the next drawing object
                position = position.slice();
                position[position.length - 1] += 1;

                // callback may want to defer processing the following drawings
                return result;
            }, 'SheetDrawingCollection.generateInsertDrawingOperations');
        };

        /**
         * Generates the operations, and the undo operations, to delete the
         * specified drawing objects from this collection.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Array<Number>>} positions
         *  An array with positions of the drawing objects to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected on any error.
         */
        this.generateDeleteDrawingOperations = function (generator, positions) {

            // sort by position, remove positions of embedded drawings whose parents will be deleted too
            DrawingUtils.optimizeDrawingPositions(positions);

            // create the delete operations (in reversed order), and the according undo operations
            return this.iterateArraySliced(positions, function (position) {

                // the drawing model to be deleted
                var drawingModel = self.getModel(position);
                // a local generator to be able to prepend all undo operations at once
                var generator2 = sheetModel.createOperationGenerator();

                // generate the operations, and insert them into the passed generator
                drawingModel.generateDeleteOperations(generator2);
                generator.appendOperations(generator2);
                generator.prependOperations(generator2, { undo: true });

            }, 'SheetDrawingCollection.generateDeleteDrawingOperations', { reverse: true });
        };

        /**
         * Generates the operations, and the undo operations, to change the
         * formatting attributes of the specified drawing objects individually.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Object>} drawingDescs
         *  An array of descriptors with the positions and attribute sets for
         *  the drawing objects to be changed. Each descriptor MUST contain the
         *  following properties:
         *  - {Array<Number>} drawingDesc.position
         *      The document position of the drawing object to be changed.
         *  - {Object} drawingDesc.attrs
         *      The (incomplete) attribute set to be applied at the drawing
         *      object.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully.
         */
        this.generateChangeDrawingOperations = function (generator, drawingDescs) {
            return this.iterateArraySliced(drawingDescs, function (drawingDesc) {
                var drawingModel = self.getModel(drawingDesc.position);
                drawingModel.generateChangeOperations(generator, drawingDesc.attrs);
            }, 'SheetDrawingCollection.generateChangeDrawingOperations');
        };

        /**
         * Generates the operations, and the undo operations, to apply the same
         * formatting attributes to multiple drawing objects.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Array<Number>>} positions
         *  An array with positions of the drawing objects to be modified.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set to be applied at all drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully.
         */
        this.generateFormatDrawingOperations = function (generator, positions, attributeSet) {
            return this.iterateArraySliced(positions, function (position) {
                var drawingModel = self.getModel(position);
                drawingModel.generateChangeOperations(generator, attributeSet);
            }, 'SheetDrawingCollection.generateFormatDrawingOperations');
        };

        this.generateReorderDrawingOperations = function (generator, positions, moveType) {

            var drawings = positions.map(this.getModel, this);
            var modelCount = self.getModelCount();
            var others = [];
            var intersections;

            _.times(modelCount, function (i) {
                var otherModel = self.getModel([i]);
                if (!_.contains(drawings, otherModel)) {
                    others.push(otherModel);
                }
            });

            if (moveType === 'backward' || moveType === 'forward') {
                intersections = [];

                drawings.forEach(function (drawing) {
                    findAllIntersections(drawing, others, intersections);
                    others = _.difference(others, intersections);
                });
            }

            if (!intersections || !intersections.length) {
                intersections = others;
            }

            if (intersections.length) {
                var afterInsertDiff;
                var i;
                var currentIndex;
                var insertIndex;

                if (moveType === 'front' || moveType === 'forward') {
                    drawings.reverse();

                    afterInsertDiff = 0;

                    if (moveType === 'front') {
                        insertIndex = null;
                    } else {
                        currentIndex = positions[0][0];
                        insertIndex = null;
                        for (i = 0; i < intersections.length; i++) {
                            if (intersections[i].getPosition()[0] > currentIndex) {
                                insertIndex = i;
                                break;
                            }
                        }

                    }
                } else {
                    afterInsertDiff = 1;

                    if (moveType === 'back') {
                        insertIndex = -1;
                    } else {
                        currentIndex = positions[0][0];
                        insertIndex = -1;
                        for (i = intersections.length - 1; i >= 0; i--) {
                            if (intersections[i].getPosition()[0] < currentIndex) {
                                insertIndex = i;
                                break;
                            }
                        }
                    }
                }

                var toIndex = 0;
                if (insertIndex === null) {
                    toIndex = modelCount - 1;
                } else if (insertIndex >= 0) {
                    toIndex = intersections[insertIndex].getPosition()[0];
                }

                positions.forEach(function (position) {
                    var fromIndex = position[0];
                    if (fromIndex !== toIndex) {
                        generator.generateMoveDrawingOperation([fromIndex], [toIndex]);
                        generator.generateMoveDrawingOperation([toIndex], [fromIndex], { undo: true, prepend: true });
                    }
                    toIndex += afterInsertDiff;
                });
            }

            return this.createResolvedPromise();
        };

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            // drawing models are owned by the base class DrawingCollection, do not destroy them here!
            self = docModel = sheetModel = null;
        });

    } }); // class SheetDrawingCollection

    // exports ================================================================

    return SheetDrawingCollection;

});
