/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/components/drawing/drawingresize', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/utils/imageutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/editconfig',
    'io.ox/office/textframework/components/drawing/imagecropframe',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/presentation/utils/presentationutils'
], function (Utils, Tracking, Rectangle, AttributeUtils, DrawingUtils, ImageUtils, DrawingFrame, Config, ImageCropFrame, Operations, DOM, Position, PresentationUtils) {

    'use strict';

    // the names of all tracking events but 'tracking:start'
    var TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';
    // the names of all tracking events
    var ALL_TRACKING_EVENT_NAMES = 'tracking:start ' + TRACKING_EVENT_NAMES;
    // class name used when dragging is active
    var ACTIVE_DRAGGING = 'activedragging';
    // class name used when resizing is active
    var ACTIVE_RESIZING = 'activeresizing';
    // class name used to define indirect modification of connector linked to active shape
    var INDIRECT_CONNECTOR_CHANGE_CLASS = 'indirectconnectorchange';
    // constant for delaying shape formatting during move/resize for different browsers
    var TEXTSHAPE_UPDATE_RESIZE_DELAY = _.browser.Chrome ? 5 : _.browser.Firefox ? 100 : 250;
    // constant for delaying connector formatting during geometry modifications for different browsers
    var CONNECTOR_UPDATE_DELAY = _.browser.Chrome ? 50 : _.browser.Firefox ? 100 : 250;

    // static class DrawingSlideResize ========================================

    var DrawingSlideResize = {};

    // static methods ---------------------------------------------------------

    /**
     * Draws a selection box for the specified drawing node and registers
     * mouse handlers for moving and resizing.
     *
     * @param {PresentationApplication} app
     *  The application instance containing the drawing.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node to be selected, as DOM node or jQuery object.
     */
    DrawingSlideResize.drawDrawingSelection = function (app, drawingNode) {

        var // the model object
            model = app.getModel(),
            // the selection object
            selection = model.getSelection(),
            // the current root node
            rootNode = model.getCurrentRootNode(),
            // the node containing the scroll bar
            scrollNode = app.getView().getContentRootNode(),
            // if it's text frame with table inside it
            containsTable = DrawingFrame.isTableDrawingFrame(drawingNode),
            // object containing information about movable and resizable
            options = { movable: true, resizable: true, rotatable: !containsTable, adjustable: true },  // -> parameter?
            // the container element used to visualize the selection
            selectionBox = null,
            // saving the selected resize node
            resizeNode = null,
            //zoom factor in floating point notation
            zoomFactor,
            // target ID of currently active root node, if existing
            target = model.getActiveTarget(),
            // an object collecting the attributes for an operation
            operationProperties = {},
            // whether this is a multi drawing selection
            isMultiSelection = false,
            // a collector for all move boxes and drawing informations
            allDrawings = null,
            allConnectors = null,
            // the empty drawing node that contains the selection node
            selectionDrawing = null,
            allConnectorsOnSlide = null;

        var updateTextFrameShapesDebounced = model.createDebouncedMethod('DrawingSlideResize.drawDrawingSelection.updateTextFrameShapesDebounced', null, updateTextFrameShapes, { delay: TEXTSHAPE_UPDATE_RESIZE_DELAY, animFrame: true });
        var updateAdjustmentShapeDebounced = model.createDebouncedMethod('DrawingSlideResize.drawDrawingSelection.updateAdjustmentShapeDebounced', null, updateAdjustmentShape, { delay: TEXTSHAPE_UPDATE_RESIZE_DELAY, animFrame: true });
        var updateConnectorsDebounced = model.createDebouncedMethod('DrawingSlideResize.drawDrawingSelection.updateConnectorsDebounced', null, updateConnectorsPreview, { delay: CONNECTOR_UPDATE_DELAY, animFrame: true });

        // debounced callback for updating text frames in allDrawings
        function updateTextFrameShapes() {
            _.each(allDrawings, function (oneDrawing) {
                if (!DrawingFrame.isTableDrawingFrame(oneDrawing.drawingNode)) {
                    updateTextFrameInShape(oneDrawing.drawingNode, oneDrawing.moveBox, { isResizeActive: true });
                }
            });
        }

        function updateAdjustmentShape() {
            _.each(allDrawings, function (oneDrawing) {
                updateTextFrameInShape(oneDrawing.drawingNode, oneDrawing.moveBox, { isResizeActive: true, modAttrs: oneDrawing.moveBox.data('modAttrs') });
            });
        }

        function updateConnectorsPreview() {
            _.each(allConnectors, function (conn) {
                updateTextFrameInShape($(conn.node), conn.moveBox, { isResizeActive: true });
            });
        }

        function isDrawingTypeConnector(drawingNode) {
            return DrawingFrame.getDrawingType(drawingNode) === 'connector';
        }

        // helper function to collect all move box information in multiple drawing selections
        function collectAndActivateAllMoveBoxes(options) {
            var tempCollection = [];

            if (selection.isMultiSelectionSupported() && selection.isMultiSelection()) {
                tempCollection = selection.getMultiSelection();
            } else {
                var drawingPos = Position.getOxoPosition(rootNode, drawingNode, 0);
                tempCollection.push({ drawing: drawingNode, startPosition: drawingPos });
            }

            _.each(tempCollection, function (drawingSelection) {
                var attrs = AttributeUtils.getExplicitAttributes(drawingSelection.drawing);
                if (PresentationUtils.isPlaceHolderAttributeSet(attrs)) {
                    attrs = model.getDrawingStyles().getElementAttributes(drawingSelection.drawing);
                }
                if (!selection.isMultiSelection() || (!isDrawingTypeConnector(drawingSelection.drawing) || !attrs.connector || !(attrs.connector.startId || attrs.connector.endId))) {
                    activateMoveBox(drawingSelection.drawing, attrs, drawingSelection.startPosition, options);
                }
            });
        }

        function activateMoveBox(drawingNode, attrs, drawingPos, options) {
            var isResize = Utils.getBooleanOption(options, 'resize', false);
            var hasNoFillNoLine = attrs && attrs.fill && attrs.fill.type === 'none' && attrs.line && attrs.line.type === 'none';
            var isNotEmptyPlaceholderDrawing = PresentationUtils.isPlaceHolderDrawing(drawingNode) && !PresentationUtils.isEmptyPlaceHolderDrawing(drawingNode);
            var isEmptyNoFillLineDrawing = hasNoFillNoLine && drawingNode.text().length === 0;
            var lockResizeAspectRatio = false;
            var id = attrs && attrs.drawing && attrs.drawing.id;
            var drawingSelectionNode = drawingNode.data('selection');

            drawingNode.addClass(ACTIVE_DRAGGING);
            if (drawingSelectionNode) { drawingSelectionNode.addClass(ACTIVE_DRAGGING); }

            if (isResize) { // don't evaluate lock ratio on move, just for resize
                drawingNode.addClass(ACTIVE_RESIZING);
                if (drawingSelectionNode) { drawingSelectionNode.addClass(ACTIVE_RESIZING); }
                lockResizeAspectRatio = (attrs && attrs.drawing && attrs.drawing.aspectLocked) || (Utils.SMALL_DEVICE && !isDrawingTypeConnector(drawingNode));
            }

            var oneMoveBox = DrawingFrame.cloneDiv(model, drawingNode, isResize);
            if ((isResize && hasNoFillNoLine) || isEmptyNoFillLineDrawing || isNotEmptyPlaceholderDrawing) {
                oneMoveBox.addClass(DrawingFrame.FRAME_WITH_TEMP_BORDER);
            }
            allDrawings.push({ drawingNode: drawingNode, id: id, attrs: attrs, moveBox: oneMoveBox, width: drawingNode.width(), height: drawingNode.height(), start: drawingPos, lockResizeAspectRatio: lockResizeAspectRatio });
            activateLinkedConnectors(allConnectorsOnSlide, id);
        }

        function clearDrawing(drawing) {
            drawing.removeClass(ACTIVE_DRAGGING + ' ' + ACTIVE_RESIZING);
            drawing.children('.copy').remove();
            var drawingSelectionNode = drawing.data('selection');
            if (drawingSelectionNode) {
                drawingSelectionNode.removeClass(ACTIVE_DRAGGING + ' ' + ACTIVE_RESIZING);
                if (isDrawingTypeConnector(drawing)) {
                    drawingSelectionNode.toggleClass(DrawingFrame.START_LINKED_CLASS, drawing.hasClass(DrawingFrame.START_LINKED_CLASS));
                    drawingSelectionNode.toggleClass(DrawingFrame.END_LINKED_CLASS, drawing.hasClass(DrawingFrame.END_LINKED_CLASS));
                }
            }
        }

        /**
         * Local method to update text frame box during resize of the shape,
         * so that text can reflow even before drawing resize is finalized.
         *
         * @param {jQuery} drawingNode
         *
         * @param {jQuery} moveBox
         *
         * @param {Object} [options]
         *  @param {Boolean} [options.isResizeActive=false]
         *     If this method is called during resizing of the shape.
         */
        function updateTextFrameInShape(drawingNode, moveBox, options) {
            var mergedAttributes = app.getModel().getDrawingStyles().getElementAttributes(drawingNode);

            if (DrawingFrame.isGroupDrawingFrame(drawingNode)) {
                moveBox.find('.textframecontent').children('img').remove();
                DrawingFrame.previewOnResizeDrawingsInGroup(app, drawingNode, mergedAttributes, options);
            } else {
                moveBox.children('img').remove();
                DrawingFrame.updateShapeFormatting(app, drawingNode, mergedAttributes, options);
            }
        }

        /**
         * Helper function to collect and prepare all drawings for rotation in multiple drawing selections
         * @param {Boolean} isMultiSelection
         *
         * @param {jQuery} drawingNode
         *
         * @param {Number} startAngle
         */
        function collectAndPrepareDrawingsForRotation(isMultiSelection, drawingNode, startAngle) {
            var tempCollection = [];
            if (isMultiSelection) {
                tempCollection = selection.getMultiSelection();
            } else {
                var drawingPos = Position.getOxoPosition(rootNode, drawingNode, 0);
                tempCollection.push({ drawing: drawingNode, startPosition: drawingPos, rotation: startAngle });

            }
            _.each(tempCollection, function (drawingSelection) {
                var drawingNode = drawingSelection.drawing;
                var drawingPos, startRotation;

                if (!DrawingFrame.isTableDrawingFrame(drawingNode)) {
                    var attrs = AttributeUtils.getExplicitAttributes(drawingNode);
                    if (PresentationUtils.isPlaceHolderAttributeSet(attrs)) {
                        attrs = model.getDrawingStyles().getElementAttributes(drawingNode);
                    }
                    if (!selection.isMultiSelection() || (!isDrawingTypeConnector(drawingNode) || !attrs.connector || !(attrs.connector.startId || attrs.connector.endId))) {
                        var hasNoFillNoLine = attrs && (!attrs.fill || attrs.fill.type === 'none') && attrs.line && attrs.line.type === 'none';
                        var id = attrs && attrs.drawing && attrs.drawing.id;

                        drawingPos = drawingSelection.startPosition;
                        startRotation = !_.isUndefined(drawingSelection.rotation) ? drawingSelection.rotation : DrawingFrame.getDrawingRotationAngle(model, drawingNode);
                        drawingNode.addClass(DrawingFrame.ROTATING_STATE_CLASSNAME);
                        drawingNode.data('selection').addClass(DrawingFrame.ROTATING_STATE_CLASSNAME);
                        if (hasNoFillNoLine) { drawingNode.addClass(DrawingFrame.FRAME_WITH_TEMP_BORDER2); }
                        allDrawings.push({ drawingNode: drawingNode, id: id, attrs: attrs, start: drawingPos, rotation: startRotation });
                        activateLinkedConnectors(allConnectorsOnSlide, id);
                    }
                }
            });
        }

        /**
         * Prepares and activates all connectors linked with given id drawing that is beeing modified currently.
         *
         * @param {Array} allConnectorsOnSlide
         *  Collection of all connectors on active slide, with properties
         * @param {String} id
         *  Currently modified drawing node's id used to search for linked connectors
         */
        function activateLinkedConnectors(allConnectorsOnSlide, id) {
            var connections = _.filter(allConnectorsOnSlide, function (c) {
                return c.attrs.drawing.id !== id &&
                    (c.startId === id || c.endId === id || c.parentStartId === id || c.parentEndId === id);
            });

            _.each(connections, function (connector) {
                var oneMoveBox = $(connector.node).children('.copy');
                oneMoveBox = oneMoveBox.length ? oneMoveBox : DrawingFrame.cloneDiv(model, $(connector.node), true);
                var connectorSelectionNode = $(connector.node).data('selection');

                $(connector.node).addClass(ACTIVE_DRAGGING + ' ' + ACTIVE_RESIZING);
                if (connectorSelectionNode) { connectorSelectionNode.addClass(ACTIVE_DRAGGING); }

                allConnectors.push({
                    node: $(connector.node),
                    moveBox: oneMoveBox,
                    attrs: connector.attrs,
                    startIndex: connector.startIndex,
                    endIndex: connector.endIndex,
                    startId: connector.startId,
                    endId: connector.endId,
                    parentStartId: connector.parentStartId,
                    parentEndId: connector.parentEndId
                });
            });
        }

        /**
         * Does the cleanup after finalize move/resize/rotate.
         *
         * @param {Array} allConnectors - array of modified connector objects
         * @returns {Array} empty array
         */
        function clearModifiedConnectors(allConnectors) {
            _.each(allConnectors, function (conn) {
                clearDrawing(conn.node);
                conn.node.removeClass(INDIRECT_CONNECTOR_CHANGE_CLASS);
            });
            return [];
        }

        function extendAndGenerateOp(generator, drawingNode, operationProperties, target) {
            PresentationUtils.extendPlaceholderProp(app, drawingNode, operationProperties);
            model.extendPropertiesWithTarget(operationProperties, target);
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

            model.updateDynFontSizeDebounced(drawingNode);
        }

        function triggerDragging() {
            app.getView().getContentRootNode().trigger('activedragging');
        }

        /**
         * Handling all tracking events while moving a drawing.
         */
        var drawingMoveHandler = (function () {

            var // the current position change (in px)
                shiftX = 0, shiftY = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the start scroll position
                startScrollX = 0, startScrollY = 0,
                // whether the move box was at least once painted
                moveBoxPainted = false,
                activeSlide = model.getSlideById(model.getActiveSlideId()),
                slideOffset,
                allSlideDrawings = null,
                // the start time of the tracking. A move operation is only generated
                // if the tracking duration is larger than Utils.MIN_MOVE_DURATION
                moveStartTime = 0;

            // initialing the moving of the drawing according to the passed tracking event
            function startMoveDrawing(event) {
                zoomFactor = app.getView().getZoomFactor();
                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();
                moveBoxPainted = false;
                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();
                allDrawings = []; // 'tracking:start' event is triggered twice consecutively, so we clean collection to prevent doubling
                allConnectors = [];
                allConnectorsOnSlide = model.getAllConnectorsOnSlide(activeSlide);
                allSlideDrawings = model.getAllDrawingsOnSlide(activeSlide);
                slideOffset = activeSlide.offset();
                moveStartTime = event.timeStamp; // saving the tracking start time

                collectAndActivateAllMoveBoxes();
            }

            // updating the drawing position according to the passed tracking event
            function updateMove(event) {

                // should be done only once
                if (drawingNode.hasClass('active-edit')) {
                    drawingNode.attr('contenteditable', 'false').removeClass('active-edit');
                    if (drawingNode.data('selection')) { drawingNode.data('selection').removeClass('active-edit'); }
                }

                // make move box visible when tracking position has moved
                DrawingFrame.toggleTracking(drawingNode.data('selection'), true);

                // reading scrollPosition again. Maybe it was not updated or not updated completely.
                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;

                shiftX = (event.pageX - event.startX + scrollX) / zoomFactor;
                shiftY = (event.pageY - event.startY + scrollY) / zoomFactor;

                // only move the moveBox, if the mouse was really moved
                if (_.isNumber(shiftX) && _.isNumber(shiftY) && ((shiftX !== 0) || (shiftY !== 0))) {
                    var updateMoveBox = function (oneDrawing) {
                        var oneDrawingNode = oneDrawing.drawingNode;
                        var flipH = DrawingFrame.isFlippedHorz(oneDrawingNode);
                        var flipV = DrawingFrame.isFlippedVert(oneDrawingNode);
                        var rotation = DrawingFrame.getDrawingRotationAngle(model, oneDrawingNode);
                        var normalizedCoords = DrawingFrame.getNormalizedMoveCoordinates(shiftX, shiftY, rotation, flipH, flipV);

                        oneDrawing.moveBox.css({ cursor: event.cursor, left: normalizedCoords.x, top: normalizedCoords.y, width: oneDrawing.width, height: oneDrawing.height });

                        var connections = DrawingUtils.getUniqueConnections(allConnectors, oneDrawing.id);
                        if (!event.ctrlKey) {
                            _.each(connections, function (connObj) {
                                var normalizedOffset = DrawingFrame.normalizeMoveOffset(model, oneDrawing.drawingNode, oneDrawing.moveBox);
                                // setting the new attributes of the drawing
                                var opDrawingAttrs = {
                                    left: Utils.convertLengthToHmm((normalizedOffset.left - slideOffset.left) / zoomFactor, 'px'),
                                    top: Utils.convertLengthToHmm((normalizedOffset.top - slideOffset.top) / zoomFactor, 'px')
                                };
                                var oneDrawingAttrs = oneDrawing.attrs.drawing;
                                var optionFlipH = DrawingFrame.isFlippedHorz(oneDrawing.drawingNode);
                                var optionFlipV = DrawingFrame.isFlippedVert(oneDrawing.drawingNode);
                                var optionObj = DrawingUtils.generateRecalcConnectorOptions(oneDrawingAttrs, opDrawingAttrs, 'move', optionFlipH, optionFlipV, oneDrawing.drawingNode, oneDrawingAttrs.rotation || 0, null, zoomFactor);
                                var newConnProp = DrawingFrame.recalcConnectorPoints(allSlideDrawings, connObj, optionObj, rootNode, zoomFactor);
                                if (newConnProp) {
                                    var newDrawingAttrs = newConnProp.attrs.drawing;
                                    var calcFlipH = !_.isUndefined(newDrawingAttrs.flipH) ? newDrawingAttrs.flipH : connObj.attrs.drawing.flipH;
                                    var calcFlipV = !_.isUndefined(newDrawingAttrs.flipV) ? newDrawingAttrs.flipV : connObj.attrs.drawing.flipV;
                                    var transformProp = DrawingUtils.getCssTransform(0, calcFlipH, calcFlipV);

                                    connObj.node.addClass(INDIRECT_CONNECTOR_CHANGE_CLASS);
                                    connObj.moveBox.css({ left: Utils.convertHmmToLength(newDrawingAttrs.left, 'px'),
                                        top: Utils.convertHmmToLength(newDrawingAttrs.top, 'px'),
                                        width: Utils.convertHmmToLength((newDrawingAttrs.width || connObj.attrs.drawing.width), 'px'),
                                        height: Utils.convertHmmToLength((newDrawingAttrs.height || connObj.attrs.drawing.height), 'px'),
                                        transform: transformProp
                                    });
                                }
                            });
                        }
                    };

                    if (event.shiftKey) { // restrict move to x or y axis only
                        if (Math.abs(shiftX) > Math.abs(shiftY)) {
                            shiftY = 0;
                        } else {
                            shiftX = 0;
                        }
                    }
                    moveBoxPainted = true;

                    _.each(allDrawings, updateMoveBox);
                    updateConnectorsDebounced();
                }
            }

            // updates scroll position according to the passed tracking event
            function updateMoveScroll(event) {

                // update scrollPosition with suggestion from event
                if (event.scrollX) {
                    scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX);
                }

                if (event.scrollY) {
                    scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY);
                }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // handling the drawing position, when moving is stopped according to the passed tracking event
            function stopMoveDrawing(event) {

                var // the operations generator
                    generator = model.createOperationGenerator(),
                    //
                    availablePos = null,
                    //
                    newPoses = null,
                    // if image crop mode is active
                    isActiveCropMode = DrawingFrame.isActiveCropping(drawingNode),
                    // if ctrl key is enabled and cropping is not active, drawing is copied
                    dragToCopy = event.ctrlKey && !isActiveCropMode,
                    // whether at least one drawing is moved above the slide
                    isMovedAboveSlide = false,
                    // the minimum time for a valid move operation (not checked in Selenium or unit tests)
                    minMoveTime = (Config.AUTOTEST || model.ignoreMinMoveTime()) ? -1 : Utils.MIN_MOVE_DURATION,
                    // whether this is a wanted move operation or just triggered by selecting the drawing
                    validOperation = (event.timeStamp - moveStartTime > minMoveTime),
                    // finalizing the move of one specified drawing
                    finalizeDrawing = function (oneDrawing) {
                        var start = oneDrawing.start;

                        if (dragToCopy) {
                            if (!availablePos) {
                                availablePos = model.getNextAvailablePositionInActiveSlide();
                                newPoses = [];
                            } else {
                                availablePos[1]++;
                            }
                            start = _.clone(availablePos);
                            newPoses.push(start);

                            var node = oneDrawing.drawingNode;
                            if (DOM.isDrawingPlaceHolderNode(node)) { node = DOM.getDrawingPlaceHolderNode(node); }

                            // for place holder drawings all attributes need to be included into the operation
                            var allAttributes = PresentationUtils.isPlaceHolderDrawing(node);

                            generator.generateDrawingOperations(node, start, { ignoreFamily: 'changes presentation', target: target, getNewDrawingId: true, allAttributes: allAttributes });
                        }

                        var normalizedOffset = DrawingFrame.normalizeMoveOffset(model, oneDrawing.drawingNode, oneDrawing.moveBox);

                        // setting the new attributes of the drawing
                        operationProperties = {
                            start: start,
                            attrs: { drawing: {
                                left: Utils.convertLengthToHmm((normalizedOffset.left - rootNode.offset().left) / zoomFactor, 'px'),
                                top: Utils.convertLengthToHmm((normalizedOffset.top - rootNode.offset().top) / zoomFactor, 'px') }
                            }
                        };

                        if (isActiveCropMode) {
                            var cropOps = ImageCropFrame.getDrawingMoveCropOp(oneDrawing.drawingNode, oneDrawing.attrs.drawing, operationProperties.attrs.drawing);
                            if (cropOps && !_.isEmpty(cropOps)) {
                                operationProperties.attrs.image = cropOps;
                            }
                        }

                        // generate attributes for disconnection of connector and shape if necessary
                        if (DrawingFrame.isConnectorDrawingFrame(oneDrawing.drawingNode) && oneDrawing.attrs && oneDrawing.attrs.connector) {
                            var startLinkId = oneDrawing.attrs.connector.startId;
                            var endLinkId = oneDrawing.attrs.connector.endId;

                            // if the linked start or end id drawing is not in the selection, disconnect
                            if (startLinkId && !_.findWhere(allDrawings, { id: startLinkId })) {
                                operationProperties.attrs.connector = { startId: null, startIndex: null };
                            }
                            if (endLinkId && !_.findWhere(allDrawings, { id: endLinkId })) {
                                operationProperties.attrs.connector = operationProperties.attrs.connector || {};
                                operationProperties.attrs.connector.endId = null;
                                operationProperties.attrs.connector.endIndex = null;
                            }
                        }

                        // marking place holder drawings in ODF as modified by the user
                        if (app.isODF() && PresentationUtils.requiresUserTransformedProperty(drawingNode)) { operationProperties.attrs.presentation = { userTransformed: true }; }

                        extendAndGenerateOp(generator, oneDrawing.drawingNode, operationProperties, target);

                        // adding operations for all drawings, that are dependent from this moved drawing
                        if (app.isODF()) { model.checkODFDependentDrawings(generator, drawingNode, operationProperties, 'move'); }

                        if (!DrawingFrame.isConnectorDrawingFrame(oneDrawing.drawingNode) && !event.ctrlKey) {
                            var connections = DrawingUtils.getUniqueConnections(allConnectors, oneDrawing.id);
                            if (connections.length) {
                                var opDrawingAttrs = operationProperties.attrs.drawing;
                                var oneDrawingAttrs = oneDrawing.attrs.drawing;
                                var optionFlipH = DrawingFrame.isFlippedHorz(oneDrawing.drawingNode);
                                var optionFlipV = DrawingFrame.isFlippedVert(oneDrawing.drawingNode);
                                var optionObj = DrawingUtils.generateRecalcConnectorOptions(oneDrawingAttrs, opDrawingAttrs, 'move', optionFlipH, optionFlipV, oneDrawing.drawingNode, oneDrawingAttrs.rotation || 0, null, zoomFactor);
                                _.each(connections, function (connObj) {
                                    var connProp = DrawingFrame.recalcConnectorPoints(allSlideDrawings, connObj, optionObj, rootNode, zoomFactor);
                                    if (connProp) {
                                        connProp.start = Position.getOxoPosition(rootNode, connObj.node, 0);
                                        extendAndGenerateOp(generator, connObj.node, connProp, target);
                                    }
                                });
                            }
                        }

                        // check whether a drawing is moved above the slide
                        if (operationProperties.attrs.drawing.top < 0) { isMovedAboveSlide = true; }
                    };

                if (validOperation && moveBoxPainted && !($(drawingNode).parent().is('.notselectable'))) { // only move drawing, if it is selectable

                    _.each(allDrawings, finalizeDrawing);

                    // apply the operations (undo group is created automatically)
                    model.applyOperations(generator);

                    if (dragToCopy) { model.getSelection().setMultiDrawingSelectionByPosition(newPoses); }

                    if (isActiveCropMode) { ImageCropFrame.refreshCropFrame(drawingNode); }

                    // update of vertical scrollbar is required, if a drawing is moved above the slide
                    if (isMovedAboveSlide) { model.trigger('update:verticalscrollbar', { keepScrollPosition: true, pos: model.getActiveSlideIndex() }); }
                }
            }

            // finalizes the move tracking
            function finalizeMoveDrawing() {
                leaveTracking();
                model.stopListeningTo(drawingNode, TRACKING_EVENT_NAMES);
                var drawingSelectionNode = drawingNode.data('selection');
                if (drawingSelectionNode) { drawingSelectionNode.off(TRACKING_EVENT_NAMES); }

                // Resetting variables for new mouseup events without mousemove
                shiftX = shiftY = scrollX = scrollY = 0;

                isMultiSelection = false;
                _.each(allDrawings, function (oneDrawing) {
                    oneDrawing.moveBox.css({ left: 0, top: 0, width: 0, height: 0 }).toggle(false);
                    updateSelectionDrawingSize(oneDrawing.drawingNode);
                    clearDrawing(oneDrawing.drawingNode);
                });
                allConnectors = clearModifiedConnectors(allConnectors);
                allDrawings = [];
            }

            function updateMoveAlways(event) {
                var dragToCopy = event.ctrlKey;
                event.cursor = dragToCopy ? 'copy' : 'move';
                // model.getNode().toggleClass('drag-to-copy', dragToCopy);
                triggerDragging();
            }

            // return the actual drawingMoveHandler() function
            return function (event) {

                // handling for text frame drawings, if the click happened inside the internal text frame element
                if ($(event.target).closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0) {
                    Tracking.disableTracking(drawingNode);
                    model.stopListeningTo(drawingNode, ALL_TRACKING_EVENT_NAMES);
                    return;
                }

                switch (event.type) {
                    case 'tracking:start':
                        // avoid moving, if the event happens for example next to a line.
                        var checkDrawingNode = DrawingFrame.isGroupDrawingFrame(drawingNode) ? DrawingFrame.getDrawingNode(event.target) : drawingNode;
                        if (!PresentationUtils.checkEventPosition(app, event, checkDrawingNode)) {
                            Tracking.disableTracking(drawingNode);
                            model.stopListeningTo(drawingNode, ALL_TRACKING_EVENT_NAMES);
                            return;
                        }
                        startMoveDrawing(event);
                        break;
                    case 'tracking:move':
                        updateMove(event);
                        event.preventDefault(); // required for avoiding moving of page on touch devices
                        event.stopPropagation();
                        break;
                    case 'tracking:scroll':
                        updateMoveScroll(event);
                        break;
                    case 'tracking:end':
                        stopMoveDrawing(event);
                        finalizeMoveDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeMoveDrawing(event);
                        break;
                }

                updateMoveAlways(event);
            };

        }()); // end of drawingMoveHandler() local scope

        /**
         * Handles all tracking events while resizing a drawing.
         */
        var drawingResizeHandler = (function () {

            var // the size of the resized drawing (in px)
                finalWidth = 0, finalHeight = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the initial scroll position
                startScrollX = 0, startScrollY = 0,
                // whether resizing is available in horizontal/vertical direction
                useX = false, useY = false,
                // whether resizing is available in left or top direction
                useLeft = false, useTop = false,
                // correction factor for resizing to the left/top
                scaleX = 0, scaleY = 0,
                // a collector for the rows of a table drawing
                allRows = null,
                // a counter for the update resize event (to check, whether the drawing was really resized, 50395)
                updateResizeCounter = 0,
                // collection of drawing rectangle properties for all drawings on active slide
                rectanglesCollection = [],
                // stored node of currently hovered shape (on connector resize over)
                hoveredShape = $(),
                // stored node of snap point in shape (on connector resize)
                storedSnapNode = null,
                // type of the drawing
                drawingType = DrawingFrame.getDrawingType(drawingNode),
                // current active slide
                activeSlide = model.getSlideById(model.getActiveSlideId()),
                // offset value of the current slide
                slideOffset,
                // start offset of the resize handle
                posStartOff,
                // collection of all connectors on slide
                allSlideDrawings = null;

            // initializes resizing the drawing according to the passed tracking event
            function startResizeDrawing(event) {

                var // evaluating which resize element is active
                    pos = $(event.target).attr('data-pos');

                // helper function to set position properties at move box(es)
                function setPositionAtMoveBox(oneMoveBox) {

                    if (useLeft) {
                        scaleX = -1;
                        oneMoveBox.css({ left: 'auto', right: 0 });
                    } else {
                        scaleX = 1;
                        oneMoveBox.css({ left: 0, right: 'auto' });
                    }
                    if (DrawingFrame.isFlippedHorz(drawingNode)) { scaleX *= -1; }

                    if (useTop) {
                        scaleY = -1;
                        oneMoveBox.css({ top: 'auto', bottom: 0 });
                    } else {
                        scaleY = 1;
                        oneMoveBox.css({ top: 0, bottom: 'auto' });
                    }
                    if (DrawingFrame.isFlippedVert(drawingNode)) { scaleY *= -1; }

                }

                updateResizeCounter = 0; // resetting the counter
                zoomFactor = app.getView().getZoomFactor();
                allConnectorsOnSlide = model.getAllConnectorsOnSlide(activeSlide);
                allSlideDrawings = model.getAllDrawingsOnSlide(activeSlide);
                slideOffset = activeSlide.offset();

                if (drawingType === 'connector') {
                    posStartOff = event.target.getBoundingClientRect();
                    posStartOff = { x: posStartOff.left + (posStartOff.width / 2), y: posStartOff.top + (posStartOff.height / 2) };
                    rectanglesCollection = model.getAllDrawingRectsOnSlide(activeSlide, slideOffset, zoomFactor);
                }
                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                // collecting information about the handle node
                useX = /[lr]/.test(pos);
                useY = /[tb]/.test(pos);

                useLeft = /[l]/.test(pos);
                useTop = /[t]/.test(pos);

                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();
                allDrawings = [];
                allConnectors = [];
                collectAndActivateAllMoveBoxes({ resize: true });

                _.each(allDrawings, function (oneDrawing) {
                    setPositionAtMoveBox(oneDrawing.moveBox);
                    oneDrawing.drawingNode.addClass(ACTIVE_DRAGGING + ' ' + ACTIVE_RESIZING);
                    var selectionDrawingNode = oneDrawing.drawingNode.data('selection');
                    if (selectionDrawingNode) { selectionDrawingNode.addClass(ACTIVE_DRAGGING + ' ' + ACTIVE_RESIZING); }
                });
            }

            // updates scroll position according to the passed tracking event
            function updateResizeScroll(event) {

                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // updates resizing the drawing according to the passed tracking event
            function updateResize(event) {

                var // the horizontal shift
                    deltaX = event.pageX / zoomFactor - event.startX / zoomFactor + scrollX,
                    // the vertical shift
                    deltaY = event.pageY / zoomFactor - event.startY / zoomFactor + scrollY,
                    // the scaling factor for the width
                    scaleWidth = 1,
                    // the scaling factor for the height
                    scaleHeight = 1,
                    // normalize and scale deltas of the dimensions
                    normalizedDeltas;

                // helper function to set the height of rows and cells in table drawing nodes
                function updateVerticalTableHeight(drawingNode, moveBox, oldHeight) {

                    var firstRun = !allRows;
                    allRows = allRows || moveBox.find('tr');
                    if (firstRun) {
                        _.each(allRows, function (row) {
                            var rowHeight = _.browser.IE ? row.offsetHeight : $(row).height(); // #57882 - jquery v3 height() doesn't return correct value in IE/Edge
                            $(row).data('startrowheight', rowHeight); // saving row height at start of resize
                        });
                    }

                    oldHeight = oldHeight || drawingNode.height;
                    var currentRatio = moveBox.height() / oldHeight; // ratio of drawing height
                    _.each(allRows, function (row) {
                        var newRowHeight = $(row).data('startrowheight') * currentRatio;
                        if (_.browser.WebKit || _.browser.IE) {
                            $(row).children('td').css('height', newRowHeight);
                        } else {
                            $(row).css('height', newRowHeight);
                        }
                    });
                }

                function clearHoveredShape() {
                    if (hoveredShape) {
                        hoveredShape.removeClass('connector-hover');
                        hoveredShape = null; // reset previously stored values
                    }
                }

                // helper function to set the current width and height of each selected drawing
                function updateDrawingDimension(drawingNode, moveBox, oldWidth, oldHeight, lockAspectRatio, oneDrawingId, oneDrawingAttrs) {
                    var canvasExpansion = 0; // an optional expansion of the canvas, for example for arrows
                    var sumWidth = oldWidth + normalizedDeltas.x;
                    var sumHeight = oldHeight + normalizedDeltas.y;
                    var signWidth = sumWidth > 0 ? 1 : -1;
                    var signHeight = sumHeight > 0 ? 1 : -1;
                    var flipH = sumWidth < 0;
                    var flipV = sumHeight < 0;
                    var isTableDrawingFrame = DrawingFrame.isTableDrawingFrame(drawingNode);

                    if (isTableDrawingFrame && (flipH || flipV)) {
                        return false;
                    }

                    var moveBoxTextframe = moveBox.find('.textframe');
                    var transformProp = 'scaleX(' + (flipH ? -1 : 1) + ') scaleY(' + (flipV ? -1 : 1) + ')';
                    var topPos, bottomPos, leftPos, rightPos;

                    // use the same scaling factor for vertical and horizontal resizing, if both are enabled
                    // -> the larger number wins
                    if (useX && useY && (lockAspectRatio || event.shiftKey) && !DrawingFrame.isActiveCropping(drawingNode)) {
                        scaleWidth = Math.abs(sumWidth / oldWidth); // scale shouldn't be negative
                        scaleHeight = Math.abs(sumHeight / oldHeight);

                        if (scaleWidth > scaleHeight) {
                            sumHeight = scaleWidth * oldHeight * signHeight;  // return sign of current width/height after scale
                        } else {
                            sumWidth = scaleHeight * oldWidth * signWidth;
                        }
                    }

                    topPos = useTop ? 'auto' : Math.min(0, sumHeight);
                    bottomPos = useTop ? Math.min(0, sumHeight) : 'auto';
                    leftPos = useLeft ? 'auto' : Math.min(0, sumWidth);
                    rightPos = useLeft ? Math.min(0, sumWidth) : 'auto';

                    // update drawing size
                    finalWidth = Math.max(1, Math.abs(sumWidth));
                    finalHeight = Math.max(1, Math.abs(sumHeight));

                    moveBox.css({ width: finalWidth, height: finalHeight, top: topPos, left: leftPos, bottom: bottomPos, right: rightPos, transform: transformProp });
                    moveBox.toggleClass(DrawingFrame.FLIPPED_HORIZONTALLY_CLASSNAME, flipH);
                    moveBox.toggleClass(DrawingFrame.FLIPPED_VERTICALLY_CLASSNAME, flipV);
                    moveBox.data({ flipH: flipH, flipV: flipV });
                    moveBoxTextframe.toggleClass('flipH', DrawingFrame.isOddFlipHVCount(moveBoxTextframe, '.slide'));

                    // make move box visible when tracking position has moved
                    DrawingFrame.toggleTracking(drawingNode.data('selection'), true);

                    // simulating height changes of tables (makes only sense for single drawing selections because of performance)
                    if (useY && !isMultiSelection && isTableDrawingFrame) { updateVerticalTableHeight(drawingNode, moveBox, oldHeight); }

                    var connections = DrawingUtils.getUniqueConnections(allConnectors, oneDrawingId);
                    _.each(connections, function (connObj) {
                        var normalizedOffset = DrawingFrame.normalizeResizeOffset(model, drawingNode, moveBox, useTop, useLeft);
                        // setting the new attributes of the drawing
                        var opDrawingAttrs = {
                            left: Utils.convertLengthToHmm((normalizedOffset.left - slideOffset.left) / zoomFactor, 'px'),
                            top: Utils.convertLengthToHmm((normalizedOffset.top - slideOffset.top) / zoomFactor, 'px'),
                            width: Utils.convertLengthToHmm(finalWidth, 'px'),
                            height: Utils.convertLengthToHmm(finalHeight, 'px')
                        };

                        var optionFlipH = flipH ? !DrawingFrame.isFlippedHorz(drawingNode) : DrawingFrame.isFlippedHorz(drawingNode);
                        var optionFlipV = flipV ? !DrawingFrame.isFlippedVert(drawingNode) : DrawingFrame.isFlippedVert(drawingNode);
                        var optionObj = DrawingUtils.generateRecalcConnectorOptions(oneDrawingAttrs.drawing, opDrawingAttrs, 'resize', optionFlipH, optionFlipV, drawingNode, oneDrawingAttrs.drawing.rotation || 0, null, zoomFactor);
                        var newConnProp = DrawingFrame.recalcConnectorPoints(allSlideDrawings, connObj, optionObj, rootNode, zoomFactor);
                        if (newConnProp) {
                            var newDrawingAttrs = newConnProp.attrs.drawing;
                            var calcFlipH = !_.isUndefined(newDrawingAttrs.flipH) ? newDrawingAttrs.flipH : connObj.attrs.drawing.flipH;
                            var calcFlipV = !_.isUndefined(newDrawingAttrs.flipV) ? newDrawingAttrs.flipV : connObj.attrs.drawing.flipV;
                            var transformProp = DrawingUtils.getCssTransform(0, calcFlipH, calcFlipV);

                            connObj.node.addClass(INDIRECT_CONNECTOR_CHANGE_CLASS);
                            connObj.moveBox.css({ left: Utils.convertHmmToLength(newDrawingAttrs.left, 'px'),
                                top: Utils.convertHmmToLength(newDrawingAttrs.top, 'px'),
                                width: Utils.convertHmmToLength((newDrawingAttrs.width || connObj.attrs.drawing.width), 'px'),
                                height: Utils.convertHmmToLength((newDrawingAttrs.height || connObj.attrs.drawing.height), 'px'),
                                transform: transformProp
                            });
                        }
                    });

                    if (!DrawingFrame.isGroupContentNode(moveBox)) {
                        canvasExpansion = drawingNode.data('canvasexpansion') || 0; // canvas expansion was saved at the drawing node
                        if (_.isObject(canvasExpansion)) { return; } // avoid flickering during resize
                        canvasExpansion *= 2;
                        moveBox.find('canvas').css({ width: finalWidth + canvasExpansion + 2, height: finalHeight + canvasExpansion + 2 }); // make smooth transition, until canvas is updated
                        // cropped images need to have calculated crop settings
                        var cropFrameImg = moveBox.find('.cropping-frame img');
                        if (cropFrameImg.length) {
                            var imageAttrs = oneDrawingAttrs.image;
                            var cropLeft = (imageAttrs && imageAttrs.cropLeft) || 0;
                            var cropRight = (imageAttrs && imageAttrs.cropRight) || 0;
                            var cropTop = (imageAttrs && imageAttrs.cropTop) || 0;
                            var cropBottom = (imageAttrs && imageAttrs.cropBottom) || 0;
                            var horzSettings = ImageUtils.calculateBitmapSettings(app, Utils.convertLengthToHmm(finalWidth, 'px'), cropLeft, cropRight);
                            var vertSettings = ImageUtils.calculateBitmapSettings(app, Utils.convertLengthToHmm(finalHeight, 'px'), cropTop, cropBottom);
                            cropFrameImg.css({ width: horzSettings.size, height: vertSettings.size, left: horzSettings.offset, top: vertSettings.offset });
                        }
                    }
                }

                if (drawingType === 'connector') {
                    var currentPoint = { x: (event.pageX - slideOffset.left), y: (event.pageY - slideOffset.top) };
                    storedSnapNode = null; // reset previously stored values
                    clearHoveredShape();
                    _.each(rectanglesCollection, function (rectObj) {
                        var isInside = DrawingUtils.pointInsideRect(currentPoint, rectObj.rect);
                        var snapPointsContainer = $(rectObj.node).children('.snap-points-container');
                        snapPointsContainer.toggleClass('connector-hover', isInside);
                        if (isInside) {
                            clearHoveredShape();
                            hoveredShape = snapPointsContainer;
                            _.each(snapPointsContainer.children(), function (snapPointNode) {
                                var snapPointRect = snapPointNode.getBoundingClientRect();
                                var snapPoint = { x: snapPointRect.left + snapPointRect.width / 2, y: snapPointRect.top + snapPointRect.height / 2 };
                                var point = { x: event.pageX, y: event.pageY };
                                // scale points according to zoom
                                $(snapPointNode).css('transform', 'scale(' + (1 / zoomFactor) + ')');

                                if (DrawingUtils.pointCloseToSnapPoint(point, snapPoint)) {
                                    // re-initialize deltas with new values of snap points
                                    deltaX = (snapPoint.x - posStartOff.x) / zoomFactor + scrollX;
                                    deltaY = (snapPoint.y - posStartOff.y) / zoomFactor + scrollY;
                                    // store value
                                    storedSnapNode = snapPointNode;
                                }
                            });
                        }
                    });
                }

                normalizedDeltas = DrawingFrame.getNormalizedResizeDeltas(deltaX, deltaY, useX, useY, scaleX, scaleY, DrawingFrame.getDrawingRotationAngle(model, drawingNode));

                _.each(allDrawings, function (oneDrawing) {
                    updateDrawingDimension(oneDrawing.drawingNode, oneDrawing.moveBox, oneDrawing.width, oneDrawing.height, oneDrawing.lockResizeAspectRatio, oneDrawing.id, oneDrawing.attrs);
                });

                // debounce updating of text frames in allDrawings on slower browsers & devices
                updateTextFrameShapesDebounced();
                updateConnectorsDebounced();

                updateResizeCounter++; // the drawing is really resized
            }

            // stop resizing of drawing
            function stopResizeDrawing() {

                var // the operations generator
                    generator = model.createOperationGenerator(),
                    // whether event was triggered while in image crop mode
                    isActiveCropMode = DrawingFrame.isActiveCropping(drawingNode),
                    // whether a operation was generated
                    operationGenerated = false;

                // helper function to generate the operation for resizing one drawing
                function generateResizeOperation(oneDrawing, drawingNode, moveBox, start) {
                    var normalizedOffset = DrawingFrame.normalizeResizeOffset(model, drawingNode, moveBox, useTop, useLeft);
                    var rootNodeOffset = rootNode.offset();

                    if (moveBox.css('display') !== 'none' && moveBox.width() && moveBox.height()) { // generate operations only if moveBox has real w/h value, measured without outer
                        // setting the new attributes of the drawing
                        operationProperties = {
                            start: start,
                            attrs: { drawing: {
                                width: Utils.convertLengthToHmm(moveBox.outerWidth(true), 'px'), // 'outer' is required to avoid shrinking in unused dimension
                                height: Utils.convertLengthToHmm(moveBox.outerHeight(true), 'px') }
                            }
                        };

                        // setting a minimum height at frames (ODP only), 55236
                        if (app.isODF() && DrawingFrame.isMinHeightDrawingFrame(drawingNode)) {
                            operationProperties.attrs.drawing.minFrameHeight = Utils.convertLengthToHmm(moveBox.outerHeight(true), 'px');
                        }

                        if (moveBox.data('flipH')) {
                            operationProperties.attrs.drawing.flipH = !DrawingFrame.isFlippedHorz(drawingNode);
                            operationProperties.attrs.drawing.left = Utils.convertLengthToHmm((normalizedOffset.left - rootNodeOffset.left) / zoomFactor, 'px');
                        }
                        if (moveBox.data('flipV')) {
                            if (app.isODF()) {
                                operationProperties.attrs.drawing.rotation = Utils.mod((oneDrawing.attrs.drawing.rotation || 0) + 180, 360);
                                operationProperties.attrs.drawing.flipH = !_.isUndefined(operationProperties.attrs.drawing.flipH) ? !operationProperties.attrs.drawing.flipH : !DrawingFrame.isFlippedHorz(drawingNode);
                            } else {
                                operationProperties.attrs.drawing.flipV = !DrawingFrame.isFlippedVert(drawingNode);
                            }
                            operationProperties.attrs.drawing.top = Utils.convertLengthToHmm((normalizedOffset.top - rootNodeOffset.top) / zoomFactor, 'px');
                        }

                        // marking place holder drawings in ODF as modified by the user
                        if (app.isODF() && PresentationUtils.requiresUserTransformedProperty(drawingNode)) { operationProperties.attrs.presentation = { userTransformed: true }; }

                        // evaluating useTop and useLeft to avoid sending of superfluous drawing information
                        if (useTop || normalizedOffset.sendMoveOperation) { operationProperties.attrs.drawing.top = Utils.convertLengthToHmm((normalizedOffset.top - rootNodeOffset.top) / zoomFactor, 'px'); }
                        if (useLeft || normalizedOffset.sendMoveOperation) { operationProperties.attrs.drawing.left = Utils.convertLengthToHmm((normalizedOffset.left - rootNodeOffset.left) / zoomFactor, 'px'); }

                        if (useX && DrawingFrame.isNoWordWrapDrawingFrame(drawingNode) && DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                            operationProperties.attrs.shape = { wordWrap: true };
                        }

                        if (drawingType === 'connector') {
                            var ind = useLeft ? 'startIndex' : 'endIndex';
                            var id = useLeft ? 'startId' : 'endId';
                            operationProperties.attrs.connector = {};
                            if (storedSnapNode) {
                                var startInd = parseInt($(storedSnapNode).data('cxnInd'), 10);
                                var drawingAttrs = model.getDrawingAttrsByNode(DrawingFrame.getDrawingNode(storedSnapNode));
                                var startId = (drawingAttrs && drawingAttrs.id) || null;
                                if (_.isNumber(startInd)) { operationProperties.attrs.connector[ind] = startInd; }
                                if (startId) {
                                    operationProperties.attrs.connector[id] = startId;
                                } else {
                                    Utils.error('DrawingResize.stopResizeDrawing(): missing drawing id for drawing: ', DrawingFrame.getDrawingNode(storedSnapNode));
                                }
                            } else { // link is disconnected
                                operationProperties.attrs.connector[ind] = null;
                                operationProperties.attrs.connector[id] = null;
                            }
                        }

                        if (isActiveCropMode) {
                            var cropOps = ImageCropFrame.getDrawingResizeCropOp(oneDrawing.drawingNode, oneDrawing.attrs.drawing, operationProperties.attrs.drawing, useLeft, useTop);
                            if (cropOps && !_.isEmpty(cropOps)) {
                                operationProperties.attrs.image = cropOps;
                            }
                        }

                        extendAndGenerateOp(generator, drawingNode, operationProperties, target);

                        // adding operations for all drawings, that are dependent from this moved drawing
                        if (app.isODF()) { model.checkODFDependentDrawings(generator, drawingNode, operationProperties, 'resize'); }

                        if (!DrawingFrame.isConnectorDrawingFrame(drawingNode)) {
                            var connections = DrawingUtils.getUniqueConnections(allConnectors, oneDrawing.id);
                            if (connections.length) {
                                var opDrawingAttrs = operationProperties.attrs.drawing;
                                var oneDrawingAttrs = oneDrawing.attrs.drawing;
                                var optionFlipH = _.isUndefined(opDrawingAttrs.flipH) ? DrawingFrame.isFlippedHorz(drawingNode) : !DrawingFrame.isFlippedHorz(drawingNode);
                                var optionFlipV = _.isUndefined(opDrawingAttrs.flipV) ? DrawingFrame.isFlippedVert(drawingNode) : !DrawingFrame.isFlippedVert(drawingNode);
                                var optionObj = DrawingUtils.generateRecalcConnectorOptions(oneDrawingAttrs, opDrawingAttrs, 'resize', optionFlipH, optionFlipV, drawingNode, oneDrawingAttrs.rotation || 0, null, zoomFactor);

                                _.each(connections, function (connObj) {
                                    var connProp = DrawingFrame.recalcConnectorPoints(allSlideDrawings, connObj, optionObj, rootNode, zoomFactor);
                                    if (connProp) {
                                        connProp.start = Position.getOxoPosition(rootNode, connObj.node, 0);
                                        extendAndGenerateOp(generator, connObj.node, connProp, target);
                                    }
                                });
                            }
                        }

                        operationGenerated = true; // setting marker for generated operation
                    }
                }

                // helper function to generate the operation for resizing one drawing of type table
                function generateTableDrawingResizeOperations(oneDrawing, drawingNode, moveBox, start, height) {

                    if (useX || useTop) {
                        // generating the drawing operation for the width or the movement upwards
                        generateResizeOperation(oneDrawing, drawingNode, moveBox, start);
                    }

                    if (useY) {
                        // increase the height of all table rows, if the drawing height has changed
                        // -> height must not be set explicitely for table drawings, because this is
                        //    determined by the height of all rows.
                        var tableNode = drawingNode.children('.content:not(.copy)').find('table');  // the table node, that is NOT inside the copy content
                        var oldHeight = height || tableNode.outerHeight(true);
                        var newHeight = moveBox.outerHeight(true);
                        var factor = newHeight / oldHeight;
                        var allRows = DOM.getTableRows(tableNode);  // using specific table node, otherwise there might be too many rows because of 'copy' node
                        var newRowHeight = 0;
                        var rowCounter = 0;

                        _.each(allRows, function (row) {
                            var rowHeight = _.browser.IE ? row.offsetHeight : $(row).outerHeight(true); // #57882 - jquery v3 height() doesn't return correct value in IE/Edge
                            newRowHeight = Utils.round(Utils.convertLengthToHmm(rowHeight, 'px') * factor, 1);
                            // -> not using the saved height in the explicit attributes, because this value might be lower
                            //    than the 'real' height, if the text content forces the row to be higher.

                            operationProperties = {
                                start: Position.appendNewIndex(start, rowCounter),
                                attrs: { row: { height: newRowHeight } }
                            };

                            extendAndGenerateOp(generator, drawingNode, operationProperties, target);

                            rowCounter++;
                        });

                        if (app.isODF() && model.useSlideMode()) { // table height must be send to the filter via operation
                            model.trigger('drawingHeight:update', drawingNode);
                        }

                        operationGenerated = true; // setting marker for generated operation
                    }
                }

                if (updateResizeCounter === 0) { return; } // no resize -> no operation

                _.each(allDrawings, function (oneDrawing) {

                    operationGenerated = false;

                    if (DrawingFrame.isTableDrawingFrame(oneDrawing.drawingNode)) {
                        generateTableDrawingResizeOperations(oneDrawing, oneDrawing.drawingNode, oneDrawing.moveBox, oneDrawing.start, oneDrawing.height);
                    } else {
                        generateResizeOperation(oneDrawing, oneDrawing.drawingNode, oneDrawing.moveBox, oneDrawing.start);
                    }

                    if (operationGenerated) { // <- do not use canvas copy, if no operation could be generated (for example after setting width to 0)
                        // prevent flickering on finalize resizing by replacing original canvas with repainted one
                        var copyCanvas = oneDrawing.moveBox.children('canvas');
                        oneDrawing.drawingNode.children('.textframecontent').not('.copy').children('canvas').replaceWith(copyCanvas);
                    }
                });

                // Applying the operations (undo group is created automatically)
                model.applyOperations(generator);
                // repaint adjustment points of shape after resize, if not multi selection
                if (!isMultiSelection && drawingNode && drawingNode.length && !isActiveCropMode) {
                    DrawingFrame.clearSelection(drawingNode);
                    selection.drawDrawingSelection(drawingNode);
                }

                if (isActiveCropMode) { ImageCropFrame.refreshCropFrame(drawingNode); }
            }

            // finalizes the resize tracking
            function finalizeResizeDrawing() {
                leaveTracking();
                resizeNode.off(TRACKING_EVENT_NAMES);

                isMultiSelection = false;
                _.each(allDrawings, function (oneDrawing) {
                    var drawingSelectionNode = oneDrawing.drawingNode.data('selection');
                    DrawingFrame.toggleTracking(drawingSelectionNode, false); // leaving tracking for all selected drawings
                    oneDrawing.moveBox.css({ left: 0, top: 0, width: 0, height: 0 }).toggle(false);
                    updateSelectionDrawingSize(oneDrawing.drawingNode);
                    // oneDrawing.drawingNode.data('selection').css({ transform: oneDrawing.drawingNode.css('transform'), left: oneDrawing.drawingNode.css('left'), top: oneDrawing.drawingNode.css('top'), width: oneDrawing.drawingNode.width(), height: oneDrawing.drawingNode.height() });
                    clearDrawing(oneDrawing.drawingNode);
                });
                allConnectors = clearModifiedConnectors(allConnectors);
                allDrawings = [];
                allRows = null;
                rectanglesCollection = [];
                if (hoveredShape && hoveredShape.length) { hoveredShape.removeClass('connector-hover'); }
                if (drawingType === 'connector') { activeSlide.find('.snap-points-container').remove(); }
            }

            // return the actual drawingResizeHandler() function
            return function (event) {

                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startResizeDrawing(event);
                        break;
                    case 'tracking:move':
                        updateResize(event);
                        break;
                    case 'tracking:scroll':
                        updateResizeScroll(event);
                        break;
                    case 'tracking:end':
                        stopResizeDrawing(event);
                        finalizeResizeDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeResizeDrawing(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();

                triggerDragging();
            };

        }()); // end of drawingResizeHandler() local scope

        var drawingRotateHandler = (function () {
            // dimensions of the drawing
            var drawingWidth = 0, drawingHeight = 0;
            // position of the center rotation of the drawingWidth
            var centerX, centerY;
            // the current scroll position
            var scrollX = 0, scrollY = 0;
            // the initial scroll position
            var startScrollX = 0, startScrollY = 0;
            // starting angle before rotation
            var startAngle = 0;
            // final angle on finish of the rotation
            var rotationAngle = null;
            // flag for marking that the operation is generated
            var generatedOp = false;
            var activeSlide = null;
            var allSlideDrawings = null;

            function startRotateDrawing() {
                var drawingOffset;
                activeSlide = model.getSlideById(model.getActiveSlideId());
                allConnectorsOnSlide = model.getAllConnectorsOnSlide(activeSlide);
                allSlideDrawings = model.getAllDrawingsOnSlide(activeSlide);
                zoomFactor = app.getView().getZoomFactor();
                startAngle = DrawingFrame.getDrawingRotationAngle(model, drawingNode);

                if (_.isNumber(startAngle) && (startAngle !== 0)) {
                    $(drawingNode).css('transform', ''); // reset to def to get values
                    drawingOffset = drawingNode.offset();
                    DrawingFrame.updateCssTransform(drawingNode, startAngle);
                } else {
                    startAngle = 0;
                    drawingOffset = drawingNode.offset();
                }
                if (selection.isAdditionalTextframeSelection()) {
                    DrawingFrame.getTextFrameNode(drawingNode, { deep: true }).attr('contenteditable', false);
                }

                drawingHeight = drawingNode.height() * zoomFactor;
                drawingWidth = drawingNode.width() * zoomFactor;
                centerX = drawingOffset.left + (drawingWidth / 2);
                centerY = drawingOffset.top + (drawingHeight / 2);

                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();
                allDrawings = []; // 'tracking:start' event is triggered twice consecutively, so we clean collection to prevent doubling
                allConnectors = [];
                collectAndPrepareDrawingsForRotation(isMultiSelection, drawingNode, startAngle);
            }

            function updateRotate(event) {
                var radians = Math.atan2(event.pageX + scrollX - centerX, event.pageY + scrollY - centerY);
                rotationAngle = Math.round(180 - radians * 180 / Math.PI);

                if (event.shiftKey) { rotationAngle = Utils.round(rotationAngle, 15); } // make 15 deg step with shift key
                if (DrawingFrame.isFlippedVert(drawingNode)) { rotationAngle += 180; }

                _.each(allDrawings, function (obj) {
                    var objAngle = Utils.mod(rotationAngle - startAngle + obj.rotation, 360);
                    var flipH = DrawingFrame.isFlippedHorz(obj.drawingNode);
                    var flipV = DrawingFrame.isFlippedVert(obj.drawingNode);
                    var scaleX = flipH ? 'scaleX(-1)' : 'scaleX(1)';
                    var scaleY = flipV ? ' scaleY(-1)' : ' scaleY(1)';
                    var selectionNode = obj.drawingNode.data('selection');
                    var oneDrawingId = obj.id;

                    DrawingFrame.updateCssTransform(obj.drawingNode, objAngle);
                    selectionNode.css('transform', 'rotate(' + objAngle + 'deg) ' + scaleX + scaleY);
                    DrawingFrame.addRotationHint(selectionNode.find(DrawingFrame.ROTATE_ANGLE_HINT_SELECTOR), objAngle, flipH, flipV);

                    var connections = DrawingUtils.getUniqueConnections(allConnectors, oneDrawingId);
                    var oneDrawingAttrs = obj.attrs.drawing;
                    var optionFlipH = DrawingFrame.isFlippedHorz(obj.drawingNode);
                    var optionFlipV = DrawingFrame.isFlippedVert(obj.drawingNode);
                    var optionObj = DrawingUtils.generateRecalcConnectorOptions(oneDrawingAttrs, oneDrawingAttrs, 'rotate', optionFlipH, optionFlipV, obj.drawingNode, objAngle || 0, obj.rotation || 0, zoomFactor);
                    _.each(connections, function (connObj) {
                        var newConnProp = DrawingFrame.recalcConnectorPoints(allSlideDrawings, connObj, optionObj, rootNode, zoomFactor);
                        if (newConnProp) {
                            var newDrawingAttrs = newConnProp.attrs.drawing;
                            var calcFlipH = !_.isUndefined(newDrawingAttrs.flipH) ? newDrawingAttrs.flipH : connObj.attrs.drawing.flipH;
                            var calcFlipV = !_.isUndefined(newDrawingAttrs.flipV) ? newDrawingAttrs.flipV : connObj.attrs.drawing.flipV;
                            var transformProp = DrawingUtils.getCssTransform(0, calcFlipH, calcFlipV);

                            connObj.node.addClass(INDIRECT_CONNECTOR_CHANGE_CLASS);
                            connObj.moveBox.css({ left: Utils.convertHmmToLength(newDrawingAttrs.left, 'px'),
                                top: Utils.convertHmmToLength(newDrawingAttrs.top, 'px'),
                                width: Utils.convertHmmToLength((newDrawingAttrs.width || connObj.attrs.drawing.width), 'px'),
                                height: Utils.convertHmmToLength((newDrawingAttrs.height || connObj.attrs.drawing.height), 'px'),
                                transform: transformProp
                            });
                        }
                    });
                });

                updateConnectorsDebounced();
            }

            function updateRotateScroll(event) {
                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            function stopRotateDrawing() {
                var generator = model.createOperationGenerator();
                var operationProperties;

                if (_.isNumber(rotationAngle)) {
                    _.each(allDrawings, function (obj) {
                        // calculate angle value for each drawing and restrict between 0 and 359 degree
                        var oneDrawingAngle = Utils.mod((rotationAngle - startAngle + obj.rotation), 360);
                        operationProperties = { start: obj.start, attrs: { drawing: { rotation: oneDrawingAngle } } };
                        // marking place holder drawings in ODF as modified by the user
                        if (app.isODF() && PresentationUtils.requiresUserTransformedProperty(drawingNode)) { operationProperties.attrs.presentation = { userTransformed: true }; }

                        if (!DrawingFrame.isConnectorDrawingFrame(obj.drawingNode)) {
                            var connections = DrawingUtils.getUniqueConnections(allConnectors, obj.id);
                            if (connections.length) {
                                var opDrawingAttrs = operationProperties.attrs.drawing;
                                var oneDrawingAttrs = obj.attrs.drawing;
                                var optionFlipH = DrawingFrame.isFlippedHorz(obj.drawingNode);
                                var optionFlipV = DrawingFrame.isFlippedVert(obj.drawingNode);
                                var optionObj = DrawingUtils.generateRecalcConnectorOptions(oneDrawingAttrs, oneDrawingAttrs, 'rotate', optionFlipH, optionFlipV, obj.drawingNode, opDrawingAttrs.rotation || 0, obj.rotation || 0, zoomFactor);

                                _.each(connections, function (connObj) {
                                    var connProp = DrawingFrame.recalcConnectorPoints(allSlideDrawings, connObj, optionObj, rootNode, zoomFactor);
                                    if (connProp) {
                                        connProp.start = Position.getOxoPosition(rootNode, connObj.node, 0);
                                        extendAndGenerateOp(generator, connObj.node, connProp, target);
                                    }
                                });
                            }
                        }

                        extendAndGenerateOp(generator, obj.drawingNode, operationProperties, target);
                    });

                    // Applying the operations (undo group is created automatically)
                    model.applyOperations(generator);
                    generatedOp = true;
                }
            }

            function finalizeRotateDrawing() {
                _.each(allDrawings, function (obj) {
                    obj.drawingNode.removeClass(DrawingFrame.ROTATING_STATE_CLASSNAME);
                    DrawingFrame.removeRotateAngleHint(obj.drawingNode);
                    if (!generatedOp) { // reset to start angle if tracking was canceled
                        DrawingFrame.updateCssTransform(obj.drawingNode, obj.rotation || 0);
                    } else {
                        DrawingFrame.updateResizersMousePointers(obj.drawingNode, DrawingFrame.getDrawingRotationAngle(model, obj.drawingNode));
                    }
                    obj.drawingNode.removeClass(DrawingFrame.FRAME_WITH_TEMP_BORDER2).data('selection').removeClass(DrawingFrame.ROTATING_STATE_CLASSNAME).css({ transform: obj.drawingNode.css('transform') });
                });

                allConnectors = clearModifiedConnectors(allConnectors);

                if (selection.isAdditionalTextframeSelection()) {
                    DrawingFrame.getTextFrameNode(drawingNode, { deep: true }).attr('contenteditable', true);
                    selection.restoreBrowserSelection();
                }
                leaveTracking();

                if (isMultiSelection) {
                    isMultiSelection = false;
                    allDrawings = [];
                }
                generatedOp = false;
            }

            // return the actual drawingRotateHandler() function
            return function (event) {

                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startRotateDrawing(event);
                        break;
                    case 'tracking:move':
                        updateRotate(event);
                        break;
                    case 'tracking:scroll':
                        updateRotateScroll(event);
                        break;
                    case 'tracking:end':
                        stopRotateDrawing(event);
                        finalizeRotateDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeRotateDrawing(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();

                triggerDragging();
            };
        }()); // end of drawingRotateHandler local scope

        /**
         * Handler for moving adjustment handle of the shape.
         */
        var drawingAdjustHandler = (function () {
            var scrollX = 0, scrollY = 0;
            var deltaX, deltaY, normDeltas;
            // the start scroll position
            var startScrollX = 0, startScrollY = 0;
            var adjHandlerNode, adjPos;
            var cxnObject, ahList, avList, gdList;
            var moveX = false, moveY = false, isPolar = false;
            var xDiffpx, yDiffpx, minMaxRdiff;
            var drawingRotation, scaleX, scaleY, snapRect;
            var minXValue, maxXValue, minYValue, maxYValue;
            var minXpos, maxXpos, minYpos, maxYpos, minAng, maxAng;
            var minR, maxR, minRPolar, maxRPolar, minRposPx, maxRposPx;
            var absMinMaxSumX, absMinMaxSumY, absMinMaxSumR;
            var actXpos, actYpos, currentPolar, currentAng, currentRadius;
            var activeSlide = model.getSlideById(model.getActiveSlideId());
            var allSlideDrawings = null;

            function startAdjust(event) {
                adjHandlerNode = $(event.target);
                zoomFactor = app.getView().getZoomFactor();
                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();
                drawingRotation = DrawingFrame.getDrawingRotationAngle(model, drawingNode);
                scaleX = DrawingFrame.isFlippedHorz(drawingNode) ? -1 : 1;
                scaleY = DrawingFrame.isFlippedVert(drawingNode) ? -1 : 1;
                cxnObject = adjHandlerNode.data('cxnObj');
                avList = cxnObject.avList;
                gdList = cxnObject.gdList;
                ahList = cxnObject.ah;
                isPolar = ahList.type === 'polar';
                moveX = !_.isUndefined(ahList.minX);
                moveY = !_.isUndefined(ahList.minY);
                snapRect = new Rectangle(0, 0, drawingNode.width(), drawingNode.height());
                adjPos = DrawingFrame.getPointPixelPosition(ahList, snapRect, avList, gdList);
                allDrawings = [];
                allConnectors = [];
                allConnectorsOnSlide = model.getAllConnectorsOnSlide(activeSlide);
                allSlideDrawings = model.getAllDrawingsOnSlide(activeSlide);
                collectAndActivateAllMoveBoxes({ resize: true });

                if (isPolar) {
                    currentPolar = snapRect.pointToPolar(adjPos);
                    if (!_.isUndefined(ahList.minAng) || !_.isUndefined(ahList.maxAng)) {
                        minAng = DrawingFrame.getGeometryValue(ahList.minAng, snapRect, avList, gdList) / 60000;
                        maxAng = DrawingFrame.getGeometryValue(ahList.maxAng, snapRect, avList, gdList) / 60000;
                        currentAng = Math.round(currentPolar.a * 180 / Math.PI);
                        if (currentAng < 0) { currentAng += 360; }
                        currentAng = Utils.minMax(currentAng, minAng, maxAng);
                    }
                    if (!_.isUndefined(ahList.minR) || !_.isUndefined(ahList.maxR)) {
                        currentRadius = currentPolar.r;
                        minR = DrawingFrame.getGeometryValue(ahList.minR, snapRect, avList, gdList);
                        minRposPx = DrawingFrame.getPxPosFromGeo(minR, snapRect, avList, ahList, gdList, ahList.gdRefR);
                        minRPolar = snapRect.pointToPolar(minRposPx).r;

                        maxR = DrawingFrame.getGeometryValue(ahList.maxR, snapRect, avList, gdList);
                        maxRposPx = DrawingFrame.getPxPosFromGeo(maxR, snapRect, avList, ahList, gdList, ahList.gdRefR);
                        maxRPolar = snapRect.pointToPolar(maxRposPx).r;

                        absMinMaxSumR = Math.abs(maxR - minR);
                        minMaxRdiff = Math.abs(maxRPolar - minRPolar);
                    }
                }

                if (moveX) {
                    minXValue = DrawingFrame.getGeometryValue(ahList.minX, snapRect, avList, gdList);
                    minXpos = DrawingFrame.getPxPosFromGeo(minXValue, snapRect, avList, ahList, gdList, ahList.gdRefX);

                    maxXValue = DrawingFrame.getGeometryValue(ahList.maxX, snapRect, avList, gdList);
                    maxXpos = DrawingFrame.getPxPosFromGeo(maxXValue, snapRect, avList, ahList, gdList, ahList.gdRefX);

                    xDiffpx = Math.abs(maxXpos.x - minXpos.x);
                    absMinMaxSumX = Math.abs(maxXValue - minXValue);
                }
                if (moveY) {
                    minYValue = DrawingFrame.getGeometryValue(ahList.minY, snapRect, avList, gdList);
                    minYpos = DrawingFrame.getPxPosFromGeo(minYValue, snapRect, avList, ahList, gdList, ahList.gdRefY);

                    maxYValue = DrawingFrame.getGeometryValue(ahList.maxY, snapRect, avList, gdList);
                    maxYpos = DrawingFrame.getPxPosFromGeo(maxYValue, snapRect, avList, ahList, gdList, ahList.gdRefY);

                    yDiffpx = Math.abs(maxYpos.y - minYpos.y);
                    absMinMaxSumY = Math.abs(maxYValue - minYValue);
                }
            }

            function updateAdjust(event) {
                var allowX = isPolar || moveX;
                var allowY = isPolar || moveY;
                var modAttrs = { avList: _.copy(avList) };

                deltaX = (event.pageX - event.startX) / zoomFactor + scrollX;
                deltaY = (event.pageY - event.startY) / zoomFactor + scrollY;
                normDeltas = DrawingFrame.getNormalizedResizeDeltas(deltaX, deltaY, allowX, allowY, scaleX, scaleY, drawingRotation);
                actXpos = normDeltas.x + adjPos.x;
                actYpos = normDeltas.y + adjPos.y;
                if (moveX) {
                    actXpos = Utils.minMax(actXpos, Math.min(maxXpos.x, minXpos.x), Math.max(maxXpos.x, minXpos.x));
                    if (_.isFinite(actXpos)) { modAttrs.avList[ahList.gdRefX] = DrawingFrame.scaleAdjustmentValue(actXpos, minXpos.x, maxXpos.x, minXValue, xDiffpx, absMinMaxSumX); }
                }
                if (moveY) {
                    actYpos = Utils.minMax(actYpos, Math.min(maxYpos.y, minYpos.y), Math.max(maxYpos.y, minYpos.y));
                    if (_.isFinite(actYpos)) { modAttrs.avList[ahList.gdRefY] = DrawingFrame.scaleAdjustmentValue(actYpos, minYpos.y, maxYpos.y, minYValue, yDiffpx, absMinMaxSumY); }

                }
                if (isPolar) {
                    currentPolar = snapRect.pointToPolar({ x: actXpos, y: actYpos });
                    if (!_.isUndefined(minAng) || !_.isUndefined(maxAng)) {
                        currentAng = Math.round(currentPolar.a * 180 / Math.PI);
                        if (currentAng < 0) { currentAng += 360; }
                        currentAng = Utils.minMax(currentAng, minAng, maxAng);
                    }
                    if (!_.isUndefined(minR) || !_.isUndefined(maxR)) {
                        currentRadius = currentPolar.r;
                        currentRadius = Utils.minMax(currentRadius, Math.min(minRPolar, maxRPolar), Math.max(minRPolar, maxRPolar));
                    }
                    if (!_.isUndefined(ahList.gdRefAng)) {
                        modAttrs.avList[ahList.gdRefAng] = currentAng * 60000;
                    }
                    if (!_.isUndefined(ahList.gdRefR)) {
                        modAttrs.avList[ahList.gdRefR] = DrawingFrame.scaleAdjustmentValue(currentRadius, minRPolar, maxRPolar, minR, minMaxRdiff, absMinMaxSumR, true);
                    }
                }

                adjHandlerNode.css({ left: actXpos - 5, top: actYpos - 5 });
                _.each(allDrawings, function (oneDrawing) {
                    oneDrawing.moveBox.data('modAttrs', modAttrs);

                    var connections = DrawingUtils.getUniqueConnections(allConnectors, oneDrawing.id);
                    var oneDrawingAttrs = oneDrawing.attrs.drawing;
                    var optionFlipH = DrawingFrame.isFlippedHorz(oneDrawing.drawingNode);
                    var optionFlipV = DrawingFrame.isFlippedVert(oneDrawing.drawingNode);
                    var optionObj = DrawingUtils.generateRecalcConnectorOptions(oneDrawingAttrs, oneDrawingAttrs, 'adjust', optionFlipH, optionFlipV, oneDrawing.drawingNode, oneDrawingAttrs.rotation || 0, null, zoomFactor);
                    optionObj.avList = modAttrs.avList;
                    _.each(connections, function (connObj) {
                        var newConnProp = DrawingFrame.recalcConnectorPoints(allSlideDrawings, connObj, optionObj, rootNode, zoomFactor);
                        if (newConnProp) {
                            var newDrawingAttrs = newConnProp.attrs.drawing;
                            var calcFlipH = !_.isUndefined(newDrawingAttrs.flipH) ? newDrawingAttrs.flipH : connObj.attrs.drawing.flipH;
                            var calcFlipV = !_.isUndefined(newDrawingAttrs.flipV) ? newDrawingAttrs.flipV : connObj.attrs.drawing.flipV;
                            var transformProp = DrawingUtils.getCssTransform(0, calcFlipH, calcFlipV);

                            connObj.node.addClass(INDIRECT_CONNECTOR_CHANGE_CLASS);
                            connObj.moveBox.css({ left: Utils.convertHmmToLength(newDrawingAttrs.left, 'px'),
                                top: Utils.convertHmmToLength(newDrawingAttrs.top, 'px'),
                                width: Utils.convertHmmToLength((newDrawingAttrs.width || connObj.attrs.drawing.width), 'px'),
                                height: Utils.convertHmmToLength((newDrawingAttrs.height || connObj.attrs.drawing.height), 'px'),
                                transform: transformProp
                            });
                        }
                    });
                });
                updateAdjustmentShapeDebounced();
                updateConnectorsDebounced();
            }

            function updateAdjustScroll(event) {
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            function stopAdjust() {
                var generator = model.createOperationGenerator();
                var operationProperties = { attrs: { geometry: { avList: _.copy(avList) } }, start: Position.getOxoPosition(rootNode, drawingNode, 0) };
                var opCreated = false;
                var scaledValue = null;

                if (moveX && _.isFinite(actXpos)) {
                    scaledValue = DrawingFrame.scaleAdjustmentValue(actXpos, minXpos.x, maxXpos.x, minXValue, xDiffpx, absMinMaxSumX);
                    if (_.isFinite(scaledValue)) {
                        operationProperties.attrs.geometry.avList[ahList.gdRefX] = scaledValue;
                        opCreated = true;
                    }
                }
                if (moveY && _.isFinite(actYpos)) {
                    scaledValue = DrawingFrame.scaleAdjustmentValue(actYpos, minYpos.y, maxYpos.y, minYValue, yDiffpx, absMinMaxSumY);
                    if (_.isFinite(scaledValue)) {
                        operationProperties.attrs.geometry.avList[ahList.gdRefY] = scaledValue;
                        opCreated = true;
                    }
                }
                if (isPolar) {
                    if (!_.isUndefined(ahList.gdRefAng) && _.isFinite(currentAng)) {
                        operationProperties.attrs.geometry.avList[ahList.gdRefAng] = currentAng * 60000;
                        opCreated = true;
                    }
                    if (!_.isUndefined(ahList.gdRefR)) {
                        scaledValue = DrawingFrame.scaleAdjustmentValue(currentRadius, minRPolar, maxRPolar, minR, minMaxRdiff, absMinMaxSumR, true);
                        if (_.isFinite(scaledValue)) {
                            operationProperties.attrs.geometry.avList[ahList.gdRefR] = scaledValue;
                            opCreated = true;
                        }
                    }
                }
                if (opCreated) {
                    extendAndGenerateOp(generator, drawingNode, operationProperties, target);

                    _.each(allDrawings, function (obj) {
                        if (!DrawingFrame.isConnectorDrawingFrame(obj.drawingNode)) {
                            var connections = DrawingUtils.getUniqueConnections(allConnectors, obj.id);
                            if (connections.length) {
                                var oneDrawingAttrs = obj.attrs.drawing;
                                var optionFlipH = DrawingFrame.isFlippedHorz(obj.drawingNode);
                                var optionFlipV = DrawingFrame.isFlippedVert(obj.drawingNode);
                                var optionObj = DrawingUtils.generateRecalcConnectorOptions(oneDrawingAttrs, oneDrawingAttrs, 'adjust', optionFlipH, optionFlipV, obj.drawingNode, oneDrawingAttrs.rotation || 0, null, zoomFactor);
                                optionObj.avList = operationProperties.attrs.geometry.avList;

                                _.each(connections, function (connObj) {
                                    var connProp = DrawingFrame.recalcConnectorPoints(allSlideDrawings, connObj, optionObj, rootNode, zoomFactor);
                                    if (connProp) {
                                        connProp.start = Position.getOxoPosition(rootNode, connObj.node, 0);
                                        extendAndGenerateOp(generator, connObj.node, connProp, target);
                                    }
                                });
                            }
                        }
                    });

                    model.applyOperations(generator);
                    if (drawingNode && drawingNode.length) {
                        DrawingFrame.clearSelection(drawingNode);
                        selection.drawDrawingSelection(drawingNode);
                    }
                }
            }

            function finalizeAdjust() {
                leaveTracking();
                model.stopListeningTo(drawingNode, TRACKING_EVENT_NAMES);
                var drawingSelectionNode = drawingNode.data('selection');
                if (drawingSelectionNode) { drawingSelectionNode.off(TRACKING_EVENT_NAMES); }

                // Resetting variables for new mouseup events without mousemove
                scrollX = scrollY = 0;

                _.each(allDrawings, function (oneDrawing) {
                    clearDrawing(oneDrawing.drawingNode);
                });
                allDrawings = [];
                allConnectors = clearModifiedConnectors(allConnectors);
            }
            // return the actual drawingAdjustHandler() function
            return function (event) {
                event.preventDefault();
                switch (event.type) {
                    case 'tracking:start':
                        startAdjust(event);
                        break;
                    case 'tracking:move':
                        updateAdjust(event);
                        break;
                    case 'tracking:scroll':
                        updateAdjustScroll(event);
                        break;
                    case 'tracking:end':
                        stopAdjust(event);
                        finalizeAdjust(event);
                        break;
                    case 'tracking:cancel':
                        finalizeAdjust(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();
                triggerDragging();
            };
        }());

        /**
         * Leaving the tracking mode.
         */
        function leaveTracking() {
            DrawingFrame.toggleTracking(drawingNode.data('selection'), false);
        }

        /**
         * Handler for 'tracking:start' events for 'div.handle' and 'div.move'
         * nodes. This handler has to decide, which node got the event. For this
         * node all other tracking events have to be registered. Additionally the
         * handler is called with the 'tracking:start' event, so that the drawing
         * initialization happens.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event, that starts the moving or the resizing
         *  of a drawing.
         */
        function trackingStartHandler(event) {

            var pos = DrawingFrame.getResizerHandleType(event.target);

            if (_.isString(pos)) {
                if (_.isString(ImageCropFrame.getCropHandlerType($(event.target)))) {
                    ImageCropFrame.handleCropEvents(app, event, drawingNode);
                } else {
                    resizeNode = $(event.target);
                    resizeNode.off(TRACKING_EVENT_NAMES);
                    resizeNode.on(TRACKING_EVENT_NAMES, drawingResizeHandler);
                    drawingResizeHandler.call(this, event);
                }
            } else if (DrawingFrame.isRotateHandle(event.target)) {
                var rotateHadleNode = $(event.target);
                rotateHadleNode.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                rotateHadleNode.on(TRACKING_EVENT_NAMES, drawingRotateHandler);
                drawingRotateHandler.call(this, event);
            } else if (DrawingFrame.isAdjustmentHandle(event.target)) {
                var adjHadleNode = $(event.target);
                adjHadleNode.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                adjHadleNode.on(TRACKING_EVENT_NAMES, drawingAdjustHandler);
                drawingAdjustHandler.call(this, event);
            } else {
                if (DrawingFrame.isActiveCropping(drawingNode) && $(event.target).closest('.crop').length) {
                    ImageCropFrame.handleCropEvents(app, event, drawingNode);
                } else {
                    selectionDrawing.on(TRACKING_EVENT_NAMES, drawingMoveHandler); // not using 'listenTo' for temporary nodes
                    model.listenTo(drawingNode, TRACKING_EVENT_NAMES, drawingMoveHandler);
                    drawingMoveHandler.call(this, event);
                }
            }
        }

        function toggleTracking(node, state) {
            if (state) {
                Tracking.enableTracking(node, {
                    autoScroll: true,
                    borderNode: scrollNode,
                    borderMargin: -30,
                    borderSize: 60,
                    minSpeed: 10,
                    maxSpeed: 250,
                    trackModifiers: true
                });
            } else {
                Tracking.disableTracking(node);
            }
        }

        // helper function to set the size, position and rotation of a selection drawing node
        function updateSelectionDrawingSize(drawing) {
            if (drawing.data('selection')) {
                drawing.data('selection').css({ transform: drawing.css('transform'), left: drawing.css('left'), top: drawing.css('top'), width: drawing.width(), height: drawing.height() });
            }
        }

        // helper function to remove adjustment points from drawing selection node, if more than one drawing is selected
        function removeAdjPointsInMultiSelection() {
            _.each(selection.getSelectedDrawings(), function (oneDrawing) {
                var selDrawing = $(oneDrawing).data('selection');
                if (selDrawing) { DrawingFrame.clearAdjPoints(selDrawing); }
            });
        }

        // shifting the selection into an overlay node
        function shiftSelectionToOverlayNode() {

            var // the overlay node that contains the selection drawings
                overlayNode = model.getNode().children('.drawingselection-overlay'),
                // the empty drawing node that contains the selection
                selDrawing =  $('<div>').addClass('drawing selectiondrawing').css({ position: 'absolute' }),
                // a helper string for class names
                classesString = drawingNode.get(0).className;

            if (classesString) { selDrawing.addClass(classesString); }  // transferring drawing class names to the selection drawing node
            drawingNode.data('selection', selDrawing); // saving the drawing selection node at the drawing
            selectionBox.css('position', 'static');
            updateSelectionDrawingSize(drawingNode);
            selDrawing.append(selectionBox); // shifting the selection to the new empty drawing node
            overlayNode.append(selDrawing);

            return selDrawing;
        }

        // starting code of static method drawDrawingSelection()
        drawingNode = $(drawingNode);

        if (drawingNode.length !== 1) {
            Utils.error('DrawingResize.drawDrawingSelection(): single drawing node expected');
            drawingNode = drawingNode.first();
        }

        if (drawingNode.hasClass('horizontal-line')) {
            options.movable = false;
        } else if (drawingNode.hasClass('grouped')) {
            // options.movable = options.resizable = false;
            drawingNode = drawingNode.closest(DrawingFrame.GROUPCONTENT_SELECTOR).parent();
        }

        if (!app.isEditable()) {
            options.movable = options.resizable = options.rotatable = options.adjustable = false;
        }

        var currentZoomFactor    = app.getView().getZoomFactor();
        options.zoomValue        = 100 * currentZoomFactor;
        options.scaleHandles     = 1 / currentZoomFactor;
        options.rotation         = DrawingFrame.getDrawingRotationAngle(model, drawingNode) || 0;
        options.isMultiSelection = selection.isMultiSelection();

        if (selection.isMultiSelection()) { removeAdjPointsInMultiSelection(); }

        // removing an existing selection (possible for example in additional text frame selections)
        if (drawingNode.data('selection')) { DrawingFrame.clearSelection(drawingNode); }

        // removing all already registered tracking listeners
        // -> required for calls of 'drawDrawingSelection' without previous call of clearSelection
        model.stopListeningTo(drawingNode, ALL_TRACKING_EVENT_NAMES);

        var isCropMode = selection.isCropMode();
        if (isCropMode) { model.exitCropMode(); }

        // the container element used to visualize the selection
        selectionBox = DrawingFrame.drawSelection(drawingNode, options);

        if (isCropMode) { model.handleDrawingCrop(true); }

        // shifting the selection into an overlay node
        selectionDrawing = shiftSelectionToOverlayNode();

        toggleTracking(drawingNode, options.movable);
        toggleTracking(selectionDrawing, options.movable);
        // initialize resize tracking
        toggleTracking(selectionBox.find('>.resizers>[data-pos]'), options.resizable);
        // initialize rotate tracking
        toggleTracking(selectionBox.find('.rotate-handle'), options.rotatable);
        // initialize adjustment tracking
        toggleTracking(selectionBox.find('.adj-handle'), options.adjustable && !options.isMultiSelection);

        if (options.movable || options.resizeable || options.rotatable || options.adjustable) {
            selectionDrawing.on('tracking:start', trackingStartHandler); // not using 'listenTo' for temporary nodes because of registration in baseObject
            model.listenTo(drawingNode, 'tracking:start', trackingStartHandler);
        }
    };

    // exports ================================================================

    return DrawingSlideResize;

});
