/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/toolbartabcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/view/viewobjectmixin'
], function (Utils, TriggerObject, ViewObjectMixin) {

    'use strict';

    // class TabDescriptor ====================================================

    /**
     * Stores all settings for a single toolbar tab.
     *
     * @constructor
     */
    function TabDescriptor(tabId, initOptions) {

        this.tabId = tabId;
        this.visibleKey = Utils.getStringOption(initOptions, 'visibleKey', null);
        this.labelKey = Utils.getStringOption(initOptions, 'labelKey', null);
        this.priority = Utils.getIntegerOption(initOptions, 'priority', 0);
        this.sticky = Utils.getBooleanOption(initOptions, 'sticky', false);
        this.visible = true;
        this.label = null;

    } // class TabDescriptor

    // class ToolBarTabCollection =============================================

    /**
     * A collection containing all tool bar tabs. Provides information about
     * the visibility of the tool bars, according to the application state.
     *
     * Instances of this class trigger the following events:
     *  - 'tab:create'
     *      After a new tab has been created. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the new tab.
     *      (3) {Object} [options]
     *          Additional options passed while creating the new tab.
     *  - 'tab:show'
     *      After a tab has been shown (and was hidden before). Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the visible tab.
     *  - 'tab:hide'
     *      After a tab has been hidden (and was visible before). Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the hidden tab.
     *  - 'tab:activate'
     *      Before a new tab will be activated. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} newTabId
     *          The identifier of the activated tab.
     *      (3) {String} oldTabId
     *          The identifier of the deactivated tab.
     *  - 'tab:activated'
     *      After a new tab has been activated. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} newTabId
     *          The identifier of the activated tab.
     *      (3) {String} oldTabId
     *          The identifier of the deactivated tab.
     *  - 'tab:label'
     *      After the label text of a tab has been changed dynamically due to
     *      the current value of a controller item. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the tab.
     *      (3) {String} label
     *          The new label text for the tab.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends ViewObjectMixin
     */
    function ToolBarTabCollection(docView) {

        // self reference
        var self = this;

        // descriptors for all existing tool bar tabs
        var tabDescs = [];

        // identifier of the active tab
        var activeTabId = null;

        // identifier of the last active tab, while no tab is active (bug 39071)
        var lastTabId = null;

        // whether the first tab must be made visible
        var forceFirstTab = false;

        // base constructors --------------------------------------------------

        TriggerObject.call(this, docView);
        ViewObjectMixin.call(this, docView);

        // private methods ----------------------------------------------------

        /**
         * Returns the descriptor of the active tab.
         *
         * @returns {TabDescriptor|Null}
         *  The descriptor of the active tab.
         */
        function getActiveTab() {
            return _.findWhere(tabDescs, { tabId: activeTabId }) || null;
        }

        /**
         * Activates the first visible tab.
         *
         * @param {Array<TabDescriptor>} [filteredTabs]
         *  If specified, activates the first visible tab in the passed array
         *  that has a higher priority than the active tab (if available).
         *  Otherwise, searches for a visible tab in all available tabs.
         *
         * @returns {Boolean}
         *  Whether a visible tab has been found and activated successfully.
         */
        function activateFirstTab(filteredTabs) {

            // filter by visible tabs
            var visibleTabs = _.where(filteredTabs || tabDescs, { visible: true });
            // the active tab
            var activeTab = getActiveTab();

            // filter by priority of the active tab
            if (activeTab && activeTab.visible) {
                visibleTabs = visibleTabs.filter(function (tab) {
                    return tab.priority < activeTab.priority;
                });
            }

            // no tab available for activation
            if (visibleTabs.length === 0) {
                return false;
            }

            // activate tab with highest priority
            visibleTabs = _.sortBy(visibleTabs, 'priority');
            self.activateTab(visibleTabs[0].tabId);

            // flag for first tab visibility no longer needed
            forceFirstTab = false;

            return true;
        }

        /**
         * Updates the visibility of the tab buttons.
         */
        function controllerChangeHandler() {

            // the application controller
            var controller = docView.getApp().getController();
            // all tabs currently visible
            var visibleTabs = _.where(tabDescs, { visible: true });
            // all tabs currently hidden
            var hiddenTabs = _.where(tabDescs, { visible: false });
            // the active tab
            var activeTab = getActiveTab();

            // process all tabs
            tabDescs.forEach(function (tabDesc) {

                // tabs without visibility key are always visible
                var visible = !tabDesc.visibleKey || controller.isItemEnabled(tabDesc.visibleKey);
                // get dynamic label text
                var label = !tabDesc.labelKey || controller.getItemValue(tabDesc.labelKey);

                // show or hide the tabs, if the state has changed
                if (tabDesc.visible !== visible) {
                    tabDesc.visible = visible;
                    self.trigger(visible ? 'tab:show' : 'tab:hide', tabDesc.tabId);
                }

                // change the button text, if the item is contained in the event data
                if (_.isString(label) && (tabDesc.label !== label)) {
                    tabDesc.label = label;
                    self.trigger('tab:label', tabDesc.tabId, label);
                }
            });

            // do nothing, if active tab is visible and sticky
            if (activeTab && activeTab.visible && activeTab.sticky) {
                return;
            }

            // do not activate a tab, if some are visible, but none is active
            // -> but do activate the tab, if the flag 'forceFirstTab' is true.
            if ((visibleTabs.length > 0) && !activeTab && !forceFirstTab) { return; }

            // if the active tab is now hidden, activate the first visible tab; otherwise
            // activate the first tab with higher priority that changed from hidden to visible
            activateFirstTab((activeTab && activeTab.visible) ? hiddenTabs : null);
        }

        // public methods -----------------------------------------------------

        /**
         * Registers a new tool bar tab.
         *
         * @param {String} tabId
         *  The unique identifier for the new tab.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.visibleKey]
         *      The key of the controller item that controls the visibility of
         *      the tool bar tab. The visibility will be bound to the 'enabled'
         *      state of the respective controller item. If omitted, the tab
         *      will always be visible.
         *  @param {String} [options.labelKey]
         *      If specified, the key of a controller item whose value will be
         *      used to update the label text of the tab dynamically.
         *  @param {Number} [options.priority=0]
         *      The priority of the tool bar tab. Used to decide which tool bar
         *      tab gets activated when the tool bar tab currently active has
         *      been hidden by a controller item. The higher the number, the
         *      lower the priority of the tab.
         *  @param {Number} [options.sticky=false]
         *      Whether the tool bar tab should stay active regardless of other
         *      tabs becoming visible.
         *
         * @returns {ToolBarTabCollection}
         *  A reference to this instance.
         */
        this.createTab = function (tabId, options) {

            // the tab descriptor
            var tabDesc = new TabDescriptor(tabId, options);

            // insert new tab and notify listeners
            tabDescs.push(tabDesc);
            this.trigger('tab:create', tabId, options);

            // initially hide all tabs with existing visibility key
            if (tabDesc.visibleKey) {
                tabDesc.visible = false;
                this.trigger('tab:hide', tabId);
            }

            return this;
        };

        /**
         * Returns the identifier of the tab currently active.
         *
         * @returns {String|Null}
         *  The identifier of the tab that is currently activated.
         */
        this.getActiveTabId = function () {
            return activeTabId;
        };

        /**
         * Activates the specified tab. If another tab was currently active, a
         * 'tab:deactivate' event will be triggered for that tab. Afterwards,
         * if the passed tab identifier is valid, a 'tab:activate' event will
         * be triggered.
         *
         * @param {String|Null} tabId
         *  The identifier of the tab to be activated.
         *
         * @returns {ToolBarTabCollection}
         *  A reference to this instance.
         */
        this.activateTab = function (tabId) {
            if (activeTabId !== tabId) {
                var oldTabId = activeTabId;
                activeTabId = tabId;
                this.trigger('tab:activate', tabId, oldTabId);
                var updatePromise = docView.getApp().getController().update();
                // bug 39071: rescue last tab identifier, to be able to restore later
                if (!tabId) { lastTabId = oldTabId; }
                updatePromise.done(function () { self.trigger('tab:activated', tabId, oldTabId); });
            }
            return this;
        };

        /**
         * Activates the first visible tab.
         *
         * @returns {ToolBarTabCollection}
         *  A reference to this instance.
         */
        this.activateFirstTab = function () {
            activateFirstTab();
            return this;
        };

        /**
         * Restores the last visible tab, if no tab is currently visible.
         *
         * @returns {ToolBarTabCollection}
         *  A reference to this instance.
         */
        this.restoreActiveTab = function () {
            if (!activeTabId) {
                if (lastTabId) {
                    this.activateTab(lastTabId);
                    lastTabId = null;
                } else {
                    activateFirstTab();
                }
            }
            return this;
        };

        /**
         * Setting the flag, that the first tab can be shown. The first tool
         * bar shall not be seen too early, to avoid changes of tool bar,
         * before the document is loaded. This was typically the case from
         * 'file' to 'format'.
         */
        this.allowFirstTab = function () {
            forceFirstTab = true;
        };

        // initialization -----------------------------------------------------

        // update visibility of the tabs
        this.handleChangedControllerItems(controllerChangeHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = tabDescs = null;
        });

    } // class ToolBarTabCollection

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: ToolBarTabCollection });

});
