/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/filenamefield', [
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/textfield',
    'gettext!io.ox/office/editframework/main'
], function (Forms, TextField, gt) {

    'use strict';

    // class FileNameField ====================================================

    /**
     * A text field used to edit the file name.
     *
     * @constructor
     *
     * @extends TextField
     *
     * @param {EditView} docView
     *  The document view instance containing this control.
     */
    function FileNameField(docView) {

        // self reference
        var self = this;

        // a special busy spinner visible while committing the file name
        var busyOverlayNode = $('<div class="busy-overlay">').hide().busy();

        // base constructor ---------------------------------------------------

        TextField.call(this, docView, {
            classes: 'file-name',
            tooltip: gt('Rename document'),
            placeholder: gt('Document name'),
            width: 230,
            select: true,
            smallerVersion: { css: { width: 100 } }
        });

        // private methods ----------------------------------------------------

        /**
         * Sets a predefined tool tip in edit mode, or the document name in
         * read-only mode.
         */
        function updateToolTip() {
            var tooltip = docView.isEditable() ? gt('Rename document') : _.noI18n(docView.getApp().getFullFileName());
            Forms.setToolTip(self.getNode(), tooltip);
        }

        /**
         * Checks the filename, when the user gets the editrights. Maybe the
         * previous user does not finish renaming with "enter".
         */
        function refreshFielName() {
            var currentName = docView.getApp().getFullFileName(),
                newName     = docView.getApp().getFileDescriptor().filename;

            if (newName !== currentName) {
                docView.getApp().setNewName(newName);
            }
        }

        // initialization -----------------------------------------------------

        // show a busy spinner as long as the control commits the value
        // (the time between change event and losing the focus)
        this.getInputNode().after(busyOverlayNode);
        this.on('group:change', function () { busyOverlayNode.show(); });
        this.on('group:focus group:blur', function () { busyOverlayNode.hide(); });

        // set document name as tool tip in read-only mode
        this.listenTo(docView.getApp(), 'docs:editmode', updateToolTip);
        self.listenTo(docView.getApp(), 'docs:editmode:enter', refreshFielName);
        updateToolTip();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = busyOverlayNode = null;
        });

    } // class FileNameField

    // exports ================================================================

    // derive this class from class TextField
    return TextField.extend({ constructor: FileNameField });

});
